/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "conflictEntryView.h"

void	CConflictEntryListView::InitializePalette(const QPalette &p)
{
	active_highlight			= p.color(	QPalette::Active,
											QColorGroup::Highlight);
	active_highlighted_text		= p.color(	QPalette::Active,
											QColorGroup::HighlightedText);
	inactive_highlight			= p.color(	QPalette::Inactive,
											QColorGroup::Highlight);
	inactive_highlighted_text	= p.color(	QPalette::Inactive,
											QColorGroup::HighlightedText);

	QPalette	focus_palette = p;

	focus_palette.setColor(	QPalette::Active,
							QColorGroup::Highlight,
							inactive_highlight);

	focus_palette.setColor(	QPalette::Active,
							QColorGroup::HighlightedText,
							inactive_highlighted_text);

	setPalette(focus_palette);
}

void	CConflictEntryListView::focusInEvent(QFocusEvent *e)
{
	QPalette	p = palette();

	p.setColor(	QPalette::Active,
				QColorGroup::Highlight,
				active_highlight);

	p.setColor(	QPalette::Active,
				QColorGroup::HighlightedText,
				active_highlighted_text);

	setPalette(p);

	QListView::focusInEvent(e);
}

void	CConflictEntryListView::focusOutEvent(QFocusEvent *e)
{
	QPalette	p = palette();

	p.setColor(	QPalette::Active,
				QColorGroup::Highlight,
				inactive_highlight);

	p.setColor(	QPalette::Active,
				QColorGroup::HighlightedText,
				inactive_highlighted_text);

	setPalette(p);

	QListView::focusOutEvent(e);
}

CConflictEntryVBoxWidget::CConflictEntryVBoxWidget(
									QWidget *parent,
									CCompilerInterface *ci,
									SMainViewControl *mvc,
									COptionData::SSubViewData *d,
									CConflictEntryView *cev)
	: CVBoxWidget(parent, ci, mvc), compiler_interface(ci),
		subview_data(d), conflict_entry_view(cev),
		module_ptr_map(cev->GetModulePtrMap())
{
	label = new CLabelFc("conflict", this, true);

	label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	list_view = new CConflictEntryListView(this);

	list_view->header()->setMovingEnabled(false);
	list_view->setSorting(-1);
	list_view->addColumn("index");
	list_view->addColumn("entry");
	list_view->addColumn("actions");
	list_view->addColumn("resolve");

	setSpacing(0);
	setStretchFactor(label, 0);
	setStretchFactor(list_view, 1);

	compiler_data = compiler_interface->GetCompilerData();

	parser_calculated	= false;
	current_entry		= -1;
}

CConflictEntryVBoxWidget::~CConflictEntryVBoxWidget()
{
}

void	CConflictEntryVBoxWidget::InitializeModulePointers()
{
	UpdateListView();
}

int		IntegerLength(int i);
string	&IntegerString(int i, string &s);

void	CConflictEntryVBoxWidget::IndexString(int index, string &s)
{
	IntegerString(index, s);
}

void	CConflictEntryVBoxWidget::EntryString(CConflict *c, string &s)
{
	char	*s_name = symbols[c->terminal_symbol_index].name;

	ostrstream		os;

	os << "State = " << setw(state_index_width) << c->machine_state_index;
	os << "  Symbol = " << setw(symbol_index_width);
	os << c->terminal_symbol_index << ", " << s_name << " " << '\0';

	s = os.str();
}

void	CConflictEntryVBoxWidget::ActionSummaryString(CConflict *c, string &s)
{
	int		action_count = c->action_set->Size();

	ostrstream		os;

	if (c->action_set->FirstData() == shift_action)
		os << "Shift, " << action_count - 2 << " Reduce";
	else
		os << action_count - 1 << " Reduce";

	os << " " << '\0';

	s = os.str();
}

void	CConflictEntryVBoxWidget::ResolveString(CConflict *c, string &s)
{
	int		resolve_action = c->action_set->Data(c->resolve_listptr);

	ostrstream		os;

	if (resolve_action == shift_action)
		os << "Shift " << symbols[c->terminal_symbol_index].name;
	else if (resolve_action == conflict_entry)
		os << "Conflict Entry";
	else
		os << "Reduce By Rule " << resolve_action;

	os << '\0';

	s = os.str();
}

void	CConflictEntryVBoxWidget::StateString(CConflict *c, string &s)
{
	IntegerString(c->machine_state_index, s);
}

void	CConflictEntryVBoxWidget::SymbolString(CConflict *c, string &s)
{
	s = symbols[c->terminal_symbol_index].name;
}

void	CConflictEntryVBoxWidget::ActionStringSet(	CConflict *c,
													list<string> &actions)
{
	int		default_action = c->action_set->Data(c->default_listptr);

	VListInt::listptr	action_listptr = c->action_set->FirstListPtr();

	while (action_listptr)
	{
		int		a = c->action_set->NextData(action_listptr);

		ostrstream		os;

		if (a == default_action)
			os << "* ";
		else
			os << "  ";

		if (a == shift_action)
			os << "Shift " << symbols[c->terminal_symbol_index].name;
		else if (a == conflict_entry)
			os << "Conflict Entry";
		else
		{
			CRuleInformation	&r = rules[a];

			os << "Rule " << setw(rule_index_width) << a << "  ";

			string	l_name(symbols[r.left_symbol_index].name);
			string	pad(rule_left_name_width - l_name.length() + 1, ' ');

			os << l_name << pad << "->";

			for (int i=0; i<r.right_symbol_count; i++)
				os << " " << symbols[r.right_symbol_indexes[i]].name;
		}

		os << '\0';

		actions.push_back(os.str());
	}
}

int		CConflictEntryVBoxWidget::ResolveIndex(CConflict *c)
{
	int		resolve_action = c->action_set->Data(c->resolve_listptr);

	VListInt::listptr	action_listptr	= c->action_set->FirstListPtr();
	int					index			= 0;

	while (action_listptr)
	{
		int		a = c->action_set->NextData(action_listptr);

		if (a == resolve_action)
			return index;

		index++;
	}

	return 0;
}

void	CConflictEntryVBoxWidget::UpdateListView()
{
	compiler_data->CalculateOutputWidths();

	vector<CSubView *>		&v0 = module_ptr_map["MACHINE_VIEW"];

	vector<CSubView *>::size_type	j;

	for (j=0; j<v0.size(); j++)
	{
		CMachineView	*mv = dynamic_cast<CMachineView *>(v0[j]);

		mv->UpdateView(j);
	}

	QObject::disconnect(list_view,
						SIGNAL(currentChanged(QListViewItem *)),
						this,
						SLOT(SelectConflictEntry(QListViewItem *)));

	ListConflict::listptr	conflict_listptr;

	parser_calculated = compiler_interface->GetParserCalculated();

	if (parser_calculated)
	{
		CParse	*p = compiler_interface->GetParser();

		symbols				= p->symbols;
		rules				= p->rules;
		conflicts			= p->parsing_conflicts;
		shift_action		= p->shift_action;
		conflict_entry		= p->conflict_entry;

		int		c_count = conflicts->Size();
		int		l_count = list_view->childCount();
		int		i;

		if (c_count < l_count)
		{
			for (i=0; i<l_count-c_count; i++)
				delete list_view->firstChild();
		}
		else if (c_count > l_count)
		{
			for (i=0; i<c_count-l_count; i++)
				new QListViewItem(list_view);
		}

		int		maximum_state	= 0;
		int		maximum_symbol	= 0;

		conflict_listptr = conflicts->FirstListPtr();

		while (conflict_listptr)
		{
			CConflict	*c = conflicts->NextData(conflict_listptr);

			if (maximum_state < c->machine_state_index)
				maximum_state = c->machine_state_index;

			if (maximum_symbol < c->terminal_symbol_index)
				maximum_symbol = c->terminal_symbol_index;
		}

		state_index_width	= IntegerLength(maximum_state);
		symbol_index_width	= IntegerLength(maximum_symbol);

		rule_index_width		= compiler_data->GetRuleIndexWidth();
		rule_left_name_width	= compiler_data->GetRuleLeftNameWidth();

		conflict_listptr = conflicts->FirstListPtr();

		QListViewItemIterator	l_iterator(list_view);

		i = 0;

		while (conflict_listptr)
		{
			CConflict	*c = conflicts->NextData(conflict_listptr);

			QListViewItem	*item = l_iterator.current();

			string			s0;
			string			s1;
			string			s2;
			string			s3;

			IndexString(i, s0);
			EntryString(c, s1);
			ActionSummaryString(c, s2);
			ResolveString(c, s3);

			item->setText(0, s0.c_str());
			item->setText(1, s1.c_str());
			item->setText(2, s2.c_str());
			item->setText(3, s3.c_str());

			l_iterator++;
			i++;
		}

		if (c_count)
		{
			list_view->setSelected(list_view->firstChild(), true);

			current_entry = 0;
		}
		else
			current_entry = -1;
	}
	else
	{
		list_view->clear();

		current_entry = -1;
	}

	vector<CSubView *>		&v1 = module_ptr_map["CONFLICT_STATUS_VIEW"];
	vector<CSubView *>		&v2 = module_ptr_map["CONFLICT_ACTION_VIEW"];
	vector<CSubView *>		&v3 = module_ptr_map["CONFLICT_CODE_VIEW"];

	if (v1.size())
		csv = dynamic_cast<CConflictStatusView *>(v1[0]);
	else
		csv = 0;

	if (v2.size())
		cav = dynamic_cast<CConflictActionView *>(v2[0]);
	else
		cav = 0;

	if (v3.size())
		ccv = dynamic_cast<CConflictCodeView *>(v3[0]);
	else
		ccv = 0;

	UpdateConflictViews();

	QObject::connect(	list_view,
						SIGNAL(currentChanged(QListViewItem *)),
						this,
						SLOT(SelectConflictEntry(QListViewItem *)));
}

void	CConflictEntryVBoxWidget::UpdateConflictViews()
{
	string			index_string;
	string			state_string;
	string			symbol_string;
	string			resolve_string;
	list<string>	action_string_set;
	int				resolve_index;

	if (parser_calculated && current_entry != -1)
	{
		ListConflict::listptr	conflict_listptr =
									conflicts->FindIndex(current_entry);

		CConflict	*c = conflicts->Data(conflict_listptr);

		if (csv)
		{
			IndexString(current_entry, index_string);
			StateString(c, state_string);
			SymbolString(c, symbol_string);
			ResolveString(c, resolve_string);

			csv->UpdateStatusControl(	index_string,
										state_string,
										symbol_string,
										resolve_string);
		}

		if (cav)
		{
			ActionStringSet(c, action_string_set);

			resolve_index = ResolveIndex(c);

			cav->UpdateListBox(action_string_set, resolve_index);
		}

		if (ccv)
		{
			SConflictResolution
				*cr = compiler_data->FindConflictResolutionIndex(current_entry);

			if (cr)
				ccv->SetMultiLineEditText(cr->code_text);
			else
				ccv->ClearMultiLineEditText();
		}
	}
	else
	{
		if (csv)
			csv->ClearStatusControl();

		if (cav)
			cav->ClearListBox();

		if (ccv)
			ccv->ClearMultiLineEditText();
	}
}

void	CConflictEntryVBoxWidget::ClearCodeCurrentEntry()
{
	if (!parser_calculated || current_entry == -1)
		return;

	CMessageDialog		*msd = OptionDialog::message_dialog;

	msd->SetTitleAndMessageText("Clear Code",
								"Clear current conflict entry code ?");

	if (msd->ExecuteDialog() == QDialog::Rejected)
		return;

	compiler_interface->SetDataModifiedFlag(true);

	SConflictResolution
		*cr = compiler_data->FindConflictResolutionIndex(current_entry);

	if (cr)
		cr->code_text.clear();

	if (ccv)
		ccv->ClearMultiLineEditText();
}

void	CConflictEntryVBoxWidget::ClearCodeAll()
{
	if (!parser_calculated || current_entry == -1)
		return;

	CMessageDialog		*msd = OptionDialog::message_dialog;

	msd->SetTitleAndMessageText("Clear All",
								"Clear all conflict entry code ?");

	if (msd->ExecuteDialog() == QDialog::Rejected)
		return;

	compiler_interface->SetDataModifiedFlag(true);

	int		c_count = conflicts->Size();

	for (int i=0; i<c_count; i++)
	{
		SConflictResolution
			*cr = compiler_data->FindConflictResolutionIndex(i);

		if (cr)
			cr->code_text.clear();
	}

	if (ccv)
		ccv->ClearMultiLineEditText();
}

void	CConflictEntryVBoxWidget::DefaultResetCurrentEntry()
{
	if (!parser_calculated || current_entry == -1)
		return;

	compiler_interface->SetDataModifiedFlag(true);

	ListConflict::listptr	conflict_listptr =
								conflicts->FindIndex(current_entry);

	CConflict	*c = conflicts->Data(conflict_listptr);

	int		default_action = c->action_set->Data(c->default_listptr);

	VListInt::listptr	action_listptr	= c->action_set->FirstListPtr();
	int					index			= 0;

	while (action_listptr)
	{
		int		a = c->action_set->NextData(action_listptr);

		if (a == default_action)
			break;

		index++;
	}

	SelectConflictAction(index);

	if (cav)
		cav->SetCurrentListBoxItem(index);
}

void	CConflictEntryVBoxWidget::DefaultResetAll()
{
	if (!parser_calculated || current_entry == -1)
		return;

	compiler_interface->SetDataModifiedFlag(true);

	PostCurrentEntryChanges();

	ListConflict::listptr	conflict_listptr	= conflicts->FirstListPtr();
	int						index				= 0;

	while (conflict_listptr)
	{
		CConflict	*c = conflicts->NextData(conflict_listptr);

		int		default_action = c->action_set->Data(c->default_listptr);

		CParse	*p = compiler_interface->GetParser();

		p->ModifyActionTable(	index,
								c->machine_state_index,
								c->terminal_symbol_index,
								default_action);

		SConflictResolution
			*cr = compiler_data->FindConflictResolutionIndex(index);

		if (cr)
		{
			cr->action = default_action;

			list<string>::const_iterator	c_iterator = cr->code_text.begin();

			while (c_iterator != cr->code_text.end())
			{
				if ((*c_iterator).find_first_not_of("\t\n ") != string::npos)
					break;

				c_iterator++;
			}

			if (c_iterator == cr->code_text.end())
				compiler_data->DeleteConflictResolution(index);
		}

		index++;
	}

	UpdateListView();
}

void	CConflictEntryVBoxWidget::SelectConflictAction(int index)
{
	if (parser_calculated && current_entry != -1)
	{
		compiler_interface->SetDataModifiedFlag(true);

		ListConflict::listptr	conflict_listptr =
									conflicts->FindIndex(current_entry);

		CConflict	*c = conflicts->Data(conflict_listptr);

		VListInt::listptr	action_listptr = c->action_set->FindIndex(index);

		int		resolve_action = c->action_set->Data(action_listptr);

		CParse	*p = compiler_interface->GetParser();

		p->ModifyActionTable(	current_entry,
								c->machine_state_index,
								c->terminal_symbol_index,
								resolve_action);

		string		index_string;
		string		state_string;
		string		symbol_string;
		string		resolve_string;

		IndexString(current_entry, index_string);
		StateString(c, state_string);
		SymbolString(c, symbol_string);
		ResolveString(c, resolve_string);

		if (csv)
			csv->UpdateStatusControl(	index_string,
										state_string,
										symbol_string,
										resolve_string);

		QListViewItem	*item = list_view->currentItem();

		if (item)
			item->setText(3, resolve_string.c_str());

		SConflictResolution
			*cr = compiler_data->FindConflictResolutionIndex(current_entry);

		if (cr)
			cr->action = resolve_action;
		else
			compiler_data->AddConflictResolution(
								SConflictResolution(current_entry,
													c->machine_state_index,
													c->terminal_symbol_index,
													resolve_action));
	}
}

void	CConflictEntryVBoxWidget::GetCurrentEntryString(string &s)
{
	if (parser_calculated && current_entry != -1)
	{
		ListConflict::listptr	conflict_listptr =
									conflicts->FindIndex(current_entry);

		CConflict	*c = conflicts->Data(conflict_listptr);

		string		index_string;
		string		entry_string;

		IndexString(current_entry, index_string);
		EntryString(c, entry_string);

		s  = index_string;
		s += " ";
		s += entry_string;
	}
	else
		s = "";
}

void	CConflictEntryVBoxWidget::PostCurrentEntryChanges()
{
	if (parser_calculated && current_entry != -1)
	{
		ListConflict::listptr	conflict_listptr =
									conflicts->FindIndex(current_entry);

		CConflict	*c = conflicts->Data(conflict_listptr);

		int		resolve_action = c->action_set->Data(c->resolve_listptr);
		int		default_action = c->action_set->Data(c->default_listptr);

		SConflictResolution
			*cr = compiler_data->FindConflictResolutionIndex(current_entry);

		if (!cr)
		{
			compiler_data->AddConflictResolution(
								SConflictResolution(current_entry,
													c->machine_state_index,
													c->terminal_symbol_index,
													resolve_action));

			cr = compiler_data->FindConflictResolutionIndex(current_entry);
		}

		if (ccv)
			ccv->GetMultiLineEditText(cr->code_text);

		list<string>::const_iterator	c_iterator = cr->code_text.begin();

		while (c_iterator != cr->code_text.end())
		{
			if ((*c_iterator).find_first_not_of("\t\n ") != string::npos)
				break;

			c_iterator++;
		}

		if (c_iterator == cr->code_text.end())
		{
			cr->code_text.clear();

			if (resolve_action == default_action)
				compiler_data->DeleteConflictResolution(current_entry);
		}
	}
}

void	CConflictEntryVBoxWidget::SelectConflictEntry(QListViewItem *item)
{
	PostCurrentEntryChanges();

	if (item)
		current_entry = item->text(0).toInt();
	else
		current_entry = -1;

	UpdateConflictViews();
}

void	CConflictEntryVBoxWidget::RestoreUndoData()
{
	compiler_data = compiler_interface->GetCompilerData();

	vector<CSubView *>		&v0 = module_ptr_map["MACHINE_VIEW"];

	vector<CSubView *>::size_type	j;

	for (j=0; j<v0.size(); j++)
	{
		CMachineView	*mv = dynamic_cast<CMachineView *>(v0[j]);

		mv->RestoreUndoData();
	}
}

void	CConflictEntryVBoxWidget::CodeTextChanged()
{
	compiler_interface->SetDataModifiedFlag(true);
}

CConflictEntryView::CConflictEntryView(	QWidget *p,
										CViewNode *n,
										CInterfaceControl *ic,
										SMainViewControl *mvc,
										const string &dfs,
										CPaletteData *dpd,
										COptionData::SSubViewData *d)
	: CSubView(p, n, ic, mvc, dfs, dpd, d)
{
	frame = new CConflictEntryVBoxWidget(
		p, dynamic_cast<CCompilerInterface *>(ic), mvc, d, this);

	QFont		view_node_font;
	QPalette	view_node_palette;

	if (SetupViewNodeFont(view_node_font))
		frame->setFont(view_node_font);
	else
		frame->setFont(frame->font());

	if (SetupViewNodePalette(view_node_palette))
	{
		frame->setPalette(view_node_palette);
		frame->InitializePalette(view_node_palette);
	}
	else
	{
		frame->setPalette(frame->palette());
		frame->InitializePalette(frame->palette());
	}
}

CConflictEntryView::~CConflictEntryView()
{
	delete frame;
}

void	CConflictEntryView::SetHighlightPalette(CPaletteData *pd)
{
	restore_palette = frame->palette();

	QPalette	highlight_palette;

	pd->SetupPalette(highlight_palette);

	frame->setPalette(highlight_palette);
}

void	CConflictEntryView::RestorePalette()
{
	frame->setPalette(restore_palette);
}

void	CConflictEntryView::InitializeModulePointers()
{
	frame->InitializeModulePointers();
}

void	CConflictEntryView::UpdateListView()
{
	frame->UpdateListView();
}

void	CConflictEntryView::ClearCodeCurrentEntry()
{
	frame->ClearCodeCurrentEntry();
}

void	CConflictEntryView::ClearCodeAll()
{
	frame->ClearCodeAll();
}

void	CConflictEntryView::DefaultResetCurrentEntry()
{
	frame->DefaultResetCurrentEntry();
}

void	CConflictEntryView::DefaultResetAll()
{
	frame->DefaultResetAll();
}

void	CConflictEntryView::SelectConflictAction(int index)
{
	frame->SelectConflictAction(index);
}

void	CConflictEntryView::GetCurrentEntryString(string &s)
{
	frame->GetCurrentEntryString(s);
}

void	CConflictEntryView::PostCurrentEntryChanges()
{
	frame->PostCurrentEntryChanges();
}

void	CConflictEntryView::RestoreUndoData()
{
	frame->RestoreUndoData();
}

void	CConflictEntryView::CodeTextChanged()
{
	frame->CodeTextChanged();
}
