/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "errorSelectView.h"

CErrorSelectControl::CErrorSelectControl(CErrorSelectVBoxWidget *v)
	: CFrameFc((QWidget *)v, 1, 4), error_select_vbox(v)
{
	grid_layout->setRowStretch(0, 1);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 0);
	grid_layout->setColStretch(2, 0);
	grid_layout->setColStretch(3, 1);

	QString     size_string("##########");

	clear_code_button		= new CPushButtonFc("Clear Code",
									this, size_string);
	clear_all_button		= new CPushButtonFc("Clear All",
									this, size_string);
	edit_token_button		= new CPushButtonFc("Edit Token",
									this, size_string);
	label					= new CLabelFc("", this);

	label->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	label->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);

	grid_layout->addWidget(clear_code_button,		0, 0);
	grid_layout->addWidget(clear_all_button,		0, 1);
	grid_layout->addWidget(edit_token_button,		0, 2);
	grid_layout->addWidget(label,					0, 3);

	connect(clear_code_button,
			SIGNAL(clicked()),
			SLOT(ClearCodeCurrentFunction()));

	connect(clear_all_button,
			SIGNAL(clicked()),
			SLOT(ClearCodeAll()));

	connect(edit_token_button,
			SIGNAL(clicked()),
			SLOT(EditErrorToken()));
}

void	CErrorSelectControl::ClearSelectControl()
{
	label->clear();
}

void	CErrorSelectControl::UpdateSelectControl(const string &s)
{
	label->setText(s.c_str());
}

void	CErrorSelectControl::ClearCodeCurrentFunction()
{
	error_select_vbox->ClearCodeCurrentFunction();
}

void	CErrorSelectControl::ClearCodeAll()
{
	error_select_vbox->ClearCodeAll();
}

void	CErrorSelectControl::EditErrorToken()
{
	error_select_vbox->EditErrorToken();
}

void	CErrorSelectListView::InitializePalette(const QPalette &p)
{
	active_highlight			= p.color(	QPalette::Active,
											QColorGroup::Highlight);
	active_highlighted_text		= p.color(	QPalette::Active,
											QColorGroup::HighlightedText);
	inactive_highlight			= p.color(	QPalette::Inactive,
											QColorGroup::Highlight);
	inactive_highlighted_text	= p.color(	QPalette::Inactive,
											QColorGroup::HighlightedText);

	QPalette	focus_palette = p;

	focus_palette.setColor(	QPalette::Active,
							QColorGroup::Highlight,
							inactive_highlight);

	focus_palette.setColor(	QPalette::Active,
							QColorGroup::HighlightedText,
							inactive_highlighted_text);

	setPalette(focus_palette);
}

void	CErrorSelectListView::focusInEvent(QFocusEvent *e)
{
	QPalette	p = palette();

	p.setColor(	QPalette::Active,
				QColorGroup::Highlight,
				active_highlight);

	p.setColor(	QPalette::Active,
				QColorGroup::HighlightedText,
				active_highlighted_text);

	setPalette(p);

	QListView::focusInEvent(e);
}

void	CErrorSelectListView::focusOutEvent(QFocusEvent *e)
{
	QPalette	p = palette();

	p.setColor(	QPalette::Active,
				QColorGroup::Highlight,
				inactive_highlight);

	p.setColor(	QPalette::Active,
				QColorGroup::HighlightedText,
				inactive_highlighted_text);

	setPalette(p);

	QListView::focusOutEvent(e);
}

void	CErrorSelectListView::mouseDoubleClickEvent(QMouseEvent *e)
{
	error_select_vbox->EditErrorToken();

	e->accept();
}

void	CErrorSelectListView::keyPressEvent(QKeyEvent *e)
{
	switch (e->key())
	{
		case Qt::Key_Return:
		if (e->state() == Qt::NoButton)
		{
			e->accept();
			error_select_vbox->EditErrorToken();
			return;
		}
		break; 

		case Qt::Key_Enter:
		if (e->state() == Qt::Keypad)
		{
			e->accept();
			error_select_vbox->EditErrorToken();
			return;
		}
		break; 
	}

	e->ignore();
	QListView::keyPressEvent(e);
}

CErrorSelectVBoxWidget::CErrorSelectVBoxWidget(
									QWidget *parent,
									CCompilerInterface *ci,
									SMainViewControl *mvc,
									COptionData::SSubViewData *d,
									CErrorSelectView *esv)
	: CVBoxWidget(parent, ci, mvc), compiler_interface(ci),
		subview_data(d), error_select_view(esv),
		module_ptr_map(esv->GetModulePtrMap())
{
	label = new CLabelFc("error", this, true);

	label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	error_select_control = new CErrorSelectControl(this);

	list_view = new CErrorSelectListView(this);

	list_view->header()->setMovingEnabled(false);
	list_view->setSorting(-1);
	list_view->addColumn("index");
	list_view->addColumn("function");
	list_view->addColumn("token");

	setSpacing(0);
	setStretchFactor(label, 0);
	setStretchFactor(error_select_control, 0);
	setStretchFactor(list_view, 1);

	compiler_data = compiler_interface->GetCompilerData();

	error_entries_numbered	= false;
	current_function		= -1;

	error_data = 0;
}

CErrorSelectVBoxWidget::~CErrorSelectVBoxWidget()
{
	delete [] error_data;
}

void	CErrorSelectVBoxWidget::InitializeModulePointers()
{
	UpdateListView();
}

int		IntegerLength(int i);
string	&IntegerString(int i, string &s);

void	CErrorSelectVBoxWidget::IndexString(int index, string &s)
{
	IntegerString(index, s);
}

void	CErrorSelectVBoxWidget::UpdateListView()
{
	QObject::disconnect(list_view,
						SIGNAL(currentChanged(QListViewItem *)),
						this,
						SLOT(SelectErrorFunction(QListViewItem *)));

	error_entries_numbered = compiler_interface->GetErrorEntriesNumbered();

	if (error_entries_numbered)
	{
		CParse	*p = compiler_interface->GetParser();

		symbols = p->symbols;

		int		maximum_state	= p->machine_count - 1;
		int		maximum_symbol	=
					compiler_data->GetEndOfFileSymbol().GetSymbolIndex();

		state_index_width	= IntegerLength(maximum_state);
		symbol_index_width	= IntegerLength(maximum_symbol);

		delete [] error_data;

		CCompilerData::ErrorCalculation
			c = compiler_data->GetParserErrorCalculation();

		CCompilerData::ErrorFunctionAggregation
			aggregation = compiler_data->GetParserErrorFunctionAggregation();

		string		error_information;
					
		if (c == CCompilerData::NoCalculation)
		{
			error_information = "No Calculation, ";

			if (aggregation == CCompilerData::OneFunction)
			{
				number_of_error_functions = 1;

				error_information += "One Function";
			}
			else if (aggregation == CCompilerData::FunctionPerState)
			{
				number_of_error_functions = p->machine_count;

				error_information += "Function Per State";
			}
			else if (aggregation == CCompilerData::FunctionPerEntry)
			{
				number_of_error_functions = p->default_error_entry_count;

				error_information += "Function Per Entry";
			}
		}
		else if (	c == CCompilerData::ApproximateEssential ||
					c == CCompilerData::ExactEssential)
		{
			if (c == CCompilerData::ApproximateEssential)
				error_information = "Approximate Essential, ";
			else if (c == CCompilerData::ExactEssential)
				error_information = "Exact Essential, ";

			if (aggregation == CCompilerData::OneFunction)
			{
				number_of_error_functions = 2;

				error_information += "One Function";
			}
			else if (aggregation == CCompilerData::FunctionPerState)
			{
				number_of_error_functions = p->machine_count + 1;

				error_information += "Function Per State";
			}
			else if (aggregation == CCompilerData::FunctionPerEntry)
			{
				number_of_error_functions = p->essential_error_entry_count + 1;

				error_information += "Function Per Entry";
			}
		}

		error_data = new SErrorData[number_of_error_functions + 1];

		error_select_control->UpdateSelectControl(error_information);

		int		i;

		for (i=1; i<=number_of_error_functions; i++)
		{
			ostrstream	os;

			if (c == CCompilerData::NoCalculation)
			{
				if (aggregation == CCompilerData::OneFunction)
				{
					error_data[i] = SErrorData(-1, -1);

					os << "error function";
				}
				else if (aggregation == CCompilerData::FunctionPerState)
				{
					error_data[i] = SErrorData(i - 1, -1);

					os << "machine state " << i - 1;
				}
				else if (aggregation == CCompilerData::FunctionPerEntry)
				{
					error_data[i] = SErrorData(
						p->default_error_state_terminals[i].state,
						p->default_error_state_terminals[i].terminal);

					os << "machine state ";
					os << setw(state_index_width);
					os << p->default_error_state_terminals[i].state;
					os << "  terminal ";
					os << setw(symbol_index_width);
					os << p->default_error_state_terminals[i].terminal;
					os << ", ";
					os << p->symbols[
							p->default_error_state_terminals[i].terminal].name;
				}
			}
			else if (	c == CCompilerData::ApproximateEssential ||
						c == CCompilerData::ExactEssential)
			{
				if (aggregation == CCompilerData::OneFunction)
				{
					if (i == 1)
					{
						error_data[i] = SErrorData(-1, -1);

						os << "inaccessible error";
					}
					else
					{
						error_data[i] = SErrorData(-2, -1);

						os << "accessible error";
					}
				}
				else if (aggregation == CCompilerData::FunctionPerState)
				{
					if (i == 1)
					{
						error_data[i] = SErrorData(-1, -1);

						os << "inaccessible error";
					}
					else
					{
						error_data[i] = SErrorData(i - 2, -1);

						os << "machine state " << i - 2;
					}
				}
				else if (aggregation == CCompilerData::FunctionPerEntry)
				{
					if (i == 1)
					{
						error_data[i] = SErrorData(-1, -1);

						os << "inaccessible error";
					}
					else
					{
						error_data[i] = SErrorData(
							p->essential_error_state_terminals[i].state,
							p->essential_error_state_terminals[i].terminal);

						os << "machine state ";
						os << setw(state_index_width);
						os << p->essential_error_state_terminals[i].state;
						os << "  terminal ";
						os << setw(symbol_index_width);
						os << p->essential_error_state_terminals[i].terminal;
						os << ", ";
						os << p->symbols[
						p->essential_error_state_terminals[i].terminal].name;
					}
				}
			}

			os << '\0';

			error_data[i].function_string = os.str();

			SErrorCode		*ec = compiler_data->FindErrorCodeIndex(i);

			if (ec)
			{
				CErrorToken		*et =
					compiler_data->FindErrorTokenIndex(ec->token);

				if (et)
				{
					error_data[i].token			= ec->token;
					error_data[i].token_string	= et->GetName();
				}
			}
		}

		int		e_count = number_of_error_functions;
		int		l_count = list_view->childCount();

		if (e_count < l_count)
		{
			for (i=0; i<l_count-e_count; i++)
				delete list_view->firstChild();
		}
		else if (e_count > l_count)
		{
			for (i=0; i<e_count-l_count; i++)
				new QListViewItem(list_view);
		}

		QListViewItemIterator	l_iterator(list_view);

		for (i=1; i<=number_of_error_functions; i++)
		{
			QListViewItem	*item = l_iterator.current();

			string	s0;

			IndexString(i, s0);

			item->setText(0, s0.c_str());
			item->setText(1, error_data[i].function_string.c_str());
			item->setText(2, error_data[i].token_string.c_str());

			l_iterator++;
		}

		if (e_count)
		{
			list_view->setSelected(list_view->firstChild(), true);

			current_function = 1;
		}
		else
			current_function = -1;
	}
	else
	{
		list_view->clear();

		error_select_control->ClearSelectControl();

		current_function = -1;
	}

	vector<CSubView *>		&v1 = module_ptr_map["ERROR_CODE_VIEW"];

	if (v1.size())
		ecv = dynamic_cast<CErrorCodeView *>(v1[0]);
	else
		ecv = 0;

	UpdateErrorViews();

	QObject::connect(	list_view,
						SIGNAL(currentChanged(QListViewItem *)),
						this,
						SLOT(SelectErrorFunction(QListViewItem *)));
}

void	CErrorSelectVBoxWidget::UpdateErrorViews()
{
	if (error_entries_numbered && current_function != -1)
	{
		if (ecv)
		{
			SErrorCode
				*ec = compiler_data->FindErrorCodeIndex(current_function);

			if (ec)
				ecv->SetMultiLineEditText(ec->code_text);
			else
				ecv->ClearMultiLineEditText();
		}
	}
	else
	{
		if (ecv)
			ecv->ClearMultiLineEditText();
	}
}

void	CErrorSelectVBoxWidget::ClearCodeCurrentFunction()
{
	if (!error_entries_numbered || current_function == -1)
		return;

	CMessageDialog		*msd = OptionDialog::message_dialog;

	msd->SetTitleAndMessageText("Clear Code",
								"Clear current error function code ?");

	if (msd->ExecuteDialog() == QDialog::Rejected)
		return;

	compiler_interface->SetDataModifiedFlag(true);

	SErrorCode		*ec = compiler_data->FindErrorCodeIndex(current_function);

	if (ec)
		ec->code_text.clear();

	if (ecv)
		ecv->ClearMultiLineEditText();
}

void	CErrorSelectVBoxWidget::ClearCodeAll()
{
	if (!error_entries_numbered || current_function == -1)
		return;

	CMessageDialog		*msd = OptionDialog::message_dialog;

	msd->SetTitleAndMessageText("Clear All",
								"Clear all error function code ?");

	if (msd->ExecuteDialog() == QDialog::Rejected)
		return;

	compiler_interface->SetDataModifiedFlag(true);

	for (int i=1; i<=number_of_error_functions; i++)
	{
		SErrorCode	*ec = compiler_data->FindErrorCodeIndex(i);

		if (ec)
			ec->code_text.clear();
	}

	if (ecv)
		ecv->ClearMultiLineEditText();
}

void	CErrorSelectVBoxWidget::EditErrorToken()
{
	if (!error_entries_numbered || current_function == -1)
		return;

	CSimulatorErrorDialog	*sed = ParserDialog::simulator_error_dialog;

	int		simulator_error_token = error_data[current_function].token;

	ParserDialog::SetSimulatorErrorPaletteAndFont(
						compiler_interface->GetOptionData());

	sed->SetTitleAndResize("Simulator Error Token");
	sed->SetValuePointer(&simulator_error_token);
	sed->SetCompilerDataPointer(compiler_data);

	if (sed->exec() == QDialog::Rejected)
		return;

	compiler_interface->SetDataModifiedFlag(true);

	error_data[current_function].token = simulator_error_token;

	if (simulator_error_token == -1)
		error_data[current_function].token_string = "(none)";
	else
	{
		CErrorToken		*et =
			compiler_data->FindErrorTokenIndex(simulator_error_token);

		error_data[current_function].token_string = et->GetName();
	}

	QListViewItem	*item = list_view->currentItem();

	item->setText(2, error_data[current_function].token_string.c_str());
}

void	CErrorSelectVBoxWidget::GetCurrentFunctionString(string &s)
{
	if (error_entries_numbered && current_function != -1)
	{
		string		index_string;

		IndexString(current_function, index_string);

		s  = index_string;
		s += " ";
		s += error_data[current_function].function_string;
	}
	else
		s = "";
}

void	CErrorSelectVBoxWidget::PostCurrentFunctionChanges()
{
	if (error_entries_numbered && current_function != -1)
	{
		SErrorCode	*ec = compiler_data->FindErrorCodeIndex(current_function);

		if (!ec)
		{
			compiler_data->AddErrorCode(
							SErrorCode(	current_function,
										error_data[current_function].state,
										error_data[current_function].symbol,
										error_data[current_function].token));

			ec = compiler_data->FindErrorCodeIndex(current_function);
		}
		else
			ec->token = error_data[current_function].token;

		if (ecv)
			ecv->GetMultiLineEditText(ec->code_text);

		list<string>::const_iterator	c_iterator = ec->code_text.begin();

		while (c_iterator != ec->code_text.end())
		{
			if ((*c_iterator).find_first_not_of("\t\n ") != string::npos)
				break;

			c_iterator++;
		}

		if (c_iterator == ec->code_text.end())
		{
			ec->code_text.clear();

			if (ec->token == -1)
				compiler_data->DeleteErrorCode(current_function);
		}
	}
}

void	CErrorSelectVBoxWidget::SelectErrorFunction(QListViewItem *item)
{
	PostCurrentFunctionChanges();

	if (item)
		current_function = item->text(0).toInt();
	else
		current_function = -1;

	UpdateErrorViews();
}

int		CErrorSelectVBoxWidget::GetSimulatorErrorToken(int index)
{
	if (error_entries_numbered &&
		index >= 1 && index <= number_of_error_functions)
		return error_data[index].token;
	else
		return -1;
}

void	CErrorSelectVBoxWidget::RestoreUndoData()
{
	compiler_data = compiler_interface->GetCompilerData();
}

void	CErrorSelectVBoxWidget::CodeTextChanged()
{
	compiler_interface->SetDataModifiedFlag(true);
}

CErrorSelectView::CErrorSelectView(	QWidget *p,
									CViewNode *n,
									CInterfaceControl *ic,
									SMainViewControl *mvc,
									const string &dfs,
									CPaletteData *dpd,
									COptionData::SSubViewData *d)
	: CSubView(p, n, ic, mvc, dfs, dpd, d)
{
	frame = new CErrorSelectVBoxWidget(
		p, dynamic_cast<CCompilerInterface *>(ic), mvc, d, this);

	QFont		view_node_font;
	QPalette	view_node_palette;

	if (SetupViewNodeFont(view_node_font))
		frame->setFont(view_node_font);
	else
		frame->setFont(frame->font());

	if (SetupViewNodePalette(view_node_palette))
	{
		frame->setPalette(view_node_palette);
		frame->InitializePalette(view_node_palette);
	}
	else
	{
		frame->setPalette(frame->palette());
		frame->InitializePalette(frame->palette());
	}
}

CErrorSelectView::~CErrorSelectView()
{
	delete frame;
}

void	CErrorSelectView::SetHighlightPalette(CPaletteData *pd)
{
	restore_palette = frame->palette();

	QPalette	highlight_palette;

	pd->SetupPalette(highlight_palette);

	frame->setPalette(highlight_palette);
}

void	CErrorSelectView::RestorePalette()
{
	frame->setPalette(restore_palette);
}

void	CErrorSelectView::InitializeModulePointers()
{
	frame->InitializeModulePointers();
}

void	CErrorSelectView::UpdateListView()
{
	frame->UpdateListView();
}

void	CErrorSelectView::ClearCodeCurrentFunction()
{
	frame->ClearCodeCurrentFunction();
}

void	CErrorSelectView::ClearCodeAll()
{
	frame->ClearCodeAll();
}

void	CErrorSelectView::EditErrorToken()
{
	frame->EditErrorToken();
}

void	CErrorSelectView::GetCurrentFunctionString(string &s)
{
	frame->GetCurrentFunctionString(s);
}

void	CErrorSelectView::PostCurrentFunctionChanges()
{
	frame->PostCurrentFunctionChanges();
}

int		CErrorSelectView::GetSimulatorErrorToken(int index)
{
	return frame->GetSimulatorErrorToken(index);
}

void	CErrorSelectView::RestoreUndoData()
{
	frame->RestoreUndoData();
}

void	CErrorSelectView::CodeTextChanged()
{
	frame->CodeTextChanged();
}
