/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "errorTokenView.h"

void	CErrorTokenListView::InitializePalette(const QPalette &p)
{
	active_highlight			= p.color(	QPalette::Active,
											QColorGroup::Highlight);
	active_highlighted_text		= p.color(	QPalette::Active,
											QColorGroup::HighlightedText);
	inactive_highlight			= p.color(	QPalette::Inactive,
											QColorGroup::Highlight);
	inactive_highlighted_text	= p.color(	QPalette::Inactive,
											QColorGroup::HighlightedText);

	QPalette	focus_palette = p;

	focus_palette.setColor(	QPalette::Active,
							QColorGroup::Highlight,
							inactive_highlight);

	focus_palette.setColor(	QPalette::Active,
							QColorGroup::HighlightedText,
							inactive_highlighted_text);

	setPalette(focus_palette);
}

void	CErrorTokenListView::focusInEvent(QFocusEvent *e)
{
	QPalette	p = palette();

	p.setColor(	QPalette::Active,
				QColorGroup::Highlight,
				active_highlight);

	p.setColor(	QPalette::Active,
				QColorGroup::HighlightedText,
				active_highlighted_text);

	setPalette(p);

	QListView::focusInEvent(e);
}

void	CErrorTokenListView::focusOutEvent(QFocusEvent *e)
{
	QPalette	p = palette();

	p.setColor(	QPalette::Active,
				QColorGroup::Highlight,
				inactive_highlight);

	p.setColor(	QPalette::Active,
				QColorGroup::HighlightedText,
				inactive_highlighted_text);

	setPalette(p);

	QListView::focusOutEvent(e);
}

void	CErrorTokenListView::mouseDoubleClickEvent(QMouseEvent *e)
{
	error_token_vbox->EditErrorToken();

	e->accept();
}

void	CErrorTokenListView::keyPressEvent(QKeyEvent *e)
{
	switch (e->key())
	{
		case Qt::Key_Insert:
		switch (e->state())
		{
			case Qt::NoButton:
			case Qt::Keypad:
			e->accept();
			error_token_vbox->InsertAfterErrorToken();
			return;

			case Qt::ShiftButton:
			e->accept();
			error_token_vbox->InsertBeforeErrorToken();
			return;
		}
		break; 

		case Qt::Key_0:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			error_token_vbox->InsertBeforeErrorToken();
			return;
		}
		break; 

		case Qt::Key_Delete:
		if (e->state() == Qt::ShiftButton)
		{
			e->accept();
			error_token_vbox->DeleteErrorToken();
			return;
		}
		break; 

		case Qt::Key_Period:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			error_token_vbox->DeleteErrorToken();
			return;
		}
		break; 

		case Qt::Key_Return:
		if (e->state() == Qt::NoButton)
		{
			e->accept();
			error_token_vbox->EditErrorToken();
			return;
		}
		break; 

		case Qt::Key_Enter:
		if (e->state() == Qt::Keypad)
		{
			e->accept();
			error_token_vbox->EditErrorToken();
			return;
		}
		break; 

		case Qt::Key_Up:
		if (e->state() == Qt::ShiftButton)
		{
			e->accept();
			error_token_vbox->MoveErrorTokenUp();
			return;
		}
		break;

		case Qt::Key_8:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			error_token_vbox->MoveErrorTokenUp();
			return;
		}
		break;

		case Qt::Key_Down:
		if (e->state() == Qt::ShiftButton)
		{
			e->accept();
			error_token_vbox->MoveErrorTokenDown();
			return;
		}
		break;

		case Qt::Key_2:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			error_token_vbox->MoveErrorTokenDown();
			return;
		}
		break;
	}

	e->ignore();
	QListView::keyPressEvent(e);
}

CErrorTokenVBoxWidget::CErrorTokenVBoxWidget(	QWidget *parent,
												CCompilerInterface *ci,
												SMainViewControl *mvc,
												COptionData::SSubViewData *d,
												CErrorTokenView *ev)
	: CVBoxWidget(parent, ci, mvc), compiler_interface(ci),
		subview_data(d), error_token_view(ev),
		module_ptr_map(ev->GetModulePtrMap())
{
	label = new CLabelFc("error token", this, true);

	label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	list_view = new CErrorTokenListView(this);

	list_view->header()->setMovingEnabled(false);
	list_view->setSorting(-1);
	list_view->addColumn("index");
	list_view->addColumn("symbol");
	list_view->addColumn("name");
	list_view->addColumn("sync");

	setSpacing(0);
	setStretchFactor(label, 0);
	setStretchFactor(list_view, 1);

	compiler_data = compiler_interface->GetCompilerData();

	UpdateListView();

	list_view->setSelected(list_view->firstChild(), true);
}

CErrorTokenVBoxWidget::~CErrorTokenVBoxWidget()
{
}

void	CErrorTokenVBoxWidget::UpdateListView()
{
	const list<CErrorToken *>	&e_list = compiler_data->GetErrorTokenList();

	int		e_count = e_list.size();
	int		l_count = list_view->childCount();
	int		i;

	if (e_count < l_count)
	{
		for (i=0; i<l_count-e_count; i++)
			delete list_view->firstChild();
	}
	else if (e_count > l_count)
	{
		for (i=0; i<e_count-l_count; i++)
			new QListViewItem(list_view);
	}

	list<CErrorToken *>::const_iterator		e_iterator	= e_list.begin();

	QListViewItemIterator	l_iterator(list_view);

	while (e_iterator != e_list.end())
	{
		const CErrorToken		*e = *e_iterator;

		QListViewItem	*item = l_iterator.current();

		string		s_index;
		string		s_symbol_index;
		string		s_synchronization_length;

		e->GetIndexString(s_index);
		e->GetSymbolIndexString(s_symbol_index);
		e->GetSynchronizationLengthString(s_synchronization_length);

		item->setText(0, s_index.c_str());
		item->setText(1, s_symbol_index.c_str());
		item->setText(2, e->GetName().c_str());
		item->setText(3, s_synchronization_length.c_str());

		e_iterator++;
		l_iterator++;
	}

	vector<CSubView *>		&v1 = module_ptr_map["NONTERMINAL_VIEW"];
	vector<CSubView *>		&v2 = module_ptr_map["IMPLICIT_VIEW"];
	vector<CSubView *>		&v3 = module_ptr_map["PRECEDENCE_VIEW"];
	vector<CSubView *>		&v4 = module_ptr_map["RULE_VIEW"];

	if (v1.size())
	{
		CNonTerminalView	*nv = dynamic_cast<CNonTerminalView *>(v1[0]);

		nv->UpdateListView();
	}

	if (v2.size())
	{
		CImplicitView		*iv = dynamic_cast<CImplicitView *>(v2[0]);

		iv->UpdateListView();
	}

	if (v3.size())
	{
		CPrecedenceView		*pv = dynamic_cast<CPrecedenceView *>(v3[0]);

		pv->UpdateListView();
	}

	if (v4.size())
	{
		CRuleView			*rv = dynamic_cast<CRuleView *>(v4[0]);

		rv->UpdateListView();
	}
}

void	CErrorTokenVBoxWidget::SetSelectedItem(int i)
{
	int		l_count = list_view->childCount();

	if (i < 0 || !l_count)
		return;

	if (i >= l_count)
		i = l_count - 1;

	QListViewItemIterator	l_iterator(list_view);
	int						count;

	for (count=0; count<i; count++)
		l_iterator++;

	list_view->setSelected(l_iterator.current(), true);
	list_view->ensureItemVisible(l_iterator.current());
}

void	CErrorTokenVBoxWidget::InitializeModulePointers()
{
}

void	CErrorTokenVBoxWidget::InsertBeforeErrorToken()
{
	GrammarDialog::SetErrorTokenPaletteAndFont(this);

	CErrorToken				error_token(0);
	CErrorTokenDialog		*ed = GrammarDialog::error_token_dialog;

	ed->SetTitleAndResize("Insert Before Error Token");
	ed->SetErrorTokenPointer(&error_token);
	ed->SetControlValues(error_token);

	if (ed->exec() == QDialog::Accepted)
	{
		QListViewItem	*current = list_view->currentItem();
		int				selected = 0;

		if (current)
			selected = current->text(0).toInt();

		compiler_interface->StoreUndoData();

		InsertBeforeErrorToken(selected, error_token);

		SetSelectedItem(selected);

		compiler_interface->SetDataModifiedFlag(true);
	}
}

void	CErrorTokenVBoxWidget::InsertAfterErrorToken()
{
	GrammarDialog::SetErrorTokenPaletteAndFont(this);

	CErrorToken				error_token(0);
	CErrorTokenDialog		*ed = GrammarDialog::error_token_dialog;

	ed->SetTitleAndResize("Insert After Error Token");
	ed->SetErrorTokenPointer(&error_token);
	ed->SetControlValues(error_token);

	if (ed->exec() == QDialog::Accepted)
	{
		QListViewItem	*current = list_view->currentItem();
		int				selected = 0;

		if (current)
			selected = current->text(0).toInt();

		compiler_interface->StoreUndoData();

		InsertAfterErrorToken(selected, error_token);

		SetSelectedItem(selected + 1);

		compiler_interface->SetDataModifiedFlag(true);
	}
}

void	CErrorTokenVBoxWidget::EditErrorToken()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	selected	= current->text(0).toInt();

		GrammarDialog::SetErrorTokenPaletteAndFont(this);

		CErrorToken			*ei = compiler_data->FindErrorTokenIndex(selected);
		CErrorToken			error_token(0);
		CErrorTokenDialog	*ed = GrammarDialog::error_token_dialog;

		ed->SetTitleAndResize("Edit Error Token");
		ed->SetErrorTokenPointer(&error_token);

		if (ei)
			ed->SetControlValues(*ei);
		else
			ed->SetControlValues(error_token);

		if (ed->exec() == QDialog::Accepted)
		{
			compiler_interface->StoreUndoData();

			UpdateErrorToken(selected, error_token);

			SetSelectedItem(selected);

			compiler_interface->SetDataModifiedFlag(true);
		}
	}
}

void	CErrorTokenVBoxWidget::DeleteErrorToken()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	selected	= current->text(0).toInt();

		compiler_interface->StoreUndoData();

		DeleteErrorToken(selected);

		SetSelectedItem(selected);

		compiler_interface->SetDataModifiedFlag(true);
	}
}

void	CErrorTokenVBoxWidget::MoveErrorTokenUp()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	selected	= current->text(0).toInt();

		if (selected > 0)
		{
			compiler_interface->StoreUndoData();

			MoveErrorTokenUp(selected);

			SetSelectedItem(selected - 1);

			compiler_interface->SetDataModifiedFlag(true);
		}
	}
}

void	CErrorTokenVBoxWidget::MoveErrorTokenDown()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	l_count		= list_view->childCount();
		int	selected	= current->text(0).toInt();

		if (selected < l_count - 1)
		{
			compiler_interface->StoreUndoData();

			MoveErrorTokenDown(selected);

			SetSelectedItem(selected + 1);

			compiler_interface->SetDataModifiedFlag(true);
		}
	}
}

void	CErrorTokenVBoxWidget::InsertBeforeErrorToken(	int i,
														const CErrorToken &e)
{
	compiler_data->InsertBeforeErrorToken(i, e);

	UpdateListView();
}

void	CErrorTokenVBoxWidget::InsertAfterErrorToken(	int i,
														const CErrorToken &e)
{
	compiler_data->InsertAfterErrorToken(i, e);

	UpdateListView();
}

void	CErrorTokenVBoxWidget::UpdateErrorToken(int i, const CErrorToken &e)
{
	compiler_data->UpdateErrorToken(i, e);

	UpdateListView();
}

void	CErrorTokenVBoxWidget::DeleteErrorToken(int i)
{
	compiler_data->DeleteErrorToken(i);

	UpdateListView();
}

void	CErrorTokenVBoxWidget::MoveErrorTokenUp(int i)
{
	compiler_data->MoveErrorTokenUp(i);

	UpdateListView();
}

void	CErrorTokenVBoxWidget::MoveErrorTokenDown(int i)
{
	compiler_data->MoveErrorTokenDown(i);

	UpdateListView();
}

void	CErrorTokenVBoxWidget::RestoreUndoData()
{
	compiler_data = compiler_interface->GetCompilerData();
}

CErrorTokenView::CErrorTokenView(	QWidget *p,
									CViewNode *n,
									CInterfaceControl *ic,
									SMainViewControl *mvc,
									const string &dfs,
									CPaletteData *dpd,
									COptionData::SSubViewData *d)
	: CSubView(p, n, ic, mvc, dfs, dpd, d)
{
	frame = new CErrorTokenVBoxWidget(
		p, dynamic_cast<CCompilerInterface *>(ic), mvc, d, this);

	QFont		view_node_font;
	QPalette	view_node_palette;

	if (SetupViewNodeFont(view_node_font))
		frame->setFont(view_node_font);
	else
		frame->setFont(frame->font());

	if (SetupViewNodePalette(view_node_palette))
	{
		frame->setPalette(view_node_palette);
		frame->InitializePalette(view_node_palette);
	}
	else
	{
		frame->setPalette(frame->palette());
		frame->InitializePalette(frame->palette());
	}
}

CErrorTokenView::~CErrorTokenView()
{
	delete frame;
}

void	CErrorTokenView::SetHighlightPalette(CPaletteData *pd)
{
	restore_palette = frame->palette();

	QPalette	highlight_palette;

	pd->SetupPalette(highlight_palette);

	frame->setPalette(highlight_palette);
}

void	CErrorTokenView::RestorePalette()
{
	frame->setPalette(restore_palette);
}

void	CErrorTokenView::InitializeModulePointers()
{
	frame->InitializeModulePointers();
}

void	CErrorTokenView::InsertAfterErrorToken()
{
	frame->InsertAfterErrorToken();
}

void	CErrorTokenView::InsertBeforeErrorToken()
{
	frame->InsertBeforeErrorToken();
}

void	CErrorTokenView::EditErrorToken()
{
	frame->EditErrorToken();
}

void	CErrorTokenView::DeleteErrorToken()
{
	frame->DeleteErrorToken();
}

void	CErrorTokenView::MoveErrorTokenUp()
{
	frame->MoveErrorTokenUp();
}

void	CErrorTokenView::MoveErrorTokenDown()
{
	frame->MoveErrorTokenDown();
}

void	CErrorTokenView::UpdateListView()
{
	frame->UpdateListView();
}

void	CErrorTokenView::RestoreUndoData()
{
	frame->RestoreUndoData();
}
