/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "fontchangeControl.h"

CPushButtonFc::CPushButtonFc(	const QString &text,
								QWidget *parent,
								const QString &s)
	: QPushButton(text, parent), size_string(s)
{
	CalculateGeometry();
}

void	CPushButtonFc::setFont(const QFont &f)
{
	QPushButton::setFont(f);

	CalculateGeometry();
}

void	CPushButtonFc::CalculateGeometry()
{
	QFontInfo		fi(font());
	QFontMetrics	fm(font());

	int		pixel_size	= fi.pixelSize();
	int		text_height	= fm.height();
	int		text_width;

	if (size_string.length())
		text_width	= fm.width(size_string);
	else
		text_width	= fm.width(text());

	int		w = text_width + push_button_width_data[pixel_size] * 2;
	int		h = text_height + push_button_height_data[pixel_size] * 2;

	setMinimumSize(w, h);
}

CCheckBoxFc::CCheckBoxFc(const QString &text, QWidget *parent)
	: QCheckBox(text, parent)
{
	CalculateGeometry();
}

void	CCheckBoxFc::setFont(const QFont &f)
{
	QCheckBox::setFont(f);

	CalculateGeometry();
}

void	CCheckBoxFc::CalculateGeometry()
{
	QFontInfo		fi(font());
	QFontMetrics	fm(font());

	int		pixel_size	= fi.pixelSize();
	int		text_width	= fm.width(text());
	int		text_height	= fm.height();

	int		w = text_width + radio_button_width_data[pixel_size] * 4;
	int		h = text_height + radio_button_height_data[pixel_size] * 2;

	setMinimumSize(w, h);
}

CRadioButtonFc::CRadioButtonFc(const QString &text, QWidget *parent)
	: QRadioButton(text, parent)
{
	CalculateGeometry();
}

void	CRadioButtonFc::setFont(const QFont &f)
{
	QRadioButton::setFont(f);

	CalculateGeometry();
}

void	CRadioButtonFc::CalculateGeometry()
{
	QFontInfo		fi(font());
	QFontMetrics	fm(font());

	int		pixel_size	= fi.pixelSize();
	int		text_width	= fm.width(text());
	int		text_height	= fm.height();

	int		w = text_width + radio_button_width_data[pixel_size] * 4;
	int		h = text_height + radio_button_height_data[pixel_size] * 2;

	setMinimumSize(w, h);
}

CLabelFc::CLabelFc(const QString &text, QWidget *parent,
					bool hh_flag, bool m_flag, bool e_flag)
	: QLabel(text, parent),
		half_height_flag(hh_flag), maximum_size_flag(m_flag),
		extra_spacing_flag(e_flag)
{
	CalculateGeometry();
}

void	CLabelFc::setFont(const QFont &f)
{
	QLabel::setFont(f);

	CalculateGeometry();
}

QSize	CLabelFc::sizeHint() const
{
	return minimumSize();
}

void	CLabelFc::CalculateGeometry()
{
	string	label_text((const char *)(text().local8Bit()));

	StringToken::TokenStatus						token_status;
	vector<StringToken::CTokenValue *>				token_vector;
	vector<StringToken::CTokenValue *>::size_type	i;

	token_status = StringToken::GetCharTokenSequence(
						label_text, string("\n"), token_vector);

	vector<string>				text_lines;
	vector<string>::size_type	line_index;

	for (i=0; i<token_vector.size(); i++)
		text_lines.push_back(token_vector[i]->GetTokenString());

	StringToken::DeleteTokenVectorElements(token_vector);

	QFontInfo		fi(font());
	QFontMetrics	fm(font());

	int		pixel_size	= fi.pixelSize();

	int		m = grid_layout_margin_data[pixel_size];
	int		s = grid_layout_spacing_data[pixel_size];

	int		text_height;

	if (extra_spacing_flag)
		text_height	= (fm.height() + s) * text_lines.size() - s + 2 * m;
	else
		text_height	= fm.height() * text_lines.size();

	int		text_width	= 0;

	for (line_index=0; line_index<text_lines.size(); line_index++)
	{
		int		tw = fm.width(text_lines[line_index].c_str());

		if (text_width < tw)
			text_width = tw;
	}

	if (extra_spacing_flag)
		text_width += 2 * m;

	int		w;
	int		h;

	if (half_height_flag)
	{
		w = text_width + label_width_data[pixel_size] / 2;
		h = text_height + label_height_data[pixel_size] / 2;

	}
	else
	{
		w = text_width + label_width_data[pixel_size] * 2;
		h = text_height + label_height_data[pixel_size] * 2;
	}

	if (half_height_flag)
		setMaximumHeight(h);
	else if (maximum_size_flag)
		setMaximumSize(w, h);

	setMinimumSize(w, h);
}

CLineEditFc::CLineEditFc(QWidget *parent, const QString &s)
	: QLineEdit(parent), size_string(s)
{
	CalculateGeometry();
}

void	CLineEditFc::setFont(const QFont &f)
{
	QLineEdit::setFont(f);

	CalculateGeometry();
}

QSize	CLineEditFc::sizeHint() const
{
	QFontInfo		fi(font());
	QFontMetrics	fm(font());

	int		pixel_size	= fi.pixelSize();
	int		text_width	= fm.width(size_string);
	int		text_height	= fm.height();

	int		w = text_width + line_edit_width_data[pixel_size] * 2;
	int		h = text_height + line_edit_height_data[pixel_size] * 2;

	return QSize(w, h);
}

void	CLineEditFc::CalculateGeometry()
{
	QFontInfo		fi(font());
	QFontMetrics	fm(font());

	int		pixel_size	= fi.pixelSize();
	int		text_width	= fm.width(size_string);
	int		text_height	= fm.height();

	int		w = text_width + line_edit_width_data[pixel_size] * 2;
	int		h = text_height + line_edit_height_data[pixel_size] * 2;

	setMinimumSize(w, h);
}

CComboBoxFc::CComboBoxFc(bool rw, QWidget *parent, const QString &s)
	: QComboBox(rw, parent), size_string(s)
{
	CalculateGeometry();
}

void	CComboBoxFc::setFont(const QFont &f)
{
	QComboBox::setFont(f);

	CalculateGeometry();
}

void	CComboBoxFc::CalculateGeometry()
{
	QFontInfo		fi(font());
	QFontMetrics	fm(font());

	int		pixel_size	= fi.pixelSize();
	int		text_width	= fm.width(size_string);
	int		text_height	= fm.height();

	int		w = text_width + combo_box_width_data[pixel_size] * 2;
	int		h = text_height + combo_box_height_data[pixel_size] * 2;

	setMinimumSize(w, h);
}

CListBoxFc::CListBoxFc(QWidget *parent, const QString &s, int c, bool m_flag)
	: QListBox(parent), size_string(s),
		line_count(c), maximum_size_flag(m_flag)
{
	CalculateGeometry();
}

void	CListBoxFc::setFont(const QFont &f)
{
	QListBox::setFont(f);

	CalculateGeometry();
}

void	CListBoxFc::CalculateGeometry()
{
	QFontInfo		fi(font());
	QFontMetrics	fm(font());

	int		pixel_size	= fi.pixelSize();
	int		text_width	= fm.width(size_string);
	int		text_height	= fm.height();

	int		w = text_width + line_edit_width_data[pixel_size] * 2;
	int		h = text_height * line_count +
				grid_layout_margin_data[pixel_size] * 4 +
				grid_layout_spacing_data[pixel_size] * line_count;

	setMinimumSize(w, h);

	if (maximum_size_flag)
		setMaximumWidth(w);
}

CMultiLineEditFc::CMultiLineEditFc(QWidget *parent, const QString &s,
									int c, bool m_flag)
	: QMultiLineEdit(parent), size_string(s),
		line_count(c), maximum_size_flag(m_flag)
{
	CalculateGeometry();
}

void	CMultiLineEditFc::setFont(const QFont &f)
{
	QMultiLineEdit::setFont(f);

	CalculateGeometry();
}

void	CMultiLineEditFc::CalculateGeometry()
{
	QFontInfo		fi(font());
	QFontMetrics	fm(font());

	int		pixel_size	= fi.pixelSize();
	int		text_width	= fm.width(size_string);
	int		text_height	= fm.height();

	int		w = text_width + line_edit_width_data[pixel_size] * 2;
	int		h = text_height * line_count +
				grid_layout_margin_data[pixel_size] * 4 +
				grid_layout_spacing_data[pixel_size] * line_count;

	setMinimumSize(w, h);

	if (maximum_size_flag)
		setMaximumWidth(w);
}

CFrameFc::CFrameFc(QWidget *parent, int nRows, int nCols,
					const char *name, WFlags f)
	: QFrame(parent, name, f)
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		m = grid_layout_margin_data[pixel_size];
	int		s = grid_layout_spacing_data[pixel_size];

	grid_layout = new QGridLayout(this, nRows, nCols, m, s);
}

void	CFrameFc::setFont(const QFont &f)
{
	QFrame::setFont(f);

	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		m = grid_layout_margin_data[pixel_size];
	int		s = grid_layout_spacing_data[pixel_size];

	grid_layout->setMargin(m);
	grid_layout->setSpacing(s);
	grid_layout->invalidate();

	setMinimumSize(grid_layout->minimumSize());
	resize(grid_layout->minimumSize());
}

CHBoxFc::CHBoxFc(QWidget *parent, bool s_flag)
	: QHBox(parent), spacing_flag(s_flag)
{
	CalculateGeometry();
}

void	CHBoxFc::setFont(const QFont &f)
{
	QHBox::setFont(f);

	CalculateGeometry();
}

void	CHBoxFc::CalculateGeometry()
{
	if (spacing_flag)
	{
		QFontInfo		fi(font());

		int		pixel_size = fi.pixelSize();

		int		m = h_box_margin_data[pixel_size];
		int		s = h_box_spacing_data[pixel_size];

		setMargin(m);
		setSpacing(s);
	}
	else
	{
		setMargin(0);
		setSpacing(0);
	}
}

CVBoxFc::CVBoxFc(QWidget *parent, bool s_flag)
	: QVBox(parent), spacing_flag(s_flag)
{
	CalculateGeometry();
}

void	CVBoxFc::setFont(const QFont &f)
{
	QVBox::setFont(f);

	CalculateGeometry();
}

void	CVBoxFc::CalculateGeometry()
{
	if (spacing_flag)
	{
		QFontInfo		fi(font());

		int		pixel_size = fi.pixelSize();

		int		m = v_box_margin_data[pixel_size];
		int		s = v_box_spacing_data[pixel_size];

		setMargin(m);
		setSpacing(s);
	}
	else
	{
		setMargin(0);
		setSpacing(0);
	}
}

CGroupBoxFc::CGroupBoxFc(	int strips,
							Orientation orientation,
							const QString &title,
							QWidget *parent,
							const char *name)
	: QGroupBox(strips, orientation, title, parent, name)
{
	CalculateGeometry();
}

void	CGroupBoxFc::setFont(const QFont &f)
{
	QGroupBox::setFont(f);

	CalculateGeometry();
}

void	CGroupBoxFc::CalculateGeometry()
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		m = group_box_margin_data[pixel_size];
	int		s = group_box_spacing_data[pixel_size];

	setInsideMargin(m);
	setInsideSpacing(s);
}

CHGroupBoxFc::CHGroupBoxFc(	const QString &title,
							QWidget *parent,
							const char *name)
	: QHGroupBox(title, parent, name)
{
	CalculateGeometry();
}

void	CHGroupBoxFc::setFont(const QFont &f)
{
	QHGroupBox::setFont(f);

	CalculateGeometry();
}

void	CHGroupBoxFc::CalculateGeometry()
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		m = group_box_margin_data[pixel_size];
	int		s = group_box_spacing_data[pixel_size];

	setInsideMargin(m);
	setInsideSpacing(s);
}

CVGroupBoxFc::CVGroupBoxFc(	const QString &title,
							QWidget *parent,
							const char *name)
	: QVGroupBox(title, parent, name)
{
	CalculateGeometry();
}

void	CVGroupBoxFc::setFont(const QFont &f)
{
	QVGroupBox::setFont(f);

	CalculateGeometry();
}

void	CVGroupBoxFc::CalculateGeometry()
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		m = group_box_margin_data[pixel_size];
	int		s = group_box_spacing_data[pixel_size];

	setInsideMargin(m);
	setInsideSpacing(s);
}

CButtonGroupFc::CButtonGroupFc(	int strips,
								Orientation orientation,
								const QString &title,
								QWidget *parent)
	: QButtonGroup(strips, orientation, title, parent)
{
	CalculateGeometry();
}

void	CButtonGroupFc::setFont(const QFont &f)
{
	QButtonGroup::setFont(f);

	CalculateGeometry();
}

void	CButtonGroupFc::CalculateGeometry()
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		m = group_box_margin_data[pixel_size];
	int		s = group_box_spacing_data[pixel_size];

	setInsideMargin(m);
	setInsideSpacing(s);
}

CHButtonGroupFc::CHButtonGroupFc(const QString &title, QWidget *parent)
	: QHButtonGroup(title, parent)
{
	CalculateGeometry();
}

CHButtonGroupFc::CHButtonGroupFc(QWidget *parent)
	: QHButtonGroup(parent)
{
	CalculateGeometry();
}

void	CHButtonGroupFc::setFont(const QFont &f)
{
	QHButtonGroup::setFont(f);

	CalculateGeometry();
}

void	CHButtonGroupFc::CalculateGeometry()
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		m = group_box_margin_data[pixel_size];
	int		s = group_box_spacing_data[pixel_size];

	setInsideMargin(m);
	setInsideSpacing(s);
}

CVButtonGroupFc::CVButtonGroupFc(const QString &title, QWidget *parent)
	: QVButtonGroup(title, parent)
{
	CalculateGeometry();
}

void	CVButtonGroupFc::setFont(const QFont &f)
{
	QVButtonGroup::setFont(f);

	CalculateGeometry();
}

void	CVButtonGroupFc::CalculateGeometry()
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		m = group_box_margin_data[pixel_size];
	int		s = group_box_spacing_data[pixel_size];

	setInsideMargin(m);
	setInsideSpacing(s);
}

CSpacerFc::CSpacerFc(QWidget *parent, bool fs)
	: QWidget(parent), fixed_size_flag(fs)
{
	CalculateGeometry();
}

void	CSpacerFc::setFont(const QFont &f)
{
	QWidget::setFont(f);

	CalculateGeometry();
}

void	CSpacerFc::CalculateGeometry()
{
	QFontInfo		fi(font());

	int		pixel_size	= fi.pixelSize();

	int		w = grid_layout_margin_data[pixel_size];
	int		h = grid_layout_margin_data[pixel_size];

	if (fixed_size_flag)
		setFixedSize(w, h);
	else
		setMinimumSize(w, h);
}

CDialogFc::CDialogFc(QWidget *parent, int nRows, int nCols,
					bool modal, const char *name, WFlags f)
	: QDialog(parent, name, modal, f)
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		m = grid_layout_margin_data[pixel_size];
	int		s = grid_layout_spacing_data[pixel_size];

	grid_layout = new QGridLayout(this, nRows, nCols, m, s);
}

void	CDialogFc::setFont(const QFont &f)
{
	QDialog::setFont(f);

	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		m = grid_layout_margin_data[pixel_size];
	int		s = grid_layout_spacing_data[pixel_size];

	grid_layout->setMargin(m);
	grid_layout->setSpacing(s);
	grid_layout->invalidate();

	setMinimumSize(grid_layout->minimumSize());
	resize(grid_layout->minimumSize());
}

int		CResizeDialogFc::ExecuteDialog()
{
	if (parentWidget())
		QApplication::postEvent(this, new QEvent((QEvent::Type)ResizeDialogFc));

	return exec();
}

bool    CResizeDialogFc::event(QEvent *ev)
{
	switch ((int)ev->type())
	{
		case ResizeDialogFc:

		show();
		updateGeometry();

		CalculateSizeAndPosition();

		if (dialog_size != QSize(-1, -1))
			setMaximumSize(dialog_size);
		else
			setMaximumSize(10, 10);

		if (dialog_position != QPoint(-1, -1))
			move(dialog_position);

		return true;
	}

	return QWidget::event(ev);
}
