/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "grammarDialog.h"

CTerminalDialog		*GrammarDialog::terminal_dialog;
CErrorTokenDialog	*GrammarDialog::error_token_dialog;
CNonTerminalDialog	*GrammarDialog::nonterminal_dialog;
CPrecedenceDialog	*GrammarDialog::precedence_dialog;
CRuleDialog			*GrammarDialog::rule_dialog;
CUnitRuleDialog		*GrammarDialog::unit_rule_dialog;

void	GrammarDialog::CreateGrammarDialogs(QWidget *parent)
{
	terminal_dialog			= new CTerminalDialog(parent);
	error_token_dialog		= new CErrorTokenDialog(parent);
	nonterminal_dialog		= new CNonTerminalDialog(parent);
	precedence_dialog		= new CPrecedenceDialog(parent);
	rule_dialog				= new CRuleDialog(parent);
	unit_rule_dialog		= new CUnitRuleDialog(parent);
}

void	GrammarDialog::DestroyGrammarDialogs()
{
	delete terminal_dialog;
	delete error_token_dialog;
	delete nonterminal_dialog;
	delete precedence_dialog;
	delete rule_dialog;
	delete unit_rule_dialog;
}

void	GrammarDialog::SetTerminalPaletteAndFont(QWidget *w)
{
	static QWidget	*current_widget = 0;

	if (current_widget == w)
		return;
	else
		current_widget = w;

	terminal_dialog->setPalette(w->palette());
	terminal_dialog->setFont(w->font());
}

void	GrammarDialog::SetErrorTokenPaletteAndFont(QWidget *w)
{
	static QWidget	*current_widget = 0;

	if (current_widget == w)
		return;
	else
		current_widget = w;

	error_token_dialog->setPalette(w->palette());
	error_token_dialog->setFont(w->font());
}

void	GrammarDialog::SetNonTerminalPaletteAndFont(QWidget *w)
{
	static QWidget	*current_widget = 0;

	if (current_widget == w)
		return;
	else
		current_widget = w;

	nonterminal_dialog->setPalette(w->palette());
	nonterminal_dialog->setFont(w->font());
}

void	GrammarDialog::SetPrecedencePaletteAndFont(QWidget *w)
{
	static QWidget	*current_widget = 0;

	if (current_widget == w)
		return;
	else
		current_widget = w;

	precedence_dialog->setPalette(w->palette());
	precedence_dialog->setFont(w->font());
}

void	GrammarDialog::SetRulePaletteAndFont(QWidget *w)
{
	static QWidget	*current_widget = 0;

	if (current_widget == w)
		return;
	else
		current_widget = w;

	rule_dialog->setPalette(w->palette());
	rule_dialog->setFont(w->font());
}

void	GrammarDialog::SetUnitRulePaletteAndFont(QWidget *w)
{
	static QWidget	*current_widget = 0;

	if (current_widget == w)
		return;
	else
		current_widget = w;

	unit_rule_dialog->setPalette(w->palette());
	unit_rule_dialog->setFont(w->font());
}

CTerminalDialog::CTerminalDialog(QWidget *parent)
	: CDialogFc(parent, 5, 3, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1 = new QGridLayout(this, 1, 3, 0, s, "subgrid1");
	subgrid_layout2 = new QGridLayout(this, 2, 2, 0, s, "subgrid2");

	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);
	grid_layout->setRowStretch(3, 0);
	grid_layout->setRowStretch(4, 1);

	grid_layout->setColStretch(0, 1);
	grid_layout->setColStretch(1, 0);
	grid_layout->setColStretch(2, 0);

	subgrid_layout1->setRowStretch(0, 1);

	subgrid_layout1->setColStretch(0, 0);
	subgrid_layout1->setColStretch(1, 0);
	subgrid_layout1->setColStretch(2, 1);

	subgrid_layout2->setRowStretch(0, 1);
	subgrid_layout2->setRowStretch(1, 1);

	subgrid_layout2->setColStretch(0, 0);
	subgrid_layout2->setColStretch(1, 1);

	QString     size_string("######");

	type_button_group		= new CHButtonGroupFc("Type", this);
	filter_radio_button		= new CRadioButtonFc("FILTER", type_button_group);
	token_radio_button		= new CRadioButtonFc("TOKEN", type_button_group);
	driver_button_group		= new CHButtonGroupFc("Driver", this);
	literal_radio_button	= new CRadioButtonFc(	"LITERAL",
													driver_button_group);
	regexp_radio_button		= new CRadioButtonFc("REGEXP", driver_button_group);
	spacer1					= new CSpacerFc(this);
	name_label				= new CLabelFc("Name", this);
	name_line_edit			= new CLineEditFc(this,
										GrammarDialog::line_edit_string_size);
	regexp_label			= new CLabelFc("Regexp", this);
	regexp_line_edit		= new CLineEditFc(this,
										GrammarDialog::line_edit_string_size);
	ok_button				= new CPushButtonFc("Ok", this, size_string);
	cancel_button			= new CPushButtonFc("Cancel", this, size_string);
	spacer2					= new CSpacerFc(this);

	name_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
	regexp_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

	subgrid_layout1->addWidget(type_button_group,		0, 0);
	subgrid_layout1->addWidget(driver_button_group,		0, 1);
	subgrid_layout1->addWidget(spacer1,					0, 2);

	subgrid_layout2->addWidget(name_label,				0, 0);
	subgrid_layout2->addWidget(name_line_edit,			0, 1);
	subgrid_layout2->addWidget(regexp_label,			1, 0);
	subgrid_layout2->addWidget(regexp_line_edit,		1, 1);

	grid_layout->addMultiCellLayout(subgrid_layout1,	0, 0, 0, 2);
	grid_layout->addMultiCellLayout(subgrid_layout2,	1, 2, 0, 2);
	grid_layout->addWidget(ok_button,					3, 1);
	grid_layout->addWidget(cancel_button,				3, 2);
	grid_layout->addMultiCellWidget(spacer2,			4, 4, 0, 2);

	connect(ok_button, SIGNAL(clicked()), SLOT(AcceptOperation()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(reject()));
}

void	CTerminalDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();

	setMaximumHeight(height());
}

void	CTerminalDialog::SetControlValues(const CTerminal &t)
{
	if (t.GetFilterFlag())
		type_button_group->setButton(0);
	else
		type_button_group->setButton(1);

	if (t.GetLiteralFlag())
		driver_button_group->setButton(0);
	else
		driver_button_group->setButton(1);

	name_line_edit->setText(t.GetName().c_str());
	regexp_line_edit->setText(t.GetRegexp().c_str());
}

void	CTerminalDialog::setFont(const QFont &f)
{
	QFontInfo       fi(f);

	int     pixel_size = fi.pixelSize();

	int     s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1->setMargin(0);
	subgrid_layout1->setSpacing(s);
	subgrid_layout1->invalidate();

	subgrid_layout2->setMargin(0);
	subgrid_layout2->setSpacing(s);
	subgrid_layout2->invalidate();

	CDialogFc::setFont(f);
}

void	CTerminalDialog::AcceptOperation()
{
	if (filter_radio_button->isChecked())
		terminal_pointer->SetFilterFlag(true);
	else
		terminal_pointer->SetFilterFlag(false);

	if (literal_radio_button->isChecked())
		terminal_pointer->SetLiteralFlag(true);
	else
		terminal_pointer->SetLiteralFlag(false);

	terminal_pointer->SetName(
		(const char *)(name_line_edit->text().local8Bit()));

	terminal_pointer->SetRegexp(
		(const char *)(regexp_line_edit->text().local8Bit()));

	accept();
}

CErrorTokenDialog::CErrorTokenDialog(QWidget *parent)
	: CDialogFc(parent, 3, 1, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1 = new QGridLayout(this, 1, 2, 0, s, "subgrid1");
	subgrid_layout2 = new QGridLayout(this, 1, 5, 0, s, "subgrid2");

	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 1);

	grid_layout->setColStretch(0, 1);

	subgrid_layout1->setRowStretch(0, 1);

	subgrid_layout1->setColStretch(0, 0);
	subgrid_layout1->setColStretch(1, 1);

	subgrid_layout2->setRowStretch(0, 1);

	subgrid_layout2->setColStretch(0, 0);
	subgrid_layout2->setColStretch(1, 0);
	subgrid_layout2->setColStretch(2, 1);
	subgrid_layout2->setColStretch(3, 0);
	subgrid_layout2->setColStretch(4, 0);

	QString     size_string("######");

	name_label				= new CLabelFc("Name", this);
	name_line_edit			= new CLineEditFc(this,
										GrammarDialog::line_edit_string_size);
	sync_length_label		= new CLabelFc("Synchronization Length", this);
	sync_length_spinbox		= new QSpinBox(1, 100, 1, this);
	spacer1					= new CSpacerFc(this);
	ok_button				= new CPushButtonFc("Ok", this, size_string);
	cancel_button			= new CPushButtonFc("Cancel", this, size_string);
	spacer2					= new CSpacerFc(this);

	name_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
	sync_length_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

	subgrid_layout1->addWidget(name_label,				0, 0);
	subgrid_layout1->addWidget(name_line_edit,			0, 1);

	subgrid_layout2->addWidget(sync_length_label,		0, 0);
	subgrid_layout2->addWidget(sync_length_spinbox,		0, 1);
	subgrid_layout2->addWidget(spacer1,					0, 2);
	subgrid_layout2->addWidget(ok_button,				0, 3);
	subgrid_layout2->addWidget(cancel_button,			0, 4);

	grid_layout->addLayout(subgrid_layout1,				0, 0);
	grid_layout->addLayout(subgrid_layout2,				1, 0);
	grid_layout->addWidget(spacer2,						2, 0);

	connect(ok_button, SIGNAL(clicked()), SLOT(AcceptOperation()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(reject()));
}

void	CErrorTokenDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();

	setMaximumHeight(height());
}

void	CErrorTokenDialog::SetControlValues(const CErrorToken &e)
{
	name_line_edit->setText(e.GetName().c_str());
	sync_length_spinbox->setValue(e.GetSynchronizationLength());
}

void	CErrorTokenDialog::setFont(const QFont &f)
{
	QFontInfo       fi(f);

	int     pixel_size = fi.pixelSize();

	int     s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1->setMargin(0);
	subgrid_layout1->setSpacing(s);
	subgrid_layout1->invalidate();

	subgrid_layout2->setMargin(0);
	subgrid_layout2->setSpacing(s);
	subgrid_layout2->invalidate();

	CDialogFc::setFont(f);
}

void	CErrorTokenDialog::AcceptOperation()
{
	error_token_pointer->SetName(
		(const char *)(name_line_edit->text().local8Bit()));

	error_token_pointer->SetSynchronizationLength(sync_length_spinbox->value());

	accept();
}

CNonTerminalDialog::CNonTerminalDialog(QWidget *parent)
	: CDialogFc(parent, 3, 1, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1 = new QGridLayout(this, 1, 2, 0, s, "subgrid1");
	subgrid_layout2 = new QGridLayout(this, 1, 5, 0, s, "subgrid2");

	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 1);

	grid_layout->setColStretch(0, 1);

	subgrid_layout1->setRowStretch(0, 1);

	subgrid_layout1->setColStretch(0, 0);
	subgrid_layout1->setColStretch(1, 1);

	subgrid_layout2->setRowStretch(0, 1);

	subgrid_layout2->setColStretch(0, 0);
	subgrid_layout2->setColStretch(1, 0);
	subgrid_layout2->setColStretch(2, 1);
	subgrid_layout2->setColStretch(3, 0);
	subgrid_layout2->setColStretch(4, 0);

	QString     size_string("######");

	name_label				= new CLabelFc("Name", this);
	name_line_edit			= new CLineEditFc(this,
										GrammarDialog::line_edit_string_size);
	spacer1					= new CSpacerFc(this, true);
	start_symbol_check_box	= new CCheckBoxFc("Start Symbol", this);
	spacer2					= new CSpacerFc(this);
	ok_button				= new CPushButtonFc("Ok", this, size_string);
	cancel_button			= new CPushButtonFc("Cancel", this, size_string);
	spacer3					= new CSpacerFc(this);

	name_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

	subgrid_layout1->addWidget(name_label,				0, 0);
	subgrid_layout1->addWidget(name_line_edit,			0, 1);

	subgrid_layout2->addWidget(spacer1,					0, 0);
	subgrid_layout2->addWidget(start_symbol_check_box,	0, 1);
	subgrid_layout2->addWidget(spacer2,					0, 2);
	subgrid_layout2->addWidget(ok_button,				0, 3);
	subgrid_layout2->addWidget(cancel_button,			0, 4);

	grid_layout->addLayout(subgrid_layout1,				0, 0);
	grid_layout->addLayout(subgrid_layout2,				1, 0);
	grid_layout->addWidget(spacer3,						2, 0);

	connect(ok_button, SIGNAL(clicked()), SLOT(AcceptOperation()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(reject()));
}

void	CNonTerminalDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();

	setMaximumHeight(height());
}

void	CNonTerminalDialog::SetControlValues(const CNonTerminal &n)
{
	name_line_edit->setText(n.GetName().c_str());
	start_symbol_check_box->setChecked(n.GetStartSymbolFlag());
}

void	CNonTerminalDialog::setFont(const QFont &f)
{
	QFontInfo       fi(f);

	int     pixel_size = fi.pixelSize();

	int     s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1->setMargin(0);
	subgrid_layout1->setSpacing(s);
	subgrid_layout1->invalidate();

	subgrid_layout2->setMargin(0);
	subgrid_layout2->setSpacing(s);
	subgrid_layout2->invalidate();

	CDialogFc::setFont(f);
}

void	CNonTerminalDialog::AcceptOperation()
{
	nonterminal_pointer->SetName(
		(const char *)(name_line_edit->text().local8Bit()));

	nonterminal_pointer->SetStartSymbolFlag(
							start_symbol_check_box->isChecked());

	accept();
}

CPrecedenceDialog::CPrecedenceDialog(QWidget *parent)
	: CDialogFc(parent, 8, 2, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1 = new QGridLayout(this, 1, 2, 0, s, "subgrid1");
	subgrid_layout2 = new QGridLayout(this, 1, 2, 0, s, "subgrid2");

	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);
	grid_layout->setRowStretch(3, 0);
	grid_layout->setRowStretch(4, 0);
	grid_layout->setRowStretch(5, 0);
	grid_layout->setRowStretch(6, 1);
	grid_layout->setRowStretch(7, 0);

	grid_layout->setColStretch(0, 1);
	grid_layout->setColStretch(1, 0);

	subgrid_layout1->setRowStretch(0, 1);

	subgrid_layout1->setColStretch(0, 0);
	subgrid_layout1->setColStretch(1, 1);

	subgrid_layout2->setRowStretch(0, 1);

	subgrid_layout2->setColStretch(0, 0);
	subgrid_layout2->setColStretch(1, 1);

	QString     size_string("######");

	associativity_button_group	= new CHButtonGroupFc("Associativity", this);
	left_button				= new CRadioButtonFc(	"Left",
													associativity_button_group);
	right_button			= new CRadioButtonFc(	"Right",
													associativity_button_group);
	nonassociative_button	= new CRadioButtonFc(	"Non Associative",
													associativity_button_group);
	spacer1					= new CSpacerFc(this);
	entry_label				= new CLabelFc("Entries", this);
	entry_line_edit			= new CLineEditFc(this,
										GrammarDialog::line_edit_string_size);
	entry_listbox			= new CListBoxFc(this, "entry list box");
	add_button				= new CPushButtonFc("Add", this, size_string);
	delete_button			= new CPushButtonFc("Delete", this, size_string);
	ok_button				= new CPushButtonFc("Ok", this, size_string);
	cancel_button			= new CPushButtonFc("Cancel", this, size_string);
	spacer2					= new CSpacerFc(this);

	entry_listbox->setColumnMode(QListBox::FitToWidth);
	entry_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

	subgrid_layout1->addWidget(associativity_button_group,	0, 0);
	subgrid_layout1->addWidget(spacer1,						0, 1);

	subgrid_layout2->addWidget(entry_label,					0, 0);
	subgrid_layout2->addWidget(entry_line_edit,				0, 1);

	grid_layout->addMultiCellLayout(subgrid_layout1,			0, 0, 0, 1);
	grid_layout->addMultiCellLayout(subgrid_layout2,			1, 1, 0, 1);
	grid_layout->addMultiCellWidget(entry_listbox,				2, 6, 0, 0);
	grid_layout->addWidget(add_button,							2, 1);
	grid_layout->addWidget(delete_button,						3, 1);
	grid_layout->addWidget(ok_button,							4, 1);
	grid_layout->addWidget(cancel_button,						5, 1);
	grid_layout->addMultiCellWidget(spacer2,					7, 7, 0, 1);

	connect(add_button, SIGNAL(clicked()), SLOT(AddEntry()));
	connect(delete_button, SIGNAL(clicked()), SLOT(DeleteEntry()));
	connect(ok_button, SIGNAL(clicked()), SLOT(AcceptOperation()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(reject()));
}

void	CPrecedenceDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();
}

void	CPrecedenceDialog::SetCompilerDataPointer(CCompilerData *d)
{
	compiler_data = d;

	entry_listbox->clear();

	const list<CTerminal *>		&t_list = d->GetTerminalList();
	const list<CErrorToken *>	&e_list = d->GetErrorTokenList();
	const list<CRule *>			&r_list = d->GetRuleList();

	list<CTerminal *>::const_iterator		t_iterator = t_list.begin();
	list<CErrorToken *>::const_iterator		e_iterator = e_list.begin();

	while (t_iterator != t_list.end())
	{
		if (!(*t_iterator)->GetFilterFlag())
			new QListBoxText(	entry_listbox,
								(*t_iterator)->GetName().c_str());

		t_iterator++;
	}

	while (e_iterator != e_list.end())
		new QListBoxText(	entry_listbox,
							(*e_iterator++)->GetName().c_str());

	for (int i=0; i<r_list.size(); i++)
	{
		ostrstream		r_text;

		r_text << "RULE:" << i << '\0';

		new QListBoxText(entry_listbox, r_text.str());
	}
}

void	CPrecedenceDialog::SetControlValues(const CPrecedence &p)
{
	switch (p.GetAssociativity())
	{
		case CPrecedence::LeftAssociative:
		associativity_button_group->setButton(0);
		break;

		case CPrecedence::RightAssociative:
		associativity_button_group->setButton(1);
		break;

		case CPrecedence::NonAssociative:
		associativity_button_group->setButton(2);
		break;
	}

	string		entry_string;

	p.GetEntryString(entry_string);

	entry_line_edit->setText(entry_string.c_str());
}

void	CPrecedenceDialog::setFont(const QFont &f)
{
	QFontInfo       fi(f);

	int     pixel_size = fi.pixelSize();

	int     s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1->setMargin(0);
	subgrid_layout1->setSpacing(s);
	subgrid_layout1->invalidate();

	subgrid_layout2->setMargin(0);
	subgrid_layout2->setSpacing(s);
	subgrid_layout2->invalidate();

	CDialogFc::setFont(f);
}

void	CPrecedenceDialog::AddEntry()
{
	QString		e = entry_line_edit->text();

	if (!e.isEmpty())
		e += " ";

	e += entry_listbox->currentText();

	entry_line_edit->setText(e);
}

void	CPrecedenceDialog::DeleteEntry()
{
	StringToken::TokenStatus				token_status;
	vector<StringToken::CTokenValue *>		token_vector;

	token_status = StringToken::GetCharTokenSequence(
					(const char *)(entry_line_edit->text().local8Bit()),
					string("\t "), token_vector);

	if (token_status == StringToken::NoError && token_vector.size())
	{
		string											e;
		vector<StringToken::CTokenValue *>::size_type	i;

		for (i=0; i<token_vector.size()-1; i++)
		{
			if (i)
				e += " ";

			e += token_vector[i]->GetTokenString();
		}

		entry_line_edit->setText(e.c_str());
	}
	else
		entry_line_edit->setText("");

	StringToken::DeleteTokenVectorElements(token_vector);
}

void	CPrecedenceDialog::AcceptOperation()
{
	string	e("0 ");

	if (left_button->isChecked())
		e += "LEFT ";
	else if (right_button->isChecked())
		e += "RIGHT ";
	else if (nonassociative_button->isChecked())
		e += "NONASSOCIATIVE ";

	e += (const char *)(entry_line_edit->text().local8Bit());

	CPrecedence		p(e, compiler_data);

	if (p.GetIndex() != -1)
	{
		*precedence_pointer = p;
		accept();
	}
	else
		reject();
}

CRuleDialog::CRuleDialog(QWidget *parent)
	: CDialogFc(parent, 8, 2, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1 = new QGridLayout(this, 1, 2, 0, s, "subgrid1");

	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);
	grid_layout->setRowStretch(3, 0);
	grid_layout->setRowStretch(4, 0);
	grid_layout->setRowStretch(5, 0);
	grid_layout->setRowStretch(6, 1);
	grid_layout->setRowStretch(7, 0);

	grid_layout->setColStretch(0, 1);
	grid_layout->setColStretch(1, 0);

	subgrid_layout1->setRowStretch(0, 1);

	subgrid_layout1->setColStretch(0, 0);
	subgrid_layout1->setColStretch(1, 1);

	QString     size_string("######");

	rule_label				= new CLabelFc("  Rule", this);
	rule_line_edit			= new CLineEditFc(this,
										GrammarDialog::line_edit_string_size);
	symbol_listbox			= new CListBoxFc(this, "symbol list box");
	add_button				= new CPushButtonFc("Add", this, size_string);
	delete_button			= new CPushButtonFc("Delete", this, size_string);
	unit_rule_button		= new CPushButtonFc("Unit", this, size_string);
	ok_button				= new CPushButtonFc("Ok", this, size_string);
	cancel_button			= new CPushButtonFc("Cancel", this, size_string);
	spacer1					= new CSpacerFc(this);

	symbol_listbox->setColumnMode(QListBox::FitToWidth);
	rule_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

	subgrid_layout1->addWidget(rule_label,					0, 0);
	subgrid_layout1->addWidget(rule_line_edit,				0, 1);

	grid_layout->addMultiCellLayout(subgrid_layout1,		0, 0, 0, 1);
	grid_layout->addMultiCellWidget(symbol_listbox,			1, 6, 0, 0);
	grid_layout->addWidget(add_button,						1, 1);
	grid_layout->addWidget(delete_button,					2, 1);
	grid_layout->addWidget(unit_rule_button,				3, 1);
	grid_layout->addWidget(ok_button,						4, 1);
	grid_layout->addWidget(cancel_button,					5, 1);
	grid_layout->addMultiCellWidget(spacer1,				7, 7, 0, 1);

	connect(add_button, SIGNAL(clicked()), SLOT(AddSymbol()));
	connect(delete_button, SIGNAL(clicked()), SLOT(DeleteSymbol()));
	connect(unit_rule_button, SIGNAL(clicked()), SLOT(UnitRuleFlag()));
	connect(ok_button, SIGNAL(clicked()), SLOT(AcceptOperation()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(reject()));
}

void	CRuleDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();
}

void	CRuleDialog::SetCompilerDataPointer(CCompilerData *d)
{
	compiler_data = d;

	symbol_listbox->clear();

	const list<CTerminal *>			&t_list = d->GetTerminalList();
	const list<CErrorToken *>		&e_list = d->GetErrorTokenList();
	const list<CNonTerminal *>		&n_list = d->GetNonTerminalList();

	list<CTerminal *>::const_iterator		t_iterator = t_list.begin();
	list<CErrorToken *>::const_iterator		e_iterator = e_list.begin();
	list<CNonTerminal *>::const_iterator	n_iterator = n_list.begin();

	while (t_iterator != t_list.end())
	{
		if (!(*t_iterator)->GetFilterFlag())
			new QListBoxText(	symbol_listbox,
								(*t_iterator)->GetName().c_str());

		t_iterator++;
	}

	while (e_iterator != e_list.end())
		new QListBoxText(	symbol_listbox,
							(*e_iterator++)->GetName().c_str());

	while (n_iterator != n_list.end())
		new QListBoxText(	symbol_listbox,
							(*n_iterator++)->GetName().c_str());
}

void	CRuleDialog::SetControlValues(const CRule &r)
{
	unit_rule_flag = r.GetUnitRuleEliminationFlag();

	if (unit_rule_flag)
		rule_label->setText("* Rule");
	else
		rule_label->setText("  Rule");

	string		s;
	string		left;
	string		right;

	r.GetLeftString(left);
	r.GetRightString(right);

	if (left.length())
	{
		s  = left;
		s += " -> ";
		s += right;
	}

	rule_line_edit->setText(s.c_str());
}

void	CRuleDialog::setFont(const QFont &f)
{
	QFontInfo       fi(f);

	int     pixel_size = fi.pixelSize();

	int     s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1->setMargin(0);
	subgrid_layout1->setSpacing(s);
	subgrid_layout1->invalidate();

	CDialogFc::setFont(f);
}

void	CRuleDialog::AddSymbol()
{
	QString		s = rule_line_edit->text();

	if (s.length() == s.contains(" "))
		s = "";

	if (!s.isEmpty())
	{
		s += " ";
		s += symbol_listbox->currentText();
	}
	else
	{
		s = symbol_listbox->currentText();
		s += " ->";
	}

	rule_line_edit->setText(s);
}

void	CRuleDialog::DeleteSymbol()
{
	StringToken::TokenStatus				token_status;
	vector<StringToken::CTokenValue *>		token_vector;

	token_status = StringToken::GetCharTokenSequence(
					(const char *)(rule_line_edit->text().local8Bit()),
					string("\t "), token_vector);

	if (token_status == StringToken::NoError && token_vector.size() > 2)
	{
		string											s;
		vector<StringToken::CTokenValue *>::size_type	i;

		for (i=0; i<token_vector.size()-1; i++)
		{
			if (i)
				s += " ";

			s += token_vector[i]->GetTokenString();
		}

		rule_line_edit->setText(s.c_str());
	}
	else
		rule_line_edit->setText("");

	StringToken::DeleteTokenVectorElements(token_vector);
}

void	CRuleDialog::UnitRuleFlag()
{
	GrammarDialog::SetUnitRulePaletteAndFont(this);

	bool				b;
	CUnitRuleDialog		*ud = GrammarDialog::unit_rule_dialog;

	ud->SetTitleAndResize("Set Unit Rule Elimination Flag");
	ud->SetFlagPointer(&b);
	ud->SetControlValues(unit_rule_flag);

	if (ud->exec() == QDialog::Accepted)
	{
		unit_rule_flag = b;

		if (unit_rule_flag)
			rule_label->setText("* Rule");
		else
			rule_label->setText("  Rule");
	}
}

void	CRuleDialog::AcceptOperation()
{
	string	s("0 ");

	if (unit_rule_flag)
		s += "* ";

	s += (const char *)(rule_line_edit->text().local8Bit());

	CRule		r(s, compiler_data);

	if (r.GetIndex() != -1)
	{
		*rule_pointer = r;
		accept();
	}
	else
		reject();
}

CUnitRuleDialog::CUnitRuleDialog(QWidget *parent)
	: CDialogFc(parent, 3, 4, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 1);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 1);
	grid_layout->setColStretch(2, 0);
	grid_layout->setColStretch(3, 0);

	QString     size_string("######");

	spacer1					= new CSpacerFc(this, true);
	unit_rule_check_box		= new CCheckBoxFc(
									"Unit Rule Elimination Flag", this);
	ok_button				= new CPushButtonFc("Ok", this, size_string);
	cancel_button			= new CPushButtonFc("Cancel", this, size_string);
	spacer2					= new CSpacerFc(this);

	grid_layout->addWidget(spacer1,							0, 0);
	grid_layout->addMultiCellWidget(unit_rule_check_box,	0, 0, 1, 3);
	grid_layout->addWidget(ok_button,						1, 2);
	grid_layout->addWidget(cancel_button,					1, 3);
	grid_layout->addMultiCellWidget(spacer2,				2, 2, 0, 3);

	connect(ok_button, SIGNAL(clicked()), SLOT(AcceptOperation()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(reject()));
}

void	CUnitRuleDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();

	setMaximumHeight(height());
	setMaximumWidth(width());
}

void	CUnitRuleDialog::SetControlValues(bool b)
{
	unit_rule_check_box->setChecked(b);
}

void	CUnitRuleDialog::AcceptOperation()
{
	*flag_pointer = unit_rule_check_box->isChecked();

	accept();
}
