/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#ifndef INTERFACECONTROL_H
#define INTERFACECONTROL_H

#include <vector>
#include <map>
#include <set>
#include <algorithm>
#include <string>
#include <iostream>
#include <strstream>
#include <fstream>

#include <qobject.h>
#include <qwidget.h>
#include <qworkspace.h>
#include <qwidgetstack.h>

struct SMainViewControl;
class CInterfaceControl;

#include "functorTemplates.h"
#include "optionData.h"
#include "mainViewMenu.h"
#include "stringToken.h"
#include "filePath.h"
#include "applicationTypes.h"
#include "applicationMainWindow.h"
#include "viewControl.h"
#include "connectionInterface.h"

using namespace ConnectionInterface;

struct SMainViewControl
{
	int							main_view_id;
	CViewNode					*view_tree;
	COptionData::SMainViewData	*view_data;
	bool						main_view_visible_flag;
	bool						widget_stack_parent_flag;

	SMainViewControl(int id, COptionData::SMainViewData *d)
		: main_view_id(id), view_tree(0), view_data(d),
			main_view_visible_flag(false) { }

	~SMainViewControl() { }

	int			GetMainViewId() const { return main_view_id; }
	CViewNode	*GetViewTree() const { return view_tree; }
	void		SetViewData(COptionData::SMainViewData *d) { view_data = d; }
	COptionData::SMainViewData  *GetViewData() const { return view_data; }
	void		SetMainViewVisibleFlag(bool f)	{ main_view_visible_flag = f; }
	bool		GetMainViewVisibleFlag() const
	{ return main_view_visible_flag; }
	void		SetWidgetStackParentFlag(bool f)
	{ widget_stack_parent_flag = f; }
	bool		GetWidgetStackParentFlag() const
	{ return widget_stack_parent_flag; }
	QWidget		*GetRootViewWidget() const;
	QWidget		*GetRootViewParentWidget() const;
	void	CreateViewTree(	QWidget *p,
							CInterfaceControl *ic,
							const string &dfs,
							CPaletteData *dpd);
	void	DestroyViewTree();
	void	ShowViewTree();
	void	HideViewTree();
};

struct destroy_view_tree
	: public unary_function<pair<int, SMainViewControl *>, void>
{
	void	operator()(pair<int, SMainViewControl *> x)
	{
		x.second->DestroyViewTree();
	}
};

struct add_views_to_widget_stack
	: public unary_function<pair<int, SMainViewControl *>, void>
{
	explicit add_views_to_widget_stack(QWidgetStack *ws)
		: widget_stack(ws) { }

	void	operator()(pair<int, SMainViewControl *> x)
	{
		if (x.second->GetViewTree())
			widget_stack->addWidget(x.second->GetRootViewWidget(),
									x.second->main_view_id);
	}

	private:
	QWidgetStack	*widget_stack;
};

struct raise_default_display_main_view
	: public unary_function<pair<int, SMainViewControl *>, void>
{
	explicit raise_default_display_main_view(CInterfaceControl *ic)
		: interface_control(ic) { }

	void	operator()(pair<int, SMainViewControl *> x);

	private:
	CInterfaceControl		*interface_control;
};

struct raise_visible_main_view
	: public unary_function<pair<int, SMainViewControl *>, void>
{
	explicit raise_visible_main_view(CInterfaceControl *ic)
		: interface_control(ic) { }

	void	operator()(pair<int, SMainViewControl *> x);

	private:
	CInterfaceControl		*interface_control;
};

struct find_default_display_main_view
	: public unary_function<pair<int, SMainViewControl *>, bool>
{
	bool	operator()(pair<int, SMainViewControl *> x)
	{
		return x.second->GetViewData()->GetDefaultDisplayFlag();
	}
};

struct find_visible_main_view
	: public unary_function<pair<int, SMainViewControl *>, bool>
{
	bool	operator()(pair<int, SMainViewControl *> x)
	{
		return x.second->main_view_visible_flag;
	}
};

struct capture_main_view_size_and_position
	: public unary_function<pair<int, SMainViewControl *>, void>
{
	explicit capture_main_view_size_and_position(	bool af = false,
													QPoint p = QPoint(-1, -1),
													QSize s = QSize(-1, -1))
		: argument_flag(af), position(p), size(s) { }

	void	operator()(pair<int, SMainViewControl *> x)
	{
		COptionData::SMainViewData	*view_data = x.second->GetViewData();

		if (argument_flag)
		{
			view_data->SetPosition(position);
			view_data->SetSize(size);
		}
		else if (	x.second->GetMainViewVisibleFlag() &&
					x.second->GetRootViewWidget())
		{
			view_data->SetPosition(
				x.second->GetRootViewWidget()->parentWidget()->pos());
			view_data->SetSize(
				x.second->GetRootViewWidget()->parentWidget()->size());
		}
	}

	private:
	bool		argument_flag;
	QPoint		position;
	QSize		size;
};

struct capture_splitter_pane_sizes
	: public unary_function<pair<int, SMainViewControl *>, void>
{
	explicit capture_splitter_pane_sizes(bool rf)
		: reset_flag(rf) { }

	void	operator()(pair<int, SMainViewControl *> x);

	private:
	bool		reset_flag;
};

class CMainWindow;

class CInterfaceControl: public QObject
{
	Q_OBJECT
	public:
	CInterfaceControl(int c_id, int i_id, CMainViewMenu *mvm, CMainWindow *mw);
	virtual ~CInterfaceControl();

	void	CreateWorkspaceViews(QWorkspace *ws);
	void	CreateWidgetStackViews(QWidgetStack *ws);
	void	CreateTitleName(string &tn);
	void	UpdateTitleName();
	void	CreateMainViews(QWidget *w);
	void	ChangeMenuItemToHidden(SMainViewControl *mvc);
	void	ChangeMenuItemToVisible(SMainViewControl *mvc);
	void	CreateMainViewHiddenText(const string &mvn, string &ht);
	void	CreateMainViewVisibleText(const string &mvn, string &vt);
	void	DestroyViews();
	void	CaptureSizeAndPosition(InterfaceType type, bool rf);

	void	SetDataModifiedFlag(bool f);
	void	SetOptionsModifiedFlag(bool f);
	void	SetDataFileName(const string &s)	{ data_file_name = s; }
	void	SetOptionFileName(const string &s)	{ option_file_name = s; }
	bool	GetDataModifiedFlag() const		{ return data_modified_flag; }
	bool	GetOptionsModifiedFlag() const	{ return options_modified_flag; }
	const string	&GetDataFileName() const	{ return data_file_name; }
	const string	&GetOptionFileName() const	{ return option_file_name; }
	COptionData		*GetOptionData() const		{ return option_data; }
	void			SetOptionData(COptionData *d)
	{ delete option_data; option_data = d; }

	CSubView	*FindSubviewTypePointer(const string &s);

	virtual void	InitializeModulePointers() = 0;
	virtual void	SetModulePointer(CSubView *sv, const string &type) = 0;
	virtual void	EnableInterfaceMenu() = 0;
	virtual void	DisableInterfaceMenu() = 0;

	virtual void	PostApplicationData() = 0;
	virtual void    SetApplicationData() = 0;
	virtual bool    SaveData(ofstream &f) = 0;

	virtual CSubView	*CreateSubView(	QWidget *p,
										CViewNode *n,
										CInterfaceControl *ic,
										SMainViewControl *mvc,
										const string &dfs,
										CPaletteData *dpd,
										COptionData::SSubViewData *d) = 0;

	int		RaiseCurrentView();
	void	RaiseMainView(int id);
	int		GetCurrentMainViewId()	const { return current_main_view_id; }
	int		GetInterfaceId()		const { return interface_id; }
	int		GetConfigurationId()	const { return configuration_id; }
	int		GetViewSeparatorId()	const { return view_separator_id; }

	const map<int, SMainViewControl *>	&GetMainViewIdToControlMap() const
	{ return main_view_id_to_control_map; }

	const pair<int, string>				&GetTitleIdAndName() const
	{ return title_id_and_name; }
	const vector< pair<int, string> >	&GetMainViewIdAndNameInfo() const
	{ return main_view_id_and_name_info; }

	bool					data_modified_flag;
	bool					options_modified_flag;
	string					data_file_name;
	string					option_file_name;
	COptionData				*option_data;
	CMainViewMenu			*main_view_menu_manager;
	CMainWindow				*main_window;
	int						interface_id;
	int						configuration_id;
	int						view_separator_id;
	int						current_main_view_id;
	bool					widget_stack_parent_flag;
	QWorkspace				*workspace;
	QWidgetStack			*widget_stack;
	RuntimeMainViewMap		*runtime_map;

	map<int, SMainViewControl *>	main_view_id_to_control_map;
	pair<int, string>				title_id_and_name;
	vector< pair<int, string> >		main_view_id_and_name_info;
};

ostream		&operator<<(ostream &s, const CInterfaceControl &ic);

#endif
