/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#ifndef MENUCONTROL_H
#define MENUCONTROL_H

#include <vector>
#include <string>

#include <qnamespace.h>
#include <qstring.h>
#include <qfont.h>
#include <qpalette.h>
#include <qobject.h>
#include <qmainwindow.h>
#include <qmenubar.h>
#include <qpopupmenu.h>

#include "stringToken.h"

struct MN
{
	enum NodeType
	{
		menu_node_type_bar			= 0,
		menu_node_type_popup		= 1,
		menu_node_type_item			= 2,
		menu_node_type_separator	= 3,
		menu_node_type_end			= 4,

		mn_bar					= 0,
		mn_popup				= 1,
		mn_item					= 2,
		mn_sep					= 3,
		mn_end					= 4
	};

	enum NodeOperation
	{
		menu_node_operation_noop	= 0,
		menu_node_operation_enable	= 1,
		menu_node_operation_disable	= 2,

		mn_noop		= 0,
		mn_enable	= 1,
		mn_disable	= 2
	};
};

template <class T>
class CMenuNode: public MN
{
	public:
	CMenuNode() { }
	virtual ~CMenuNode() { }

	virtual bool		AddMenuNode(const T &d) = 0;
	virtual bool		AddMenuNode(const T &d, vector<int> &iv) = 0;
	virtual QPopupMenu	*GetSubMenu(const T &d) = 0;
	virtual QPopupMenu	*GetSubMenu(const T &d, vector<int> &iv) = 0;
	virtual bool		ModifyMenu(int op, QObject *obj) = 0;
	virtual void		SetMenuFont(QFont &f) = 0;
	virtual void		SetMenuPalette(QPalette &p) = 0;
	void				SetData(const T &d);
	int					GetNodeType() { return node_type; }

	int					node_type;
	string				tree_index;
	int					node_identifier;
	QString				text;
	int					accelerator;
	string				slot;
	vector<int>			operation_list;
	QObject				*slot_object;
};

template <class T>
class CBranchNode: public CMenuNode<T>
{
	public:
	CBranchNode() { }
	virtual ~CBranchNode() { }

	virtual bool		AddMenuNode(const T &d);
	virtual bool		AddMenuNode(const T &d, vector<int> &iv)
	{ return false; }
	virtual QPopupMenu	*GetSubMenu(const T &d);
	virtual QPopupMenu	*GetSubMenu(const T &d, vector<int> &iv);
	virtual bool		ModifyMenu(int op, QObject *obj) { return false; }

	vector<CMenuNode<T> *>	descendants;
};

template <class T>
class CMenuBarNode: public CBranchNode<T>
{
	public:
	CMenuBarNode(QMenuBar *b, const T &d);
	virtual ~CMenuBarNode();

	virtual bool		AddMenuNode(const T &d)
	{ return CBranchNode<T>::AddMenuNode(d); }
	virtual bool		AddMenuNode(const T &d, vector<int> &iv);
	virtual bool		ModifyMenu(int op, QObject *obj);
	virtual void		SetMenuFont(QFont &f);
	virtual void		SetMenuPalette(QPalette &p);
	QMenuBar			*GetMenuBar() { return menu_bar; }

	QWidget			*parent;
	QMenuBar		*menu_bar;
};

template <class T>
class CPopupMenuNode: public CBranchNode<T>
{
	public:
	CPopupMenuNode() { }
	virtual ~CPopupMenuNode() { }

	virtual bool		AddMenuNode(const T &d)
	{ return CBranchNode<T>::AddMenuNode(d); }
	virtual bool		AddMenuNode(const T &d, vector<int> &iv);
	virtual void		SetMenuFont(QFont &f);
	virtual void		SetMenuPalette(QPalette &p);
	QPopupMenu			*GetPopupMenu() { return popup; }

	QPopupMenu		*popup;
};

template <class T>
class CMenuBarPopupMenuNode: public CPopupMenuNode<T>
{
	public:
	CMenuBarPopupMenuNode(QMenuBar *p, const T &d);
	virtual ~CMenuBarPopupMenuNode();

	virtual bool		ModifyMenu(int op, QObject *obj);
	void				SetParentMenuBar(QMenuBar *mb) { parent = mb; }

	QMenuBar		*parent;
};

template <class T>
class CPopupMenuPopupMenuNode: public CPopupMenuNode<T>
{
	public:
	CPopupMenuPopupMenuNode(QPopupMenu *p, const T &d);
	virtual ~CPopupMenuPopupMenuNode();

	virtual bool		ModifyMenu(int op, QObject *obj);
	void				SetParentPopupMenu(QPopupMenu *pm) { parent = pm; }

	QPopupMenu		*parent;
};

template <class T>
class CItemNode: public CMenuNode<T>
{
	public:
	CItemNode() { }
	virtual ~CItemNode() { }

	virtual bool	AddMenuNode(const T &d) { return false; }
	virtual bool	AddMenuNode(const T &d, vector<int> &iv) { return false; }
	virtual QPopupMenu	*GetSubMenu(const T &d) { return 0; }
	virtual QPopupMenu	*GetSubMenu(const T &d, vector<int> &iv) { return 0; }
	virtual void		SetMenuFont(QFont &f) { }
	virtual void		SetMenuPalette(QPalette &p) { }
};

template <class T>
class CMenuBarItemNode: public CItemNode<T>
{
	public:
	CMenuBarItemNode(QMenuBar *p, const T &d);
	virtual ~CMenuBarItemNode();

	virtual bool	ModifyMenu(int op, QObject *obj);

	QMenuBar		*parent;
};

template <class T>
class CPopupMenuItemNode: public CItemNode<T>
{
	public:
	CPopupMenuItemNode(QPopupMenu *p, const T &d);
	virtual ~CPopupMenuItemNode();

	virtual bool	ModifyMenu(int op, QObject *obj);

	QPopupMenu		*parent;
};

template <class T>
class CSeparatorNode: public CMenuNode<T>
{
	public:
	CSeparatorNode() { }
	virtual ~CSeparatorNode() { }

	virtual bool	AddMenuNode(const T &d) { return false; }
	virtual bool	AddMenuNode(const T &d, vector<int> &iv) { return false; }
	virtual QPopupMenu	*GetSubMenu(const T &d) { return 0; }
	virtual QPopupMenu	*GetSubMenu(const T &d, vector<int> &iv) { return 0; }
	virtual bool		ModifyMenu(int op, QObject *obj) { return true; }
	virtual void		SetMenuFont(QFont &f) { }
	virtual void		SetMenuPalette(QPalette &p) { }
};

template <class T>
class CMenuBarSeparatorNode: public CSeparatorNode<T>
{
	public:
	CMenuBarSeparatorNode(QMenuBar *p, const T &d);
	virtual ~CMenuBarSeparatorNode();

	QMenuBar		*parent;
};

template <class T>
class CPopupMenuSeparatorNode: public CSeparatorNode<T>
{
	public:
	CPopupMenuSeparatorNode(QPopupMenu *p, const T &d);
	virtual ~CPopupMenuSeparatorNode();

	QPopupMenu		*parent;
};

template <class T>
bool		CBranchNode<T>::AddMenuNode(const T &d)
{
	StringToken::TokenStatus						token_status;
	vector<StringToken::CTokenValue *>				token_vector;
	vector<StringToken::CTokenValue *>::size_type	i;

	token_status = StringToken::GetIntegerTokenSequence(
						string(d.tree_index), string(", "), token_vector);

	vector<int>		index_vector;

	for (i=0; i<token_vector.size(); i++)
		index_vector.push_back(token_vector[i]->GetIntegerValue());

	StringToken::DeleteTokenVectorElements(token_vector);

	return this->AddMenuNode(d, index_vector);
}

template <class T>
QPopupMenu	*CBranchNode<T>::GetSubMenu(const T &d)
{
	if (d.node_type != menu_node_type_popup)
		return false;

	StringToken::TokenStatus						token_status;
	vector<StringToken::CTokenValue *>				token_vector;
	vector<StringToken::CTokenValue *>::size_type	i;

	token_status = StringToken::GetIntegerTokenSequence(
						string(d.tree_index), string(", "), token_vector);

	vector<int>		index_vector;

	for (i=0; i<token_vector.size(); i++)
		index_vector.push_back(token_vector[i]->GetIntegerValue());

	StringToken::DeleteTokenVectorElements(token_vector);

	return this->GetSubMenu(d, index_vector);
}

template <class T>
QPopupMenu	*CBranchNode<T>::GetSubMenu(const T &d, vector<int> &iv)
{
	if (!iv.size())
		return false;

	int		index = iv[0];

	if (iv.size() == 1)
	{
		if (descendants[index]->GetNodeType() == menu_node_type_popup)
			return ((CPopupMenuNode<T> *)descendants[index])->GetPopupMenu();
		else
			return 0;
	}
	else
	{
		if (index < 0 || index >= descendants.size())
			return false;

		iv.erase(iv.begin());

		return descendants[index]->GetSubMenu(d, iv);
	}
}

template <class T>
bool		CMenuBarNode<T>::AddMenuNode(const T &d, vector<int> &iv)
{
	if (!iv.size())
		return false;

	int		index = iv[0];

	if (iv.size() == 1)
	{
		if (index != descendants.size())
			return false;

		switch (d.node_type)
		{
			case menu_node_type_popup:
			descendants.push_back(new CMenuBarPopupMenuNode<T>(menu_bar, d));
			break;

			case menu_node_type_item:
			descendants.push_back(new CMenuBarItemNode<T>(menu_bar, d));
			break;

			case menu_node_type_separator:
			descendants.push_back(new CMenuBarSeparatorNode<T>(menu_bar, d));
			break;

			default:
			return false;
		}
	}
	else
	{
		if (index < 0 || index >= descendants.size())
			return false;

		iv.erase(iv.begin());

		return descendants[index]->AddMenuNode(d, iv);
	}

	return true;
}

template <class T>
bool		CPopupMenuNode<T>::AddMenuNode(const T &d, vector<int> &iv)
{
	if (!iv.size())
		return false;

	int		index = iv[0];

	if (iv.size() == 1)
	{
		if (index != descendants.size())
			return false;

		switch (d.node_type)
		{
			case menu_node_type_popup:
			descendants.push_back(new CPopupMenuPopupMenuNode<T>(popup, d));
			break;

			case menu_node_type_item:
			descendants.push_back(new CPopupMenuItemNode<T>(popup, d));
			break;

			case menu_node_type_separator:
			descendants.push_back(new CPopupMenuSeparatorNode<T>(popup, d));
			break;

			default:
			return false;
		}
	}
	else
	{
		if (index < 0 || index >= descendants.size())
			return false;

		iv.erase(iv.begin());

		return descendants[index]->AddMenuNode(d, iv);
	}

	return true;
}

template <class T>
void		CMenuNode<T>::SetData(const T &d)
{
	node_type			= d.node_type;
	tree_index			= d.tree_index;
	node_identifier		= d.node_identifier;
	text				= d.text;
	accelerator			= d.accelerator;

	if (!d.slot)
		slot = string("");
	else
		slot = d.slot;

	int		i;
	for (i=0; i<d.GetOperationCount(); i++)
		operation_list.push_back(d.operation_list[i]);

	slot_object = 0;
}

template <class T>
CMenuBarNode<T>::CMenuBarNode(QMenuBar *b, const T &d)
{
	SetData(d);

	parent		= b->parentWidget();
	menu_bar	= b;
}

template <class T>
CMenuBarNode<T>::~CMenuBarNode()
{
	typename vector<CMenuNode<T> *>::size_type		index;

	for (index=0; index<descendants.size(); index++)
		delete descendants[index];
}

template <class T>
CMenuBarPopupMenuNode<T>::CMenuBarPopupMenuNode(QMenuBar *p, const T &d)
{
	SetData(d);

	parent		= p;
	popup		= new QPopupMenu(p);

	if (parent)
		parent->insertItem(text, popup, node_identifier);
}

template <class T>
CMenuBarPopupMenuNode<T>::~CMenuBarPopupMenuNode()
{
	typename vector<CMenuNode<T> *>::size_type		index;

	for (index=0; index<descendants.size(); index++)
		delete descendants[index];

	delete popup;
}

template <class T>
CPopupMenuPopupMenuNode<T>::CPopupMenuPopupMenuNode(QPopupMenu *p, const T &d)
{
	SetData(d);

	parent		= p;
	popup		= new QPopupMenu(p);

	if (parent)
		parent->insertItem(text, popup, node_identifier);
}

template <class T>
CPopupMenuPopupMenuNode<T>::~CPopupMenuPopupMenuNode()
{
	typename vector<CMenuNode<T> *>::size_type		index;

	for (index=0; index<descendants.size(); index++)
		delete descendants[index];

	delete popup;
}

template <class T>
CMenuBarItemNode<T>::CMenuBarItemNode(QMenuBar *p, const T &d)
{
	SetData(d);

	parent		= p;

	parent->insertItem(text, node_identifier);
	parent->setAccel(accelerator, node_identifier);
}

template <class T>
CMenuBarItemNode<T>::~CMenuBarItemNode()
{
}

template <class T>
CPopupMenuItemNode<T>::CPopupMenuItemNode(QPopupMenu *p, const T &d)
{
	SetData(d);

	parent		= p;

	parent->insertItem(text, node_identifier);
	parent->setAccel(accelerator, node_identifier);
}

template <class T>
CPopupMenuItemNode<T>::~CPopupMenuItemNode()
{
}

template <class T>
CMenuBarSeparatorNode<T>::CMenuBarSeparatorNode(QMenuBar *p, const T &d)
{
	SetData(d);

	parent		= p;

	parent->insertSeparator();
}

template <class T>
CMenuBarSeparatorNode<T>::~CMenuBarSeparatorNode()
{
}

template <class T>
CPopupMenuSeparatorNode<T>::CPopupMenuSeparatorNode(QPopupMenu *p, const T &d)
{
	SetData(d);

	parent		= p;

	parent->insertSeparator();
}

template <class T>
CPopupMenuSeparatorNode<T>::~CPopupMenuSeparatorNode()
{
}

template <class T>
bool		CMenuBarNode<T>::ModifyMenu(int op, QObject *obj)
{
	bool	rvalue = true;

	switch (operation_list[op])
	{
		case menu_node_operation_noop:
		break;

		case menu_node_operation_enable:
		if (slot.length())
		{
			if (slot_object)
				rvalue &= QObject::disconnect(	menu_bar,
												SIGNAL(activated(int)),
												slot_object, slot.c_str());
			slot_object = obj;

			if (slot_object)
				rvalue &= QObject::connect(	menu_bar,
											SIGNAL(activated(int)),
											slot_object, slot.c_str());
		}
		break;

		case menu_node_operation_disable:
		if (slot.length())
		{
			if (slot_object)
				rvalue &= QObject::disconnect(	menu_bar,
												SIGNAL(activated(int)),
												slot_object, slot.c_str());
			slot_object = 0;
		}
		break;
	}

	typename vector<CMenuNode<T> *>::size_type		index;

	for (index=0; index<descendants.size(); index++)
		rvalue &= descendants[index]->ModifyMenu(op, obj);

	return rvalue;
}

template <class T>
bool		CMenuBarPopupMenuNode<T>::ModifyMenu(int op, QObject *obj)
{
	bool	rvalue = true;

	switch (operation_list[op])
	{
		case menu_node_operation_noop:
		break;

		case menu_node_operation_enable:
		if (slot.length())
		{
			if (slot_object)
				rvalue &= QObject::disconnect(	popup,
												SIGNAL(activated(int)),
												slot_object, slot.c_str());
			slot_object = obj;

			if (slot_object)
				rvalue &= QObject::connect(	popup,
											SIGNAL(activated(int)),
											slot_object, slot.c_str());
		}

		parent->setItemEnabled(node_identifier, true);
		break;

		case menu_node_operation_disable:
		parent->setItemEnabled(node_identifier, false);

		if (slot.length())
		{
			if (slot_object)
				rvalue &= QObject::disconnect(	popup,
												SIGNAL(activated(int)),
												slot_object, slot.c_str());
			slot_object = 0;
		}
		break;
	}

	typename vector<CMenuNode<T> *>::size_type		index;

	for (index=0; index<descendants.size(); index++)
		rvalue &= descendants[index]->ModifyMenu(op, obj);

	return rvalue;
}

template <class T>
bool		CPopupMenuPopupMenuNode<T>::ModifyMenu(int op, QObject *obj)
{
	bool	rvalue = true;

	switch (operation_list[op])
	{
		case menu_node_operation_noop:
		break;

		case menu_node_operation_enable:
		if (slot.length())
		{
			if (slot_object)
				rvalue &= QObject::disconnect(	popup,
												SIGNAL(activated(int)),
												slot_object, slot.c_str());
			slot_object = obj;

			if (slot_object)
				rvalue &= QObject::connect(	popup,
											SIGNAL(activated(int)),
											slot_object, slot.c_str());
		}

		parent->setItemEnabled(node_identifier, true);
		break;

		case menu_node_operation_disable:
		parent->setItemEnabled(node_identifier, false);

		if (slot.length())
		{
			if (slot_object)
				rvalue &= QObject::disconnect(	popup,
												SIGNAL(activated(int)),
												slot_object, slot.c_str());
			slot_object = 0;
		}
		break;
	}

	typename vector<CMenuNode<T> *>::size_type		index;

	for (index=0; index<descendants.size(); index++)
		rvalue &= descendants[index]->ModifyMenu(op, obj);

	return rvalue;
}

template <class T>
bool		CMenuBarItemNode<T>::ModifyMenu(int op, QObject *obj)
{
	bool	rvalue = true;

	switch (operation_list[op])
	{
		case menu_node_operation_noop:
		break;

		case menu_node_operation_enable:
		if (slot.length())
		{
			if (slot_object)
				rvalue &= parent->disconnectItem(	node_identifier,
													slot_object, slot.c_str());
			slot_object = obj;

			if (slot_object)
				rvalue &= parent->connectItem(	node_identifier,
												slot_object, slot.c_str());
		}

		parent->setItemEnabled(node_identifier, true);
		break;

		case menu_node_operation_disable:
		parent->setItemEnabled(node_identifier, false);

		if (slot.length())
		{
			if (slot_object)
				rvalue &= parent->disconnectItem(	node_identifier,
													slot_object, slot.c_str());
			slot_object = 0;
		}
		break;
	}

	return rvalue;
}

template <class T>
bool		CPopupMenuItemNode<T>::ModifyMenu(int op, QObject *obj)
{
	bool	rvalue = true;

	switch (operation_list[op])
	{
		case menu_node_operation_noop:
		break;

		case menu_node_operation_enable:
		if (slot.length())
		{
			if (slot_object)
				rvalue &= parent->disconnectItem(	node_identifier,
													slot_object, slot.c_str());
			slot_object = obj;

			if (slot_object)
				rvalue &= parent->connectItem(	node_identifier,
												slot_object, slot.c_str());
		}

		parent->setItemEnabled(node_identifier, true);
		break;

		case menu_node_operation_disable:
		parent->setItemEnabled(node_identifier, false);

		if (slot.length())
		{
			if (slot_object)
				rvalue &= parent->disconnectItem(	node_identifier,
													slot_object, slot.c_str());
			slot_object = 0;
		}
		break;
	}

	return rvalue;
}

template <class T>
void		CMenuBarNode<T>::SetMenuFont(QFont &f)
{
	menu_bar->setFont(f);

	typename vector<CMenuNode<T> *>::size_type		index;

	for (index=0; index<descendants.size(); index++)
		descendants[index]->SetMenuFont(f);
}

template <class T>
void		CPopupMenuNode<T>::SetMenuFont(QFont &f)
{
	popup->setFont(f);

	typename vector<CMenuNode<T> *>::size_type		index;

	for (index=0; index<descendants.size(); index++)
		descendants[index]->SetMenuFont(f);
}

template <class T>
void		CMenuBarNode<T>::SetMenuPalette(QPalette &p)
{
	menu_bar->setPalette(p);

	typename vector<CMenuNode<T> *>::size_type		index;

	for (index=0; index<descendants.size(); index++)
		descendants[index]->SetMenuPalette(p);
}

template <class T>
void		CPopupMenuNode<T>::SetMenuPalette(QPalette &p)
{
	popup->setPalette(p);

	typename vector<CMenuNode<T> *>::size_type		index;

	for (index=0; index<descendants.size(); index++)
		descendants[index]->SetMenuPalette(p);
}

#endif
