/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "nonterminalView.h"

void	CNonTerminalListView::InitializePalette(const QPalette &p)
{
	active_highlight			= p.color(	QPalette::Active,
											QColorGroup::Highlight);
	active_highlighted_text		= p.color(	QPalette::Active,
											QColorGroup::HighlightedText);
	inactive_highlight			= p.color(	QPalette::Inactive,
											QColorGroup::Highlight);
	inactive_highlighted_text	= p.color(	QPalette::Inactive,
											QColorGroup::HighlightedText);

	QPalette	focus_palette = p;

	focus_palette.setColor(	QPalette::Active,
							QColorGroup::Highlight,
							inactive_highlight);

	focus_palette.setColor(	QPalette::Active,
							QColorGroup::HighlightedText,
							inactive_highlighted_text);

	setPalette(focus_palette);
}

void	CNonTerminalListView::focusInEvent(QFocusEvent *e)
{
	QPalette	p = palette();

	p.setColor(	QPalette::Active,
				QColorGroup::Highlight,
				active_highlight);

	p.setColor(	QPalette::Active,
				QColorGroup::HighlightedText,
				active_highlighted_text);

	setPalette(p);

	QListView::focusInEvent(e);
}

void	CNonTerminalListView::focusOutEvent(QFocusEvent *e)
{
	QPalette	p = palette();

	p.setColor(	QPalette::Active,
				QColorGroup::Highlight,
				inactive_highlight);

	p.setColor(	QPalette::Active,
				QColorGroup::HighlightedText,
				inactive_highlighted_text);

	setPalette(p);

	QListView::focusOutEvent(e);
}

void	CNonTerminalListView::mouseDoubleClickEvent(QMouseEvent *e)
{
	nonterminal_vbox->EditNonTerminal();

	e->accept();
}

void	CNonTerminalListView::keyPressEvent(QKeyEvent *e)
{
	switch (e->key())
	{
		case Qt::Key_Insert:
		switch (e->state())
		{
			case Qt::NoButton:
			case Qt::Keypad:
			e->accept();
			nonterminal_vbox->InsertAfterNonTerminal();
			return;

			case Qt::ShiftButton:
			e->accept();
			nonterminal_vbox->InsertBeforeNonTerminal();
			return;
		}
		break; 

		case Qt::Key_0:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			nonterminal_vbox->InsertBeforeNonTerminal();
			return;
		}
		break; 

		case Qt::Key_Delete:
		if (e->state() == Qt::ShiftButton)
		{
			e->accept();
			nonterminal_vbox->DeleteNonTerminal();
			return;
		}
		break; 

		case Qt::Key_Period:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			nonterminal_vbox->DeleteNonTerminal();
			return;
		}
		break; 

		case Qt::Key_Return:
		if (e->state() == Qt::NoButton)
		{
			e->accept();
			nonterminal_vbox->EditNonTerminal();
			return;
		}
		break; 

		case Qt::Key_Enter:
		if (e->state() == Qt::Keypad)
		{
			e->accept();
			nonterminal_vbox->EditNonTerminal();
			return;
		}
		break; 

		case Qt::Key_Up:
		if (e->state() == Qt::ShiftButton)
		{
			e->accept();
			nonterminal_vbox->MoveNonTerminalUp();
			return;
		}
		break;

		case Qt::Key_8:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			nonterminal_vbox->MoveNonTerminalUp();
			return;
		}
		break;

		case Qt::Key_Down:
		if (e->state() == Qt::ShiftButton)
		{
			e->accept();
			nonterminal_vbox->MoveNonTerminalDown();
			return;
		}
		break;

		case Qt::Key_2:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			nonterminal_vbox->MoveNonTerminalDown();
			return;
		}
		break;
	}

	e->ignore();
	QListView::keyPressEvent(e);
}

CNonTerminalVBoxWidget::CNonTerminalVBoxWidget(	QWidget *parent,
												CCompilerInterface *ci,
												SMainViewControl *mvc,
												COptionData::SSubViewData *d,
												CNonTerminalView *nv)
	: CVBoxWidget(parent, ci, mvc), compiler_interface(ci),
		subview_data(d), nonterminal_view(nv),
		module_ptr_map(nv->GetModulePtrMap())
{
	label = new CLabelFc("nonterminal", this, true);

	label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	list_view = new CNonTerminalListView(this);

	list_view->header()->setMovingEnabled(false);
	list_view->setSorting(-1);
	list_view->addColumn("index");
	list_view->addColumn("symbol");
	list_view->addColumn("name");
	list_view->addColumn("start");

	setSpacing(0);
	setStretchFactor(label, 0);
	setStretchFactor(list_view, 1);

	compiler_data = compiler_interface->GetCompilerData();

	UpdateListView();

	list_view->setSelected(list_view->firstChild(), true);
}

CNonTerminalVBoxWidget::~CNonTerminalVBoxWidget()
{
}

void	CNonTerminalVBoxWidget::UpdateListView()
{
	const list<CNonTerminal *>	&n_list = compiler_data->GetNonTerminalList();

	int		n_count = n_list.size();
	int		l_count = list_view->childCount();
	int		i;

	if (n_count < l_count)
	{
		for (i=0; i<l_count-n_count; i++)
			delete list_view->firstChild();
	}
	else if (n_count > l_count)
	{
		for (i=0; i<n_count-l_count; i++)
			new QListViewItem(list_view);
	}

	list<CNonTerminal *>::const_iterator	n_iterator	= n_list.begin();

	QListViewItemIterator	l_iterator(list_view);

	while (n_iterator != n_list.end())
	{
		const CNonTerminal		*n = *n_iterator;

		QListViewItem	*item = l_iterator.current();

		string		s_index;
		string		s_symbol_index;

		n->GetIndexString(s_index);
		n->GetSymbolIndexString(s_symbol_index);

		item->setText(0, s_index.c_str());
		item->setText(1, s_symbol_index.c_str());
		item->setText(2, n->GetName().c_str());

		if (n->GetStartSymbolFlag())
			item->setText(3, "*");
		else
			item->setText(3, "");

		n_iterator++;
		l_iterator++;
	}

	vector<CSubView *>		&v1 = module_ptr_map["IMPLICIT_VIEW"];
	vector<CSubView *>		&v2 = module_ptr_map["RULE_VIEW"];

	if (v1.size())
	{
		CImplicitView		*iv = dynamic_cast<CImplicitView *>(v1[0]);

		iv->UpdateListView();
	}


	if (v2.size())
	{
		CRuleView			*rv = dynamic_cast<CRuleView *>(v2[0]);

		rv->UpdateListView();
	}
}

void	CNonTerminalVBoxWidget::SetSelectedItem(int i)
{
	int		l_count = list_view->childCount();

	if (i < 0 || !l_count)
		return;

	if (i >= l_count)
		i = l_count - 1;

	QListViewItemIterator	l_iterator(list_view);
	int						count;

	for (count=0; count<i; count++)
		l_iterator++;

	list_view->setSelected(l_iterator.current(), true);
	list_view->ensureItemVisible(l_iterator.current());
}

void	CNonTerminalVBoxWidget::InitializeModulePointers()
{
}

void	CNonTerminalVBoxWidget::InsertBeforeNonTerminal()
{
	GrammarDialog::SetNonTerminalPaletteAndFont(this);

	CNonTerminal			nonterminal(0);
	CNonTerminalDialog		*nd = GrammarDialog::nonterminal_dialog;

	nd->SetTitleAndResize("Insert Before NonTerminal");
	nd->SetNonTerminalPointer(&nonterminal);
	nd->SetControlValues(nonterminal);

	if (nd->exec() == QDialog::Accepted)
	{
		QListViewItem	*current = list_view->currentItem();
		int				selected = 0;

		if (current)
			selected = current->text(0).toInt();

		compiler_interface->StoreUndoData();

		InsertBeforeNonTerminal(selected, nonterminal);

		SetSelectedItem(selected);

		compiler_interface->SetDataModifiedFlag(true);
	}
}

void	CNonTerminalVBoxWidget::InsertAfterNonTerminal()
{
	GrammarDialog::SetNonTerminalPaletteAndFont(this);

	CNonTerminal			nonterminal(0);
	CNonTerminalDialog		*nd = GrammarDialog::nonterminal_dialog;

	nd->SetTitleAndResize("Insert After NonTerminal");
	nd->SetNonTerminalPointer(&nonterminal);
	nd->SetControlValues(nonterminal);

	if (nd->exec() == QDialog::Accepted)
	{
		QListViewItem	*current = list_view->currentItem();
		int				selected = 0;

		if (current)
			selected = current->text(0).toInt();

		compiler_interface->StoreUndoData();

		InsertAfterNonTerminal(selected, nonterminal);

		SetSelectedItem(selected + 1);

		compiler_interface->SetDataModifiedFlag(true);
	}
}

void	CNonTerminalVBoxWidget::EditNonTerminal()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	selected	= current->text(0).toInt();

		GrammarDialog::SetNonTerminalPaletteAndFont(this);

		CNonTerminal		*ni = compiler_data->FindNonTerminalIndex(selected);
		CNonTerminal		nonterminal(0);
		CNonTerminalDialog	*nd = GrammarDialog::nonterminal_dialog;

		nd->SetTitleAndResize("Edit NonTerminal");
		nd->SetNonTerminalPointer(&nonterminal);

		if (ni)
			nd->SetControlValues(*ni);
		else
			nd->SetControlValues(nonterminal);

		if (nd->exec() == QDialog::Accepted)
		{
			compiler_interface->StoreUndoData();

			UpdateNonTerminal(selected, nonterminal);

			SetSelectedItem(selected);

			compiler_interface->SetDataModifiedFlag(true);
		}
	}
}

void	CNonTerminalVBoxWidget::DeleteNonTerminal()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	selected	= current->text(0).toInt();

		compiler_interface->StoreUndoData();

		DeleteNonTerminal(selected);

		SetSelectedItem(selected);

		compiler_interface->SetDataModifiedFlag(true);
	}
}

void	CNonTerminalVBoxWidget::MoveNonTerminalUp()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	selected	= current->text(0).toInt();

		if (selected > 0)
		{
			compiler_interface->StoreUndoData();

			MoveNonTerminalUp(selected);

			SetSelectedItem(selected - 1);

			compiler_interface->SetDataModifiedFlag(true);
		}
	}
}

void	CNonTerminalVBoxWidget::MoveNonTerminalDown()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	l_count		= list_view->childCount();
		int	selected	= current->text(0).toInt();

		if (selected < l_count - 1)
		{
			compiler_interface->StoreUndoData();

			MoveNonTerminalDown(selected);

			SetSelectedItem(selected + 1);

			compiler_interface->SetDataModifiedFlag(true);
		}
	}
}

void	CNonTerminalVBoxWidget::InsertBeforeNonTerminal(int i,
														const CNonTerminal &n)
{
	compiler_data->InsertBeforeNonTerminal(i, n);

	UpdateListView();
}

void	CNonTerminalVBoxWidget::InsertAfterNonTerminal(	int i,
														const CNonTerminal &n)
{
	compiler_data->InsertAfterNonTerminal(i, n);

	UpdateListView();
}

void	CNonTerminalVBoxWidget::UpdateNonTerminal(int i, const CNonTerminal &n)
{
	compiler_data->UpdateNonTerminal(i, n);

	UpdateListView();
}

void	CNonTerminalVBoxWidget::DeleteNonTerminal(int i)
{
	compiler_data->DeleteNonTerminal(i);

	UpdateListView();
}

void	CNonTerminalVBoxWidget::MoveNonTerminalUp(int i)
{
	compiler_data->MoveNonTerminalUp(i);

	UpdateListView();
}

void	CNonTerminalVBoxWidget::MoveNonTerminalDown(int i)
{
	compiler_data->MoveNonTerminalDown(i);

	UpdateListView();
}

void	CNonTerminalVBoxWidget::RestoreUndoData()
{
	compiler_data = compiler_interface->GetCompilerData();
}

CNonTerminalView::CNonTerminalView(	QWidget *p,
									CViewNode *n,
									CInterfaceControl *ic,
									SMainViewControl *mvc,
									const string &dfs,
									CPaletteData *dpd,
									COptionData::SSubViewData *d)
	: CSubView(p, n, ic, mvc, dfs, dpd, d)
{
	frame = new CNonTerminalVBoxWidget(
		p, dynamic_cast<CCompilerInterface *>(ic), mvc, d, this);

	QFont		view_node_font;
	QPalette	view_node_palette;

	if (SetupViewNodeFont(view_node_font))
		frame->setFont(view_node_font);
	else
		frame->setFont(frame->font());

	if (SetupViewNodePalette(view_node_palette))
	{
		frame->setPalette(view_node_palette);
		frame->InitializePalette(view_node_palette);
	}
	else
	{
		frame->setPalette(frame->palette());
		frame->InitializePalette(frame->palette());
	}
}

CNonTerminalView::~CNonTerminalView()
{
	delete frame;
}

void	CNonTerminalView::SetHighlightPalette(CPaletteData *pd)
{
	restore_palette = frame->palette();

	QPalette	highlight_palette;

	pd->SetupPalette(highlight_palette);

	frame->setPalette(highlight_palette);
}

void	CNonTerminalView::RestorePalette()
{
	frame->setPalette(restore_palette);
}

void	CNonTerminalView::InitializeModulePointers()
{
	frame->InitializeModulePointers();
}

void	CNonTerminalView::InsertAfterNonTerminal()
{
	frame->InsertAfterNonTerminal();
}

void	CNonTerminalView::InsertBeforeNonTerminal()
{
	frame->InsertBeforeNonTerminal();
}

void	CNonTerminalView::EditNonTerminal()
{
	frame->EditNonTerminal();
}

void	CNonTerminalView::DeleteNonTerminal()
{
	frame->DeleteNonTerminal();
}

void	CNonTerminalView::MoveNonTerminalUp()
{
	frame->MoveNonTerminalUp();
}

void	CNonTerminalView::MoveNonTerminalDown()
{
	frame->MoveNonTerminalDown();
}

void	CNonTerminalView::UpdateListView()
{
	frame->UpdateListView();
}

void	CNonTerminalView::RestoreUndoData()
{
	frame->RestoreUndoData();
}
