/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#ifndef OPTIONDATA_H
#define OPTIONDATA_H

#include <string.h>
#include <iostream>
#include <iomanip>
#include <strstream>
#include <string>
#include <vector>
#include <list>
#include <map>
#include <algorithm>

#include <qnamespace.h>
#include <qregexp.h>
#include <qpixmap.h>
#include <qbitmap.h>
#include <qcolor.h>
#include <qbrush.h>
#include <qpalette.h>
#include <qfont.h>
#include <qfontdatabase.h>
#include <qsize.h>
#include <qpoint.h>
#include <qclipboard.h>
#include <qapplication.h>

#include "roleData.h"
#include "functorTemplates.h"
#include "stringToken.h"
#include "optionLexicalDriver.h"
#include "optionParserDriver.h"

namespace FontSpecifier
{
	bool					SetFontSpecifier(QFont &f, const string &s);
	void					NormalizeString(string &s1, const string &s2);
};

class CXpmImage
{
	public:
	CXpmImage() : pixmap(0) { }
	~CXpmImage() { delete pixmap; }
	CXpmImage(const CXpmImage &xi) : xpm_data(xi.xpm_data), pixmap(0) { }

	void	PushBackXpmCharData(const char *c)
	{ xpm_data.push_back(string(c)); }
	const list<string>	&GetXpmData() const { return xpm_data; }
	QPixmap	*CreatePixmap();
	bool	ValidateXpmData(char *data[], int count);

	list<string>	xpm_data;
	QPixmap			*pixmap;
};

ostream	&operator<<(ostream &os, const CXpmImage &p);

ostream		&OutputStringData(ostream &os, const string &s);

class CPaletteData
{
	public:
	enum ColorSpecifierType
	{
		RgbSpecifier,
		HsvSpecifier,
		ColorNameSpecifier
	};

	struct SColorSpecifier
	{
		virtual ColorSpecifierType	GetSpecifierType() const = 0;
		virtual ~SColorSpecifier() { }
		virtual void	SetupColor(QColor &c) const = 0;
		virtual ostream	&OutputData(ostream &os) const = 0;
		virtual SColorSpecifier *clone() const = 0;
	};

	struct SRgbSpecifier: public SColorSpecifier
	{
		int		red;
		int		green;
		int		blue;

		virtual ColorSpecifierType	GetSpecifierType() const
		{ return RgbSpecifier; }

		SRgbSpecifier(int r, int g, int b) : red(r), green(g), blue(b) { }
		virtual ~SRgbSpecifier() { }
		virtual void	SetupColor(QColor &c) const
		{ c.setRgb(red, green, blue); }
		virtual ostream	&OutputData(ostream &os) const
		{
			return os	<< "RGB "
						<< setw(4) << red
						<< setw(4) << green
						<< setw(4) << blue;
		}
		virtual SRgbSpecifier *clone() const
		{ return new SRgbSpecifier(*this); }
	};

	struct SHsvSpecifier: public SColorSpecifier
	{
		int		hue;
		int		saturation;
		int		value;

		virtual ColorSpecifierType	GetSpecifierType() const
		{ return HsvSpecifier; }

		SHsvSpecifier(int h, int s, int v)
			: hue(h), saturation(s), value(v) { }
		virtual ~SHsvSpecifier() { }
		virtual void	SetupColor(QColor &c) const
		{ c.setHsv(hue, saturation, value); }
		virtual ostream	&OutputData(ostream &os) const
		{
			return os	<< "HSV "
						<< setw(4) << hue
						<< setw(4) << saturation
						<< setw(4) << value;
		}
		virtual SHsvSpecifier *clone() const
		{ return new SHsvSpecifier(*this); }
	};

	struct SColorNameSpecifier: public SColorSpecifier
	{
		string	color_name;

		virtual ColorSpecifierType	GetSpecifierType() const
		{ return ColorNameSpecifier; }

		SColorNameSpecifier(const string &cn) : color_name(cn) { }
		virtual ~SColorNameSpecifier() { }
		virtual void	SetupColor(QColor &c) const
		{ c.setNamedColor(color_name.c_str()); }
		virtual ostream	&OutputData(ostream &os) const
		{
			os << "COLOR_NAME  ";

			return OutputStringData(os, color_name);
		}
		virtual SColorNameSpecifier *clone() const
		{ return new SColorNameSpecifier(*this); }
	};

	enum BrushSpecifierType
	{
		StyleBrushSpecifier,
		XpmImageBrushSpecifier
	};

	struct SBrushSpecifier
	{
		SColorSpecifier		*color_specifier;

		virtual BrushSpecifierType	GetSpecifierType() const = 0;
		virtual const string 		&GetBrushStyle() const = 0;
		SBrushSpecifier(SColorSpecifier *cs) : color_specifier(cs) { }
		virtual ~SBrushSpecifier() { delete color_specifier; }
		virtual void	SetupBrush(QBrush &b) const = 0;
		virtual ostream	&OutputData(ostream &os) const = 0;
		virtual SBrushSpecifier *clone() const = 0;
	};

	struct SStyleBrushSpecifier: public SBrushSpecifier
	{
		Qt::BrushStyle	style;

		virtual BrushSpecifierType	GetSpecifierType() const
		{ return StyleBrushSpecifier; }
		virtual const string 		&GetBrushStyle() const
		{ return brush_style_to_string_map[style]; }

		SStyleBrushSpecifier(SColorSpecifier *cs, Qt::BrushStyle bs)
			: SBrushSpecifier(cs), style(bs) { }

		SStyleBrushSpecifier(SColorSpecifier *cs, const string &s)
			: SBrushSpecifier(cs), style(string_to_brush_style_map[s]) { }
		virtual ~SStyleBrushSpecifier() { }
		virtual void	SetupBrush(QBrush &b) const
		{
			QColor	color;

			color_specifier->SetupColor(color);

			b.setColor(color);
			b.setStyle(style);
		}
		virtual ostream	&OutputData(ostream &os) const;
		virtual SStyleBrushSpecifier *clone() const
		{
			return new SStyleBrushSpecifier(color_specifier->clone(), style);
		}
	};

	struct SXpmImageBrushSpecifier: public SBrushSpecifier
	{
		CXpmImage	*xpm_image;

		virtual BrushSpecifierType	GetSpecifierType() const
		{ return XpmImageBrushSpecifier; }
		virtual const string 		&GetBrushStyle() const
		{ return brush_style_to_string_map[Qt::CustomPattern]; }

		SXpmImageBrushSpecifier(SColorSpecifier *cs, CXpmImage *xi)
			: SBrushSpecifier(cs), xpm_image(xi) { }

		virtual ~SXpmImageBrushSpecifier() { delete xpm_image; }
		virtual void	SetupBrush(QBrush &b) const
		{
			QColor	color;

			color_specifier->SetupColor(color);

			b.setColor(color);
//			b.setPixmap(*xpm_image->CreatePixmap());

			QBitmap		bitmap;

			bitmap = *xpm_image->CreatePixmap();

			b.setPixmap(bitmap);
		}
		virtual ostream	&OutputData(ostream &os) const;
		virtual SXpmImageBrushSpecifier *clone() const
		{
			return new SXpmImageBrushSpecifier(	color_specifier->clone(),
												new CXpmImage(*xpm_image));
		}
	};

	struct SGroupRoleEntry
	{
		QPalette::ColorGroup	group;
		QColorGroup::ColorRole	role;

		SGroupRoleEntry(QPalette::ColorGroup g, QColorGroup::ColorRole r)
			: group(g), role(r) { }

		SGroupRoleEntry(const string &g, const string &r)
			: 	group(string_to_color_group_map[g]),
				role(string_to_color_role_map[r]) { }

		bool	operator<(const SGroupRoleEntry &gre) const
		{
			if (group < gre.group)
				return true;
			else if (group == gre.group)
			{
				if (role < gre.role)
					return true;
				else
					return false;
			}
			else
				return false;
		}

		ostream	&OutputData(ostream &os) const;
	};

	struct SGroupRoleSpecifier
	{
		virtual ~SGroupRoleSpecifier() { }
		virtual void	SetupGroupRoleEntry(QPalette &p,
											SGroupRoleEntry &gre) const = 0;
		virtual ostream	&OutputData(ostream &os) const = 0;
		virtual SGroupRoleSpecifier *clone() const = 0;
	};

	struct SColorGroupRoleSpecifier: public SGroupRoleSpecifier
	{
		SColorSpecifier		*color_specifier;

		SColorGroupRoleSpecifier(SColorSpecifier *cs) : color_specifier(cs) { }
		virtual ~SColorGroupRoleSpecifier() { delete color_specifier; }
		virtual void	SetupGroupRoleEntry(QPalette &p,
											SGroupRoleEntry &gre) const
		{
			QColor		color;

			color_specifier->SetupColor(color);

			p.setColor(gre.group, gre.role, color);
		}
		virtual ostream	&OutputData(ostream &os) const;
		virtual SColorGroupRoleSpecifier *clone() const
		{
			return new SColorGroupRoleSpecifier(color_specifier->clone());
		}
	};

	struct SBrushGroupRoleSpecifier: public SGroupRoleSpecifier
	{
		SBrushSpecifier		*brush_specifier;

		SBrushGroupRoleSpecifier(SBrushSpecifier *bs) : brush_specifier(bs) { }
		virtual ~SBrushGroupRoleSpecifier() { delete brush_specifier; }
		virtual void	SetupGroupRoleEntry(QPalette &p,
											SGroupRoleEntry &gre) const
		{
			QBrush		brush;

			brush_specifier->SetupBrush(brush);

			p.setBrush(gre.group, gre.role, brush);
		}
		virtual ostream	&OutputData(ostream &os) const;
		virtual SBrushGroupRoleSpecifier *clone() const
		{
			return new SBrushGroupRoleSpecifier(brush_specifier->clone());
		}
	};

	CPaletteData() : precalculate_button_color(0),
		precalculate_background_color(0), group_role_specifiers(0) { }
	virtual ~CPaletteData();
	CPaletteData(const CPaletteData &pd);

	static void		SetupGroupRoleMaps();

	void	SetPrecalculateButtonColor(SColorSpecifier *cs)
	{
		delete precalculate_button_color;
		precalculate_button_color = cs;
	}
	void	SetPrecalculateBackgroundColor(SColorSpecifier *cs)
	{
		delete precalculate_background_color;
		precalculate_background_color = cs;
	}
	void	SetGroupRoleSpecifierMap(
						map<SGroupRoleEntry, SGroupRoleSpecifier *> *m);

	SColorSpecifier		*GetPrecalculateButtonColor()
	{ return precalculate_button_color; }
	SColorSpecifier		*GetPrecalculateBackgroundColor()
	{ return precalculate_background_color; }
	map<SGroupRoleEntry, SGroupRoleSpecifier *>		*GetGroupRoleSpecifierMap()
	{ return group_role_specifiers; }

	void			SetGroupRoleSpecifier(	const SGroupRoleEntry &gre,
											SGroupRoleSpecifier *grs);
	void			DeleteGroupRoleSpecifier(const SGroupRoleEntry &gre);

	void	SetupPalette(QPalette &p);

	SColorSpecifier								*precalculate_button_color;
	SColorSpecifier								*precalculate_background_color;
	map<SGroupRoleEntry, SGroupRoleSpecifier *>	*group_role_specifiers;

	static map<string, QPalette::ColorGroup>	string_to_color_group_map;
	static map<string, QColorGroup::ColorRole>	string_to_color_role_map;
	static map<string, Qt::BrushStyle>			string_to_brush_style_map;
	static map<QPalette::ColorGroup, string>	color_group_to_string_map;
	static map<QColorGroup::ColorRole, string>	color_role_to_string_map;
	static map<Qt::BrushStyle, string>			brush_style_to_string_map;

	struct set_group_role_entries
	: public unary_function<pair<SGroupRoleEntry, SGroupRoleSpecifier *>, void>
	{
		explicit set_group_role_entries(QPalette &p) : palette(p) { }

		void	operator()(pair<CPaletteData::SGroupRoleEntry,
								CPaletteData::SGroupRoleSpecifier *> x)
		{
			x.second->SetupGroupRoleEntry(palette, x.first);
		}

		private:
		QPalette	&palette;
	};
};

typedef map<CPaletteData::SGroupRoleEntry,
			CPaletteData::SGroupRoleSpecifier *>
										GroupRoleToSpecifierMap;

typedef GroupRoleToSpecifierMap::iterator	GroupRoleToSpecifierMapIterator;

ostream	&operator<<(ostream &os, const CPaletteData::SColorSpecifier &d);
ostream	&operator<<(ostream &os, const CPaletteData::SBrushSpecifier &d);
ostream	&operator<<(ostream &os, const CPaletteData::SGroupRoleEntry &d);
ostream	&operator<<(ostream &os, const CPaletteData::SGroupRoleSpecifier &d);
ostream	&operator<<(ostream &os, const CPaletteData &d);

struct output_group_role_specifiers
	: public unary_function<
		pair<	CPaletteData::SGroupRoleEntry,
				CPaletteData::SGroupRoleSpecifier *>, void>
{
	explicit output_group_role_specifiers(ostream &os) : output_stream(os) { }

	void	operator()(pair<CPaletteData::SGroupRoleEntry,
							CPaletteData::SGroupRoleSpecifier *> x)
	{
		output_stream << x.first << "  " << *x.second << endl;
	}

	private:
	ostream		&output_stream;
};

struct output_auxiliary_palettes
	: public unary_function<pair<string, CPaletteData *>, void>
{
	explicit output_auxiliary_palettes(ostream &os) : output_stream(os) { }

	void	operator()(pair<string, CPaletteData *> x)
	{
		output_stream << "AUXILIARY_PALETTE  ";
		OutputStringData(output_stream, x.first);
		output_stream << endl << *x.second << endl;
	}

	private:
	ostream		&output_stream;
};

struct output_auxiliary_colors
	: public unary_function<
		pair<string, CPaletteData::SColorSpecifier *>, void>
{
	explicit output_auxiliary_colors(ostream &os) : output_stream(os) { }

	void	operator()(pair<string, CPaletteData::SColorSpecifier *> x)
	{
		output_stream << "AUXILIARY_COLOR  ";
		OutputStringData(output_stream, x.first);
		output_stream << "  " << *x.second << endl;
	}

	private:
	ostream		&output_stream;
};

struct output_auxiliary_brushes
	: public unary_function<
		pair<string, CPaletteData::SBrushSpecifier *>, void>
{
	explicit output_auxiliary_brushes(ostream &os) : output_stream(os) { }

	void	operator()(pair<string, CPaletteData::SBrushSpecifier *> x)
	{
		output_stream << "AUXILIARY_BRUSH  ";
		OutputStringData(output_stream, x.first);
		output_stream << endl << *x.second << endl;
	}

	private:
	ostream		&output_stream;
};

struct output_auxiliary_xpm_images
	: public unary_function<pair<string, CXpmImage *>, void>
{
	explicit output_auxiliary_xpm_images(ostream &os) : output_stream(os) { }

	void	operator()(pair<string, CXpmImage *> x)
	{
		output_stream << "AUXILIARY_XPM_IMAGE  ";
		OutputStringData(output_stream, x.first);
		output_stream << endl << *x.second;
	}

	private:
	ostream		&output_stream;
};

struct output_auxiliary_fonts
	: public unary_function<pair<string, string>, void>
{
	explicit output_auxiliary_fonts(ostream &os) : output_stream(os) { }

	void	operator()(pair<string, string> x)
	{
		output_stream << "AUXILIARY_FONT  ";
		OutputStringData(output_stream, x.first);
		output_stream << endl;
		OutputStringData(output_stream, x.second);
		output_stream << endl;
	}

	private:
	ostream		&output_stream;
};

struct output_auxiliary_strings
	: public unary_function<pair<string, string>, void>
{
	explicit output_auxiliary_strings(ostream &os) : output_stream(os) { }

	void	operator()(pair<string, string> x)
	{
		output_stream << "AUXILIARY_STRING  ";
		OutputStringData(output_stream, x.first);
		output_stream << endl;
		OutputStringData(output_stream, x.second);
		output_stream << endl;
	}

	private:
	ostream		&output_stream;
};

class COptionData
{
	public:
	struct SViewNodeData
	{
		enum ViewNodeType
		{
			SplitterNode,
			SubViewNode
		};

		vector<int>		*index;
		string			font_string;
		CPaletteData	*palette_data;

		SViewNodeData() : index(0), palette_data(0) { }
		virtual ~SViewNodeData()	{ delete index; delete palette_data; }
		SViewNodeData(const SViewNodeData &vnd);
		virtual SViewNodeData *clone() const = 0;

		virtual ViewNodeType	GetNodeType() = 0;
		virtual ostream			&OutputData(ostream &os) const = 0;

		void	SetIndex(vector<int> *v)
		{ delete index; index = v; }
		void	SetFontString(const string &s)
		{ font_string = s; }
		void	SetPaletteData(CPaletteData *p)
		{ delete palette_data; palette_data = p; }

		vector<int>		*GetIndex()				{ return index; }
		string			&GetFontString()		{ return font_string; }
		CPaletteData	*GetPaletteData()		{ return palette_data; }
	};

	struct SSplitterData: public SViewNodeData
	{
		Qt::Orientation		orientation;
		vector<int>			*pane_size;

		SSplitterData() : orientation(Qt::Horizontal), pane_size(0) { }
		virtual ~SSplitterData()	{ delete pane_size; }
		SSplitterData(const SSplitterData &sd);
		virtual SSplitterData *clone() const
		{ return new SSplitterData(*this); }

		virtual ViewNodeType	GetNodeType() { return SplitterNode; }
		virtual ostream			&OutputData(ostream &os) const;

		void	SetOrientation(Qt::Orientation o) { orientation = o; }
		void	SetPaneSize(vector<int> *v)
		{ delete pane_size; pane_size = v; }

		Qt::Orientation		GetOrientation()	{ return orientation; }
		vector<int>			*GetPaneSize()		{ return pane_size; }
	};

	struct SSubViewData: public SViewNodeData
	{
		struct SInterfaceSpecifier
		{
			string			main_view_name;
			string			subview_name;
			vector<int>		*index;

			SInterfaceSpecifier() : index(0) { }
			virtual ~SInterfaceSpecifier()	{ delete index; }
			SInterfaceSpecifier(const SInterfaceSpecifier &is);

			void	SetMainViewName(const string &s)	{ main_view_name = s; }
			void	SetSubViewName(const string &s)		{ subview_name = s; }
			void	SetIndex(vector<int> *v)	{ delete index; index = v; }

			string			&GetMainViewName()	{ return main_view_name; }
			string			&GetSubViewName()	{ return subview_name; }
			vector<int>		*GetIndex()			{ return index; }
		};

		string							subview_name;
		vector<SInterfaceSpecifier *>	*interface_specifiers;

		SSubViewData() : interface_specifiers(0) { }
		virtual ~SSubViewData();
		SSubViewData(const SSubViewData &svd);
		virtual SSubViewData *clone() const
		{ return new SSubViewData(*this); }

		virtual ViewNodeType	GetNodeType() { return SubViewNode; }
		virtual ostream			&OutputData(ostream &os) const;

		void		SetSubViewName(const string &s)	{ subview_name = s; }
		void		SetInterfaceSpecifiers(vector<SInterfaceSpecifier *> *v);

		string		&GetSubViewName()	{ return subview_name; }
		vector<SInterfaceSpecifier *>	*GetInterfaceSpecifiers()
		{ return interface_specifiers; }

		void		AddInterfaceSpecifier(SInterfaceSpecifier *is);
	};

	struct compare_view_nodes
		: public binary_function<SViewNodeData *, SViewNodeData *, bool>
	{
		bool	operator()(SViewNodeData *x, SViewNodeData *y) const
		{
			return *x->GetIndex() < *y->GetIndex();
		}
	};

	struct SMainViewData
	{
		string						main_view_name;
		bool						default_display_flag;
		QPoint						position;
		QSize						size;
		vector<SViewNodeData *>		*view_node_data_ptrs;

		SMainViewData()
			: default_display_flag(false), view_node_data_ptrs(0),
				position(-1, -1), size(-1, -1) { }
		~SMainViewData();
		SMainViewData(const SMainViewData &mvd);

		void	SetMainViewName(const string &s) { main_view_name = s; }
		void	SetDefaultDisplayFlag(bool f)	{ default_display_flag = f; }
		void	SetPosition(const QPoint &p)	{ position = p; }
		void	SetSize(const QSize &s)			{ size = s; }
		void	SetViewNodeDataPtrs(vector<SViewNodeData *> *v);

		const string	&GetMainViewName()		const { return main_view_name; }
		bool			GetDefaultDisplayFlag()	const
		{ return default_display_flag; }
		const QPoint	&GetPosition()			const { return position; }
		const QSize		&GetSize()				const { return size; }
		vector<SViewNodeData *>		*GetViewNodeDataPtrs() const
		{ return view_node_data_ptrs; }

		void		AddViewNodeDataPtr(SViewNodeData *vnd);
	};

	struct SApplicationData
	{
		QPoint				position;
		QSize				size;
		string				font_string;
		CPaletteData		*palette_data;

		SApplicationData() : palette_data(0), position(-1, -1), size(-1, -1) { }
		virtual ~SApplicationData() { delete palette_data; }
		SApplicationData(const SApplicationData &ad);

		void	SetPosition(const QPoint &p)	{ position = p; }
		void	SetSize(const QSize &s)			{ size = s; }
		void	SetFontString(const string &s)	{ font_string = s; }
		void	SetPaletteData(CPaletteData *p)
		{ delete palette_data; palette_data = p; }

		QPoint			&GetPosition()			{ return position; }
		QSize			&GetSize()				{ return size; }
		string			&GetFontString()		{ return font_string; }
		CPaletteData	*GetPaletteData()		{ return palette_data; }
	};

	struct SHelpData : public SApplicationData
	{
		SHelpData() { }
		SHelpData(const SHelpData &hd): SApplicationData(hd) { }
	};

	SApplicationData			*application_data;
	SHelpData					*help_data;
	vector<SMainViewData *>		*main_view_ptrs;

	map<string, CPaletteData *>						*auxiliary_palettes;
	map<string, CPaletteData::SColorSpecifier *>	*auxiliary_colors;
	map<string, CPaletteData::SBrushSpecifier *>	*auxiliary_brushes;
	map<string, CXpmImage *>						*auxiliary_xpm_images;
	map<string, string>								*auxiliary_fonts;
	map<string, string>								*auxiliary_strings;

	COptionData()
		:	application_data(0), help_data(0), main_view_ptrs(0),
			auxiliary_palettes(0), auxiliary_colors(0), auxiliary_brushes(0),
			auxiliary_xpm_images(0), auxiliary_fonts(0),
			auxiliary_strings(0) { }
	~COptionData();
	COptionData(const COptionData &od);

	void	SetApplicationData(SApplicationData *ad)
	{ delete application_data; application_data = ad; }
	void	SetHelpData(SHelpData *hd)
	{ delete help_data; help_data = hd; }
	void	SetMainViewPtrs(vector<SMainViewData *> *v, bool df = true);
	void	SetAuxiliaryPalettes(map<string, CPaletteData *> *m);
	void	SetAuxiliaryColors(map<string,
								CPaletteData::SColorSpecifier *> *m);
	void	SetAuxiliaryBrushes(map<string,
								CPaletteData::SBrushSpecifier *> *m);
	void	SetAuxiliaryXpmImages(map<string, CXpmImage *> *m);
	void	SetAuxiliaryFonts(map<string, string> *m)
	{ delete auxiliary_fonts; auxiliary_fonts  = m; }
	void	SetAuxiliaryStrings(map<string, string> *m)
	{ delete auxiliary_strings; auxiliary_strings = m; }

	SApplicationData								*GetApplicationData()
	{ return application_data; }
	SHelpData										*GetHelpData()
	{ return help_data; }
	vector<SMainViewData *>							*GetMainViewPtrs()
	{ return main_view_ptrs; }
	map<string, CPaletteData *>						*GetAuxiliaryPalettes()
	{ return auxiliary_palettes; }
	map<string, CPaletteData::SColorSpecifier *>	*GetAuxiliaryColors()
	{ return auxiliary_colors; }
	map<string, CPaletteData::SBrushSpecifier *>	*GetAuxiliaryBrushes()
	{ return auxiliary_brushes; }
	map<string, CXpmImage *>						*GetAuxiliaryXpmImages()
	{ return auxiliary_xpm_images; }
	map<string, string>								*GetAuxiliaryFonts()
	{ return auxiliary_fonts; }
	map<string, string>								*GetAuxiliaryStrings()
	{ return auxiliary_strings; }

	void	AddMainViewPtr(SMainViewData *p);
	void	AddAuxiliaryPalette(	const string &id,
									CPaletteData *pd);
	void	AddAuxiliaryColor(		const string &id,
									CPaletteData::SColorSpecifier *cs);
	void	AddAuxiliaryBrush(		const string &id,
									CPaletteData::SBrushSpecifier *bs);
	void	AddAuxiliaryXpmImage(	const string &id,
									CXpmImage *xi);
	void	AddAuxiliaryFont(		const string &id,
									const string &fs);
	void	AddAuxiliaryString(		const string &id,
									const string &s);

	CPaletteData					*GetAuxiliaryPalette(const string &id);
	CPaletteData::SColorSpecifier	*GetAuxiliaryColor(const string &id);
	CPaletteData::SBrushSpecifier	*GetAuxiliaryBrush(const string &id);
	CXpmImage						*GetAuxiliaryXpmImage(const string &id);
	string							*GetAuxiliaryFont(const string &id);
	string							*GetAuxiliaryString(const string &id);

	void	DeleteAuxiliaryPalette(		const string &id);
	void	DeleteAuxiliaryColor(		const string &id);
	void	DeleteAuxiliaryBrush(		const string &id);
	void	DeleteAuxiliaryXpmImage(	const string &id);
	void	DeleteAuxiliaryFont(		const string &id);
	void	DeleteAuxiliaryString(		const string &id);

	void	OrderViewNodes();
	void	AdjustMainViewNames(bool omit_last_flag = false);

	static CParserDriver::StatusCode
		InputOptionData(istream &is, COptionData **ptr_ptr_od);

	static CParserDriver::StatusCode
		InputOptionData(const unsigned char *i_buffer,
						int i_size, COptionData **ptr_ptr_od);

	static CParserDriver::StatusCode
		InputOptionDataFromClipboard(COptionData **ptr_ptr_od);

	void	CopyOptionDataToClipboard();
	void	GetSubviewCountMap(map<string, int> &cm);
};

ostream	&operator<<(ostream &os,
					const COptionData::SSubViewData::SInterfaceSpecifier &d);
ostream	&operator<<(ostream &os, const COptionData::SViewNodeData &d);
ostream	&operator<<(ostream &os, const COptionData::SMainViewData &d);
ostream	&operator<<(ostream &os, const COptionData::SApplicationData &d);
ostream	&operator<<(ostream &os, const COptionData::SHelpData &d);
ostream	&operator<<(ostream &os, const COptionData &d);

#endif
