/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "optionDialog.h"

CMessageDialog					*OptionDialog::message_dialog;
CInformationDialog				*OptionDialog::information_dialog;
CRootOrientationDialog			*OptionDialog::root_orientation_dialog;
CViewOrderingDialog				*OptionDialog::view_ordering_dialog;
CValueDialog					*OptionDialog::value_dialog;
CFileDirectoryDialog			*OptionDialog::file_directory_dialog;
CAuxiliaryStringDialog			*OptionDialog::auxiliary_string_dialog;
CAuxiliaryFontDialog         	*OptionDialog::auxiliary_font_dialog;
CMainviewTreeFontDialog      	*OptionDialog::mainview_tree_font_dialog;
CAuxiliaryColorDialog        	*OptionDialog::auxiliary_color_dialog;
CPaletteColorDialog          	*OptionDialog::palette_color_dialog;
CAuxiliaryPaletteDialog			*OptionDialog::auxiliary_palette_dialog;
CMainviewTreePaletteDialog		*OptionDialog::mainview_tree_palette_dialog;
CAuxiliaryXpmImageDialog		*OptionDialog::auxiliary_xpm_image_dialog;
CBrushXpmImageDialog			*OptionDialog::brush_xpm_image_dialog;
CAuxiliaryBrushDialog			*OptionDialog::auxiliary_brush_dialog;
CSizePositionDialog				*OptionDialog::size_position_dialog;

void	OptionDialog::CreateOptionDialogs(QWidget *parent)
{
	message_dialog					= new CMessageDialog(parent);
	information_dialog				= new CInformationDialog(parent);
	root_orientation_dialog			= new CRootOrientationDialog(parent);
	view_ordering_dialog			= new CViewOrderingDialog(parent);
	value_dialog					= new CValueDialog(parent);
	file_directory_dialog			= new CFileDirectoryDialog(parent);
	auxiliary_string_dialog			= new CAuxiliaryStringDialog(parent);
	auxiliary_font_dialog			= new CAuxiliaryFontDialog(parent);
	mainview_tree_font_dialog		= new CMainviewTreeFontDialog(parent);
	auxiliary_color_dialog			= new CAuxiliaryColorDialog(parent);
	palette_color_dialog			= new CPaletteColorDialog(parent);
	auxiliary_palette_dialog		= new CAuxiliaryPaletteDialog(parent);
	mainview_tree_palette_dialog	= new CMainviewTreePaletteDialog(parent);
	auxiliary_xpm_image_dialog		= new CAuxiliaryXpmImageDialog(parent);
	brush_xpm_image_dialog			= new CBrushXpmImageDialog(parent);
	auxiliary_brush_dialog			= new CAuxiliaryBrushDialog(parent);
	size_position_dialog			= new CSizePositionDialog(parent);
}

void	OptionDialog::DestroyOptionDialogs()
{
	delete message_dialog;
	delete information_dialog;
	delete root_orientation_dialog;
	delete view_ordering_dialog;
	delete value_dialog;
	delete file_directory_dialog;
	delete auxiliary_string_dialog;
	delete auxiliary_font_dialog;
	delete mainview_tree_font_dialog;
	delete auxiliary_color_dialog;
	delete palette_color_dialog;
	delete auxiliary_palette_dialog;
	delete mainview_tree_palette_dialog;
	delete auxiliary_xpm_image_dialog;
	delete brush_xpm_image_dialog;
	delete auxiliary_brush_dialog;
	delete size_position_dialog;
}

void	OptionDialog::SetPaletteAndFont(QWidget *w)
{
	message_dialog->setPalette(w->palette());
	message_dialog->setFont(w->font());
	information_dialog->setPalette(w->palette());
	information_dialog->setFont(w->font());
	root_orientation_dialog->setPalette(w->palette());
	root_orientation_dialog->setFont(w->font());
	view_ordering_dialog->setPalette(w->palette());
	view_ordering_dialog->setFont(w->font());
	value_dialog->setPalette(w->palette());
	value_dialog->setFont(w->font());
	file_directory_dialog->setPalette(w->palette());
	file_directory_dialog->setFont(w->font());
	auxiliary_string_dialog->setPalette(w->palette());
	auxiliary_string_dialog->setFont(w->font());
	auxiliary_font_dialog->setPalette(w->palette());
	auxiliary_font_dialog->setFont(w->font());
	mainview_tree_font_dialog->setPalette(w->palette());
	mainview_tree_font_dialog->setFont(w->font());
	auxiliary_color_dialog->setPalette(w->palette());
	auxiliary_color_dialog->setFont(w->font());
	palette_color_dialog->setPalette(w->palette());
	palette_color_dialog->setFont(w->font());
	auxiliary_palette_dialog->setPalette(w->palette());
	auxiliary_palette_dialog->setFont(w->font());
	mainview_tree_palette_dialog->setPalette(w->palette());
	mainview_tree_palette_dialog->setFont(w->font());
	auxiliary_xpm_image_dialog->setPalette(w->palette());
	auxiliary_xpm_image_dialog->setFont(w->font());
	brush_xpm_image_dialog->setPalette(w->palette());
	brush_xpm_image_dialog->setFont(w->font());
	auxiliary_brush_dialog->setPalette(w->palette());
	auxiliary_brush_dialog->setFont(w->font());
	size_position_dialog->setPalette(w->palette());
	size_position_dialog->setFont(w->font());
}

CMessageDialog::CMessageDialog(QWidget *parent)
	: CResizeDialogFc(parent, 3, 3, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 1);

	grid_layout->setColStretch(0, 1);
	grid_layout->setColStretch(1, 0);
	grid_layout->setColStretch(2, 0);

	QString     size_string("######");

	message_label		= new CLabelFc("", this);
	ok_button			= new CPushButtonFc("Ok", this, size_string);
	cancel_button		= new CPushButtonFc("Cancel", this, size_string);
	spacer				= new CSpacerFc(this);

	message_label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	grid_layout->addMultiCellWidget(message_label,		0, 0, 0, 2);
	grid_layout->addWidget(ok_button,					1, 1);
	grid_layout->addWidget(cancel_button,				1, 2);
	grid_layout->addMultiCellWidget(spacer,				2, 2, 0, 2);

	connect(ok_button, SIGNAL(clicked()), SLOT(accept()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(reject()));
}

void	CMessageDialog::SetTitleAndMessageText(	const string &title,
												const string &msg)
{
	message_label->setText(msg.c_str());
	message_label->CalculateGeometry();

	QPalette	p(palette());

	message_label->setPaletteBackgroundColor(
							p.color(QPalette::Active, QColorGroup::Button));

	message_label->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);

	setCaption(title.c_str());
	updateGeometry();

	setMaximumSize(size());
}

void    CMessageDialog::CalculateSizeAndPosition()
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		m = grid_layout_margin_data[pixel_size];
	int		s = grid_layout_spacing_data[pixel_size];

	int		w1 = message_label->minimumWidth() + 2 * m;
	int		w2 = ok_button->minimumWidth() +
					cancel_button->minimumWidth() + s + 2 * m;

	int		w = w1 < w2 ? w2 : w1;
	int		h = message_label->minimumHeight() + ok_button->minimumHeight() +
					spacer->minimumHeight() + 2 * s + 2 * m;

	dialog_size = QSize(w, h);

	QSize	parent_size = parentWidget()->size();

	dialog_position = QPoint(	(parent_size.width() - w) / 2,
								(parent_size.height() - h) / 2);
}

CInformationDialog::CInformationDialog(QWidget *parent)
	: CResizeDialogFc(parent, 3, 2, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 1);

	grid_layout->setColStretch(0, 1);
	grid_layout->setColStretch(1, 0);

	QString     size_string("######");

	message_label		= new CLabelFc("", this, false, false, true);
	ok_button			= new CPushButtonFc("Ok", this, size_string);
	spacer				= new CSpacerFc(this);

	message_label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	grid_layout->addMultiCellWidget(message_label,		0, 0, 0, 1);
	grid_layout->addWidget(ok_button,					1, 1);
	grid_layout->addMultiCellWidget(spacer,				2, 2, 0, 1);

	connect(ok_button, SIGNAL(clicked()), SLOT(accept()));
}

void	CInformationDialog::SetTitleAndMessageText(	const string &title,
												const string &msg)
{
	message_label->setText(msg.c_str());
	message_label->CalculateGeometry();

	QPalette	p(palette());

	message_label->setPaletteBackgroundColor(
							p.color(QPalette::Active, QColorGroup::Button));

	message_label->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);

	setCaption(title.c_str());
	updateGeometry();

	setMaximumSize(size());
}

void    CInformationDialog::SetTextAlignment(int f)
{
	message_label->setAlignment(f);
}

void    CInformationDialog::CalculateSizeAndPosition()
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		m = grid_layout_margin_data[pixel_size];
	int		s = grid_layout_spacing_data[pixel_size];

	int		w1 = message_label->minimumWidth() + 2 * m;
	int		w2 = ok_button->minimumWidth() + 2 * m;

	int		w = w1 < w2 ? w2 : w1;
	int		h = message_label->minimumHeight() + ok_button->minimumHeight() +
					spacer->minimumHeight() + 2 * s + 2 * m;

	dialog_size = QSize(w, h);

	QSize	parent_size = parentWidget()->size();

	dialog_position = QPoint(	(parent_size.width() - w) / 2,
								(parent_size.height() - h) / 2);
}

CRootOrientationDialog::CRootOrientationDialog(QWidget *parent)
	: CResizeDialogFc(parent, 3, 4, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		), add_node(0)
{
	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 1);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 1);
	grid_layout->setColStretch(2, 0);
	grid_layout->setColStretch(3, 0);

	QString     size_string("######");

	message_label		= new CLabelFc("Create Root Node", this);
	button_group		= new CHButtonGroupFc("Orientation", this);
	horizontal_radio_button	= new CRadioButtonFc("Horizontal", button_group);
	vertical_radio_button	= new CRadioButtonFc("Vertical", button_group);
	ok_button			= new CPushButtonFc("Ok", this, size_string);
	cancel_button		= new CPushButtonFc("Cancel", this, size_string);
	spacer				= new CSpacerFc(this);

	setCaption("Create Main View Root Node");
	message_label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
	button_group->setButton(0);

	grid_layout->addWidget(message_label,				0, 0);
	grid_layout->addMultiCellWidget(button_group,		0, 0, 1, 3);
	grid_layout->addWidget(ok_button,					1, 2);
	grid_layout->addWidget(cancel_button,				1, 3);
	grid_layout->addMultiCellWidget(spacer,				2, 2, 0, 3);

	connect(ok_button, SIGNAL(clicked()), SLOT(AcceptOperation()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(reject()));
}

void    CRootOrientationDialog::CalculateSizeAndPosition()
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		m = grid_layout_margin_data[pixel_size];
	int		s = grid_layout_spacing_data[pixel_size];

	int		w = message_label->minimumWidth() +
					button_group->minimumWidth() + s + 2 * m;

	int		h = message_label->minimumHeight() + ok_button->minimumHeight() +
					spacer->minimumHeight() + 2 * s + 2 * m;

	dialog_size = QSize(w, h);

	QSize	parent_size = parentWidget()->size();

	dialog_position = QPoint(	(parent_size.width() - w) / 2,
								(parent_size.height() - h) / 2);
}

void    CRootOrientationDialog::AcceptOperation()
{
	if (add_node)
	{
		if (horizontal_radio_button->isChecked())
			*add_node = Qt::Horizontal;
		else
			*add_node = Qt::Vertical;
	}

	accept();
}

CViewOrderingDialog::CViewOrderingDialog(QWidget *parent)
	: CDialogFc(parent, 3, 4, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	grid_layout->setRowStretch(0, 1);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);

	grid_layout->setColStretch(0, 1);
	grid_layout->setColStretch(1, 0);
	grid_layout->setColStretch(2, 0);
	grid_layout->setColStretch(3, 0);

	QString     size_string("######");

	main_view_listbox		= new CListBoxFc(this, "main view listbox", 6);
	move_up_button			= new CPushButtonFc("Up", this, size_string);
	move_down_button		= new CPushButtonFc("Down", this, size_string);
	move_to_first_button	= new CPushButtonFc("First", this, size_string);
	move_to_last_button		= new CPushButtonFc("Last", this, size_string);
	ok_button				= new CPushButtonFc("Ok", this, size_string);
	cancel_button			= new CPushButtonFc("Cancel", this, size_string);

	grid_layout->addMultiCellWidget(main_view_listbox,		0, 0, 0, 3);
	grid_layout->addWidget(move_up_button,					1, 1);
	grid_layout->addWidget(move_to_first_button,			1, 2);
	grid_layout->addWidget(ok_button,						1, 3);
	grid_layout->addWidget(move_down_button,				2, 1);
	grid_layout->addWidget(move_to_last_button,				2, 2);
	grid_layout->addWidget(cancel_button,					2, 3);

	connect(ok_button, SIGNAL(clicked()), SLOT(accept()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(reject()));

	connect(move_up_button,
			SIGNAL(clicked()),
			SLOT(MoveUp()));

	connect(move_down_button,
			SIGNAL(clicked()),
			SLOT(MoveDown()));

	connect(move_to_first_button,
			SIGNAL(clicked()),
			SLOT(MoveToFirst()));

	connect(move_to_last_button,
			SIGNAL(clicked()),
			SLOT(MoveToLast()));
}

void    CViewOrderingDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();
}

void    CViewOrderingDialog::SetPointers(	MainViewPtrs *p1,
											vector<MainViewPtrsIndex> *p2)
{
	main_view_ptrs	= p1;
	ordering_array	= p2;

	main_view_listbox->clear();
	ordering_array->clear();

	MainViewPtrsIndex	i;

	for (i=0; i<main_view_ptrs->size(); i++)
	{
		main_view_listbox->insertItem(
			(*main_view_ptrs)[i]->GetMainViewName().c_str());

		ordering_array->push_back(i);
	}

	main_view_listbox->setCurrentItem(0);
}

void    CViewOrderingDialog::MoveUp()
{
	int		current_item = main_view_listbox->currentItem();

	if (current_item <= 0)
		return;

	QString		s1 = main_view_listbox->text(current_item - 1);
	QString		s2 = main_view_listbox->text(current_item);

	main_view_listbox->changeItem(s2, current_item - 1);
	main_view_listbox->changeItem(s1, current_item);

	swap((*ordering_array)[current_item - 1], (*ordering_array)[current_item]);

	main_view_listbox->setCurrentItem(current_item - 1);
}

void    CViewOrderingDialog::MoveDown()
{
	int		current_item = main_view_listbox->currentItem();

	if (current_item == -1 || current_item == main_view_listbox->count() - 1)
		return;

	QString		s1 = main_view_listbox->text(current_item);
	QString		s2 = main_view_listbox->text(current_item + 1);

	main_view_listbox->changeItem(s2, current_item);
	main_view_listbox->changeItem(s1, current_item + 1);

	swap((*ordering_array)[current_item], (*ordering_array)[current_item + 1]);

	main_view_listbox->setCurrentItem(current_item + 1);
}

void    CViewOrderingDialog::MoveToFirst()
{
	int		current_item = main_view_listbox->currentItem();

	if (current_item <= 0)
		return;

	QString		s = main_view_listbox->text(current_item);

	main_view_listbox->removeItem(current_item);
	main_view_listbox->insertItem(s, 0);

	rotate(	ordering_array->begin(),
			ordering_array->begin() + current_item,
			ordering_array->begin() + current_item + 1);

	main_view_listbox->setCurrentItem(0);
}

void    CViewOrderingDialog::MoveToLast()
{
	int		current_item	= main_view_listbox->currentItem();
	int		last_index		= main_view_listbox->count() - 1;

	if (current_item == -1 || current_item == last_index)
		return;

	QString		s = main_view_listbox->text(current_item);

	main_view_listbox->removeItem(current_item);
	main_view_listbox->insertItem(s, last_index);

	rotate(	ordering_array->begin() + current_item,
			ordering_array->begin() + current_item + 1,
			ordering_array->end());

	main_view_listbox->setCurrentItem(last_index);
}

CValueDialog::CValueDialog(QWidget *parent)
	: CDialogFc(parent, 3, 4, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 1);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 1);
	grid_layout->setColStretch(2, 0);
	grid_layout->setColStretch(3, 0);

	QString     size_string("######");

	value_label			= new CLabelFc("Value ", this);
	value_line_edit		= new CLineEditFc(this, 
										OptionDialog::auxiliary_string_size);
	ok_button			= new CPushButtonFc("Ok", this, size_string);
	cancel_button		= new CPushButtonFc("Cancel", this, size_string);
	spacer				= new CSpacerFc(this);

	value_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

	grid_layout->addWidget(value_label,					0, 0);
	grid_layout->addMultiCellWidget(value_line_edit,	0, 0, 1, 3);
	grid_layout->addWidget(ok_button,					1, 2);
	grid_layout->addWidget(cancel_button,				1, 3);
	grid_layout->addMultiCellWidget(spacer,				2, 2, 0, 3);

	connect(ok_button, SIGNAL(clicked()), SLOT(AcceptOperation()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(reject()));
}

void	CValueDialog::SetTitleAndLabelText(	const string &title,
											const string &lt)
{
	setCaption(title.c_str());
	value_label->setText(lt.c_str());
	value_label->CalculateGeometry();
	updateGeometry();

	setMaximumHeight(height());
}

void	CValueDialog::AcceptOperation()
{
	*value_pointer =
		string((const char *)(value_line_edit->text().local8Bit()));

	accept();
}

CFileDirectoryDialog::CFileDirectoryDialog(QWidget *parent)
	: CDialogFc(parent, 4, 4, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		),	current_entry(0),
			unselected_file_pixmap(0), selected_file_pixmap(0),
			unselected_directory_pixmap(0), selected_directory_pixmap(0),
			unselected_alternate_pixmap(0), selected_alternate_pixmap(0)
{
	grid_layout->setRowStretch(0, 1);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);
	grid_layout->setRowStretch(3, 0);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 1);
	grid_layout->setColStretch(2, 0);
	grid_layout->setColStretch(3, 0);

	QString     size_string("######");

	directory_entry_listbox	= new CListBoxFc(this, "directory listbox", 6);
	directory_label			= new CLabelFc("Directory ", this);
	directory_line_edit		= new CLineEditFc(this,
										OptionDialog::path_string_size);
	file_label				= new CLabelFc("File ", this);
	file_line_edit			= new CLineEditFc(this,
										OptionDialog::path_string_size);
	status_button			= new CPushButtonFc("Status", this, size_string);
	type_label				= new CLabelFc("Type ", this);
	type_combo_box			= new CComboBoxFc(false, this,
										OptionDialog::path_string_size);
	ok_button				= new CPushButtonFc("Ok", this, size_string);
	cancel_button			= new CPushButtonFc("Cancel", this, size_string);

	directory_entry_listbox->setRowMode(QListBox::FitToHeight);
	type_combo_box->setInsertionPolicy(QComboBox::NoInsertion);
	directory_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
	file_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
	type_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
	ok_button->setDefault(true);

	SetupPixmaps();

	grid_layout->addMultiCellWidget(directory_entry_listbox,	0, 0, 0, 3);
	grid_layout->addWidget(directory_label,						1, 0);
	grid_layout->addMultiCellWidget(directory_line_edit,		1, 1, 1, 3);
	grid_layout->addWidget(file_label,							2, 0);
	grid_layout->addMultiCellWidget(file_line_edit,				2, 2, 1, 2);
	grid_layout->addWidget(status_button,						2, 3);
	grid_layout->addWidget(type_label,							3, 0);
	grid_layout->addWidget(type_combo_box,						3, 1);
	grid_layout->addWidget(ok_button,							3, 2);
	grid_layout->addWidget(cancel_button,						3, 3);

	connect(ok_button, SIGNAL(clicked()), SLOT(AcceptOperation()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(reject()));

	connect(status_button,
			SIGNAL(clicked()),
			SLOT(DisplayStatus()));
}

CFileDirectoryDialog::~CFileDirectoryDialog()
{
	delete unselected_file_pixmap;
	delete selected_file_pixmap;
	delete unselected_directory_pixmap;
	delete selected_directory_pixmap;
	delete unselected_alternate_pixmap;
	delete selected_alternate_pixmap;
}

void	CFileDirectoryDialog::ConnectListBox()
{
	connect(directory_entry_listbox,
			SIGNAL(highlighted(QListBoxItem *)),
			SLOT(DirectoryEntryHighlighted(QListBoxItem *)));

	connect(directory_entry_listbox,
			SIGNAL(selected(QListBoxItem *)),
			SLOT(DirectoryEntrySelected(QListBoxItem *)));
}

void	CFileDirectoryDialog::DisconnectListBox()
{
	disconnect(	directory_entry_listbox,
				SIGNAL(highlighted(QListBoxItem *)),
				this,
				SLOT(DirectoryEntryHighlighted(QListBoxItem *)));

	disconnect(	directory_entry_listbox,
				SIGNAL(selected(QListBoxItem *)),
				this,
				SLOT(DirectoryEntrySelected(QListBoxItem *)));
}

void	CFileDirectoryDialog::ConnectComboBox()
{
	connect(type_combo_box,
			SIGNAL(activated(int)),
			SLOT(FileTypeSelected(int)));
}

void	CFileDirectoryDialog::DisconnectComboBox()
{
	disconnect(	type_combo_box,
				SIGNAL(activated(int)),
				this,
				SLOT(FileTypeSelected(int)));
}

void	CFileDirectoryDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();
}

void	CFileDirectoryDialog::SetFileTypes(
									vector< pair<string, string> > *types)
{
	DisconnectComboBox();

	type_combo_box->clear();

	if (types)
		file_types = *types;
	else
		file_types.clear();

	vector< pair<string, string> >::size_type	i;

	for (i=0; i<file_types.size(); i++)
	{
		string	item_text(file_types[i].first);
		item_text += " (";
		item_text += file_types[i].second;
		item_text += ")";

		type_combo_box->insertItem(item_text.c_str());
	}

	if (file_types.size())
		type_combo_box->setCurrentItem(0);

	ConnectComboBox();
}

void	CFileDirectoryDialog::SetDirectoryPath(const string &p)
{
	string		directory_path;

	if (p.length())
	{
		directory_path = p;

		directory_line_edit->setText(directory_path.c_str());
	}
	else
	{
		if (!directory_line_edit->text().length())
		{
			FilePath::GetCurrentWorkingDirectory(directory_path);

			directory_line_edit->setText(directory_path.c_str());
		}
		else
			directory_path = (const char *)
								(directory_line_edit->text().local8Bit());
	}

	directory_entries.clear();

	list<string>	entry_strings;

	DIR		*directory_stream = opendir(directory_path.c_str());

	if (directory_stream)
	{
		struct dirent	*directory_entry;

		while (directory_entry = readdir(directory_stream))
			entry_strings.push_back(directory_entry->d_name);

		entry_strings.sort();

		list<string>::iterator	l_iterator = entry_strings.begin();

		while (l_iterator != entry_strings.end())
		{
			struct stat		file_status;
			string			full_path = directory_path;
		
			full_path += "/";
			full_path += *l_iterator;

			if (lstat(full_path.c_str(), &file_status) != -1)
			{
				if ((file_status.st_mode & S_IFMT) == S_IFREG)
				{
					directory_entries.push_back(
						CEntryData(	directory_path,
									*l_iterator,
									full_path,
									file_status,
									unselected_file_pixmap,
									selected_file_pixmap));
				}
				else if ((file_status.st_mode & S_IFMT) == S_IFDIR)
				{
					directory_entries.push_back(
						CEntryData(	directory_path,
									*l_iterator,
									full_path,
									file_status,
									unselected_directory_pixmap,
									selected_directory_pixmap));
				}
				else
				{
					directory_entries.push_back(
						CEntryData(	directory_path,
									*l_iterator,
									full_path,
									file_status,
									unselected_alternate_pixmap,
									selected_alternate_pixmap));
				}
			}

			l_iterator++;
		}
	}

	closedir(directory_stream);

	CreateListBoxItems();
}

void	CFileDirectoryDialog::CreateListBoxItems()
{
	DisconnectListBox();

	directory_entry_listbox->clear();

	string		wildcard_pattern;
	QRegExp		regexp;

	if (file_types.size())
	{
		wildcard_pattern	= file_types[type_combo_box->currentItem()].second;
		regexp				= QRegExp(wildcard_pattern.c_str(), true, true);
	}

	list<CEntryData>::iterator	l_iterator = directory_entries.begin();

	while (l_iterator != directory_entries.end())
	{
		CEntryData		&entry_data = *l_iterator;

		if (!((entry_data.status_data.st_mode & S_IFMT) == S_IFREG) ||
			!wildcard_pattern.length() ||
			regexp.exactMatch(entry_data.entry_name.c_str()))
		{
			directory_entry_listbox->insertItem(
				new CDirectoryEntry(entry_data, *entry_data.unselected_pixmap));
		}

		l_iterator++;
	}

	current_entry = 0;

	ConnectListBox();
}

void	CFileDirectoryDialog::SetFileName(const string &f)
{
	DisconnectListBox();

	file_line_edit->setText(f.c_str());

	QListBoxItem	*item = directory_entry_listbox->findItem(f.c_str(),
										Qt::CaseSensitive | Qt::ExactMatch);

	if (item)
	{
		int     index = directory_entry_listbox->index(item);

		CDirectoryEntry		*de = dynamic_cast<CDirectoryEntry *>(item);

		CEntryData		&entry_data = de->entry_data;

		current_entry = new CDirectoryEntry(entry_data,
											*entry_data.selected_pixmap);

		directory_entry_listbox->changeItem(current_entry, index);
	}
	else
	{
		item = directory_entry_listbox->firstItem();

		if (item)
		{
			CDirectoryEntry		*de = dynamic_cast<CDirectoryEntry *>(item);

			CEntryData		&entry_data = de->entry_data;

			current_entry = new CDirectoryEntry(entry_data,
												*entry_data.selected_pixmap);

			directory_entry_listbox->changeItem(current_entry, 0);
		}
		else
			current_entry = 0;
	}

	ConnectListBox();
}

int		CFileDirectoryDialog::DisplayOpenFileDialog(
								string *path,
								const string &title,
								const string &directory,
								const string &file,
								vector< pair<string, string> > *types)
{
	SetOpenFileMode();
	SetPathPointer(path);
	SetTitleAndResize(title);
	SetFileTypes(types);
	SetDirectoryPath(directory);
	SetFileName(file);

	return exec();
}

int		CFileDirectoryDialog::DisplaySaveAsFileDialog(
								string *path,
								const string &title,
								const string &directory,
								const string &file,
								vector< pair<string, string> > *types)
{
	SetSaveAsFileMode();
	SetPathPointer(path);
	SetTitleAndResize(title);
	SetFileTypes(types);
	SetDirectoryPath(directory);
	SetFileName(file);

	return exec();
}

void	CFileDirectoryDialog::setFont(const QFont &f)
{
	CDialogFc::setFont(f);

	SetupPixmaps();
}

void    CFileDirectoryDialog::SetupPixmaps()
{
	QFontInfo       fi(font());

	int     pixel_size  = fi.pixelSize();

	delete unselected_file_pixmap;
	delete selected_file_pixmap;
	delete unselected_directory_pixmap;
	delete selected_directory_pixmap;
	delete unselected_alternate_pixmap;
	delete selected_alternate_pixmap;

	char	*pixmap_data[pixel_size + 3];

	ostrstream		row0_data;

	row0_data << pixel_size << " " << pixel_size << " 2 1" << '\0';

	pixmap_data[0] = new char[strlen(row0_data.str()) + 1];

	strcpy(pixmap_data[0], row0_data.str());

	string		row1_data("# c None");
	string		row2_color1_data(". c #000000");
	string		row2_color2_data(". c #c0c0c0");

	pixmap_data[1] = new char[row1_data.length() + 1];
	pixmap_data[2] = new char[row2_color1_data.length() + 1];

	strcpy(pixmap_data[1], row1_data.c_str());
	strcpy(pixmap_data[2], row2_color1_data.c_str());

	string		solid_bar(pixel_size + 2, '.');
	string		space_bar(pixel_size + 2, '#');

	solid_bar[pixel_size] = '#';
	solid_bar[pixel_size + 1] = '#';
	space_bar[0] = '.';
	space_bar[pixel_size - 1] = '.';

	int		bar_index = pixel_size / 4 + 3;
	int		i;

	for (i=3; i<pixel_size + 3; i++)
	{
		pixmap_data[i] = new char[pixel_size + 3];

		if (i == 3 || i == bar_index || i == pixel_size + 2)
			strcpy(pixmap_data[i], solid_bar.c_str());
		else
			strcpy(pixmap_data[i], space_bar.c_str());
	}

	unselected_file_pixmap =
		new QPixmap(const_cast<const char **>(pixmap_data));

	strcpy(pixmap_data[2], row2_color2_data.c_str());

	selected_file_pixmap =
		new QPixmap(const_cast<const char **>(pixmap_data));

	strcpy(pixmap_data[2], row2_color1_data.c_str());

	for (i=bar_index + 1; i<pixel_size + 2; i++)
		strcpy(pixmap_data[i], solid_bar.c_str());

	unselected_directory_pixmap =
		new QPixmap(const_cast<const char **>(pixmap_data));

	strcpy(pixmap_data[2], row2_color2_data.c_str());

	selected_directory_pixmap =
		new QPixmap(const_cast<const char **>(pixmap_data));

	strcpy(pixmap_data[2], row2_color1_data.c_str());

	for (i=4; i<pixel_size + 2; i++)
	{
		strcpy(pixmap_data[i], solid_bar.c_str());

		int		j;
		int		j1 = i - 2;
		int		j2 = pixel_size + 1 - i;

		if (j1 <= j2)
			for (j=j1; j<=j2; j++)
				pixmap_data[i][j] = '#';
		else
			for (j=j2+2; j<=j1-2; j++)
				pixmap_data[i][j] = '#';
	}

	unselected_alternate_pixmap =
		new QPixmap(const_cast<const char **>(pixmap_data));

	strcpy(pixmap_data[2], row2_color2_data.c_str());

	selected_alternate_pixmap =
		new QPixmap(const_cast<const char **>(pixmap_data));

	for (i=0; i<pixel_size + 3; i++)
		delete [] pixmap_data[i];
}

void	CFileDirectoryDialog::AcceptOperation()
{
	string		full_path =
				(const char *)(directory_line_edit->text().local8Bit());
		
	if (file_line_edit->text().isEmpty())
	{
		SetDirectoryPath(full_path);
		SetFileName(string());
	}
	else
	{
		full_path += "/";
		full_path += (const char *)(file_line_edit->text().local8Bit());

		if (check_file_exists_flag && !FilePath::FileExists(full_path))
			return;

		*path_pointer = full_path;

		accept();
	}
}

void	CFileDirectoryDialog::DirectoryEntryHighlighted(QListBoxItem *entry)
{
	int     current_entry_index	= directory_entry_listbox->index(current_entry);
	int		next_entry_index	= directory_entry_listbox->index(entry);

	if (current_entry_index == next_entry_index)
		return;

	DisconnectListBox();

	if (current_entry)
	{
		CEntryData		&entry_data = current_entry->entry_data;

		directory_entry_listbox->changeItem(
			new CDirectoryEntry(entry_data, *entry_data.unselected_pixmap),
			current_entry_index);
	}

	CDirectoryEntry		*de = dynamic_cast<CDirectoryEntry *>(entry);

	CEntryData		&entry_data = de->entry_data;

	current_entry = new CDirectoryEntry(entry_data,
										*entry_data.selected_pixmap);

	directory_entry_listbox->changeItem(current_entry, next_entry_index);

	if ((entry_data.status_data.st_mode & S_IFMT) == S_IFREG)
		file_line_edit->setText(entry_data.entry_name.c_str());
	else
		file_line_edit->clear();

	ConnectListBox();
}

void	CFileDirectoryDialog::DirectoryEntrySelected(QListBoxItem *entry)
{
	CDirectoryEntry		*de			= dynamic_cast<CDirectoryEntry *>(entry);
	CEntryData			&entry_data	= de->entry_data;

	if (entry_data.entry_name == ".")
	{
		SetDirectoryPath(string());
		SetFileName(string());
		return;
	}

	if (entry_data.entry_name == "..")
	{
		string	parent_directory;
		string	sub_directory;

		FilePath::GetPathAndFileName(
			(const char *)(directory_line_edit->text().local8Bit()),
			parent_directory, sub_directory);

		SetDirectoryPath(parent_directory);
		SetFileName(string());
		return;
	}

	if ((entry_data.status_data.st_mode & S_IFMT) == S_IFREG)
	{
		*path_pointer = entry_data.full_path;

		accept();
	}
	else if ((entry_data.status_data.st_mode & S_IFMT) == S_IFDIR)
	{
		SetDirectoryPath(entry_data.full_path);
		SetFileName(string());
	}
}

void	CFileDirectoryDialog::FileTypeSelected(int index)
{
	CreateListBoxItems();
}

void	CFileDirectoryDialog::DisplayStatus()
{
	CEntryData		&entry_data		= current_entry->entry_data;
	struct stat		&status_data	= entry_data.status_data;

	string		filesystem_type;
	string		set_uid_bit(	"set uid bit = ");
	string		set_gid_bit(	"set gid bit = ");
	string		sticky_bit(		"sticky bit  = ");
	string		owner(	"owner  = ");
	string		group(	"group  = ");
	string		others(	"others = ");

	switch (status_data.st_mode & S_IFMT)
	{
		case S_IFSOCK:	filesystem_type = "socket";				break;
		case S_IFLNK:	filesystem_type = "symbolic link";		break;
		case S_IFREG:	filesystem_type = "regular file";		break;
		case S_IFBLK:	filesystem_type = "block device";		break;
		case S_IFDIR:	filesystem_type = "directory";			break;
		case S_IFCHR:	filesystem_type = "character device";	break;
		case S_IFIFO:	filesystem_type = "fifo";				break;
	}

	if (status_data.st_mode & S_ISUID)
		set_uid_bit += '1';
	else
		set_uid_bit += '0';

	if (status_data.st_mode & S_ISGID)
		set_gid_bit += '1';
	else
		set_gid_bit += '0';

	if (status_data.st_mode & S_ISVTX)
		sticky_bit += '1';
	else
		sticky_bit += '0';

	if (status_data.st_mode & S_IRUSR)
		owner += "read";
	else
		owner += "    ";

	if (status_data.st_mode & S_IWUSR)
		owner += " write";
	else
		owner += "      ";

	if (status_data.st_mode & S_IXUSR)
		owner += " execute";
	else
		owner += "        ";

	if (status_data.st_mode & S_IRGRP)
		group += "read";
	else
		group += "    ";

	if (status_data.st_mode & S_IWGRP)
		group += " write";
	else
		group += "      ";

	if (status_data.st_mode & S_IXGRP)
		group += " execute";
	else
		group += "        ";

	if (status_data.st_mode & S_IROTH)
		others += "read";
	else
		others += "    ";

	if (status_data.st_mode & S_IWOTH)
		others += " write";
	else
		others += "      ";

	if (status_data.st_mode & S_IXOTH)
		others += " execute";
	else
		others += "        ";

	ostrstream		status_text;

	status_text	<< "directory: " << entry_data.directory_path << endl;
	status_text << "file name: " << entry_data.entry_name << endl;
	status_text << "device           = " << status_data.st_dev << endl;
	status_text << "device type      = " << status_data.st_rdev << endl;
	status_text << "inode            = " << status_data.st_ino << endl;
	status_text << "# hard links     = " << status_data.st_nlink << endl;
	status_text << "blocksize        = " << status_data.st_blksize << endl;
	status_text << "# blocks         = " << status_data.st_blocks << endl;
	status_text << "total size       = " << status_data.st_size << endl;
	status_text << "user id          = " << status_data.st_uid << endl;
	status_text << "group id         = " << status_data.st_gid << endl;

	status_text << "file system type = " << filesystem_type << endl;
	status_text << set_uid_bit << endl;
	status_text << set_gid_bit << endl;
	status_text << sticky_bit << endl;
	status_text << owner << endl;
	status_text << group << endl;
	status_text << others << endl;

	status_text << "last access       " << ctime(&status_data.st_atime);
	status_text << "last modification " << ctime(&status_data.st_mtime);
	status_text << "last change       " << ctime(&status_data.st_ctime);

	status_text << '\0';

	CInformationDialog	*ifd = OptionDialog::information_dialog;

	QFont	current_font(ifd->font());
	QFont	source_code_font(current_font);

	source_code_font.setFamily("Courier");
	source_code_font.setItalic(false);

	ifd->setFont(source_code_font);
	ifd->SetTextAlignment(Qt::AlignLeft | Qt::AlignTop);

	ifd->SetTitleAndMessageText("Filesystem Status", status_text.str());

	ifd->ExecuteDialog();

	ifd->SetTextAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
	ifd->setFont(current_font);
}

CAuxiliaryStringDialog::CAuxiliaryStringDialog(QWidget *parent)
	: CDialogFc(parent, 4, 4, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);
	grid_layout->setRowStretch(3, 1);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 1);
	grid_layout->setColStretch(2, 0);
	grid_layout->setColStretch(3, 0);

	QString     size_string("######");

	name_label			= new CLabelFc("Name ", this);
	name_line_edit		= new CLineEditFc(this,
										OptionDialog::auxiliary_string_size);
	value_label			= new CLabelFc("Value ", this);
	value_line_edit		= new CLineEditFc(this, 
										OptionDialog::auxiliary_string_size);
	ok_button			= new CPushButtonFc("Ok", this, size_string);
	cancel_button		= new CPushButtonFc("Cancel", this, size_string);
	spacer				= new CSpacerFc(this);

	name_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
	value_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

	grid_layout->addWidget(name_label,					0, 0);
	grid_layout->addMultiCellWidget(name_line_edit,		0, 0, 1, 3);
	grid_layout->addWidget(value_label,					1, 0);
	grid_layout->addMultiCellWidget(value_line_edit,	1, 1, 1, 3);
	grid_layout->addWidget(ok_button,					2, 2);
	grid_layout->addWidget(cancel_button,				2, 3);
	grid_layout->addMultiCellWidget(spacer,				3, 3, 0, 3);

	connect(ok_button, SIGNAL(clicked()), SLOT(AcceptOperation()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(reject()));

	connect(name_line_edit,
			SIGNAL(textChanged(const QString &)),
			SLOT(NameChanged(const QString &)));

	connect(value_line_edit,
			SIGNAL(textChanged(const QString &)),
			SLOT(ValueChanged(const QString &)));
}

void	CAuxiliaryStringDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();

	setMaximumHeight(height());
}

void	CAuxiliaryStringDialog::AcceptOperation()
{
	*name_pointer =
		string((const char *)(name_line_edit->text().local8Bit()));

	*value_pointer =
		string((const char *)(value_line_edit->text().local8Bit()));

	accept();
}

void    CAuxiliaryStringDialog::NameChanged(const QString &s)
{
	name_changed_flag = true;
}

void    CAuxiliaryStringDialog::ValueChanged(const QString &s)
{
	value_changed_flag = true;
}

void	CBaseFontDialog::CreateFontDatabaseMap()
{
	QStringList		family_list = font_database.families();

	QStringList::Iterator	f_iterator = family_list.begin();

	while (f_iterator != family_list.end())
	{
		QString		family = *f_iterator;
		string		f_name((const char *)(family.local8Bit()));

		StyleToPointSizeMap		style_map;

		database_map[f_name] = style_map;

		StyleToPointSizeMap		&s_map = (*database_map.find(f_name)).second;

		QStringList		style_list = font_database.styles(family);

		QStringList::Iterator	s_iterator = style_list.begin();

		while (s_iterator != style_list.end())
		{
			QString		style = *s_iterator;
			string		s_name((const char *)(style.local8Bit()));

			vector<int>		point_vector;

			s_map[s_name] = point_vector;

			vector<int>		&p_vector = (*s_map.find(s_name)).second;

			QValueList<int>		point_size_list =
				font_database.pointSizes(family, style);

			QValueList<int>::Iterator	p_iterator = point_size_list.begin();

			while (p_iterator != point_size_list.end())
			{
				int		point = *p_iterator;

				p_vector.push_back(point);

				p_iterator++;
			}

			s_iterator++;
		}

		f_iterator++;
	}
}

CBaseFontDialog::CBaseFontDialog(QWidget *parent)
	: CDialogFc(parent, 2, 1, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	QFontInfo   fi(font());

	int     pixel_size = fi.pixelSize();

	int     s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1 = new QGridLayout(this, 2, 4, 0, s, "subgrid1");
	subgrid_layout2 = new QGridLayout(this, 2, 4, 0, s, "subgrid2");

	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 1);

	grid_layout->setColStretch(0, 1);

	subgrid_layout1->setRowStretch(0, 0);
	subgrid_layout1->setRowStretch(1, 0);

	subgrid_layout1->setColStretch(0, 0);
	subgrid_layout1->setColStretch(1, 1);
	subgrid_layout1->setColStretch(2, 0);
	subgrid_layout1->setColStretch(3, 0);

	subgrid_layout2->setRowStretch(0, 0);
	subgrid_layout2->setRowStretch(1, 1);

	subgrid_layout2->setColStretch(0, 1);
	subgrid_layout2->setColStretch(1, 0);
	subgrid_layout2->setColStretch(2, 0);
	subgrid_layout2->setColStretch(3, 2);

	QString     size_string("######");

	name_label			= new CLabelFc("Name", this, false, true);
	ok_button			= new CPushButtonFc("Ok", this, size_string);
	value_label			= new CLabelFc("Value", this, false, true);
	value_line_edit		= new CLineEditFc(this,
										OptionDialog::auxiliary_string_size);
	display_font_button	= new CPushButtonFc("Display", this);
	cancel_button		= new CPushButtonFc("Cancel", this, size_string);
	family_label		= new CLabelFc("Family", this);
	style_label			= new CLabelFc("Style", this);
	point_size_label	= new CLabelFc("Size", this);
	character_display_label	= new CLabelFc("Character Display", this);
	family_listbox		= new CListBoxFc(this, "family name size string ##", 6);
	style_listbox		= new CListBoxFc(this, "style size ###", 6, true);
	point_size_listbox	= new CListBoxFc(this, "size #", 6, true);
	scroll_view			= new QScrollView(this);
	font_display_label	= new CLabelFc("", scroll_view->viewport());

	scroll_view->addChild(font_display_label);
	scroll_view->setResizePolicy(QScrollView::AutoOneFit);
	scroll_view->setHScrollBarMode(QScrollView::AlwaysOn);
	scroll_view->setVScrollBarMode(QScrollView::AlwaysOn);

	name_label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
	value_label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
	family_label->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
	style_label->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
	point_size_label->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
	character_display_label->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
	font_display_label->setAlignment(Qt::AlignLeft | Qt::AlignTop);

	display_font_button->setDefault(true);

	subgrid_layout1->addWidget(name_label,					0, 0);
	subgrid_layout1->addWidget(ok_button,					0, 3);
	subgrid_layout1->addWidget(value_label,					1, 0);
	subgrid_layout1->addWidget(value_line_edit,				1, 1);
	subgrid_layout1->addWidget(display_font_button,			1, 2);
	subgrid_layout1->addWidget(cancel_button,				1, 3);

	subgrid_layout2->addWidget(family_label,				0, 0);
	subgrid_layout2->addWidget(style_label,					0, 1);
	subgrid_layout2->addWidget(point_size_label,			0, 2);
	subgrid_layout2->addWidget(character_display_label,		0, 3);
	subgrid_layout2->addWidget(family_listbox,				1, 0);
	subgrid_layout2->addWidget(style_listbox,				1, 1);
	subgrid_layout2->addWidget(point_size_listbox,			1, 2);
	subgrid_layout2->addWidget(scroll_view,					1, 3);

	grid_layout->addLayout(subgrid_layout1,			0, 0);
	grid_layout->addLayout(subgrid_layout2,			1, 0);

	connect(ok_button, SIGNAL(clicked()), SLOT(AcceptOperation()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(reject()));

	connect(value_line_edit,
			SIGNAL(textChanged(const QString &)),
			SLOT(ValueChanged(const QString &)));

	connect(display_font_button,
			SIGNAL(clicked()),
			SLOT(DisplayFont()));

	connect(family_listbox,
			SIGNAL(highlighted(int)),
			SLOT(SelectFamily(int)));

	connect(style_listbox,
			SIGNAL(highlighted(int)),
			SLOT(SelectStyle(int)));

	connect(point_size_listbox,
			SIGNAL(highlighted(int)),
			SLOT(SelectPointSize(int)));

	QPalette	p;

	p.setColor(QColorGroup::Background, QColor(255, 255, 255));
	p.setColor(QColorGroup::Text, QColor(0, 0, 0));

	font_display_label->setPalette(p);

	string	font_display_string;
	int		index = 32;

	for (int i=2; i<8; i++)
	{
		for (int j=0; j<16; j++)
		{
			font_display_string += (char)index;

			index++;
		}

		font_display_string += '\n';
	}

	index = 160;

	for (int i=10; i<16; i++)
	{
		for (int j=0; j<16; j++)
		{
			font_display_string += (char)index;

			index++;
		}

		font_display_string += '\n';
	}

	font_display_label->setText(font_display_string.c_str());

	CreateFontDatabaseMap();

	for_each(	database_map.begin(),
				database_map.end(),
				add_family_types_to_listbox(family_listbox));

	family_listbox->setCurrentItem(0);
}

void	CBaseFontDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();
}

void	CBaseFontDialog::SetValueLineEdit(const QString &v)
{
	value_line_edit->setText(v);
	value_changed_flag = false;

	DisplayFont();
}

void    CBaseFontDialog::ValueChanged(const QString &s)
{
	value_changed_flag = true;
}

void	CBaseFontDialog::DisplayFont()
{
	QFont	f;

	FontSpecifier::SetFontSpecifier(f,
		string((const char *)(value_line_edit->text().local8Bit())));
	
	font_display_label->setFont(f);
}

void	CBaseFontDialog::SelectFamily(int index)
{
	style_listbox->clear();

	current_style_map =
		database_map[(const char *)
								(family_listbox->currentText().local8Bit())];

	for_each(	current_style_map.begin(),
				current_style_map.end(),
				add_style_types_to_listbox(style_listbox));

	style_listbox->setCurrentItem(0);
}

void	CBaseFontDialog::SelectStyle(int index)
{
	point_size_listbox->clear();

	current_point_size_vector =
		current_style_map[(const char *)
								(style_listbox->currentText().local8Bit())];

	vector<int>::size_type		i;

	for (i=0; i<current_point_size_vector.size(); i++)
	{
		ostrstream		s;

		s << current_point_size_vector[i] << '\0';

		point_size_listbox->insertItem(s.str());
	}

	point_size_listbox->setCurrentItem(0);
}

void	CBaseFontDialog::SelectPointSize(int index)
{
	QFont	f = font_database.font(	family_listbox->currentText(),
									style_listbox->currentText(),
									current_point_size_vector[index]);
	
	ostrstream		s;

	s << string((const char *)(family_listbox->currentText().local8Bit()));
	s << ";";
	s << string((const char *)(style_listbox->currentText().local8Bit()));
	s << ";";
	s << current_point_size_vector[index];
	s << '\0';

	value_line_edit->setText(s.str());

	font_display_label->setFont(f);
}

CAuxiliaryFontDialog::CAuxiliaryFontDialog(QWidget *parent)
	: CBaseFontDialog(parent)
{
	name_line_edit	= new CLineEditFc(this, 
										OptionDialog::auxiliary_string_size);

	subgrid_layout1->addWidget(name_line_edit,		0, 1);

	connect(name_line_edit,
			SIGNAL(textChanged(const QString &)),
			SLOT(NameChanged(const QString &)));
}

void    CAuxiliaryFontDialog::SetNameLineEdit(const QString &n)
{
	name_line_edit->setText(n);
	name_changed_flag = false;
}

void	CAuxiliaryFontDialog::AcceptOperation()
{
	*name_pointer =
		string((const char *)(name_line_edit->text().local8Bit()));

	*value_pointer =
		string((const char *)(value_line_edit->text().local8Bit()));

	accept();
}

void    CAuxiliaryFontDialog::NameChanged(const QString &s)
{
	name_changed_flag = true;
}

CMainviewTreeFontDialog::CMainviewTreeFontDialog(QWidget *parent)
	: CBaseFontDialog(parent)
{
	name_combo_box			= new CComboBoxFc(true, this,
										OptionDialog::auxiliary_string_size);
	add_auxiliary_button	= new CPushButtonFc("Add Auxiliary", this);

	subgrid_layout1->addWidget(name_combo_box,				0, 1);
	subgrid_layout1->addWidget(add_auxiliary_button,		0, 2);

	name_combo_box->setInsertionPolicy(QComboBox::NoInsertion);

	connect(name_combo_box,
			SIGNAL(activated(const QString &)),
			SLOT(NameSelected(const QString &)));

	connect(add_auxiliary_button,
			SIGNAL(clicked()),
			SLOT(AddAuxiliaryFont()));
}

void    CMainviewTreeFontDialog::SetNameComboBox(CAuxiliaryFontView *afv)
{
	auxiliary_font_view = afv;

	UpdateComboBox(0);
}

void	CMainviewTreeFontDialog::AcceptOperation()
{
	*value_pointer =
		string((const char *)(value_line_edit->text().local8Bit()));

	accept();
}

void	CMainviewTreeFontDialog::AddAuxiliaryFont()
{
	string	name((const char *)
						(name_combo_box->lineEdit()->text().local8Bit()));
	string	value((const char *)(value_line_edit->text().local8Bit()));

	if (auxiliary_font_view->AddAuxiliaryFont(name, value))
		UpdateComboBox(&name);
}

void	CMainviewTreeFontDialog::NameSelected(const QString &s)
{
	QListViewItem		*lvi =
		auxiliary_font_view->GetListView()->findItem(s, 0);

	if (lvi)
		SetValueLineEdit(lvi->text(1));
}

void	CMainviewTreeFontDialog::UpdateComboBox(string *s)
{
	name_combo_box->clear();

	COptionData		*d = auxiliary_font_view->GetOptionData();

	map<string, string>     *auxiliary_fonts = d->GetAuxiliaryFonts();

	if (!auxiliary_fonts || !auxiliary_fonts->size())
		return;

	map<string, string>::iterator   m_iterator = auxiliary_fonts->begin();

	while (m_iterator != auxiliary_fonts->end())
	{
		name_combo_box->insertItem((*m_iterator).first.c_str());

		m_iterator++;
	}

	if (s)
		name_combo_box->setEditText(s->c_str());
	else
		name_combo_box->clearEdit();
}

CColorSelectionWidget::CColorSelectionWidget(CBaseColorDialog *parent,
												int b, int iw, int ih)
	: QWidget(parent), base_color_dialog(parent), border_width(b),
		interior_width(iw), interior_height(ih), focus_flag(false)
{
	QSize	s(	interior_width + 2 * border_width,
				interior_height + 2 * border_width);

	setMinimumSize(s);
	setMaximumSize(s);

	setFocusPolicy(QWidget::StrongFocus);

	setPaletteBackgroundColor(QColor(0, 0, 0, QColor::Rgb));
}

void	CColorSelectionWidget::focusInEvent(QFocusEvent *e)
{
	focus_flag = true;

	base_color_dialog->SetupSpinboxConnections(
		CColorComponentSpinBox::ColorSelectionId);

	DrawBorder();
}

void	CColorSelectionWidget::focusOutEvent(QFocusEvent *e)
{
	focus_flag = false;

	DrawBorder();
}

void	CColorSelectionWidget::DrawBorder()
{
	QPainter	painter(this);

	painter.setPen(QColor(0, 0, 0, QColor::Rgb));

	for (int i=0; i<border_width; i++)
	{
		painter.drawLine (	0,
							i,
							interior_width + 2 * border_width - 1,
							i);

		painter.drawLine (	0,
							interior_height + 2 * border_width - 1 - i,
							interior_width + 2 * border_width - 1,
							interior_height + 2 * border_width - 1 - i);

		painter.drawLine (	i,
							border_width,
							i,
							interior_height + border_width - 1);

		painter.drawLine (	interior_width + 2 * border_width - 1 - i,
							border_width,
							interior_width + 2 * border_width - 1 - i,
							interior_height + border_width - 1);
	}

	if (focus_flag)
	{
		painter.setPen(QColor(255, 255, 255, QColor::Rgb));

		int		offset = border_width / 2;

		painter.drawLine (	offset,
							offset,
							interior_width + 2 * border_width - 1 - offset,
							offset);

		painter.drawLine (	offset,
							interior_height + 2 * border_width - 1 - offset,
							interior_width + 2 * border_width - 1 - offset,
							interior_height + 2 * border_width - 1 - offset);

		painter.drawLine (	offset,
							offset,
							offset,
							interior_height + 2 * border_width - 1 - offset);

		painter.drawLine (	interior_width + 2 * border_width - 1 - offset,
							offset,
							interior_width + 2 * border_width - 1 - offset,
							interior_height + 2 * border_width - 1 - offset);
	}
}

QPixmap		*CHueSaturationWidget::hue_saturation_pixmap;
int			CHueSaturationWidget::object_count = 0;

CHueSaturationWidget::CHueSaturationWidget(CBaseColorDialog *parent)
	: CColorSelectionWidget(parent, 5, 360, 256),
		current_hue(0), current_saturation(0), indicator_radius(16)
{
	if (!object_count)
	{
		hue_saturation_pixmap = new QPixmap(360, 256);

		QPainter	painter(hue_saturation_pixmap);

		for (int i=0; i<360; i++)
		{
			for (int j=0; j<256; j++)
			{
				painter.setPen(QColor(i, j, 255, QColor::Hsv));

				painter.drawPoint(i, 255 - j);
			}
		}

		object_count++;
	}
	else
		object_count++;
}

CHueSaturationWidget::~CHueSaturationWidget()
{
	object_count--;

	if (!object_count)
		delete hue_saturation_pixmap;
}

void	CHueSaturationWidget::SetColorSelection(int h, int s, int v)
{
	RestorePixels();

	current_hue			= h;
	current_saturation	= s;

	if (current_hue == -1)
		current_hue = 0;

	DrawIndicator();
}

void	CHueSaturationWidget::mousePressEvent(QMouseEvent *e)
{
	UpdateSelection(e->x(), e->y());

	e->accept();
}

void	CHueSaturationWidget::mouseMoveEvent(QMouseEvent *e)
{
	UpdateSelection(e->x(), e->y());

	e->accept();
}

void	CHueSaturationWidget::keyPressEvent(QKeyEvent *e)
{
	switch (e->key())
	{
		case Qt::Key_Left:
		switch (e->state())
		{
			case Qt::NoButton:
			UpdateSelection(current_hue + border_width - 1,
							255 - current_saturation + border_width);
			break;

			case Qt::ControlButton:
			UpdateSelection(current_hue + border_width - 10,
							255 - current_saturation + border_width);
			break;
		}
		break;

		case Qt::Key_Right:
		switch (e->state())
		{
			case Qt::NoButton:
			UpdateSelection(current_hue + border_width + 1,
							255 - current_saturation + border_width);
			break;

			case Qt::ControlButton:
			UpdateSelection(current_hue + border_width + 10,
							255 - current_saturation + border_width);
			break;
		}
		break;

		case Qt::Key_Up:
		switch (e->state())
		{
			case Qt::NoButton:
			UpdateSelection(current_hue + border_width,
							255 - current_saturation + border_width - 1);
			break;

			case Qt::ControlButton:
			UpdateSelection(current_hue + border_width,
							255 - current_saturation + border_width - 10);
			break;
		}
		break;

		case Qt::Key_Down:
		switch (e->state())
		{
			case Qt::NoButton:
			UpdateSelection(current_hue + border_width,
							255 - current_saturation + border_width + 1);
			break;

			case Qt::ControlButton:
			UpdateSelection(current_hue + border_width,
							255 - current_saturation + border_width + 10);
			break;
		}
		break;
	}

	e->accept();
}

void	CHueSaturationWidget::paintEvent(QPaintEvent *e)
{
	DrawBorder();

	QPainter	painter(this);

	painter.drawPixmap(border_width, border_width, *hue_saturation_pixmap);

	DrawIndicator();
}

void	CHueSaturationWidget::UpdateSelection(int x, int y)
{
	RestorePixels();

	current_hue = x - border_width;

	if (current_hue < 0)
		current_hue = 0;
	else if (current_hue > 359)
		current_hue = 359;

	int		current_saturation_inverted = y - border_width;

	if (current_saturation_inverted < 0)
		current_saturation_inverted = 0;
	else if (current_saturation_inverted > 255)
		current_saturation_inverted = 255;

	current_saturation = 255 - current_saturation_inverted;

	DrawIndicator();

	base_color_dialog->HueSaturationSelection(	current_hue,
												current_saturation);
}

void	CHueSaturationWidget::RestorePixels()
{
	QPainter	painter(this);

	int		h1;
	int		s1;
	int		h2;
	int		s2;

	int		current_saturation_inverted = 255 - current_saturation;

	if (current_hue > 0)
	{
		h1 = current_hue - 1;
		s1 = current_saturation_inverted - indicator_radius;
		h2 = current_hue - 1;
		s2 = current_saturation_inverted + indicator_radius;

		if (s1 < 0)
			s1 = 0;

		if (s2 > 255)
			s2 = 255;

		for (int i=s1; i<=s2; i++)
		{
			painter.setPen(QColor(h1, 255 - i, 255, QColor::Hsv));
			
			painter.drawPoint(h1 + border_width, i + border_width);
		}
	}

	if (current_hue < 359)
	{
		h1 = current_hue + 1;
		s1 = current_saturation_inverted - indicator_radius;
		h2 = current_hue + 1;
		s2 = current_saturation_inverted + indicator_radius;

		if (s1 < 0)
			s1 = 0;

		if (s2 > 255)
			s2 = 255;

		for (int i=s1; i<=s2; i++)
		{
			painter.setPen(QColor(h1, 255 - i, 255, QColor::Hsv));
			
			painter.drawPoint(h1 + border_width, i + border_width);
		}
	}

	if (current_saturation_inverted > 0)
	{
		h1 = current_hue - indicator_radius;
		s1 = current_saturation_inverted - 1;
		h2 = current_hue + indicator_radius;
		s2 = current_saturation_inverted - 1;

		if (h1 < 0)
			h1 = 0;

		if (h2 > 359)
			h2 = 359;

		for (int i=h1; i<=h2; i++)
		{
			painter.setPen(QColor(i, 255 - s1, 255, QColor::Hsv));
			
			painter.drawPoint(i + border_width, s1 + border_width);
		}
	}

	if (current_saturation_inverted < 255)
	{
		h1 = current_hue - indicator_radius;
		s1 = current_saturation_inverted + 1;
		h2 = current_hue + indicator_radius;
		s2 = current_saturation_inverted + 1;

		if (h1 < 0)
			h1 = 0;

		if (h2 > 359)
			h2 = 359;

		for (int i=h1; i<=h2; i++)
		{
			painter.setPen(QColor(i, 255 - s1, 255, QColor::Hsv));
			
			painter.drawPoint(i + border_width, s1 + border_width);
		}
	}
}

void	CHueSaturationWidget::DrawIndicator()
{
	QPainter	painter(this);

	painter.setPen(QColor(0, 0, 0, QColor::Rgb));

	int		h1;
	int		s1;
	int		h2;
	int		s2;

	int		current_saturation_inverted = 255 - current_saturation;

	if (current_hue > 0)
	{
		h1 = current_hue - 1;
		s1 = current_saturation_inverted - indicator_radius;
		h2 = current_hue - 1;
		s2 = current_saturation_inverted + indicator_radius;

		if (s1 < 0)
			s1 = 0;

		if (s2 > 255)
			s2 = 255;

		painter.drawLine (	h1 + border_width, s1 + border_width,
							h2 + border_width, s2 + border_width);
	}

	if (current_hue < 359)
	{
		h1 = current_hue + 1;
		s1 = current_saturation_inverted - indicator_radius;
		h2 = current_hue + 1;
		s2 = current_saturation_inverted + indicator_radius;

		if (s1 < 0)
			s1 = 0;

		if (s2 > 255)
			s2 = 255;

		painter.drawLine (	h1 + border_width, s1 + border_width,
							h2 + border_width, s2 + border_width);
	}

	if (current_saturation_inverted > 0)
	{
		h1 = current_hue - indicator_radius;
		s1 = current_saturation_inverted - 1;
		h2 = current_hue + indicator_radius;
		s2 = current_saturation_inverted - 1;

		if (h1 < 0)
			h1 = 0;

		if (h2 > 359)
			h2 = 359;

		painter.drawLine (	h1 + border_width, s1 + border_width,
							h2 + border_width, s2 + border_width);
	}

	if (current_saturation_inverted < 255)
	{
		h1 = current_hue - indicator_radius;
		s1 = current_saturation_inverted + 1;
		h2 = current_hue + indicator_radius;
		s2 = current_saturation_inverted + 1;

		if (h1 < 0)
			h1 = 0;

		if (h2 > 359)
			h2 = 359;

		painter.drawLine (	h1 + border_width, s1 + border_width,
							h2 + border_width, s2 + border_width);
	}
}

CValueWidget::CValueWidget(CBaseColorDialog *parent)
	: CColorSelectionWidget(parent, 5, 40, 256),
		current_hue(0), current_saturation(0), current_value(0)
{
	indicator_width = 6;

	indicator_pixel_count = indicator_width * indicator_width;

	indicator_x = new int[2 * indicator_pixel_count];
	indicator_y = new int[2 * indicator_pixel_count];

	int		index		= 0;
	int		y_offset	= 1 - indicator_width;
	int		i;
	int		j;

	for (i=1; i<=indicator_width; i++)
	{
		for (j=0; j<i; j++)
		{
			indicator_x[index]		= j;
			indicator_y[index++]	= y_offset;
		}

		y_offset++;
	}
	
	for (i=indicator_width-1; i>=1; i--)
	{
		for (j=0; j<i; j++)
		{
			indicator_x[index]		= j;
			indicator_y[index++]	= y_offset;
		}

		y_offset++;
	}

	y_offset = 1 - indicator_width;

	for (i=1; i<=indicator_width; i++)
	{
		for (j=0; j<i; j++)
		{
			indicator_x[index]		= interior_width - 1 - j;
			indicator_y[index++]	= y_offset;
		}

		y_offset++;
	}
	
	for (i=indicator_width-1; i>=1; i--)
	{
		for (j=0; j<i; j++)
		{
			indicator_x[index]		= interior_width - 1 - j;
			indicator_y[index++]	= y_offset;
		}

		y_offset++;
	}
}

CValueWidget::~CValueWidget()
{
	delete [] indicator_x;
	delete [] indicator_y;
}

void	CValueWidget::SetColorSelection(int h, int s, int v)
{
	current_hue			= h;
	current_saturation	= s;
	current_value		= v;

	DrawValuePixmap();
	DrawIndicator();
}

void	CValueWidget::mousePressEvent(QMouseEvent *e)
{
	UpdateSelection(e->y());

	e->accept();
}

void	CValueWidget::mouseMoveEvent(QMouseEvent *e)
{
	UpdateSelection(e->y());

	e->accept();
}

void	CValueWidget::keyPressEvent(QKeyEvent *e)
{
	switch (e->key())
	{
		case Qt::Key_Up:
		switch (e->state())
		{
			case Qt::NoButton:
			UpdateSelection(255 - current_value + border_width - 1);
			break;

			case Qt::ControlButton:
			UpdateSelection(255 - current_value + border_width - 10);
			break;
		}
		break;

		case Qt::Key_Down:
		switch (e->state())
		{
			case Qt::NoButton:
			UpdateSelection(255 - current_value + border_width + 1);
			break;

			case Qt::ControlButton:
			UpdateSelection(255 - current_value + border_width + 10);
			break;
		}
		break;
	}

	e->accept();
}

void	CValueWidget::paintEvent(QPaintEvent *e)
{
	DrawBorder();
	DrawValuePixmap();
	DrawIndicator();
}

void	CValueWidget::UpdateSelection(int z)
{
	int		current_value_inverted = z - border_width;

	if (current_value_inverted < 0)
		current_value_inverted = 0;
	else if (current_value_inverted > 255)
		current_value_inverted = 255;

	current_value = 255 - current_value_inverted;

	DrawValuePixmap();
	DrawIndicator();

	base_color_dialog->ValueSelection(current_value);
}

void	CValueWidget::DrawValuePixmap()
{
	QPainter	painter(this);

	for (int i=0; i<256; i++)
	{
		painter.setPen(QColor(	current_hue,
								current_saturation,
								255 - i,
								QColor::Hsv));

		painter.drawLine (	border_width,
							i + border_width,
							border_width + interior_width - 1,
							i + border_width);
	}
}

void	CValueWidget::DrawIndicator()
{
	QPainter	painter(this);
	int			i;

	painter.setPen(QColor(0, 0, 0, QColor::Rgb));

	for (i=0; i<indicator_pixel_count; i++)
	{
		int		y = 255 - current_value + indicator_y[i];

		if (y >= 0 && y <= 255)
			painter.drawPoint(	indicator_x[i] + border_width,
								y + border_width);
	}

	painter.setPen(QColor(255, 255, 255, QColor::Rgb));

	for (i=indicator_pixel_count; i<2*indicator_pixel_count; i++)
	{
		int		y = 255 - current_value + indicator_y[i];

		if (y >= 0 && y <= 255)
			painter.drawPoint(	indicator_x[i] + border_width,
								y + border_width);
	}
}

CColorComponentSpinBox::CColorComponentSpinBox(	ComponentId id,
												int minValue,
												int maxValue,
												int step,
												CBaseColorDialog *parent)
	: QSpinBox(minValue, maxValue, step, parent),
		base_color_dialog(parent), component_id(id)
{
	setFocusPolicy(QWidget::StrongFocus);
}

bool	CColorComponentSpinBox::eventFilter(QObject *obj, QEvent *ev)
{
	if (ev->type() == QEvent::FocusIn)
	{
		base_color_dialog->SetupSpinboxConnections(component_id);
		return false;
	}
	else if (ev->type() == QEvent::KeyPress)
	{
		QKeyEvent	*k = (QKeyEvent *)ev;

		if (k->key() == Qt::Key_Return)
		{
			interpretText();
			return true;
		}
	}

	return QSpinBox::eventFilter(obj, ev);
}

CColorDisplayWidget::CColorDisplayWidget(	QWidget *parent,
											int bw,
											const QString &s)
	: QWidget(parent), color(QColor(0, 0, 0, QColor::Rgb)),
		border_width(bw), size_string(s)
{
	setPaletteBackgroundColor(color);

	CalculateGeometry();
}

void	CColorDisplayWidget::DrawColor(const QColor &c)
{
	QPainter	painter(this);

	painter.fillRect(	border_width,
						border_width,
						size().width() - 2 * border_width,
						size().height() - 2 * border_width,
						c);

	color = c;
}

void	CColorDisplayWidget::DrawColor(CPaletteData::SColorSpecifier *cs)
{
	QColor	c;

	cs->SetupColor(c);

	DrawColor(c);
}

void    CColorDisplayWidget::CalculateGeometry()
{
	QFontInfo       fi(font());
	QFontMetrics    fm(font());

	int     pixel_size  = fi.pixelSize();
	int     text_height = fm.height();
	int     text_width;

	if (size_string.length())
		text_width  = fm.width(size_string);
	else
		text_width  = text_height;

	int     w = text_width + push_button_width_data[pixel_size] * 2;
	int     h = text_height + push_button_height_data[pixel_size] * 2;

	setMinimumSize(w, h);
}

void    CColorDisplayWidget::setFont(const QFont &f)
{
	QWidget::setFont(f);

	CalculateGeometry();
}

void	CColorDisplayWidget::paintEvent(QPaintEvent *e)
{
	DrawColor(color);
}

CBaseColorDialog::CBaseColorDialog(QWidget *parent)
	: CDialogFc(parent, 4, 3, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		), current_hue(0), current_saturation(0), current_value(0),
			current_connection_id(CColorComponentSpinBox::NullId)
{
	QFontInfo   fi(font());

	int     pixel_size = fi.pixelSize();

	int     s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1 = new QGridLayout(this, 3, 5, 0, s, "subgrid1");
	subgrid_layout2 = new QGridLayout(this, 1, 3, 0, s, "subgrid2");
	subgrid_layout3 = new QGridLayout(this, 1, 5, 0, s, "subgrid3");

	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 1);
	grid_layout->setRowStretch(3, 0);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 0);
	grid_layout->setColStretch(2, 1);

	subgrid_layout1->setRowStretch(0, 0);
	subgrid_layout1->setRowStretch(1, 0);
	subgrid_layout1->setRowStretch(2, 0);

	subgrid_layout1->setColStretch(0, 1);
	subgrid_layout1->setColStretch(1, 0);
	subgrid_layout1->setColStretch(2, 0);
	subgrid_layout1->setColStretch(3, 0);
	subgrid_layout1->setColStretch(4, 0);

	subgrid_layout2->setRowStretch(0, 0);

	subgrid_layout2->setColStretch(0, 0);
	subgrid_layout2->setColStretch(1, 1);
	subgrid_layout2->setColStretch(2, 0);

	subgrid_layout3->setRowStretch(0, 0);

	subgrid_layout3->setColStretch(0, 0);
	subgrid_layout3->setColStretch(1, 1);
	subgrid_layout3->setColStretch(2, 0);
	subgrid_layout3->setColStretch(3, 0);
	subgrid_layout3->setColStretch(4, 0);

	QString     button_size_string("######");
	QString     color_name_size_string("################");

	button_group			= new CHButtonGroupFc("Specifier Type", this);
	rgb_radio_button		= new CRadioButtonFc("Rgb", button_group);
	hsv_radio_button		= new CRadioButtonFc("Hsv", button_group);
	color_name_radio_button	= new CRadioButtonFc("Color Name", button_group);
	hue_saturation_widget	= new CHueSaturationWidget(this);
	value_widget			= new CValueWidget(this);
	color_display_widget	= new CColorDisplayWidget(this, 5, "######");
	red_label				= new CLabelFc("Red ", this);
	red_spinbox				= new CColorComponentSpinBox(
						CColorComponentSpinBox::RedId, 0, 255, 1, this);
	green_label				= new CLabelFc("Green ", this);
	green_spinbox			= new CColorComponentSpinBox(
						CColorComponentSpinBox::GreenId, 0, 255, 1, this);
	blue_label				= new CLabelFc("Blue ", this);
	blue_spinbox			= new CColorComponentSpinBox(
						CColorComponentSpinBox::BlueId, 0, 255, 1, this);
	hue_label				= new CLabelFc("Hue ", this);
	hue_spinbox				= new CColorComponentSpinBox(
						CColorComponentSpinBox::HueId, -1, 359, 1, this);
	saturation_label		= new CLabelFc("Saturation ", this);
	saturation_spinbox		= new CColorComponentSpinBox(
						CColorComponentSpinBox::SaturationId, 0, 255, 1, this);
	value_label				= new CLabelFc("Value ", this);
	value_spinbox			= new CColorComponentSpinBox(
						CColorComponentSpinBox::ValueId, 0, 255, 1, this);
	color_name_label		= new CLabelFc("Color Name", this);
	color_name_line_edit	= new CLineEditFc(this, color_name_size_string);
	display_button			= new CPushButtonFc("Display", this);
	name_label				= new CLabelFc("Name", this);
	ok_button		= new CPushButtonFc("Ok", this, button_size_string);
	cancel_button	= new CPushButtonFc("Cancel", this, button_size_string);

	red_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
	green_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
	blue_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
	hue_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
	saturation_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
	value_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
	color_name_label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
	name_label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	display_button->setDefault(true);

	subgrid_layout1->addMultiCellWidget(color_display_widget,	0, 2, 0, 0);
	subgrid_layout1->addWidget(red_label,						0, 1);
	subgrid_layout1->addWidget(red_spinbox,						0, 2);
	subgrid_layout1->addWidget(green_label,						1, 1);
	subgrid_layout1->addWidget(green_spinbox,					1, 2);
	subgrid_layout1->addWidget(blue_label,						2, 1);
	subgrid_layout1->addWidget(blue_spinbox,					2, 2);
	subgrid_layout1->addWidget(hue_label,						0, 3);
	subgrid_layout1->addWidget(hue_spinbox,						0, 4);
	subgrid_layout1->addWidget(saturation_label,				1, 3);
	subgrid_layout1->addWidget(saturation_spinbox,				1, 4);
	subgrid_layout1->addWidget(value_label,						2, 3);
	subgrid_layout1->addWidget(value_spinbox,					2, 4);

	subgrid_layout2->addWidget(color_name_label,		0, 0, Qt::AlignTop);
	subgrid_layout2->addWidget(color_name_line_edit,	0, 1, Qt::AlignTop);
	subgrid_layout2->addWidget(display_button,			0, 2, Qt::AlignTop);

	subgrid_layout3->addWidget(name_label,						0, 0);
	subgrid_layout3->addWidget(ok_button,						0, 3);
	subgrid_layout3->addWidget(cancel_button,					0, 4);

	grid_layout->addMultiCellWidget(hue_saturation_widget,		0, 2, 0, 0);
	grid_layout->addMultiCellWidget(value_widget,				0, 2, 1, 1);
	grid_layout->addLayout(subgrid_layout1,						0, 2);
	grid_layout->addLayout(subgrid_layout2,						1, 2);
	grid_layout->addWidget(button_group,	2, 2, Qt::AlignLeft | Qt::AlignTop);
	grid_layout->addMultiCellLayout(subgrid_layout3,			3, 3, 0, 2);

	connect(ok_button, SIGNAL(clicked()), SLOT(AcceptOperation()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(reject()));

	connect(display_button,
			SIGNAL(clicked()),
			SLOT(DisplayColorName()));
}

void	CBaseColorDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();

	setMaximumHeight(height());
}

void	CBaseColorDialog::SetColorValue(CPaletteData::SColorSpecifier *v)
{
	SetupSpinboxConnections(CColorComponentSpinBox::NullId);

	if (!v)
	{
		current_hue			= -1;
		current_saturation	= 0;
		current_value		= 0;

		hue_spinbox->setValue(-1);
		saturation_spinbox->setValue(0);
		value_spinbox->setValue(0);
		SetRgbColorComponents();
		button_group->setButton(0);
		rgb_radio_button->setFocus();
		return;
	}

	switch (v->GetSpecifierType())
	{
		case CPaletteData::RgbSpecifier:
		{
			CPaletteData::SRgbSpecifier		*s =
				dynamic_cast<CPaletteData::SRgbSpecifier *>(v);

			red_spinbox->setValue(s->red);
			green_spinbox->setValue(s->green);
			blue_spinbox->setValue(s->blue);
			color_name_line_edit->clear();
			SetHsvColorComponents();
			button_group->setButton(0);
			rgb_radio_button->setFocus();
		}
		break;

		case CPaletteData::HsvSpecifier:
		{
			CPaletteData::SHsvSpecifier		*s =
				dynamic_cast<CPaletteData::SHsvSpecifier *>(v);

			current_hue			= s->hue;
			current_saturation	= s->saturation;
			current_value		= s->value;

			hue_spinbox->setValue(s->hue);
			saturation_spinbox->setValue(s->saturation);
			value_spinbox->setValue(s->value);
			color_name_line_edit->clear();
			SetRgbColorComponents();
			button_group->setButton(1);
			hsv_radio_button->setFocus();
		}
		break;

		case CPaletteData::ColorNameSpecifier:
		{
			CPaletteData::SColorNameSpecifier	*s =
				dynamic_cast<CPaletteData::SColorNameSpecifier *>(v);

			color_name_line_edit->setText(s->color_name.c_str());

			QColor	c;

			c.setNamedColor(color_name_line_edit->text());

			red_spinbox->setValue(c.red());
			green_spinbox->setValue(c.green());
			blue_spinbox->setValue(c.blue());
			SetHsvColorComponents();
			button_group->setButton(2);
			color_name_radio_button->setFocus();
		}
		break;
	}
}

void	CBaseColorDialog::HueSaturationSelection(int h, int s)
{
	current_hue			= h;
	current_saturation	= s;

	SetColorComponents();

	value_widget->SetColorSelection(current_hue,
									current_saturation,
									current_value);
}

void	CBaseColorDialog::ValueSelection(int v)
{
	current_value = v;

	SetColorComponents();

	hue_saturation_widget->SetColorSelection(	current_hue,
												current_saturation,
												current_value);
}

void    CBaseColorDialog::SetWidgetColorSelections()
{
	hue_saturation_widget->SetColorSelection(	current_hue,
												current_saturation,
												current_value);

	value_widget->SetColorSelection(current_hue,
									current_saturation,
									current_value);
}

void	CBaseColorDialog::SetColorComponents()
{
	hue_spinbox->setValue(current_hue);
	saturation_spinbox->setValue(current_saturation);
	value_spinbox->setValue(current_value);

	QColor	c(current_hue, current_saturation, current_value, QColor::Hsv);

	red_spinbox->setValue(c.red());
	green_spinbox->setValue(c.green());
	blue_spinbox->setValue(c.blue());

	color_display_widget->DrawColor(c);
}

void	CBaseColorDialog::SetRgbColorComponents()
{
	QColor	c(current_hue, current_saturation, current_value, QColor::Hsv);

	red_spinbox->setValue(c.red());
	green_spinbox->setValue(c.green());
	blue_spinbox->setValue(c.blue());

	color_display_widget->DrawColor(c);

	SetWidgetColorSelections();
}

void	CBaseColorDialog::SetHsvColorComponents()
{
	QColor	c(	red_spinbox->value(),
				green_spinbox->value(),
				blue_spinbox->value(),
				QColor::Rgb);

	c.hsv(&current_hue, &current_saturation, &current_value);

	hue_spinbox->setValue(current_hue);
	saturation_spinbox->setValue(current_saturation);
	value_spinbox->setValue(current_value);

	color_display_widget->DrawColor(c);

	SetWidgetColorSelections();
}

void    CBaseColorDialog::SetupSpinboxConnections(
										CColorComponentSpinBox::ComponentId id)
{
	switch (current_connection_id)
	{
		case CColorComponentSpinBox::RedId:
		disconnect(	red_spinbox,
					SIGNAL(valueChanged(int)),
					this,
					SLOT(RedChanged(int)));
		break;

		case CColorComponentSpinBox::GreenId:
		disconnect(	green_spinbox,
					SIGNAL(valueChanged(int)),
					this,
					SLOT(GreenChanged(int)));
		break;

		case CColorComponentSpinBox::BlueId:
		disconnect(	blue_spinbox,
					SIGNAL(valueChanged(int)),
					this,
					SLOT(BlueChanged(int)));
		break;

		case CColorComponentSpinBox::HueId:
		disconnect(	hue_spinbox,
					SIGNAL(valueChanged(int)),
					this,
					SLOT(HueChanged(int)));
		break;

		case CColorComponentSpinBox::SaturationId:
		disconnect(	saturation_spinbox,
					SIGNAL(valueChanged(int)),
					this,
					SLOT(SaturationChanged(int)));
		break;

		case CColorComponentSpinBox::ValueId:
		disconnect(	value_spinbox,
					SIGNAL(valueChanged(int)),
					this,
					SLOT(ValueChanged(int)));
		break;
	}

	switch (id)
	{
		case CColorComponentSpinBox::ColorSelectionId:
		if (color_name_radio_button->isChecked())
			button_group->setButton(1);
		break;

		case CColorComponentSpinBox::ColorNameId:
		button_group->setButton(2);
		break;

		case CColorComponentSpinBox::RedId:
		button_group->setButton(0);
		connect(red_spinbox,
				SIGNAL(valueChanged(int)),
				SLOT(RedChanged(int)));
		break;

		case CColorComponentSpinBox::GreenId:
		button_group->setButton(0);
		connect(green_spinbox,
				SIGNAL(valueChanged(int)),
				SLOT(GreenChanged(int)));
		break;

		case CColorComponentSpinBox::BlueId:
		button_group->setButton(0);
		connect(blue_spinbox,
				SIGNAL(valueChanged(int)),
				SLOT(BlueChanged(int)));
		break;

		case CColorComponentSpinBox::HueId:
		button_group->setButton(1);
		connect(hue_spinbox,
				SIGNAL(valueChanged(int)),
				SLOT(HueChanged(int)));
		break;

		case CColorComponentSpinBox::SaturationId:
		button_group->setButton(1);
		connect(saturation_spinbox,
				SIGNAL(valueChanged(int)),
				SLOT(SaturationChanged(int)));
		break;

		case CColorComponentSpinBox::ValueId:
		button_group->setButton(1);
		connect(value_spinbox,
				SIGNAL(valueChanged(int)),
				SLOT(ValueChanged(int)));
		break;
	}

	current_connection_id = id;
}

void    CBaseColorDialog::RedChanged(int r)
{
	SetHsvColorComponents();
}

void    CBaseColorDialog::GreenChanged(int g)
{
	SetHsvColorComponents();
}

void    CBaseColorDialog::BlueChanged(int b)
{
	SetHsvColorComponents();
}

void    CBaseColorDialog::HueChanged(int h)
{
	current_hue = h;

	SetRgbColorComponents();
}

void    CBaseColorDialog::SaturationChanged(int s)
{
	current_saturation = s;

	SetRgbColorComponents();
}

void    CBaseColorDialog::ValueChanged(int v)
{
	current_value = v;

	SetRgbColorComponents();
}

void    CBaseColorDialog::DisplayColorName()
{
	SetupSpinboxConnections(CColorComponentSpinBox::ColorNameId);

	QColor	c;

	c.setNamedColor(color_name_line_edit->text());

	red_spinbox->setValue(c.red());
	green_spinbox->setValue(c.green());
	blue_spinbox->setValue(c.blue());

	SetHsvColorComponents();

	color_name_line_edit->setFocus();
}

void    CBaseColorDialog::StoreColorValue(CPaletteData::SColorSpecifier **p)
{
	if (rgb_radio_button->isChecked())
	{
		*p = new CPaletteData::SRgbSpecifier(	red_spinbox->value(),
												green_spinbox->value(),
												blue_spinbox->value());
	}
	else if (hsv_radio_button->isChecked())
	{
		*p = new CPaletteData::SHsvSpecifier(	hue_spinbox->value(),
												saturation_spinbox->value(),
												value_spinbox->value());
	}
	else if (color_name_radio_button->isChecked())
	{
		*p = new CPaletteData::SColorNameSpecifier(
				(const char *)(color_name_line_edit->text().local8Bit()));
	}
}

CAuxiliaryColorDialog::CAuxiliaryColorDialog(QWidget *parent)
	: CBaseColorDialog(parent)
{
	name_line_edit	= new CLineEditFc(this, 
										OptionDialog::auxiliary_string_size);

	subgrid_layout3->addWidget(name_line_edit,		0, 1);

	connect(name_line_edit,
			SIGNAL(textChanged(const QString &)),
			SLOT(NameChanged(const QString &)));
}

void    CAuxiliaryColorDialog::SetNameLineEdit(const QString &n)
{
	name_line_edit->setText(n);
	name_changed_flag = false;
}

void	CAuxiliaryColorDialog::AcceptOperation()
{
	*name_pointer =
		string((const char *)(name_line_edit->text().local8Bit()));

	value_changed_flag = true;

	StoreColorValue(value_pointer);

	accept();
}

void    CAuxiliaryColorDialog::NameChanged(const QString &s)
{
	name_changed_flag = true;
}

CPaletteColorDialog::CPaletteColorDialog(QWidget *parent)
	: CBaseColorDialog(parent)
{
	name_combo_box			= new CComboBoxFc(true, this, "0123456789");
	add_auxiliary_button	= new CPushButtonFc("Add Auxiliary", this);

	subgrid_layout3->addWidget(name_combo_box,				0, 1);
	subgrid_layout3->addWidget(add_auxiliary_button,		0, 2);

	name_combo_box->setInsertionPolicy(QComboBox::NoInsertion);

	connect(name_combo_box,
			SIGNAL(activated(const QString &)),
			SLOT(NameSelected(const QString &)));

	connect(add_auxiliary_button,
			SIGNAL(clicked()),
			SLOT(AddAuxiliaryColor()));
}

void    CPaletteColorDialog::SetNameComboBox(CAuxiliaryColorView *acv)
{
	auxiliary_color_view = acv;

	UpdateComboBox(0);
}

void	CPaletteColorDialog::AcceptOperation()
{
	StoreColorValue(value_pointer);

	accept();
}

void	CPaletteColorDialog::AddAuxiliaryColor()
{
	string	name((const char *)
						(name_combo_box->lineEdit()->text().local8Bit()));

	CPaletteData::SColorSpecifier   *value;

	StoreColorValue(&value);

	if (auxiliary_color_view->AddAuxiliaryColor(name, value))
		UpdateComboBox(&name);
	else
		delete value;
}

void	CPaletteColorDialog::NameSelected(const QString &s)
{
	QListViewItem		*lvi =
		auxiliary_color_view->GetListView()->findItem(s, 0);

	if (lvi)
	{
		COptionData		*d = auxiliary_color_view->GetOptionData();

		map<string, CPaletteData::SColorSpecifier *>	*auxiliary_colors =
													d->GetAuxiliaryColors();

		string		name((const char *)(s.local8Bit()));

		map<string, CPaletteData::SColorSpecifier *>::iterator  m_iterator =
			auxiliary_colors->find(name);

		SetColorValue((*m_iterator).second);
	}
}

void	CPaletteColorDialog::UpdateComboBox(string *s)
{
	name_combo_box->clear();

	COptionData		*d = auxiliary_color_view->GetOptionData();

	map<string, CPaletteData::SColorSpecifier *>	*auxiliary_colors =
													d->GetAuxiliaryColors();

	if (!auxiliary_colors || !auxiliary_colors->size())
		return;

	map<string, CPaletteData::SColorSpecifier *>::iterator   m_iterator =
													auxiliary_colors->begin();

	while (m_iterator != auxiliary_colors->end())
	{
		name_combo_box->insertItem((*m_iterator).first.c_str());

		m_iterator++;
	}

	if (s)
		name_combo_box->setEditText(s->c_str());
	else
		name_combo_box->clearEdit();
}

CPaletteDemoWidget::CPaletteDemoWidget(QWidget *parent)
	: CFrameFc(parent, 6, 2)
{
	QString		hypertext1;
	QString		hypertext2;

	hypertext1 += "<html>\n<body>\n";
	hypertext1 += "<p>\n";
	hypertext1 += "Active/Inactive palette color group\n";
	hypertext1 += "</p>\n";
	hypertext1 += "<ol>\n";
	hypertext1 += "<li><a href=\"#hyperlink1\">Hyperlink 1</a>\n";
	hypertext1 += "<li><a href=\"#hyperlink2\">Hyperlink 2</a>\n";
	hypertext1 += "</ol>\n";
	hypertext1 += "</body>\n</html>\n";

	hypertext2 += "<html>\n<body>\n";
	hypertext2 += "<p>\n";
	hypertext2 += "Disabled palette color group\n";
	hypertext2 += "</p>\n";
	hypertext2 += "<ol>\n";
	hypertext2 += "<li><a href=\"#hyperlink1\">Hyperlink 1</a>\n";
	hypertext2 += "<li><a href=\"#hyperlink2\">Hyperlink 2</a>\n";
	hypertext2 += "</ol>\n";
	hypertext2 += "</body>\n</html>\n";

	QString		active_inactive_text("Active/Inactive");
	QString		disabled_text("Disabled");

	active_raised_label		= new CLabelFc(active_inactive_text, this);
	active_sunken_label		= new CLabelFc(active_inactive_text, this);
	active_button			= new CPushButtonFc(active_inactive_text, this);
	active_line_edit		= new CLineEditFc(this, active_inactive_text);
	active_text_edit		= new QTextEdit(hypertext1, QString::null, this);
	disabled_raised_label	= new CLabelFc(disabled_text, this);
	disabled_sunken_label	= new CLabelFc(disabled_text, this);
	disabled_button			= new CPushButtonFc(disabled_text, this);
	disabled_line_edit		= new CLineEditFc(this, disabled_text);
	disabled_text_edit		= new QTextEdit(hypertext2, QString::null, this);

	setFrameStyle(QFrame::StyledPanel | QFrame::Raised);

	active_raised_label->setFrameStyle(QFrame::StyledPanel | QFrame::Raised);
	active_sunken_label->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);
	disabled_raised_label->setFrameStyle(QFrame::StyledPanel | QFrame::Raised);
	disabled_sunken_label->setFrameStyle(QFrame::StyledPanel | QFrame::Sunken);

	active_raised_label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
	active_sunken_label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
	disabled_raised_label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
	disabled_sunken_label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	active_line_edit->setText(active_inactive_text);
	disabled_line_edit->setText(disabled_text);

	disabled_raised_label->setEnabled(false);
	disabled_sunken_label->setEnabled(false);
	disabled_button->setEnabled(false);
	disabled_line_edit->setEnabled(false);
	disabled_text_edit->setEnabled(false);
}

void	CPaletteDemoWidget::SetDemoPalette(CPaletteData *p)
{
	QPalette    demo_palette;

	p->SetupPalette(demo_palette);

	setPalette(demo_palette);
}

CListViewPaletteDemo::CListViewPaletteDemo(QWidget *parent)
	: CPaletteDemoWidget(parent)
{
	QString		hypertext1;
	QString		hypertext2;

	hypertext1 += "<html>\n<body>\n";
	hypertext1 += "<p>\n";
	hypertext1 += "Active/Inactive\n";
	hypertext1 += "<a href=\"#hyperlink1\">Hyperlink</a>\n";
	hypertext1 += "</p>\n";
	hypertext1 += "</body>\n</html>\n";

	hypertext2 += "<html>\n<body>\n";
	hypertext2 += "<p>\n";
	hypertext2 += "Disabled\n";
	hypertext2 += "<a href=\"#hyperlink1\">Hyperlink</a>\n";
	hypertext2 += "</p>\n";
	hypertext2 += "</body>\n</html>\n";

	active_text_edit->setText(hypertext1);
	disabled_text_edit->setText(hypertext2);
}

void	CListViewPaletteDemo::CalculateGeometry()
{
	QFontInfo       fi(font());

	int     pixel_size = fi.pixelSize();

	int     m = grid_layout_margin_data[pixel_size];
	int     s = grid_layout_spacing_data[pixel_size];

	QSize	s0 = active_raised_label->size();
	QSize	s1 = active_sunken_label->size();
	QSize	s2 = active_button->size();
	QSize	s3 = active_line_edit->size();

	active_text_edit->setMinimumSize(	s0.width(),
										s0.height() + s + s1.height());

	active_text_edit->viewport()->setMinimumSize(
										s0.width(),
										s0.height() + s + s1.height());

	QSize	s4 = active_text_edit->size();

	QSize	s5 = disabled_raised_label->size();
	QSize	s6 = disabled_sunken_label->size();
	QSize	s7 = disabled_button->size();
	QSize	s8 = disabled_line_edit->size();

	disabled_text_edit->setMinimumSize(	s5.width(),
										s5.height() + s + s6.height());

	disabled_text_edit->viewport()->setMinimumSize(
										s5.width(),
										s5.height() + s + s6.height());

	QSize	s9 = disabled_text_edit->size();

	int		x0 = m;
	int		x1 = x0 + s0.width() + s;
	int		x2 = x1 + s2.width() + s;
	int		x3 = x2 + s4.width() + s;
	int		x4 = x3 + s5.width() + s;
	int		x5 = x4 + s7.width() + s;

	int		y0 = m;
	int		y1 = y0 + s0.height() + s;

	int		w = x5 + s9.width() + m;
	int		h = y1 + s1.height() + m;

	setMinimumSize(w, h);

	active_raised_label->move(		x0, y0);
	active_sunken_label->move(		x0, y1);
	active_button->move(			x1, y0);
	active_line_edit->move(			x1, y1);
	active_text_edit->move(			x2, y0);
	disabled_raised_label->move(	x3, y0);
	disabled_sunken_label->move(	x3, y1);
	disabled_button->move(			x4, y0);
	disabled_line_edit->move(		x4, y1);
	disabled_text_edit->move(		x5, y0);
}

CDialogPaletteDemo::CDialogPaletteDemo(QWidget *parent)
	: CPaletteDemoWidget(parent)
{
	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 1);
	grid_layout->setRowStretch(3, 0);
	grid_layout->setRowStretch(4, 0);
	grid_layout->setRowStretch(5, 1);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 0);

	grid_layout->addWidget(active_raised_label,					0, 0);
	grid_layout->addWidget(active_sunken_label,					0, 1);
	grid_layout->addWidget(active_button,						1, 0);
	grid_layout->addWidget(active_line_edit,					1, 1);
	grid_layout->addMultiCellWidget(active_text_edit,			2, 2, 0, 1);
	grid_layout->addWidget(disabled_raised_label,				3, 0);
	grid_layout->addWidget(disabled_sunken_label,				3, 1);
	grid_layout->addWidget(disabled_button,						4, 0);
	grid_layout->addWidget(disabled_line_edit,					4, 1);
	grid_layout->addMultiCellWidget(disabled_text_edit,			5, 5, 0, 1);

	connect(active_button,
			SIGNAL(clicked()),
			SLOT(InactiveDisplay()));
}

void	CDialogPaletteDemo::InactiveDisplay()
{
	CInformationDialog	*ifd = OptionDialog::information_dialog;

	ifd->SetTitleAndMessageText("Inactive Display",
								"Inactive palette\ndisplayed in dialog.");

	ifd->ExecuteDialog();
}

CColorResetButton::CColorResetButton(QWidget *parent, int bw)
	: QPushButton(parent), border_width(bw)
{
	CalculateGeometry();

	SetColor(QColor(0, 0, 0, QColor::Rgb));
}

void	CColorResetButton::SetColor(const QColor &c)
{
	color = c;

	int		pixmap_edge_length = size().height() - 2 * border_width;

	QPixmap		button_pixmap(pixmap_edge_length, pixmap_edge_length);

	button_pixmap.fill(color);

	setPixmap(button_pixmap);
}

void	CColorResetButton::CalculateGeometry()
{
	QFontInfo       fi(font());
	QFontMetrics    fm(font());

	int     pixel_size  = fi.pixelSize();
	int     text_height = fm.height();

	int     h = text_height + push_button_height_data[pixel_size] * 2;

	setMinimumSize(h, h);
	setMaximumSize(h, h);
}

void	CColorResetButton::setFont(const QFont &f)
{
	QPushButton::setFont(f);

	CalculateGeometry();

	SetColor(QColor(color));
}

CColorRoleWidget::CColorRoleWidget(	CColorGroupWidget *parent,
									QColorGroup::ColorRole id)
	: CFrameFc(parent, 1, 4), color_group_widget(parent), role_id(id),
		color_specifier(0)
{
	grid_layout->setRowStretch(0, 0);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 0);
	grid_layout->setColStretch(2, 1);
	grid_layout->setColStretch(3, 0);

	activate_check_box		= new QCheckBox(this);
	color_reset_button		= new CColorResetButton(this, 5);
	color_selection_button	= new CPushButtonFc(
		CPaletteData::color_role_to_string_map[role_id].c_str(), this);
	color_display_widget	= new CColorDisplayWidget(this, 2, "###");

	grid_layout->addWidget(activate_check_box,			0, 0);
	grid_layout->addWidget(color_reset_button,			0, 1);
	grid_layout->addWidget(color_selection_button,		0, 2);
	grid_layout->addWidget(color_display_widget,		0, 3);

	connect(color_reset_button,
			SIGNAL(clicked()),
			SLOT(ResetColor()));

	connect(color_selection_button,
			SIGNAL(clicked()),
			SLOT(SelectAndActivateColor()));

	connect(activate_check_box,
			SIGNAL(toggled(bool)),
			SLOT(ActivateToggled(bool)));
}

CColorRoleWidget::~CColorRoleWidget()
{
	delete color_specifier;
}

CAuxiliaryColorView     *CColorRoleWidget::GetAuxiliaryColorView()
{
	return color_group_widget->GetAuxiliaryColorView();
}

void    CColorRoleWidget::InitializeColorRole()
{
	delete color_specifier;

	color_specifier = 0;

	SetActivateIndicator(false);
}

void    CColorRoleWidget::SetPrecalculatedColor(const QColor &c)
{
	precalculate_color = c;

	color_reset_button->SetColor(c);

	if (!activate_check_box->isChecked())
		color_display_widget->DrawColor(c);
}

void    CColorRoleWidget::SetColorSpecifier(CPaletteData::SColorSpecifier *cs)
{
	delete color_specifier;

	color_specifier = cs->clone();

	SetActivateIndicator(true);

	color_display_widget->DrawColor(color_specifier);
}

void    CColorRoleWidget::SetActivateIndicator(bool a)
{
	disconnect(	activate_check_box,
				SIGNAL(toggled(bool)),
				this,
				SLOT(ActivateToggled(bool)));

	activate_check_box->setChecked(a);

	connect(activate_check_box,
			SIGNAL(toggled(bool)),
			SLOT(ActivateToggled(bool)));
}

void    CColorRoleWidget::ResetColor()
{
	delete color_specifier;

	color_specifier =
		new CPaletteData::SRgbSpecifier(
			precalculate_color.red(),
			precalculate_color.green(),
			precalculate_color.blue());

	SetActivateIndicator(true);

	color_display_widget->DrawColor(color_specifier);

	color_group_widget->StoreColorSpecifier(role_id,
											color_specifier->clone());
}

void    CColorRoleWidget::SelectAndActivateColor()
{
	CPaletteColorDialog		*pcd = OptionDialog::palette_color_dialog;

	CPaletteData::SColorSpecifier   *value;

	pcd->SetTitleAndResize("Select Group/Role Color");
	pcd->SetValuePointer(&value);
	pcd->SetNameComboBox(GetAuxiliaryColorView());
	pcd->SetColorValue(color_specifier);

	if (pcd->exec() == QDialog::Accepted)
	{
		delete color_specifier;

		color_specifier = value;

		SetActivateIndicator(true);

		color_display_widget->DrawColor(color_specifier);

		color_group_widget->StoreColorSpecifier(role_id,
												color_specifier->clone());
	}
}

void    CColorRoleWidget::ActivateToggled(bool on)
{
	if (on)
	{
		if (!color_specifier)
			color_specifier =
				new CPaletteData::SRgbSpecifier(
					precalculate_color.red(),
					precalculate_color.green(),
					precalculate_color.blue());

		color_display_widget->DrawColor(color_specifier);

		color_group_widget->StoreColorSpecifier(role_id,
												color_specifier->clone());
	}
	else
	{
		color_display_widget->DrawColor(precalculate_color);

		color_group_widget->StoreColorSpecifier(role_id, 0);
	}
}

CColorGroupWidget::CColorGroupWidget(	QWidget *parent,
										CBasePaletteDialog *bpd,
										QPalette::ColorGroup id)
	: CFrameFc(parent, 8, 2), base_palette_dialog(bpd), group_id(id)
{
	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);
	grid_layout->setRowStretch(3, 0);
	grid_layout->setRowStretch(4, 0);
	grid_layout->setRowStretch(5, 0);
	grid_layout->setRowStretch(6, 0);
	grid_layout->setRowStretch(7, 0);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 0);

	background_color_role_widget		= new CColorRoleWidget(this,
												QColorGroup::Background);
	foreground_color_role_widget		= new CColorRoleWidget(this,
												QColorGroup::Foreground);
	base_color_role_widget				= new CColorRoleWidget(this,
												QColorGroup::Base);
	text_color_role_widget				= new CColorRoleWidget(this,
												QColorGroup::Text);
	button_color_role_widget			= new CColorRoleWidget(this,
												QColorGroup::Button);
	button_text_color_role_widget		= new CColorRoleWidget(this,
												QColorGroup::ButtonText);
	light_color_role_widget				= new CColorRoleWidget(this,
												QColorGroup::Light);
	midlight_color_role_widget			= new CColorRoleWidget(this,
												QColorGroup::Midlight);
	dark_color_role_widget				= new CColorRoleWidget(this,
												QColorGroup::Dark);
	mid_color_role_widget				= new CColorRoleWidget(this,
												QColorGroup::Mid);
	shadow_color_role_widget			= new CColorRoleWidget(this,
												QColorGroup::Shadow);
	highlight_color_role_widget			= new CColorRoleWidget(this,
												QColorGroup::Highlight);
	highlighted_text_color_role_widget	= new CColorRoleWidget(this,
												QColorGroup::HighlightedText);
	bright_text_color_role_widget		= new CColorRoleWidget(this,
												QColorGroup::BrightText);
	link_color_role_widget				= new CColorRoleWidget(this,
												QColorGroup::Link);
	link_visited_color_role_widget		= new CColorRoleWidget(this,
												QColorGroup::LinkVisited);

	grid_layout->addWidget(background_color_role_widget,		0, 0);
	grid_layout->addWidget(foreground_color_role_widget,		1, 0);
	grid_layout->addWidget(base_color_role_widget,				2, 0);
	grid_layout->addWidget(text_color_role_widget,				3, 0);
	grid_layout->addWidget(button_color_role_widget,			4, 0);
	grid_layout->addWidget(button_text_color_role_widget,		5, 0);
	grid_layout->addWidget(light_color_role_widget,				6, 0);
	grid_layout->addWidget(midlight_color_role_widget,			7, 0);
	grid_layout->addWidget(dark_color_role_widget,				0, 1);
	grid_layout->addWidget(mid_color_role_widget,				1, 1);
	grid_layout->addWidget(shadow_color_role_widget,			2, 1);
	grid_layout->addWidget(highlight_color_role_widget,			3, 1);
	grid_layout->addWidget(highlighted_text_color_role_widget,	4, 1);
	grid_layout->addWidget(bright_text_color_role_widget,		5, 1);
	grid_layout->addWidget(link_color_role_widget,				6, 1);
	grid_layout->addWidget(link_visited_color_role_widget,		7, 1);
}

CAuxiliaryColorView     *CColorGroupWidget::GetAuxiliaryColorView()
{
	return base_palette_dialog->GetAuxiliaryColorView();
}

void    CColorGroupWidget::InitializeColorGroup()
{
	background_color_role_widget->InitializeColorRole();
	foreground_color_role_widget->InitializeColorRole();
	base_color_role_widget->InitializeColorRole();
	text_color_role_widget->InitializeColorRole();
	button_color_role_widget->InitializeColorRole();
	button_text_color_role_widget->InitializeColorRole();
	light_color_role_widget->InitializeColorRole();
	midlight_color_role_widget->InitializeColorRole();
	dark_color_role_widget->InitializeColorRole();
	mid_color_role_widget->InitializeColorRole();
	shadow_color_role_widget->InitializeColorRole();
	highlight_color_role_widget->InitializeColorRole();
	highlighted_text_color_role_widget->InitializeColorRole();
	bright_text_color_role_widget->InitializeColorRole();
	link_color_role_widget->InitializeColorRole();
	link_visited_color_role_widget->InitializeColorRole();
}

void    CColorGroupWidget::SetPrecalculatedColor(const QPalette &p)
{
	background_color_role_widget->SetPrecalculatedColor(
		p.color(group_id, background_color_role_widget->GetRoleId()));
	foreground_color_role_widget->SetPrecalculatedColor(
		p.color(group_id, foreground_color_role_widget->GetRoleId()));
	base_color_role_widget->SetPrecalculatedColor(
		p.color(group_id, base_color_role_widget->GetRoleId()));
	text_color_role_widget->SetPrecalculatedColor(
		p.color(group_id, text_color_role_widget->GetRoleId()));
	button_color_role_widget->SetPrecalculatedColor(
		p.color(group_id, button_color_role_widget->GetRoleId()));
	button_text_color_role_widget->SetPrecalculatedColor(
		p.color(group_id, button_text_color_role_widget->GetRoleId()));
	light_color_role_widget->SetPrecalculatedColor(
		p.color(group_id, light_color_role_widget->GetRoleId()));
	midlight_color_role_widget->SetPrecalculatedColor(
		p.color(group_id, midlight_color_role_widget->GetRoleId()));
	dark_color_role_widget->SetPrecalculatedColor(
		p.color(group_id, dark_color_role_widget->GetRoleId()));
	mid_color_role_widget->SetPrecalculatedColor(
		p.color(group_id, mid_color_role_widget->GetRoleId()));
	shadow_color_role_widget->SetPrecalculatedColor(
		p.color(group_id, shadow_color_role_widget->GetRoleId()));
	highlight_color_role_widget->SetPrecalculatedColor(
		p.color(group_id, highlight_color_role_widget->GetRoleId()));
	highlighted_text_color_role_widget->SetPrecalculatedColor(
		p.color(group_id, highlighted_text_color_role_widget->GetRoleId()));
	bright_text_color_role_widget->SetPrecalculatedColor(
		p.color(group_id, bright_text_color_role_widget->GetRoleId()));
	link_color_role_widget->SetPrecalculatedColor(
		p.color(group_id, link_color_role_widget->GetRoleId()));
	link_visited_color_role_widget->SetPrecalculatedColor(
		p.color(group_id, link_visited_color_role_widget->GetRoleId()));
}

void    CColorGroupWidget::SetColorSpecifier(QColorGroup::ColorRole role_id,
											CPaletteData::SColorSpecifier *cs)
{
	switch (role_id)
	{
		case QColorGroup::Background:
		background_color_role_widget->SetColorSpecifier(cs);
		break;

		case QColorGroup::Foreground:
		foreground_color_role_widget->SetColorSpecifier(cs);
		break;

		case QColorGroup::Base:
		base_color_role_widget->SetColorSpecifier(cs);
		break;

		case QColorGroup::Text:
		text_color_role_widget->SetColorSpecifier(cs);
		break;

		case QColorGroup::Button:
		button_color_role_widget->SetColorSpecifier(cs);
		break;

		case QColorGroup::ButtonText:
		button_text_color_role_widget->SetColorSpecifier(cs);
		break;

		case QColorGroup::Light:
		light_color_role_widget->SetColorSpecifier(cs);
		break;

		case QColorGroup::Midlight:
		midlight_color_role_widget->SetColorSpecifier(cs);
		break;

		case QColorGroup::Dark:
		dark_color_role_widget->SetColorSpecifier(cs);
		break;

		case QColorGroup::Mid:
		mid_color_role_widget->SetColorSpecifier(cs);
		break;

		case QColorGroup::Shadow:
		shadow_color_role_widget->SetColorSpecifier(cs);
		break;

		case QColorGroup::Highlight:
		highlight_color_role_widget->SetColorSpecifier(cs);
		break;

		case QColorGroup::HighlightedText:
		highlighted_text_color_role_widget->SetColorSpecifier(cs);
		break;

		case QColorGroup::BrightText:
		bright_text_color_role_widget->SetColorSpecifier(cs);
		break;

		case QColorGroup::Link:
		link_color_role_widget->SetColorSpecifier(cs);
		break;

		case QColorGroup::LinkVisited:
		link_visited_color_role_widget->SetColorSpecifier(cs);
		break;
	}
}

void    CColorGroupWidget::StoreColorSpecifier(QColorGroup::ColorRole role_id,
											CPaletteData::SColorSpecifier *cs)
{
	base_palette_dialog->StoreColorSpecifier(group_id, role_id, cs);
}

CBasePaletteDialog::CBasePaletteDialog(QWidget *parent)
	: CDialogFc(parent, 3, 2, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		), palette_data(0),
			button_color_specifier(0), background_color_specifier(0)
{
	QFontInfo   fi(font());

	int     pixel_size = fi.pixelSize();

	int     s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1 = new QGridLayout(this, 1, 2, 0, s, "subgrid1");
	subgrid_layout2 = new QGridLayout(this, 1, 5, 0, s, "subgrid2");

	grid_layout->setRowStretch(0, 1);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 1);

	subgrid_layout1->setRowStretch(0, 0);

	subgrid_layout1->setColStretch(0, 0);
	subgrid_layout1->setColStretch(1, 1);

	subgrid_layout2->setRowStretch(0, 0);

	subgrid_layout2->setColStretch(0, 0);
	subgrid_layout2->setColStretch(1, 1);
	subgrid_layout2->setColStretch(2, 0);
	subgrid_layout2->setColStretch(3, 0);
	subgrid_layout2->setColStretch(4, 0);

	QString     button_size_string("######");

	tab_widget              	= new QTabWidget(this);
	active_color_group_widget	= new CColorGroupWidget(tab_widget, this,
														QPalette::Active);
	inactive_color_group_widget = new CColorGroupWidget(tab_widget, this,
														QPalette::Inactive);
	disabled_color_group_widget	= new CColorGroupWidget(tab_widget, this,
														QPalette::Disabled);
	dialog_palette_demo			= new CDialogPaletteDemo(this);
	reset_button				= new CPushButtonFc("Reset", this);
	group_box					= new CHGroupBoxFc(	"Precalculated Colors",
													this);
	precalculated_check_box		= new CCheckBoxFc("Calculate", group_box);
	precalculated_button_button	= new CPushButtonFc("Button", group_box);
	precalculated_button_color	= new CColorDisplayWidget(group_box, 2, "###");
	precalculated_background_button	= new CPushButtonFc("Background",
														group_box);
	precalculated_background_color	= new CColorDisplayWidget(	group_box, 2,
																"###");
	name_label		= new CLabelFc(" Name", this);
	ok_button		= new CPushButtonFc("Ok", this, button_size_string);
	cancel_button	= new CPushButtonFc("Cancel", this, button_size_string);

	tab_widget->insertTab(active_color_group_widget, "Active");
	tab_widget->insertTab(inactive_color_group_widget, "Inactive");
	tab_widget->insertTab(disabled_color_group_widget, "Disabled");

	ok_button->setDefault(true);

	subgrid_layout1->addWidget(reset_button,				0, 0, Qt::AlignTop);
	subgrid_layout1->addWidget(group_box,					0, 1);

	subgrid_layout2->addWidget(name_label,					0, 0);
	subgrid_layout2->addWidget(ok_button,					0, 3);
	subgrid_layout2->addWidget(cancel_button,				0, 4);

	grid_layout->addWidget(tab_widget,						0, 0);
	grid_layout->addWidget(dialog_palette_demo,				0, 1);
	grid_layout->addMultiCellLayout(subgrid_layout1,		1, 1, 0, 1);
	grid_layout->addMultiCellLayout(subgrid_layout2,		2, 2, 0, 1);

	connect(ok_button, SIGNAL(clicked()), SLOT(AcceptOperation()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(CancelOperation()));

	connect(reset_button,
			SIGNAL(clicked()),
			SLOT(ResetColors()));

	connect(precalculated_button_button,
			SIGNAL(clicked()),
			SLOT(SelectPrecalculatedButtonColor()));

	connect(precalculated_background_button,
			SIGNAL(clicked()),
			SLOT(SelectPrecalculatedBackgroundColor()));
}

CBasePaletteDialog::~CBasePaletteDialog()
{
	delete palette_data;
	delete button_color_specifier;
	delete background_color_specifier;
}

void	CBasePaletteDialog::StoreColorSpecifier(
								QPalette::ColorGroup group_id,
								QColorGroup::ColorRole role_id,
								CPaletteData::SColorSpecifier *cs)
{
	if (cs)
		palette_data->SetGroupRoleSpecifier(
			CPaletteData::SGroupRoleEntry(group_id, role_id),
			new CPaletteData::SColorGroupRoleSpecifier(cs));
	else
		palette_data->DeleteGroupRoleSpecifier(
			CPaletteData::SGroupRoleEntry(group_id, role_id));

	dialog_palette_demo->SetDemoPalette(palette_data);
}

void	CBasePaletteDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();

	setMaximumHeight(height());
}

void	CBasePaletteDialog::SetPaletteValue(CPaletteData *v)
{
	delete palette_data;

	if (!v)
	{
		palette_data = new CPaletteData();

		palette_data->SetPrecalculateButtonColor(
			new CPaletteData::SRgbSpecifier(192, 192, 192));

		palette_data->SetPrecalculateBackgroundColor(
			new CPaletteData::SRgbSpecifier(192, 192, 192));
	}
	else
		palette_data = new CPaletteData(*v);

	delete button_color_specifier;
	delete background_color_specifier;

	if (palette_data->GetPrecalculateButtonColor())
		button_color_specifier =
			palette_data->GetPrecalculateButtonColor()->clone();
	else
		button_color_specifier = new CPaletteData::SRgbSpecifier(0, 0, 0);

	if (palette_data->GetPrecalculateBackgroundColor())
		background_color_specifier =
			palette_data->GetPrecalculateBackgroundColor()->clone();
	else
		background_color_specifier = new CPaletteData::SRgbSpecifier(0, 0, 0);

	QPalette	precalculated;

	disconnect(	precalculated_check_box,
				SIGNAL(toggled(bool)),
				this,
				SLOT(PrecalculatedToggled(bool)));

	if (palette_data->GetPrecalculateButtonColor() &&
		palette_data->GetPrecalculateBackgroundColor())
	{
		precalculated_check_box->setChecked(true);

		QColor		button;
		QColor		background;

		button_color_specifier->SetupColor(button);
		background_color_specifier->SetupColor(background);

		precalculated = QPalette(button, background);
	}
	else
	{
		precalculated_check_box->setChecked(false);

		precalculated = QPalette();
	}

	connect(precalculated_check_box,
			SIGNAL(toggled(bool)),
			SLOT(PrecalculatedToggled(bool)));

	precalculated_button_color->DrawColor(button_color_specifier);
	precalculated_background_color->DrawColor(background_color_specifier);

	active_color_group_widget->InitializeColorGroup();
	inactive_color_group_widget->InitializeColorGroup();
	disabled_color_group_widget->InitializeColorGroup();

	active_color_group_widget->SetPrecalculatedColor(precalculated);
	inactive_color_group_widget->SetPrecalculatedColor(precalculated);
	disabled_color_group_widget->SetPrecalculatedColor(precalculated);

	GroupRoleToSpecifierMap		*grtsm =
		palette_data->GetGroupRoleSpecifierMap();

	if (grtsm)
	{
		GroupRoleToSpecifierMapIterator		grtsm_iterator = grtsm->begin();

		while (grtsm_iterator != grtsm->end())
		{
			CPaletteData::SGroupRoleEntry
				group_role_entry = (*grtsm_iterator).first;

			CPaletteData::SGroupRoleSpecifier
				*group_role_specifier = (*grtsm_iterator).second;

			CPaletteData::SColorGroupRoleSpecifier	*cgrs =
				dynamic_cast<CPaletteData::SColorGroupRoleSpecifier *>
													(group_role_specifier);

			if (cgrs)
			{
				CPaletteData::SColorSpecifier	*cs = cgrs->color_specifier;

				switch (group_role_entry.group)
				{
					case QPalette::Active:
					active_color_group_widget->SetColorSpecifier(
													group_role_entry.role, cs);
					break;

					case QPalette::Inactive:
					inactive_color_group_widget->SetColorSpecifier(
													group_role_entry.role, cs);
					break;

					case QPalette::Disabled:
					disabled_color_group_widget->SetColorSpecifier(
													group_role_entry.role, cs);
					break;
				}
			}

			grtsm_iterator++;
		}
	}

	dialog_palette_demo->SetDemoPalette(palette_data);
}

void	CBasePaletteDialog::StorePaletteValue(CPaletteData **p)
{
	*p = new CPaletteData(*palette_data);
}

void	CBasePaletteDialog::CancelOperation()
{
	reject();
}

void	CBasePaletteDialog::ResetColors()
{
	SetPaletteValue(0);
}

void	CBasePaletteDialog::setFont(const QFont &f)
{
	QFontInfo       fi(f);

	int     pixel_size = fi.pixelSize();

	int     m = grid_layout_margin_data[pixel_size];
	int     s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1->setMargin(m);
	subgrid_layout1->setSpacing(s);
	subgrid_layout1->invalidate();

	subgrid_layout2->setMargin(m);
	subgrid_layout2->setSpacing(s);
	subgrid_layout2->invalidate();

	CDialogFc::setFont(f);
}

void	CBasePaletteDialog::PrecalculatedToggled(bool on)
{
	QPalette	precalculated;

	if (on)
	{
		palette_data->SetPrecalculateButtonColor(
			button_color_specifier->clone());
		palette_data->SetPrecalculateBackgroundColor(
			background_color_specifier->clone());

		QColor		button;
		QColor		background;

		button_color_specifier->SetupColor(button);
		background_color_specifier->SetupColor(background);

		precalculated = QPalette(button, background);
	}
	else
	{
		palette_data->SetPrecalculateButtonColor(0);
		palette_data->SetPrecalculateBackgroundColor(0);

		precalculated = QPalette();
	}

	active_color_group_widget->SetPrecalculatedColor(precalculated);
	inactive_color_group_widget->SetPrecalculatedColor(precalculated);
	disabled_color_group_widget->SetPrecalculatedColor(precalculated);

	dialog_palette_demo->SetDemoPalette(palette_data);
}

void	CBasePaletteDialog::SelectPrecalculatedButtonColor()
{
	CPaletteColorDialog		*pcd = OptionDialog::palette_color_dialog;

	CPaletteData::SColorSpecifier   *value;

	pcd->SetTitleAndResize("Select Precalculate Button Color");
	pcd->SetValuePointer(&value);
	pcd->SetNameComboBox(auxiliary_color_view);
	pcd->SetColorValue(button_color_specifier);

	if (pcd->exec() == QDialog::Accepted)
	{
		delete button_color_specifier;

		button_color_specifier = value;

		palette_data->SetPrecalculateButtonColor(
			button_color_specifier->clone());

		precalculated_button_color->DrawColor(button_color_specifier);
		
		if (precalculated_check_box->isChecked())
		{
			QColor		button;
			QColor		background;

			button_color_specifier->SetupColor(button);
			background_color_specifier->SetupColor(background);

			QPalette	precalculated = QPalette(button, background);

			active_color_group_widget->SetPrecalculatedColor(precalculated);
			inactive_color_group_widget->SetPrecalculatedColor(precalculated);
			disabled_color_group_widget->SetPrecalculatedColor(precalculated);

			dialog_palette_demo->SetDemoPalette(palette_data);
		}
	}
}

void	CBasePaletteDialog::SelectPrecalculatedBackgroundColor()
{
	CPaletteColorDialog		*pcd = OptionDialog::palette_color_dialog;

	CPaletteData::SColorSpecifier   *value;

	pcd->SetTitleAndResize("Select Precalculate Background Color");
	pcd->SetValuePointer(&value);
	pcd->SetNameComboBox(auxiliary_color_view);
	pcd->SetColorValue(background_color_specifier);

	if (pcd->exec() == QDialog::Accepted)
	{
		delete background_color_specifier;

		background_color_specifier = value;

		palette_data->SetPrecalculateBackgroundColor(
			background_color_specifier->clone());

		precalculated_background_color->DrawColor(background_color_specifier);
		
		if (precalculated_check_box->isChecked())
		{
			QColor		button;
			QColor		background;

			button_color_specifier->SetupColor(button);
			background_color_specifier->SetupColor(background);

			QPalette	precalculated = QPalette(button, background);

			active_color_group_widget->SetPrecalculatedColor(precalculated);
			inactive_color_group_widget->SetPrecalculatedColor(precalculated);
			disabled_color_group_widget->SetPrecalculatedColor(precalculated);

			dialog_palette_demo->SetDemoPalette(palette_data);
		}
	}
}

CAuxiliaryPaletteDialog::CAuxiliaryPaletteDialog(QWidget *parent)
	: CBasePaletteDialog(parent)
{
	name_line_edit	= new CLineEditFc(this, 
										OptionDialog::auxiliary_string_size);

	subgrid_layout2->addWidget(name_line_edit,		0, 1);

	connect(name_line_edit,
			SIGNAL(textChanged(const QString &)),
			SLOT(NameChanged(const QString &)));
}

void    CAuxiliaryPaletteDialog::SetNameLineEdit(const QString &n)
{
	name_line_edit->setText(n);
	name_changed_flag = false;
}

void	CAuxiliaryPaletteDialog::AcceptOperation()
{
	*name_pointer =
		string((const char *)(name_line_edit->text().local8Bit()));

	value_changed_flag = true;

	StorePaletteValue(value_pointer);

	accept();
}

void    CAuxiliaryPaletteDialog::NameChanged(const QString &s)
{
	name_changed_flag = true;
}

CMainviewTreePaletteDialog::CMainviewTreePaletteDialog(QWidget *parent)
	: CBasePaletteDialog(parent)
{
	name_combo_box			= new CComboBoxFc(true, this, "0123456789");
	add_auxiliary_button	= new CPushButtonFc("Add Auxiliary", this);

	subgrid_layout2->addWidget(name_combo_box,			0, 1);
	subgrid_layout2->addWidget(add_auxiliary_button,	0, 2);

	name_combo_box->setInsertionPolicy(QComboBox::NoInsertion);

	connect(name_combo_box,
			SIGNAL(activated(const QString &)),
			SLOT(NameSelected(const QString &)));

	connect(add_auxiliary_button,
			SIGNAL(clicked()),
			SLOT(AddAuxiliaryPalette()));
}

void    CMainviewTreePaletteDialog::SetNameComboBox(CAuxiliaryPaletteView *apv)
{
	auxiliary_palette_view = apv;

	UpdateComboBox(0);
}

void	CMainviewTreePaletteDialog::AcceptOperation()
{
	StorePaletteValue(value_pointer);

	accept();
}

void	CMainviewTreePaletteDialog::AddAuxiliaryPalette()
{
	string	name((const char *)
						(name_combo_box->lineEdit()->text().local8Bit()));

	CPaletteData	*value;

	StorePaletteValue(&value);

	if (auxiliary_palette_view->AddAuxiliaryPalette(name, value))
		UpdateComboBox(&name);
	else
		delete value;
}

void	CMainviewTreePaletteDialog::NameSelected(const QString &s)
{
	QListViewItem		*lvi =
		auxiliary_palette_view->GetListView()->findItem(s, 0);

	if (lvi)
	{
		COptionData		*d = auxiliary_palette_view->GetOptionData();

		map<string, CPaletteData *>		*auxiliary_palettes =
													d->GetAuxiliaryPalettes();

		string		name((const char *)(s.local8Bit()));

		map<string, CPaletteData *>::iterator  m_iterator =
			auxiliary_palettes->find(name);
			
		SetPaletteValue((*m_iterator).second);
	}
}

void	CMainviewTreePaletteDialog::UpdateComboBox(string *s)
{
	name_combo_box->clear();

	COptionData		*d = auxiliary_palette_view->GetOptionData();

	map<string, CPaletteData *>		*auxiliary_palettes =
										d->GetAuxiliaryPalettes();

	if (!auxiliary_palettes || !auxiliary_palettes->size())
		return;

	map<string, CPaletteData *>::iterator   m_iterator =
												auxiliary_palettes->begin();

	while (m_iterator != auxiliary_palettes->end())
	{
		name_combo_box->insertItem((*m_iterator).first.c_str());

		m_iterator++;
	}

	if (s)
		name_combo_box->setEditText(s->c_str());
	else
		name_combo_box->clearEdit();
}

CXpmImageDisplayWidget::CXpmImageDisplayWidget(QWidget *parent, int bw,
																QColor bg)
	: QWidget(parent), border_width(bw), xpm_image(0)
{
	setPaletteBackgroundColor(bg);
}

void	CXpmImageDisplayWidget::DrawXpmImage(CXpmImage *x)
{
	xpm_image = x;

	CalculateGeometry();

	if (xpm_image)
	{
		QPainter	painter(this);

		QPixmap		*p = xpm_image->CreatePixmap();

		if (p)
			painter.drawPixmap(border_width, border_width, *p);
	}
}

void    CXpmImageDisplayWidget::CalculateGeometry()
{
	int     w = border_width * 2;
	int     h = border_width * 2;

	if (xpm_image)
	{
		QPixmap		*p = xpm_image->CreatePixmap();

		if (p)
		{
			w += p->width();
			h += p->height();
		}
	}

	setMinimumSize(w, h);
	setMaximumSize(w, h);
}

void	CXpmImageDisplayWidget::paintEvent(QPaintEvent *e)
{
	DrawXpmImage(xpm_image);
}

CBrushDisplayWidget::CBrushDisplayWidget(	QWidget *parent,
											int bw,
											const QString &s,
											QColor bg)
	: QWidget(parent), border_width(bw), size_string(s),
		brush(QColor(0, 0, 0, QColor::Rgb))
{
	setPaletteBackgroundColor(bg);

	CalculateGeometry();
}

void	CBrushDisplayWidget::DrawBrush(const QBrush &b)
{
	brush = b;

	QPainter	painter(this);

	painter.fillRect(	border_width,
						border_width,
						size().width() - 2 * border_width,
						size().height() - 2 * border_width,
						brush);
}

void	CBrushDisplayWidget::DrawBrush(CPaletteData::SBrushSpecifier *bs)
{
	QBrush	b;

	bs->SetupBrush(b);

	erase();

	DrawBrush(b);
}

void    CBrushDisplayWidget::CalculateGeometry()
{
	QFontInfo       fi(font());
	QFontMetrics    fm(font());

	int     pixel_size  = fi.pixelSize();
	int     text_height = fm.height();
	int     text_width;

	if (size_string.length())
		text_width  = fm.width(size_string);
	else
		text_width  = text_height;

	int     w = text_width + push_button_width_data[pixel_size] * 2;
	int     h = text_height + push_button_height_data[pixel_size] * 2;

	setMinimumSize(w, h);
}

void    CBrushDisplayWidget::setFont(const QFont &f)
{
	QWidget::setFont(f);

	CalculateGeometry();
}

void	CBrushDisplayWidget::paintEvent(QPaintEvent *e)
{
	DrawBrush(brush);
}

CBaseXpmImageDialog::CBaseXpmImageDialog(QWidget *parent)
	: CDialogFc(parent, 4, 2, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		), xpm_image(0)
{
	QFontInfo   fi(font());

	int     pixel_size = fi.pixelSize();

	int     s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1 = new QGridLayout(this, 1, 4, 0, s, "subgrid1");
	subgrid_layout2 = new QGridLayout(this, 1, 5, 0, s, "subgrid2");

	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 1);
	grid_layout->setRowStretch(2, 0);
	grid_layout->setRowStretch(3, 0);

	grid_layout->setColStretch(0, 1);
	grid_layout->setColStretch(1, 0);

	subgrid_layout1->setRowStretch(0, 0);

	subgrid_layout1->setColStretch(0, 0);
	subgrid_layout1->setColStretch(1, 0);
	subgrid_layout1->setColStretch(2, 0);
	subgrid_layout1->setColStretch(3, 1);

	subgrid_layout2->setRowStretch(0, 0);

	subgrid_layout2->setColStretch(0, 0);
	subgrid_layout2->setColStretch(1, 1);
	subgrid_layout2->setColStretch(2, 0);
	subgrid_layout2->setColStretch(3, 0);
	subgrid_layout2->setColStretch(4, 0);

	QString     button_size_string("######");

	xpm_data_multi_line_edit	= new CMultiLineEditFc(this,
									"XPM Image Data #################", 10);
	xpm_image_display_widget1	= new CXpmImageDisplayWidget(this, 5);
	xpm_image_display_widget2	= new CXpmImageDisplayWidget(this, 5,
										QColor(255, 255, 255, QColor::Rgb));
	display_xpm_image_button	= new CPushButtonFc("Display", this);
	load_file_button			= new CPushButtonFc("Load", this);
	save_file_button			= new CPushButtonFc("Save", this);
	spacer						= new CSpacerFc(this);
	name_label		= new CLabelFc(" Name", this);
	ok_button		= new CPushButtonFc("Ok", this, button_size_string);
	cancel_button	= new CPushButtonFc("Cancel", this, button_size_string);

	ok_button->setDefault(true);

	subgrid_layout1->addWidget(display_xpm_image_button,		0, 0);
	subgrid_layout1->addWidget(load_file_button,				0, 1);
	subgrid_layout1->addWidget(save_file_button,				0, 2);
	subgrid_layout1->addWidget(spacer,							0, 3);

	subgrid_layout2->addWidget(name_label,						0, 0);
	subgrid_layout2->addWidget(ok_button,						0, 3);
	subgrid_layout2->addWidget(cancel_button,					0, 4);

	grid_layout->addMultiCellWidget(xpm_data_multi_line_edit,	0, 2, 0, 0);
	grid_layout->addWidget(xpm_image_display_widget1,			0, 1,
												Qt::AlignLeft | Qt::AlignTop);
	grid_layout->addWidget(xpm_image_display_widget2,			1, 1,
												Qt::AlignLeft | Qt::AlignTop);
	grid_layout->addLayout(subgrid_layout1,						2, 1);
	grid_layout->addMultiCellLayout(subgrid_layout2,			3, 3, 0, 1);

	connect(ok_button, SIGNAL(clicked()), SLOT(AcceptOperation()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(CancelOperation()));

	connect(display_xpm_image_button,
			SIGNAL(clicked()),
			SLOT(DisplayXpmImage()));

	connect(load_file_button,
			SIGNAL(clicked()),
			SLOT(LoadFile()));

	connect(save_file_button,
			SIGNAL(clicked()),
			SLOT(SaveFile()));
}

CBaseXpmImageDialog::~CBaseXpmImageDialog()
{
	delete xpm_image;
}

void	CBaseXpmImageDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();
}

void	CBaseXpmImageDialog::SetXpmImageValue(CXpmImage *v)
{
	xpm_data_multi_line_edit->clear();

	if (v)
	{
		xpm_data_multi_line_edit->append("/* XPM */");
		xpm_data_multi_line_edit->append("const char *xpm_image_data[] = {");

		string		line;

		list<string>::const_iterator	l_iterator = v->GetXpmData().begin();

		while (l_iterator != --(v->GetXpmData().end()))
		{
			line = "\"";
			line += *l_iterator;
			line += "\",";

			xpm_data_multi_line_edit->append(line.c_str());

			l_iterator++;
		}

		line = "\"";
		line += *l_iterator;
		line += "\"};";

		xpm_data_multi_line_edit->append(line.c_str());
	}

	delete xpm_image;

	if (v)
		xpm_image = new CXpmImage(*v);
	else
		xpm_image = 0;

	xpm_image_display_widget1->DrawXpmImage(xpm_image);
	xpm_image_display_widget2->DrawXpmImage(xpm_image);
}

void	CBaseXpmImageDialog::StoreXpmImageValue(CXpmImage **p)
{
	SetXpmData();

	if (xpm_image)
		*p = new CXpmImage(*xpm_image);
	else
		*p = 0;
}

void	CBaseXpmImageDialog::setFont(const QFont &f)
{
	QFontInfo       fi(f);

	int     pixel_size = fi.pixelSize();

	int     m = grid_layout_margin_data[pixel_size];
	int     s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1->setMargin(m);
	subgrid_layout1->setSpacing(s);
	subgrid_layout1->invalidate();

	subgrid_layout2->setMargin(m);
	subgrid_layout2->setSpacing(s);
	subgrid_layout2->invalidate();

	CDialogFc::setFont(f);

	QFont	source_code_font(f);

	source_code_font.setFamily("Courier");

	xpm_data_multi_line_edit->setFont(source_code_font);
}

void	CBaseXpmImageDialog::CancelOperation()
{
	reject();
}

void	CBaseXpmImageDialog::LoadFile()
{
	string		xpm_data_file_name;

	CFileDirectoryDialog	*fdd = OptionDialog::file_directory_dialog;

	vector< pair<string, string> >		types;

	types.push_back(pair<string, string>("XPM Image Files", "*.xpm"));
	types.push_back(pair<string, string>("All Files", "*"));

	if (fdd->DisplayOpenFileDialog(	&xpm_data_file_name,
									"Open XPM Image File",
									string(),
									string(),
									&types)     != QDialog::Accepted)
		return;

	ifstream    d_file(xpm_data_file_name.c_str());
	string		line;

	xpm_data_multi_line_edit->clear();

	while (getline(d_file, line))
		xpm_data_multi_line_edit->append(line.c_str());

	DisplayXpmImage();
}

void	CBaseXpmImageDialog::SaveFile()
{
	string		xpm_data_file_name;

	CFileDirectoryDialog	*fdd = OptionDialog::file_directory_dialog;

	vector< pair<string, string> >		types;

	types.push_back(pair<string, string>("XPM Image Files", "*.xpm"));
	types.push_back(pair<string, string>("All Files", "*"));

	if (fdd->DisplaySaveAsFileDialog(	&xpm_data_file_name,
										"Save As XPM Image File",
										string(),
										string(),
										&types)     != QDialog::Accepted)
		return;

	if (FilePath::FileExists(xpm_data_file_name))
	{
		CMessageDialog		*msd = OptionDialog::message_dialog;

		string	s("Overwrite existing file named ");
		s += xpm_data_file_name;
		s += " ?";

		msd->SetTitleAndMessageText("Save As XPM Image File", s);

		if (msd->ExecuteDialog() == QDialog::Accepted)
		{
			ofstream	d_file(xpm_data_file_name.c_str());

			d_file << (const char *)
						(xpm_data_multi_line_edit->text().local8Bit()) << endl;

			d_file.close();
		}
		else
			return;
	}
	else
	{
		ofstream	d_file(xpm_data_file_name.c_str());

		d_file << (const char *)
					(xpm_data_multi_line_edit->text().local8Bit()) << endl;

		d_file.close();
	}
}

void	CBaseXpmImageDialog::DisplayXpmImage()
{
	SetXpmData();

	xpm_image_display_widget1->DrawXpmImage(xpm_image);
	xpm_image_display_widget2->DrawXpmImage(xpm_image);
}

void	CBaseXpmImageDialog::SetXpmData()
{
	string		xpm_text_data((const char *)
							(xpm_data_multi_line_edit->text().local8Bit()));

	delete xpm_image;

	xpm_image = new CXpmImage();

	const int	left_quote		= 0;
	const int	right_quote		= 1;
	const int	line_char		= 2;
	const int	discard_char	= 3;

	int					state = discard_char;
	string::size_type	i;
	string::size_type	begin_index;

	for (i=0; i<xpm_text_data.length(); i++)
	{
		char	c = xpm_text_data[i];

		switch (state)
		{
			case left_quote:
			begin_index = i;
			if (c == '\"')
				state = right_quote;
			else
				state = line_char;
			break;

			case right_quote:
			if (c == '\"')
				state = left_quote;
			else
				state = discard_char;
			break;

			case line_char:
			if (c == '\"')
			{
				state = right_quote;
				xpm_image->xpm_data.push_back(
					string(xpm_text_data, begin_index, i - begin_index));
			}
			break;

			case discard_char:
			if (c == '\"')
				state = left_quote;
			break;
		}
	}
}

CAuxiliaryXpmImageDialog::CAuxiliaryXpmImageDialog(QWidget *parent)
	: CBaseXpmImageDialog(parent)
{
	name_line_edit	= new CLineEditFc(this, 
										OptionDialog::auxiliary_string_size);

	subgrid_layout2->addWidget(name_line_edit,		0, 1);

	connect(name_line_edit,
			SIGNAL(textChanged(const QString &)),
			SLOT(NameChanged(const QString &)));
}

void	CAuxiliaryXpmImageDialog::SetNameLineEdit(const QString &n)
{
	name_line_edit->setText(n);
	name_changed_flag = false;
}

void	CAuxiliaryXpmImageDialog::AcceptOperation()
{
	*name_pointer =
		string((const char *)(name_line_edit->text().local8Bit()));

	value_changed_flag = true;

	StoreXpmImageValue(value_pointer);

	accept();
}

void	CAuxiliaryXpmImageDialog::NameChanged(const QString &s)
{
	name_changed_flag = true;
}

CBrushXpmImageDialog::CBrushXpmImageDialog(QWidget *parent)
	: CBaseXpmImageDialog(parent)
{
	name_combo_box			= new CComboBoxFc(true, this, "0123456789");
	add_auxiliary_button	= new CPushButtonFc("Add Auxiliary", this);

	subgrid_layout2->addWidget(name_combo_box,			0, 1);
	subgrid_layout2->addWidget(add_auxiliary_button,	0, 2);

	name_combo_box->setInsertionPolicy(QComboBox::NoInsertion);

	connect(name_combo_box,
			SIGNAL(activated(const QString &)),
			SLOT(NameSelected(const QString &)));

	connect(add_auxiliary_button,
			SIGNAL(clicked()),
			SLOT(AddAuxiliaryXpmImage()));
}

void    CBrushXpmImageDialog::SetNameComboBox(CAuxiliaryXpmImageView *axiv)
{
	auxiliary_xpm_image_view = axiv;

	UpdateComboBox(0);
}

void	CBrushXpmImageDialog::AcceptOperation()
{
	StoreXpmImageValue(value_pointer);

	accept();
}

void	CBrushXpmImageDialog::AddAuxiliaryXpmImage()
{
	string	name((const char *)
						(name_combo_box->lineEdit()->text().local8Bit()));

	CXpmImage	*value;

	StoreXpmImageValue(&value);

	if (auxiliary_xpm_image_view->AddAuxiliaryXpmImage(name, value))
		UpdateComboBox(&name);
	else
		delete value;
}

void	CBrushXpmImageDialog::NameSelected(const QString &s)
{
	QListViewItem		*lvi =
		auxiliary_xpm_image_view->GetListView()->findItem(s, 0);

	if (lvi)
	{
		COptionData		*d = auxiliary_xpm_image_view->GetOptionData();

		map<string, CXpmImage *>		*auxiliary_xpm_images =
													d->GetAuxiliaryXpmImages();

		string		name((const char *)(s.local8Bit()));

		map<string, CXpmImage *>::iterator  m_iterator =
			auxiliary_xpm_images->find(name);
			
		SetXpmImageValue((*m_iterator).second);
	}
}

void	CBrushXpmImageDialog::UpdateComboBox(string *s)
{
	name_combo_box->clear();

	COptionData		*d = auxiliary_xpm_image_view->GetOptionData();

	map<string, CXpmImage *>		*auxiliary_xpm_images =
										d->GetAuxiliaryXpmImages();

	if (!auxiliary_xpm_images || !auxiliary_xpm_images->size())
		return;

	map<string, CXpmImage *>::iterator   m_iterator =
												auxiliary_xpm_images->begin();

	while (m_iterator != auxiliary_xpm_images->end())
	{
		name_combo_box->insertItem((*m_iterator).first.c_str());

		m_iterator++;
	}

	if (s)
		name_combo_box->setEditText(s->c_str());
	else
		name_combo_box->clearEdit();
}

CAuxiliaryBrushDialog::CAuxiliaryBrushDialog(QWidget *parent)
	: CDialogFc(parent, 7, 2, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		),  brush_data(0), xpm_image(0), color_specifier(0)
{
	QFontInfo   fi(font());

	int     pixel_size = fi.pixelSize();

	int     s = grid_layout_spacing_data[pixel_size];

	subgrid_layout = new QGridLayout(this, 1, 4, 0, s, "subgrid");

	grid_layout->setRowStretch(0, 1);
	grid_layout->setRowStretch(1, 1);
	grid_layout->setRowStretch(2, 0);
	grid_layout->setRowStretch(3, 0);
	grid_layout->setRowStretch(4, 0);
	grid_layout->setRowStretch(5, 0);
	grid_layout->setRowStretch(6, 0);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 1);

	subgrid_layout->setRowStretch(0, 0);

	subgrid_layout->setColStretch(0, 0);
	subgrid_layout->setColStretch(1, 1);
	subgrid_layout->setColStretch(2, 0);
	subgrid_layout->setColStretch(3, 0);

	QString     button_size_string1("######");
	QString     button_size_string2("##########");

	brush_style_button_group	= new CButtonGroupFc(2, Qt::Horizontal,
														"Brush Style", this);

	nobrush_button				= new CRadioButtonFc("NoBrush",
													brush_style_button_group);
	int_to_brush_style_map[0]					= Qt::NoBrush;
	brush_style_to_int_map[Qt::NoBrush]			= 0;

	solidpattern_button			= new CRadioButtonFc("SolidPattern",
													brush_style_button_group);
	int_to_brush_style_map[1]					= Qt::SolidPattern;
	brush_style_to_int_map[Qt::SolidPattern]	= 1;

	dense1pattern_button		= new CRadioButtonFc("Dense1Pattern",
													brush_style_button_group);
	int_to_brush_style_map[2]					= Qt::Dense1Pattern;
	brush_style_to_int_map[Qt::Dense1Pattern]	= 2;

	dense2pattern_button		= new CRadioButtonFc("Dense2Pattern",
													brush_style_button_group);
	int_to_brush_style_map[3]					= Qt::Dense2Pattern;
	brush_style_to_int_map[Qt::Dense2Pattern]	= 3;

	dense3pattern_button		= new CRadioButtonFc("Dense3Pattern",
													brush_style_button_group);
	int_to_brush_style_map[4]					= Qt::Dense3Pattern;
	brush_style_to_int_map[Qt::Dense3Pattern]	= 4;

	dense4pattern_button		= new CRadioButtonFc("Dense4Pattern",
													brush_style_button_group);
	int_to_brush_style_map[5]					= Qt::Dense4Pattern;
	brush_style_to_int_map[Qt::Dense4Pattern]	= 5;

	dense5pattern_button		= new CRadioButtonFc("Dense5Pattern",
													brush_style_button_group);
	int_to_brush_style_map[6]					= Qt::Dense5Pattern;
	brush_style_to_int_map[Qt::Dense5Pattern]	= 6;

	dense6pattern_button		= new CRadioButtonFc("Dense6Pattern",
													brush_style_button_group);
	int_to_brush_style_map[7]					= Qt::Dense6Pattern;
	brush_style_to_int_map[Qt::Dense6Pattern]	= 7;

	dense7pattern_button		= new CRadioButtonFc("Dense7Pattern",
													brush_style_button_group);
	int_to_brush_style_map[8]					= Qt::Dense7Pattern;
	brush_style_to_int_map[Qt::Dense7Pattern]	= 8;

	horpattern_button			= new CRadioButtonFc("HorPattern",
													brush_style_button_group);
	int_to_brush_style_map[9]					= Qt::HorPattern;
	brush_style_to_int_map[Qt::HorPattern]		= 9;

	verpattern_button			= new CRadioButtonFc("VerPattern",
													brush_style_button_group);
	int_to_brush_style_map[10]					= Qt::VerPattern;
	brush_style_to_int_map[Qt::VerPattern]		= 10;

	crosspattern_button			= new CRadioButtonFc("CrossPattern",
													brush_style_button_group);
	int_to_brush_style_map[11]					= Qt::CrossPattern;
	brush_style_to_int_map[Qt::CrossPattern]	= 11;

	bdiagpattern_button			= new CRadioButtonFc("BDiagPattern",
													brush_style_button_group);
	int_to_brush_style_map[12]					= Qt::BDiagPattern;
	brush_style_to_int_map[Qt::BDiagPattern]	= 12;

	fdiagpattern_button			= new CRadioButtonFc("FDiagPattern",
													brush_style_button_group);
	int_to_brush_style_map[13]					= Qt::FDiagPattern;
	brush_style_to_int_map[Qt::FDiagPattern]	= 13;

	diagcrosspattern_button		= new CRadioButtonFc("DiagCrossPattern",
													brush_style_button_group);
	int_to_brush_style_map[14]						= Qt::DiagCrossPattern;
	brush_style_to_int_map[Qt::DiagCrossPattern]	= 14;

	custompattern_button		= new CRadioButtonFc("CustomPattern",
													brush_style_button_group);
	int_to_brush_style_map[15]					= Qt::CustomPattern;
	brush_style_to_int_map[Qt::CustomPattern]	= 15;

	brush_display_widget1		= new CBrushDisplayWidget(this, 5);
	brush_display_widget2		= new CBrushDisplayWidget(this, 5, "",
										QColor(255, 255, 255, QColor::Rgb));
	brush_color_button			= new CPushButtonFc("Color", this,
													button_size_string2);
	color_display_widget		= new CColorDisplayWidget(this, 5);
	brush_xpm_image_button		= new CPushButtonFc("Xpm Image", this,
													button_size_string2);
	xpm_image_display_widget	= new CXpmImageDisplayWidget(this, 5);
	name_label		= new CLabelFc(" Name", this);
	name_line_edit	= new CLineEditFc(this, "");
	ok_button		= new CPushButtonFc("Ok", this, button_size_string1);
	cancel_button	= new CPushButtonFc("Cancel", this, button_size_string1);

	ok_button->setDefault(true);

	subgrid_layout->addWidget(name_label,				0, 0);
	subgrid_layout->addWidget(name_line_edit,			0, 1);
	subgrid_layout->addWidget(ok_button,				0, 2);
	subgrid_layout->addWidget(cancel_button,			0, 3);

	grid_layout->addMultiCellWidget(brush_style_button_group,	0, 5, 0, 0,
												Qt::AlignLeft | Qt::AlignTop);
	grid_layout->addWidget(brush_display_widget1,				0, 1);
	grid_layout->addWidget(brush_display_widget2,				1, 1);
	grid_layout->addWidget(brush_color_button,					2, 1,
												Qt::AlignLeft | Qt::AlignTop);
	grid_layout->addWidget(color_display_widget,				3, 1);
	grid_layout->addWidget(brush_xpm_image_button,				4, 1,
												Qt::AlignLeft | Qt::AlignTop);
	grid_layout->addWidget(xpm_image_display_widget,			5, 1,
												Qt::AlignLeft | Qt::AlignTop);
	grid_layout->addMultiCellLayout(subgrid_layout,				6, 6, 0, 1);

	connect(ok_button, SIGNAL(clicked()), SLOT(AcceptOperation()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(CancelOperation()));

	connect(brush_style_button_group,
			SIGNAL(clicked(int)),
			SLOT(SelectStyle(int)));

	connect(brush_color_button,
			SIGNAL(clicked()),
			SLOT(SelectBrushColor()));

	connect(brush_xpm_image_button,
			SIGNAL(clicked()),
			SLOT(SelectXpmImage()));

	connect(name_line_edit,
			SIGNAL(textChanged(const QString &)),
			SLOT(NameChanged(const QString &)));
}

CAuxiliaryBrushDialog::~CAuxiliaryBrushDialog()
{
	delete brush_data;
	delete xpm_image;
	delete color_specifier;
}

void    CAuxiliaryBrushDialog::SetNameLineEdit(const QString &n)
{
	name_line_edit->setText(n);
	name_changed_flag = false;
}

void    CAuxiliaryBrushDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();
}

void    CAuxiliaryBrushDialog::SetBrushValue(CPaletteData::SBrushSpecifier *v)
{
	delete brush_data;

	if (v)
		brush_data = v->clone();
	else
		brush_data =
			new CPaletteData::SStyleBrushSpecifier(
				new CPaletteData::SRgbSpecifier(0, 0, 0), Qt::NoBrush);

	int		id;

	switch(brush_data->GetSpecifierType())
	{
		case CPaletteData::StyleBrushSpecifier:
		{
			CPaletteData::SStyleBrushSpecifier	*sbs =
				dynamic_cast<CPaletteData::SStyleBrushSpecifier *>(brush_data);

			brush_style = sbs->style;

			id = brush_style_to_int_map[brush_style];

			delete color_specifier;
			color_specifier = sbs->color_specifier->clone();
			delete xpm_image;
			xpm_image = 0;
		}
		break;

		case CPaletteData::XpmImageBrushSpecifier:
		{
			CPaletteData::SXpmImageBrushSpecifier	*xibs =
				dynamic_cast<CPaletteData::SXpmImageBrushSpecifier *>(
																brush_data);

			brush_style = Qt::CustomPattern;

			id = brush_style_to_int_map[brush_style];

			delete color_specifier;
			color_specifier = xibs->color_specifier->clone();
			delete xpm_image;
			xpm_image = new CXpmImage(*xibs->xpm_image);
		}
		break;
	}

	brush_style_button_group->setButton(id);
	brush_style_button_group->find(id)->setFocus();

	color_display_widget->DrawColor(color_specifier);
	xpm_image_display_widget->DrawXpmImage(xpm_image);

	brush_display_widget1->DrawBrush(brush_data);
	brush_display_widget2->DrawBrush(brush_data);
}

void    CAuxiliaryBrushDialog::StoreBrushValue(
										CPaletteData::SBrushSpecifier **p)
{
	*p = brush_data->clone();
}

void    CAuxiliaryBrushDialog::setFont(const QFont &f)
{
	QFontInfo       fi(f);

	int     pixel_size = fi.pixelSize();

	int     m = grid_layout_margin_data[pixel_size];
	int     s = grid_layout_spacing_data[pixel_size];

	subgrid_layout->setMargin(m);
	subgrid_layout->setSpacing(s);
	subgrid_layout->invalidate();

	CDialogFc::setFont(f);
}

void    CAuxiliaryBrushDialog::AcceptOperation()
{
	*name_pointer =
		string((const char *)(name_line_edit->text().local8Bit()));

	value_changed_flag = true;

	StoreBrushValue(value_pointer);

	accept();
}

void    CAuxiliaryBrushDialog::CancelOperation()
{
	reject();
}

void    CAuxiliaryBrushDialog::CreateNewBrush()
{
	delete brush_data;

	if (brush_style == Qt::CustomPattern)
	{
		if (!xpm_image)
		{
			xpm_image = new CXpmImage();

			xpm_image->PushBackXpmCharData("2 2 1 1");
			xpm_image->PushBackXpmCharData(". c #000000");
			xpm_image->PushBackXpmCharData("..");
			xpm_image->PushBackXpmCharData("..");
		}

		brush_data = new CPaletteData::SXpmImageBrushSpecifier(
						color_specifier->clone(), new CXpmImage(*xpm_image));
	}
	else
	{
		brush_data = new CPaletteData::SStyleBrushSpecifier(
						color_specifier->clone(), brush_style);
	}

	brush_display_widget1->DrawBrush(brush_data);
	brush_display_widget2->DrawBrush(brush_data);
}

void    CAuxiliaryBrushDialog::SelectStyle(int id)
{
	brush_style_button_group->find(id)->setFocus();

	brush_style = int_to_brush_style_map[id];

	CreateNewBrush();
}

void    CAuxiliaryBrushDialog::SelectBrushColor()
{
	CPaletteColorDialog		*pcd = OptionDialog::palette_color_dialog;

	CPaletteData::SColorSpecifier   *value;

	pcd->SetTitleAndResize("Select Brush Color");
	pcd->SetValuePointer(&value);
	pcd->SetNameComboBox(GetAuxiliaryColorView());
	pcd->SetColorValue(color_specifier);

	if (pcd->exec() == QDialog::Accepted)
	{
		delete color_specifier;

		color_specifier = value;

		color_display_widget->DrawColor(color_specifier);

		CreateNewBrush();
	}
}

void    CAuxiliaryBrushDialog::SelectXpmImage()
{
	CBrushXpmImageDialog	*bxid = OptionDialog::brush_xpm_image_dialog;

	CXpmImage	*value;

	bxid->SetTitleAndResize("Select Brush Xpm Image");
	bxid->SetValuePointer(&value);
	bxid->SetNameComboBox(GetAuxiliaryXpmImageView());
	bxid->SetXpmImageValue(xpm_image);

	if (bxid->exec() == QDialog::Accepted)
	{
		delete xpm_image;

		xpm_image = value;

		xpm_image_display_widget->DrawXpmImage(xpm_image);

		CreateNewBrush();
	}
}

void    CAuxiliaryBrushDialog::NameChanged(const QString &s)
{
	name_changed_flag = true;
}

CSizePositionDialog::CSizePositionDialog(QWidget *parent)
	: CDialogFc(parent, 6, 3, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1 = new QGridLayout(this, 1, 2, 0, s, "subgrid1");
	subgrid_layout2 = new QGridLayout(this, 1, 2, 0, s, "subgrid2");
	subgrid_layout3 = new QGridLayout(this, 1, 2, 0, s, "subgrid3");
	subgrid_layout4 = new QGridLayout(this, 1, 2, 0, s, "subgrid4");

	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);
	grid_layout->setRowStretch(3, 0);
	grid_layout->setRowStretch(4, 1);
	grid_layout->setRowStretch(5, 0);

	grid_layout->setColStretch(0, 1);
	grid_layout->setColStretch(1, 0);
	grid_layout->setColStretch(2, 0);

	subgrid_layout1->setRowStretch(0, 1);

	subgrid_layout1->setColStretch(0, 0);
	subgrid_layout1->setColStretch(1, 1);

	subgrid_layout2->setRowStretch(0, 1);

	subgrid_layout2->setColStretch(0, 0);
	subgrid_layout2->setColStretch(1, 1);

	subgrid_layout3->setRowStretch(0, 1);

	subgrid_layout3->setColStretch(0, 0);
	subgrid_layout3->setColStretch(1, 1);

	subgrid_layout4->setRowStretch(0, 1);

	subgrid_layout4->setColStretch(0, 0);
	subgrid_layout4->setColStretch(1, 1);

	QString     size_string("######");

	spacer1					= new CSpacerFc(this, true);
	reset_data_check_box	= new CCheckBoxFc(
									"Reset Size/Position Data", this);
	spacer2					= new CSpacerFc(this, true);
	application_check_box	= new CCheckBoxFc(
									"Capture Application Window ", this);
	spacer3					= new CSpacerFc(this, true);
	help_check_box			= new CCheckBoxFc(
									"Capture Help Window", this);
	spacer4					= new CSpacerFc(this, true);
	option_file_label		= new CLabelFc("Capture Main View Windows", this);
	option_file_listbox		= new CListBoxFc(this, "option file listbox", 6);
	spacer5					= new CSpacerFc(this);
	ok_button				= new CPushButtonFc("Ok", this, size_string);
	cancel_button			= new CPushButtonFc("Cancel", this, size_string);

	subgrid_layout1->addWidget(spacer1,							0, 0);
	subgrid_layout1->addWidget(reset_data_check_box,			0, 1);

	subgrid_layout2->addWidget(spacer2,							0, 0);
	subgrid_layout2->addWidget(application_check_box,			0, 1);

	subgrid_layout3->addWidget(spacer3,							0, 0);
	subgrid_layout3->addWidget(help_check_box,					0, 1);

	subgrid_layout4->addWidget(spacer4,							0, 0);
	subgrid_layout4->addWidget(option_file_label,				0, 1);

	grid_layout->addMultiCellLayout(subgrid_layout1,			0, 0, 0, 2);
	grid_layout->addMultiCellLayout(subgrid_layout2,			1, 1, 0, 2);
	grid_layout->addMultiCellLayout(subgrid_layout3,			2, 2, 0, 2);
	grid_layout->addMultiCellLayout(subgrid_layout4,			3, 3, 0, 2);
	grid_layout->addMultiCellWidget(option_file_listbox,		4, 4, 0, 2);
	grid_layout->addWidget(spacer5,								5, 0);
	grid_layout->addWidget(ok_button,							5, 1);
	grid_layout->addWidget(cancel_button,						5, 2);

	connect(ok_button, SIGNAL(clicked()), SLOT(AcceptOperation()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(reject()));

	connect(reset_data_check_box,
			SIGNAL(toggled(bool)),
			SLOT(ResetDataToggled(bool)));
}

void	CSizePositionDialog::SetInterfaceIdAndNameVector(
											vector< pair<int, string> > *v)
{
	option_file_listbox->clear();
	option_file_listbox->setSelectionMode(QListBox::Extended);

	vector< pair<int, string> >::size_type		i;

	for (i=0; i<v->size(); i++)
	{
		string		path;
		string		file_name;

		FilePath::GetPathAndFileName((*v)[i].second, path, file_name);

		new SOptionFileItem(option_file_listbox, file_name, (*v)[i].first);
	}
}

void    CSizePositionDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();
}

void	CSizePositionDialog::setFont(const QFont &f)
{
	QFontInfo       fi(f);

	int     pixel_size = fi.pixelSize();

	int     s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1->setMargin(0);
	subgrid_layout1->setSpacing(s);
	subgrid_layout1->invalidate();

	subgrid_layout2->setMargin(0);
	subgrid_layout2->setSpacing(s);
	subgrid_layout2->invalidate();

	subgrid_layout3->setMargin(0);
	subgrid_layout3->setSpacing(s);
	subgrid_layout3->invalidate();

	subgrid_layout4->setMargin(0);
	subgrid_layout4->setSpacing(s);
	subgrid_layout4->invalidate();

	CDialogFc::setFont(f);
}

void	CSizePositionDialog::AcceptOperation()
{
	if (reset_data_check_box->isChecked())
		*reset_data_flag_pointer = true;
	else
		*reset_data_flag_pointer = false;

	if (application_check_box->isChecked())
		*capture_application_flag_pointer = true;
	else
		*capture_application_flag_pointer = false;

	if (help_check_box->isChecked())
		*capture_help_flag_pointer = true;
	else
		*capture_help_flag_pointer = false;

	selected_id_vector_pointer->clear();

	for (int i=0; i<option_file_listbox->count(); i++)
	{
		SOptionFileItem		*item =
			dynamic_cast<SOptionFileItem *>(option_file_listbox->item(i));

		if (option_file_listbox->isSelected(item))
			selected_id_vector_pointer->push_back(item->interface_id);
	}

	accept();
}

void    CSizePositionDialog::ResetDataToggled(bool on)
{
	if (on)
	{
		application_check_box->setText("Reset Application Window ");
		help_check_box->setText("Reset Help Window");
		option_file_label->setText("Reset Main View Windows");
	}
	else
	{
		application_check_box->setText("Capture Application Window ");
		help_check_box->setText("Capture Help Window");
		option_file_label->setText("Capture Main View Windows");
	}
}
