/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "optionEditView.h"

COptionEditFrame::COptionEditFrame(	QWidget *parent,
									CMainWindow *mw,
									DefaultOptionsIndexMode dm)
	: CFrameFc(parent, 2, 9),
		main_window(mw),
		default_options_index_mode(dm),
		current_option_data(0),
		undo_option_data(0),
		mainview_page_index(0),
		application_page_index(1),
		connection_page_index(2)
{
	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 1);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 5);
	grid_layout->setColStretch(2, 1);
	grid_layout->setColStretch(3, 0);
	grid_layout->setColStretch(4, 0);
	grid_layout->setColStretch(5, 0);
	grid_layout->setColStretch(6, 0);
	grid_layout->setColStretch(7, 0);
	grid_layout->setColStretch(8, 0);

	option_file_label		= new CLabelFc("Option File", this);
	option_file_combo_box	= new CComboBoxFc(false, this,
											"option_file_size_string");
	spacer					= new CSpacerFc(this);
	
	QString		size_string("#######");

	undo_button				= new CPushButtonFc("Undo", this, size_string);
	reload_button			= new CPushButtonFc("Reload", this, size_string);
	default_button			= new CPushButtonFc("Default", this, size_string);
	apply_button			= new CPushButtonFc("Apply", this, size_string);
	save_button				= new CPushButtonFc("Save", this, size_string);
	quit_button				= new CPushButtonFc("Quit", this, size_string);

	tab_widget				= new QTabWidget(this);
	mainview_edit_view	= new CMainviewEditView(tab_widget, main_window, this);
	application_edit_view	= new CApplicationEditView(tab_widget, this);
	connection_edit_view	= new CConnectionEditView(tab_widget, this);

	mainview_edit_view->SetAuxiliaryPaletteView(
		application_edit_view->GetAuxiliaryPaletteView());
	mainview_edit_view->SetAuxiliaryColorView(
		application_edit_view->GetAuxiliaryColorView());
	mainview_edit_view->SetAuxiliaryFontView(
		application_edit_view->GetAuxiliaryFontView());

	main_view_list_box = mainview_edit_view->GetMainViewListBox();

	option_file_label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	tab_widget->insertTab(	mainview_edit_view, "Main View Edit",
							mainview_page_index);
	tab_widget->insertTab(	application_edit_view, "Application Edit",
							application_page_index);
	tab_widget->insertTab(	connection_edit_view, "Connection Edit",
							connection_page_index);

	grid_layout->addWidget(option_file_label,			0, 0);
	grid_layout->addWidget(option_file_combo_box,		0, 1);
	grid_layout->addWidget(spacer,						0, 2);
	grid_layout->addWidget(undo_button,					0, 3);
	grid_layout->addWidget(reload_button,				0, 4);
	grid_layout->addWidget(default_button,				0, 5);
	grid_layout->addWidget(apply_button,				0, 6);
	grid_layout->addWidget(save_button,					0, 7);
	grid_layout->addWidget(quit_button,					0, 8);
	grid_layout->addMultiCellWidget(tab_widget,			1, 1, 0, 8);

	vector<SInterfaceConfiguration>::size_type		c_index;
	vector<SInterfaceConfiguration>
		&interface_configurations =
			main_window->GetApplication()->GetInterfaceConfigurations();

	if (default_options_index_mode == AtTop)
	{
		for (c_index=0; c_index<interface_configurations.size(); c_index++)
			option_file_items.push_back(
				SOptionFileItem(interface_configurations[c_index].
									GetDefaultOptionFileName(),
								&interface_configurations[c_index],
								0));
	}

	const map<int, CInterfaceControl *>		&iitim =
		main_window->GetInterfaceIdToInterfaceMap();

	map<int, CInterfaceControl *>::const_iterator	m_iterator = iitim.begin();

	while (m_iterator != iitim.end())
	{
		CInterfaceControl	*ic = (*m_iterator).second;

		option_file_items.push_back(
			SOptionFileItem(ic->GetOptionFileName(),
							&interface_configurations[ic->GetConfigurationId()],
							ic));

		m_iterator++;
	}

	if (default_options_index_mode == AtBottom ||
		default_options_index_mode == Sorted)
	{
		for (c_index=0; c_index<interface_configurations.size(); c_index++)
			option_file_items.push_back(
				SOptionFileItem(interface_configurations[c_index].
									GetDefaultOptionFileName(),
								&interface_configurations[c_index],
								0));
	}

	current_index = 0;

	if (option_file_items.size() > 1)
	{
		int		sort_index_first;
		int		sort_index_last;

		switch (default_options_index_mode)
		{
			case AtTop:
			sort_index_first	= interface_configurations.size();
			sort_index_last		= option_file_items.size();
			break;

			case AtBottom:
			sort_index_first	= 0;
			sort_index_last		= option_file_items.size() -
									interface_configurations.size();
			break;

			case Sorted:
			sort_index_first	= 0;
			sort_index_last		= option_file_items.size();
			break;
		}

		sort(	option_file_items.begin() + sort_index_first,
				option_file_items.begin() + sort_index_last);

		int		current_interface_id = main_window->GetCurrentInterfaceId();

		for (	vector<SOptionFileItem>::size_type i=0;
				i<option_file_items.size(); i++)
		{
			CInterfaceControl	*ic = option_file_items[i].interface_control;

			if (!ic)
				continue;
			else if (ic->GetInterfaceId() == current_interface_id)
			{
				current_index = i;
				break;
			}
		}
	}

	current_option_file_item = &option_file_items[current_index];

	for (	vector<SOptionFileItem>::size_type i=0;
			i<option_file_items.size(); i++)
	{
		string		path;
		string		file_name;

		FilePath::GetPathAndFileName(option_file_items[i].option_file_name,
										path, file_name);

		option_file_combo_box->insertItem(file_name.c_str());
	}

	option_file_combo_box->setCurrentItem(current_index);

	show();

	QObject::connect(	tab_widget,
						SIGNAL(currentChanged(QWidget *)),
						this,
						SLOT(CurrentTabChanged(QWidget *)));

	QObject::connect(	option_file_combo_box,
						SIGNAL(activated(int)),
						this,
						SLOT(SelectOptionFile(int)));

	QObject::connect(	undo_button,
						SIGNAL(clicked()),
						this,
						SLOT(UndoChanges()));

	QObject::connect(	reload_button,
						SIGNAL(clicked()),
						this,
						SLOT(ReloadOptionData()));

	QObject::connect(	default_button,
						SIGNAL(clicked()),
						this,
						SLOT(SetOptionDataToDefault()));

	QObject::connect(	apply_button,
						SIGNAL(clicked()),
						this,
						SLOT(ApplyChanges()));

	QObject::connect(	save_button,
						SIGNAL(clicked()),
						this,
						SLOT(SaveChanges()));

	QObject::connect(	quit_button,
						SIGNAL(clicked()),
						this,
						SLOT(Quit()));
}

void	COptionEditFrame::UpdateOptionEditFrameGeometry()
{
	mainview_edit_view->UpdateOptionEditFrameGeometry();
	application_edit_view->UpdateOptionEditFrameGeometry();
	connection_edit_view->UpdateOptionEditFrameGeometry();
	tab_widget->updateGeometry();
	updateGeometry();
}

void	COptionEditFrame::SetEditViewOptionData(bool undo_flag)
{
	mainview_edit_view->SetOptionData(	current_option_file_item,
										current_option_data, undo_flag);
	application_edit_view->SetOptionData(	current_option_file_item,
											current_option_data);
	connection_edit_view->SetOptionData(	current_option_file_item,
											current_option_data);
}

void	COptionEditFrame::SetConnectionEditViewOptionData()
{
	connection_edit_view->SetOptionData(current_option_file_item,
										current_option_data);
}

void	COptionEditFrame::CreateUndoOptionData()
{
	delete undo_option_data;

	undo_option_data = new COptionData(*current_option_data);
}

void	COptionEditFrame::CreateInitialViewTree(int tab_widget_height)
{
	if (tab_widget_height > 0)
		tab_widget->setMaximumHeight(tab_widget_height);

	mainview_edit_view->SetupNavigateTreeListView();
	mainview_edit_view->SetupSubviewTypeListView();
	application_edit_view->SetupAuxiliaryListViews();

	SelectOptionFile(current_index);

	mainview_edit_view->CreateViewTree();
}

void	COptionEditFrame::CurrentTabChanged(QWidget *tab_page)
{
	int		page_index = tab_widget->indexOf(tab_page);
}

void	COptionEditFrame::SelectOptionFile(int index)
{
	current_index = index;

	current_option_file_item = &option_file_items[current_index];

	CInterfaceControl		*ic = current_option_file_item->interface_control;
	SInterfaceConfiguration	*cf = current_option_file_item->
										interface_configuration;

	delete current_option_data;

	if (ic)
		current_option_data = new COptionData(*ic->GetOptionData());
	else
		current_option_data = new COptionData(*cf->GetDefaultOptions());

	SetEditViewOptionData(false);
}

void	COptionEditFrame::UndoChanges()
{
	if (!undo_option_data)
		return;

	bool	list_box_has_focus = main_view_list_box->hasFocus();

	if (list_box_has_focus)
		undo_button->setFocus();

	COptionData		*d = current_option_data;

	current_option_data	= undo_option_data;
	undo_option_data	= d;

	SetEditViewOptionData(true);

	if (list_box_has_focus)
		main_view_list_box->setFocus();
}

void	COptionEditFrame::ReloadOptionData()
{
	CInterfaceControl		*ic = current_option_file_item->interface_control;
	SInterfaceConfiguration	*cf = current_option_file_item->
										interface_configuration;

	delete current_option_data;

	if (ic)
		current_option_data = new COptionData(*ic->GetOptionData());
	else
		current_option_data = new COptionData(*cf->GetDefaultOptions());

	SetEditViewOptionData(true);
}

void	COptionEditFrame::SetOptionDataToDefault()
{
	CInterfaceControl		*ic = current_option_file_item->interface_control;
	SInterfaceConfiguration	*cf = current_option_file_item->
										interface_configuration;

	delete current_option_data;

	if (ic)
		current_option_data = new COptionData(*cf->GetDefaultOptions());
	else
		cf->CreateDefaultOptions(&current_option_data);

	SetEditViewOptionData(false);
}

void	COptionEditFrame::ApplyChanges()
{
	connection_edit_view->StoreConnectionData();

	CInterfaceControl		*ic = current_option_file_item->interface_control;
	SInterfaceConfiguration	*cf = current_option_file_item->
										interface_configuration;

	if (ic)
	{
		ic->SetOptionData(new COptionData(*current_option_data));

		ic->SetOptionsModifiedFlag(true);
	}
	else
	{
		cf->SetDefaultOptions(new COptionData(*current_option_data));

		cf->SetOptionsModifiedFlag(true);
	}
}

void	COptionEditFrame::SaveChanges()
{
	ApplyChanges();

	CInterfaceControl		*ic = current_option_file_item->interface_control;
	SInterfaceConfiguration	*cf = current_option_file_item->
										interface_configuration;

	if (ic)
	{
		ofstream    o_file(ic->GetOptionFileName().c_str());

		main_window->SaveOptions(o_file, ic);
	}
	else
	{
		cf->SaveDefaultOptionFile();
	}
}

void    COptionEditFrame::Quit()
{
	QApplication::postEvent(main_window,
		new QEvent((QEvent::Type)CMainWindow::InterfaceRestore));
}

CMainviewEditView::CMainviewEditView(	QWidget *parent,
										CMainWindow *mw,
										COptionEditFrame *oef)
	: CFrameFc(parent, 1, 2), tree_widget_id(-1),
		main_window(mw), option_edit_frame(oef),
		main_view_control(0), interface_control(0),
		view_tree(0)
{
	grid_layout->setRowStretch(0, 1);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 1);

	tab_widget			= new QTabWidget(this);
	mainview_select		= new CMainviewSelect(tab_widget, this, oef);
	mainview_tree_edit	= new CMainviewTreeEdit(tab_widget, this, oef);

	tab_widget->addTab(mainview_select, "View Select");
	tab_widget->addTab(mainview_tree_edit, "Tree Edit");

	tree_scroll_view	= new QScrollView(this);
	widget_stack		= new QWidgetStack(tree_scroll_view->viewport());

	tree_scroll_view->addChild(widget_stack);
	tree_scroll_view->setResizePolicy(QScrollView::AutoOneFit);
	tree_scroll_view->setHScrollBarMode(QScrollView::AlwaysOn);
	tree_scroll_view->setVScrollBarMode(QScrollView::AlwaysOn);

	grid_layout->addWidget(tab_widget,			0, 0);
	grid_layout->addWidget(tree_scroll_view,	0, 1);

	QObject::connect(	tab_widget,
						SIGNAL(currentChanged(QWidget *)),
						this,
						SLOT(CurrentTabChanged(QWidget *)));
}

CMainviewEditView::~CMainviewEditView()
{
	delete main_view_control;
	delete interface_control;

	if (view_tree)
		widget_stack->removeWidget(view_tree->GetWidget());

	tree_scroll_view->removeChild(widget_stack);

	delete view_tree;
	delete widget_stack;
	delete tree_scroll_view;
}

void	CMainviewEditView::SetAuxiliaryPaletteView(CAuxiliaryPaletteView *apv)
{
	mainview_tree_edit->SetAuxiliaryPaletteView(apv);
}

void	CMainviewEditView::SetAuxiliaryColorView(CAuxiliaryColorView *acv)
{
	mainview_tree_edit->SetAuxiliaryColorView(acv);
}

void	CMainviewEditView::SetAuxiliaryFontView(CAuxiliaryFontView *afv)
{
	mainview_tree_edit->SetAuxiliaryFontView(afv);
}

void	CMainviewEditView::SetupNavigateTreeListView()
{
	mainview_tree_edit->SetupNavigateTreeListView();
}

void	CMainviewEditView::SetupSubviewTypeListView()
{
	mainview_tree_edit->SetupSubviewTypeListView();
}

void	CMainviewEditView::UpdateOptionEditFrameGeometry()
{
	mainview_select->updateGeometry();
	mainview_tree_edit->UpdateOptionEditFrameGeometry();
	tab_widget->updateGeometry();
	widget_stack->updateGeometry();
	tree_scroll_view->updateGeometry();
	updateGeometry();
}

void	CMainviewEditView::CurrentTabChanged(QWidget *tab_page)
{
	UpdateOptionEditFrameGeometry();
}

void	CMainviewEditView::SetOptionData(	SOptionFileItem *ofi,
											COptionData *d, bool undo_flag)
{
	option_file_item	= ofi;
	option_data			= d;

	delete interface_control;

	SInterfaceConfiguration	*cf = option_file_item->interface_configuration;

	interface_control = (*cf->create_interface_control)(0, 0, 0, main_window);

	(*cf->create_new_data)();

	interface_control->SetApplicationData();

	mainview_tree_edit->SetOptionData(option_file_item, option_data);
	mainview_select->SetOptionData(option_file_item, option_data, undo_flag);
}

void    CMainviewEditView::SetCurrentMainViewIndex(int index)
{
	current_mainview_index = index;

	mainview_tree_edit->SetCurrentMainViewIndex(current_mainview_index);
}

void	CMainviewEditView::CreateViewTree(vector<int> *highlight_index)
{
	COptionData::SApplicationData   	*main_option_data =
											option_data->GetApplicationData();

	vector<COptionData::SMainViewData *>	*main_view_ptrs =
												option_data->GetMainViewPtrs();

	delete main_view_control;

	main_view_control = 0;

	if (!main_view_ptrs || !main_view_ptrs->size())
	{
		switch (tree_widget_id)
		{
			case -1:
			break;

			case 0:
			case 1:
			widget_stack->removeWidget(view_tree->GetWidget());
			delete view_tree;
			break;
		}

		tree_widget_id = -1;

		view_tree = 0;

		mainview_tree_edit->SetViewTree(view_tree, 0);
		UpdateOptionEditFrameGeometry();

		return;
	}

	main_view_control =
		new SMainViewControl(0, (*main_view_ptrs)[current_mainview_index]);

	main_view_control->CreateViewTree(	widget_stack,
										interface_control,
										main_option_data->GetFontString(),
										main_option_data->GetPaletteData());

	if (main_view_control->GetViewTree())
	{
		switch (tree_widget_id)
		{
			case -1:
			tree_widget_id = 0;
			widget_stack->addWidget(main_view_control->GetRootViewWidget(), 0);
			widget_stack->raiseWidget(0);
			break;

			case 0:
			tree_widget_id = 1;
			widget_stack->addWidget(main_view_control->GetRootViewWidget(), 1);
			widget_stack->raiseWidget(1);
			widget_stack->removeWidget(view_tree->GetWidget());
			delete view_tree;
			break;

			case 1:
			tree_widget_id = 0;
			widget_stack->addWidget(main_view_control->GetRootViewWidget(), 0);
			widget_stack->raiseWidget(0);
			widget_stack->removeWidget(view_tree->GetWidget());
			delete view_tree;
			break;
		}
	}
	else
	{
		switch (tree_widget_id)
		{
			case -1:
			break;

			case 0:
			case 1:
			widget_stack->removeWidget(view_tree->GetWidget());
			delete view_tree;
			break;
		}

		tree_widget_id = -1;
	}

	view_tree = main_view_control->GetViewTree();

	mainview_tree_edit->SetViewTree(view_tree, highlight_index);
	UpdateOptionEditFrameGeometry();
}

void    CMainviewEditView::ClearInterfaceSpecifiers(bool sdf)
{
	vector<COptionData::SMainViewData *>::size_type		i;
	vector<COptionData::SViewNodeData *>::size_type		j;

	vector<COptionData::SMainViewData *>	*mv =
											option_data->GetMainViewPtrs();

	if (!mv)
	{
		if (sdf)
			option_edit_frame->SetConnectionEditViewOptionData();

		return;
	}

	for (i=0; i<mv->size(); i++)
	{
		vector<COptionData::SViewNodeData *>	*vn =
											(*mv)[i]->GetViewNodeDataPtrs();

		if (!vn)
			continue;

		for (j=0; j<vn->size(); j++)
		{
			COptionData::SViewNodeData	*view_node_data = (*vn)[j];

			if (view_node_data->GetNodeType() ==
				COptionData::SViewNodeData::SubViewNode)
			{
				COptionData::SSubViewData	*subview_data =
					dynamic_cast<COptionData::SSubViewData *>(view_node_data);

				subview_data->SetInterfaceSpecifiers(0);
			}
		}
	}

	if (sdf)
		option_edit_frame->SetConnectionEditViewOptionData();
}

QListBox    *CMainviewEditView::GetMainViewListBox()
{
	return mainview_select->GetMainViewListBox();
}

CMainviewSelect::CMainviewSelect(	QWidget *parent,
									CMainviewEditView *mvev,
									COptionEditFrame *oef)
	: CFrameFc(parent, 5, 2), mainview_edit_view(mvev), option_edit_frame(oef)
{
	grid_layout->setRowStretch(0, 1);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);
	grid_layout->setRowStretch(3, 0);
	grid_layout->setRowStretch(4, 0);

	grid_layout->setColStretch(0, 1);
	grid_layout->setColStretch(1, 1);

	main_view_list_box			= new QListBox(this);
	sort_main_views_button		= new CPushButtonFc("Sort Views", this);
	set_default_flag_button		= new CPushButtonFc("Default Flag", this);
	add_main_view_before_button	= new CPushButtonFc("Add View Before", this);
	add_main_view_after_button	= new CPushButtonFc("Add View After", this);
	delete_main_view_button		= new CPushButtonFc("Delete View", this);
	rename_main_view_button		= new CPushButtonFc("Rename View", this);
	copy_main_view_button		= new CPushButtonFc("Copy View", this);
	paste_main_view_button		= new CPushButtonFc("Paste View", this);

	grid_layout->addMultiCellWidget(main_view_list_box,		0, 0, 0, 1);
	grid_layout->addWidget(sort_main_views_button,			1, 0);
	grid_layout->addWidget(set_default_flag_button,			1, 1);
	grid_layout->addWidget(add_main_view_before_button,		2, 0);
	grid_layout->addWidget(add_main_view_after_button,		2, 1);
	grid_layout->addWidget(delete_main_view_button,			3, 0);
	grid_layout->addWidget(rename_main_view_button,			3, 1);
	grid_layout->addWidget(copy_main_view_button,			4, 0);
	grid_layout->addWidget(paste_main_view_button,			4, 1);

	QObject::connect(	main_view_list_box,
						SIGNAL(highlighted(int)),
						this,
						SLOT(SelectMainView(int)));

	QObject::connect(	sort_main_views_button,
						SIGNAL(clicked()),
						this,
						SLOT(SortMainViews()));

	QObject::connect(	set_default_flag_button,
						SIGNAL(clicked()),
						this,
						SLOT(SetDefaultFlag()));

	QObject::connect(	add_main_view_before_button,
						SIGNAL(clicked()),
						this,
						SLOT(AddMainViewBefore()));

	QObject::connect(	add_main_view_after_button,
						SIGNAL(clicked()),
						this,
						SLOT(AddMainViewAfter()));

	QObject::connect(	delete_main_view_button,
						SIGNAL(clicked()),
						this,
						SLOT(DeleteMainView()));

	QObject::connect(	rename_main_view_button,
						SIGNAL(clicked()),
						this,
						SLOT(RenameMainView()));

	QObject::connect(	copy_main_view_button,
						SIGNAL(clicked()),
						this,
						SLOT(CopyMainView()));

	QObject::connect(	paste_main_view_button,
						SIGNAL(clicked()),
						this,
						SLOT(PasteMainView()));
}

void	CMainviewSelect::SetOptionData(	SOptionFileItem *ofi,
										COptionData *d, bool undo_flag)
{
	option_file_item	= ofi;
	option_data			= d;

	QObject::disconnect(main_view_list_box,
						SIGNAL(highlighted(int)),
						this,
						SLOT(SelectMainView(int)));

	main_view_list_box->clear();

	vector<COptionData::SMainViewData *>	*main_view_ptrs =
												option_data->GetMainViewPtrs();

	vector<COptionData::SMainViewData *>::size_type		i;

	if (main_view_ptrs && main_view_ptrs->size())
	{
		for (i=0; i<main_view_ptrs->size(); i++)
		{
			string		list_box_text;

			if ((*main_view_ptrs)[i]->GetDefaultDisplayFlag())
				list_box_text = "*";
			else
				list_box_text = " ";

			list_box_text += (*main_view_ptrs)[i]->GetMainViewName();

			main_view_list_box->insertItem(list_box_text.c_str());
		}

		if (undo_flag)
		{
			main_view_list_box->setCurrentItem(current_mainview_index);
			SelectMainView(current_mainview_index);
		}
		else
		{
			main_view_list_box->setCurrentItem(0);
			SelectMainView(0);
		}
	}
	else
		SelectMainView(-1);

	QObject::connect(	main_view_list_box,
						SIGNAL(highlighted(int)),
						this,
						SLOT(SelectMainView(int)));
}

void	CMainviewSelect::SelectMainView(int index)
{
	current_mainview_index = index;

	mainview_edit_view->SetCurrentMainViewIndex(current_mainview_index);
	mainview_edit_view->CreateViewTree();
}

void	CMainviewSelect::SortMainViews()
{
	CViewOrderingDialog		*d = OptionDialog::view_ordering_dialog;

	vector<COptionData::SMainViewData *>	*main_view_ptrs =
												option_data->GetMainViewPtrs();

	vector<vector<COptionData::SMainViewData *>::size_type>   ordering_array;

	d->SetTitleAndResize("Sort Main Views");
	d->SetPointers(main_view_ptrs, &ordering_array);

	if (d->exec() == QDialog::Rejected)
		return;

	sort_main_views_button->setFocus();

	option_edit_frame->CreateUndoOptionData();

	vector<COptionData::SMainViewData *>    *ordered_main_view_ptrs =
		new vector<COptionData::SMainViewData *>;

	vector<COptionData::SMainViewData *>::size_type		i;

	for (i=0; i<main_view_ptrs->size(); i++)
		ordered_main_view_ptrs->push_back((*main_view_ptrs)[ordering_array[i]]);

	option_data->SetMainViewPtrs(ordered_main_view_ptrs, false);

	option_edit_frame->SetEditViewOptionData(true);

	main_view_list_box->setFocus();
}

void	CMainviewSelect::SetDefaultFlag()
{
	vector<COptionData::SMainViewData *>	*main_view_ptrs =
												option_data->GetMainViewPtrs();

	if (main_view_ptrs &&
		current_mainview_index >= 0 &&
		current_mainview_index < main_view_ptrs->size())
	{
		set_default_flag_button->setFocus();

		option_edit_frame->CreateUndoOptionData();

		COptionData::SMainViewData		*main_view =
			(*main_view_ptrs)[current_mainview_index];

		if (main_view->GetDefaultDisplayFlag())
			main_view->SetDefaultDisplayFlag(false);
		else
			main_view->SetDefaultDisplayFlag(true);

		option_edit_frame->SetEditViewOptionData(true);

		main_view_list_box->setFocus();
	}
}

void	CMainviewSelect::AddMainViewBefore()
{
	CValueDialog	*d = OptionDialog::value_dialog;

	string		name;

	d->SetTitleAndLabelText("Add Main View Before", "Main View Name ");
	d->SetValuePointer(&name);
	d->SetValueLineEdit("");

	if (d->exec() == QDialog::Rejected)
		return;

	add_main_view_after_button->setFocus();

	option_edit_frame->CreateUndoOptionData();

	COptionData::SMainViewData	*mvd = new COptionData::SMainViewData();

	mvd->SetMainViewName(name);

	vector<COptionData::SMainViewData *>	*main_view_ptrs =
												option_data->GetMainViewPtrs();

	if (main_view_ptrs)
	{
		if (current_mainview_index >= 0 &&
			current_mainview_index < main_view_ptrs->size())
			main_view_ptrs->insert(
				main_view_ptrs->begin() + current_mainview_index, mvd);
	}
	else
		option_data->AddMainViewPtr(mvd);

	option_data->AdjustMainViewNames(true);

	mainview_edit_view->ClearInterfaceSpecifiers();

	option_edit_frame->SetEditViewOptionData(true);

	main_view_list_box->setFocus();
}

void	CMainviewSelect::AddMainViewAfter()
{
	CValueDialog	*d = OptionDialog::value_dialog;

	string		name;

	d->SetTitleAndLabelText("Add Main View After", "Main View Name ");
	d->SetValuePointer(&name);
	d->SetValueLineEdit("");

	if (d->exec() == QDialog::Rejected)
		return;

	add_main_view_after_button->setFocus();

	option_edit_frame->CreateUndoOptionData();

	COptionData::SMainViewData	*mvd = new COptionData::SMainViewData();

	mvd->SetMainViewName(name);

	vector<COptionData::SMainViewData *>	*main_view_ptrs =
												option_data->GetMainViewPtrs();

	if (main_view_ptrs)
	{
		if (current_mainview_index >= 0 &&
			current_mainview_index < main_view_ptrs->size())
		{
			current_mainview_index++;

			main_view_ptrs->insert(
				main_view_ptrs->begin() + current_mainview_index, mvd);
		}
	}
	else
		option_data->AddMainViewPtr(mvd);

	option_data->AdjustMainViewNames(true);

	mainview_edit_view->ClearInterfaceSpecifiers();

	option_edit_frame->SetEditViewOptionData(true);

	main_view_list_box->setFocus();
}

void	CMainviewSelect::DeleteMainView()
{
	vector<COptionData::SMainViewData *>	*main_view_ptrs =
												option_data->GetMainViewPtrs();

	if (main_view_ptrs &&
		current_mainview_index >= 0 &&
		current_mainview_index < main_view_ptrs->size())
	{
		CopyMainView();

		delete_main_view_button->setFocus();

		option_edit_frame->CreateUndoOptionData();

		main_view_ptrs->erase(main_view_ptrs->begin() + current_mainview_index);

		if (current_mainview_index == main_view_ptrs->size())
			current_mainview_index--;

		mainview_edit_view->ClearInterfaceSpecifiers();

		option_edit_frame->SetEditViewOptionData(true);

		main_view_list_box->setFocus();
	}
}

void	CMainviewSelect::RenameMainView()
{
	vector<COptionData::SMainViewData *>	*main_view_ptrs =
												option_data->GetMainViewPtrs();

	if (main_view_ptrs &&
		current_mainview_index >= 0 &&
		current_mainview_index < main_view_ptrs->size())
	{
		CValueDialog	*d = OptionDialog::value_dialog;

		string		name;

		d->SetTitleAndLabelText("Rename Main View", "Main View Name ");
		d->SetValuePointer(&name);
		d->SetValueLineEdit("");

		if (d->exec() == QDialog::Accepted)
		{
			rename_main_view_button->setFocus();

			option_edit_frame->CreateUndoOptionData();

			(*main_view_ptrs)[current_mainview_index]->SetMainViewName(name);

			option_data->AdjustMainViewNames(true);

			mainview_edit_view->ClearInterfaceSpecifiers();

			option_edit_frame->SetEditViewOptionData(true);

			main_view_list_box->setFocus();
		}
	}
}

void	CMainviewSelect::CopyMainView()
{
	vector<COptionData::SMainViewData *>	*main_view_ptrs =
												option_data->GetMainViewPtrs();

	if (main_view_ptrs &&
		current_mainview_index >= 0 &&
		current_mainview_index < main_view_ptrs->size())
	{
		COptionData		*od = new COptionData();

		od->SetApplicationData(new COptionData::SApplicationData());
		od->SetHelpData(new COptionData::SHelpData());
		od->AddMainViewPtr(
			new COptionData::SMainViewData(
				*(*main_view_ptrs)[current_mainview_index]));

		od->CopyOptionDataToClipboard();

		delete od;
	}
}

void	CMainviewSelect::PasteMainView()
{
	COptionData		*od;

	CParserDriver::StatusCode   sc =
		COptionData::InputOptionDataFromClipboard(&od);

	if (sc == CParserDriver::EndOfFile)
	{
		vector<COptionData::SMainViewData *>    *mvd = od->GetMainViewPtrs();

		if (mvd->size())
		{
			paste_main_view_button->setFocus();

			option_edit_frame->CreateUndoOptionData();

			option_data->AddMainViewPtr(
				new COptionData::SMainViewData(*(*mvd)[0]));

			option_data->AdjustMainViewNames(true);

			vector<COptionData::SMainViewData *>	*main_view_ptrs =
												option_data->GetMainViewPtrs();

			current_mainview_index = main_view_ptrs->size() - 1;

			mainview_edit_view->ClearInterfaceSpecifiers();

			option_edit_frame->SetEditViewOptionData(true);

			main_view_list_box->setFocus();
		}

		delete od;
	}
}

CMainviewTreeEdit::CMainviewTreeEdit(	QWidget *parent,
										CMainviewEditView *mvev,
										COptionEditFrame *oef)
	: CFrameFc(parent, 4, 2), mainview_edit_view(mvev), option_edit_frame(oef),
		highlight_palette_data(0), delete_palette_data_flag(false)
{
	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);
	grid_layout->setRowStretch(3, 1);

	grid_layout->setColStretch(0, 1);
	grid_layout->setColStretch(1, 1);

	reset_options_button		= new CPushButtonFc("Reset Options", this);
	highlight_tree_button		= new CPushButtonFc("Highlight Tree", this);
	move_down_subview_tree_button	= new CPushButtonFc("Descend", this);
	move_up_subview_tree_button		= new CPushButtonFc("Ascend", this);
	move_left_subview_tree_button	= new CPushButtonFc("Previous", this);
	move_right_subview_tree_button	= new CPushButtonFc("Next", this);

	tab_widget		= new QTabWidget(this);
	navigate_tree	= new CNavigateTree(tab_widget, this);
	subview_edit	= new CSubviewEdit(tab_widget, this);

	tab_widget->addTab(navigate_tree, "Navigate");
	tab_widget->addTab(subview_edit, "Subview");

	grid_layout->addWidget(reset_options_button,			0, 0);
	grid_layout->addWidget(highlight_tree_button,			0, 1);
	grid_layout->addWidget(move_down_subview_tree_button,	1, 0);
	grid_layout->addWidget(move_up_subview_tree_button,		1, 1);
	grid_layout->addWidget(move_left_subview_tree_button,	2, 0);
	grid_layout->addWidget(move_right_subview_tree_button,	2, 1);
	grid_layout->addMultiCellWidget(tab_widget,				3, 3, 0, 1);

	QObject::connect(	reset_options_button,
						SIGNAL(clicked()),
						this,
						SLOT(ResetOptions()));

	QObject::connect(	highlight_tree_button,
						SIGNAL(clicked()),
						this,
						SLOT(HighlightTree()));

	QObject::connect(	move_down_subview_tree_button,
						SIGNAL(clicked()),
						this,
						SLOT(MoveDown()));

	QObject::connect(	move_up_subview_tree_button,
						SIGNAL(clicked()),
						this,
						SLOT(MoveUp()));

	QObject::connect(	move_left_subview_tree_button,
						SIGNAL(clicked()),
						this,
						SLOT(MoveLeft()));

	QObject::connect(	move_right_subview_tree_button,
						SIGNAL(clicked()),
						this,
						SLOT(MoveRight()));

	QObject::connect(	tab_widget,
						SIGNAL(currentChanged(QWidget *)),
						this,
						SLOT(CurrentTabChanged(QWidget *)));
}

CMainviewTreeEdit::~CMainviewTreeEdit()
{
	if (delete_palette_data_flag)
		delete highlight_palette_data;
}

void	CMainviewTreeEdit::SetupNavigateTreeListView()
{
	navigate_tree->SetupListView();
}

void	CMainviewTreeEdit::SetupSubviewTypeListView()
{
	subview_edit->SetupListView();
}

void	CMainviewTreeEdit::UpdateOptionEditFrameGeometry()
{
	navigate_tree->updateGeometry();
	subview_edit->updateGeometry();
	tab_widget->updateGeometry();
	updateGeometry();
}

void	CMainviewTreeEdit::CurrentTabChanged(QWidget *tab_page)
{
//	QApplication::postEvent(main_window,
//		new QEvent((QEvent::Type)CMainWindow::UpdateGeometry));
}

void	CMainviewTreeEdit::SetOptionData(SOptionFileItem *ofi, COptionData *d)
{
	option_file_item	= ofi;
	option_data			= d;

	subview_edit->SetSubviewTypes(
		option_file_item->interface_configuration->subview_types);

	if (delete_palette_data_flag)
		delete highlight_palette_data;

	map<string, CPaletteData *>		*auxiliary_palettes =
										option_data->GetAuxiliaryPalettes();

	if (auxiliary_palettes)
	{
		map<string, CPaletteData *>::iterator	m_iterator =
			auxiliary_palettes->find("MAINVIEW_TREE_EDIT_HIGHLIGHT_PALETTE");

		if (m_iterator != auxiliary_palettes->end())
		{
			highlight_palette_data		= (*m_iterator).second;
			delete_palette_data_flag	= false;
		}
		else
		{
			CreateHighlightPaletteData();
		}
	}
	else
	{
		CreateHighlightPaletteData();
	}
}

void	CMainviewTreeEdit::SetViewTree(	CViewNode *n,
										vector<int> *highlight_index)
{
	view_tree = n;

	if (view_tree)
	{
		if (highlight_index)
			highlight_view_node	= view_tree->LocateViewNode(highlight_index);
		else
			highlight_view_node	= view_tree;

		highlight_tree_mode = true;

		highlight_view_node->SetHighlightPalette(highlight_palette_data);
		highlight_view_node->SetNodeIndicator(true);
	}

	navigate_tree->SetViewTree(n, highlight_index);
}

void	CMainviewTreeEdit::SetCurrentMainViewIndex(int index)
{
	current_mainview_index = index;
}

void    CMainviewTreeEdit::CreateHighlightPaletteData()
{
	highlight_palette_data		= new CPaletteData();
	delete_palette_data_flag	= true;

	CPaletteData::SColorNameSpecifier	*button =
		new CPaletteData::SColorNameSpecifier("yellow");

	CPaletteData::SColorNameSpecifier	*background =
		new CPaletteData::SColorNameSpecifier("red");

	highlight_palette_data->SetPrecalculateButtonColor(button);
	highlight_palette_data->SetPrecalculateBackgroundColor(background);
}

void	CMainviewTreeEdit::MoveHighlight(NavigationType nt)
{
	if (!view_tree)
		return;

	CViewNode	*next_view_node =
		highlight_view_node->NavigateViewNodeTree(nt);

	if (next_view_node)
	{
		highlight_view_node->RestorePalette();
		highlight_view_node->SetNodeIndicator(false);

		highlight_view_node	= next_view_node;
		highlight_tree_mode = true;

		highlight_view_node->SetHighlightPalette(highlight_palette_data);
		highlight_view_node->SetNodeIndicator(true);

		navigate_tree->MoveHighlightToIndex(
			highlight_view_node->GetViewNodeIndex());
	}
}

void	CMainviewTreeEdit::MoveHighlightToIndex(vector<int> *highlight_index)
{
	if (!view_tree)
		return;

	CViewNode	*next_view_node;

	if (highlight_index)
		next_view_node	= view_tree->LocateViewNode(highlight_index);
	else
		next_view_node	= view_tree;

	if (next_view_node)
	{
		highlight_view_node->RestorePalette();
		highlight_view_node->SetNodeIndicator(false);

		highlight_view_node	= next_view_node;
		highlight_tree_mode = true;

		highlight_view_node->SetHighlightPalette(highlight_palette_data);
		highlight_view_node->SetNodeIndicator(true);
	}
}

void	CMainviewTreeEdit::SelectFont(bool s_flag)
{
	if (!view_tree)
		return;

	CMainviewTreeFontDialog	*mtfd = OptionDialog::mainview_tree_font_dialog;

	string		value;

	mtfd->SetTitleAndResize("Select Font");
	mtfd->SetValuePointer(&value);
	mtfd->SetNameComboBox(auxiliary_font_view);
	mtfd->SetValueLineEdit(highlight_view_node->GetDefaultFontData().c_str());

	if (mtfd->exec() == QDialog::Accepted)
	{
		option_edit_frame->CreateUndoOptionData();

		if (s_flag)
			highlight_view_node->ResetFontData();

		highlight_view_node->SetFontData(value);

		vector<int>		*index1 = highlight_view_node->GetViewNodeIndex();
		vector<int>		*index2 = new vector<int>(index1->size());

		copy(index1->begin(), index1->end(), index2->begin());

		mainview_edit_view->CreateViewTree(index2);

		delete index2;
	}
}

void	CMainviewTreeEdit::SelectPalette(bool s_flag)
{
	if (!view_tree)
		return;

	CMainviewTreePaletteDialog	*mtpd =
		OptionDialog::mainview_tree_palette_dialog;

	mtpd->SetAuxiliaryColorView(auxiliary_color_view);

	CPaletteData	*value;

	mtpd->SetTitleAndResize("Select Palette");
	mtpd->SetValuePointer(&value);
	mtpd->SetNameComboBox(auxiliary_palette_view);
	mtpd->SetPaletteValue(highlight_view_node->GetDefaultPaletteData());

	if (mtpd->exec() == QDialog::Accepted)
	{
		option_edit_frame->CreateUndoOptionData();

		if (s_flag)
			highlight_view_node->ResetPaletteData();

		highlight_view_node->SetPaletteData(value);

		vector<int>		*index1 = highlight_view_node->GetViewNodeIndex();
		vector<int>		*index2 = new vector<int>(index1->size());

		copy(index1->begin(), index1->end(), index2->begin());

		mainview_edit_view->CreateViewTree(index2);

		delete index2;
	}
}

void	CMainviewTreeEdit::AddTreeNode(SAddNode &an, AddNodeType t)
{
	vector<COptionData::SMainViewData *>	*main_view_ptrs =
												option_data->GetMainViewPtrs();

	if (!main_view_ptrs || !main_view_ptrs->size())
		return;

	COptionData::SMainViewData	*main_view_data =
		(*main_view_ptrs)[current_mainview_index];

	string	mvn = main_view_data->GetMainViewName();
	bool	ddf = main_view_data->GetDefaultDisplayFlag();

	if (view_tree)
	{
		if (!highlight_view_node->GetParentNode())
		{
			CRootOrientationDialog		*rod =
				OptionDialog::root_orientation_dialog;

			Qt::Orientation		orientation;

			rod->SetAddNodePointer(&orientation);

			if (rod->ExecuteDialog() == QDialog::Rejected)
				return;

			an.SetRootOrientation(orientation);
		}

		option_edit_frame->CreateUndoOptionData();

		CViewNode	*new_root_node;
		CViewNode	*next_view_node;
		
		if (t == AddBefore)
			next_view_node =
				highlight_view_node->AddNodeBefore(an, &new_root_node);
		else
			next_view_node =
				highlight_view_node->AddNodeAfter(an, &new_root_node);

		if (new_root_node)
			view_tree = new_root_node;

		COptionData::SMainViewData	*mvd =
			view_tree->ReindexAndCreateMainViewData(mvn, ddf);

		(*main_view_ptrs)[current_mainview_index] = mvd;

		vector<int>		*index1 = next_view_node->GetViewNodeIndex();
		vector<int>		*index2 = new vector<int>(index1->size());

		copy(index1->begin(), index1->end(), index2->begin());

		mainview_edit_view->CreateViewTree(index2);

		delete main_view_data;
		delete index2;

		mainview_edit_view->ClearInterfaceSpecifiers(true);
	}
	else
	{
		option_edit_frame->CreateUndoOptionData();

		COptionData::SMainViewData	*mvd = new COptionData::SMainViewData();

		mvd->SetMainViewName(mvn);
		mvd->SetDefaultDisplayFlag(ddf);

		vector<int>     *index = new vector<int>;

		CViewNode::CreateReplacementSubtree(mvd, an, index);

		(*main_view_ptrs)[current_mainview_index] = mvd;

		mainview_edit_view->CreateViewTree();

		delete main_view_data;
		delete index;

		mainview_edit_view->ClearInterfaceSpecifiers(true);
	}
}

void	CMainviewTreeEdit::ReplaceNode(SReplaceNode &rn)
{
	if (!view_tree)
		return;

	option_edit_frame->CreateUndoOptionData();

	vector<COptionData::SMainViewData *>	*main_view_ptrs =
												option_data->GetMainViewPtrs();

	COptionData::SMainViewData	*main_view_data =
		(*main_view_ptrs)[current_mainview_index];

	string	mvn = main_view_data->GetMainViewName();
	bool	ddf = main_view_data->GetDefaultDisplayFlag();

	COptionData::SMainViewData	*mvd =
		view_tree->CreateMainViewDataWithReplace(mvn, ddf, rn);

	(*main_view_ptrs)[current_mainview_index] = mvd;

	vector<int>		*index1 = highlight_view_node->GetViewNodeIndex();
	vector<int>		*index2 = new vector<int>(index1->size());

	copy(index1->begin(), index1->end(), index2->begin());

	mainview_edit_view->CreateViewTree(index2);

	delete main_view_data;
	delete index2;

	mainview_edit_view->ClearInterfaceSpecifiers(true);
}

void	CMainviewTreeEdit::DeleteNode()
{
	if (!view_tree)
		return;

	option_edit_frame->CreateUndoOptionData();

	vector<COptionData::SMainViewData *>	*main_view_ptrs =
												option_data->GetMainViewPtrs();

	COptionData::SMainViewData	*main_view_data =
		(*main_view_ptrs)[current_mainview_index];

	string	mvn = main_view_data->GetMainViewName();
	bool	ddf = main_view_data->GetDefaultDisplayFlag();

	CViewNode	*next_view_node = highlight_view_node->DeleteNode();

	if (next_view_node)
	{
		delete highlight_view_node;

		COptionData::SMainViewData	*mvd =
			view_tree->ReindexAndCreateMainViewData(mvn, ddf);

		(*main_view_ptrs)[current_mainview_index] = mvd;

		vector<int>		*index1 = next_view_node->GetViewNodeIndex();
		vector<int>		*index2 = new vector<int>(index1->size());

		copy(index1->begin(), index1->end(), index2->begin());

		mainview_edit_view->CreateViewTree(index2);

		delete main_view_data;
		delete index2;

		mainview_edit_view->ClearInterfaceSpecifiers(true);
	}
	else
	{
		COptionData::SMainViewData	*mvd = new COptionData::SMainViewData();

		mvd->SetMainViewName(mvn);
		mvd->SetDefaultDisplayFlag(ddf);

		(*main_view_ptrs)[current_mainview_index] = mvd;

		mainview_edit_view->CreateViewTree();

		delete main_view_data;

		mainview_edit_view->ClearInterfaceSpecifiers(true);
	}
}

void	CMainviewTreeEdit::MoveDown()
{
	MoveHighlight(MoveDownTree);
}

void	CMainviewTreeEdit::MoveUp()
{
	MoveHighlight(MoveUpTree);
}

void	CMainviewTreeEdit::MoveLeft()
{
	MoveHighlight(MoveLeftTree);
}

void	CMainviewTreeEdit::MoveRight()
{
	MoveHighlight(MoveRightTree);
}

void	CMainviewTreeEdit::ResetOptions()
{
	if (!view_tree)
		return;

	option_edit_frame->CreateUndoOptionData();

	highlight_view_node->ResetFontData();
	highlight_view_node->ResetPaletteData();

	vector<int>		*index1 = highlight_view_node->GetViewNodeIndex();
	vector<int>		*index2 = new vector<int>(index1->size());

	copy(index1->begin(), index1->end(), index2->begin());

	mainview_edit_view->CreateViewTree(index2);

	delete index2;
}

void	CMainviewTreeEdit::HighlightTree()
{
	if (!view_tree)
		return;

	switch (highlight_tree_mode)
	{
		case true:
		highlight_view_node->RestorePalette();
		highlight_tree_mode = false;
		break;

		case false:
		highlight_view_node->SetHighlightPalette(highlight_palette_data);
		highlight_tree_mode = true;
		break;
	}
}

CNavigateTree::CNavigateTree(QWidget *parent, CMainviewTreeEdit *mvte)
	: QListView(parent), mainview_tree_edit(mvte), navigate_tree(0),
		foward_current_item_changed_flag(false)
{
	QObject::connect(	this,
						SIGNAL(currentChanged(QListViewItem *)),
						this,
						SLOT(CurrentItemChanged(QListViewItem *)));
}

CNavigateTree::~CNavigateTree()
{
	delete navigate_tree;
}

void	CNavigateTree::SetupListView()
{
	addColumn("Subview");
	addColumn("Index");
	setSorting(-1);
	setRootIsDecorated(true);
	setSelectionMode(QListView::Single);
}

void	CNavigateTree::SetViewTree(CViewNode *n, vector<int> *highlight_index)
{
	delete navigate_tree;

	clear();

	view_tree = n;

	if (view_tree)
	{
		navigate_tree = CreateNavigateTree(0, view_tree);

		OpenNavigateTree(navigate_tree);

		highlight_navigate_tree_node =
			FindNode(navigate_tree, highlight_index);

		foward_current_item_changed_flag = true;

		if (highlight_navigate_tree_node)
			setSelected(highlight_navigate_tree_node->item, true);
	}
	else
		navigate_tree = 0;
}

void	CNavigateTree::CreateIndexText(ostrstream &s, vector<int> *index)
{
	if (index && index->size())
	{
		vector<int>::size_type	i;

		for(i=0; i<index->size()-1; i++)
			s << (*index)[i] << ", ";

		s << (*index)[index->size()-1];
	}

	s << '\0';
}

SNavigateTreeNode	*CNavigateTree::CreateNavigateTree(
						SNavigateTreeNode *parent,
						CViewNode *vn)
{
	SNavigateTreeNode	*tree;

	switch (vn->GetNodeType())
	{
		case COptionData::SViewNodeData::SplitterNode:
		if (parent)
		{
			CSplitterView	*sv = dynamic_cast<CSplitterView *>(vn);

			COptionData::SSplitterData	*sd = sv->GetSplitterOptionData();

			string			subview_text	= "SPLITTER";
			vector<int>		*node_index 	= sd->GetIndex();

			ostrstream		index_text;

			CreateIndexText(index_text, node_index);

			SNavigateTreeItem	*item =
				new SNavigateTreeItem(	0,
										parent->item,
										subview_text.c_str(),
										index_text.str());

			tree = new SNavigateTreeNode(
				parent, item, subview_text, node_index);

			item->SetNode(tree);

			int		descendant_count = sv->GetDescendantCount();

			for (int i=descendant_count-1; i>=0; i--)
			{
				CViewNode	*descendant = sv->GetDescendant(i);

				tree->AddChild(CreateNavigateTree(tree, descendant));
			}

			reverse(tree->descendants.begin(), tree->descendants.end());
		}
		else
		{
			CSplitterView	*sv = dynamic_cast<CSplitterView *>(vn);

			COptionData::SSplitterData	*sd = sv->GetSplitterOptionData();

			string			subview_text	= "SPLITTER";
			vector<int>		*node_index 	= sd->GetIndex();

			ostrstream		index_text;

			CreateIndexText(index_text, node_index);

			SNavigateTreeItem	*item =
				new SNavigateTreeItem(	0,
										this,
										subview_text.c_str(),
										index_text.str());

			tree = new SNavigateTreeNode(
				0, item, subview_text, node_index);

			item->SetNode(tree);

			int		descendant_count = sv->GetDescendantCount();

			for (int i=descendant_count-1; i>=0; i--)
			{
				CViewNode	*descendant = sv->GetDescendant(i);

				tree->AddChild(CreateNavigateTree(tree, descendant));
			}

			reverse(tree->descendants.begin(), tree->descendants.end());
		}
		break;

		case COptionData::SViewNodeData::SubViewNode:
		if (parent)
		{
			CSubView	*sv = dynamic_cast<CSubView *>(vn);

			COptionData::SSubViewData	*sd = sv->GetSubViewOptionData();

			string			subview_text	= sd->GetSubViewName();
			vector<int>		*node_index 	= sd->GetIndex();

			ostrstream		index_text;

			CreateIndexText(index_text, node_index);

			SNavigateTreeItem	*item =
				new SNavigateTreeItem(	0,
										parent->item,
										subview_text.c_str(),
										index_text.str());

			tree = new SNavigateTreeNode(
				parent, item, subview_text, node_index);

			item->SetNode(tree);
		}
		else
		{
			CSubView	*sv = dynamic_cast<CSubView *>(vn);

			COptionData::SSubViewData	*sd = sv->GetSubViewOptionData();

			string			subview_text	= sd->GetSubViewName();
			vector<int>		*node_index 	= sd->GetIndex();

			ostrstream		index_text;

			CreateIndexText(index_text, node_index);

			SNavigateTreeItem	*item =
				new SNavigateTreeItem(	0,
										this,
										subview_text.c_str(),
										index_text.str());

			tree = new SNavigateTreeNode(
				0, item, subview_text, node_index);

			item->SetNode(tree);
		}
		break;
	}

	return tree;
}

void	CNavigateTree::OpenNavigateTree(SNavigateTreeNode *n)
{
	n->item->setOpen(true);

	vector<SNavigateTreeNode *>		&d = n->descendants;

	vector<SNavigateTreeNode *>::size_type	i;

	for (i=0; i<d.size(); i++)
		OpenNavigateTree(d[i]);
}

SNavigateTreeNode	*CNavigateTree::FindNode(
						SNavigateTreeNode *n, vector<int> *index)
{
	if (!index)
		return n;

	vector<int>		index_vector;

	copy(index->begin(), index->end(), back_inserter(index_vector));

	return FindNode(n, index_vector);
}

SNavigateTreeNode	*CNavigateTree::FindNode(
						SNavigateTreeNode *n, vector<int> &iv)
{
	if (!iv.size())
		return n;
	else
	{
		int		index = iv[0];

		if (index < 0 || index >= n->descendants.size())
			return 0;

		iv.erase(iv.begin());

		return FindNode(n->descendants[index], iv);
	}
}

void	CNavigateTree::MoveHighlightToIndex(vector<int> *highlight_index)
{
	if (view_tree)
	{
		highlight_navigate_tree_node =
			FindNode(navigate_tree, highlight_index);

		foward_current_item_changed_flag = false;

		if (highlight_navigate_tree_node)
			setSelected(highlight_navigate_tree_node->item, true);
	}
}

void	CNavigateTree::CurrentItemChanged(QListViewItem *item)
{
	if (foward_current_item_changed_flag == false)
		foward_current_item_changed_flag = true;
	else
	{
		SNavigateTreeItem	*nti = dynamic_cast<SNavigateTreeItem *>(item);

		highlight_navigate_tree_node = nti->node;

		mainview_tree_edit->MoveHighlightToIndex(nti->node->node_index);
	}
}

CSubviewEdit::CSubviewEdit(QWidget *parent, CMainviewTreeEdit *mvte)
	: CFrameFc(parent, 7, 4), mainview_tree_edit(mvte), subview_types(0)
{
	QFontInfo       fi(font());

	int     pixel_size = fi.pixelSize();

	int     s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1	= new QGridLayout(this, 1, 4, 0, s, "subgrid1");
	subgrid_layout2	= new QGridLayout(this, 1, 4, 0, s, "subgrid2");
	subgrid_layout3	= new QGridLayout(this, 1, 4, 0, s, "subgrid3");

	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);
	grid_layout->setRowStretch(3, 0);
	grid_layout->setRowStretch(4, 0);
	grid_layout->setRowStretch(5, 0);
	grid_layout->setRowStretch(6, 1);

	grid_layout->setColStretch(0, 1);
	grid_layout->setColStretch(1, 1);
	grid_layout->setColStretch(2, 1);
	grid_layout->setColStretch(3, 0);

	subgrid_layout1->setRowStretch(0, 1);

	subgrid_layout1->setColStretch(0, 0);
	subgrid_layout1->setColStretch(1, 1);
	subgrid_layout1->setColStretch(2, 0);
	subgrid_layout1->setColStretch(3, 1);

	subgrid_layout2->setRowStretch(0, 1);

	subgrid_layout2->setColStretch(0, 0);
	subgrid_layout2->setColStretch(1, 1);
	subgrid_layout2->setColStretch(2, 1);
	subgrid_layout2->setColStretch(3, 0);

	subgrid_layout3->setRowStretch(0, 1);

	subgrid_layout3->setColStretch(0, 0);
	subgrid_layout3->setColStretch(1, 1);
	subgrid_layout3->setColStretch(2, 1);
	subgrid_layout3->setColStretch(3, 0);

	select_font_button			= new CPushButtonFc("Select Font", this);
	select_palette_button		= new CPushButtonFc("Select Palette", this);

	spacer1_1					= new CSpacerFc(this, true);
	subtree_font_check_box		= new CCheckBoxFc("Subtree Font", this);
	spacer1_2					= new CSpacerFc(this, true);
	subtree_palette_check_box	= new CCheckBoxFc("Subtree Palette ", this);

	add_node_before_button		= new CPushButtonFc("Add Before", this);
	add_node_after_button		= new CPushButtonFc("Add After", this);

	replace_node_button			= new CPushButtonFc("Replace", this);
	delete_node_button			= new CPushButtonFc("Delete", this);

	spacer2_1					= new CSpacerFc(this, true);
	splitter_radio_button		= new CRadioButtonFc("Splitter", this);
	subview_radio_button		= new CRadioButtonFc("Subview", this);
	spacer2_2					= new CSpacerFc(this, true);

	spacer3_1					= new CSpacerFc(this, true);
	horizontal_radio_button		= new CRadioButtonFc("Hztl ", this);
	vertical_radio_button		= new CRadioButtonFc("Vrtl", this);
	spacer3_2					= new CSpacerFc(this, true);

	splitter_pane_count_label	= new CLabelFc("  # Panes", this);
	splitter_pane_count_spinbox	= new QSpinBox(1, 8, 1, this);

	subview_type_list_view		= new QListView(this);

	splitter_pane_count_label->setAlignment(
									Qt::AlignHCenter | Qt::AlignVCenter);

	splitter_radio_button->setState(QButton::On);
	subview_radio_button->setState(QButton::Off);

	horizontal_radio_button->setState(QButton::On);
	vertical_radio_button->setState(QButton::Off);

	subgrid_layout1->addWidget(spacer1_1,						0, 0);
	subgrid_layout1->addWidget(subtree_font_check_box,			0, 1);
	subgrid_layout1->addWidget(spacer1_2,						0, 2);
	subgrid_layout1->addWidget(subtree_palette_check_box,		0, 3);

	subgrid_layout2->addWidget(spacer2_1,						0, 0);
	subgrid_layout2->addWidget(splitter_radio_button,			0, 1);
	subgrid_layout2->addWidget(subview_radio_button,			0, 2);
	subgrid_layout2->addWidget(spacer2_2,						0, 3);

	subgrid_layout3->addWidget(spacer3_1,						0, 0);
	subgrid_layout3->addWidget(horizontal_radio_button,			0, 1);
	subgrid_layout3->addWidget(vertical_radio_button,			0, 2);
	subgrid_layout3->addWidget(spacer3_2,						0, 3);

	grid_layout->addMultiCellWidget(select_font_button,				0, 0, 0, 1);
	grid_layout->addMultiCellWidget(select_palette_button,			0, 0, 2, 3);
	grid_layout->addMultiCellLayout(subgrid_layout1,				1, 1, 0, 3);
	grid_layout->addMultiCellWidget(add_node_before_button,			2, 2, 0, 1);
	grid_layout->addMultiCellWidget(add_node_after_button,			2, 2, 2, 3);
	grid_layout->addMultiCellWidget(replace_node_button,			3, 3, 0, 1);
	grid_layout->addMultiCellWidget(delete_node_button,				3, 3, 2, 3);
	grid_layout->addMultiCellLayout(subgrid_layout2,				4, 4, 0, 3);
	grid_layout->addMultiCellLayout(subgrid_layout3,				5, 5, 0, 1);
	grid_layout->addWidget(splitter_pane_count_label,				5, 2);
	grid_layout->addWidget(splitter_pane_count_spinbox,				5, 3);
	grid_layout->addMultiCellWidget(subview_type_list_view,			6, 6, 0, 3);

	QObject::connect(	select_font_button,
						SIGNAL(clicked()),
						this,
						SLOT(SelectFont()));

	QObject::connect(	select_palette_button,
						SIGNAL(clicked()),
						this,
						SLOT(SelectPalette()));

	QObject::connect(	add_node_before_button,
						SIGNAL(clicked()),
						this,
						SLOT(AddNodeBefore()));

	QObject::connect(	add_node_after_button,
						SIGNAL(clicked()),
						this,
						SLOT(AddNodeAfter()));

	QObject::connect(	replace_node_button,
						SIGNAL(clicked()),
						this,
						SLOT(ReplaceNode()));

	QObject::connect(	delete_node_button,
						SIGNAL(clicked()),
						this,
						SLOT(DeleteNode()));

	QObject::connect(	splitter_radio_button,
						SIGNAL(toggled(bool)),
						this,
						SLOT(SplitterRadioToggled(bool)));

	QObject::connect(	subview_radio_button,
						SIGNAL(toggled(bool)),
						this,
						SLOT(SubviewRadioToggled(bool)));

	QObject::connect(	horizontal_radio_button,
						SIGNAL(toggled(bool)),
						this,
						SLOT(HorizontalRadioToggled(bool)));

	QObject::connect(	vertical_radio_button,
						SIGNAL(toggled(bool)),
						this,
						SLOT(VerticalRadioToggled(bool)));
}

CSubviewEdit::~CSubviewEdit()
{
}

void	CSubviewEdit::SetupListView()
{
	subview_type_list_view->header()->setMovingEnabled(false);
	subview_type_list_view->setSorting(-1);
	subview_type_list_view->addColumn("Subview");
	subview_type_list_view->addColumn("M");
	subview_type_list_view->addColumn("Count");
	subview_type_list_view->setSelectionMode(QListView::Single);
}

void	CSubviewEdit::SetSubviewTypes(const vector< pair<string, int> > *st)
{
	subview_types = st;

	subview_type_list_view->clear();
	subview_multiplicity_count_map.clear();

	int		i;

	for (i=subview_types->size()-1; i>=0; i--)
	{
		ostrstream		os;
		const string	&type			= (*subview_types)[i].first;
		int				multiplicity	= (*subview_types)[i].second;

		subview_multiplicity_count_map[type] = pair<int, int>(multiplicity, 0);

		if (multiplicity != -1)
			os << multiplicity << '\0';
		else
			os << "*" << '\0';

		new QListViewItem(subview_type_list_view, type.c_str(), os.str(), "0");
	}

	UpdateMultiplicityCountMap();

	subview_type_list_view->setSelected(
		subview_type_list_view->firstChild(), true);

	subview_type_list_view->setCurrentItem(
		subview_type_list_view->firstChild());
}

void    CSubviewEdit::UpdateMultiplicityCountMap()
{
	int		i;

	for (i=subview_types->size()-1; i>=0; i--)
	{
		const string	&type = (*subview_types)[i].first;

		subview_multiplicity_count_map[type].second = 0;

		QListViewItem	*item =
			subview_type_list_view->findItem(type.c_str(), 0);

		item->setText(2, "0");
	}

	COptionData		*option_data = mainview_tree_edit->GetOptionData();

	map<string, int>				count_map;
	map<string, int>::iterator		m_iterator;

	option_data->GetSubviewCountMap(count_map);

	m_iterator = count_map.begin();

	while (m_iterator != count_map.end())
	{
		const string	&subview_name	= (*m_iterator).first;
		int				count			= (*m_iterator).second;

		pair<int, int>	&mc_pair = subview_multiplicity_count_map[subview_name];

		mc_pair.second = count;

		QListViewItem	*item =
			subview_type_list_view->findItem(subview_name.c_str(), 0);

		ostrstream		os;

		if (count <= mc_pair.first || mc_pair.first == -1)
			os << count << '\0';
		else
			os << "*" << count << '\0';

		item->setText(2, os.str());

		m_iterator++;
	}
}

bool    CSubviewEdit::MultiplicityCountReached(const string &sv)
{
	pair<int, int>	&mc_pair = subview_multiplicity_count_map[sv];

	if (mc_pair.first != -1 && mc_pair.first <= mc_pair.second)
		return true;
	else
		return false;
}

void    CSubviewEdit::setFont(const QFont &f)
{
	QFontInfo		fi(f);

	int		pixel_size = fi.pixelSize();

	int		s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1->setMargin(0);
	subgrid_layout1->setSpacing(s);
	subgrid_layout1->invalidate();

	subgrid_layout2->setMargin(0);
	subgrid_layout2->setSpacing(s);
	subgrid_layout2->invalidate();

	subgrid_layout3->setMargin(0);
	subgrid_layout3->setSpacing(s);
	subgrid_layout3->invalidate();

	CFrameFc::setFont(f);
}

void	CSubviewEdit::SelectFont()
{
	mainview_tree_edit->SelectFont(subtree_font_check_box->isChecked());
}

void	CSubviewEdit::SelectPalette()
{
	mainview_tree_edit->SelectPalette(subtree_palette_check_box->isChecked());
}

SReplaceNode     CSubviewEdit::CreateModifyStruct()
{
	if (splitter_radio_button->state() == QButton::On)
	{
		Qt::Orientation		orientation;
		int					count;

		if (horizontal_radio_button->state() == QButton::On)
			orientation = Qt::Horizontal;
		else
			orientation = Qt::Vertical;

		count = splitter_pane_count_spinbox->value();

		return SReplaceNode(orientation, count);
	}
	else
	{
		QListViewItem	*current_item = subview_type_list_view->currentItem();

		QString	listview_string = current_item->text(0);

		string	subview_string = (const char *)(listview_string.local8Bit());

		return SReplaceNode(subview_string);
	}
}

void	CSubviewEdit::AddNodeBefore()
{
	SAddNode		an = CreateModifyStruct();

	if (!an.modify_with_splitter && MultiplicityCountReached(an.subview_name))
	{
		CInformationDialog	*ifd = OptionDialog::information_dialog;

		string		text;

		text += "Multiplicity count reached for\n";
		text += "subview type ";
		text += an.subview_name;
		text += ".";

		ifd->SetTitleAndMessageText("Add Node Before", text);
		ifd->ExecuteDialog();

		return;
	}

	mainview_tree_edit->AddTreeNode(an, AddBefore);

	UpdateMultiplicityCountMap();
}

void	CSubviewEdit::AddNodeAfter()
{
	SAddNode		an = CreateModifyStruct();

	if (!an.modify_with_splitter && MultiplicityCountReached(an.subview_name))
	{
		CInformationDialog	*ifd = OptionDialog::information_dialog;

		string		text;

		text += "Multiplicity count reached for\n";
		text += "subview type ";
		text += an.subview_name;
		text += ".";

		ifd->SetTitleAndMessageText("Add Node After", text);
		ifd->ExecuteDialog();

		return;
	}

	mainview_tree_edit->AddTreeNode(an, AddAfter);

	UpdateMultiplicityCountMap();
}

void	CSubviewEdit::ReplaceNode()
{
	SReplaceNode		rn = CreateModifyStruct();

	if (!rn.modify_with_splitter && MultiplicityCountReached(rn.subview_name))
	{
		CInformationDialog	*ifd = OptionDialog::information_dialog;

		string		text;

		text += "Multiplicity count reached for\n";
		text += "subview type ";
		text += rn.subview_name;
		text += ".";

		ifd->SetTitleAndMessageText("Replace Node", text);
		ifd->ExecuteDialog();

		return;
	}

	mainview_tree_edit->ReplaceNode(rn);

	UpdateMultiplicityCountMap();
}

void	CSubviewEdit::DeleteNode()
{
	mainview_tree_edit->DeleteNode();

	UpdateMultiplicityCountMap();
}

void	CSubviewEdit::SplitterRadioToggled(bool on)
{
	if (on)
		subview_radio_button->setState(QButton::Off);
	else
		subview_radio_button->setState(QButton::On);
}

void	CSubviewEdit::SubviewRadioToggled(bool on)
{
	if (on)
		splitter_radio_button->setState(QButton::Off);
	else
		splitter_radio_button->setState(QButton::On);
}

void	CSubviewEdit::HorizontalRadioToggled(bool on)
{
	if (on)
		vertical_radio_button->setState(QButton::Off);
	else
		vertical_radio_button->setState(QButton::On);
}

void	CSubviewEdit::VerticalRadioToggled(bool on)
{
	if (on)
		horizontal_radio_button->setState(QButton::Off);
	else
		horizontal_radio_button->setState(QButton::On);
}

CApplicationEditView::CApplicationEditView(	QWidget *parent,
											COptionEditFrame *oef)
	: CFrameFc(parent, 9, 3), option_edit_frame(oef)
{
	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);
	grid_layout->setRowStretch(3, 0);
	grid_layout->setRowStretch(4, 0);
	grid_layout->setRowStretch(5, 0);
	grid_layout->setRowStretch(6, 0);
	grid_layout->setRowStretch(7, 0);
	grid_layout->setRowStretch(8, 1);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 0);
	grid_layout->setColStretch(2, 1);

	select_main_font_button		= new CPushButtonFc("Main Font", this);
	default_main_font_button	= new CPushButtonFc("Default", this);
	main_font_line_edit			= new CLineEditFc(this, "Main Font Line Edit");
	select_main_palette_button	= new CPushButtonFc("Main Palette", this);
	default_main_palette_button	= new CPushButtonFc("Default", this);
	copy_main_palette_button	= new CPushButtonFc("Copy", this);
	paste_main_palette_button	= new CPushButtonFc("Paste", this);
	spacer1						= new CSpacerFc(this);

	main_scrollview				= new QScrollView(this);
	main_palette_label			= new QLabel(main_scrollview->viewport());

	main_scrollview->addChild(main_palette_label);
	main_scrollview->setResizePolicy(QScrollView::AutoOneFit);

	select_help_font_button		= new CPushButtonFc("Help Font", this);
	default_help_font_button	= new CPushButtonFc("Default", this);
	help_font_line_edit			= new CLineEditFc(this, "Help Font Line Edit");
	select_help_palette_button	= new CPushButtonFc("Help Palette", this);
	default_help_palette_button	= new CPushButtonFc("Default", this);
	copy_help_palette_button	= new CPushButtonFc("Copy", this);
	paste_help_palette_button	= new CPushButtonFc("Paste", this);
	spacer2						= new CSpacerFc(this);

	help_scrollview				= new QScrollView(this);
	help_palette_label			= new QLabel(help_scrollview->viewport());

	help_scrollview->addChild(help_palette_label);
	help_scrollview->setResizePolicy(QScrollView::AutoOneFit);

	tab_widget					= new QTabWidget(this);
	auxiliary_palette_view		= new CAuxiliaryPaletteView(tab_widget, oef);
	auxiliary_color_view		= new CAuxiliaryColorView(tab_widget, oef);
	auxiliary_brush_view		= new CAuxiliaryBrushView(tab_widget, oef);
	auxiliary_xpm_image_view	= new CAuxiliaryXpmImageView(tab_widget, oef);
	auxiliary_font_view			= new CAuxiliaryFontView(tab_widget, oef);
	auxiliary_string_view		= new CAuxiliaryStringView(tab_widget, oef);

	auxiliary_palette_view->SetAuxiliaryColorView(auxiliary_color_view);
	auxiliary_brush_view->SetAuxiliaryColorView(auxiliary_color_view);
	auxiliary_brush_view->SetAuxiliaryXpmImageView(auxiliary_xpm_image_view);

	tab_widget->addTab(auxiliary_palette_view, "Auxiliary Palette");
	tab_widget->addTab(auxiliary_color_view, "Auxiliary Color");
	tab_widget->addTab(auxiliary_brush_view, "Auxiliary Brush");
	tab_widget->addTab(auxiliary_xpm_image_view, "Auxiliary Xpm Image");
	tab_widget->addTab(auxiliary_font_view, "Auxiliary Font");
	tab_widget->addTab(auxiliary_string_view, "Auxiliary String");

	grid_layout->addWidget(select_main_font_button,			0, 0);
	grid_layout->addWidget(default_main_font_button,		0, 1);
	grid_layout->addWidget(main_font_line_edit,				0, 2);
	grid_layout->addWidget(select_main_palette_button,		1, 0);
	grid_layout->addWidget(default_main_palette_button,		1, 1);
	grid_layout->addWidget(copy_main_palette_button,		2, 0);
	grid_layout->addWidget(paste_main_palette_button,		2, 1);
	grid_layout->addMultiCellWidget(spacer1,				3, 3, 0, 1);
	grid_layout->addMultiCellWidget(main_scrollview,		1, 3, 2, 2);
	grid_layout->addWidget(select_help_font_button,			4, 0);
	grid_layout->addWidget(default_help_font_button,		4, 1);
	grid_layout->addWidget(help_font_line_edit,				4, 2);
	grid_layout->addWidget(select_help_palette_button,		5, 0);
	grid_layout->addWidget(default_help_palette_button,		5, 1);
	grid_layout->addWidget(copy_help_palette_button,		6, 0);
	grid_layout->addWidget(paste_help_palette_button,		6, 1);
	grid_layout->addMultiCellWidget(spacer2,				7, 7, 0, 1);
	grid_layout->addMultiCellWidget(help_scrollview,		5, 7, 2, 2);
	grid_layout->addMultiCellWidget(tab_widget,				8, 8, 0, 2);

	QObject::connect(	select_main_font_button,
						SIGNAL(clicked()),
						this,
						SLOT(SelectMainFont()));

	QObject::connect(	default_main_font_button,
						SIGNAL(clicked()),
						this,
						SLOT(DefaultMainFont()));

	QObject::connect(	select_main_palette_button,
						SIGNAL(clicked()),
						this,
						SLOT(SelectMainPalette()));

	QObject::connect(	default_main_palette_button,
						SIGNAL(clicked()),
						this,
						SLOT(DefaultMainPalette()));

	QObject::connect(	copy_main_palette_button,
						SIGNAL(clicked()),
						this,
						SLOT(CopyMainPalette()));

	QObject::connect(	paste_main_palette_button,
						SIGNAL(clicked()),
						this,
						SLOT(PasteMainPalette()));

	QObject::connect(	select_help_font_button,
						SIGNAL(clicked()),
						this,
						SLOT(SelectHelpFont()));

	QObject::connect(	default_help_font_button,
						SIGNAL(clicked()),
						this,
						SLOT(DefaultHelpFont()));

	QObject::connect(	select_help_palette_button,
						SIGNAL(clicked()),
						this,
						SLOT(SelectHelpPalette()));

	QObject::connect(	default_help_palette_button,
						SIGNAL(clicked()),
						this,
						SLOT(DefaultHelpPalette()));

	QObject::connect(	copy_help_palette_button,
						SIGNAL(clicked()),
						this,
						SLOT(CopyHelpPalette()));

	QObject::connect(	paste_help_palette_button,
						SIGNAL(clicked()),
						this,
						SLOT(PasteHelpPalette()));

	default_palette = new CPaletteData();

	default_palette->SetPrecalculateButtonColor(
		new CPaletteData::SRgbSpecifier(192, 192, 192));

	default_palette->SetPrecalculateBackgroundColor(
		new CPaletteData::SRgbSpecifier(192, 192, 192));
}

CApplicationEditView::~CApplicationEditView()
{
	delete default_palette;
}

void	CApplicationEditView::UpdateOptionEditFrameGeometry()
{
	tab_widget->updateGeometry();
	updateGeometry();
}

void	CApplicationEditView::SetupAuxiliaryListViews()
{
	auxiliary_palette_view->SetupListView("Palette");
	auxiliary_color_view->SetupListView("Color");
	auxiliary_brush_view->SetupListView("Brush");
	auxiliary_xpm_image_view->SetupListView("XpmImage");
	auxiliary_font_view->SetupListView("Font");
	auxiliary_string_view->SetupListView("String");
}

void	CApplicationEditView::SetOptionData(SOptionFileItem *ofi,
											COptionData *d)
{
	option_file_item	= ofi;
	option_data			= d;

	application_data	= option_data->GetApplicationData();
	help_data			= option_data->GetHelpData();

	DisplayMainOptionData();
	DisplayHelpOptionData();

	auxiliary_palette_view->SetOptionData(option_data);
	auxiliary_color_view->SetOptionData(option_data);
	auxiliary_brush_view->SetOptionData(option_data);
	auxiliary_xpm_image_view->SetOptionData(option_data);
	auxiliary_font_view->SetOptionData(option_data);
	auxiliary_string_view->SetOptionData(option_data);
}

void	CApplicationEditView::DisplayMainOptionData()
{
	CListViewPaletteDemo	*palette_demo = new CListViewPaletteDemo(this);
	QFont					demo_font;

	if (application_data->GetFontString().length())
	{
		main_font_line_edit->setText(application_data->GetFontString().c_str());

		FontSpecifier::SetFontSpecifier(demo_font,
										application_data->GetFontString());

		palette_demo->setFont(demo_font);
	}
	else
	{
		main_font_line_edit->setText(font().rawName());

		palette_demo->setFont(font());
	}

	palette_demo->CalculateGeometry();

	main_scrollview->setMinimumHeight(palette_demo->height() + 4);
	main_scrollview->setMaximumHeight(palette_demo->height() + 4);

	if (application_data->GetPaletteData())
		palette_demo->SetDemoPalette(application_data->GetPaletteData());
	else
		palette_demo->SetDemoPalette(default_palette);

	main_palette_label->setPixmap(QPixmap::grabWidget(palette_demo));

	delete palette_demo;
}

void	CApplicationEditView::DisplayHelpOptionData()
{
	CListViewPaletteDemo	*palette_demo = new CListViewPaletteDemo(this);
	QFont					demo_font;

	if (help_data->GetFontString().length())
	{
		help_font_line_edit->setText(help_data->GetFontString().c_str());

		FontSpecifier::SetFontSpecifier(demo_font,
										help_data->GetFontString());

		palette_demo->setFont(demo_font);
	}
	else
	{
		help_font_line_edit->setText(font().rawName());

		palette_demo->setFont(font());
	}

	palette_demo->CalculateGeometry();

	help_scrollview->setMinimumHeight(palette_demo->height() + 4);
	help_scrollview->setMaximumHeight(palette_demo->height() + 4);

	if (help_data->GetPaletteData())
		palette_demo->SetDemoPalette(help_data->GetPaletteData());
	else
		palette_demo->SetDemoPalette(default_palette);

	help_palette_label->setPixmap(QPixmap::grabWidget(palette_demo));

	delete palette_demo;
}

void	CApplicationEditView::SelectMainFont()
{
	CMainviewTreeFontDialog	*mtfd = OptionDialog::mainview_tree_font_dialog;

	string		value;

	mtfd->SetTitleAndResize("Select Font");
	mtfd->SetValuePointer(&value);
	mtfd->SetNameComboBox(auxiliary_font_view);
	mtfd->SetValueLineEdit(main_font_line_edit->text());

	if (mtfd->exec() == QDialog::Accepted)
	{
		option_edit_frame->CreateUndoOptionData();

		application_data->SetFontString(value);

		DisplayMainOptionData();
	}
}

void	CApplicationEditView::DefaultMainFont()
{
	option_edit_frame->CreateUndoOptionData();

	application_data->SetFontString(string());

	DisplayMainOptionData();
}

void	CApplicationEditView::SelectMainPalette()
{
	CMainviewTreePaletteDialog	*mtpd =
		OptionDialog::mainview_tree_palette_dialog;

	mtpd->SetAuxiliaryColorView(auxiliary_color_view);

	CPaletteData	*value;

	mtpd->SetTitleAndResize("Select Palette");
	mtpd->SetValuePointer(&value);
	mtpd->SetNameComboBox(auxiliary_palette_view);
	mtpd->SetPaletteValue(application_data->GetPaletteData());

	if (mtpd->exec() == QDialog::Accepted)
	{
		option_edit_frame->CreateUndoOptionData();

		application_data->SetPaletteData(value);

		DisplayMainOptionData();
	}
}

void	CApplicationEditView::DefaultMainPalette()
{
	option_edit_frame->CreateUndoOptionData();

	application_data->SetPaletteData(0);

	DisplayMainOptionData();
}

void	CApplicationEditView::CopyMainPalette()
{
	COptionData		*od = new COptionData();

	od->SetApplicationData(new COptionData::SApplicationData());
	od->SetHelpData(new COptionData::SHelpData());

	if (application_data->GetPaletteData())
		od->AddAuxiliaryPalette("Main_Palette",
						new CPaletteData(*application_data->GetPaletteData()));

	od->CopyOptionDataToClipboard();

	delete od;
}

void	CApplicationEditView::PasteMainPalette()
{
	COptionData		*od;

	CParserDriver::StatusCode   sc =
		COptionData::InputOptionDataFromClipboard(&od);

	if (sc == CParserDriver::EndOfFile)
	{
		map<string, CPaletteData *>		*auxiliary_palettes =
			od->GetAuxiliaryPalettes();

		if (auxiliary_palettes && auxiliary_palettes->size())
		{
			option_edit_frame->CreateUndoOptionData();

			application_data->SetPaletteData(
				new CPaletteData(*(*auxiliary_palettes->begin()).second));

			DisplayMainOptionData();
		}

		delete od;
	}
}

void	CApplicationEditView::SelectHelpFont()
{
	CMainviewTreeFontDialog	*mtfd = OptionDialog::mainview_tree_font_dialog;

	string		value;

	mtfd->SetTitleAndResize("Select Font");
	mtfd->SetValuePointer(&value);
	mtfd->SetNameComboBox(auxiliary_font_view);
	mtfd->SetValueLineEdit(help_font_line_edit->text());

	if (mtfd->exec() == QDialog::Accepted)
	{
		option_edit_frame->CreateUndoOptionData();

		help_data->SetFontString(value);

		DisplayHelpOptionData();
	}
}

void	CApplicationEditView::DefaultHelpFont()
{
	option_edit_frame->CreateUndoOptionData();

	help_data->SetFontString(string());

	DisplayHelpOptionData();
}

void	CApplicationEditView::SelectHelpPalette()
{
	CMainviewTreePaletteDialog	*mtpd =
		OptionDialog::mainview_tree_palette_dialog;

	mtpd->SetAuxiliaryColorView(auxiliary_color_view);

	CPaletteData	*value;

	mtpd->SetTitleAndResize("Select Palette");
	mtpd->SetValuePointer(&value);
	mtpd->SetNameComboBox(auxiliary_palette_view);
	mtpd->SetPaletteValue(help_data->GetPaletteData());

	if (mtpd->exec() == QDialog::Accepted)
	{
		option_edit_frame->CreateUndoOptionData();

		help_data->SetPaletteData(value);

		DisplayHelpOptionData();
	}
}

void	CApplicationEditView::DefaultHelpPalette()
{
	option_edit_frame->CreateUndoOptionData();

	help_data->SetPaletteData(0);

	DisplayHelpOptionData();
}

void	CApplicationEditView::CopyHelpPalette()
{
	COptionData		*od = new COptionData();

	od->SetApplicationData(new COptionData::SApplicationData());
	od->SetHelpData(new COptionData::SHelpData());

	if (help_data->GetPaletteData())
		od->AddAuxiliaryPalette("Help_Palette",
						new CPaletteData(*help_data->GetPaletteData()));

	od->CopyOptionDataToClipboard();

	delete od;
}

void	CApplicationEditView::PasteHelpPalette()
{
	COptionData		*od;

	CParserDriver::StatusCode   sc =
		COptionData::InputOptionDataFromClipboard(&od);

	if (sc == CParserDriver::EndOfFile)
	{
		map<string, CPaletteData *>		*auxiliary_palettes =
			od->GetAuxiliaryPalettes();

		if (auxiliary_palettes && auxiliary_palettes->size())
		{
			option_edit_frame->CreateUndoOptionData();

			help_data->SetPaletteData(
				new CPaletteData(*(*auxiliary_palettes->begin()).second));

			DisplayHelpOptionData();
		}

		delete od;
	}
}

CAuxiliaryView::CAuxiliaryView(QWidget *parent, COptionEditFrame *oef)
	: CFrameFc(parent, 2, 8), option_edit_frame(oef)
{
	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 1);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 0);
	grid_layout->setColStretch(2, 0);
	grid_layout->setColStretch(3, 0);
	grid_layout->setColStretch(4, 0);
	grid_layout->setColStretch(5, 0);
	grid_layout->setColStretch(6, 0);
	grid_layout->setColStretch(7, 1);
	
	QString		size_string("#######");

	add_auxiliary_button	= new CPushButtonFc("Add", this, size_string);
	edit_auxiliary_button	= new CPushButtonFc("Edit", this, size_string);
	rename_auxiliary_button	= new CPushButtonFc("Rename", this, size_string);
	delete_auxiliary_button	= new CPushButtonFc("Delete", this, size_string);
	copy_auxiliary_button	= new CPushButtonFc("Copy", this, size_string);
	paste_auxiliary_button	= new CPushButtonFc("Paste", this, size_string);
	auxiliary_name_label	= new CLabelFc("Auxiliary Name", this);
	auxiliary_name_line_edit
		= new CLineEditFc(this, "Auxiliary Name Line Edit");
	auxiliary_list_view		= new QListView(this);

	grid_layout->addWidget(add_auxiliary_button,			0, 0);
	grid_layout->addWidget(edit_auxiliary_button,			0, 1);
	grid_layout->addWidget(rename_auxiliary_button,			0, 2);
	grid_layout->addWidget(delete_auxiliary_button,			0, 3);
	grid_layout->addWidget(copy_auxiliary_button,			0, 4);
	grid_layout->addWidget(paste_auxiliary_button,			0, 5);
	grid_layout->addWidget(auxiliary_name_label,			0, 6);
	grid_layout->addWidget(auxiliary_name_line_edit,		0, 7);
	grid_layout->addMultiCellWidget(auxiliary_list_view,	1, 1, 0, 7);

	auxiliary_name_label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	QObject::connect(	add_auxiliary_button,
						SIGNAL(clicked()),
						this,
						SLOT(AddAuxiliary()));

	QObject::connect(	edit_auxiliary_button,
						SIGNAL(clicked()),
						this,
						SLOT(EditAuxiliary()));

	QObject::connect(	rename_auxiliary_button,
						SIGNAL(clicked()),
						this,
						SLOT(RenameAuxiliary()));

	QObject::connect(	delete_auxiliary_button,
						SIGNAL(clicked()),
						this,
						SLOT(DeleteAuxiliary()));

	QObject::connect(	copy_auxiliary_button,
						SIGNAL(clicked()),
						this,
						SLOT(CopyAuxiliary()));

	QObject::connect(	paste_auxiliary_button,
						SIGNAL(clicked()),
						this,
						SLOT(PasteAuxiliary()));

	QObject::connect(	auxiliary_name_line_edit,
						SIGNAL(returnPressed()),
						this,
						SLOT(NameEntry()));

	QObject::connect(	auxiliary_list_view,
						SIGNAL(currentChanged(QListViewItem *)),
						this,
						SLOT(CurrentItemChanged(QListViewItem *)));
}

void	CAuxiliaryView::SetupListView(const string &column1)
{
	string			column0("Name");

	auxiliary_list_view->addColumn(column0.c_str());
	auxiliary_list_view->addColumn(column1.c_str());
	auxiliary_list_view->setSorting(0);
	auxiliary_list_view->setRootIsDecorated(false);
	auxiliary_list_view->setSelectionMode(QListView::Extended);
}

void	CAuxiliaryView::NameEntry()
{
	QListViewItem	*lvi = auxiliary_list_view->findItem(
		auxiliary_name_line_edit->text(), 0);

	if (lvi)
	{
		auxiliary_list_view->setSelected(lvi, true);
		auxiliary_list_view->setCurrentItem(lvi);
	}
}

void	CAuxiliaryView::CurrentItemChanged(QListViewItem *item)
{
	auxiliary_name_line_edit->setText(item->text(0));
}

void    CAuxiliaryPaletteView::UpdateListView(const string *s)
{
	auxiliary_name_line_edit->clear();
	auxiliary_list_view->clear();

	auxiliary_palettes = option_data->GetAuxiliaryPalettes();

	if (!auxiliary_palettes || !auxiliary_palettes->size())
		return;

	QListViewItem	*lvi;
	QListViewItem	*highlight_item = 0;

	map<string, CPaletteData *>::iterator	m_iterator =
		auxiliary_palettes->begin();

	CListViewPaletteDemo	*listview_demo = new CListViewPaletteDemo(this);

	listview_demo->setFont(font());
	listview_demo->CalculateGeometry();

	while (m_iterator != auxiliary_palettes->end())
	{
		listview_demo->SetDemoPalette((*m_iterator).second);

		lvi = new QListViewItem(auxiliary_list_view,
								(*m_iterator).first.c_str());

		lvi->setPixmap(1, QPixmap::grabWidget(listview_demo));

		if (s && (*m_iterator).first == *s)
			highlight_item = lvi;

		m_iterator++;
	}

	delete listview_demo;

	if (!highlight_item)
	{
		auxiliary_name_line_edit->setText(
			auxiliary_list_view->firstChild()->text(0));

		auxiliary_list_view->setSelected(
			auxiliary_list_view->firstChild(), true);

		auxiliary_list_view->setCurrentItem(auxiliary_list_view->firstChild());
	}
	else
	{
		auxiliary_name_line_edit->setText(highlight_item->text(0));
		auxiliary_list_view->setSelected(highlight_item, true);
		auxiliary_list_view->setCurrentItem(highlight_item);
	}
}

bool    CAuxiliaryPaletteView::AddAuxiliaryPalette(	const string &name,
													CPaletteData *value)
{
	CMessageDialog		*msd = OptionDialog::message_dialog;
	ostrstream			message;

	QListViewItem	*lvi =
		auxiliary_list_view->findItem(name.c_str(), 0);

	if (lvi)
	{
		message	<< "Auxiliary palette name = " << name
				<< " already exists." << endl
				<< "Replace with new value ?" << '\0';
	}
	else
	{
		message	<< "Add new auxiliary palette name = "
				<< name << " ?" << '\0';
	}

	msd->SetTitleAndMessageText("Add Auxiliary Palette",
								message.str());

	if (msd->ExecuteDialog() == QDialog::Rejected)
		return false;

	option_edit_frame->CreateUndoOptionData();

	option_data->AddAuxiliaryPalette(name, value);
	UpdateListView(&name);

	return true;
}

void	CAuxiliaryPaletteView::AddAuxiliary()
{
	CAuxiliaryPaletteDialog	*apd = OptionDialog::auxiliary_palette_dialog;
	CMessageDialog			*msd = OptionDialog::message_dialog;

	apd->SetAuxiliaryColorView(auxiliary_color_view);

	string			name;
	CPaletteData	*value;

	apd->SetNameLineEdit(auxiliary_name_line_edit->text());

	QListViewItem	*current_item = auxiliary_list_view->currentItem();

	if (current_item)
	{
		string	current_name =
						(const char *)(current_item->text(0).local8Bit());

		apd->SetPaletteValue(option_data->GetAuxiliaryPalette(current_name));
	}
	else
		apd->SetPaletteValue(0);

	apd->SetTitleAndResize("Add Auxiliary Palette");
	apd->SetNamePointer(&name);
	apd->SetValuePointer(&value);

	if (apd->exec() == QDialog::Accepted)
	{
		QListViewItem	*lvi = auxiliary_list_view->findItem(name.c_str(), 0);

		if (lvi)
		{
			ostrstream		message;

			message	<< "Auxiliary palette name = " << name
					<< " already exists." << endl
					<< "Replace with new value ?" << '\0';

			msd->SetTitleAndMessageText("Add Auxiliary Palette", message.str());

			if (msd->ExecuteDialog() == QDialog::Rejected)
			{
				delete value;

				return;
			}
		}

		option_edit_frame->CreateUndoOptionData();

		option_data->AddAuxiliaryPalette(name, value);
		UpdateListView(&name);
	}
}

void	CAuxiliaryPaletteView::EditAuxiliary()
{
	if (!auxiliary_palettes || !auxiliary_palettes->size())
		return;

	QListViewItem	*current_item = auxiliary_list_view->currentItem();

	if (!current_item)
		return;

	CAuxiliaryPaletteDialog	*apd = OptionDialog::auxiliary_palette_dialog;
	CMessageDialog			*msd = OptionDialog::message_dialog;

	apd->SetAuxiliaryColorView(auxiliary_color_view);

	string			name;
	CPaletteData	*value;

	apd->SetNameLineEdit(current_item->text(0));

	string		current_name =
						(const char *)(current_item->text(0).local8Bit());

	apd->SetPaletteValue(option_data->GetAuxiliaryPalette(current_name));
	apd->SetTitleAndResize("Edit Auxiliary Palette");
	apd->SetNamePointer(&name);
	apd->SetValuePointer(&value);

	if (apd->exec() == QDialog::Accepted)
	{
		if (apd->GetNameChangedFlag())
		{
			ostrstream		message;

			QListViewItem	*lvi =
				auxiliary_list_view->findItem(name.c_str(), 0);

			if (lvi)
			{
				message	<< "Auxiliary palette name = " << name
						<< " already exists." << endl
						<< "Replace with new value ?" << '\0';
			}
			else
			{
				message	<< "Add new auxiliary palette name = "
						<< name << " ?" << '\0';
			}

			msd->SetTitleAndMessageText("Edit Auxiliary Palette",
															message.str());

			if (msd->ExecuteDialog() == QDialog::Rejected)
			{
				delete value;

				return;
			}

			option_edit_frame->CreateUndoOptionData();

			option_data->AddAuxiliaryPalette(name, value);
			UpdateListView(&name);
		}
		else if (apd->GetValueChangedFlag())
		{
			option_edit_frame->CreateUndoOptionData();

			option_data->AddAuxiliaryPalette(name, value);
			UpdateListView(&name);
		}
	}
}

void	CAuxiliaryPaletteView::RenameAuxiliary()
{
	if (!auxiliary_palettes || !auxiliary_palettes->size())
		return;

	QListViewItem	*current_item = auxiliary_list_view->currentItem();
	string			current_name;

	if (current_item)
		current_name = (const char *)(current_item->text(0).local8Bit());
	else
		return;

	CAuxiliaryPaletteDialog	*apd = OptionDialog::auxiliary_palette_dialog;
	CMessageDialog			*msd = OptionDialog::message_dialog;

	apd->SetAuxiliaryColorView(auxiliary_color_view);

	string			name;
	CPaletteData	*value;

	apd->SetNameLineEdit(current_item->text(0));
	apd->SetPaletteValue(option_data->GetAuxiliaryPalette(current_name));
	apd->SetTitleAndResize("Rename Auxiliary Palette");
	apd->SetNamePointer(&name);
	apd->SetValuePointer(&value);

	if (apd->exec() == QDialog::Accepted)
	{
		if (apd->GetNameChangedFlag())
		{
			QListViewItem	*lvi =
				auxiliary_list_view->findItem(name.c_str(), 0);

			if (lvi)
			{
				ostrstream		message;

				message	<< "Auxiliary palette name = " << name
						<< " already exists." << endl
						<< "Replace with new value ?" << '\0';

				msd->SetTitleAndMessageText("Rename Auxiliary Palette",
											message.str());

				if (msd->ExecuteDialog() == QDialog::Rejected)
				{
					delete value;

					return;
				}
			}

			option_edit_frame->CreateUndoOptionData();

			option_data->DeleteAuxiliaryPalette(current_name);
			option_data->AddAuxiliaryPalette(name, value);
			UpdateListView(&name);
		}
	}
}

void	CAuxiliaryPaletteView::DeleteAuxiliary()
{
	if (!auxiliary_palettes || !auxiliary_palettes->size())
		return;

	CopyAuxiliary();

	QListViewItem	*list_view_item		= auxiliary_list_view->firstChild();
	bool			create_undo_flag	= false;
	string			item_name;

	while (list_view_item)
	{
		if (list_view_item->isSelected())
		{
			item_name = (const char *)(list_view_item->text(0).local8Bit());

			if (!create_undo_flag)
			{
				create_undo_flag = true;

				option_edit_frame->CreateUndoOptionData();
			}

			option_data->DeleteAuxiliaryPalette(item_name);
		}

		list_view_item = list_view_item->itemBelow();
	}

	if (!create_undo_flag)
		return;

	QListViewItem	*current_item	= auxiliary_list_view->currentItem();
	QListViewItem	*next_item		= current_item->itemBelow();
	string			next_name;

	if (!next_item)
		next_item = current_item->itemAbove();

	if (next_item)
	{
		next_name = (const char *)(next_item->text(0).local8Bit());

		UpdateListView(&next_name);
	}
	else
		UpdateListView(0);
}

void	CAuxiliaryPaletteView::CopyAuxiliary()
{
	COptionData		*od = new COptionData();

	od->SetApplicationData(new COptionData::SApplicationData());
	od->SetHelpData(new COptionData::SHelpData());

	QListViewItem	*list_view_item = auxiliary_list_view->firstChild();
	string			item_name;

	while (list_view_item)
	{
		if (list_view_item->isSelected())
		{
			item_name = (const char *)(list_view_item->text(0).local8Bit());

			CPaletteData	*palette_data =
				option_data->GetAuxiliaryPalette(item_name);

			if (palette_data)
				od->AddAuxiliaryPalette(item_name,
					new CPaletteData(*palette_data));
		}

		list_view_item = list_view_item->itemBelow();
	}

	od->CopyOptionDataToClipboard();

	delete od;
}

void	CAuxiliaryPaletteView::PasteAuxiliary()
{
	COptionData		*od;

	CParserDriver::StatusCode   sc =
		COptionData::InputOptionDataFromClipboard(&od);

	if (sc == CParserDriver::EndOfFile)
	{
		CMessageDialog					*msd = OptionDialog::message_dialog;
		map<string, CPaletteData *>		*ap = od->GetAuxiliaryPalettes();

		if (!ap || !ap->size())
		{
			delete od;

			return;
		}

		bool	create_undo_flag = false;

		map<string, CPaletteData *>::iterator	m_iterator = ap->begin();

		while (m_iterator != ap->end())
		{
			string			name((*m_iterator).first);
			CPaletteData	*pd((*m_iterator).second);

			QListViewItem	*lvi =
				auxiliary_list_view->findItem(name.c_str(), 0);

			if (lvi)
			{
				ostrstream		message;

				message	<< "Auxiliary palette name = " << name
						<< " already exists." << endl
						<< "Replace with new value ?" << '\0';

				msd->SetTitleAndMessageText(
					"Paste Auxiliary Palette", message.str());

				if (msd->ExecuteDialog() == QDialog::Accepted)
				{
					if (!create_undo_flag)
					{
						create_undo_flag = true;

						option_edit_frame->CreateUndoOptionData();
					}

					option_data->AddAuxiliaryPalette(
												name, new CPaletteData(*pd));
				}
			}
			else
			{
				if (!create_undo_flag)
				{
					create_undo_flag = true;

					option_edit_frame->CreateUndoOptionData();
				}

				option_data->AddAuxiliaryPalette(name, new CPaletteData(*pd));
			}

			m_iterator++;
		}

		UpdateListView(0);

		delete od;
	}
}

void    CAuxiliaryColorView::UpdateListView(const string *s)
{
	auxiliary_name_line_edit->clear();
	auxiliary_list_view->clear();

	auxiliary_colors = option_data->GetAuxiliaryColors();

	if (!auxiliary_colors || !auxiliary_colors->size())
		return;

	QListViewItem	*lvi;
	QListViewItem	*highlight_item = 0;

	map<string, CPaletteData::SColorSpecifier *>::iterator	m_iterator =
		auxiliary_colors->begin();

	CLabelFc	*color_display_label =
		new CLabelFc(	"0123456789012345678901234567890123456789",
						this, false, true);

	color_display_label->CalculateGeometry();

	while (m_iterator != auxiliary_colors->end())
	{
		ostrstream		specifier_text;

		specifier_text << " " << *(*m_iterator).second << '\0';

		color_display_label->setText(specifier_text.str());

		QColor		specifier_color;

		(*m_iterator).second->SetupColor(specifier_color);

		color_display_label->setPaletteBackgroundColor(specifier_color);

		lvi = new QListViewItem(auxiliary_list_view,
								(*m_iterator).first.c_str());

		lvi->setPixmap(1, QPixmap::grabWidget(color_display_label));

		if (s && (*m_iterator).first == *s)
			highlight_item = lvi;

		m_iterator++;
	}

	delete color_display_label;

	if (!highlight_item)
	{
		auxiliary_name_line_edit->setText(
			auxiliary_list_view->firstChild()->text(0));

		auxiliary_list_view->setSelected(
			auxiliary_list_view->firstChild(), true);

		auxiliary_list_view->setCurrentItem(auxiliary_list_view->firstChild());
	}
	else
	{
		auxiliary_name_line_edit->setText(highlight_item->text(0));
		auxiliary_list_view->setSelected(highlight_item, true);
		auxiliary_list_view->setCurrentItem(highlight_item);
	}
}

bool    CAuxiliaryColorView::AddAuxiliaryColor(const string &name,
										CPaletteData::SColorSpecifier *value)
{
	CMessageDialog		*msd = OptionDialog::message_dialog;
	ostrstream			message;

	QListViewItem	*lvi =
		auxiliary_list_view->findItem(name.c_str(), 0);

	if (lvi)
	{
		message	<< "Auxiliary color name = " << name
				<< " already exists." << endl
				<< "Replace with value = " << *value << "?" << '\0';
	}
	else
	{
		message	<< "Add new auxiliary color name = " << name
				<< "," << endl
				<< "value = " << *value << "?" << '\0';
	}

	msd->SetTitleAndMessageText("Add Auxiliary Color",
								message.str());

	if (msd->ExecuteDialog() == QDialog::Rejected)
		return false;

	option_edit_frame->CreateUndoOptionData();

	option_data->AddAuxiliaryColor(name, value);
	UpdateListView(&name);

	return true;
}

void	CAuxiliaryColorView::AddAuxiliary()
{
	CAuxiliaryColorDialog	*acd = OptionDialog::auxiliary_color_dialog;
	CMessageDialog			*msd = OptionDialog::message_dialog;

	string							name;
	CPaletteData::SColorSpecifier	*value;

	acd->SetNameLineEdit(auxiliary_name_line_edit->text());

	QListViewItem	*current_item = auxiliary_list_view->currentItem();

	if (current_item)
	{
		string	current_name =
						(const char *)(current_item->text(0).local8Bit());

		acd->SetColorValue(option_data->GetAuxiliaryColor(current_name));
	}
	else
		acd->SetColorValue(0);

	acd->SetTitleAndResize("Add Auxiliary Color");
	acd->SetNamePointer(&name);
	acd->SetValuePointer(&value);

	if (acd->exec() == QDialog::Accepted)
	{
		QListViewItem	*lvi = auxiliary_list_view->findItem(name.c_str(), 0);

		if (lvi)
		{
			ostrstream		message;

			message	<< "Auxiliary color name = " << name
					<< " already exists." << endl
					<< "Replace with value = " << *value << "?" << '\0';

			msd->SetTitleAndMessageText("Add Auxiliary Color", message.str());

			if (msd->ExecuteDialog() == QDialog::Rejected)
			{
				delete value;

				return;
			}
		}

		option_edit_frame->CreateUndoOptionData();

		option_data->AddAuxiliaryColor(name, value);
		UpdateListView(&name);
	}
}

void	CAuxiliaryColorView::EditAuxiliary()
{
	if (!auxiliary_colors || !auxiliary_colors->size())
		return;

	QListViewItem	*current_item = auxiliary_list_view->currentItem();

	if (!current_item)
		return;

	CAuxiliaryColorDialog	*acd = OptionDialog::auxiliary_color_dialog;
	CMessageDialog			*msd = OptionDialog::message_dialog;

	string							name;
	CPaletteData::SColorSpecifier	*value;

	acd->SetNameLineEdit(current_item->text(0));

	string		current_name =
						(const char *)(current_item->text(0).local8Bit());

	acd->SetColorValue(option_data->GetAuxiliaryColor(current_name));
	acd->SetTitleAndResize("Edit Auxiliary Color");
	acd->SetNamePointer(&name);
	acd->SetValuePointer(&value);

	if (acd->exec() == QDialog::Accepted)
	{
		if (acd->GetNameChangedFlag())
		{
			ostrstream		message;

			QListViewItem	*lvi =
				auxiliary_list_view->findItem(name.c_str(), 0);

			if (lvi)
			{
				message	<< "Auxiliary color name = " << name
						<< " already exists." << endl
						<< "Replace with value = " << *value << "?" << '\0';
			}
			else
			{
				message	<< "Add new auxiliary color name = " << name
						<< "," << endl
						<< "value = " << *value << "?" << '\0';
			}

			msd->SetTitleAndMessageText("Edit Auxiliary Color",
															message.str());

			if (msd->ExecuteDialog() == QDialog::Rejected)
			{
				delete value;

				return;
			}

			option_edit_frame->CreateUndoOptionData();

			option_data->AddAuxiliaryColor(name, value);
			UpdateListView(&name);
		}
		else if (acd->GetValueChangedFlag())
		{
			option_edit_frame->CreateUndoOptionData();

			option_data->AddAuxiliaryColor(name, value);
			UpdateListView(&name);
		}
	}
}

void	CAuxiliaryColorView::RenameAuxiliary()
{
	if (!auxiliary_colors || !auxiliary_colors->size())
		return;

	QListViewItem	*current_item = auxiliary_list_view->currentItem();
	string			current_name;

	if (current_item)
		current_name = (const char *)(current_item->text(0).local8Bit());
	else
		return;

	CAuxiliaryColorDialog	*acd = OptionDialog::auxiliary_color_dialog;
	CMessageDialog			*msd = OptionDialog::message_dialog;

	string							name;
	CPaletteData::SColorSpecifier	*value;

	acd->SetNameLineEdit(current_item->text(0));
	acd->SetColorValue(option_data->GetAuxiliaryColor(current_name));
	acd->SetTitleAndResize("Rename Auxiliary Color");
	acd->SetNamePointer(&name);
	acd->SetValuePointer(&value);

	if (acd->exec() == QDialog::Accepted)
	{
		if (acd->GetNameChangedFlag())
		{
			QListViewItem	*lvi =
				auxiliary_list_view->findItem(name.c_str(), 0);

			if (lvi)
			{
				ostrstream		message;

				message	<< "Auxiliary color name = " << name
						<< " already exists." << endl
						<< "Replace with value = " << *value << "?" << '\0';

				msd->SetTitleAndMessageText("Rename Auxiliary Color",
											message.str());

				if (msd->ExecuteDialog() == QDialog::Rejected)
				{
					delete value;

					return;
				}
			}

			option_edit_frame->CreateUndoOptionData();

			option_data->DeleteAuxiliaryColor(current_name);
			option_data->AddAuxiliaryColor(name, value);
			UpdateListView(&name);
		}
	}
}

void	CAuxiliaryColorView::DeleteAuxiliary()
{
	if (!auxiliary_colors || !auxiliary_colors->size())
		return;

	CopyAuxiliary();

	QListViewItem	*list_view_item		= auxiliary_list_view->firstChild();
	bool			create_undo_flag	= false;
	string			item_name;

	while (list_view_item)
	{
		if (list_view_item->isSelected())
		{
			item_name = (const char *)(list_view_item->text(0).local8Bit());

			if (!create_undo_flag)
			{
				create_undo_flag = true;

				option_edit_frame->CreateUndoOptionData();
			}

			option_data->DeleteAuxiliaryColor(item_name);
		}

		list_view_item = list_view_item->itemBelow();
	}

	if (!create_undo_flag)
		return;

	QListViewItem	*current_item	= auxiliary_list_view->currentItem();
	QListViewItem	*next_item		= current_item->itemBelow();
	string			next_name;

	if (!next_item)
		next_item = current_item->itemAbove();

	if (next_item)
	{
		next_name = (const char *)(next_item->text(0).local8Bit());

		UpdateListView(&next_name);
	}
	else
		UpdateListView(0);
}

void	CAuxiliaryColorView::CopyAuxiliary()
{
	COptionData		*od = new COptionData();

	od->SetApplicationData(new COptionData::SApplicationData());
	od->SetHelpData(new COptionData::SHelpData());

	QListViewItem	*list_view_item = auxiliary_list_view->firstChild();
	string			item_name;

	while (list_view_item)
	{
		if (list_view_item->isSelected())
		{
			item_name = (const char *)(list_view_item->text(0).local8Bit());

			CPaletteData::SColorSpecifier	*color_specifier =
				option_data->GetAuxiliaryColor(item_name);

			if (color_specifier)
				od->AddAuxiliaryColor(item_name, color_specifier->clone());
		}

		list_view_item = list_view_item->itemBelow();
	}

	od->CopyOptionDataToClipboard();

	delete od;
}

void	CAuxiliaryColorView::PasteAuxiliary()
{
	COptionData		*od;

	CParserDriver::StatusCode   sc =
		COptionData::InputOptionDataFromClipboard(&od);

	if (sc == CParserDriver::EndOfFile)
	{
		CMessageDialog		*msd = OptionDialog::message_dialog;

		map<string, CPaletteData::SColorSpecifier *>	*ac =
											od->GetAuxiliaryColors();

		if (!ac || !ac->size())
		{
			delete od;

			return;
		}

		bool	create_undo_flag = false;

		map<string, CPaletteData::SColorSpecifier *>::iterator	m_iterator =
																ac->begin();

		while (m_iterator != ac->end())
		{
			string							name((*m_iterator).first);
			CPaletteData::SColorSpecifier	*cs((*m_iterator).second);

			QListViewItem	*lvi =
				auxiliary_list_view->findItem(name.c_str(), 0);

			if (lvi)
			{
				ostrstream		message;

				message	<< "Auxiliary color name = " << name
						<< " already exists." << endl
						<< "Replace with new value ?" << '\0';

				msd->SetTitleAndMessageText(
					"Paste Auxiliary Color", message.str());

				if (msd->ExecuteDialog() == QDialog::Accepted)
				{
					if (!create_undo_flag)
					{
						create_undo_flag = true;

						option_edit_frame->CreateUndoOptionData();
					}

					option_data->AddAuxiliaryColor(name, cs->clone());
				}
			}
			else
			{
				if (!create_undo_flag)
				{
					create_undo_flag = true;

					option_edit_frame->CreateUndoOptionData();
				}

				option_data->AddAuxiliaryColor(name, cs->clone());
			}

			m_iterator++;
		}

		UpdateListView(0);

		delete od;
	}
}

void    CAuxiliaryBrushView::UpdateListView(const string *s)
{
	auxiliary_name_line_edit->clear();
	auxiliary_list_view->clear();

	auxiliary_brushes = option_data->GetAuxiliaryBrushes();

	if (!auxiliary_brushes || !auxiliary_brushes->size())
		return;

	QListViewItem	*lvi;
	QListViewItem	*highlight_item = 0;

	map<string, CPaletteData::SBrushSpecifier *>::iterator	m_iterator =
		auxiliary_brushes->begin();

	CBrushDisplayWidget		*brush_display_widget =
		new CBrushDisplayWidget(this, 0,
								"0123456789012345678901234567890123456789",
								palette().color(QPalette::Active,
												QColorGroup::Base));

	while (m_iterator != auxiliary_brushes->end())
	{
		brush_display_widget->DrawBrush((*m_iterator).second);

		lvi = new QListViewItem(auxiliary_list_view,
								(*m_iterator).first.c_str());

		lvi->setPixmap(1, QPixmap::grabWidget(brush_display_widget));

		if (s && (*m_iterator).first == *s)
			highlight_item = lvi;

		m_iterator++;
	}

	delete brush_display_widget;

	if (!highlight_item)
	{
		auxiliary_name_line_edit->setText(
			auxiliary_list_view->firstChild()->text(0));

		auxiliary_list_view->setSelected(
			auxiliary_list_view->firstChild(), true);

		auxiliary_list_view->setCurrentItem(auxiliary_list_view->firstChild());
	}
	else
	{
		auxiliary_name_line_edit->setText(highlight_item->text(0));
		auxiliary_list_view->setSelected(highlight_item, true);
		auxiliary_list_view->setCurrentItem(highlight_item);
	}
}

void	CAuxiliaryBrushView::AddAuxiliary()
{
	CAuxiliaryBrushDialog	*abd = OptionDialog::auxiliary_brush_dialog;
	CMessageDialog			*msd = OptionDialog::message_dialog;

	abd->SetAuxiliaryColorView(auxiliary_color_view);
	abd->SetAuxiliaryXpmImageView(auxiliary_xpm_image_view);

	string							name;
	CPaletteData::SBrushSpecifier	*value;

	abd->SetNameLineEdit(auxiliary_name_line_edit->text());

	QListViewItem	*current_item = auxiliary_list_view->currentItem();

	if (current_item)
	{
		string	current_name =
						(const char *)(current_item->text(0).local8Bit());

		abd->SetBrushValue(option_data->GetAuxiliaryBrush(current_name));
	}
	else
		abd->SetBrushValue(0);

	abd->SetTitleAndResize("Add Auxiliary Brush");
	abd->SetNamePointer(&name);
	abd->SetValuePointer(&value);

	if (abd->exec() == QDialog::Accepted)
	{
		QListViewItem	*lvi = auxiliary_list_view->findItem(name.c_str(), 0);

		if (lvi)
		{
			ostrstream		message;

			message	<< "Auxiliary brush name = " << name
					<< " already exists." << endl
					<< "Replace with new value ?" << '\0';

			msd->SetTitleAndMessageText("Add Auxiliary Brush", message.str());

			if (msd->ExecuteDialog() == QDialog::Rejected)
			{
				delete value;

				return;
			}
		}

		option_edit_frame->CreateUndoOptionData();

		option_data->AddAuxiliaryBrush(name, value);
		UpdateListView(&name);
	}
}

void	CAuxiliaryBrushView::EditAuxiliary()
{
	if (!auxiliary_brushes || !auxiliary_brushes->size())
		return;

	QListViewItem	*current_item = auxiliary_list_view->currentItem();

	if (!current_item)
		return;

	CAuxiliaryBrushDialog	*abd = OptionDialog::auxiliary_brush_dialog;
	CMessageDialog			*msd = OptionDialog::message_dialog;

	abd->SetAuxiliaryColorView(auxiliary_color_view);
	abd->SetAuxiliaryXpmImageView(auxiliary_xpm_image_view);

	string							name;
	CPaletteData::SBrushSpecifier	*value;

	abd->SetNameLineEdit(current_item->text(0));

	string		current_name =
						(const char *)(current_item->text(0).local8Bit());

	abd->SetBrushValue(option_data->GetAuxiliaryBrush(current_name));
	abd->SetTitleAndResize("Edit Auxiliary Brush");
	abd->SetNamePointer(&name);
	abd->SetValuePointer(&value);

	if (abd->exec() == QDialog::Accepted)
	{
		if (abd->GetNameChangedFlag())
		{
			ostrstream		message;

			QListViewItem	*lvi =
				auxiliary_list_view->findItem(name.c_str(), 0);

			if (lvi)
			{
				message	<< "Auxiliary brush name = " << name
						<< " already exists." << endl
						<< "Replace with new value ?" << '\0';
			}
			else
			{
				message	<< "Add new auxiliary brush name = "
						<< name << " ?" << '\0';
			}

			msd->SetTitleAndMessageText("Edit Auxiliary Brush",
															message.str());

			if (msd->ExecuteDialog() == QDialog::Rejected)
			{
				delete value;

				return;
			}

			option_edit_frame->CreateUndoOptionData();

			option_data->AddAuxiliaryBrush(name, value);
			UpdateListView(&name);
		}
		else if (abd->GetValueChangedFlag())
		{
			option_edit_frame->CreateUndoOptionData();

			option_data->AddAuxiliaryBrush(name, value);
			UpdateListView(&name);
		}
	}
}

void	CAuxiliaryBrushView::RenameAuxiliary()
{
	if (!auxiliary_brushes || !auxiliary_brushes->size())
		return;

	QListViewItem	*current_item = auxiliary_list_view->currentItem();
	string			current_name;

	if (current_item)
		current_name = (const char *)(current_item->text(0).local8Bit());
	else
		return;

	CAuxiliaryBrushDialog	*abd = OptionDialog::auxiliary_brush_dialog;
	CMessageDialog			*msd = OptionDialog::message_dialog;

	abd->SetAuxiliaryColorView(auxiliary_color_view);
	abd->SetAuxiliaryXpmImageView(auxiliary_xpm_image_view);

	string							name;
	CPaletteData::SBrushSpecifier	*value;

	abd->SetNameLineEdit(current_item->text(0));
	abd->SetBrushValue(option_data->GetAuxiliaryBrush(current_name));
	abd->SetTitleAndResize("Rename Auxiliary Brush");
	abd->SetNamePointer(&name);
	abd->SetValuePointer(&value);

	if (abd->exec() == QDialog::Accepted)
	{
		if (abd->GetNameChangedFlag())
		{
			QListViewItem	*lvi =
				auxiliary_list_view->findItem(name.c_str(), 0);

			if (lvi)
			{
				ostrstream		message;

				message	<< "Auxiliary brush name = " << name
						<< " already exists." << endl
						<< "Replace with new value ?" << '\0';

				msd->SetTitleAndMessageText("Rename Auxiliary Brush",
											message.str());

				if (msd->ExecuteDialog() == QDialog::Rejected)
				{
					delete value;

					return;
				}
			}

			option_edit_frame->CreateUndoOptionData();

			option_data->DeleteAuxiliaryBrush(current_name);
			option_data->AddAuxiliaryBrush(name, value);
			UpdateListView(&name);
		}
	}
}

void	CAuxiliaryBrushView::DeleteAuxiliary()
{
	if (!auxiliary_brushes || !auxiliary_brushes->size())
		return;

	CopyAuxiliary();

	QListViewItem	*list_view_item		= auxiliary_list_view->firstChild();
	bool			create_undo_flag	= false;
	string			item_name;

	while (list_view_item)
	{
		if (list_view_item->isSelected())
		{
			item_name = (const char *)(list_view_item->text(0).local8Bit());

			if (!create_undo_flag)
			{
				create_undo_flag = true;

				option_edit_frame->CreateUndoOptionData();
			}

			option_data->DeleteAuxiliaryBrush(item_name);
		}

		list_view_item = list_view_item->itemBelow();
	}

	if (!create_undo_flag)
		return;

	QListViewItem	*current_item	= auxiliary_list_view->currentItem();
	QListViewItem	*next_item		= current_item->itemBelow();
	string			next_name;

	if (!next_item)
		next_item = current_item->itemAbove();

	if (next_item)
	{
		next_name = (const char *)(next_item->text(0).local8Bit());

		UpdateListView(&next_name);
	}
	else
		UpdateListView(0);
}

void	CAuxiliaryBrushView::CopyAuxiliary()
{
	COptionData		*od = new COptionData();

	od->SetApplicationData(new COptionData::SApplicationData());
	od->SetHelpData(new COptionData::SHelpData());

	QListViewItem	*list_view_item = auxiliary_list_view->firstChild();
	string			item_name;

	while (list_view_item)
	{
		if (list_view_item->isSelected())
		{
			item_name = (const char *)(list_view_item->text(0).local8Bit());

			CPaletteData::SBrushSpecifier	*brush_specifier =
				option_data->GetAuxiliaryBrush(item_name);

			if (brush_specifier)
				od->AddAuxiliaryBrush(item_name, brush_specifier->clone());
		}

		list_view_item = list_view_item->itemBelow();
	}

	od->CopyOptionDataToClipboard();

	delete od;
}

void	CAuxiliaryBrushView::PasteAuxiliary()
{
	COptionData		*od;

	CParserDriver::StatusCode   sc =
		COptionData::InputOptionDataFromClipboard(&od);

	if (sc == CParserDriver::EndOfFile)
	{
		CMessageDialog		*msd = OptionDialog::message_dialog;

		map<string, CPaletteData::SBrushSpecifier *>	*ab =
											od->GetAuxiliaryBrushes();

		if (!ab || !ab->size())
		{
			delete od;

			return;
		}

		bool	create_undo_flag = false;

		map<string, CPaletteData::SBrushSpecifier *>::iterator	m_iterator =
																ab->begin();

		while (m_iterator != ab->end())
		{
			string							name((*m_iterator).first);
			CPaletteData::SBrushSpecifier	*bs((*m_iterator).second);

			QListViewItem	*lvi =
				auxiliary_list_view->findItem(name.c_str(), 0);

			if (lvi)
			{
				ostrstream		message;

				message	<< "Auxiliary brush name = " << name
						<< " already exists." << endl
						<< "Replace with new value ?" << '\0';

				msd->SetTitleAndMessageText(
					"Paste Auxiliary Brush", message.str());

				if (msd->ExecuteDialog() == QDialog::Accepted)
				{
					if (!create_undo_flag)
					{
						create_undo_flag = true;

						option_edit_frame->CreateUndoOptionData();
					}

					option_data->AddAuxiliaryBrush(name, bs->clone());
				}
			}
			else
			{
				if (!create_undo_flag)
				{
					create_undo_flag = true;

					option_edit_frame->CreateUndoOptionData();
				}

				option_data->AddAuxiliaryBrush(name, bs->clone());
			}

			m_iterator++;
		}

		UpdateListView(0);

		delete od;
	}
}

void    CAuxiliaryXpmImageView::UpdateListView(const string *s)
{
	auxiliary_name_line_edit->clear();
	auxiliary_list_view->clear();

	auxiliary_xpm_images = option_data->GetAuxiliaryXpmImages();

	if (!auxiliary_xpm_images || !auxiliary_xpm_images->size())
		return;

	QListViewItem	*lvi;
	QListViewItem	*highlight_item = 0;

	map<string, CXpmImage *>::iterator	m_iterator =
		auxiliary_xpm_images->begin();

	while (m_iterator != auxiliary_xpm_images->end())
	{
		lvi = new QListViewItem(auxiliary_list_view,
								(*m_iterator).first.c_str());

		QPixmap		*p = (*m_iterator).second->CreatePixmap();

		if (p)
			lvi->setPixmap(1, *p);
		else
			lvi->setText(1, "Invalid xpm format.");

		if (s && (*m_iterator).first == *s)
			highlight_item = lvi;

		m_iterator++;
	}

	if (!highlight_item)
	{
		auxiliary_name_line_edit->setText(
			auxiliary_list_view->firstChild()->text(0));

		auxiliary_list_view->setSelected(
			auxiliary_list_view->firstChild(), true);

		auxiliary_list_view->setCurrentItem(auxiliary_list_view->firstChild());
	}
	else
	{
		auxiliary_name_line_edit->setText(highlight_item->text(0));
		auxiliary_list_view->setSelected(highlight_item, true);
		auxiliary_list_view->setCurrentItem(highlight_item);
	}
}

bool    CAuxiliaryXpmImageView::AddAuxiliaryXpmImage(	const string &name,
														CXpmImage *value)
{
	CMessageDialog		*msd = OptionDialog::message_dialog;
	ostrstream			message;

	QListViewItem	*lvi =
		auxiliary_list_view->findItem(name.c_str(), 0);

	if (lvi)
	{
		message	<< "Auxiliary XPM image name = " << name
				<< " already exists." << endl
				<< "Replace with new value ?" << '\0';
	}
	else
	{
		message	<< "Add new auxiliary XPM image name = "
				<< name << " ?" << '\0';
	}

	msd->SetTitleAndMessageText("Add Auxiliary XPM Image",
								message.str());

	if (msd->ExecuteDialog() == QDialog::Rejected)
		return false;

	option_edit_frame->CreateUndoOptionData();

	option_data->AddAuxiliaryXpmImage(name, value);
	UpdateListView(&name);

	return true;
}

void	CAuxiliaryXpmImageView::AddAuxiliary()
{
	CAuxiliaryXpmImageDialog	*axid =
		OptionDialog::auxiliary_xpm_image_dialog;

	CMessageDialog		*msd = OptionDialog::message_dialog;

	string		name;
	CXpmImage	*value;

	axid->SetNameLineEdit(auxiliary_name_line_edit->text());

	QListViewItem	*current_item = auxiliary_list_view->currentItem();

	if (current_item)
	{
		string	current_name =
						(const char *)(current_item->text(0).local8Bit());

		axid->SetXpmImageValue(option_data->GetAuxiliaryXpmImage(current_name));
	}
	else
		axid->SetXpmImageValue(0);

	axid->SetTitleAndResize("Add Auxiliary XPM Image");
	axid->SetNamePointer(&name);
	axid->SetValuePointer(&value);

	if (axid->exec() == QDialog::Accepted)
	{
		QListViewItem	*lvi = auxiliary_list_view->findItem(name.c_str(), 0);

		if (lvi)
		{
			ostrstream		message;

			message	<< "Auxiliary XPM image name = " << name
					<< " already exists." << endl
					<< "Replace with new value ?" << '\0';

			msd->SetTitleAndMessageText("Add Auxiliary XPM Image",
										message.str());

			if (msd->ExecuteDialog() == QDialog::Rejected)
			{
				delete value;

				return;
			}
		}

		option_edit_frame->CreateUndoOptionData();

		option_data->AddAuxiliaryXpmImage(name, value);
		UpdateListView(&name);
	}
}

void	CAuxiliaryXpmImageView::EditAuxiliary()
{
	if (!auxiliary_xpm_images || !auxiliary_xpm_images->size())
		return;

	QListViewItem	*current_item = auxiliary_list_view->currentItem();

	if (!current_item)
		return;

	CAuxiliaryXpmImageDialog	*axid =
		OptionDialog::auxiliary_xpm_image_dialog;

	CMessageDialog		*msd = OptionDialog::message_dialog;

	string		name;
	CXpmImage	*value;

	axid->SetNameLineEdit(current_item->text(0));

	string		current_name =
						(const char *)(current_item->text(0).local8Bit());

	axid->SetXpmImageValue(option_data->GetAuxiliaryXpmImage(current_name));
	axid->SetTitleAndResize("Edit Auxiliary XPM Image");
	axid->SetNamePointer(&name);
	axid->SetValuePointer(&value);

	if (axid->exec() == QDialog::Accepted)
	{
		if (axid->GetNameChangedFlag())
		{
			ostrstream		message;

			QListViewItem	*lvi =
				auxiliary_list_view->findItem(name.c_str(), 0);

			if (lvi)
			{
				message	<< "Auxiliary XPM image name = " << name
						<< " already exists." << endl
						<< "Replace with new value ?" << '\0';
			}
			else
			{
				message	<< "Add new auxiliary XPM image name = "
						<< name << " ?" << '\0';
			}

			msd->SetTitleAndMessageText("Edit Auxiliary XPM Image",
															message.str());

			if (msd->ExecuteDialog() == QDialog::Rejected)
			{
				delete value;

				return;
			}

			option_edit_frame->CreateUndoOptionData();

			option_data->AddAuxiliaryXpmImage(name, value);
			UpdateListView(&name);
		}
		else if (axid->GetValueChangedFlag())
		{
			option_edit_frame->CreateUndoOptionData();

			option_data->AddAuxiliaryXpmImage(name, value);
			UpdateListView(&name);
		}
	}
}

void	CAuxiliaryXpmImageView::RenameAuxiliary()
{
	if (!auxiliary_xpm_images || !auxiliary_xpm_images->size())
		return;

	QListViewItem	*current_item = auxiliary_list_view->currentItem();
	string			current_name;

	if (current_item)
		current_name = (const char *)(current_item->text(0).local8Bit());
	else
		return;

	CAuxiliaryXpmImageDialog	*axid =
		OptionDialog::auxiliary_xpm_image_dialog;

	CMessageDialog		*msd = OptionDialog::message_dialog;

	string		name;
	CXpmImage	*value;

	axid->SetNameLineEdit(current_item->text(0));
	axid->SetXpmImageValue(option_data->GetAuxiliaryXpmImage(current_name));
	axid->SetTitleAndResize("Rename Auxiliary XPM Image");
	axid->SetNamePointer(&name);
	axid->SetValuePointer(&value);

	if (axid->exec() == QDialog::Accepted)
	{
		if (axid->GetNameChangedFlag())
		{
			QListViewItem	*lvi =
				auxiliary_list_view->findItem(name.c_str(), 0);

			if (lvi)
			{
				ostrstream		message;

				message	<< "Auxiliary XPM image name = " << name
						<< " already exists." << endl
						<< "Replace with new value ?" << '\0';

				msd->SetTitleAndMessageText("Rename Auxiliary XPM Image",
											message.str());

				if (msd->ExecuteDialog() == QDialog::Rejected)
				{
					delete value;

					return;
				}
			}

			option_edit_frame->CreateUndoOptionData();

			option_data->DeleteAuxiliaryXpmImage(current_name);
			option_data->AddAuxiliaryXpmImage(name, value);
			UpdateListView(&name);
		}
	}
}

void	CAuxiliaryXpmImageView::DeleteAuxiliary()
{
	if (!auxiliary_xpm_images || !auxiliary_xpm_images->size())
		return;

	CopyAuxiliary();

	QListViewItem	*list_view_item		= auxiliary_list_view->firstChild();
	bool			create_undo_flag	= false;
	string			item_name;

	while (list_view_item)
	{
		if (list_view_item->isSelected())
		{
			item_name = (const char *)(list_view_item->text(0).local8Bit());

			if (!create_undo_flag)
			{
				create_undo_flag = true;

				option_edit_frame->CreateUndoOptionData();
			}

			option_data->DeleteAuxiliaryXpmImage(item_name);
		}

		list_view_item = list_view_item->itemBelow();
	}

	if (!create_undo_flag)
		return;

	QListViewItem	*current_item	= auxiliary_list_view->currentItem();
	QListViewItem	*next_item		= current_item->itemBelow();
	string			next_name;

	if (!next_item)
		next_item = current_item->itemAbove();

	if (next_item)
	{
		next_name = (const char *)(next_item->text(0).local8Bit());

		UpdateListView(&next_name);
	}
	else
		UpdateListView(0);
}

void	CAuxiliaryXpmImageView::CopyAuxiliary()
{
	COptionData		*od = new COptionData();

	od->SetApplicationData(new COptionData::SApplicationData());
	od->SetHelpData(new COptionData::SHelpData());

	QListViewItem	*list_view_item = auxiliary_list_view->firstChild();
	string			item_name;

	while (list_view_item)
	{
		if (list_view_item->isSelected())
		{
			item_name = (const char *)(list_view_item->text(0).local8Bit());

			CXpmImage		*xpm_image_data =
				option_data->GetAuxiliaryXpmImage(item_name);

			if (xpm_image_data)
				od->AddAuxiliaryXpmImage(item_name,
					new CXpmImage(*xpm_image_data));
		}

		list_view_item = list_view_item->itemBelow();
	}

	od->CopyOptionDataToClipboard();

	delete od;
}

void	CAuxiliaryXpmImageView::PasteAuxiliary()
{
	COptionData		*od;

	CParserDriver::StatusCode   sc =
		COptionData::InputOptionDataFromClipboard(&od);

	if (sc == CParserDriver::EndOfFile)
	{
		CMessageDialog					*msd = OptionDialog::message_dialog;
		map<string, CXpmImage *>		*axi = od->GetAuxiliaryXpmImages();

		if (!axi || !axi->size())
		{
			delete od;

			return;
		}

		bool	create_undo_flag = false;

		map<string, CXpmImage *>::iterator	m_iterator = axi->begin();

		while (m_iterator != axi->end())
		{
			string			name((*m_iterator).first);
			CXpmImage		*xi((*m_iterator).second);

			QListViewItem	*lvi =
				auxiliary_list_view->findItem(name.c_str(), 0);

			if (lvi)
			{
				ostrstream		message;

				message	<< "Auxiliary XPM image name = " << name
						<< " already exists." << endl
						<< "Replace with new value ?" << '\0';

				msd->SetTitleAndMessageText(
					"Paste Auxiliary XPM Image", message.str());

				if (msd->ExecuteDialog() == QDialog::Accepted)
				{
					if (!create_undo_flag)
					{
						create_undo_flag = true;

						option_edit_frame->CreateUndoOptionData();
					}

					option_data->AddAuxiliaryXpmImage(
												name, new CXpmImage(*xi));
				}
			}
			else
			{
				if (!create_undo_flag)
				{
					create_undo_flag = true;

					option_edit_frame->CreateUndoOptionData();
				}

				option_data->AddAuxiliaryXpmImage(name, new CXpmImage(*xi));
			}

			m_iterator++;
		}

		UpdateListView(0);

		delete od;
	}
}

void    CAuxiliaryFontView::UpdateListView(const string *s)
{
	auxiliary_name_line_edit->clear();
	auxiliary_list_view->clear();

	auxiliary_fonts = option_data->GetAuxiliaryFonts();

	if (!auxiliary_fonts || !auxiliary_fonts->size())
		return;

	QListViewItem	*lvi;
	QListViewItem	*highlight_item = 0;

	map<string, string>::iterator	m_iterator = auxiliary_fonts->begin();

	while (m_iterator != auxiliary_fonts->end())
	{
		lvi = new QListViewItem(auxiliary_list_view,
								(*m_iterator).first.c_str(),
								(*m_iterator).second.c_str());

		if (s && (*m_iterator).first == *s)
			highlight_item = lvi;

		m_iterator++;
	}

	if (!highlight_item)
	{
		auxiliary_name_line_edit->setText(
			auxiliary_list_view->firstChild()->text(0));

		auxiliary_list_view->setSelected(
			auxiliary_list_view->firstChild(), true);

		auxiliary_list_view->setCurrentItem(auxiliary_list_view->firstChild());
	}
	else
	{
		auxiliary_name_line_edit->setText(highlight_item->text(0));
		auxiliary_list_view->setSelected(highlight_item, true);
		auxiliary_list_view->setCurrentItem(highlight_item);
	}
}

bool	CAuxiliaryFontView::AddAuxiliaryFont(	const string &name,
												const string &value)
{
	CMessageDialog		*msd = OptionDialog::message_dialog;
	ostrstream			message;

	QListViewItem	*lvi =
		auxiliary_list_view->findItem(name.c_str(), 0);

	if (lvi)
	{
		message	<< "Auxiliary font name = " << name
				<< " already exists." << endl
				<< "Replace with value = " << value << "?" << '\0';
	}
	else
	{
		message	<< "Add new auxiliary font name = " << name
				<< "," << endl
				<< "value = " << value << "?" << '\0';
	}

	msd->SetTitleAndMessageText("Add Auxiliary Font",
								message.str());

	if (msd->ExecuteDialog() == QDialog::Rejected)
		return false;

	option_edit_frame->CreateUndoOptionData();

	option_data->AddAuxiliaryFont(name, value);
	UpdateListView(&name);

	return true;
}

void	CAuxiliaryFontView::AddAuxiliary()
{
	CAuxiliaryFontDialog	*afd = OptionDialog::auxiliary_font_dialog;
	CMessageDialog			*msd = OptionDialog::message_dialog;
	string					name;
	string					value;

	afd->SetNameLineEdit(auxiliary_name_line_edit->text());

	QListViewItem	*current_item = auxiliary_list_view->currentItem();

	if (current_item)
		afd->SetValueLineEdit(current_item->text(1));
	else
		afd->SetValueLineEdit(QString());

	afd->SetTitleAndResize("Add Auxiliary Font");
	afd->SetNamePointer(&name);
	afd->SetValuePointer(&value);

	if (afd->exec() == QDialog::Accepted)
	{
		QListViewItem	*lvi = auxiliary_list_view->findItem(name.c_str(), 0);

		if (lvi)
		{
			ostrstream		message;

			message	<< "Auxiliary font name = " << name
					<< " already exists." << endl
					<< "Replace with value = " << value << "?" << '\0';

			msd->SetTitleAndMessageText("Add Auxiliary Font", message.str());

			if (msd->ExecuteDialog() == QDialog::Rejected)
				return;
		}

		option_edit_frame->CreateUndoOptionData();

		option_data->AddAuxiliaryFont(name, value);
		UpdateListView(&name);
	}
}

void	CAuxiliaryFontView::EditAuxiliary()
{
	if (!auxiliary_fonts || !auxiliary_fonts->size())
		return;

	QListViewItem	*current_item = auxiliary_list_view->currentItem();

	if (!current_item)
		return;

	CAuxiliaryFontDialog	*afd = OptionDialog::auxiliary_font_dialog;
	CMessageDialog			*msd = OptionDialog::message_dialog;
	string					name;
	string					value;

	afd->SetNameLineEdit(current_item->text(0));
	afd->SetValueLineEdit(current_item->text(1));
	afd->SetTitleAndResize("Edit Auxiliary Font");
	afd->SetNamePointer(&name);
	afd->SetValuePointer(&value);

	if (afd->exec() == QDialog::Accepted)
	{
		if (afd->GetNameChangedFlag())
		{
			ostrstream		message;

			QListViewItem	*lvi =
				auxiliary_list_view->findItem(name.c_str(), 0);

			if (lvi)
			{
				message	<< "Auxiliary font name = " << name
						<< " already exists." << endl
						<< "Replace with value = " << value << "?" << '\0';
			}
			else
			{
				message	<< "Add new auxiliary font name = " << name
						<< "," << endl
						<< "value = " << value << "?" << '\0';
			}

			msd->SetTitleAndMessageText("Edit Auxiliary Font",
															message.str());

			if (msd->ExecuteDialog() == QDialog::Rejected)
				return;

			option_edit_frame->CreateUndoOptionData();

			option_data->AddAuxiliaryFont(name, value);
			UpdateListView(&name);
		}
		else if (afd->GetValueChangedFlag())
		{
			option_edit_frame->CreateUndoOptionData();

			option_data->AddAuxiliaryFont(name, value);
			UpdateListView(&name);
		}
	}
}

void	CAuxiliaryFontView::RenameAuxiliary()
{
	if (!auxiliary_fonts || !auxiliary_fonts->size())
		return;

	QListViewItem	*current_item = auxiliary_list_view->currentItem();
	string			current_name;

	if (current_item)
		current_name = (const char *)(current_item->text(0).local8Bit());
	else
		return;

	CAuxiliaryFontDialog	*afd = OptionDialog::auxiliary_font_dialog;
	CMessageDialog			*msd = OptionDialog::message_dialog;
	string					name;
	string					value;

	afd->SetNameLineEdit(current_item->text(0));
	afd->SetValueLineEdit(current_item->text(1));
	afd->SetTitleAndResize("Rename Auxiliary Font");
	afd->SetNamePointer(&name);
	afd->SetValuePointer(&value);

	if (afd->exec() == QDialog::Accepted)
	{
		if (afd->GetNameChangedFlag())
		{
			QListViewItem	*lvi =
				auxiliary_list_view->findItem(name.c_str(), 0);

			if (lvi)
			{
				ostrstream		message;

				message	<< "Auxiliary font name = " << name
						<< " already exists." << endl
						<< "Replace with value = " << value << "?" << '\0';

				msd->SetTitleAndMessageText("Rename Auxiliary Font",
											message.str());

				if (msd->ExecuteDialog() == QDialog::Rejected)
					return;
			}

			option_edit_frame->CreateUndoOptionData();

			option_data->DeleteAuxiliaryFont(current_name);
			option_data->AddAuxiliaryFont(name, value);
			UpdateListView(&name);
		}
	}
}

void	CAuxiliaryFontView::DeleteAuxiliary()
{
	if (!auxiliary_fonts || !auxiliary_fonts->size())
		return;

	CopyAuxiliary();

	QListViewItem	*list_view_item		= auxiliary_list_view->firstChild();
	bool			create_undo_flag	= false;
	string			item_name;

	while (list_view_item)
	{
		if (list_view_item->isSelected())
		{
			item_name = (const char *)(list_view_item->text(0).local8Bit());

			if (!create_undo_flag)
			{
				create_undo_flag = true;

				option_edit_frame->CreateUndoOptionData();
			}

			option_data->DeleteAuxiliaryFont(item_name);
		}

		list_view_item = list_view_item->itemBelow();
	}

	if (!create_undo_flag)
		return;

	QListViewItem	*current_item	= auxiliary_list_view->currentItem();
	QListViewItem	*next_item		= current_item->itemBelow();
	string			next_name;

	if (!next_item)
		next_item = current_item->itemAbove();

	if (next_item)
	{
		next_name = (const char *)(next_item->text(0).local8Bit());

		UpdateListView(&next_name);
	}
	else
		UpdateListView(0);
}

void	CAuxiliaryFontView::CopyAuxiliary()
{
	COptionData		*od = new COptionData();

	od->SetApplicationData(new COptionData::SApplicationData());
	od->SetHelpData(new COptionData::SHelpData());

	QListViewItem	*list_view_item = auxiliary_list_view->firstChild();
	string			item_name;

	while (list_view_item)
	{
		if (list_view_item->isSelected())
		{
			item_name = (const char *)(list_view_item->text(0).local8Bit());

			string		*font_data =
				option_data->GetAuxiliaryFont(item_name);

			if (font_data)
				od->AddAuxiliaryFont(item_name, *font_data);
		}

		list_view_item = list_view_item->itemBelow();
	}

	od->CopyOptionDataToClipboard();

	delete od;
}

void	CAuxiliaryFontView::PasteAuxiliary()
{
	COptionData		*od;

	CParserDriver::StatusCode   sc =
		COptionData::InputOptionDataFromClipboard(&od);

	if (sc == CParserDriver::EndOfFile)
	{
		CMessageDialog			*msd = OptionDialog::message_dialog;
		map<string, string>		*af = od->GetAuxiliaryFonts();

		if (!af || !af->size())
		{
			delete od;

			return;
		}

		bool	create_undo_flag = false;

		map<string, string>::iterator	m_iterator = af->begin();

		while (m_iterator != af->end())
		{
			string		name((*m_iterator).first);
			string		value((*m_iterator).second);

			QListViewItem	*lvi =
				auxiliary_list_view->findItem(name.c_str(), 0);

			if (lvi)
			{
				ostrstream		message;

				message	<< "Auxiliary font name = " << name
						<< " already exists." << endl
						<< "Replace with new value ?" << '\0';

				msd->SetTitleAndMessageText(
					"Paste Auxiliary Font", message.str());

				if (msd->ExecuteDialog() == QDialog::Accepted)
				{
					if (!create_undo_flag)
					{
						create_undo_flag = true;

						option_edit_frame->CreateUndoOptionData();
					}

					option_data->AddAuxiliaryFont(name, value);
				}
			}
			else
			{
				if (!create_undo_flag)
				{
					create_undo_flag = true;

					option_edit_frame->CreateUndoOptionData();
				}

				option_data->AddAuxiliaryFont(name, value);
			}

			m_iterator++;
		}

		UpdateListView(0);

		delete od;
	}
}

void    CAuxiliaryStringView::UpdateListView(const string *s)
{
	auxiliary_name_line_edit->clear();
	auxiliary_list_view->clear();

	auxiliary_strings = option_data->GetAuxiliaryStrings();

	if (!auxiliary_strings || !auxiliary_strings->size())
		return;

	QListViewItem	*lvi;
	QListViewItem	*highlight_item = 0;

	map<string, string>::iterator	m_iterator = auxiliary_strings->begin();

	while (m_iterator != auxiliary_strings->end())
	{
		lvi = new QListViewItem(auxiliary_list_view,
								(*m_iterator).first.c_str(),
								(*m_iterator).second.c_str());

		if (s && (*m_iterator).first == *s)
			highlight_item = lvi;

		m_iterator++;
	}

	if (!highlight_item)
	{
		auxiliary_name_line_edit->setText(
			auxiliary_list_view->firstChild()->text(0));

		auxiliary_list_view->setSelected(
			auxiliary_list_view->firstChild(), true);

		auxiliary_list_view->setCurrentItem(auxiliary_list_view->firstChild());
	}
	else
	{
		auxiliary_name_line_edit->setText(highlight_item->text(0));
		auxiliary_list_view->setSelected(highlight_item, true);
		auxiliary_list_view->setCurrentItem(highlight_item);
	}
}

void	CAuxiliaryStringView::AddAuxiliary()
{
	CAuxiliaryStringDialog	*asd = OptionDialog::auxiliary_string_dialog;
	CMessageDialog			*msd = OptionDialog::message_dialog;
	string					name;
	string					value;

	asd->SetAuxiliaryName(auxiliary_name_line_edit->text());

	QListViewItem	*current_item = auxiliary_list_view->currentItem();

	if (current_item)
		asd->SetAuxiliaryValue(current_item->text(1));
	else
		asd->SetAuxiliaryValue(QString());

	asd->SetTitleAndResize("Add Auxiliary String");
	asd->SetNamePointer(&name);
	asd->SetValuePointer(&value);

	if (asd->exec() == QDialog::Accepted)
	{
		QListViewItem	*lvi = auxiliary_list_view->findItem(name.c_str(), 0);

		if (lvi)
		{
			ostrstream		message;

			message	<< "Auxiliary string name = " << name
					<< " already exists." << endl
					<< "Replace with value = " << value << "?" << '\0';

			msd->SetTitleAndMessageText("Add Auxiliary String", message.str());

			if (msd->ExecuteDialog() == QDialog::Rejected)
				return;
		}

		option_edit_frame->CreateUndoOptionData();

		option_data->AddAuxiliaryString(name, value);
		UpdateListView(&name);
	}
}

void	CAuxiliaryStringView::EditAuxiliary()
{
	if (!auxiliary_strings || !auxiliary_strings->size())
		return;

	QListViewItem	*current_item = auxiliary_list_view->currentItem();

	if (!current_item)
		return;

	CAuxiliaryStringDialog	*asd = OptionDialog::auxiliary_string_dialog;
	CMessageDialog			*msd = OptionDialog::message_dialog;
	string					name;
	string					value;

	asd->SetAuxiliaryName(current_item->text(0));
	asd->SetAuxiliaryValue(current_item->text(1));
	asd->SetTitleAndResize("Edit Auxiliary String");
	asd->SetNamePointer(&name);
	asd->SetValuePointer(&value);

	if (asd->exec() == QDialog::Accepted)
	{
		if (asd->GetNameChangedFlag())
		{
			ostrstream		message;

			QListViewItem	*lvi =
				auxiliary_list_view->findItem(name.c_str(), 0);

			if (lvi)
			{
				message	<< "Auxiliary string name = " << name
						<< " already exists." << endl
						<< "Replace with value = " << value << "?" << '\0';
			}
			else
			{
				message	<< "Add new auxiliary string name = " << name
						<< "," << endl
						<< "value = " << value << "?" << '\0';
			}

			msd->SetTitleAndMessageText("Edit Auxiliary String",
															message.str());

			if (msd->ExecuteDialog() == QDialog::Rejected)
				return;

			option_edit_frame->CreateUndoOptionData();

			option_data->AddAuxiliaryString(name, value);
			UpdateListView(&name);
		}
		else if (asd->GetValueChangedFlag())
		{
			option_edit_frame->CreateUndoOptionData();

			option_data->AddAuxiliaryString(name, value);
			UpdateListView(&name);
		}
	}
}

void	CAuxiliaryStringView::RenameAuxiliary()
{
	if (!auxiliary_strings || !auxiliary_strings->size())
		return;

	QListViewItem	*current_item = auxiliary_list_view->currentItem();
	string			current_name;

	if (current_item)
		current_name = (const char *)(current_item->text(0).local8Bit());
	else
		return;

	CAuxiliaryStringDialog	*asd = OptionDialog::auxiliary_string_dialog;
	CMessageDialog			*msd = OptionDialog::message_dialog;
	string					name;
	string					value;

	asd->SetAuxiliaryName(current_item->text(0));
	asd->SetAuxiliaryValue(current_item->text(1));
	asd->SetTitleAndResize("Rename Auxiliary String");
	asd->SetNamePointer(&name);
	asd->SetValuePointer(&value);

	if (asd->exec() == QDialog::Accepted)
	{
		if (asd->GetNameChangedFlag())
		{
			QListViewItem	*lvi =
				auxiliary_list_view->findItem(name.c_str(), 0);

			if (lvi)
			{
				ostrstream		message;

				message	<< "Auxiliary string name = " << name
						<< " already exists." << endl
						<< "Replace with value = " << value << "?" << '\0';

				msd->SetTitleAndMessageText("Rename Auxiliary String",
											message.str());

				if (msd->ExecuteDialog() == QDialog::Rejected)
					return;
			}

			option_edit_frame->CreateUndoOptionData();

			option_data->DeleteAuxiliaryString(current_name);
			option_data->AddAuxiliaryString(name, value);
			UpdateListView(&name);
		}
	}
}

void	CAuxiliaryStringView::DeleteAuxiliary()
{
	if (!auxiliary_strings || !auxiliary_strings->size())
		return;

	CopyAuxiliary();

	QListViewItem	*list_view_item		= auxiliary_list_view->firstChild();
	bool			create_undo_flag	= false;
	string			item_name;

	while (list_view_item)
	{
		if (list_view_item->isSelected())
		{
			item_name = (const char *)(list_view_item->text(0).local8Bit());

			if (!create_undo_flag)
			{
				create_undo_flag = true;

				option_edit_frame->CreateUndoOptionData();
			}

			option_data->DeleteAuxiliaryString(item_name);
		}

		list_view_item = list_view_item->itemBelow();
	}

	if (!create_undo_flag)
		return;

	QListViewItem	*current_item	= auxiliary_list_view->currentItem();
	QListViewItem	*next_item		= current_item->itemBelow();
	string			next_name;

	if (!next_item)
		next_item = current_item->itemAbove();

	if (next_item)
	{
		next_name = (const char *)(next_item->text(0).local8Bit());

		UpdateListView(&next_name);
	}
	else
		UpdateListView(0);
}

void	CAuxiliaryStringView::CopyAuxiliary()
{
	COptionData		*od = new COptionData();

	od->SetApplicationData(new COptionData::SApplicationData());
	od->SetHelpData(new COptionData::SHelpData());

	QListViewItem	*list_view_item = auxiliary_list_view->firstChild();
	string			item_name;

	while (list_view_item)
	{
		if (list_view_item->isSelected())
		{
			item_name = (const char *)(list_view_item->text(0).local8Bit());

			string		*str_data =
				option_data->GetAuxiliaryString(item_name);

			if (str_data)
				od->AddAuxiliaryString(item_name, *str_data);
		}

		list_view_item = list_view_item->itemBelow();
	}

	od->CopyOptionDataToClipboard();

	delete od;
}

void	CAuxiliaryStringView::PasteAuxiliary()
{
	COptionData		*od;

	CParserDriver::StatusCode   sc =
		COptionData::InputOptionDataFromClipboard(&od);

	if (sc == CParserDriver::EndOfFile)
	{
		CMessageDialog			*msd = OptionDialog::message_dialog;
		map<string, string>		*as = od->GetAuxiliaryStrings();

		if (!as || !as->size())
		{
			delete od;

			return;
		}

		bool	create_undo_flag = false;

		map<string, string>::iterator	m_iterator = as->begin();

		while (m_iterator != as->end())
		{
			string		name((*m_iterator).first);
			string		value((*m_iterator).second);

			QListViewItem	*lvi =
				auxiliary_list_view->findItem(name.c_str(), 0);

			if (lvi)
			{
				ostrstream		message;

				message	<< "Auxiliary string name = " << name
						<< " already exists." << endl
						<< "Replace with new value ?" << '\0';

				msd->SetTitleAndMessageText(
					"Paste Auxiliary String", message.str());

				if (msd->ExecuteDialog() == QDialog::Accepted)
				{
					if (!create_undo_flag)
					{
						create_undo_flag = true;

						option_edit_frame->CreateUndoOptionData();
					}

					option_data->AddAuxiliaryString(name, value);
				}
			}
			else
			{
				if (!create_undo_flag)
				{
					create_undo_flag = true;

					option_edit_frame->CreateUndoOptionData();
				}

				option_data->AddAuxiliaryString(name, value);
			}

			m_iterator++;
		}

		UpdateListView(0);

		delete od;
	}
}

CConnectionEditView::CConnectionEditView(	QWidget *parent,
											COptionEditFrame *oef)
	: CFrameFc(parent, 2, 5), option_edit_frame(oef),
		option_data_main_view_map(0)
{
	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 1);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 0);
	grid_layout->setColStretch(2, 0);
	grid_layout->setColStretch(3, 0);
	grid_layout->setColStretch(4, 1);
	
	QString		size_string("##############");

	connect_button			= new CPushButtonFc(
										"Connect", this, size_string);
	disconnect_button		= new CPushButtonFc(
										"Disconnect", this, size_string);
	connect_all_button		= new CPushButtonFc(
										"Connect All", this, size_string);
	disconnect_all_button	= new CPushButtonFc(
										"Disconnect All", this, size_string);
	spacer					= new CSpacerFc(this);
	connection_list_view	= new QListView(this);

	connection_list_view->addColumn("Connection Tree");
	connection_list_view->setSorting(-1);
	connection_list_view->setRootIsDecorated(false);
	connection_list_view->setSelectionMode(QListView::Extended);

	grid_layout->addWidget(connect_button,					0, 0);
	grid_layout->addWidget(disconnect_button,				0, 1);
	grid_layout->addWidget(connect_all_button,				0, 2);
	grid_layout->addWidget(disconnect_all_button,			0, 3);
	grid_layout->addWidget(spacer,							0, 4);
	grid_layout->addMultiCellWidget(connection_list_view,	1, 1, 0, 4);

	QObject::connect(	connect_button,
						SIGNAL(clicked()),
						this,
						SLOT(Connect()));

	QObject::connect(	disconnect_button,
						SIGNAL(clicked()),
						this,
						SLOT(Disconnect()));

	QObject::connect(	connect_all_button,
						SIGNAL(clicked()),
						this,
						SLOT(ConnectAll()));

	QObject::connect(	disconnect_all_button,
						SIGNAL(clicked()),
						this,
						SLOT(DisconnectAll()));
}

CConnectionEditView::~CConnectionEditView()
{
	DestroyOptionDataMainViewMap(option_data_main_view_map);
}

void	CConnectionEditView::UpdateOptionEditFrameGeometry()
{
	updateGeometry();
}

void	CConnectionEditView::SetOptionData(	SOptionFileItem *ofi,
											COptionData *d)
{
	option_file_item	= ofi;
	option_data			= d;

	DestroyOptionDataMainViewMap(option_data_main_view_map);

	option_data_main_view_map = CreateOptionDataMainViewMap(option_data,
		option_file_item->interface_configuration->subview_connection_map);

	CreateOptionDataMainViewTree(option_data_main_view_map);
}

void    CConnectionEditView::CreateOptionDataMainViewTree(
												OptionDataMainViewMap *odmvm)
{
	connection_list_view->clear();

	ReverseOptionDataMainViewMapIterator		odmvm_iterator;
	ReverseOptionDataSubtypeMapIterator			odsm_iterator;
	ReverseOptionDataIndexMapIterator			odim_iterator;

	odmvm_iterator = odmvm->rbegin();

	while (odmvm_iterator != odmvm->rend())
	{
		string					main_view_name("Main View: ");
		OptionDataSubtypeMap	&odsm = (*odmvm_iterator).second;

		main_view_name += (*odmvm_iterator).first;

		QListViewItem	*main_view_item =
			new QListViewItem(connection_list_view, main_view_name.c_str());

		odsm_iterator = odsm.rbegin();

		while (odsm_iterator != odsm.rend())
		{
			string					subview_name("Subview: ");
			OptionDataIndexMap		&odim = (*odsm_iterator).second;

			subview_name += (*odsm_iterator).first;

			QListViewItem	*subview_item =
				new QListViewItem(main_view_item, subview_name.c_str());

			odim_iterator = odim.rbegin();

			while (odim_iterator != odim.rend())
			{
				ostrstream				index_text;
				const vector<int>		&index = (*odim_iterator).first;
				ConnectionMainViewMap	*cmvm =
					(*odim_iterator).second.second;

				CreateIndexText(index_text, index);

				QListViewItem	*index_item =
					new QListViewItem(subview_item, index_text.str());

				CreateConnectionMainViewTree(index_item, cmvm);

				index_item->setOpen(true);
				index_item->setSelectable(false);

				odim_iterator++;
			}

			subview_item->setOpen(true);
			subview_item->setSelectable(false);

			odsm_iterator++;
		}

		main_view_item->setOpen(true);
		main_view_item->setSelectable(false);

		odmvm_iterator++;
	}
}

void    CConnectionEditView::CreateConnectionMainViewTree(
												QListViewItem *item,
												ConnectionMainViewMap *cmvm)
{
	ReverseConnectionMainViewMapIterator		cmvm_iterator;
	ReverseConnectionSubtypeMapIterator			csm_iterator;
	ReverseConnectionIndexMapIterator			cim_iterator;

	cmvm_iterator = cmvm->rbegin();

	while (cmvm_iterator != cmvm->rend())
	{
		string					main_view_name("Main View: ");
		ConnectionSubtypeMap	&csm = (*cmvm_iterator).second;

		main_view_name += (*cmvm_iterator).first;

		QListViewItem	*main_view_item =
			new QListViewItem(item, main_view_name.c_str());

		csm_iterator = csm.rbegin();

		while (csm_iterator != csm.rend())
		{
			string					subview_name("Subview: ");
			ConnectionIndexMap		&cim = (*csm_iterator).second;

			subview_name += (*csm_iterator).first;

			QListViewItem	*subview_item =
				new QListViewItem(main_view_item, subview_name.c_str());

			cim_iterator = cim.rbegin();

			while (cim_iterator != cim.rend())
			{
				ostrstream				index_text;
				const vector<int>		&index = (*cim_iterator).first;
				bool					&connection_flag =
											(*cim_iterator).second;

				CreateIndexText(index_text, index);

				SConnectionTreeItem		*index_item =
					new SConnectionTreeItem(subview_item,
											index_text.str(),
											connection_flag);

				cim_iterator++;
			}

			subview_item->setOpen(true);
			subview_item->setSelectable(false);

			csm_iterator++;
		}

		main_view_item->setOpen(true);
		main_view_item->setSelectable(false);

		cmvm_iterator++;
	}
}

void	CConnectionEditView::CreateIndexText(	ostrstream &s,
												const vector<int> &index)
{
	s << "Index = (";

	if (index.size())
	{
		vector<int>::size_type	i;

		for(i=0; i<index.size()-1; i++)
			s << index[i] << ", ";

		s << index[index.size()-1];
	}

	s << ")" << '\0';
}

void	CConnectionEditView::StoreConnectionData()
{
	StoreOptionDataMainViewMap(option_data_main_view_map);
}

void	CConnectionEditView::Connect()
{
	QListViewItem	*item = connection_list_view->firstChild();

	while (item)
	{
		SConnectionTreeItem		*check_item =
			dynamic_cast<SConnectionTreeItem *>(item);

		if (check_item && check_item->isSelected())
		{
			check_item->setOn(true);
			check_item->connection_flag = true;
		}

		item = item->itemBelow();
	}
}

void	CConnectionEditView::Disconnect()
{
	QListViewItem	*item = connection_list_view->firstChild();

	while (item)
	{
		SConnectionTreeItem		*check_item =
			dynamic_cast<SConnectionTreeItem *>(item);

		if (check_item && check_item->isSelected())
		{
			check_item->setOn(false);
			check_item->connection_flag = false;
		}

		item = item->itemBelow();
	}
}

void	CConnectionEditView::ConnectAll()
{
	QListViewItem	*item = connection_list_view->firstChild();

	while (item)
	{
		SConnectionTreeItem		*check_item =
			dynamic_cast<SConnectionTreeItem *>(item);

		if (check_item)
		{
			check_item->setOn(true);
			check_item->connection_flag = true;
		}

		item = item->itemBelow();
	}
}

void	CConnectionEditView::DisconnectAll()
{
	QListViewItem	*item = connection_list_view->firstChild();

	while (item)
	{
		SConnectionTreeItem		*check_item =
			dynamic_cast<SConnectionTreeItem *>(item);

		if (check_item)
		{
			check_item->setOn(false);
			check_item->connection_flag = false;
		}

		item = item->itemBelow();
	}
}
