/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "precedenceView.h"

void	CPrecedenceListView::InitializePalette(const QPalette &p)
{
	active_highlight			= p.color(	QPalette::Active,
											QColorGroup::Highlight);
	active_highlighted_text		= p.color(	QPalette::Active,
											QColorGroup::HighlightedText);
	inactive_highlight			= p.color(	QPalette::Inactive,
											QColorGroup::Highlight);
	inactive_highlighted_text	= p.color(	QPalette::Inactive,
											QColorGroup::HighlightedText);

	QPalette	focus_palette = p;

	focus_palette.setColor(	QPalette::Active,
							QColorGroup::Highlight,
							inactive_highlight);

	focus_palette.setColor(	QPalette::Active,
							QColorGroup::HighlightedText,
							inactive_highlighted_text);

	setPalette(focus_palette);
}

void	CPrecedenceListView::focusInEvent(QFocusEvent *e)
{
	QPalette	p = palette();

	p.setColor(	QPalette::Active,
				QColorGroup::Highlight,
				active_highlight);

	p.setColor(	QPalette::Active,
				QColorGroup::HighlightedText,
				active_highlighted_text);

	setPalette(p);

	QListView::focusInEvent(e);
}

void	CPrecedenceListView::focusOutEvent(QFocusEvent *e)
{
	QPalette	p = palette();

	p.setColor(	QPalette::Active,
				QColorGroup::Highlight,
				inactive_highlight);

	p.setColor(	QPalette::Active,
				QColorGroup::HighlightedText,
				inactive_highlighted_text);

	setPalette(p);

	QListView::focusOutEvent(e);
}

void	CPrecedenceListView::mouseDoubleClickEvent(QMouseEvent *e)
{
	precedence_vbox->EditPrecedence();

	e->accept();
}

void	CPrecedenceListView::keyPressEvent(QKeyEvent *e)
{
	switch (e->key())
	{
		case Qt::Key_Insert:
		switch (e->state())
		{
			case Qt::NoButton:
			case Qt::Keypad:
			e->accept();
			precedence_vbox->InsertAfterPrecedence();
			return;

			case Qt::ShiftButton:
			e->accept();
			precedence_vbox->InsertBeforePrecedence();
			return;
		}
		break; 

		case Qt::Key_0:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			precedence_vbox->InsertBeforePrecedence();
			return;
		}
		break; 

		case Qt::Key_Delete:
		if (e->state() == Qt::ShiftButton)
		{
			e->accept();
			precedence_vbox->DeletePrecedence();
			return;
		}
		break; 

		case Qt::Key_Period:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			precedence_vbox->DeletePrecedence();
			return;
		}
		break; 

		case Qt::Key_Return:
		if (e->state() == Qt::NoButton)
		{
			e->accept();
			precedence_vbox->EditPrecedence();
			return;
		}
		break; 

		case Qt::Key_Enter:
		if (e->state() == Qt::Keypad)
		{
			e->accept();
			precedence_vbox->EditPrecedence();
			return;
		}
		break; 

		case Qt::Key_Up:
		if (e->state() == Qt::ShiftButton)
		{
			e->accept();
			precedence_vbox->MovePrecedenceUp();
			return;
		}
		break;

		case Qt::Key_8:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			precedence_vbox->MovePrecedenceUp();
			return;
		}
		break;

		case Qt::Key_Down:
		if (e->state() == Qt::ShiftButton)
		{
			e->accept();
			precedence_vbox->MovePrecedenceDown();
			return;
		}
		break;

		case Qt::Key_2:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			precedence_vbox->MovePrecedenceDown();
			return;
		}
		break;
	}

	e->ignore();
	QListView::keyPressEvent(e);
}

CPrecedenceVBoxWidget::CPrecedenceVBoxWidget(	QWidget *parent,
												CCompilerInterface *ci,
												SMainViewControl *mvc,
												COptionData::SSubViewData *d,
												CPrecedenceView *pv)
	: CVBoxWidget(parent, ci, mvc), compiler_interface(ci),
		subview_data(d), precedence_view(pv),
		module_ptr_map(pv->GetModulePtrMap())
{
	label = new CLabelFc("precedence", this, true);

	label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	list_view = new CPrecedenceListView(this);

	list_view->header()->setMovingEnabled(false);
	list_view->setSorting(-1);
	list_view->addColumn("index");
	list_view->addColumn("associativity");
	list_view->addColumn("entries");

	setSpacing(0);
	setStretchFactor(label, 0);
	setStretchFactor(list_view, 1);

	compiler_data = compiler_interface->GetCompilerData();

	UpdateListView();

	list_view->setSelected(list_view->firstChild(), true);
}

CPrecedenceVBoxWidget::~CPrecedenceVBoxWidget()
{
}

void	CPrecedenceVBoxWidget::UpdateListView()
{
	const list<CPrecedence *>	&p_list = compiler_data->GetPrecedenceList();

	int		p_count = p_list.size();
	int		l_count = list_view->childCount();
	int		i;

	if (p_count < l_count)
	{
		for (i=0; i<l_count-p_count; i++)
			delete list_view->firstChild();
	}
	else if (p_count > l_count)
	{
		for (i=0; i<p_count-l_count; i++)
			new QListViewItem(list_view);
	}

	list<CPrecedence *>::const_iterator		p_iterator	= p_list.begin();

	QListViewItemIterator	l_iterator(list_view);

	while (p_iterator != p_list.end())
	{
		const CPrecedence		*p = *p_iterator;

		QListViewItem	*item = l_iterator.current();

		string		s_index;
		string		s_associativity;
		string		s_entries;

		p->GetIndexString(s_index);
		p->GetAssociativityString(s_associativity);
		p->GetEntryString(s_entries);

		item->setText(0, s_index.c_str());
		item->setText(1, s_associativity.c_str());
		item->setText(2, s_entries.c_str());

		p_iterator++;
		l_iterator++;
	}
}

void	CPrecedenceVBoxWidget::SetSelectedItem(int i)
{
	int		l_count = list_view->childCount();

	if (i < 0 || !l_count)
		return;

	if (i >= l_count)
		i = l_count - 1;

	QListViewItemIterator	l_iterator(list_view);
	int						count;

	for (count=0; count<i; count++)
		l_iterator++;

	list_view->setSelected(l_iterator.current(), true);
	list_view->ensureItemVisible(l_iterator.current());
}

void	CPrecedenceVBoxWidget::InitializeModulePointers()
{
}

void	CPrecedenceVBoxWidget::InsertBeforePrecedence()
{
	GrammarDialog::SetPrecedencePaletteAndFont(this);

	CPrecedence			precedence(0);
	CPrecedenceDialog	*pd = GrammarDialog::precedence_dialog;

	pd->SetTitleAndResize("Insert Before Precedence");
	pd->SetPrecedencePointer(&precedence);
	pd->SetControlValues(precedence);
	pd->SetCompilerDataPointer(compiler_data);

	if (pd->exec() == QDialog::Accepted)
	{
		QListViewItem	*current = list_view->currentItem();
		int				selected = 0;

		if (current)
			selected = current->text(0).toInt();

		compiler_interface->StoreUndoData();

		InsertBeforePrecedence(selected, precedence);

		SetSelectedItem(selected);

		compiler_interface->SetDataModifiedFlag(true);
	}
}

void	CPrecedenceVBoxWidget::InsertAfterPrecedence()
{
	GrammarDialog::SetPrecedencePaletteAndFont(this);

	CPrecedence			precedence(0);
	CPrecedenceDialog	*pd = GrammarDialog::precedence_dialog;

	pd->SetTitleAndResize("Insert After Precedence");
	pd->SetPrecedencePointer(&precedence);
	pd->SetControlValues(precedence);
	pd->SetCompilerDataPointer(compiler_data);

	if (pd->exec() == QDialog::Accepted)
	{
		QListViewItem	*current = list_view->currentItem();
		int				selected = 0;

		if (current)
			selected = current->text(0).toInt();

		compiler_interface->StoreUndoData();

		InsertAfterPrecedence(selected, precedence);

		SetSelectedItem(selected + 1);

		compiler_interface->SetDataModifiedFlag(true);
	}
}

void	CPrecedenceVBoxWidget::EditPrecedence()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	selected	= current->text(0).toInt();

		GrammarDialog::SetPrecedencePaletteAndFont(this);

		CPrecedence			*pi = compiler_data->FindPrecedenceIndex(selected);
		CPrecedence			precedence(0);
		CPrecedenceDialog	*pd = GrammarDialog::precedence_dialog;

		pd->SetTitleAndResize("Edit Precedence");
		pd->SetPrecedencePointer(&precedence);
		pd->SetCompilerDataPointer(compiler_data);

		if (pi)
			pd->SetControlValues(*pi);
		else
			pd->SetControlValues(precedence);

		if (pd->exec() == QDialog::Accepted)
		{
			compiler_interface->StoreUndoData();

			UpdatePrecedence(selected, precedence);

			SetSelectedItem(selected);

			compiler_interface->SetDataModifiedFlag(true);
		}
	}
}

void	CPrecedenceVBoxWidget::DeletePrecedence()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	selected	= current->text(0).toInt();

		compiler_interface->StoreUndoData();

		DeletePrecedence(selected);

		SetSelectedItem(selected);

		compiler_interface->SetDataModifiedFlag(true);
	}
}

void	CPrecedenceVBoxWidget::MovePrecedenceUp()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	selected	= current->text(0).toInt();

		if (selected > 0)
		{
			compiler_interface->StoreUndoData();

			MovePrecedenceUp(selected);

			SetSelectedItem(selected - 1);

			compiler_interface->SetDataModifiedFlag(true);
		}
	}
}

void	CPrecedenceVBoxWidget::MovePrecedenceDown()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	l_count		= list_view->childCount();
		int	selected	= current->text(0).toInt();

		if (selected < l_count - 1)
		{
			compiler_interface->StoreUndoData();

			MovePrecedenceDown(selected);

			SetSelectedItem(selected + 1);

			compiler_interface->SetDataModifiedFlag(true);
		}
	}
}

void	CPrecedenceVBoxWidget::InsertBeforePrecedence(	int i,
														const CPrecedence &p)
{
	compiler_data->InsertBeforePrecedence(i, p);

	UpdateListView();
}

void	CPrecedenceVBoxWidget::InsertAfterPrecedence(	int i,
														const CPrecedence &p)
{
	compiler_data->InsertAfterPrecedence(i, p);

	UpdateListView();
}

void	CPrecedenceVBoxWidget::UpdatePrecedence(int i, const CPrecedence &p)
{
	compiler_data->UpdatePrecedence(i, p);

	UpdateListView();
}

void	CPrecedenceVBoxWidget::DeletePrecedence(int i)
{
	compiler_data->DeletePrecedence(i);

	UpdateListView();
}

void	CPrecedenceVBoxWidget::MovePrecedenceUp(int i)
{
	compiler_data->MovePrecedenceUp(i);

	UpdateListView();
}

void	CPrecedenceVBoxWidget::MovePrecedenceDown(int i)
{
	compiler_data->MovePrecedenceDown(i);

	UpdateListView();
}

void	CPrecedenceVBoxWidget::RestoreUndoData()
{
	compiler_data = compiler_interface->GetCompilerData();
}

CPrecedenceView::CPrecedenceView(	QWidget *p,
									CViewNode *n,
									CInterfaceControl *ic,
									SMainViewControl *mvc,
									const string &dfs,
									CPaletteData *dpd,
									COptionData::SSubViewData *d)
	: CSubView(p, n, ic, mvc, dfs, dpd, d)
{
	frame = new CPrecedenceVBoxWidget(
		p, dynamic_cast<CCompilerInterface *>(ic), mvc, d, this);

	QFont		view_node_font;
	QPalette	view_node_palette;

	if (SetupViewNodeFont(view_node_font))
		frame->setFont(view_node_font);
	else
		frame->setFont(frame->font());

	if (SetupViewNodePalette(view_node_palette))
	{
		frame->setPalette(view_node_palette);
		frame->InitializePalette(view_node_palette);
	}
	else
	{
		frame->setPalette(frame->palette());
		frame->InitializePalette(frame->palette());
	}
}

CPrecedenceView::~CPrecedenceView()
{
	delete frame;
}

void	CPrecedenceView::SetHighlightPalette(CPaletteData *pd)
{
	restore_palette = frame->palette();

	QPalette	highlight_palette;

	pd->SetupPalette(highlight_palette);

	frame->setPalette(highlight_palette);
}

void	CPrecedenceView::RestorePalette()
{
	frame->setPalette(restore_palette);
}

void	CPrecedenceView::InitializeModulePointers()
{
	frame->InitializeModulePointers();
}

void	CPrecedenceView::InsertAfterPrecedence()
{
	frame->InsertAfterPrecedence();
}

void	CPrecedenceView::InsertBeforePrecedence()
{
	frame->InsertBeforePrecedence();
}

void	CPrecedenceView::EditPrecedence()
{
	frame->EditPrecedence();
}

void	CPrecedenceView::DeletePrecedence()
{
	frame->DeletePrecedence();
}

void	CPrecedenceView::MovePrecedenceUp()
{
	frame->MovePrecedenceUp();
}

void	CPrecedenceView::MovePrecedenceDown()
{
	frame->MovePrecedenceDown();
}

void	CPrecedenceView::UpdateListView()
{
	frame->UpdateListView();
}

void	CPrecedenceView::RestoreUndoData()
{
	frame->RestoreUndoData();
}
