/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "ruleView.h"

void	CRuleListView::InitializePalette(const QPalette &p)
{
	active_highlight			= p.color(	QPalette::Active,
											QColorGroup::Highlight);
	active_highlighted_text		= p.color(	QPalette::Active,
											QColorGroup::HighlightedText);
	inactive_highlight			= p.color(	QPalette::Inactive,
											QColorGroup::Highlight);
	inactive_highlighted_text	= p.color(	QPalette::Inactive,
											QColorGroup::HighlightedText);

	QPalette	focus_palette = p;

	focus_palette.setColor(	QPalette::Active,
							QColorGroup::Highlight,
							inactive_highlight);

	focus_palette.setColor(	QPalette::Active,
							QColorGroup::HighlightedText,
							inactive_highlighted_text);

	setPalette(focus_palette);
}

void	CRuleListView::focusInEvent(QFocusEvent *e)
{
	QPalette	p = palette();

	p.setColor(	QPalette::Active,
				QColorGroup::Highlight,
				active_highlight);

	p.setColor(	QPalette::Active,
				QColorGroup::HighlightedText,
				active_highlighted_text);

	setPalette(p);

	QListView::focusInEvent(e);
}

void	CRuleListView::focusOutEvent(QFocusEvent *e)
{
	QPalette	p = palette();

	p.setColor(	QPalette::Active,
				QColorGroup::Highlight,
				inactive_highlight);

	p.setColor(	QPalette::Active,
				QColorGroup::HighlightedText,
				inactive_highlighted_text);

	setPalette(p);

	QListView::focusOutEvent(e);
}

void	CRuleListView::mouseDoubleClickEvent(QMouseEvent *e)
{
	rule_vbox->EditRule();

	e->accept();
}

void	CRuleListView::keyPressEvent(QKeyEvent *e)
{
	switch (e->key())
	{
		case Qt::Key_Insert:
		switch (e->state())
		{
			case Qt::NoButton:
			case Qt::Keypad:
			e->accept();
			rule_vbox->InsertAfterRule();
			return;

			case Qt::ShiftButton:
			e->accept();
			rule_vbox->InsertBeforeRule();
			return;
		}
		break; 

		case Qt::Key_0:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			rule_vbox->InsertBeforeRule();
			return;
		}
		break; 

		case Qt::Key_Delete:
		if (e->state() == Qt::ShiftButton)
		{
			e->accept();
			rule_vbox->DeleteRule();
			return;
		}
		break; 

		case Qt::Key_Period:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			rule_vbox->DeleteRule();
			return;
		}
		break; 

		case Qt::Key_Return:
		if (e->state() == Qt::NoButton)
		{
			e->accept();
			rule_vbox->EditRule();
			return;
		}
		break; 

		case Qt::Key_Enter:
		if (e->state() == Qt::Keypad)
		{
			e->accept();
			rule_vbox->EditRule();
			return;
		}
		break; 

		case Qt::Key_Up:
		if (e->state() == Qt::ShiftButton)
		{
			e->accept();
			rule_vbox->MoveRuleUp();
			return;
		}
		break;

		case Qt::Key_8:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			rule_vbox->MoveRuleUp();
			return;
		}
		break;

		case Qt::Key_Down:
		if (e->state() == Qt::ShiftButton)
		{
			e->accept();
			rule_vbox->MoveRuleDown();
			return;
		}
		break;

		case Qt::Key_2:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			rule_vbox->MoveRuleDown();
			return;
		}
		break;
	}

	e->ignore();
	QListView::keyPressEvent(e);
}

CRuleVBoxWidget::CRuleVBoxWidget(	QWidget *parent,
									CCompilerInterface *ci,
									SMainViewControl *mvc,
									COptionData::SSubViewData *d,
									CRuleView *rv)
	: CVBoxWidget(parent, ci, mvc), compiler_interface(ci),
		subview_data(d), rule_view(rv),
		module_ptr_map(rv->GetModulePtrMap())
{
	label = new CLabelFc("rule", this, true);

	label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	list_view = new CRuleListView(this);

	list_view->header()->setMovingEnabled(false);
	list_view->setSorting(-1);
	list_view->addColumn("index");
	list_view->addColumn("unit");
	list_view->addColumn("left");
	list_view->addColumn("right");

	list_view->setColumnWidth(0, 70);
	list_view->setColumnWidth(1, 60);
	list_view->setColumnWidth(2, 60);
	list_view->setColumnWidth(3, 70);

	setSpacing(0);
	setStretchFactor(label, 0);
	setStretchFactor(list_view, 1);

	compiler_data = compiler_interface->GetCompilerData();

	UpdateListView();

	list_view->setSelected(list_view->firstChild(), true);
}

CRuleVBoxWidget::~CRuleVBoxWidget()
{
}

void	CRuleVBoxWidget::UpdateListView()
{
	const list<CRule *>		&r_list = compiler_data->GetRuleList();

	int		r_count = r_list.size();
	int		l_count = list_view->childCount();
	int		i;

	if (r_count < l_count)
	{
		for (i=0; i<l_count-r_count; i++)
			delete list_view->firstChild();
	}
	else if (r_count > l_count)
	{
		for (i=0; i<r_count-l_count; i++)
			new QListViewItem(list_view);
	}

	list<CRule *>::const_iterator		r_iterator	= r_list.begin();

	QListViewItemIterator	l_iterator(list_view);

	while (r_iterator != r_list.end())
	{
		const CRule		*r = *r_iterator;

		QListViewItem	*item = l_iterator.current();

		string		s_index;
		string		s_left;
		string		s_right;

		r->GetIndexString(s_index);
		r->GetLeftString(s_left);
		r->GetRightString(s_right);

		item->setText(0, s_index.c_str());

		if (r->GetUnitRuleEliminationFlag())
			item->setText(1, "*");
		else
			item->setText(1, "");

		item->setText(2, s_left.c_str());
		item->setText(3, s_right.c_str());

		r_iterator++;
		l_iterator++;
	}

	vector<CSubView *>		&v1 = module_ptr_map["IMPLICIT_VIEW"];
	vector<CSubView *>		&v2 = module_ptr_map["PRECEDENCE_VIEW"];
	vector<CSubView *>		&v3 = module_ptr_map["RULE_SELECT_VIEW"];

	if (v1.size())
	{
		CImplicitView		*iv = dynamic_cast<CImplicitView *>(v1[0]);

		iv->UpdateListView();
	}

	if (v2.size())
	{
		CPrecedenceView		*pv = dynamic_cast<CPrecedenceView *>(v2[0]);

		pv->UpdateListView();
	}

	if (v3.size())
	{
		CRuleSelectView		*rv = dynamic_cast<CRuleSelectView *>(v3[0]);

		rv->UpdateListView();
	}
}

void	CRuleVBoxWidget::SetSelectedItem(int i)
{
	int		l_count = list_view->childCount();

	if (i < 0 || !l_count)
		return;

	if (i >= l_count)
		i = l_count - 1;

	QListViewItemIterator	l_iterator(list_view);
	int						count;

	for (count=0; count<i; count++)
		l_iterator++;

	list_view->setSelected(l_iterator.current(), true);
	list_view->ensureItemVisible(l_iterator.current());
}

void	CRuleVBoxWidget::InitializeModulePointers()
{
}

void	CRuleVBoxWidget::InsertBeforeRule()
{
	GrammarDialog::SetRulePaletteAndFont(this);

	CRule			rule(0);
	CRuleDialog		*rd = GrammarDialog::rule_dialog;

	rd->SetTitleAndResize("Insert Before Rule");
	rd->SetRulePointer(&rule);
	rd->SetControlValues(rule);
	rd->SetCompilerDataPointer(compiler_data);

	if (rd->exec() == QDialog::Accepted)
	{
		QListViewItem	*current = list_view->currentItem();
		int				selected = 0;

		if (current)
			selected = current->text(0).toInt();

		compiler_interface->StoreUndoData();

		InsertBeforeRule(selected, rule);

		SetSelectedItem(selected);

		compiler_interface->SetDataModifiedFlag(true);
	}
}

void	CRuleVBoxWidget::InsertAfterRule()
{
	GrammarDialog::SetRulePaletteAndFont(this);

	CRule			rule(0);
	CRuleDialog		*rd = GrammarDialog::rule_dialog;

	rd->SetTitleAndResize("Insert After Rule");
	rd->SetRulePointer(&rule);
	rd->SetControlValues(rule);
	rd->SetCompilerDataPointer(compiler_data);

	if (rd->exec() == QDialog::Accepted)
	{
		QListViewItem	*current = list_view->currentItem();
		int				selected = 0;

		if (current)
			selected = current->text(0).toInt();

		compiler_interface->StoreUndoData();

		InsertAfterRule(selected, rule);

		SetSelectedItem(selected + 1);

		compiler_interface->SetDataModifiedFlag(true);
	}
}

void	CRuleVBoxWidget::EditRule()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	selected	= current->text(0).toInt();

		GrammarDialog::SetRulePaletteAndFont(this);

		CRule			*ri = compiler_data->FindRuleIndex(selected);
		CRule			rule(0);
		CRuleDialog		*rd = GrammarDialog::rule_dialog;

		rd->SetTitleAndResize("Edit Rule");
		rd->SetRulePointer(&rule);
		rd->SetCompilerDataPointer(compiler_data);

		if (ri)
			rd->SetControlValues(*ri);
		else
			rd->SetControlValues(rule);

		if (rd->exec() == QDialog::Accepted)
		{
			compiler_interface->StoreUndoData();

			UpdateRule(selected, rule);

			SetSelectedItem(selected);

			compiler_interface->SetDataModifiedFlag(true);
		}
	}
}

void	CRuleVBoxWidget::DeleteRule()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	selected	= current->text(0).toInt();

		compiler_interface->StoreUndoData();

		DeleteRule(selected);

		SetSelectedItem(selected);

		compiler_interface->SetDataModifiedFlag(true);
	}
}

void	CRuleVBoxWidget::MoveRuleUp()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	selected	= current->text(0).toInt();

		if (selected > 0)
		{
			compiler_interface->StoreUndoData();

			MoveRuleUp(selected);

			SetSelectedItem(selected - 1);

			compiler_interface->SetDataModifiedFlag(true);
		}
	}
}

void	CRuleVBoxWidget::MoveRuleDown()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	l_count		= list_view->childCount();
		int	selected	= current->text(0).toInt();

		if (selected < l_count - 1)
		{
			compiler_interface->StoreUndoData();

			MoveRuleDown(selected);

			SetSelectedItem(selected + 1);

			compiler_interface->SetDataModifiedFlag(true);
		}
	}
}

void	CRuleVBoxWidget::InsertBeforeRule(int i, const CRule &r)
{
	compiler_data->InsertBeforeRule(i, r);

	UpdateListView();
}

void	CRuleVBoxWidget::InsertAfterRule(int i, const CRule &r)
{
	compiler_data->InsertAfterRule(i, r);

	UpdateListView();
}

void	CRuleVBoxWidget::UpdateRule(int i, const CRule &r)
{
	compiler_data->UpdateRule(i, r);

	UpdateListView();
}

void	CRuleVBoxWidget::DeleteRule(int i)
{
	compiler_data->DeleteRule(i);

	UpdateListView();
}

void	CRuleVBoxWidget::MoveRuleUp(int i)
{
	compiler_data->MoveRuleUp(i);

	UpdateListView();
}

void	CRuleVBoxWidget::MoveRuleDown(int i)
{
	compiler_data->MoveRuleDown(i);

	UpdateListView();
}

void	CRuleVBoxWidget::RestoreUndoData()
{
	compiler_data = compiler_interface->GetCompilerData();
}

CRuleView::CRuleView(	QWidget *p,
						CViewNode *n,
						CInterfaceControl *ic,
						SMainViewControl *mvc,
						const string &dfs,
						CPaletteData *dpd,
						COptionData::SSubViewData *d)
	: CSubView(p, n, ic, mvc, dfs, dpd, d)
{
	frame = new CRuleVBoxWidget(
		p, dynamic_cast<CCompilerInterface *>(ic), mvc, d, this);

	QFont		view_node_font;
	QPalette	view_node_palette;

	if (SetupViewNodeFont(view_node_font))
		frame->setFont(view_node_font);
	else
		frame->setFont(frame->font());

	if (SetupViewNodePalette(view_node_palette))
	{
		frame->setPalette(view_node_palette);
		frame->InitializePalette(view_node_palette);
	}
	else
	{
		frame->setPalette(frame->palette());
		frame->InitializePalette(frame->palette());
	}
}

CRuleView::~CRuleView()
{
	delete frame;
}

void	CRuleView::SetHighlightPalette(CPaletteData *pd)
{
	restore_palette = frame->palette();

	QPalette	highlight_palette;

	pd->SetupPalette(highlight_palette);

	frame->setPalette(highlight_palette);
}

void	CRuleView::RestorePalette()
{
	frame->setPalette(restore_palette);
}

void	CRuleView::InitializeModulePointers()
{
	frame->InitializeModulePointers();
}

void	CRuleView::InsertAfterRule()
{
	frame->InsertAfterRule();
}

void	CRuleView::InsertBeforeRule()
{
	frame->InsertBeforeRule();
}

void	CRuleView::EditRule()
{
	frame->EditRule();
}

void	CRuleView::DeleteRule()
{
	frame->DeleteRule();
}

void	CRuleView::MoveRuleUp()
{
	frame->MoveRuleUp();
}

void	CRuleView::MoveRuleDown()
{
	frame->MoveRuleDown();
}

void	CRuleView::UpdateListView()
{
	frame->UpdateListView();
}

void	CRuleView::RestoreUndoData()
{
	frame->RestoreUndoData();
}
