/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "simulateInputView.h"

bool	CSimulateInputMultiLineEdit::event(QEvent *e)
{
	bool	tab_flag = simulate_input_vbox->GetTabIndicator();

	if (e->type() == QEvent::KeyPress)
	{
		if (((QKeyEvent *)e)->key() == Qt::Key_Escape)
		{
			simulate_input_vbox->ToggleTabIndicator();

			if (!tab_flag)
			{
				this->keyPressEvent((QKeyEvent *)e);
				return true;
			}
		}
		else
		{
			if (((QKeyEvent *)e)->key() == Qt::Key_F1)
				simulate_input_vbox->ToggleSimulateInputIndicator();
			else if (((QKeyEvent *)e)->key() == Qt::Key_G &&
				((QKeyEvent *)e)->state() == Qt::ControlButton)
				simulate_input_vbox->FindString();
			else if (((QKeyEvent *)e)->key() == Qt::Key_O &&
				((QKeyEvent *)e)->state() == Qt::ControlButton)
				simulate_input_vbox->OpenSimulateInputFile();
			else if (((QKeyEvent *)e)->key() == Qt::Key_S &&
				((QKeyEvent *)e)->state() == Qt::ControlButton)
				simulate_input_vbox->SaveSimulateInputFile();

			if (tab_flag)
			{
				this->keyPressEvent((QKeyEvent *)e);
				return true;
			}
		}
	}
	else if (e->type() == QEvent::KeyRelease)
	{
		if (tab_flag)
		{
			this->keyReleaseEvent((QKeyEvent *)e);
			return true;
		}
	}

	return CMultiLineEditFc::event(e);
}

void	CSimulateInputMultiLineEdit::focusInEvent(QFocusEvent *e)
{
	simulate_input_vbox->SetModulePointer();

	CMultiLineEditFc::focusInEvent(e);
}

void	CSimulateInputMultiLineEdit::InputCursorRight(bool mark, bool wrap)
{
	cursorRight(mark, wrap);
}

CSimulateInputVBoxWidget::CSimulateInputVBoxWidget(
										QWidget *parent,
										CCompilerInterface *ci,
										SMainViewControl *mvc,
										COptionData::SSubViewData *d,
										CSimulateInputView *siv)
	: CVBoxWidget(parent, ci, mvc), compiler_interface(ci),
		subview_data(d), simulate_input_view(siv),
		module_ptr_map(siv->GetModulePtrMap())
{
	label = new CLabelFc(" input", this, true);

	label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	multi_line_edit = new CSimulateInputMultiLineEdit(this);

	multi_line_edit->setWordWrap(QMultiLineEdit::NoWrap);
	multi_line_edit->setWrapPolicy(QMultiLineEdit::AtWhiteSpace);

	setSpacing(0);
	setStretchFactor(label, 0);
	setStretchFactor(multi_line_edit, 1);

	tab_indicator				= false;
	simulate_input_indicator	= false;
	file_modified_flag			= false;
	current_file_path			= "(none)";
	current_paragraph			= 1;
	current_column				= 1;

	compiler_data = compiler_interface->GetCompilerData();

	connect(multi_line_edit,
			SIGNAL(cursorPositionChanged(int, int)),
			SLOT(UpdateCursorPosition(int, int)));

	connect(multi_line_edit,
			SIGNAL(textChanged()),
			SLOT(CodeTextChanged()));

	ToggleSimulateInputIndicator();
}

CSimulateInputVBoxWidget::~CSimulateInputVBoxWidget()
{
}

void	CSimulateInputVBoxWidget::ClearMultiLineEditText()
{
	disconnect(	multi_line_edit,
				SIGNAL(textChanged()),
				this,
				SLOT(CodeTextChanged()));

	multi_line_edit->clear();

	connect(multi_line_edit,
			SIGNAL(textChanged()),
			this,
			SLOT(CodeTextChanged()));

	if (simulate_input_indicator)
	{
		simulate_input_indicator = false;

		ToggleSimulateInputIndicator();
	}
}

void	CSimulateInputVBoxWidget::SetTabStopWidth(int count, bool f)
{
	QFontMetrics	fm(font());

	int		w = fm.width(QChar('Z'));

	multi_line_edit->setTabStopWidth(count * w);

	if (f)
	{
		vector<CSubView *>		&v1 = module_ptr_map["SIMULATE_INPUT_VIEW"];

		vector<CSubView *>::size_type	i;

		for (i=0; i<v1.size(); i++)
		{
			CSimulateInputView	*sv = dynamic_cast<CSimulateInputView *>(v1[i]);

			if (sv != simulate_input_view)
				sv->SetTabStopWidth(count, false);
		}
	}
}

void	CSimulateInputVBoxWidget::SetWordWrap(bool wrap, bool f)
{
	if (wrap)
	{
		multi_line_edit->setWordWrap(QMultiLineEdit::WidgetWidth);
		multi_line_edit->setWrapPolicy(QMultiLineEdit::Anywhere);
	}
	else
	{
		multi_line_edit->setWordWrap(QMultiLineEdit::NoWrap);
		multi_line_edit->setWrapPolicy(QMultiLineEdit::AtWhiteSpace);
	}

	if (f)
	{
		vector<CSubView *>		&v1 = module_ptr_map["SIMULATE_INPUT_VIEW"];

		vector<CSubView *>::size_type	i;

		for (i=0; i<v1.size(); i++)
		{
			CSimulateInputView	*sv = dynamic_cast<CSimulateInputView *>(v1[i]);

			if (sv != simulate_input_view)
				sv->SetWordWrap(wrap, false);
		}
	}
}

void	CSimulateInputVBoxWidget::SetMultiLineEditText(const list<string> &code)
{
	disconnect(	multi_line_edit,
				SIGNAL(textChanged()),
				this,
				SLOT(CodeTextChanged()));

	multi_line_edit->clear();

	list<string>::const_iterator	s_iterator = code.begin();

	while (s_iterator != code.end())
	{
		const string	&s = *s_iterator++;

		if (!s.length())
			multi_line_edit->append("\n");
		else
			multi_line_edit->append(s.c_str());
	}

	multi_line_edit->setCursorPosition(0, 0);

	connect(multi_line_edit,
			SIGNAL(textChanged()),
			this,
			SLOT(CodeTextChanged()));

	if (simulate_input_indicator)
	{
		simulate_input_indicator = false;

		ToggleSimulateInputIndicator();
	}
}

void	CSimulateInputVBoxWidget::GetMultiLineEditText(list<string> &code)
{
	code.clear();

	int		count = multi_line_edit->numLines();
	int		i;

	for (i=0; i<count; i++)
		code.push_back(
			(const char *)(multi_line_edit->textLine(i).local8Bit()));

	if (code.back() == string(""))
		code.pop_back();
}

bool	CSimulateInputVBoxWidget::GetCodeEditedFlag()
{
	return multi_line_edit->edited();
}

void	CSimulateInputVBoxWidget::SetCodeEditedFlag(bool f)
{
	multi_line_edit->setEdited(f);
}

bool	CSimulateInputVBoxWidget::GetTabIndicator()
{
	return tab_indicator;
}

void	CSimulateInputVBoxWidget::ToggleTabIndicator()
{
	string	s1(" input");
	string	s2;
	string	s3;

	if (tab_indicator)
	{
		tab_indicator = false;
	}
	else
	{
		s1 += " (tab entry)";

		tab_indicator = true;
	}

	if (simulate_input_indicator)
	{
		GetCurrentSimulateInputString(s2);
		GetCursorPositionString(s3);

		s1 += ": ";

		if (file_modified_flag)
			s1 += "*";

		s1 += s2;
		s1 += s3;

		label->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
	}
	else
		label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	label->setText(s1.c_str());
}

void	CSimulateInputVBoxWidget::ToggleSimulateInputIndicator()
{
	string	s1(" input");
	string	s2;
	string	s3;

	if (tab_indicator)
		s1 += " (tab entry)";

	if (simulate_input_indicator)
	{
		label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

		simulate_input_indicator = false;
	}
	else
	{
		GetCurrentSimulateInputString(s2);
		GetCursorPositionString(s3);

		s1 += ": ";

		if (file_modified_flag)
			s1 += "*";

		s1 += s2;
		s1 += s3;

		label->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);

		simulate_input_indicator = true;
	}

	label->setText(s1.c_str());
}

void	CSimulateInputVBoxWidget::GetCurrentSimulateInputString(string &s)
{
	s = "file - " + current_file_path;
}

void	CSimulateInputVBoxWidget::GetCursorPositionString(string &s)
{
	ostrstream	os;

	os << "  " << current_paragraph << "," << current_column << '\0';

	s = os.str();
}

void	CSimulateInputVBoxWidget::OpenSimulateInputFile()
{
	vector< pair<string, string> >		file_types;
	string								simulate_input_file_name;
	string								line;

	file_types.push_back(pair<string, string>("Input Files", "*.input"));
	file_types.push_back(pair<string, string>("Text Files", "*.txt"));
	file_types.push_back(pair<string, string>("Include Files", "*.h"));
	file_types.push_back(pair<string, string>("Source Files", "*.cpp"));
	file_types.push_back(pair<string, string>("All Files", "*"));

	CFileDirectoryDialog	*fdd = OptionDialog::file_directory_dialog;

	if (fdd->DisplayOpenFileDialog(	&simulate_input_file_name,
									"Open Simulate Input File",
									string(),
									string(),
									&file_types) != QDialog::Accepted)
		return;

	ifstream	s_file(simulate_input_file_name.c_str());

	if (s_file)
	{
		current_file_path = simulate_input_file_name;

		list<string>	simulate_input;

		while (getline(s_file, line, '\0'))
			simulate_input.push_back(line);

		file_modified_flag = false;

		SetMultiLineEditText(simulate_input);
	}
}

void	CSimulateInputVBoxWidget::ClearSimulateInputFile()
{
	current_file_path = "(none)";

	file_modified_flag = false;

	ClearMultiLineEditText();
}

void	CSimulateInputVBoxWidget::SaveSimulateInputFile()
{
	vector< pair<string, string> >		file_types;
	string								simulate_input_file_name;
	string								d_path;
	string								f_name;

	FilePath::GetPathAndFileName(current_file_path, d_path, f_name);

	file_types.push_back(pair<string, string>("Input Files", "*.input"));
	file_types.push_back(pair<string, string>("Text Files", "*.txt"));
	file_types.push_back(pair<string, string>("Include Files", "*.h"));
	file_types.push_back(pair<string, string>("Source Files", "*.cpp"));
	file_types.push_back(pair<string, string>("All Files", "*"));

	CFileDirectoryDialog	*fdd = OptionDialog::file_directory_dialog;

	if (fdd->DisplaySaveAsFileDialog(	&simulate_input_file_name,
										"Save As Simulate Input File",
										d_path,
										f_name,
										&file_types) != QDialog::Accepted)
		return;

	if (simulate_input_file_name != current_file_path &&
		FilePath::FileExists(simulate_input_file_name))
	{
		CMessageDialog		*msd = OptionDialog::message_dialog;

		string	s("Overwrite existing file named ");
		s += simulate_input_file_name;
		s += " ?";

		msd->SetTitleAndMessageText("Save As File", s);

		if (msd->ExecuteDialog() == QDialog::Accepted)
			SaveSimulateInputFile(simulate_input_file_name);
		else
			return;
	}
	else
		SaveSimulateInputFile(simulate_input_file_name);
}

void	CSimulateInputVBoxWidget::SaveSimulateInputFile(const string &s)
{
	current_file_path = s;

	ofstream	s_file(current_file_path.c_str());

	list<string>	simulate_input;

	GetMultiLineEditText(simulate_input);

	list<string>::iterator	s_iterator = simulate_input.begin();

	while (s_iterator != simulate_input.end())
		s_file << *s_iterator++ << endl;

	file_modified_flag = false;

	if (simulate_input_indicator)
	{
		simulate_input_indicator = false;

		ToggleSimulateInputIndicator();
	}
}

void	CSimulateInputVBoxWidget::UndoEdit()
{
	multi_line_edit->undo();
}

void	CSimulateInputVBoxWidget::RedoEdit()
{
	multi_line_edit->redo();
}

void	FindMultiLineEditString(QMultiLineEdit *multi_line_edit);

void	CSimulateInputVBoxWidget::FindString()
{
	FindMultiLineEditString(multi_line_edit);
}

void	CSimulateInputVBoxWidget::SetModulePointer()
{
	compiler_interface->SetModulePointer(	simulate_input_view,
											"SIMULATE_INPUT_VIEW");
}

void	CSimulateInputVBoxWidget::UpdateCursorPosition(int p, int c)
{
	current_paragraph	= p + 1;
	current_column		= c + 1;

	if (simulate_input_indicator)
	{
		simulate_input_indicator = false;

		ToggleSimulateInputIndicator();
	}
}

void	CSimulateInputVBoxWidget::InitializeModulePointers()
{
}

void	CSimulateInputVBoxWidget::CodeTextChanged()
{
	file_modified_flag = true;

	if (simulate_input_indicator)
	{
		simulate_input_indicator = false;

		ToggleSimulateInputIndicator();
	}
}

void	CSimulateInputVBoxWidget::GetInputText(QString &s)
{
	s = multi_line_edit->text();
}

void	CSimulateInputVBoxWidget::SetInputCursorPosition(int line, int column)
{
	multi_line_edit->setCursorPosition(line, column, false);
}

void	CSimulateInputVBoxWidget::HighlightInputText(int length)
{
	int		line;
	int		column;

	multi_line_edit->getCursorPosition(&line, &column);
	multi_line_edit->setCursorPosition(line, column, false);

	for (int i=0; i<length; i++)
		multi_line_edit->InputCursorRight(true, true);
}

void	CSimulateInputVBoxWidget::RestoreUndoData()
{
	compiler_data = compiler_interface->GetCompilerData();
}

CSimulateInputView::CSimulateInputView(
							QWidget *p,
							CViewNode *n,
							CInterfaceControl *ic,
							SMainViewControl *mvc,
							const string &dfs,
							CPaletteData *dpd,
							COptionData::SSubViewData *d)
	: CSubView(p, n, ic, mvc, dfs, dpd, d)
{
	frame = new CSimulateInputVBoxWidget(
		p, dynamic_cast<CCompilerInterface *>(ic), mvc, d, this);

	QFont		view_node_font;
	QPalette	view_node_palette;

	if (SetupViewNodeFont(view_node_font))
		frame->setFont(view_node_font);
	else
		frame->setFont(frame->font());

	if (SetupViewNodePalette(view_node_palette))
		frame->setPalette(view_node_palette);
	else
		frame->setPalette(frame->palette());
}

CSimulateInputView::~CSimulateInputView()
{
	delete frame;
}

void	CSimulateInputView::SetHighlightPalette(CPaletteData *pd)
{
	restore_palette = frame->palette();

	QPalette	highlight_palette;

	pd->SetupPalette(highlight_palette);

	frame->setPalette(highlight_palette);
}

void	CSimulateInputView::RestorePalette()
{
	frame->setPalette(restore_palette);
}

void	CSimulateInputView::InitializeModulePointers()
{
	frame->InitializeModulePointers();
}

void	CSimulateInputView::ClearMultiLineEditText()
{
	frame->ClearMultiLineEditText();
}

void	CSimulateInputView::SetTabStopWidth(int count, bool f)
{
	frame->SetTabStopWidth(count, f);
}

void	CSimulateInputView::SetWordWrap(bool wrap, bool f)
{
	frame->SetWordWrap(wrap, f);
}

void	CSimulateInputView::SetMultiLineEditText(const list<string> &code)
{
	frame->SetMultiLineEditText(code);
}

void	CSimulateInputView::GetMultiLineEditText(list<string> &code)
{
	frame->GetMultiLineEditText(code);
}

bool	CSimulateInputView::GetCodeEditedFlag()
{
	return frame->GetCodeEditedFlag();
}

void	CSimulateInputView::SetCodeEditedFlag(bool f)
{
	frame->SetCodeEditedFlag(f);
}

void	CSimulateInputView::OpenSimulateInputFile()
{
	frame->OpenSimulateInputFile();
}

void	CSimulateInputView::ClearSimulateInputFile()
{
	frame->ClearSimulateInputFile();
}

void	CSimulateInputView::SaveSimulateInputFile()
{
	frame->SaveSimulateInputFile();
}

void	CSimulateInputView::UndoEdit()
{
	frame->UndoEdit();
}

void	CSimulateInputView::RedoEdit()
{
	frame->RedoEdit();
}

void	CSimulateInputView::FindString()
{
	frame->FindString();
}

void	CSimulateInputView::GetInputText(QString &s)
{
	frame->GetInputText(s);
}

void	CSimulateInputView::SetInputCursorPosition(int line, int column)
{
	frame->SetInputCursorPosition(line, column);
}

void	CSimulateInputView::HighlightInputText(int length)
{
	frame->HighlightInputText(length);
}

void	CSimulateInputView::RestoreUndoData()
{
	frame->RestoreUndoData();
}
