/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "sourceView.h"

bool	CSourceMultiLineEdit::event(QEvent *e)
{
	bool	tab_flag = source_vbox->GetTabIndicator();

	if (e->type() == QEvent::KeyPress)
	{
		if (((QKeyEvent *)e)->key() == Qt::Key_Escape)
		{
			source_vbox->ToggleTabIndicator();

			if (!tab_flag)
			{
				this->keyPressEvent((QKeyEvent *)e);
				return true;
			}
		}
		else
		{
			if (((QKeyEvent *)e)->key() == Qt::Key_F1)
				source_vbox->ToggleSourceIndicator();
			else if (((QKeyEvent *)e)->key() == Qt::Key_G &&
				((QKeyEvent *)e)->state() == Qt::ControlButton)
				source_vbox->FindString();
			else if (((QKeyEvent *)e)->key() == Qt::Key_O &&
				((QKeyEvent *)e)->state() == Qt::ControlButton)
				source_vbox->OpenSourceFile();
			else if (((QKeyEvent *)e)->key() == Qt::Key_S &&
				((QKeyEvent *)e)->state() == Qt::ControlButton)
				source_vbox->SaveSourceFile();

			if (tab_flag)
			{
				this->keyPressEvent((QKeyEvent *)e);
				return true;
			}
		}
	}
	else if (e->type() == QEvent::KeyRelease)
	{
		if (tab_flag)
		{
			this->keyReleaseEvent((QKeyEvent *)e);
			return true;
		}
	}

	return CMultiLineEditFc::event(e);
}

void	CSourceMultiLineEdit::focusInEvent(QFocusEvent *e)
{
	source_vbox->SetModulePointer();

	CMultiLineEditFc::focusInEvent(e);
}

CSourceVBoxWidget::CSourceVBoxWidget(	QWidget *parent,
										CCompilerInterface *ci,
										SMainViewControl *mvc,
										COptionData::SSubViewData *d,
										CSourceView *sv)
	: CVBoxWidget(parent, ci, mvc), compiler_interface(ci),
		subview_data(d), source_view(sv),
		module_ptr_map(sv->GetModulePtrMap())
{
	label = new CLabelFc(" source code", this, true);

	label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	multi_line_edit = new CSourceMultiLineEdit(this);

	multi_line_edit->setWordWrap(QMultiLineEdit::NoWrap);

	setSpacing(0);
	setStretchFactor(label, 0);
	setStretchFactor(multi_line_edit, 1);

	tab_indicator		= false;
	source_indicator	= false;
	file_modified_flag	= false;
	current_file_path	= "(none)";
	current_paragraph	= 1;
	current_column		= 1;

	compiler_data = compiler_interface->GetCompilerData();

	connect(multi_line_edit,
			SIGNAL(cursorPositionChanged(int, int)),
			SLOT(UpdateCursorPosition(int, int)));

	connect(multi_line_edit,
			SIGNAL(textChanged()),
			SLOT(CodeTextChanged()));

	ToggleSourceIndicator();
}

CSourceVBoxWidget::~CSourceVBoxWidget()
{
}

void	CSourceVBoxWidget::ClearMultiLineEditText()
{
	disconnect(	multi_line_edit,
				SIGNAL(textChanged()),
				this,
				SLOT(CodeTextChanged()));

	multi_line_edit->clear();

	connect(multi_line_edit,
			SIGNAL(textChanged()),
			this,
			SLOT(CodeTextChanged()));

	if (source_indicator)
	{
		source_indicator = false;

		ToggleSourceIndicator();
	}
}

void	CSourceVBoxWidget::SetTabStopWidth(int count, bool f)
{
	QFontMetrics	fm(font());

	int		w = fm.width(QChar('Z'));

	multi_line_edit->setTabStopWidth(count * w);

	if (f)
	{
		vector<CSubView *>		&v1 = module_ptr_map["SOURCE_VIEW"];

		vector<CSubView *>::size_type	i;

		for (i=0; i<v1.size(); i++)
		{
			CSourceView		*sv = dynamic_cast<CSourceView *>(v1[i]);

			if (sv != source_view)
				sv->SetTabStopWidth(count, false);
		}
	}
}

void	CSourceVBoxWidget::SetMultiLineEditText(const list<string> &code)
{
	disconnect(	multi_line_edit,
				SIGNAL(textChanged()),
				this,
				SLOT(CodeTextChanged()));

	multi_line_edit->clear();

	list<string>::const_iterator	s_iterator = code.begin();

	while (s_iterator != code.end())
	{
		const string	&s = *s_iterator++;

		if (!s.length())
			multi_line_edit->append("\n");
		else
			multi_line_edit->append(s.c_str());
	}

	multi_line_edit->setCursorPosition(0, 0);

	connect(multi_line_edit,
			SIGNAL(textChanged()),
			this,
			SLOT(CodeTextChanged()));

	if (source_indicator)
	{
		source_indicator = false;

		ToggleSourceIndicator();
	}
}

void	CSourceVBoxWidget::GetMultiLineEditText(list<string> &code)
{
	code.clear();

	int		count = multi_line_edit->numLines();
	int		i;

	for (i=0; i<count; i++)
		code.push_back(
			(const char *)(multi_line_edit->textLine(i).local8Bit()));

	if (code.back() == string(""))
		code.pop_back();
}

bool	CSourceVBoxWidget::GetCodeEditedFlag()
{
	return multi_line_edit->edited();
}

void	CSourceVBoxWidget::SetCodeEditedFlag(bool f)
{
	multi_line_edit->setEdited(f);
}

bool	CSourceVBoxWidget::GetTabIndicator()
{
	return tab_indicator;
}

void	CSourceVBoxWidget::ToggleTabIndicator()
{
	string	s1(" source code");
	string	s2;
	string	s3;

	if (tab_indicator)
	{
		tab_indicator = false;
	}
	else
	{
		s1 += " (tab entry)";

		tab_indicator = true;
	}

	if (source_indicator)
	{
		GetCurrentSourceString(s2);
		GetCursorPositionString(s3);

		s1 += ": ";

		if (file_modified_flag)
			s1 += "*";

		s1 += s2;
		s1 += s3;

		label->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
	}
	else
		label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	label->setText(s1.c_str());
}

void	CSourceVBoxWidget::ToggleSourceIndicator()
{
	string	s1(" source code");
	string	s2;
	string	s3;

	if (tab_indicator)
		s1 += " (tab entry)";

	if (source_indicator)
	{
		label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

		source_indicator = false;
	}
	else
	{
		GetCurrentSourceString(s2);
		GetCursorPositionString(s3);

		s1 += ": ";

		if (file_modified_flag)
			s1 += "*";

		s1 += s2;
		s1 += s3;

		label->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);

		source_indicator = true;
	}

	label->setText(s1.c_str());
}

void	CSourceVBoxWidget::GetCurrentSourceString(string &s)
{
	s = "file - " + current_file_path;
}

void	CSourceVBoxWidget::GetCursorPositionString(string &s)
{
	ostrstream	os;

	os << "  " << current_paragraph << "," << current_column << '\0';

	s = os.str();
}

void	CSourceVBoxWidget::OpenSourceFile()
{
	vector< pair<string, string> >		file_types;
	string								source_file_name;
	string								line;

	file_types.push_back(pair<string, string>("Text Files", "*.txt"));
	file_types.push_back(pair<string, string>("Include Files", "*.h"));
	file_types.push_back(pair<string, string>("Source Files", "*.cpp"));
	file_types.push_back(pair<string, string>("All Files", "*"));

	CFileDirectoryDialog	*fdd = OptionDialog::file_directory_dialog;

	if (fdd->DisplayOpenFileDialog(	&source_file_name,
									"Open Source File",
									string(),
									string(),
									&file_types) != QDialog::Accepted)
		return;

	ifstream	s_file(source_file_name.c_str());

	if (s_file)
	{
		current_file_path = source_file_name;

		list<string>	source;

		while (getline(s_file, line, '\0'))
			source.push_back(line);

		file_modified_flag = false;

		SetMultiLineEditText(source);
	}
}

void	CSourceVBoxWidget::ClearSourceFile()
{
	current_file_path = "(none)";

	file_modified_flag = false;

	ClearMultiLineEditText();
}

void	CSourceVBoxWidget::SaveSourceFile()
{
	vector< pair<string, string> >		file_types;
	string								source_file_name;
	string								d_path;
	string								f_name;

	FilePath::GetPathAndFileName(current_file_path, d_path, f_name);

	file_types.push_back(pair<string, string>("Text Files", "*.txt"));
	file_types.push_back(pair<string, string>("Include Files", "*.h"));
	file_types.push_back(pair<string, string>("Source Files", "*.cpp"));
	file_types.push_back(pair<string, string>("All Files", "*"));

	CFileDirectoryDialog	*fdd = OptionDialog::file_directory_dialog;

	if (fdd->DisplaySaveAsFileDialog(	&source_file_name,
										"Save As Source File",
										d_path,
										f_name,
										&file_types) != QDialog::Accepted)
		return;

	if (source_file_name != current_file_path &&
		FilePath::FileExists(source_file_name))
	{
		CMessageDialog		*msd = OptionDialog::message_dialog;

		string	s("Overwrite existing file named ");
		s += source_file_name;
		s += " ?";

		msd->SetTitleAndMessageText("Save As File", s);

		if (msd->ExecuteDialog() == QDialog::Accepted)
			SaveSourceFile(source_file_name);
		else
			return;
	}
	else
		SaveSourceFile(source_file_name);
}

void	CSourceVBoxWidget::SaveSourceFile(const string &s)
{
	current_file_path = s;

	ofstream	s_file(current_file_path.c_str());

	list<string>	source;

	GetMultiLineEditText(source);

	list<string>::iterator	s_iterator = source.begin();

	while (s_iterator != source.end())
		s_file << *s_iterator++ << endl;

	file_modified_flag = false;

	if (source_indicator)
	{
		source_indicator = false;

		ToggleSourceIndicator();
	}
}

void	CSourceVBoxWidget::UndoEdit()
{
	multi_line_edit->undo();
}

void	CSourceVBoxWidget::RedoEdit()
{
	multi_line_edit->redo();
}

void	FindMultiLineEditString(QMultiLineEdit *multi_line_edit)
{
	CValueDialog	*d = OptionDialog::value_dialog;

	static string	last_find_string;
	string			value;

	d->SetTitleAndLabelText("Find String In Source", "String");
	d->SetValuePointer(&value);
	d->SetValueLineEdit(last_find_string.c_str());

	if (d->exec() == QDialog::Rejected)
		return;

	last_find_string = value;

	int		paragraph;
	int		index;

	multi_line_edit->getCursorPosition(&paragraph, &index);

	if (multi_line_edit->find(value.c_str(), true, false, true,
								&paragraph, &index))
	{
		multi_line_edit->setSelection(paragraph, index, paragraph,
										index + value.size());
	}
	else
	{
		CInformationDialog	*ifd = OptionDialog::information_dialog;

		string		s("String \"");

		s += value;
		s += "\" not found.";

		ifd->SetTitleAndMessageText("Find String", s);

		ifd->ExecuteDialog();
	}
}

void	CSourceVBoxWidget::FindString()
{
	FindMultiLineEditString(multi_line_edit);
}

void	CSourceVBoxWidget::SetModulePointer()
{
	compiler_interface->SetModulePointer(source_view, "SOURCE_VIEW");
}

void	CSourceVBoxWidget::UpdateCursorPosition(int p, int c)
{
	current_paragraph	= p + 1;
	current_column		= c + 1;

	if (source_indicator)
	{
		source_indicator = false;

		ToggleSourceIndicator();
	}
}

void	CSourceVBoxWidget::InitializeModulePointers()
{
}

void	CSourceVBoxWidget::CodeTextChanged()
{
	file_modified_flag = true;

	if (source_indicator)
	{
		source_indicator = false;

		ToggleSourceIndicator();
	}
}

CSourceView::CSourceView(	QWidget *p,
							CViewNode *n,
							CInterfaceControl *ic,
							SMainViewControl *mvc,
							const string &dfs,
							CPaletteData *dpd,
							COptionData::SSubViewData *d)
	: CSubView(p, n, ic, mvc, dfs, dpd, d)
{
	frame = new CSourceVBoxWidget(
		p, dynamic_cast<CCompilerInterface *>(ic), mvc, d, this);

	QFont		view_node_font;
	QPalette	view_node_palette;

	if (SetupViewNodeFont(view_node_font))
		frame->setFont(view_node_font);
	else
		frame->setFont(frame->font());

	if (SetupViewNodePalette(view_node_palette))
		frame->setPalette(view_node_palette);
	else
		frame->setPalette(frame->palette());
}

CSourceView::~CSourceView()
{
	delete frame;
}

void	CSourceView::SetHighlightPalette(CPaletteData *pd)
{
	restore_palette = frame->palette();

	QPalette	highlight_palette;

	pd->SetupPalette(highlight_palette);

	frame->setPalette(highlight_palette);
}

void	CSourceView::RestorePalette()
{
	frame->setPalette(restore_palette);
}

void	CSourceView::InitializeModulePointers()
{
	frame->InitializeModulePointers();
}

void	CSourceView::ClearMultiLineEditText()
{
	frame->ClearMultiLineEditText();
}

void	CSourceView::SetTabStopWidth(int count, bool f)
{
	frame->SetTabStopWidth(count, f);
}

void	CSourceView::SetMultiLineEditText(const list<string> &code)
{
	frame->SetMultiLineEditText(code);
}

void	CSourceView::GetMultiLineEditText(list<string> &code)
{
	frame->GetMultiLineEditText(code);
}

bool	CSourceView::GetCodeEditedFlag()
{
	return frame->GetCodeEditedFlag();
}

void	CSourceView::SetCodeEditedFlag(bool f)
{
	frame->SetCodeEditedFlag(f);
}

void	CSourceView::OpenSourceFile()
{
	frame->OpenSourceFile();
}

void	CSourceView::ClearSourceFile()
{
	frame->ClearSourceFile();
}

void	CSourceView::SaveSourceFile()
{
	frame->SaveSourceFile();
}

void	CSourceView::UndoEdit()
{
	frame->UndoEdit();
}

void	CSourceView::RedoEdit()
{
	frame->RedoEdit();
}

void	CSourceView::FindString()
{
	frame->FindString();
}
