/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "stringToken.h"

StringToken::TokenStatus		StringToken::GetCharTokenSequence(
									const string &s, const string &d,
									vector<CTokenValue *> &tv)
{
	TInfoVector		info_vector;

	GetInfoVector(s, d, info_vector);

	TInfoVector::size_type	i;

	for (i=0; i<info_vector.size(); i++)
		tv.push_back(CreateNewCharToken(s, info_vector[i]));

	return NoError;
}

StringToken::TokenStatus		StringToken::GetIntegerTokenSequence(
									const string &s, const string &d,
									vector<CTokenValue *> &tv)
{
	TInfoVector		info_vector;
	int				operation_status = NoError;

	GetInfoVector(s, d, info_vector);

	TInfoVector::size_type	i;

	for (i=0; i<info_vector.size(); i++)
	{
		CIntegerToken	*it = CreateNewIntegerToken(s, info_vector[i]);

		tv.push_back(it);

		operation_status |= it->GetTokenStatus();
	}

	return (TokenStatus)operation_status;
}

StringToken::TokenStatus		StringToken::GetDoubleTokenSequence(
									const string &s, const string &d,
									vector<CTokenValue *> &tv)
{
	TInfoVector		info_vector;
	int				operation_status = NoError;

	GetInfoVector(s, d, info_vector);

	TInfoVector::size_type	i;

	for (i=0; i<info_vector.size(); i++)
	{
		CDoubleToken	*dt = CreateNewDoubleToken(s, info_vector[i]);

		tv.push_back(dt);

		operation_status |= dt->GetTokenStatus();
	}

	return (TokenStatus)operation_status;
}

StringToken::TokenStatus		StringToken::GetMappedTokenSequence(
									const string &s, const string &d,
									const map<string, long int> &tm,
									vector<CTokenValue *> &tv)
{
	TInfoVector		info_vector;
	int				operation_status = NoError;

	GetInfoVector(s, d, info_vector);

	TInfoVector::size_type	i;

	for (i=0; i<info_vector.size(); i++)
	{
		CMappedToken	*mt = CreateNewMappedToken(s, info_vector[i], tm);

		tv.push_back(mt);

		operation_status |= mt->GetTokenStatus();
	}

	return (TokenStatus)operation_status;
}

StringToken::TokenStatus		StringToken::GetTokenSequence(
									const string &s, const string &d,
									const vector<TokenType> &tt,
									vector<CTokenValue *> &tv)
{
	const map<string, long int>		empty_token_map;

	return GetTokenSequence(s, d, tt, empty_token_map, tv);
}

StringToken::TokenStatus		StringToken::GetTokenSequence(
									const string &s, const string &d,
									const vector<TokenType> &tt,
									const map<string, long int> &tm, 
									vector<CTokenValue *> &tv)
{
	TInfoVector		info_vector;
	int				operation_status = NoError;
	CTokenValue		*tv_ptr;

	GetInfoVector(s, d, info_vector);

	TInfoVector::size_type			iv_size = info_vector.size();
	vector<TokenType>::size_type	tt_size = tt.size();
	vector<TokenType>::size_type	i;

	if (tt_size <= iv_size)
	{
		for (i=0; i<tt_size; i++)
		{
			switch (tt[i])
			{
				case CharToken:
				tv_ptr = CreateNewCharToken(s, info_vector[i]);
				break;

				case IntegerToken:
				tv_ptr = CreateNewIntegerToken(s, info_vector[i]);
				break;

				case DoubleToken:
				tv_ptr = CreateNewDoubleToken(s, info_vector[i]);
				break;

				case MappedToken:
				tv_ptr = CreateNewMappedToken(s, info_vector[i], tm);
				break;
			}

			tv.push_back(tv_ptr);

			operation_status |= tv_ptr->GetTokenStatus();
		}

		if (tt_size < iv_size)
			operation_status |= ExcessFields;
	}
	else
	{
		for (i=0; i<iv_size; i++)
		{
			switch (tt[i])
			{
				case CharToken:
				tv_ptr = CreateNewCharToken(s, info_vector[i]);
				break;

				case IntegerToken:
				tv_ptr = CreateNewIntegerToken(s, info_vector[i]);
				break;

				case DoubleToken:
				tv_ptr = CreateNewDoubleToken(s, info_vector[i]);
				break;

				case MappedToken:
				tv_ptr = CreateNewMappedToken(s, info_vector[i], tm);
				break;
			}

			tv.push_back(tv_ptr);

			operation_status |= tv_ptr->GetTokenStatus();
		}

		for (i=iv_size; i<tt_size; i++)
		{
			switch (tt[i])
			{
				case CharToken:
				tv_ptr = new CCharToken();
				break;

				case IntegerToken:
				tv_ptr = new CIntegerToken();
				break;

				case DoubleToken:
				tv_ptr = new CDoubleToken();
				break;

				case MappedToken:
				tv_ptr = new CMappedToken();
				break;
			}

			tv.push_back(tv_ptr);

			operation_status |= tv_ptr->GetTokenStatus();
		}

		operation_status |= InsufficientFields;
	}

	return (TokenStatus)operation_status;
}

void	StringToken::DeleteTokenVectorElements(vector<CTokenValue *> &tv)
{
	vector<CTokenValue *>::size_type	i;

	for (i=0; i<tv.size(); i++)
		delete tv[i];

	tv.clear();
}

void	StringToken::GetInfoVector(	const string &s, const string &d,
									TInfoVector &iv)
{
	string::size_type	start_position	= 0;
	string::size_type	end_position	= 0;

	while (start_position < s.length())
	{
		start_position = s.find_first_not_of(d, start_position);

		if (start_position != string::npos)
		{
			end_position = s.find_first_of(d, start_position + 1);

			if (end_position != string::npos)
			{
				string::size_type	n = end_position - start_position;

				iv.push_back(TSubstringInfo(start_position, n));

				start_position	= end_position + 1;
			}
			else
			{
				string::size_type	n = s.length() - start_position;

				iv.push_back(TSubstringInfo(start_position, n));

				break;
			}
		}
		else
			break;
	}
}

StringToken::CCharToken		*StringToken::CreateNewCharToken(
								const string &s, const TSubstringInfo &si)
{
	return new CCharToken(s.substr(si.first, si.second), NoError);
}

StringToken::CIntegerToken	*StringToken::CreateNewIntegerToken(
								const string &s, const TSubstringInfo &si)
{
	long int	value;
	char		*endptr;
	int			token_status = NoError;

	string	ts(s.substr(si.first, si.second));

	errno = 0;
	value = strtol(ts.c_str(), &endptr, 0);

	if (*endptr != '\0')
		token_status |= BadIntegerType;

	if (errno == ERANGE)
		token_status |= OutOfRange;

	return new CIntegerToken(ts, (TokenStatus)token_status, value);
}

StringToken::CDoubleToken	*StringToken::CreateNewDoubleToken(
								const string &s, const TSubstringInfo &si)
{
	double		value;
	char		*endptr;
	int			token_status = NoError;

	string	ts(s.substr(si.first, si.second));

	errno = 0;
	value = strtod(ts.c_str(), &endptr);

	if (*endptr != '\0')
		token_status |= BadDoubleType;

	if (errno == ERANGE)
		token_status |= OutOfRange;

	return new CDoubleToken(ts, (TokenStatus)token_status, value);
}

StringToken::CMappedToken	*StringToken::CreateNewMappedToken(
								const string &s,
								const TSubstringInfo &si,
								const map<string, long int> &tm)
{
	long int	value;
	int			token_status = NoError;

	string	ts(s.substr(si.first, si.second));

	map<string, long int>::const_iterator		map_iterator = tm.find(ts);

	if (map_iterator != tm.end())
	{
		value = (*map_iterator).second;
	}
	else
	{
		value = 0;
		token_status |= NoCharMapping;

		if (!tm.size())
			token_status |= NoTokenMap;
	}

	return new CMappedToken(ts, (TokenStatus)token_status, value);
}

ostream		&operator<<(ostream &s, const StringToken::CTokenValue &tv)
{
	return tv.OutputData(s);
}

ostream		&operator<<(ostream &s,
				const vector<StringToken::CTokenValue *> &tvv)
{
	int		i;
	int		digit_count;

	vector<StringToken::CTokenValue *>::size_type	n = tvv.size();

	if (!n)
		return s << "empty token vector" << endl;
	else if (n == 1)
		digit_count = 1;
	else
	{
		n--;
		digit_count	= 0;
		while (n)
		{
			digit_count++;
			n /= 10;
		}
	}

	for (i=0; i<tvv.size(); i++)
	{
		s.width(digit_count);
		s << i << ") status = ";

		StringToken::TokenStatus	status = tvv[i]->GetTokenStatus();

		if (status == StringToken::NoError)
			s << "NoError            ";
		else
		{
			if (status & StringToken::BadIntegerType)
				s << "BadIntegerType     ";

			if (status & StringToken::BadDoubleType)
				s << "BadDoubleType      ";

			if (status & StringToken::OutOfRange)
				s << "OutOfRange         ";

			if (status & StringToken::NoCharMapping)
				s << "NoCharMapping      ";

			if (status & StringToken::NoTokenMap)
				s << "NoTokenMap         ";

			if (status & StringToken::NoData)
				s << "NoData             ";

			if (status & StringToken::InsufficientFields)
				s << "InsufficientFields ";

			if (status & StringToken::ExcessFields)
				s << "ExcessFields       ";
		}

		switch (tvv[i]->GetTokenType())
		{
			case StringToken::CharToken:
			s << "CharToken    = ";
			break;

			case StringToken::IntegerToken:
			s << "IntegerToken = ";
			break;

			case StringToken::DoubleToken:
			s << "DoubleToken  = ";
			break;

			case StringToken::MappedToken:
			s << "MappedToken  = ";
			break;
		}

		s << *tvv[i] << endl;
	}

	return s;
}
