/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "terminalCodeView.h"

bool	CTerminalCodeMultiLineEdit::event(QEvent *e)
{
	bool	tab_flag = terminal_code_vbox->GetTabIndicator();

	if (e->type() == QEvent::KeyPress)
	{
		if (((QKeyEvent *)e)->key() == Qt::Key_Escape)
		{
			terminal_code_vbox->ToggleTabIndicator();

			if (!tab_flag)
			{
				this->keyPressEvent((QKeyEvent *)e);
				return true;
			}
		}
		else
		{
			if (((QKeyEvent *)e)->key() == Qt::Key_F1)
				terminal_code_vbox->ToggleTerminalIndicator();
			else if (((QKeyEvent *)e)->key() == Qt::Key_G &&
				((QKeyEvent *)e)->state() == Qt::ControlButton)
				terminal_code_vbox->FindString();

			if (tab_flag)
			{
				this->keyPressEvent((QKeyEvent *)e);
				return true;
			}
		}
	}
	else if (e->type() == QEvent::KeyRelease)
	{
		if (tab_flag)
		{
			this->keyReleaseEvent((QKeyEvent *)e);
			return true;
		}
	}

	return CMultiLineEditFc::event(e);
}

CTerminalCodeVBoxWidget::CTerminalCodeVBoxWidget(
									QWidget *parent,
									CCompilerInterface *ci,
									SMainViewControl *mvc,
									COptionData::SSubViewData *d,
									CTerminalCodeView *tcv)
	: CVBoxWidget(parent, ci, mvc), compiler_interface(ci),
		subview_data(d), terminal_code_view(tcv),
		module_ptr_map(tcv->GetModulePtrMap())
{
	label = new CLabelFc("terminal code", this, true);

	label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	tab_indicator		= false;
	terminal_indicator	= false;

	multi_line_edit = new CTerminalCodeMultiLineEdit(this);

	multi_line_edit->setWordWrap(QMultiLineEdit::NoWrap);

	connect(multi_line_edit,
			SIGNAL(textChanged()),
			this,
			SLOT(CodeTextChanged()));

	setSpacing(0);
	setStretchFactor(label, 0);
	setStretchFactor(multi_line_edit, 1);

	compiler_data = compiler_interface->GetCompilerData();
}

CTerminalCodeVBoxWidget::~CTerminalCodeVBoxWidget()
{
}

void	CTerminalCodeVBoxWidget::ClearMultiLineEditText()
{
	disconnect(	multi_line_edit,
				SIGNAL(textChanged()),
				this,
				SLOT(CodeTextChanged()));

	multi_line_edit->clear();

	connect(multi_line_edit,
			SIGNAL(textChanged()),
			this,
			SLOT(CodeTextChanged()));

	if (terminal_indicator)
	{
		terminal_indicator = false;

		ToggleTerminalIndicator();
	}
}

void	CTerminalCodeVBoxWidget::SetTabStopWidth(int count)
{
	QFontMetrics	fm(font());

	int		w = fm.width(QChar('Z'));

	multi_line_edit->setTabStopWidth(count * w);
}

void	CTerminalCodeVBoxWidget::SetMultiLineEditText(const list<string> &code)
{
	disconnect(	multi_line_edit,
				SIGNAL(textChanged()),
				this,
				SLOT(CodeTextChanged()));

	multi_line_edit->clear();

	list<string>::const_iterator	s_iterator = code.begin();

	while (s_iterator != code.end())
	{
		const string	&s = *s_iterator++;

		if (!s.length())
			multi_line_edit->append("\n");
		else
			multi_line_edit->append(s.c_str());
	}

	multi_line_edit->setCursorPosition(0, 0);

	connect(multi_line_edit,
			SIGNAL(textChanged()),
			this,
			SLOT(CodeTextChanged()));

	if (terminal_indicator)
	{
		terminal_indicator = false;

		ToggleTerminalIndicator();
	}
}

void	CTerminalCodeVBoxWidget::GetMultiLineEditText(list<string> &code)
{
	code.clear();

	int		count = multi_line_edit->numLines();
	int		i;

	for (i=0; i<count; i++)
		code.push_back(
			(const char *)(multi_line_edit->textLine(i).local8Bit()));

	if (code.back() == string(""))
		code.pop_back();
}

bool	CTerminalCodeVBoxWidget::GetCodeEditedFlag()
{
	return multi_line_edit->edited();
}

void	CTerminalCodeVBoxWidget::SetCodeEditedFlag(bool f)
{
	multi_line_edit->setEdited(f);
}

bool	CTerminalCodeVBoxWidget::GetTabIndicator()
{
	return tab_indicator;
}

void	CTerminalCodeVBoxWidget::ToggleTabIndicator()
{
	string	s1("terminal code");
	string	s2;

	if (tab_indicator)
	{
		tab_indicator = false;
	}
	else
	{
		s1 += " (tab entry)";

		tab_indicator = true;
	}

	if (terminal_indicator)
	{
		if (tsv)
			tsv->GetCurrentFunctionString(s2);

		s1 += ": ";
		s1 += s2;

		label->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
	}
	else
		label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	label->setText(s1.c_str());
}

void	CTerminalCodeVBoxWidget::ToggleTerminalIndicator()
{
	string	s1("terminal code");
	string	s2;

	if (tab_indicator)
		s1 += " (tab entry)";

	if (terminal_indicator)
	{
		label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

		terminal_indicator = false;
	}
	else
	{
		if (tsv)
			tsv->GetCurrentFunctionString(s2);

		s1 += ": ";
		s1 += s2;

		label->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);

		terminal_indicator = true;
	}

	label->setText(s1.c_str());
}

void	CTerminalCodeVBoxWidget::UndoEdit()
{
	multi_line_edit->undo();
}

void	CTerminalCodeVBoxWidget::RedoEdit()
{
	multi_line_edit->redo();
}

void	FindMultiLineEditString(QMultiLineEdit *multi_line_edit);

void	CTerminalCodeVBoxWidget::FindString()
{
	FindMultiLineEditString(multi_line_edit);
}

void	CTerminalCodeVBoxWidget::InitializeModulePointers()
{
	vector<CSubView *>		&v1 = module_ptr_map["TERMINAL_SELECT_VIEW"];

	if (v1.size())
		tsv = dynamic_cast<CTerminalSelectView *>(v1[0]);
	else
		tsv = 0;
}

void	CTerminalCodeVBoxWidget::RestoreUndoData()
{
	compiler_data = compiler_interface->GetCompilerData();
}

void	CTerminalCodeVBoxWidget::CodeTextChanged()
{
	if (tsv)
		tsv->CodeTextChanged();
}

CTerminalCodeView::CTerminalCodeView(	QWidget *p,
										CViewNode *n,
										CInterfaceControl *ic,
										SMainViewControl *mvc,
										const string &dfs,
										CPaletteData *dpd,
										COptionData::SSubViewData *d)
	: CSubView(p, n, ic, mvc, dfs, dpd, d)
{
	frame = new CTerminalCodeVBoxWidget(
		p, dynamic_cast<CCompilerInterface *>(ic), mvc, d, this);

	QFont		view_node_font;
	QPalette	view_node_palette;

	if (SetupViewNodeFont(view_node_font))
		frame->setFont(view_node_font);
	else
		frame->setFont(frame->font());

	if (SetupViewNodePalette(view_node_palette))
		frame->setPalette(view_node_palette);
	else
		frame->setPalette(frame->palette());
}

CTerminalCodeView::~CTerminalCodeView()
{
	delete frame;
}

void	CTerminalCodeView::SetHighlightPalette(CPaletteData *pd)
{
	restore_palette = frame->palette();

	QPalette	highlight_palette;

	pd->SetupPalette(highlight_palette);

	frame->setPalette(highlight_palette);
}

void	CTerminalCodeView::RestorePalette()
{
	frame->setPalette(restore_palette);
}

void	CTerminalCodeView::InitializeModulePointers()
{
	frame->InitializeModulePointers();
}

void	CTerminalCodeView::ClearMultiLineEditText()
{
	frame->ClearMultiLineEditText();
}

void	CTerminalCodeView::SetTabStopWidth(int count)
{
	frame->SetTabStopWidth(count);
}

void	CTerminalCodeView::SetMultiLineEditText(const list<string> &code)
{
	frame->SetMultiLineEditText(code);
}

void	CTerminalCodeView::GetMultiLineEditText(list<string> &code)
{
	frame->GetMultiLineEditText(code);
}

bool	CTerminalCodeView::GetCodeEditedFlag()
{
	return frame->GetCodeEditedFlag();
}

void	CTerminalCodeView::SetCodeEditedFlag(bool f)
{
	frame->SetCodeEditedFlag(f);
}

void	CTerminalCodeView::UndoEdit()
{
	frame->UndoEdit();
}

void	CTerminalCodeView::RedoEdit()
{
	frame->RedoEdit();
}

void	CTerminalCodeView::FindString()
{
	frame->FindString();
}

void	CTerminalCodeView::RestoreUndoData()
{
	frame->RestoreUndoData();
}
