/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "terminalView.h"

CStartContextWidget::CStartContextWidget(CTerminalVBoxWidget *v)
	: CFrameFc((QWidget *)v, 1, 2), terminal_vbox(v)
{
	grid_layout->setRowStretch(0, 1);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 1);

	start_context_button	= new CPushButtonFc("start context", this);
	start_context_line_edit	= new CLineEditFc(this, "Start Context");

	start_context_line_edit->setReadOnly(true);

	grid_layout->addWidget(start_context_button,			0, 0);
	grid_layout->addWidget(start_context_line_edit,			0, 1);

	connect(start_context_button,
			SIGNAL(clicked()),
			SLOT(ModifyStartContext()));
}

void	CStartContextWidget::ModifyStartContext()
{
	terminal_vbox->ModifyStartContext();
}

void	CTerminalListView::InitializePalette(const QPalette &p)
{
	active_highlight			= p.color(	QPalette::Active,
											QColorGroup::Highlight);
	active_highlighted_text		= p.color(	QPalette::Active,
											QColorGroup::HighlightedText);
	inactive_highlight			= p.color(	QPalette::Inactive,
											QColorGroup::Highlight);
	inactive_highlighted_text	= p.color(	QPalette::Inactive,
											QColorGroup::HighlightedText);

	QPalette	focus_palette = p;

	focus_palette.setColor(	QPalette::Active,
							QColorGroup::Highlight,
							inactive_highlight);

	focus_palette.setColor(	QPalette::Active,
							QColorGroup::HighlightedText,
							inactive_highlighted_text);

	setPalette(focus_palette);
}

void	CTerminalListView::focusInEvent(QFocusEvent *e)
{
	QPalette	p = palette();

	p.setColor(	QPalette::Active,
				QColorGroup::Highlight,
				active_highlight);

	p.setColor(	QPalette::Active,
				QColorGroup::HighlightedText,
				active_highlighted_text);

	setPalette(p);

	QListView::focusInEvent(e);
}

void	CTerminalListView::focusOutEvent(QFocusEvent *e)
{
	QPalette	p = palette();

	p.setColor(	QPalette::Active,
				QColorGroup::Highlight,
				inactive_highlight);

	p.setColor(	QPalette::Active,
				QColorGroup::HighlightedText,
				inactive_highlighted_text);

	setPalette(p);

	QListView::focusOutEvent(e);
}

void	CTerminalListView::mouseDoubleClickEvent(QMouseEvent *e)
{
	terminal_vbox->EditTerminal();

	e->accept();
}

void	CTerminalListView::keyPressEvent(QKeyEvent *e)
{
	switch (e->key())
	{
		case Qt::Key_Insert:
		switch (e->state())
		{
			case Qt::NoButton:
			case Qt::Keypad:
			e->accept();
			terminal_vbox->InsertAfterTerminal();
			return;

			case Qt::ShiftButton:
			e->accept();
			terminal_vbox->InsertBeforeTerminal();
			return;
		}
		break; 

		case Qt::Key_0:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			terminal_vbox->InsertBeforeTerminal();
			return;
		}
		break; 

		case Qt::Key_Delete:
		if (e->state() == Qt::ShiftButton)
		{
			e->accept();
			terminal_vbox->DeleteTerminal();
			return;
		}
		break; 

		case Qt::Key_Period:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			terminal_vbox->DeleteTerminal();
			return;
		}
		break; 

		case Qt::Key_Return:
		if (e->state() == Qt::NoButton)
		{
			e->accept();
			terminal_vbox->EditTerminal();
			return;
		}
		break; 

		case Qt::Key_Enter:
		if (e->state() == Qt::Keypad)
		{
			e->accept();
			terminal_vbox->EditTerminal();
			return;
		}
		break; 

		case Qt::Key_Up:
		if (e->state() == Qt::ShiftButton)
		{
			e->accept();
			terminal_vbox->MoveTerminalUp();
			return;
		}
		break;

		case Qt::Key_8:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			terminal_vbox->MoveTerminalUp();
			return;
		}
		break;

		case Qt::Key_Down:
		if (e->state() == Qt::ShiftButton)
		{
			e->accept();
			terminal_vbox->MoveTerminalDown();
			return;
		}
		break;

		case Qt::Key_2:
		if (e->state() == Qt::ShiftButton | Qt::Keypad)
		{
			e->accept();
			terminal_vbox->MoveTerminalDown();
			return;
		}
		break;
	}

	e->ignore();
	QListView::keyPressEvent(e);
}

CTerminalVBoxWidget::CTerminalVBoxWidget(	QWidget *parent,
											CCompilerInterface *ci,
											SMainViewControl *mvc,
											COptionData::SSubViewData *d,
											CTerminalView *tv)
	: CVBoxWidget(parent, ci, mvc), compiler_interface(ci),
		subview_data(d), terminal_view(tv),
		module_ptr_map(tv->GetModulePtrMap())
{
	compiler_data = compiler_interface->GetCompilerData();

	label = new CLabelFc("terminal", this, true);

	label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	start_context_widget = new CStartContextWidget(this);

	CLineEditFc	*line_edit = start_context_widget->GetStartContextLineEdit();

	line_edit->setText(compiler_data->GetStartContext().c_str());

	list_view = new CTerminalListView(this);

	list_view->header()->setMovingEnabled(false);
	list_view->setSorting(-1);
	list_view->addColumn("index");
	list_view->addColumn("symbol");
	list_view->addColumn("type");
	list_view->addColumn("driver");
	list_view->addColumn("name");
	list_view->addColumn("regexp");

	setSpacing(0);
	setStretchFactor(label, 0);
	setStretchFactor(start_context_widget, 0);
	setStretchFactor(list_view, 1);

	UpdateListView();

	list_view->setSelected(list_view->firstChild(), true);
}

CTerminalVBoxWidget::~CTerminalVBoxWidget()
{
}

void	CTerminalVBoxWidget::ModifyStartContext()
{
	CValueDialog	*d = OptionDialog::value_dialog;

	QFont		d_font		= d->font();
	QPalette	d_palette	= d->palette();

	d->setFont(font());
	d->setPalette(palette());

	string		context_string;
	CLineEditFc	*line_edit = start_context_widget->GetStartContextLineEdit();

	d->SetTitleAndLabelText("Modify Start Context", "Start Context");
	d->SetValuePointer(&context_string);
	d->SetValueLineEdit(line_edit->text());

	int		code = d->exec();

	d->setFont(d_font);
	d->setPalette(d_palette);

	if (code == QDialog::Rejected)
		return;

	line_edit->setText(context_string.c_str());

	compiler_interface->StoreUndoData();

	compiler_data->SetStartContext(context_string);

	compiler_interface->SetDataModifiedFlag(true);
}

void	CTerminalVBoxWidget::UpdateListView()
{
	const list<CTerminal *>		&t_list = compiler_data->GetTerminalList();

	int		t_count = t_list.size();
	int		l_count = list_view->childCount();
	int		i;

	if (t_count < l_count)
	{
		for (i=0; i<l_count-t_count; i++)
			delete list_view->firstChild();
	}
	else if (t_count > l_count)
	{
		for (i=0; i<t_count-l_count; i++)
			new QListViewItem(list_view);
	}

	list<CTerminal *>::const_iterator	t_iterator	= t_list.begin();

	QListViewItemIterator	l_iterator(list_view);

	while (t_iterator != t_list.end())
	{
		const CTerminal		*t = *t_iterator;

		QListViewItem	*item = l_iterator.current();

		string		s_index;
		string		s_symbol_index;
		string		s_filter_flag;
		string		s_literal_flag;

		t->GetIndexString(s_index);
		t->GetSymbolIndexString(s_symbol_index);
		t->GetFilterFlagString(s_filter_flag);
		t->GetLiteralFlagString(s_literal_flag);

		item->setText(0, s_index.c_str());
		item->setText(1, s_symbol_index.c_str());
		item->setText(2, s_filter_flag.c_str());
		item->setText(3, s_literal_flag.c_str());
		item->setText(4, t->GetName().c_str());
		item->setText(5, t->GetRegexp().c_str());

		t_iterator++;
		l_iterator++;
	}

	vector<CSubView *>		&v1 = module_ptr_map["ERROR_TOKEN_VIEW"];
	vector<CSubView *>		&v2 = module_ptr_map["NONTERMINAL_VIEW"];
	vector<CSubView *>		&v3 = module_ptr_map["IMPLICIT_VIEW"];
	vector<CSubView *>		&v4 = module_ptr_map["PRECEDENCE_VIEW"];
	vector<CSubView *>		&v5 = module_ptr_map["RULE_VIEW"];
	vector<CSubView *>		&v6 = module_ptr_map["TERMINAL_SELECT_VIEW"];

	if (v1.size())
	{
		CErrorTokenView		*ev = dynamic_cast<CErrorTokenView *>(v1[0]);

		ev->UpdateListView();
	}

	if (v2.size())
	{
		CNonTerminalView	*nv = dynamic_cast<CNonTerminalView *>(v2[0]);

		nv->UpdateListView();
	}

	if (v3.size())
	{
		CImplicitView		*iv = dynamic_cast<CImplicitView *>(v3[0]);

		iv->UpdateListView();
	}

	if (v4.size())
	{
		CPrecedenceView		*pv = dynamic_cast<CPrecedenceView *>(v4[0]);

		pv->UpdateListView();
	}

	if (v5.size())
	{
		CRuleView			*rv = dynamic_cast<CRuleView *>(v5[0]);

		rv->UpdateListView();
	}

	if (v6.size())
	{
		CTerminalSelectView	*tv = dynamic_cast<CTerminalSelectView *>(v6[0]);

		tv->UpdateListView();
	}
}

void	CTerminalVBoxWidget::SetSelectedItem(int i)
{
	int		l_count = list_view->childCount();

	if (i < 0 || !l_count)
		return;

	if (i >= l_count)
		i = l_count - 1;

	QListViewItemIterator	l_iterator(list_view);
	int						count;

	for (count=0; count<i; count++)
		l_iterator++;

	list_view->setSelected(l_iterator.current(), true);
	list_view->ensureItemVisible(l_iterator.current());
}

void	CTerminalVBoxWidget::InitializeModulePointers()
{
}

void	CTerminalVBoxWidget::InsertBeforeTerminal()
{
	GrammarDialog::SetTerminalPaletteAndFont(this);

	CTerminal			terminal(0);
	CTerminalDialog		*td = GrammarDialog::terminal_dialog;

	td->SetTitleAndResize("Insert Before Terminal");
	td->SetTerminalPointer(&terminal);
	td->SetControlValues(terminal);

	if (td->exec() == QDialog::Accepted)
	{
		QListViewItem	*current = list_view->currentItem();
		int				selected = 0;

		if (current)
			selected = current->text(0).toInt();

		compiler_interface->StoreUndoData();

		InsertBeforeTerminal(selected, terminal);

		SetSelectedItem(selected);

		compiler_interface->SetDataModifiedFlag(true);
	}
}

void	CTerminalVBoxWidget::InsertAfterTerminal()
{
	GrammarDialog::SetTerminalPaletteAndFont(this);

	CTerminal			terminal(0);
	CTerminalDialog		*td = GrammarDialog::terminal_dialog;

	td->SetTitleAndResize("Insert After Terminal");
	td->SetTerminalPointer(&terminal);
	td->SetControlValues(terminal);

	if (td->exec() == QDialog::Accepted)
	{
		QListViewItem	*current = list_view->currentItem();
		int				selected = 0;

		if (current)
			selected = current->text(0).toInt();

		compiler_interface->StoreUndoData();

		InsertAfterTerminal(selected, terminal);

		SetSelectedItem(selected + 1);

		compiler_interface->SetDataModifiedFlag(true);
	}
}

void	CTerminalVBoxWidget::EditTerminal()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	selected	= current->text(0).toInt();

		GrammarDialog::SetTerminalPaletteAndFont(this);

		CTerminal			*ti = compiler_data->FindTerminalIndex(selected);
		CTerminal			terminal(0);
		CTerminalDialog		*td = GrammarDialog::terminal_dialog;

		td->SetTitleAndResize("Edit Terminal");
		td->SetTerminalPointer(&terminal);

		if (ti)
			td->SetControlValues(*ti);
		else
			td->SetControlValues(terminal);

		if (td->exec() == QDialog::Accepted)
		{
			compiler_interface->StoreUndoData();

			UpdateTerminal(selected, terminal);

			SetSelectedItem(selected);

			compiler_interface->SetDataModifiedFlag(true);
		}
	}
}

void	CTerminalVBoxWidget::DeleteTerminal()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	selected	= current->text(0).toInt();

		compiler_interface->StoreUndoData();

		DeleteTerminal(selected);

		SetSelectedItem(selected);

		compiler_interface->SetDataModifiedFlag(true);
	}
}

void	CTerminalVBoxWidget::MoveTerminalUp()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	selected	= current->text(0).toInt();

		if (selected > 0)
		{
			compiler_interface->StoreUndoData();

			MoveTerminalUp(selected);

			SetSelectedItem(selected - 1);

			compiler_interface->SetDataModifiedFlag(true);
		}
	}
}

void	CTerminalVBoxWidget::MoveTerminalDown()
{
	QListViewItem	*current = list_view->currentItem();

	if (current)
	{
		int	l_count		= list_view->childCount();
		int	selected	= current->text(0).toInt();

		if (selected < l_count - 1)
		{
			compiler_interface->StoreUndoData();

			MoveTerminalDown(selected);

			SetSelectedItem(selected + 1);

			compiler_interface->SetDataModifiedFlag(true);
		}
	}
}

void	CTerminalVBoxWidget::InsertBeforeTerminal(int i, const CTerminal &t)
{
	compiler_data->InsertBeforeTerminal(i, t);

	UpdateListView();
}

void	CTerminalVBoxWidget::InsertAfterTerminal(int i, const CTerminal &t)
{
	compiler_data->InsertAfterTerminal(i, t);

	UpdateListView();
}

void	CTerminalVBoxWidget::UpdateTerminal(int i, const CTerminal &t)
{
	compiler_data->UpdateTerminal(i, t);

	UpdateListView();
}

void	CTerminalVBoxWidget::DeleteTerminal(int i)
{
	compiler_data->DeleteTerminal(i);

	UpdateListView();
}

void	CTerminalVBoxWidget::MoveTerminalUp(int i)
{
	compiler_data->MoveTerminalUp(i);

	UpdateListView();
}

void	CTerminalVBoxWidget::MoveTerminalDown(int i)
{
	compiler_data->MoveTerminalDown(i);

	UpdateListView();
}

void	CTerminalVBoxWidget::RestoreUndoData()
{
	compiler_data = compiler_interface->GetCompilerData();
}

CTerminalView::CTerminalView(	QWidget *p,
								CViewNode *n,
								CInterfaceControl *ic,
								SMainViewControl *mvc,
								const string &dfs,
								CPaletteData *dpd,
								COptionData::SSubViewData *d)
	: CSubView(p, n, ic, mvc, dfs, dpd, d)
{
	frame = new CTerminalVBoxWidget(
		p, dynamic_cast<CCompilerInterface *>(ic), mvc, d, this);

	QFont		view_node_font;
	QPalette	view_node_palette;

	if (SetupViewNodeFont(view_node_font))
		frame->setFont(view_node_font);
	else
		frame->setFont(frame->font());

	if (SetupViewNodePalette(view_node_palette))
	{
		frame->setPalette(view_node_palette);
		frame->InitializePalette(view_node_palette);
	}
	else
	{
		frame->setPalette(frame->palette());
		frame->InitializePalette(frame->palette());
	}
}

CTerminalView::~CTerminalView()
{
	delete frame;
}

void	CTerminalView::SetHighlightPalette(CPaletteData *pd)
{
	restore_palette = frame->palette();

	QPalette	highlight_palette;

	pd->SetupPalette(highlight_palette);

	frame->setPalette(highlight_palette);
}

void	CTerminalView::RestorePalette()
{
	frame->setPalette(restore_palette);
}

void	CTerminalView::InitializeModulePointers()
{
	frame->InitializeModulePointers();
}

void	CTerminalView::ModifyStartContext()
{
	frame->ModifyStartContext();
}

void	CTerminalView::InsertAfterTerminal()
{
	frame->InsertAfterTerminal();
}

void	CTerminalView::InsertBeforeTerminal()
{
	frame->InsertBeforeTerminal();
}

void	CTerminalView::EditTerminal()
{
	frame->EditTerminal();
}

void	CTerminalView::DeleteTerminal()
{
	frame->DeleteTerminal();
}

void	CTerminalView::MoveTerminalUp()
{
	frame->MoveTerminalUp();
}

void	CTerminalView::MoveTerminalDown()
{
	frame->MoveTerminalDown();
}

void	CTerminalView::UpdateListView()
{
	frame->UpdateListView();
}

void	CTerminalView::RestoreUndoData()
{
	frame->RestoreUndoData();
}
