/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#ifndef COMPILERDATA_H
#define COMPILERDATA_H

#include <string>
#include <vector>
#include <list>
#include <map>
#include <set>
#include <algorithm>
#include <iostream>
#include <strstream>
#include <iomanip>

#include "stringToken.h"

class CSymbol
{
	public:
	CSymbol(int i, const string &n = "") : index(i), name(n) { }

	int				GetIndex() const			{ return index; }
	void			SetIndex(int i)				{ index = i; }
	int				GetSymbolIndex() const		{ return symbol_index; }
	void			SetSymbolIndex(int i)		{ symbol_index = i; }
	const string	&GetName() const			{ return name; }
	void			SetName(const string &n)	{ name = n; }
	bool			IsUndefined() const 		{ return index < 0; }
	virtual bool	IsSymbolATerminal() const	{ return false; }

	void	GetIndexString(string &s) const
	{
		ostrstream		os;

		os << index << '\0';

		s = os.str();
	}

	void	GetSymbolIndexString(string &s) const
	{
		ostrstream		os;

		if (symbol_index >= 0)
			os << symbol_index << '\0';
		else
			os << "*" << '\0';

		s = os.str();
	}

	protected:
	int			index;
	int			symbol_index;
	string		name;
};

class CTerminal : public CSymbol
{
	public:
	CTerminal(const string &s);
	CTerminal(int i, const string &n = "",
				bool ff = false, bool lf = false, const string &r = "")
		: CSymbol(i, n), filter_flag(ff), literal_flag(lf),
			regular_expression(r), code_flag(false) { }

	bool			GetFilterFlag() const		{ return filter_flag; }
	void			SetFilterFlag(bool f)		{ filter_flag = f; }
	bool			GetLiteralFlag() const		{ return literal_flag; }
	void			SetLiteralFlag(bool f)		{ literal_flag = f; }
	const string	&GetRegexp() const			{ return regular_expression; }
	void			SetRegexp(const string &r)	{ regular_expression = r; }
	bool			GetCodeFlag() const			{ return code_flag; }
	void			SetCodeFlag(bool f)			{ code_flag = f; }
	list<string>	&GetCodeText()				{ return code_text; }
	const list<string>	&GetCodeTextConst() const	{ return code_text; }
	virtual bool	IsSymbolATerminal() const	{ return true; }

	void			UpdateTerminal(const CTerminal &t)
	{
		SetIndex(t.GetIndex());
		SetName(t.GetName());

		filter_flag			= t.filter_flag;
		literal_flag		= t.literal_flag;
		regular_expression	= t.regular_expression;
	}

	void	GetFilterFlagString(string &s) const
	{
		if (filter_flag)
			s = "FILTER";
		else
			s = "TOKEN";
	}

	void	GetLiteralFlagString(string &s) const
	{
		if (literal_flag)
			s = "LITERAL";
		else
			s = "REGEXP";
	}

	private:
	bool			filter_flag;
	bool			literal_flag;
	string			regular_expression;
	bool			code_flag;
	list<string>	code_text;
};

class CErrorToken : public CSymbol
{
	public:
	CErrorToken(const string &s);
	CErrorToken(int i, const string &n = "", int sl = 0)
		: CSymbol(i, n), synchronization_length(sl) { }

	int		GetSynchronizationLength() const { return synchronization_length; }
	void	SetSynchronizationLength(int sl) { synchronization_length = sl; }

	virtual bool	IsSymbolATerminal() const	{ return true; }

	void	GetSynchronizationLengthString(string &s) const
	{
		ostrstream		os;

		os << synchronization_length << '\0';

		s = os.str();
	}

	private:
	int		synchronization_length;
};

class CNonTerminal : public CSymbol
{
	public:
	CNonTerminal(const string &s);
	CNonTerminal(int i, const string &n = "", bool sf = false)
		: CSymbol(i, n), start_symbol_flag(sf) { }

	bool	GetStartSymbolFlag() const { return start_symbol_flag; }
	void	SetStartSymbolFlag(int sf) { start_symbol_flag = sf; }

	virtual bool	IsSymbolATerminal() const	{ return false; }

	private:
	bool	start_symbol_flag;
};

class CCompilerData;

class CRule
{
	public:
	CRule(int i)
		: index(i), unit_rule_elimination_flag(false), left(0),
			undefined_symbol_flag(false), code_flag(false) { }
	CRule(const string &s, CCompilerData *d);

	int								GetIndex() const	{ return index; }
	void							SetIndex(int i)		{ index = i; }
	bool							IsUndefined() const { return index < 0; }

	bool	GetUnitRuleEliminationFlag() const
	{ return unit_rule_elimination_flag; }
	void	SetUnitRuleEliminationFlag(bool f)
	{ unit_rule_elimination_flag = f; }

	const CSymbol					*GetLeft() const	{ return left; }
	const vector<const CSymbol *>	&GetRight() const	{ return right; }
	void		SetLeft(CSymbol *l)						{ left = l; }
	void		SetRight(vector<const CSymbol *> &r)	{ right = r; }

	bool	GetUndefinedSymbolFlag() const { return undefined_symbol_flag; }
	void	SetUndefinedSymbolFlag(bool f) { undefined_symbol_flag = f; }

	bool				GetCodeFlag() const			{ return code_flag; }
	void				SetCodeFlag(bool f)			{ code_flag = f; }
	list<string>		&GetCodeText()				{ return code_text; }
	const list<string>	&GetCodeTextConst() const	{ return code_text; }

	bool	ReplaceSymbol(const CSymbol *current, const CSymbol *new_symbol);
	string	&GetLeftString(string &s) const;
	string	&GetRightString(string &s) const;
	string	&GetRightLessOne(string &s);
	string	&GetStringPadded(string &s, int rule_left_name_width) const;
	string	&GetStringIndexed(string &s) const;

	void	UpdateRule(const CRule &r)
	{
		index						= r.index;
		unit_rule_elimination_flag	= r.unit_rule_elimination_flag;
		left						= r.left;
		right						= r.right;
		undefined_symbol_flag		= r.undefined_symbol_flag;
	}

	void	GetIndexString(string &s) const
	{
		ostrstream		os;

		os << index << '\0';

		s = os.str();
	}

	private:
	int						index;
	bool					unit_rule_elimination_flag;
	const CSymbol			*left;
	vector<const CSymbol *>	right;
	bool					undefined_symbol_flag;
	bool					code_flag;
	list<string>			code_text;
};

struct SPrecedenceEntry
{
	enum EntryType
	{
		TerminalEntry,
		ErrorTokenEntry,
		RuleEntry,
		UndefinedEntry
	};

	SPrecedenceEntry(CTerminal *t)
	{ type = TerminalEntry; terminal = t; }
	SPrecedenceEntry(CErrorToken *e)
	{ type = ErrorTokenEntry; error_token = e; }
	SPrecedenceEntry(CRule *r)
	{ type = RuleEntry; rule = r; }
	SPrecedenceEntry()
	{ type = UndefinedEntry; }

	EntryType			type;
	const CTerminal		*terminal;
	const CErrorToken	*error_token;
	const CRule			*rule;
};

class CPrecedence
{
	public:
	enum Associativity
	{
		LeftAssociative,
		RightAssociative,
		NonAssociative,
		NullAssociative
	};

	CPrecedence(int i)
		: index(i), associativity(NullAssociative),
			undefined_entry_flag(false) {}
	CPrecedence(const string &s, CCompilerData *d);

	int		GetIndex() const		{ return index; }
	void	SetIndex(int i)			{ index = i; }
	bool	IsUndefined() const 	{ return index < 0; }

	const Associativity				GetAssociativity() const
	{ return associativity; }
	const vector<SPrecedenceEntry>	&GetEntries() const
	{ return entries; }

	bool	GetUndefinedEntryFlag() const	{ return undefined_entry_flag; }
	void	SetUndefinedEntryFlag(bool f)	{ undefined_entry_flag = f; }
	bool	ReplaceTerminal(const CTerminal *current,
							const CTerminal *new_terminal);
	bool	ReplaceErrorToken(	const CErrorToken *current,
								const CErrorToken *new_error_token);
	bool	ReplaceRule(const CRule *current,
						const CRule *new_rule);
	string	&GetAssociativityString(string &s) const;
	string	&GetEntryString(string &s) const;
	string	&GetEntryLessOne(string &s);

	void	GetIndexString(string &s) const
	{
		ostrstream		os;

		os << index << '\0';

		s = os.str();
	}

	private:
	int							index;
	Associativity				associativity;
	vector<SPrecedenceEntry>	entries;
	bool						undefined_entry_flag;
};

struct SConflictResolution
{
	SConflictResolution() : index(-1) { }
	SConflictResolution(const string &s);
	SConflictResolution(int i, int st, int sy, int a)
		: index(i), state(st), symbol(sy), action(a) { }

	int		GetIndex() const	{ return index; }
	bool	IsUndefined() const { return index < 0; }

	int				index;
	int				state;
	int				symbol;
	int				action;
	list<string>	code_text;
};

struct SErrorCode
{
	SErrorCode() : entry(0) { }
	SErrorCode(const string &s);
	SErrorCode(int e, int st, int sy, int t = -1)
		: entry(e), state(st), symbol(sy), token(t) { }

	int		GetEntry() const	{ return entry; }
	bool	IsUndefined() const { return !entry; }

	int				entry;
	int				state;
	int				symbol;
	int				token;
	list<string>	code_text;
};

namespace OutputWidths
{
	extern int		terminal_index_width;
	extern int		terminal_name_width;
	extern int		error_token_index_width;
	extern int		error_token_name_width;
	extern int		nonterminal_index_width;
	extern int		nonterminal_name_width;
	extern int		symbol_index_width;
	extern int		symbol_name_width;
	extern int		rule_index_width;
	extern int		rule_left_name_width;
	extern int		precedence_index_width;
	extern int		conflict_index_width;
	extern int		error_entry_width;
};

class CCompilerData
{
	public:
	enum FiniteStateMachineType
	{
		RegularExpression,
		SingleCharacter
	};

	enum ParserType
	{
		LR_1,
		LALR_1,
		SLR_1
	};

	enum ErrorCalculation
	{
		NoCalculation,
		ApproximateEssential,
		ExactEssential
	};

	enum ErrorFunctionNumbering
	{
		OneNumber,
		NumberPerState,
		NumberPerEntry
	};

	enum ErrorFunctionAggregation
	{
		OneFunction,
		FunctionPerState,
		FunctionPerEntry
	};

	CCompilerData()
		:	end_of_file_symbol(CTerminal(0, "EOF", false, true)),
			augmented_start_symbol(CNonTerminal(0, "ASTART")),
			augmented_start_rule(CRule(0)),
			start_symbol(0),
			lexical_automaton_calculate(false),
			lexical_prefix_name("default"),
			lexical_finite_state_machine_type(RegularExpression),
			lexical_input_range(256),
			parser_automaton_calculate(false),
			parser_prefix_name("default"),
			parser_type(LALR_1),
			base_prefix_name("default"),
			generate_base_classes(true),
			parser_error_calculation(NoCalculation),
			parser_error_function_numbering(OneNumber),
			parser_error_function_aggregation(OneFunction),
			parser_optimize_unit_rule_reduction(false),
			generate_tty_code(false),
			regexp_match_table(false)
			{
				index_upper_bound = 1000;

				SetupSymbolIndexes();
			}

	~CCompilerData();

	static int		index_upper_bound;

	int		terminal_index_width;
	int		terminal_name_width;
	int		error_token_index_width;
	int		error_token_name_width;
	int		nonterminal_index_width;
	int		nonterminal_name_width;
	int		symbol_index_width;
	int		symbol_name_width;
	int		rule_index_width;
	int		rule_left_name_width;
	int		precedence_index_width;
	int		conflict_index_width;
	int		error_entry_width;

	int		GetIndexUpperBound() const		{ return index_upper_bound; }
	int		GetTerminalIndexWidth() const	{ return terminal_index_width; }
	int		GetTerminalNameWidth() const	{ return terminal_name_width; }
	int		GetErrorTokenIndexWidth() const	{ return error_token_index_width; }
	int		GetErrorTokenNameWidth() const	{ return error_token_name_width; }
	int		GetNonterminalIndexWidth() const{ return nonterminal_index_width; }
	int		GetNonterminalNameWidth() const	{ return nonterminal_name_width; }
	int		GetSymbolIndexWidth() const		{ return symbol_index_width; }
	int		GetSymbolNameWidth() const		{ return symbol_name_width; }
	int		GetRuleIndexWidth() const		{ return rule_index_width; }
	int		GetRuleLeftNameWidth() const	{ return rule_left_name_width; }
	int		GetPrecedenceIndexWidth() const	{ return precedence_index_width; }
	int		GetConflictIndexWidth() const	{ return conflict_index_width; }
	int		GetErrorEntryWidth() const		{ return error_entry_width; }

	void	CalculateOutputWidths();

	bool						GetLexicalAutomatonCalculate() const
	{ return lexical_automaton_calculate; }
	const string				&GetLexicalPrefixName() const
	{ return lexical_prefix_name; }
	FiniteStateMachineType		GetLexicalFiniteStateMachineType() const
	{ return lexical_finite_state_machine_type; }
	int							GetLexicalInputRange() const
	{ return lexical_input_range; }
	bool						GetParserAutomatonCalculate() const
	{ return parser_automaton_calculate; }
	const string				&GetParserPrefixName() const
	{ return parser_prefix_name; }
	ParserType					GetParserType() const
	{ return parser_type; }
	const string				&GetBasePrefixName() const
	{ return base_prefix_name; }
	bool						GetGenerateBaseClasses() const
	{ return generate_base_classes; }
	ErrorCalculation			GetParserErrorCalculation() const
	{ return parser_error_calculation; }
	ErrorFunctionNumbering		GetParserErrorFunctionNumbering() const
	{ return parser_error_function_numbering; }
	ErrorFunctionAggregation	GetParserErrorFunctionAggregation() const
	{ return parser_error_function_aggregation; }
	bool						GetParserOptimizeUnitRuleReduction() const
	{ return parser_optimize_unit_rule_reduction; }
	bool						GetGenerateTTYCode() const
	{ return generate_tty_code; }
	bool						GetRegexpMatchTable() const
	{ return regexp_match_table; }

	void	SetLexicalAutomatonCalculate(bool c)
	{ lexical_automaton_calculate = c; }
	void	SetLexicalPrefixName(const string &n)
	{ lexical_prefix_name = n; }
	void	SetLexicalFiniteStateMachineType(FiniteStateMachineType t)
	{ lexical_finite_state_machine_type = t; }
	void	SetLexicalInputRange(int r)
	{ lexical_input_range = r; }
	void	SetParserAutomatonCalculate(bool c)
	{ parser_automaton_calculate = c; }
	void	SetParserPrefixName(const string &n)
	{ parser_prefix_name = n; }
	void	SetParserType(ParserType t)
	{ parser_type = t; }
	void	SetBasePrefixName(const string &n)
	{ base_prefix_name = n; }
	void	SetGenerateBaseClasses(bool g)
	{ generate_base_classes = g; }
	void	SetParserErrorCalculation(ErrorCalculation c)
	{ parser_error_calculation = c; }
	void	SetParserErrorFunctionNumbering(ErrorFunctionNumbering n)
	{ parser_error_function_numbering = n; }
	void	SetParserErrorFunctionAggregation(ErrorFunctionAggregation a)
	{ parser_error_function_aggregation = a; }
	void	SetParserOptimizeUnitRuleReduction(bool r)
	{ parser_optimize_unit_rule_reduction = r; }
	void	SetGenerateTTYCode(bool f)
	{ generate_tty_code = f; }
	void	SetRegexpMatchTable(bool f)
	{ regexp_match_table = f; }

	const string							&GetStartContext() const
	{ return start_context; }
	const list<CTerminal *>					&GetTerminalList() const
	{ return terminal_list; }
	const list<CErrorToken *>				&GetErrorTokenList() const
	{ return error_token_list; }
	const list<CNonTerminal *>				&GetNonTerminalList() const
	{ return nonterminal_list; }
	const list<CRule *>						&GetRuleList() const
	{ return rule_list; }
	const list<CPrecedence *>				&GetPrecedenceList() const
	{ return precedence_list; }
	const list<string>						&GetLexicalCode() const
	{ return lexical_code; }
	const list<string>						&GetParserCode() const
	{ return parser_code; }
	const list<string>						&GetStackCode() const
	{ return stack_code; }
	const list<string>						&GetCommentList() const
	{ return comment_list; }
	const map<int, SConflictResolution *>	&GetConflictResolutionMap() const
	{ return conflict_resolution_map; }
	const map<int, SErrorCode *>			&GetErrorCodeMap() const
	{ return error_code_map; }
	const list<int>							&GetUnitRuleOrderList() const
	{ return unit_rule_order_list; }

	const CTerminal							&GetEndOfFileSymbol() const
	{ return end_of_file_symbol; }
	const CNonTerminal						&GetAugmentedStartSymbol() const
	{ return augmented_start_symbol; }
	const CRule								&GetAugmentedStartRule() const
	{ return augmented_start_rule; }
	const CNonTerminal						*GetStartSymbol() const
	{ return start_symbol; }

	void	SetStartContext(const string &s)
	{ start_context = s; }

	void	AppendTerminal(const CTerminal &t);
	void	InsertBeforeTerminal(int i, const CTerminal &t);
	void	InsertAfterTerminal(int i, const CTerminal &t);
	void	UpdateTerminal(int i, const CTerminal &t);
	void	DeleteTerminal(int i);
	void	MoveTerminalUp(int i);
	void	MoveTerminalDown(int i);

	void	AppendErrorToken(const CErrorToken &e);
	void	InsertBeforeErrorToken(int i, const CErrorToken &e);
	void	InsertAfterErrorToken(int i, const CErrorToken &e);
	void	UpdateErrorToken(int i, const CErrorToken &e);
	void	DeleteErrorToken(int i);
	void	MoveErrorTokenUp(int i);
	void	MoveErrorTokenDown(int i);

	void	AppendNonTerminal(const CNonTerminal &n);
	void	InsertBeforeNonTerminal(int i, const CNonTerminal &n);
	void	InsertAfterNonTerminal(int i, const CNonTerminal &n);
	void	UpdateNonTerminal(int i, const CNonTerminal &n);
	void	DeleteNonTerminal(int i);
	void	MoveNonTerminalUp(int i);
	void	MoveNonTerminalDown(int i);

	void	AppendRule(const CRule &r);
	void	InsertBeforeRule(int i, const CRule &r);
	void	InsertAfterRule(int i, const CRule &r);
	void	UpdateRule(int i, const CRule &r);
	void	DeleteRule(int i);
	void	MoveRuleUp(int i);
	void	MoveRuleDown(int i);

	void	AppendPrecedence(const CPrecedence &p);
	void	InsertBeforePrecedence(int i, const CPrecedence &p);
	void	InsertAfterPrecedence(int i, const CPrecedence &p);
	void	UpdatePrecedence(int i, const CPrecedence &p);
	void	DeletePrecedence(int i);
	void	MovePrecedenceUp(int i);
	void	MovePrecedenceDown(int i);

	void	UpdateLexicalCode(const char *c);
	void	AddLexicalCode(const string &s);
	void	ClearLexicalCode();
	void	GetLexicalCodeSections(	list<string> &include_code,
									list<string> &class_code,
									list<string> &source_code,
									list<string> &constructor_code,
									list<string> &destructor_code);
	void	StoreLexicalCodeSections(	const list<string> &include_code,
										const list<string> &class_code,
										const list<string> &source_code,
										const list<string> &constructor_code,
										const list<string> &destructor_code);

	void	UpdateParserCode(const char *c);
	void	AddParserCode(const string &s);
	void	ClearParserCode();
	void	GetParserCodeSections(	list<string> &include_code,
									list<string> &class_code,
									list<string> &source_code,
									list<string> &constructor_code,
									list<string> &destructor_code);
	void	StoreParserCodeSections(const list<string> &include_code,
									const list<string> &class_code,
									const list<string> &source_code,
									const list<string> &constructor_code,
									const list<string> &destructor_code);

	void	AddStackCode(const string &s);
	void	StoreStackCode(const list<string> &code);
	void	ClearStackCode();
	void	InitializeStackCode();

	void	AddCommentString(const string &s);

	void	AddConflictResolution(const SConflictResolution &c);
	void	DeleteConflictResolution(int i);
	void	ClearConflictResolution();

	void	AddErrorCode(const SErrorCode &e);
	void	DeleteErrorCode(int i);
	void	ClearErrorCode();

	void	InitializeUnitRuleOrderList();
	bool	ValidateUnitRuleOrderList();

	void	AddUnitRuleOrderIndex(int i)
	{ unit_rule_order_list.push_back(i); }
	void	SetUnitRuleOrderList(const list<int> &u)
	{ unit_rule_order_list = u; }
	void	ClearUnitRuleOrderList()
	{ unit_rule_order_list.clear(); }

	void	ReplaceUpdatedTerminal(CTerminal *t);
	void	ReplaceDeletedTerminal(CTerminal *t);
	void	ReplaceDeletedErrorToken(CErrorToken *e);
	void	ReplaceDeletedNonTerminal(CNonTerminal *n);
	void	ReplaceDeletedRule(CRule *r);

	void	ClearAllData();

	CSymbol			*FindSymbolName(const string &s);
	CTerminal		*FindTerminalName(const string &s);
	CErrorToken		*FindErrorTokenName(const string &s);
	CNonTerminal	*FindNonTerminalName(const string &s);

	list<CTerminal *>::iterator		FindTerminalIterator(int i);
	list<CErrorToken *>::iterator	FindErrorTokenIterator(int i);
	list<CNonTerminal *>::iterator	FindNonTerminalIterator(int i);
	list<CRule *>::iterator			FindRuleIterator(int i);
	list<CPrecedence *>::iterator	FindPrecedenceIterator(int i);

	CTerminal				*FindTerminalIndex(int i);
	CErrorToken				*FindErrorTokenIndex(int i);
	CNonTerminal			*FindNonTerminalIndex(int i);
	CRule					*FindRuleIndex(int i);
	CPrecedence				*FindPrecedenceIndex(int i);
	SConflictResolution		*FindConflictResolutionIndex(int i);
	SErrorCode				*FindErrorCodeIndex(int i);

	void	ClearStartSymbolFlag();
	void	SetupTerminalIndexes();
	void	SetupErrorTokenIndexes();
	void	SetupNonTerminalIndexes();
	void	SetupRuleIndexes();
	void	SetupPrecedenceIndexes();
	void	SetupSymbolIndexes();

	bool		GetUndefinedSymbolRules();
	bool		GetUndefinedEntryPrecedences();

	const list<int>		&GetUndefinedSymbolRulesList() const
	{ return undefined_symbol_rules; }
	const list<int>		&GetUndefinedEntryPrecedencesList() const
	{ return undefined_entry_precedences; }

	private:
	string							start_context;
	CTerminal						end_of_file_symbol;
	CNonTerminal					augmented_start_symbol;
	CRule							augmented_start_rule;
	CNonTerminal					*start_symbol;
	list<CTerminal *>				terminal_list;
	list<CErrorToken *>				error_token_list;
	list<CNonTerminal *>			nonterminal_list;
	list<CRule *>					rule_list;
	list<CPrecedence *>				precedence_list;
	list<string>					lexical_code;
	list<string>					parser_code;
	list<string>					stack_code;
	map<int, SConflictResolution *>	conflict_resolution_map;
	map<int, SErrorCode *>			error_code_map;
	list<int>						unit_rule_order_list;
	list<int>						undefined_symbol_rules;
	list<int>						undefined_entry_precedences;
	list<string>					comment_list;

	bool						lexical_automaton_calculate;
	string						lexical_prefix_name;
	FiniteStateMachineType		lexical_finite_state_machine_type;
	int							lexical_input_range;
	bool						parser_automaton_calculate;
	string						parser_prefix_name;
	ParserType					parser_type;
	string						base_prefix_name;
	bool						generate_base_classes;
	ErrorCalculation			parser_error_calculation;
	ErrorFunctionNumbering		parser_error_function_numbering;
	ErrorFunctionAggregation	parser_error_function_aggregation;
	bool						parser_optimize_unit_rule_reduction;
	bool						generate_tty_code;
	bool						regexp_match_table;
};

const char * const undefined_text = "****";

const CSymbol 		undefined_symbol(-1, undefined_text);
const CTerminal		undefined_terminal(-1, undefined_text);
const CErrorToken	undefined_error_token(-1, undefined_text);
const CRule			undefined_rule(-1);

istream		&operator>>(istream &is, CCompilerData &d);
ostream		&operator<<(ostream &os, const CTerminal &d);
ostream		&operator<<(ostream &os, const CErrorToken &d);
ostream		&operator<<(ostream &os, const CNonTerminal &d);
ostream		&operator<<(ostream &os, const CRule &d);
ostream		&operator<<(ostream &os, const CPrecedence &d);
ostream		&operator<<(ostream &os, const SConflictResolution &d);
ostream		&operator<<(ostream &os, const SErrorCode &d);
ostream		&operator<<(ostream &os, const CCompilerData &d);

#endif
