/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "parserDriver.h"

CParserDriver::CParserDriver(	int	tce, int sca, int rca, int est, int sa,
								int aa, int ce, int sc, int nce, int nei,
								const int *a_table, const int *g_table,
								const int *rls_table, const int *rrl_table)
	:	lexical_stack(10)
{
	terminal_count_e				= tce;
	symbol_count_a					= sca;
	rule_count_a					= rca;
	empty_symbol_token				= est;
	shift_action					= sa;
	accept_action					= aa;
	conflict_entry					= ce;
	state_count						= sc;
	number_of_conflict_entries		= nce;
	number_of_error_indexes			= nei;
	action_table					= a_table;
	goto_table						= g_table;
	rule_left_symbol_table			= rls_table;
	rule_right_length_table			= rrl_table;

	stack		= 0;
	error_stack	= 0;
}

CParserDriver::~CParserDriver()
{
	delete [] stack;
	delete [] error_stack;
}

CParserDriver::StatusCode	CParserDriver::ReturnStatusCode()
{
	int		code = NoError;

	if (lexical_error_flag)
		code = LexicalError;

	if (parser_error_flag)
		code |= ParserError;

	if (code_generation_error_flag)
		code |= CodeGenerationError;

	if (halt_parser_flag)
		code |= ParserHalted;

	if (end_of_file_flag)
		code |= EndOfFile;

	return StatusCode(code);
}

void	CParserDriver::PushLexical(CLexicalDriver *l)
{
	lexical_stack.PushBack(lexical);

	lexical = l;

	if (!code_generation_flag && syntax_parsing_mode)
		lexical->HaltProcessTerminal();
}

void	CParserDriver::PopLexical()
{
	delete lexical;

	lexical = lexical_stack.LastData();

	if (!code_generation_flag && syntax_parsing_mode)
		lexical->HaltProcessTerminal();

	lexical_stack.PopBack();
}

void		CParserDriver::Initialize(	int iss, bool icgf, 
										bool dlef, bool erf, bool spm)
{
	initial_stack_size				= iss;
	initial_code_generation_flag	= icgf;
	discard_lexical_error_flag		= dlef;
	error_reporting_flag			= erf;
	syntax_parsing_mode				= spm;

	delete [] stack;
	delete [] error_stack;

	stack_size	= initial_stack_size;
	stack		= new CStackElement[stack_size];
	stack_ptr	= stack;

	stack_ptr[0].state	= 0;
	stack_ptr[0].type	= StackLiteralToken;
	stack_count			= 1;

	error_stack_size	= initial_stack_size;
	error_stack			= new CStackElement[error_stack_size];
	error_stack_ptr		= error_stack;
	error_stack_count	= 0;

	driver_state = NonErrorParse;

	code_generation_flag		= initial_code_generation_flag;

	if (!code_generation_flag && syntax_parsing_mode)
		call_reduce_function_flag = false;
	else
		call_reduce_function_flag = true;

	halt_parser_flag			= false;
	lexical_error_flag			= false;
	parser_error_flag			= false;
	code_generation_error_flag	= false;
	end_of_file_flag			= false;

	error_output_stream = &cout;
}

void		CParserDriver::DoubleStack()
{
	CStackElement	*double_stack = new CStackElement[2 * stack_size];

	for (int i=0; i<stack_size; i++)
		double_stack[i] = stack[i];

	stack_size *= 2;

	delete stack;
	stack		= double_stack;
	stack_ptr	= stack + (stack_count - 1);
}

void		CParserDriver::DoubleErrorStack()
{
	CStackElement	*double_stack = new CStackElement[2 * error_stack_size];

	for (int i=0; i<error_stack_size; i++)
		double_stack[i] = error_stack[i];

	error_stack_size *= 2;

	delete error_stack;
	error_stack		= double_stack;
	error_stack_ptr	= error_stack + (error_stack_count - 1);
}

CParserDriver::ActionType	CParserDriver::MapActionToType(int action)
{
	if (action == shift_action)						return Shift;
	else if (0 <= action && action < rule_count_a)	return Reduce;
	else if (action == accept_action)				return Accept;
	else if (action < 0)							return Error;
	else if (action >= conflict_entry)				return Conflict;
}

void	CParserDriver::HaltCodeGeneration()
{
	if (code_generation_flag)
	{
		code_generation_flag = false;

		if (syntax_parsing_mode)
		{
			call_reduce_function_flag = false;

			lexical->HaltProcessTerminal();
		}

		DeleteGeneratedCode(stack, stack_count);
	}
}

CParserDriver::StatusCode	CParserDriver::ParseStream()
{
	if (!code_generation_flag && syntax_parsing_mode)
		lexical->HaltProcessTerminal();

	while (driver_state != ParseStreamDone)
		ParseStreamSingleStep();

	return ReturnStatusCode();
}

CParserDriver::ActionType	CParserDriver::ParseStreamSingleStep()
{
	ActionType	action_type;

	switch (driver_state)
	{
		case NonErrorParse:
		action_type = ProcessNonErrorParse();
		switch (action_type)
		{
			case Shift:
			case Reduce:
			break;

			case Accept:
			case Quit:
			driver_state = ParseStreamDone;
			break;

			case Error:
			case Conflict:
			driver_state = FindErrorInputState;
			break;
		}
		break;

		case FindErrorInputState:
		action_type = ProcessFindErrorInputState();
		switch (action_type)
		{
			case Quit:
			driver_state = ParseStreamDone;
			break;

			case SetupErrorStack:
			driver_state = ErrorParseThruErrorShift;
			break;
		}
		break;

		case ErrorParseThruErrorShift:
		action_type = ProcessErrorParseThruErrorShift();
		switch (action_type)
		{
			case Reduce:
			break;

			case Quit:
			driver_state = ParseStreamDone;
			break;

			case ShiftErrorToken:
			driver_state = ErrorParseAfterErrorShift;
			break;
		}
		break;

		case ErrorParseAfterErrorShift:
		action_type = ProcessErrorParseAfterErrorShift();
		switch (action_type)
		{
			case Shift:
			case Reduce:
			break;

			case Accept:
			case Quit:
			driver_state = ParseStreamDone;
			break;

			case InputSynchronized:
			driver_state = ErrorParseSynchronization;
			break;

			case DiscardToken:
			break;
		}
		break;

		case ErrorParseSynchronization:
		action_type = ProcessErrorParseSynchronized();
		switch (action_type)
		{
			case Shift:
			case Reduce:
			break;

			case SynchronizationComplete:
			driver_state = NonErrorParse;
			break;
		}
		break;
	}

	return action_type;
}

CParserDriver::ActionType	CParserDriver::ProcessNonErrorParse()
{
	int		reduce_function_return;

	while (true)
	{
		token = lexical->GetToken();

		if (halt_parser_flag)
		{
			HaltCodeGeneration();

			if (error_reporting_flag)
			{
				string	s;

				*error_output_stream << "Parser Halted: ";
				*error_output_stream << lexical->GetLexicalStatusString(s);
				*error_output_stream << endl;
			}

			return Quit;
		}
		else if (token < 0)
		{
			lexical_error_flag = true;
			HaltCodeGeneration();

			if (error_reporting_flag)
			{
				string	s;

				*error_output_stream << "Lexical Error: ";
				*error_output_stream << lexical->GetLexicalStatusString(s);
				*error_output_stream << endl;
			}

			if (!discard_lexical_error_flag)
				return Quit;
		}
		else if (token == empty_symbol_token && lexical_stack.Size())
		{
			PopLexical();
		}
		else
			break;
	}

	machine_state	= stack_ptr->state;
	action			= action_table[machine_state * terminal_count_e + token];
		
	switch (MapActionToType(action))
	{
		case Shift:
		{
			machine_state = goto_table[machine_state * symbol_count_a + token];

			stack_ptr++;
			stack_ptr->state = machine_state;
			stack_count++;

			if (stack_count == stack_size)
				DoubleStack();

			lexical->LoadStackElement(stack_ptr);
		}
		return Shift;

		case Reduce:
		{
			int		rule_right_length = rule_right_length_table[action];

			stack_ptr -= rule_right_length;
				
			machine_state = stack_ptr->state;
			machine_state = goto_table[machine_state * symbol_count_a +
											rule_left_symbol_table[action]];
			stack_ptr++;
			stack_ptr->state			= machine_state;
			stack_ptr->reduce_action	= action;
			stack_count -= (rule_right_length - 1);

			if (stack_count == stack_size)
				DoubleStack();

			lexical->PushCurrentTokenBack();

			if (call_reduce_function_flag)
			{
				reduce_function_return = CallReduceFunction(action, 0);

				if (reduce_function_return == -1)
				{
					code_generation_error_flag = true;
					HaltCodeGeneration();
				}
			}
			else
			{
				stack_ptr->type = StackNonTerminal;
			}
		}
		return Reduce;

		case Accept:
		end_of_file_flag = true;
		return Accept;

		case Error:
		{
			parser_error_flag = true;
			HaltCodeGeneration();

			lexical->PushCurrentTokenBack();

			error_token = -1;
			CallErrorFunction(-action);
		}
		return Error;

		case Conflict:
		{
			parser_error_flag = true;
			HaltCodeGeneration();

			lexical->PushCurrentTokenBack();

			error_token = -1;
			CallConflictFunction(action - conflict_entry);
		}
		return Conflict;
	}
}

CParserDriver::ActionType	CParserDriver::ProcessFindErrorInputState()
{
	int		i;

	if (error_token == -1)
		return Quit;

	for (i=stack_count-1; i>=0; i--)
	{
		machine_state = stack[i].state;

		int	error_token_action =
				action_table[machine_state * terminal_count_e + error_token];

		if (error_token_action >= 0 && error_token_action <= shift_action)
			break;
	}

	if (i >= 0)
	{
		stack_count = i + 1;

		stack_ptr = stack + i;

		return SetupErrorStack;
	}
	else
		return Quit;
}

CParserDriver::ActionType	CParserDriver::ProcessErrorParseThruErrorShift()
{
	int		reduce_function_return;

	machine_state	= stack_ptr->state;
	action		= action_table[machine_state * terminal_count_e + error_token];
		
	switch (MapActionToType(action))
	{
		case Shift:
		{
			machine_state =
					goto_table[machine_state * symbol_count_a + error_token];

			stack_ptr++;
			stack_ptr->state	= machine_state;
			stack_ptr->type		= StackError;
			stack_ptr->token	= error_token;
			stack_count++;

			if (stack_count == stack_size)
				DoubleStack();

			error_stack_count = stack_count;

			while (error_stack_count >= error_stack_size)
				DoubleErrorStack();

			for (int i=error_stack_count-1; i>=0; i--)
				error_stack[i] = stack[i];

			error_stack_ptr	= error_stack + (error_stack_count - 1);

			input_token_fifo.Clear();
			input_token_fifo_listptr	= 0;
			synchronization_count		= 0;
			stack_modification_flag		= false;
		}
		return ShiftErrorToken;

		case Reduce:
		{
			int		rule_right_length = rule_right_length_table[action];

			stack_ptr -= rule_right_length;

			machine_state = stack_ptr->state;
			machine_state = goto_table[machine_state * symbol_count_a +
											rule_left_symbol_table[action]];
			stack_ptr++;
			stack_ptr->state			= machine_state;
			stack_ptr->reduce_action	= action;
			stack_count -= (rule_right_length - 1);

			if (stack_count == stack_size)
				DoubleStack();

			if (call_reduce_function_flag)
			{
				reduce_function_return = CallReduceFunction(action, 0);

				if (reduce_function_return == -1)
				{
					code_generation_error_flag = true;
				}
			}
			else
			{
				stack_ptr->type = StackNonTerminal;
			}
		}
		return Reduce;

		case Accept:
		case Error:
		case Conflict:
		return Quit;
	}
}

CParserDriver::ActionType	CParserDriver::ProcessErrorParseAfterErrorShift()
{
	CStackElement	stack_element;

	if (input_token_fifo_listptr)
	{
		CStackElement	&se = input_token_fifo.Data(input_token_fifo_listptr);

		token = se.token;
	}
	else
	{
		while (true)
		{
			token = lexical->GetToken();

			if (halt_parser_flag)
			{
				if (error_reporting_flag)
				{
					string	s;

					*error_output_stream << "Parser Halted: ";
					*error_output_stream << lexical->GetLexicalStatusString(s);
					*error_output_stream << endl;
				}

				return Quit;
			}
			else if (token < 0)
			{
				lexical_error_flag = true;

				if (error_reporting_flag)
				{
					string	s;

					*error_output_stream << "Lexical Error: ";
					*error_output_stream << lexical->GetLexicalStatusString(s);
					*error_output_stream << endl;
				}

				if (!discard_lexical_error_flag)
					return Quit;
			}
			else if (token == empty_symbol_token && lexical_stack.Size())
			{
				PopLexical();
			}
			else
				break;
		}

		lexical->LoadStackElement(&stack_element);

		input_token_fifo.PushBack(stack_element);

		input_token_fifo_listptr = input_token_fifo.LastListPtr();
	}

	machine_state	= error_stack_ptr->state;
	action			= action_table[machine_state * terminal_count_e + token];
		
	switch (MapActionToType(action))
	{
		case Shift:
		{
			machine_state =
					goto_table[machine_state * symbol_count_a + token];

			error_stack_ptr++;
			error_stack_ptr->state	= machine_state;
			error_stack_ptr->type	= StackLiteralToken;
			error_stack_ptr->token	= token;
			error_stack_count++;

			if (error_stack_count == error_stack_size)
				DoubleErrorStack();

			synchronization_count++;

			if (synchronization_count == synchronization_length)
			{
				input_token_fifo_listptr = input_token_fifo.FirstListPtr();

				synchronization_count = 0;

				return InputSynchronized;
			}
			else
				input_token_fifo.NextListPtr(input_token_fifo_listptr);

			stack_modification_flag = true;
		}
		return Shift;

		case Reduce:
		{
			int		rule_right_length = rule_right_length_table[action];

			error_stack_ptr -= rule_right_length;
				
			machine_state = error_stack_ptr->state;
			machine_state = goto_table[machine_state * symbol_count_a +
											rule_left_symbol_table[action]];
			error_stack_ptr++;
			error_stack_ptr->state			= machine_state;
			error_stack_ptr->type			= StackNonTerminal;
			error_stack_ptr->reduce_action	= action;
			error_stack_count -= (rule_right_length - 1);

			if (error_stack_count == error_stack_size)
				DoubleErrorStack();

			stack_modification_flag = true;
		}
		return Reduce;

		case Accept:
		end_of_file_flag = true;
		return Accept;

		case Error:
		case Conflict:
		{
			if (token == empty_symbol_token)
				return Quit;

			input_token_fifo_listptr	= input_token_fifo.PopFront();
			synchronization_count		= 0;

			if (stack_modification_flag)
			{
				error_stack_count = stack_count;

				for (int i=error_stack_count-1; i>=0; i--)
					error_stack[i] = stack[i];

				error_stack_ptr = error_stack + (error_stack_count - 1);

				stack_modification_flag = false;
			}
		}
		return DiscardToken;
	}
}

CParserDriver::ActionType	CParserDriver::ProcessErrorParseSynchronized()
{
	int		reduce_function_return;

	CStackElement	&se = input_token_fifo.Data(input_token_fifo_listptr);

	token = se.token;

	machine_state	= stack_ptr->state;
	action			= action_table[machine_state * terminal_count_e + token];
		
	switch (MapActionToType(action))
	{
		case Shift:
		{
			machine_state = goto_table[machine_state * symbol_count_a + token];

			stack_ptr++;

			*stack_ptr = se;

			stack_ptr->state = machine_state;

			stack_count++;

			if (stack_count == stack_size)
				DoubleStack();

			input_token_fifo.NextListPtr(input_token_fifo_listptr);

			synchronization_count++;

			if (synchronization_count == synchronization_length)
				return SynchronizationComplete;
		}
		return Shift;

		case Reduce:
		{
			int		rule_right_length = rule_right_length_table[action];

			stack_ptr -= rule_right_length;

			machine_state = stack_ptr->state;
			machine_state = goto_table[machine_state * symbol_count_a +
											rule_left_symbol_table[action]];
			stack_ptr++;
			stack_ptr->state			= machine_state;
			stack_ptr->reduce_action	= action;
			stack_count -= (rule_right_length - 1);

			if (stack_count == stack_size)
				DoubleStack();

			if (call_reduce_function_flag)
			{
				reduce_function_return = CallReduceFunction(action, 0);

				if (reduce_function_return == -1)
				{
					code_generation_error_flag	= true;
				}
			}
			else
			{
				stack_ptr->type = StackNonTerminal;
			}
		}
		return Reduce;

		case Accept:
		case Error:
		case Conflict:
		return Quit;
	}
}
