/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "ruleSelectView.h"

void	CRuleSelectListView::InitializePalette(const QPalette &p)
{
	active_highlight			= p.color(	QPalette::Active,
											QColorGroup::Highlight);
	active_highlighted_text		= p.color(	QPalette::Active,
											QColorGroup::HighlightedText);
	inactive_highlight			= p.color(	QPalette::Inactive,
											QColorGroup::Highlight);
	inactive_highlighted_text	= p.color(	QPalette::Inactive,
											QColorGroup::HighlightedText);

	QPalette	focus_palette = p;

	focus_palette.setColor(	QPalette::Active,
							QColorGroup::Highlight,
							inactive_highlight);

	focus_palette.setColor(	QPalette::Active,
							QColorGroup::HighlightedText,
							inactive_highlighted_text);

	setPalette(focus_palette);
}

void	CRuleSelectListView::focusInEvent(QFocusEvent *e)
{
	QPalette	p = palette();

	p.setColor(	QPalette::Active,
				QColorGroup::Highlight,
				active_highlight);

	p.setColor(	QPalette::Active,
				QColorGroup::HighlightedText,
				active_highlighted_text);

	setPalette(p);

	Q3ListView::focusInEvent(e);
}

void	CRuleSelectListView::focusOutEvent(QFocusEvent *e)
{
	QPalette	p = palette();

	p.setColor(	QPalette::Active,
				QColorGroup::Highlight,
				inactive_highlight);

	p.setColor(	QPalette::Active,
				QColorGroup::HighlightedText,
				inactive_highlighted_text);

	setPalette(p);

	Q3ListView::focusOutEvent(e);
}

CRuleSelectVBoxWidget::CRuleSelectVBoxWidget(
									QWidget *parent,
									CCompilerInterface *ci,
									SMainViewControl *mvc,
									COptionData::SSubViewData *d,
									CRuleSelectView *rsv)
	: CVBoxWidget(parent, ci, mvc), compiler_interface(ci),
		subview_data(d), rule_select_view(rsv),
		module_ptr_map(rsv->GetModulePtrMap())
{
	label = new CLabelFc("rule select", this, true);

	label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	list_view = new CRuleSelectListView(this);

	list_view->header()->setMovingEnabled(false);
	list_view->setSorting(-1);
	list_view->addColumn("index");
	list_view->addColumn("function");

	setSpacing(0);
	setStretchFactor(label, 0);
	setStretchFactor(list_view, 1);

	compiler_data = compiler_interface->GetCompilerData();

	rcv = 0;

	UpdateListView();
}

CRuleSelectVBoxWidget::~CRuleSelectVBoxWidget()
{
}

void	CRuleSelectVBoxWidget::InitializeModulePointers()
{
	vector<CSubView *>		&v1 = module_ptr_map["RULE_CODE_VIEW"];

	if (v1.size())
		rcv = dynamic_cast<CRuleCodeView *>(v1[0]);
	else
		rcv = 0;

	UpdateRuleViews();
}

string	&IntegerString(int i, string &s);

void	CRuleSelectVBoxWidget::IndexString(int index, string &s)
{
	IntegerString(index, s);
}

void	CRuleSelectVBoxWidget::FunctionString(const CRule *r, string &s, bool f)
{
	ostrstream	os;

	if (r->GetUnitRuleEliminationFlag())
		os << "  * ";
	else
		os << "    ";

	string	pad1(compiler_data->GetRuleLeftNameWidth() -
					r->GetLeft()->GetName().length() + 2, ' ');

	if (!f)
		os  << r->GetLeft()->GetName() << pad1 << "->";
	else
		os  << r->GetLeft()->GetName() << "  ->";

	const vector<const CSymbol *>			&sv = r->GetRight();
	vector<const CSymbol *>::size_type		v_index;

	for (v_index=0; v_index<sv.size(); v_index++)
		if (!f)
			os << "  " << sv[v_index]->GetName();
		else
			os << "  (" << v_index << ")" << sv[v_index]->GetName();

	os << '\0';

	s = os.str();
}

void	CRuleSelectVBoxWidget::UpdateListView()
{
	QObject::disconnect(list_view,
						SIGNAL(currentChanged(Q3ListViewItem *)),
						this,
						SLOT(SelectRuleFunction(Q3ListViewItem *)));

	compiler_data->CalculateOutputWidths();

	const list<CRule *>		&r_list = compiler_data->GetRuleList();

	int		r_count = r_list.size();
	int		l_count = list_view->childCount();
	int		i;

	if (r_count < l_count)
	{
		for (i=0; i<l_count-r_count; i++)
			delete list_view->firstChild();
	}
	else if (r_count > l_count)
	{
		for (i=0; i<r_count-l_count; i++)
			new Q3ListViewItem(list_view);
	}

	i = 0;

	list<CRule *>::const_iterator	r_iterator  = r_list.begin();

	Q3ListViewItemIterator	l_iterator(list_view);

	while (r_iterator != r_list.end())
	{
		const CRule		*r = *r_iterator;

		Q3ListViewItem	*item = l_iterator.current();

		string	s0;
		string	s1;

		IndexString(i, s0);
		FunctionString(r, s1);

		item->setText(0, s0.c_str());
		item->setText(1, s1.c_str());

		i++;
		r_iterator++;
		l_iterator++;
	}

	if (r_count)
	{
		list_view->setSelected(list_view->firstChild(), true);

		current_function = 0;
	}
	else
		current_function = -1;

	QObject::connect(	list_view,
						SIGNAL(currentChanged(Q3ListViewItem *)),
						this,
						SLOT(SelectRuleFunction(Q3ListViewItem *)));

	UpdateRuleViews();
}

void	CRuleSelectVBoxWidget::UpdateRuleViews()
{
	if (current_function != -1)
	{
		if (rcv)
		{
			CRule	*r = compiler_data->FindRuleIndex(current_function);

			if (r && r->GetCodeFlag())
				rcv->SetMultiLineEditText(r->GetCodeText());
			else
				rcv->ClearMultiLineEditText();
		}
	}
	else
	{
		if (rcv)
			rcv->ClearMultiLineEditText();
	}
}

void	CRuleSelectVBoxWidget::GetCurrentFunctionString(string &s)
{
	if (current_function != -1)
	{
		string		index_string;
		string		function_string;

		IndexString(current_function, index_string);

		CRule	*r = compiler_data->FindRuleIndex(current_function);

		FunctionString(r, function_string, true);

		s  = index_string;
		s += " ";
		s += function_string;
	}
	else
		s = "";
}

void	CRuleSelectVBoxWidget::PostCurrentFunctionChanges()
{
	if (current_function != -1)
	{
		CRule	*r = compiler_data->FindRuleIndex(current_function);

		if (!r)
			return;

		list<string>	&c_list = r->GetCodeText();

		if (rcv)
			rcv->GetMultiLineEditText(c_list);

		list<string>::const_iterator	c_iterator = c_list.begin();

		while (c_iterator != c_list.end())
		{
			if ((*c_iterator).find_first_not_of("\t\n ") != string::npos)
				break;

			c_iterator++;
		}

		if (c_iterator == c_list.end())
		{
			c_list.clear();

			r->SetCodeFlag(false);
		}
		else
			r->SetCodeFlag(true);
	}
}

void	CRuleSelectVBoxWidget::SelectRuleFunction(Q3ListViewItem *item)
{
	PostCurrentFunctionChanges();

	if (item)
		current_function = item->text(0).toInt();
	else
		current_function = -1;

	UpdateRuleViews();
}

void	CRuleSelectVBoxWidget::RestoreUndoData()
{
	compiler_data = compiler_interface->GetCompilerData();
}

void	CRuleSelectVBoxWidget::CodeTextChanged()
{
	compiler_interface->SetDataModifiedFlag(true);
}

CRuleSelectView::CRuleSelectView(	QWidget *p,
									CViewNode *n,
									CInterfaceControl *ic,
									SMainViewControl *mvc,
									const string &dfs,
									CPaletteData *dpd,
									COptionData::SSubViewData *d)
	: CSubView(p, n, ic, mvc, dfs, dpd, d)
{
	frame = new CRuleSelectVBoxWidget(
		p, dynamic_cast<CCompilerInterface *>(ic), mvc, d, this);

	QFont		view_node_font;
	QPalette	view_node_palette;

	if (SetupViewNodeFont(view_node_font))
		frame->setFont(view_node_font);
	else
		frame->setFont(frame->font());

	if (SetupViewNodePalette(view_node_palette))
	{
		frame->setPalette(view_node_palette);
		frame->InitializePalette(view_node_palette);
	}
	else
	{
		frame->setPalette(frame->palette());
		frame->InitializePalette(frame->palette());
	}
}

CRuleSelectView::~CRuleSelectView()
{
	delete frame;
}

void	CRuleSelectView::SetHighlightPalette(CPaletteData *pd)
{
	restore_palette = frame->palette();

	QPalette	highlight_palette;

	pd->SetupPalette(highlight_palette);

	frame->setPalette(highlight_palette);
}

void	CRuleSelectView::RestorePalette()
{
	frame->setPalette(restore_palette);
}

void	CRuleSelectView::InitializeModulePointers()
{
	frame->InitializeModulePointers();
}

void	CRuleSelectView::UpdateListView()
{
	frame->UpdateListView();
}

void	CRuleSelectView::GetCurrentFunctionString(string &s)
{
	frame->GetCurrentFunctionString(s);
}

void	CRuleSelectView::PostCurrentFunctionChanges()
{
	frame->PostCurrentFunctionChanges();
}

void	CRuleSelectView::RestoreUndoData()
{
	frame->RestoreUndoData();
}

void	CRuleSelectView::CodeTextChanged()
{
	frame->CodeTextChanged();
}
