/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "simulateControlView.h"

CSimulateControlControl::CSimulateControlControl(CSimulateControlVBoxWidget *v)
	: CFrameFc((QWidget *)v, 4, 2), simulate_control_vbox(v)
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		s = grid_layout_spacing_data[pixel_size];

	subgrid_layout = new QGridLayout(this, 3, 5, 0, s, "subgrid");

	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);
	grid_layout->setRowStretch(3, 1);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 1);

	subgrid_layout->setRowStretch(0, 0);
	subgrid_layout->setRowStretch(1, 0);
	subgrid_layout->setRowStretch(2, 1);

	subgrid_layout->setColStretch(0, 1);
	subgrid_layout->setColStretch(1, 0);
	subgrid_layout->setColStretch(2, 0);
	subgrid_layout->setColStretch(3, 0);
	subgrid_layout->setColStretch(4, 0);

	QString     size_string("#######");

	state_label			= new CLabelFc("State", this);
	state_value			= new CLabelFc("", this);
	terminal_label		= new CLabelFc("Terminal", this);
	terminal_value		= new CLabelFc("", this);
	action_label		= new CLabelFc("Action", this);
	action_value		= new CLabelFc("", this);
	timer_label			= new CLabelFc("Timer", this);
	timer_line_edit		= new CLineEditFc(this, "####");
	expand_button		= new CPushButtonFc("Expand", this, size_string);
	expand_spinbox		= new QSpinBox(0, 9999, 1, this);
	options_button		= new CPushButtonFc("Options", this, size_string);
	reset_button		= new CPushButtonFc("Reset", this, size_string);
	next_button			= new CPushButtonFc("Next", this, size_string);
	run_button			= new CPushButtonFc("Run", this, size_string);

	state_value->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	terminal_value->setFrameStyle(QFrame::Panel | QFrame::Sunken);
	action_value->setFrameStyle(QFrame::Panel | QFrame::Sunken);

	state_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
	state_value->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
	terminal_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
	terminal_value->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
	action_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
	action_value->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
	timer_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

	timer_line_edit->setText("500");

	subgrid_layout->addWidget(timer_label,			0, 1);
	subgrid_layout->addWidget(timer_line_edit,		0, 2);
	subgrid_layout->addWidget(expand_button,		0, 3);
	subgrid_layout->addWidget(expand_spinbox,		0, 4);
	subgrid_layout->addWidget(options_button,		1, 1);
	subgrid_layout->addWidget(reset_button,			1, 2);
	subgrid_layout->addWidget(next_button,			1, 3);
	subgrid_layout->addWidget(run_button,			1, 4);

	grid_layout->addWidget(state_label,					0, 0);
	grid_layout->addWidget(state_value,					0, 1);
	grid_layout->addWidget(terminal_label,				1, 0);
	grid_layout->addWidget(terminal_value,				1, 1);
	grid_layout->addWidget(action_label,				2, 0);
	grid_layout->addWidget(action_value,				2, 1);
	grid_layout->addMultiCellLayout(subgrid_layout,		3, 3, 0, 1);

	connect(options_button,
			SIGNAL(clicked()),
			SLOT(SetControlOptions()));

	connect(reset_button,
			SIGNAL(clicked()),
			SLOT(ResetSimulator()));

	connect(next_button,
			SIGNAL(clicked()),
			SLOT(NextOperation()));

	connect(run_button,
			SIGNAL(clicked()),
			SLOT(RunSimulator()));

	connect(expand_button,
			SIGNAL(clicked()),
			SLOT(ExpandParseTree()));
}

void	CSimulateControlControl::setFont(const QFont &f)
{
	QFontInfo       fi(f);

	int     pixel_size = fi.pixelSize();

	int     s = grid_layout_spacing_data[pixel_size];

	subgrid_layout->setMargin(0);
	subgrid_layout->setSpacing(s);
	subgrid_layout->invalidate();

	CFrameFc::setFont(f);
}

void	CSimulateControlControl::ClearSimulateControl()
{
	state_value->clear();
	terminal_value->clear();
	action_value->clear();
	expand_spinbox->setValue(0);
}

void	CSimulateControlControl::UpdateSimulateControl(	const string &state,
														const string &terminal,
														const string &action,
														int level)
{
	state_value->setText(state.c_str());
	terminal_value->setText(terminal.c_str());
	action_value->setText(action.c_str());
	expand_spinbox->setValue(level);
}

int		CSimulateControlControl::GetTimerInterval()
{
	return timer_line_edit->text().toInt();
}

void	CSimulateControlControl::SetTimerInterval(int t)
{
	ostrstream	os;

	os << t << '\0';

	timer_line_edit->setText(os.str());
}

int		CSimulateControlControl::GetExpandLevel()
{
	return expand_spinbox->value();
}

void	CSimulateControlControl::SetControlOptions()
{
	simulate_control_vbox->SetControlOptions();
}

void	CSimulateControlControl::ResetSimulator()
{
	simulate_control_vbox->ResetSimulator();
}

void	CSimulateControlControl::NextOperation()
{
	simulate_control_vbox->NextOperation();
}

void	CSimulateControlControl::RunSimulator()
{
	simulate_control_vbox->RunSimulator();
}

void	CSimulateControlControl::ExpandParseTree()
{
	simulate_control_vbox->ExpandParseTree();
}

CSimulateControlVBoxWidget::CSimulateControlVBoxWidget(
								QWidget *parent,
								CCompilerInterface *ci,
								SMainViewControl *mvc,
								COptionData::SSubViewData *d,
								CSimulateControlView *scv)
	: CVBoxWidget(parent, ci, mvc), compiler_interface(ci),
		subview_data(d), simulate_control_view(scv),
		module_ptr_map(scv->GetModulePtrMap())
{
	label = new CLabelFc("control", this, true);

	label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	simulate_control_control = new CSimulateControlControl(this);

	simulate_control_control->setFrameStyle(QFrame::Panel | QFrame::Sunken);

	setSpacing(0);
	setStretchFactor(label, 0);
	setStretchFactor(simulate_control_control, 1);

	compiler_data	= compiler_interface->GetCompilerData();
	option_data		= compiler_interface->GetOptionData();

	error_function_index_table		= 0;
	lexical_symbol_trap_list		= 0;
	parser_symbol_trap_list			= 0;

	default_filter_xpm_image		= CreateXpmImage(
										default_filter_data);
	default_token_literal_xpm_image	= CreateXpmImage(
										default_token_literal_data);
	default_token_regexp_xpm_image	= CreateXpmImage(
										default_token_regexp_data);
	default_error_token_xpm_image	= CreateXpmImage(
										default_error_token_data);
	default_nonterminal_xpm_image	= CreateXpmImage(
										default_nonterminal_data);

	end_of_file_xpm_image			= 0;
	lexical_xpm_image_table			= 0;
	parser_xpm_image_table			= 0;

	lexical	= 0;
	parser	= 0;

	timer = new QTimer(this);

	ready = false;

	connect(timer, SIGNAL(timeout()), SLOT(TimerOperation()));

	siv = 0;
	stv = 0;
	esv = 0;

	InitializeSimulator();
}

CSimulateControlVBoxWidget::~CSimulateControlVBoxWidget()
{
	delete [] error_function_index_table;
	delete [] lexical_symbol_trap_list;
	delete [] parser_symbol_trap_list;
	delete default_filter_xpm_image;
	delete default_token_literal_xpm_image;
	delete default_token_regexp_xpm_image;
	delete default_error_token_xpm_image;
	delete default_nonterminal_xpm_image;
	delete [] lexical_xpm_image_table;
	delete [] parser_xpm_image_table;

	delete lexical;
	delete parser;
}

void	CSimulateControlVBoxWidget::SetControlOptions()
{
	CSimulatorControlDialog		*scd = ParserDialog::simulator_control_dialog;

	ParserDialog::SetSimulatorControlPaletteAndFont(option_data);

	scd->SetLexicalFlagPointer(&simulate_lexical_flag);
	scd->SetParserFlagPointer(&simulate_parser_flag);
	scd->SetDiscardLexicalErrorFlagPointer(&discard_lexical_error_flag);
	scd->SetWithUnitRuleEliminationsFlagPointer(
											&with_unit_rule_eliminations_flag);

	list< pair<bool, string> >		trap_lexical_list;
	list< pair<bool, string> >		trap_parser_list;

	int		i;

	if (compiler_interface->GetLexicalCalculated())
	{
		CAutomaton	*a = compiler_interface->GetAutomaton();

		for (i=0; i<a->regexp_count; i++)
			trap_lexical_list.push_back(
				pair<bool, string>(	lexical_symbol_trap_list[i],
									a->terminals[i].name));
	}

	if (compiler_interface->GetParserCalculated())
	{
		CParse		*p = compiler_interface->GetParser();

		for (i=0; i<p->symbol_count_a; i++)
			trap_parser_list.push_back(
				pair<bool, string>(	parser_symbol_trap_list[i],
									p->symbols[i].name));
	}

	scd->SetTrapLexicalListPointer(&trap_lexical_list);
	scd->SetTrapParserListPointer(&trap_parser_list);
	scd->SetTitleAndResize("Simulator Options");
	scd->SetCompilerInterfacePointer(compiler_interface);

	if (scd->exec() == QDialog::Rejected)
		return;

	list< pair<bool, string> >::iterator	l_iterator;

	if (compiler_interface->GetLexicalCalculated())
	{
		l_iterator = trap_lexical_list.begin();

		i = 0;

		while (l_iterator != trap_lexical_list.end())
		{
			lexical_symbol_trap_list[i] = (*l_iterator++).first;

			i++;
		}
	}

	if (compiler_interface->GetParserCalculated())
	{
		l_iterator = trap_parser_list.begin();

		i = 0;

		while (l_iterator != trap_parser_list.end())
		{
			parser_symbol_trap_list[i] = (*l_iterator++).first;

			i++;
		}
	}

	CreateAutomatons();
	ResetSimulator();
}

void	CSimulateControlVBoxWidget::ResetSimulator()
{
	if (stv)
		stv->ClearTree(0);

	if (!ready) return;

	timer->stop();

	running		= false;
	finished	= false;

	if (siv)
	{
		siv->GetInputText(input_text);
		siv->SetInputCursorPosition(0, 0);
	}

	input_buffer = string(	(const char *)(input_text.local8Bit()),
							input_text.length());

	lexical->SetInputString(input_buffer);
	lexical->Initialize();

	if (simulate_parser_flag)
	{
		parser->Initialize();
		parser->SetLexical(lexical);

		driver_state		= CParserDriver::NonErrorParse;
		push_back_flag		= false;
		stack_count			= 1;
		expand_level		= 1;
	}

	simulate_control_control->ClearSimulateControl();
}

void	CSimulateControlVBoxWidget::NextOperation()
{
	if (simulate_lexical_flag)
		NextLexicalOperation();
	else if (simulate_parser_flag)
		NextParserOperation();
}

void	CSimulateControlVBoxWidget::NextLexicalOperation()
{
	if (!ready) return;

	if (finished)
	{
		timer->stop();

		running = false;

		return;
	}

	int				i;
	int				terminal_code;
	unsigned char	*terminal_ptr;
	int				terminal_size;
	string			control_state_string;
	string			control_terminal_string;
	string			control_action_string;
	string			tree_symbol_string;
	string			tree_state_string;
	string			tree_action_string;
	string			tree_value_string;

	terminal_code = lexical->GetFilterOrToken(terminal_ptr, terminal_size);

	for (i=0; i<terminal_size; i++)
		if (terminal_ptr[i] < 32)
			terminal_ptr[i] = 32;

	if (siv)
		siv->HighlightInputText(terminal_size);

	ostrstream	os;

	os << lexical->GetCurrentTerminalStateIndex() << '\0';

	control_state_string	= os.str();
	tree_state_string		= control_state_string;

	if (terminal_code == lexical->lexical_error_code)
	{
		if (discard_lexical_error_flag)
		{
			simulate_control_control->UpdateSimulateControl(
				control_state_string, (char *)terminal_ptr,
				"Discard Lexical Error", 0);
		}
		else
		{
			simulate_control_control->UpdateSimulateControl(
				control_state_string, (char *)terminal_ptr,
				"Lexical Error", 0);

			finished = true;
		}

		return;
	}

	CAutomaton	*a = compiler_interface->GetAutomaton();

	if (terminal_code != a->empty_symbol_index)
	{
		int			match_index = lexical->GetCurrentTerminalMatchIndex();
		CXpmImage	*xi;

		if (match_index == lexical->filter_terminal)
		{
			xi						= default_filter_xpm_image;
			control_terminal_string	= "Filter";
			tree_action_string		= "Filter";
		}
		else
		{
			xi						= lexical_xpm_image_table[match_index];
			control_terminal_string	= a->terminals[match_index].name;

			if (a->terminals[match_index].filter_flag)
				tree_action_string = "Filter";
			else
				tree_action_string = "Token";

			if (lexical_symbol_trap_list[match_index])
			{
				timer->stop();

				running = false;
			}
		}

		control_action_string	= "Recognize Terminal";
		tree_symbol_string		= control_terminal_string;
		tree_value_string		= (char *)terminal_ptr;

		if (stv)
			stv->ShiftSymbol(	tree_symbol_string,
								tree_state_string,
								tree_action_string,
								tree_value_string,
								xi);

		simulate_control_control->UpdateSimulateControl(
			control_state_string, control_terminal_string,
			control_action_string, 0);
	}
	else
	{
		if (stv)
			stv->ShiftSymbol(	"End Of File",
								tree_state_string,
								"Token",
								"EOF",
								0);

		simulate_control_control->UpdateSimulateControl(
			control_state_string, "End Of File",
			"Recognize Terminal", 0);

		finished = true;
	}
}

void	CSimulateControlVBoxWidget::NextParserOperation()
{
	if (!ready) return;

	if (finished)
	{
		timer->stop();

		running = false;

		return;
	}

	switch (driver_state)
	{
		case CParserDriver::NonErrorParse:
		ProcessNonErrorParse();
		break;

		case CParserDriver::FindErrorInputState:
		ProcessFindErrorInputState();
		break;

		case CParserDriver::ErrorParseThruErrorShift:
		ProcessErrorParseThruErrorShift();
		break;

		case CParserDriver::ErrorParseAfterErrorShift:
		ProcessErrorParseAfterErrorShift();
		break;

		case CParserDriver::ErrorParseSynchronization:
		ProcessErrorParseSynchronized();
		break;

		case CParserDriver::ParseStreamDone:
		finished = true;
		break;
	}

	driver_state = parser->GetDriverState();
}

bool	CSimulateControlVBoxWidget::ProcessLexical()
{
	int		i;

	if (push_back_flag)
	{
		push_back_flag = false;

		return true;
	}

	terminal_code = lexical->GetFilterOrToken(terminal_ptr, terminal_size);

	for (i=0; i<terminal_size; i++)
		if (terminal_ptr[i] < 32)
			terminal_ptr[i] = 32;

	if (siv)
		siv->HighlightInputText(terminal_size);

	if (terminal_code == lexical->lexical_error_code)
	{
		if (discard_lexical_error_flag)
		{
			simulate_control_control->UpdateSimulateControl(
			"", (char *)terminal_ptr, "Discard Lexical Error", expand_level);
		}
		else
		{
			simulate_control_control->UpdateSimulateControl(
			"", (char *)terminal_ptr, "Lexical Error", expand_level);

			finished = true;
		}

		return false;
	}

	CAutomaton	*a = compiler_interface->GetAutomaton();

	if (terminal_code != a->empty_symbol_index)
	{
		int		match_index = lexical->GetCurrentTerminalMatchIndex();

		if (match_index == lexical->filter_terminal)
		{
			simulate_control_control->UpdateSimulateControl(
				"", (char *)terminal_ptr,
				"Discard Filter Terminal", expand_level);

			return false;
		}
		else if (a->terminals[match_index].filter_flag)
		{
			simulate_control_control->UpdateSimulateControl(
				"", a->terminals[match_index].name,
				"Discard Filter Terminal", expand_level);

			return false;
		}
	}

	lexical->PushCurrentTokenBack();

	return true;
}

void	CSimulateControlVBoxWidget::ProcessNonErrorParse()
{
	CParse		*p = compiler_interface->GetParser();

	if (!ProcessLexical())
		return;

	machine_state_before	= parser->GetMachineState();
	action_type				= parser->ParseStreamSingleStep();
	machine_state_after		= parser->GetMachineState();

	switch (action_type)
	{
		case CParserDriver::Shift:
		{
			ostrstream	os1;
			ostrstream	os2;

			os1	<< machine_state_before << " -> "
				<< machine_state_after << '\0';

			control_state_string	= os1.str();
			control_terminal_string	= p->symbols[terminal_code].name;
			control_action_string	= "Shift";

			os2 << machine_state_after << '\0';

			tree_symbol_string		= control_terminal_string;
			tree_state_string		= os2.str();
			tree_action_string		= control_action_string;
			tree_value_string		= (char *)terminal_ptr;

			if (stv)
				stv->ShiftSymbol(	tree_symbol_string,
									tree_state_string,
									tree_action_string,
									tree_value_string,
									parser_xpm_image_table[terminal_code]);

			simulate_control_control->UpdateSimulateControl(
				control_state_string, control_terminal_string,
				control_action_string, expand_level);

			stack_count++;

			if (parser_symbol_trap_list[terminal_code])
			{
				timer->stop();

				running = false;
			}
		}
		break;

		case CParserDriver::Reduce:
		{
			push_back_flag = true;

			int		rule_index = parser->GetAction();

			CRuleInformation	&ri = p->rules[rule_index];

			int		goto_symbol		= ri.left_symbol_index;
			int		reduce_length	= ri.right_symbol_count;

			ostrstream	os1;
			ostrstream	os2;
			ostrstream	os3;

			os1	<< machine_state_before << " -> "
				<< machine_state_after << '\0';

			os2	<< "Reduce " << rule_index << '\0';

			control_state_string	= os1.str();
			control_terminal_string	= p->symbols[terminal_code].name;
			control_action_string	= os2.str();

			os3 << machine_state_after << '\0';

			tree_symbol_string		= p->symbols[goto_symbol].name;
			tree_state_string		= os3.str();
			tree_action_string		= control_action_string;
			tree_value_string		= "";

			int		subtree_depth = 0;

			if (stv)
				subtree_depth =
					stv->ReduceTop(	tree_symbol_string,
									tree_state_string,
									tree_action_string,
									tree_value_string,
									parser_xpm_image_table[goto_symbol],
									reduce_length);

			if (expand_level < subtree_depth)
				expand_level = subtree_depth;

			simulate_control_control->UpdateSimulateControl(
				control_state_string, control_terminal_string,
				control_action_string, expand_level);

			stack_count -= (reduce_length - 1);

			if (parser_symbol_trap_list[goto_symbol])
			{
				timer->stop();

				running = false;
			}
		}
		break;

		case CParserDriver::Accept:
		{
			ostrstream	os1;
			ostrstream	os2;

			os1	<< machine_state_before << " -> "
				<< machine_state_after << '\0';

			control_state_string	= os1.str();
			control_terminal_string	= "End Of File";
			control_action_string	= "Accept";

			os2 << machine_state_after << '\0';

			tree_symbol_string		= control_terminal_string;
			tree_state_string		= os2.str();
			tree_action_string		= control_action_string;
			tree_value_string		= "EOF";

			if (stv)
				stv->ShiftSymbol(	tree_symbol_string,
									tree_state_string,
									tree_action_string,
									tree_value_string,
									0);

			simulate_control_control->UpdateSimulateControl(
				control_state_string, control_terminal_string,
				control_action_string, expand_level);
		}
		break;

		case CParserDriver::Error:
		{
			push_back_flag = true;

			ostrstream	os1;

			os1	<< machine_state_before << '\0';

			control_state_string	= os1.str();
			control_terminal_string	= p->symbols[terminal_code].name;
			control_action_string	= "Error";

			simulate_control_control->UpdateSimulateControl(
				control_state_string, control_terminal_string,
				control_action_string, expand_level);

			int		error_token_index =
						esv->GetSimulatorErrorToken(- parser->GetAction());

			CErrorToken		*et =
				compiler_data->FindErrorTokenIndex(error_token_index);

			if (et)
			{
				error_token_code	= et->GetSymbolIndex();
				error_token_string	= et->GetName();

				parser->SetErrorToken(error_token_code);
				parser->SetSynchronizationLength(
										et->GetSynchronizationLength());
			}
		}
		break;

		case CParserDriver::Conflict:
		{
			push_back_flag = true;

			ostrstream	os1;

			os1	<< machine_state_before << '\0';

			control_state_string	= os1.str();
			control_terminal_string	= p->symbols[terminal_code].name;
			control_action_string	= "Conflict";

			simulate_control_control->UpdateSimulateControl(
				control_state_string, control_terminal_string,
				control_action_string, expand_level);

			finished = true;
		}
		break;

		case CParserDriver::Quit:
		{
			ostrstream	os1;

			os1	<< machine_state_before << '\0';

			control_state_string	= os1.str();
			control_terminal_string	= p->symbols[terminal_code].name;
			control_action_string	= "Quit On Halt";

			simulate_control_control->UpdateSimulateControl(
				control_state_string, control_terminal_string,
				control_action_string, expand_level);
		}
		break;
	}
}

void	CSimulateControlVBoxWidget::ProcessFindErrorInputState()
{
	CParse		*p = compiler_interface->GetParser();

	pop_top_length			= parser->GetStackCount();
	machine_state_before	= parser->GetMachineState();
	action_type				= parser->ParseStreamSingleStep();
	machine_state_after		= parser->GetMachineState();

	switch (action_type)
	{
		case CParserDriver::Quit:
		{
			ostrstream	os1;

			os1	<< machine_state_before << '\0';

			control_state_string	= os1.str();
			control_terminal_string	= p->symbols[terminal_code].name;
			control_action_string	= "Quit On Find Error State";

			simulate_control_control->UpdateSimulateControl(
				control_state_string, control_terminal_string,
				control_action_string, expand_level);
		}
		break;

		case CParserDriver::SetupErrorStack:
		{
			pop_top_length -= parser->GetStackCount();

			if (stv)
				stv->PopTop(pop_top_length);

			ostrstream	os1;

			os1	<< machine_state_before << '\0';

			control_state_string	= os1.str();
			control_terminal_string	= error_token_string;
			control_action_string	= "Setup Error Stack";

			simulate_control_control->UpdateSimulateControl(
				control_state_string, control_terminal_string,
				control_action_string, expand_level);
		}
		break;
	}
}

void	CSimulateControlVBoxWidget::ProcessErrorParseThruErrorShift()
{
	CParse		*p = compiler_interface->GetParser();

	machine_state_before	= parser->GetMachineState();
	action_type				= parser->ParseStreamSingleStep();
	machine_state_after		= parser->GetMachineState();

	switch (action_type)
	{
		case CParserDriver::ShiftErrorToken:
		{
			if (stv)
				stv->RestoreTop();

			ostrstream	os1;
			ostrstream	os2;

			os1	<< machine_state_before << " -> "
				<< machine_state_after << '\0';

			control_state_string	= os1.str();
			control_terminal_string	= p->symbols[error_token_code].name;
			control_action_string	= "Shift Error Token";

			os2 << machine_state_after << '\0';

			tree_symbol_string		= control_terminal_string;
			tree_state_string		= os2.str();
			tree_action_string		= control_action_string;
			tree_value_string		= "";

			int		subtree_depth = 0;

			if (stv)
				subtree_depth =
					stv->ReduceTop(	tree_symbol_string,
									tree_state_string,
									tree_action_string,
									tree_value_string,
									parser_xpm_image_table[error_token_code],
									pop_top_length);

			if (expand_level < subtree_depth)
				expand_level = subtree_depth;

			simulate_control_control->UpdateSimulateControl(
				control_state_string, control_terminal_string,
				control_action_string, expand_level);

			stack_count++;

			synchronization_tokens.clear();

			synchronization_iterator = synchronization_tokens.end();

			if (parser_symbol_trap_list[error_token_code])
			{
				timer->stop();

				running = false;
			}
		}
		break;

		case CParserDriver::Reduce:
		{
			int		rule_index = parser->GetAction();

			CRuleInformation	&ri = p->rules[rule_index];

			int		goto_symbol		= ri.left_symbol_index;
			int		reduce_length	= ri.right_symbol_count;

			ostrstream	os1;
			ostrstream	os2;
			ostrstream	os3;

			os1	<< machine_state_before << " -> "
				<< machine_state_after << '\0';

			os2	<< "Reduce " << rule_index << '\0';

			control_state_string	= os1.str();
			control_terminal_string	= p->symbols[error_token_code].name;
			control_action_string	= os2.str();

			os3 << machine_state_after << '\0';

			tree_symbol_string		= p->symbols[goto_symbol].name;
			tree_state_string		= os3.str();
			tree_action_string		= control_action_string;
			tree_value_string		= "";

			int		subtree_depth = 0;

			if (stv)
				subtree_depth =
					stv->ReduceTop(	tree_symbol_string,
									tree_state_string,
									tree_action_string,
									tree_value_string,
									parser_xpm_image_table[goto_symbol],
									reduce_length);

			if (expand_level < subtree_depth)
				expand_level = subtree_depth;

			simulate_control_control->UpdateSimulateControl(
				control_state_string, control_terminal_string,
				control_action_string, expand_level);

			stack_count -= (reduce_length - 1);

			if (parser_symbol_trap_list[goto_symbol])
			{
				timer->stop();

				running = false;
			}
		}
		break;

		case CParserDriver::Quit:
		{
			ostrstream	os1;

			os1	<< machine_state_before << '\0';

			control_state_string	= os1.str();
			control_terminal_string	= error_token_string;
			control_action_string	= "Quit Before Error Shift";

			simulate_control_control->UpdateSimulateControl(
				control_state_string, control_terminal_string,
				control_action_string, expand_level);
		}
		break;
	}
}

void	CSimulateControlVBoxWidget::ProcessErrorParseAfterErrorShift()
{
	if (synchronization_iterator == synchronization_tokens.end())
	{
		if (!ProcessLexical())
			return;

		synchronization_tokens.push_back(
			pair<int, string>(terminal_code, (char *)terminal_ptr));

		synchronization_iterator = --synchronization_tokens.end();
	}

	machine_state_before	= parser->GetMachineState();
	action_type				= parser->ParseStreamSingleStep();
	machine_state_after		= parser->GetMachineState();

	ostrstream	os1;

	os1	<< machine_state_before << " -> "
		<< machine_state_after << '\0';

	control_state_string	= os1.str();
	control_terminal_string	= (*synchronization_iterator).second;

	switch (action_type)
	{
		case CParserDriver::Shift:
		synchronization_iterator++;
		control_action_string = "Error Trial Shift";
		break;

		case CParserDriver::Reduce:
		control_action_string = "Error Trial Reduce";
		break;

		case CParserDriver::Accept:
		control_action_string = "Error Trial Accept";
		break;

		case CParserDriver::Quit:
		control_action_string = "Error Trial Quit";
		break;

		case CParserDriver::InputSynchronized:
		synchronization_iterator = synchronization_tokens.begin();
		control_action_string = "Error Trial Input Synchronized";
		break;

		case CParserDriver::DiscardToken:
		control_terminal_string	= (*synchronization_tokens.begin()).second;
		synchronization_tokens.pop_front();
		synchronization_iterator = synchronization_tokens.begin();
		control_action_string = "Error Trial Discard Token";
		break;
	}

	simulate_control_control->UpdateSimulateControl(
		control_state_string, control_terminal_string,
		control_action_string, expand_level);
}

void	CSimulateControlVBoxWidget::ProcessErrorParseSynchronized()
{
	CParse		*p = compiler_interface->GetParser();

	machine_state_before	= parser->GetMachineState();
	action_type				= parser->ParseStreamSingleStep();
	machine_state_after		= parser->GetMachineState();

	terminal_code	= (*synchronization_iterator).first;
	terminal_ptr	=
		(unsigned char *)(*synchronization_iterator).second.c_str();

	switch (action_type)
	{
		case CParserDriver::Shift:
		{
			ostrstream	os1;
			ostrstream	os2;

			os1	<< machine_state_before << " -> "
				<< machine_state_after << '\0';

			control_state_string	= os1.str();
			control_terminal_string	= p->symbols[terminal_code].name;
			control_action_string	= "Synchronization Shift";

			os2 << machine_state_after << '\0';

			tree_symbol_string		= control_terminal_string;
			tree_state_string		= os2.str();
			tree_action_string		= control_action_string;
			tree_value_string		= (char *)terminal_ptr;

			if (stv)
				stv->ShiftSymbol(	tree_symbol_string,
									tree_state_string,
									tree_action_string,
									tree_value_string,
									parser_xpm_image_table[terminal_code]);

			simulate_control_control->UpdateSimulateControl(
				control_state_string, control_terminal_string,
				control_action_string, expand_level);

			stack_count++;

			synchronization_iterator++;

			if (parser_symbol_trap_list[terminal_code])
			{
				timer->stop();

				running = false;
			}
		}
		break;

		case CParserDriver::Reduce:
		{
			int		rule_index = parser->GetAction();

			CRuleInformation	&ri = p->rules[rule_index];

			int		goto_symbol		= ri.left_symbol_index;
			int		reduce_length	= ri.right_symbol_count;

			ostrstream	os1;
			ostrstream	os2;
			ostrstream	os3;

			os1	<< machine_state_before << " -> "
				<< machine_state_after << '\0';

			os2	<< "Synchronization Reduce " << rule_index << '\0';

			control_state_string	= os1.str();
			control_terminal_string	= p->symbols[terminal_code].name;
			control_action_string	= os2.str();

			os3 << machine_state_after << '\0';

			tree_symbol_string		= p->symbols[goto_symbol].name;
			tree_state_string		= os3.str();
			tree_action_string		= control_action_string;
			tree_value_string		= "";

			int		subtree_depth = 0;

			if (stv)
				subtree_depth =
					stv->ReduceTop(	tree_symbol_string,
									tree_state_string,
									tree_action_string,
									tree_value_string,
									parser_xpm_image_table[goto_symbol],
									reduce_length);

			if (expand_level < subtree_depth)
				expand_level = subtree_depth;

			simulate_control_control->UpdateSimulateControl(
				control_state_string, control_terminal_string,
				control_action_string, expand_level);

			stack_count -= (reduce_length - 1);

			if (parser_symbol_trap_list[goto_symbol])
			{
				timer->stop();

				running = false;
			}
		}
		break;

		case CParserDriver::SynchronizationComplete:
		{
			ostrstream	os1;
			ostrstream	os2;

			os1	<< machine_state_before << " -> "
				<< machine_state_after << '\0';

			control_state_string	= os1.str();
			control_terminal_string	= p->symbols[terminal_code].name;
			control_action_string	= "Synchronization Shift Complete";

			os2 << machine_state_after << '\0';

			tree_symbol_string		= control_terminal_string;
			tree_state_string		= os2.str();
			tree_action_string		= control_action_string;
			tree_value_string		= (char *)terminal_ptr;

			if (stv)
				stv->ShiftSymbol(	tree_symbol_string,
									tree_state_string,
									tree_action_string,
									tree_value_string,
									parser_xpm_image_table[terminal_code]);

			simulate_control_control->UpdateSimulateControl(
				control_state_string, control_terminal_string,
				control_action_string, expand_level);

			stack_count++;

			synchronization_iterator++;

			if (parser_symbol_trap_list[terminal_code])
			{
				timer->stop();

				running = false;
			}
		}
		break;
	}
}

void	CSimulateControlVBoxWidget::RunSimulator()
{
	if (!ready) return;

	if (!running)
	{
		NextOperation();

		timer->start(simulate_control_control->GetTimerInterval());

		running = true;
	}
	else
	{
		timer->stop();

		running = false;
	}
}

void	CSimulateControlVBoxWidget::ExpandParseTree()
{
	if (stv)
		stv->ExpandTree(simulate_control_control->GetExpandLevel());
}

void	CSimulateControlVBoxWidget::InitializeSimulator()
{
	if (compiler_interface->GetLexicalCalculated())
	{
		if (compiler_interface->GetParserCalculated())
		{
			simulate_lexical_flag	= false;
			simulate_parser_flag	= true;
		}
		else
		{
			simulate_lexical_flag	= true;
			simulate_parser_flag	= false;
		}

		discard_lexical_error_flag	= false;

		if (compiler_interface->GetUnitRuleEliminationCalculated())
			with_unit_rule_eliminations_flag = true;
		else
			with_unit_rule_eliminations_flag = false;
	}
	else
	{
		simulate_lexical_flag				= false;
		simulate_parser_flag				= false;
		discard_lexical_error_flag			= false;
		with_unit_rule_eliminations_flag	= false;
	}

	int		i;

	if (compiler_interface->GetLexicalCalculated())
	{
		CAutomaton	*a = compiler_interface->GetAutomaton();

		delete [] lexical_symbol_trap_list;

		lexical_symbol_trap_list = new bool[a->regexp_count];

		for (i=0; i<a->regexp_count; i++)
			lexical_symbol_trap_list[i] = false;

		delete [] lexical_xpm_image_table;

		lexical_xpm_image_table = new (CXpmImage *[a->regexp_count]);

		for (i=0; i<a->regexp_count; i++)
		{
			string	s("XPM_IMAGE_");

			s += a->terminals[i].name;

			CXpmImage	*xi = option_data->GetAuxiliaryXpmImage(s);

			if (xi)
			{
				xi->CreatePixmap();

				lexical_xpm_image_table[i] = xi;
			}
			else
			{
				if (a->terminals[i].filter_flag)
				lexical_xpm_image_table[i] = default_filter_xpm_image;
				else if (a->terminals[i].literal_flag)
				lexical_xpm_image_table[i] = default_token_literal_xpm_image;
				else
				lexical_xpm_image_table[i] = default_token_regexp_xpm_image;
			}
		}

		CXpmImage	*xi = option_data->GetAuxiliaryXpmImage("XPM_IMAGE_EOF");

		if (xi)
		{
			xi->CreatePixmap();

			end_of_file_xpm_image = xi;
		}
		else
			end_of_file_xpm_image = default_token_literal_xpm_image;
	}

	if (compiler_interface->GetParserCalculated())
	{
		CParse		*p = compiler_interface->GetParser();

		CCompilerData::ErrorCalculation
			c = compiler_data->GetParserErrorCalculation();

		CCompilerData::ErrorFunctionNumbering
			n = compiler_data->GetParserErrorFunctionNumbering();

		CCompilerData::ErrorFunctionAggregation
			aggregation = compiler_data->GetParserErrorFunctionAggregation();

		if (!compiler_interface->GetErrorEntriesNumbered())
		{
			number_of_error_indexes		= 1;
			number_of_error_functions	= 1;
		}
		else
		{
			if (c == CCompilerData::NoCalculation)
			{
				if (n == CCompilerData::OneNumber)
				{
					number_of_error_indexes = 1;
				}
				else if (n == CCompilerData::NumberPerState)
				{
					number_of_error_indexes = p->machine_count;
				}
				else if (n == CCompilerData::NumberPerEntry)
				{
					number_of_error_indexes = p->default_error_entry_count;
				}

				if (aggregation == CCompilerData::OneFunction)
				{
					number_of_error_functions = 1;
				}
				else if (aggregation == CCompilerData::FunctionPerState)
				{
					number_of_error_functions = p->machine_count;
				}
				else if (aggregation == CCompilerData::FunctionPerEntry)
				{
					number_of_error_functions = p->default_error_entry_count;
				}
			}
			else if (	c == CCompilerData::ApproximateEssential ||
						c == CCompilerData::ExactEssential)
			{
				if (n == CCompilerData::OneNumber)
				{
					number_of_error_indexes = 2;
				}
				else if (n == CCompilerData::NumberPerState)
				{
					number_of_error_indexes = p->machine_count + 1;
				}
				else if (n == CCompilerData::NumberPerEntry)
				{
					number_of_error_indexes =
						p->essential_error_entry_count + 1;
				}
	
				if (aggregation == CCompilerData::OneFunction)
				{
					number_of_error_functions = 2;
				}
				else if (aggregation == CCompilerData::FunctionPerState)
				{
					number_of_error_functions = p->machine_count + 1;
				}
				else if (aggregation == CCompilerData::FunctionPerEntry)
				{
					number_of_error_functions =
						p->essential_error_entry_count + 1;
				}
			}
		}

		delete [] error_function_index_table;

		error_function_index_table	= new int[number_of_error_indexes + 1];

		for (i=1; i<=number_of_error_indexes; i++)
		{
			if (!compiler_interface->GetErrorEntriesNumbered())
			{
				error_function_index_table[i] = 1;
			}
			else
			{
				if (c == CCompilerData::NoCalculation)
				{
					if (aggregation == CCompilerData::OneFunction)
					{
						error_function_index_table[i] = 1;
					}
					else if (aggregation == CCompilerData::FunctionPerState)
					{
						if (n == CCompilerData::NumberPerState)
						{
							error_function_index_table[i] = i;
						}
						else if (n == CCompilerData::NumberPerEntry)
						{
							error_function_index_table[i] =
								p->default_error_state_terminals[i].state + 1;
						}
					}
					else if (aggregation == CCompilerData::FunctionPerEntry)
					{
						error_function_index_table[i] = i;
					}
				}
				else if (	c == CCompilerData::ApproximateEssential ||
							c == CCompilerData::ExactEssential)
				{
					if (aggregation == CCompilerData::OneFunction)
					{
						if (i == 1)
							error_function_index_table[i] = 1;
						else
							error_function_index_table[i] = 2;
					}
					else if (aggregation == CCompilerData::FunctionPerState)
					{
						if (i == 1 || n == CCompilerData::NumberPerState)
						{
							error_function_index_table[i] = i;
						}
						else if (n == CCompilerData::NumberPerEntry)
						{
							error_function_index_table[i] =
								p->essential_error_state_terminals[i].state + 2;
						}
					}
					else if (aggregation == CCompilerData::FunctionPerEntry)
					{
						error_function_index_table[i] = i;
					}
				}
			}
		}

		delete [] parser_symbol_trap_list;

		parser_symbol_trap_list = new bool[p->symbol_count_a];

		for (i=0; i<p->symbol_count_a; i++)
			parser_symbol_trap_list[i] = false;

		delete [] parser_xpm_image_table;

		parser_xpm_image_table = new (CXpmImage *[p->symbol_count_a]);

		for (i=0; i<p->symbol_count_a; i++)
		{
			string	s("XPM_IMAGE_");

			s += p->symbols[i].name;

			CXpmImage	*xi = option_data->GetAuxiliaryXpmImage(s);

			if (xi)
			{
				xi->CreatePixmap();

				parser_xpm_image_table[i] = xi;
			}
			else
			{
				if (p->symbols[i].nonterminal_flag)
				parser_xpm_image_table[i] = default_nonterminal_xpm_image;
				else if (p->symbols[i].error_flag)
				parser_xpm_image_table[i] = default_error_token_xpm_image;
				else if (p->symbols[i].literal_flag)
				parser_xpm_image_table[i] = default_token_literal_xpm_image;
				else
				parser_xpm_image_table[i] = default_token_regexp_xpm_image;
			}
		}
	}

	CreateAutomatons();
	ResetSimulator();
}

void	CSimulateControlVBoxWidget::InitializeModulePointers()
{
	vector<CSubView *>		&v1 = module_ptr_map["SIMULATE_INPUT_VIEW"];
	vector<CSubView *>		&v2 = module_ptr_map["SIMULATE_TREE_VIEW"];
	vector<CSubView *>		&v3 = module_ptr_map["ERROR_SELECT_VIEW"];

	if (v1.size())
		siv = dynamic_cast<CSimulateInputView *>(v1[0]);
	else
		siv = 0;

	if (v2.size())
		stv = dynamic_cast<CSimulateTreeView *>(v2[0]);
	else
		stv = 0;

	if (v3.size())
		esv = dynamic_cast<CErrorSelectView *>(v3[0]);
	else
		esv = 0;
}

CXpmImage	*CSimulateControlVBoxWidget::CreateXpmImage(
											const char * const data[])
{
	CXpmImage	*xi	= new CXpmImage();
	int			i	= 0;

	while (data[i])
		xi->PushBackXpmCharData(data[i++]);

	xi->CreatePixmap();

	return xi;
}

void	CSimulateControlVBoxWidget::CreateAutomatons()
{
	delete lexical;
	delete parser;

	int		i;

	if (compiler_interface->GetLexicalCalculated())
	{
		CAutomaton	*a = compiler_interface->GetAutomaton();

		if (a->scan_type == CLexicalDriver::SingleCharLexical)
		{
			char	*lex_char_token_names[a->regexp_count + 1];

			for (i=0; i<a->regexp_count; i++)
				lex_char_token_names[i] = a->terminals[i].name;

			lex_char_token_names[i] = 0;

			lexical = new CLexicalSingleCharDriver(lex_char_token_names);
		}
		else if (a->scan_type == CLexicalDriver::RegexpLexical)
		{
			lexical = new CLexicalRegexpDriver(	a->token_count,
												a->filter_count,
												a->regexp_count,
												a->empty_symbol_index,
												a->input_range,
												a->fsm_count,
												a->fsm_table,
												a->regexp_match,
												a->prefix_match,
												a->terminal_match,
												a->start_assignment,
												a->trailing_context,
												a->token_map,
												a->filter_map,
												a->literal_terminal,
												0);
		}

		ready = true;
	}
	else
	{
		lexical = 0;

		ready = false;
	}

	if (compiler_interface->GetParserCalculated())
	{
		CParse		*p = compiler_interface->GetParser();

		int		m_count;
		int		*action_table;
		int		*goto_table;

		if (!with_unit_rule_eliminations_flag)
			m_count = p->machine_count;
		else
			m_count = p->unit_machine_count;

		if (!compiler_interface->GetErrorEntriesNumbered())
			action_table = p->action_table;
		else if (!with_unit_rule_eliminations_flag)
			action_table = p->error_action_table;
		else
			action_table = p->unit_action_table;

		if (!with_unit_rule_eliminations_flag)
			goto_table = p->goto_table;
		else
			goto_table = p->unit_goto_table;

		parser = new CParserDriver(	p->terminal_count_e,
									p->symbol_count_a,
									p->rule_count_a,
									p->empty_symbol_index,
									p->shift_action,
									p->accept_action,
									p->conflict_entry,
									m_count,
									p->parsing_conflicts->Size(),
									number_of_error_indexes,
									action_table,
									goto_table,
									p->rule_left_symbol_table,
									p->rule_right_length_table);
	}
	else
		parser = 0;
}

void	CSimulateControlVBoxWidget::TimerOperation()
{
	NextOperation();
}

void	CSimulateControlVBoxWidget::RestoreUndoData()
{
	compiler_data = compiler_interface->GetCompilerData();
}

CSimulateControlView::CSimulateControlView(	QWidget *p,
											CViewNode *n,
											CInterfaceControl *ic,
											SMainViewControl *mvc,
											const string &dfs,
											CPaletteData *dpd,
											COptionData::SSubViewData *d)
	: CSubView(p, n, ic, mvc, dfs, dpd, d)
{
	frame = new CSimulateControlVBoxWidget(
		p, dynamic_cast<CCompilerInterface *>(ic), mvc, d, this);

	QFont		view_node_font;
	QPalette	view_node_palette;

	if (SetupViewNodeFont(view_node_font))
		frame->setFont(view_node_font);
	else
		frame->setFont(frame->font());

	if (SetupViewNodePalette(view_node_palette))
		frame->setPalette(view_node_palette);
	else
		frame->setPalette(frame->palette());
}

CSimulateControlView::~CSimulateControlView()
{
	delete frame;
}

void	CSimulateControlView::SetHighlightPalette(CPaletteData *pd)
{
	restore_palette = frame->palette();

	QPalette	highlight_palette;

	pd->SetupPalette(highlight_palette);

	frame->setPalette(highlight_palette);
}

void	CSimulateControlView::RestorePalette()
{
	frame->setPalette(restore_palette);
}

void	CSimulateControlView::InitializeModulePointers()
{
	frame->InitializeModulePointers();
}

void	CSimulateControlView::SetControlOptions()
{
	frame->SetControlOptions();
}

void	CSimulateControlView::ResetSimulator()
{
	frame->ResetSimulator();
}

void	CSimulateControlView::NextOperation()
{
	frame->NextOperation();
}

void	CSimulateControlView::RunSimulator()
{
	frame->RunSimulator();
}

void	CSimulateControlView::ExpandParseTree()
{
	frame->ExpandParseTree();
}

void	CSimulateControlView::InitializeSimulator()
{
	frame->InitializeSimulator();
}

void	CSimulateControlView::RestoreUndoData()
{
	frame->RestoreUndoData();
}
