/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "classCodeView.h"

CClassCodeControl::CClassCodeControl(CClassCodeVBoxWidget *v)
	: CFrameFc((QWidget *)v, 1, 4), class_code_vbox(v)
{
	grid_layout->setRowStretch(0, 1);

	grid_layout->setColStretch(0, 1);
	grid_layout->setColStretch(1, 0);
	grid_layout->setColStretch(2, 0);
	grid_layout->setColStretch(3, 0);

	view_label				= new CLabelFc("class code", this);
	code_section_label		= new CLabelFc("Class Code Section", this);
	code_section_combo_box	= new CComboBoxFc(false, this,
									"class code section string");
	spacer					= new CSpacerFc(this);

	view_label->setFrameStyle(QFrame::Panel | QFrame::Raised);
	view_label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
	code_section_label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	grid_layout->addWidget(view_label,				0, 0);
	grid_layout->addWidget(code_section_label,		0, 1);
	grid_layout->addWidget(code_section_combo_box,	0, 2);
	grid_layout->addWidget(spacer,					0, 3);
}

void	CClassCodeControl::ResetControl(const list<string> &s, int section)
{
	disconnect(	code_section_combo_box,
				SIGNAL(activated(int)),
				this,
				SLOT(SelectCodeSection(int)));

	code_section_combo_box->clear();

	list<string>::const_iterator	s_iterator = s.begin();

	while (s_iterator != s.end())
		code_section_combo_box->insertItem((*s_iterator++).c_str());

	code_section_combo_box->setCurrentItem(section);

	connect(code_section_combo_box,
			SIGNAL(activated(int)),
			this,
			SLOT(SelectCodeSection(int)));
}

void	CClassCodeControl::SelectCodeSection(int section)
{
	class_code_vbox->SelectCodeSection(section);
}

CClassCodeVBoxWidget::CClassCodeVBoxWidget(
									QWidget *parent,
									CCompilerInterface *ci,
									SMainViewControl *mvc,
									COptionData::SSubViewData *d,
									CClassCodeView *ccv)
	: CVBoxWidget(parent, ci, mvc), compiler_interface(ci),
		subview_data(d), class_code_view(ccv),
		module_ptr_map(ccv->GetModulePtrMap())
{
	class_code_control	= new CClassCodeControl(this);
	multi_line_edit = new CMultiLineEditFc(this, "");

	multi_line_edit->setWordWrap(Q3MultiLineEdit::NoWrap);

	setSpacing(0);
	setStretchFactor(class_code_control, 0);
	setStretchFactor(multi_line_edit, 1);

	compiler_data = compiler_interface->GetCompilerData();

	connect(multi_line_edit,
			SIGNAL(textChanged()),
			this,
			SLOT(CodeTextChanged()));

	UpdateView();
}

CClassCodeVBoxWidget::~CClassCodeVBoxWidget()
{
}

void	CClassCodeVBoxWidget::InitializeModulePointers()
{
}

void	CClassCodeVBoxWidget::UpdateView()
{
	class_code_section = 0;

	list<string>	c_types;

	c_types.push_back("Lexical Include");
	c_types.push_back("Lexical Class");
	c_types.push_back("Lexical Source");
	c_types.push_back("Lexical Constructor");
	c_types.push_back("Lexical Destructor");
	c_types.push_back("Parser Include");
	c_types.push_back("Parser Class");
	c_types.push_back("Parser Source");
	c_types.push_back("Parser Constructor");
	c_types.push_back("Parser Destructor");
	c_types.push_back("Stack Element");

	class_code_control->ResetControl(c_types, class_code_section);

	UpdateMultiLineEdit();
}

void	CClassCodeVBoxWidget::UpdateMultiLineEdit()
{
	list<string>	lexical_include_code;
	list<string>	lexical_class_code;
	list<string>	lexical_source_code;
	list<string>	lexical_constructor_code;
	list<string>	lexical_destructor_code;
	list<string>	parser_include_code;
	list<string>	parser_class_code;
	list<string>	parser_source_code;
	list<string>	parser_constructor_code;
	list<string>	parser_destructor_code;

	switch (class_code_section)
	{
		case 0:
		case 1:
		case 2:
		case 3:
		case 4:
		{
			compiler_data->GetLexicalCodeSections(	lexical_include_code,
													lexical_class_code,
													lexical_source_code,
													lexical_constructor_code,
													lexical_destructor_code);

			switch (class_code_section)
			{
				case 0:
				SetMultiLineEditText(lexical_include_code);
				break;

				case 1:
				SetMultiLineEditText(lexical_class_code);
				break;

				case 2:
				SetMultiLineEditText(lexical_source_code);
				break;

				case 3:
				SetMultiLineEditText(lexical_constructor_code);
				break;

				case 4:
				SetMultiLineEditText(lexical_destructor_code);
				break;
			}
		}
		break;

		case 5:
		case 6:
		case 7:
		case 8:
		case 9:
		{
			compiler_data->GetParserCodeSections(	parser_include_code,
													parser_class_code,
													parser_source_code,
													parser_constructor_code,
													parser_destructor_code);

			switch (class_code_section)
			{
				case 5:
				SetMultiLineEditText(parser_include_code);
				break;

				case 6:
				SetMultiLineEditText(parser_class_code);
				break;

				case 7:
				SetMultiLineEditText(parser_source_code);
				break;

				case 8:
				SetMultiLineEditText(parser_constructor_code);
				break;

				case 9:
				SetMultiLineEditText(parser_destructor_code);
				break;
			}
		}
		break;

		case 10:
		SetMultiLineEditText(compiler_data->GetStackCode());
		break;
	}
}

void	CClassCodeVBoxWidget::SetMultiLineEditText(const list<string> &code)
{
	disconnect(	multi_line_edit,
				SIGNAL(textChanged()),
				this,
				SLOT(CodeTextChanged()));

	multi_line_edit->clear();

	list<string>::const_iterator	s_iterator = code.begin();

	while (s_iterator != code.end())
	{
		const string	&s = *s_iterator++;

		if (!s.length())
			multi_line_edit->append("\n");
		else
			multi_line_edit->append(s.c_str());
	}

	multi_line_edit->setCursorPosition(0, 0);

	connect(multi_line_edit,
			SIGNAL(textChanged()),
			this,
			SLOT(CodeTextChanged()));
}

void	CClassCodeVBoxWidget::PostCurrentCodeChanges()
{
	list<string>	lexical_include_code;
	list<string>	lexical_class_code;
	list<string>	lexical_source_code;
	list<string>	lexical_constructor_code;
	list<string>	lexical_destructor_code;
	list<string>	parser_include_code;
	list<string>	parser_class_code;
	list<string>	parser_source_code;
	list<string>	parser_constructor_code;
	list<string>	parser_destructor_code;
	list<string>	stack_code;

	switch (class_code_section)
	{
		case 0:
		case 1:
		case 2:
		case 3:
		case 4:
		{
			compiler_data->GetLexicalCodeSections(	lexical_include_code,
													lexical_class_code,
													lexical_source_code,
													lexical_constructor_code,
													lexical_destructor_code);

			switch (class_code_section)
			{
				case 0:
				GetMultiLineEditText(lexical_include_code);
				break;

				case 1:
				GetMultiLineEditText(lexical_class_code);
				break;

				case 2:
				GetMultiLineEditText(lexical_source_code);
				break;

				case 3:
				GetMultiLineEditText(lexical_constructor_code);
				break;

				case 4:
				GetMultiLineEditText(lexical_destructor_code);
				break;
			}

			compiler_data->StoreLexicalCodeSections(lexical_include_code,
													lexical_class_code,
													lexical_source_code,
													lexical_constructor_code,
													lexical_destructor_code);
		}
		break;

		case 5:
		case 6:
		case 7:
		case 8:
		case 9:
		{
			compiler_data->GetParserCodeSections(	parser_include_code,
													parser_class_code,
													parser_source_code,
													parser_constructor_code,
													parser_destructor_code);

			switch (class_code_section)
			{
				case 5:
				GetMultiLineEditText(parser_include_code);
				break;

				case 6:
				GetMultiLineEditText(parser_class_code);
				break;

				case 7:
				GetMultiLineEditText(parser_source_code);
				break;

				case 8:
				GetMultiLineEditText(parser_constructor_code);
				break;

				case 9:
				GetMultiLineEditText(parser_destructor_code);
				break;
			}

			compiler_data->StoreParserCodeSections(	parser_include_code,
													parser_class_code,
													parser_source_code,
													parser_constructor_code,
													parser_destructor_code);
		}
		break;

		case 10:
		GetMultiLineEditText(stack_code);
		compiler_data->StoreStackCode(stack_code);
		break;
	}
}

void	CClassCodeVBoxWidget::GetMultiLineEditText(list<string> &code)
{
	code.clear();

	int		count = multi_line_edit->numLines();
	int		i;

	for (i=0; i<count; i++)
		code.push_back(
			(const char *)(multi_line_edit->textLine(i).local8Bit()));

	if (code.back() == string(""))
		code.pop_back();

	list<string>::const_iterator	c_iterator = code.begin();

	while (c_iterator != code.end())
	{
		if ((*c_iterator).find_first_not_of("\t\n ") != string::npos)
			break;

		c_iterator++;
	}

	if (c_iterator == code.end())
		code.clear();
}

void	CClassCodeVBoxWidget::SetTabStopWidth(int count)
{
	QFontMetrics	fm(font());

	int		w = fm.width(QChar('Z'));

	multi_line_edit->setTabStopWidth(count * w);
}

void	CClassCodeVBoxWidget::SelectCodeSection(int section)
{
	PostCurrentCodeChanges();

	class_code_section = section;

	UpdateMultiLineEdit();
}

void	CClassCodeVBoxWidget::ClearCodeSection()
{
	disconnect(	multi_line_edit,
				SIGNAL(textChanged()),
				this,
				SLOT(CodeTextChanged()));

	multi_line_edit->clear();

	if (class_code_section == 10)
	{
		compiler_data->InitializeStackCode();

		SetMultiLineEditText(compiler_data->GetStackCode());
	}

	connect(multi_line_edit,
			SIGNAL(textChanged()),
			this,
			SLOT(CodeTextChanged()));
}

void	CClassCodeVBoxWidget::RestoreUndoData()
{
	compiler_data = compiler_interface->GetCompilerData();
}

void	FindMultiLineEditString(Q3MultiLineEdit *multi_line_edit);

void	CClassCodeVBoxWidget::FindString()
{
	FindMultiLineEditString(multi_line_edit);
}

void	CClassCodeVBoxWidget::UndoEdit()
{
	multi_line_edit->undo();
}

void	CClassCodeVBoxWidget::RedoEdit()
{
	multi_line_edit->redo();
}

void	CClassCodeVBoxWidget::CodeTextChanged()
{
	compiler_interface->SetDataModifiedFlag(true);
}

CClassCodeView::CClassCodeView(	QWidget *p,
								CViewNode *n,
								CInterfaceControl *ic,
								SMainViewControl *mvc,
								const string &dfs,
								CPaletteData *dpd,
								COptionData::SSubViewData *d)
	: CSubView(p, n, ic, mvc, dfs, dpd, d)
{
	frame = new CClassCodeVBoxWidget(
		p, dynamic_cast<CCompilerInterface *>(ic), mvc, d, this);

	QFont		view_node_font;
	QPalette	view_node_palette;

	if (SetupViewNodeFont(view_node_font))
		frame->setFont(view_node_font);
	else
		frame->setFont(frame->font());

	if (SetupViewNodePalette(view_node_palette))
		frame->setPalette(view_node_palette);
	else
		frame->setPalette(frame->palette());
}

CClassCodeView::~CClassCodeView()
{
	delete frame;
}

void	CClassCodeView::SetHighlightPalette(CPaletteData *pd)
{
	restore_palette = frame->palette();

	QPalette	highlight_palette;

	pd->SetupPalette(highlight_palette);

	frame->setPalette(highlight_palette);
}

void	CClassCodeView::RestorePalette()
{
	frame->setPalette(restore_palette);
}

void	CClassCodeView::InitializeModulePointers()
{
	frame->InitializeModulePointers();
}

void	CClassCodeView::UpdateView()
{
	frame->UpdateView();
}

void	CClassCodeView::SetTabStopWidth(int count)
{
	frame->SetTabStopWidth(count);
}

void	CClassCodeView::ClearCodeSection()
{
	frame->ClearCodeSection();
}

void	CClassCodeView::PostCurrentCodeChanges()
{
	frame->PostCurrentCodeChanges();
}

void	CClassCodeView::RestoreUndoData()
{
	frame->RestoreUndoData();
}

void	CClassCodeView::FindString()
{
	frame->FindString();
}

void	CClassCodeView::UndoEdit()
{
	frame->UndoEdit();
}

void	CClassCodeView::RedoEdit()
{
	frame->RedoEdit();
}
