/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "compilerCalculate.h"

bool	CCompilerCalculate::CalculateLexical()
{
	bool									calculate;
	CCompilerData::FiniteStateMachineType	type;
	int										range;

	calculate	= compiler_data->GetLexicalAutomatonCalculate();
	type		= compiler_data->GetLexicalFiniteStateMachineType();
	range		= compiler_data->GetLexicalInputRange();

	delete automaton;

	automaton = 0;

	if (!calculate)
	{
		lexical_calculated = false;

		return true;
	}

	VListPtrChar									start_context;
	StringToken::TokenStatus						token_status;
	vector<StringToken::CTokenValue *>				token_vector;
	vector<StringToken::CTokenValue *>::size_type	v_index;

	token_status = StringToken::GetCharTokenSequence(
		compiler_data->GetStartContext(), string("\t ,"), token_vector);

	if (token_status == StringToken::NoError)
	{
		for (v_index=0; v_index<token_vector.size(); v_index++)
		{
			const string	&s = token_vector[v_index]->GetTokenString();

			char	*s_char = new char[s.length() + 1];

			strcpy(s_char, s.c_str());

			start_context.PushBack(s_char);
		}
	}

	StringToken::DeleteTokenVectorElements(token_vector);

	const list<CTerminal *>		&t_list = compiler_data->GetTerminalList();

	CTerminalInformation	*terminals =
								new CTerminalInformation[t_list.size()];

	list<CTerminal *>::const_iterator	t_iterator = t_list.begin();

	int		i = 0;

	while (t_iterator != t_list.end())
	{
		CTerminal		*t = *t_iterator;

		terminals[i].Initialize(t->GetSymbolIndex(),
								t->GetFilterFlag(),
								t->GetLiteralFlag(),
								t->GetName().c_str(),
								t->GetRegexp().c_str());

		i++;
		t_iterator++;
	}

	CLexicalDriver::LexicalScanType		l_type;

	switch (type)
	{
		case CCompilerData::RegularExpression:
		l_type = CLexicalDriver::RegexpLexical;
		break;

		case CCompilerData::SingleCharacter:
		l_type = CLexicalDriver::SingleCharLexical;
		break;
	}

	automaton = new CAutomaton(
					l_type,
					range,
					t_list.size(),
					terminals,
					compiler_data->GetEndOfFileSymbol().GetSymbolIndex(),
					start_context);

	return lexical_calculated = automaton->CalculateLexical();
}

bool	CCompilerCalculate::CalculateParser()
{
	bool						calculate;
	CCompilerData::ParserType	type;

	calculate	= compiler_data->GetParserAutomatonCalculate();
	type		= compiler_data->GetParserType();

	error_entries_numbered				= false;
	error_entries_calculated			= false;
	unit_rule_elimination_calculated	= false;

	delete parser;

	parser = 0;

	undefined_symbol_rules		= false;
	undefined_entry_precedences	= false;

	if (!calculate)
	{
		parser_calculated = false;

		return true;
	}

	undefined_symbol_rules		= compiler_data->GetUndefinedSymbolRules();
	undefined_entry_precedences	= compiler_data->GetUndefinedEntryPrecedences();

	if (undefined_symbol_rules || undefined_entry_precedences)
	{
		parser_calculated = false;

		return false;
	}

	const list<CTerminal *>		&t_list = compiler_data->GetTerminalList();
	const list<CErrorToken *>	&e_list = compiler_data->GetErrorTokenList();
	const list<CNonTerminal *>	&n_list = compiler_data->GetNonTerminalList();
	const list<CRule *>			&r_list = compiler_data->GetRuleList();
	const list<CPrecedence *>	&p_list = compiler_data->GetPrecedenceList();

	const CTerminal			&end_of_file_symbol =
									compiler_data->GetEndOfFileSymbol();
	const CNonTerminal		*start_symbol =
									compiler_data->GetStartSymbol();
	const CNonTerminal		&augmented_start_symbol =
									compiler_data->GetAugmentedStartSymbol();
	const CRule				&augmented_start_rule =
									compiler_data->GetAugmentedStartRule();

	int		token_count = 0;

	list<CTerminal *>::const_iterator	t_iterator = t_list.begin();

	while (t_iterator != t_list.end())
		if (!(*t_iterator++)->GetFilterFlag())
			token_count++;

	token_count += e_list.size();

	int		token_count_e		= token_count + 1;
	int		nonterminal_count	= n_list.size();
	int		nonterminal_count_a	= nonterminal_count + 1;
	int		symbol_count		= token_count + nonterminal_count;
	int		symbol_count_a		= token_count_e + nonterminal_count_a;
	int		rule_count			= r_list.size();
	int		rule_count_a		= rule_count + 1;

	CSymbolInformation	*symbols	= new CSymbolInformation[symbol_count_a];
	CRuleInformation	*rules		= new CRuleInformation[rule_count_a];

	int		empty_symbol_index		= end_of_file_symbol.GetSymbolIndex();
	int		start_symbol_index		= start_symbol->GetSymbolIndex();
	int		augmented_symbol_index	= augmented_start_symbol.GetSymbolIndex();
	int		augmented_rule_index	= augmented_start_rule.GetIndex();

	int		i = 0;

	t_iterator = t_list.begin();

	while (t_iterator != t_list.end())
	{
		CTerminal		*t = *t_iterator;

		if (!t->GetFilterFlag())
		{
			symbols[i].Initialize(	t->GetName().c_str(),
									false, 0, 0,
									t->GetRegexp().c_str(),
									t->GetLiteralFlag(), false);

			i++;
		}

		t_iterator++;
	}

	list<CErrorToken *>::const_iterator		e_iterator = e_list.begin();

	while (e_iterator != e_list.end())
	{
		CErrorToken		*e = *e_iterator;

		symbols[i].Initialize(	e->GetName().c_str(),
								false, 0, 0, "", true, true);

		i++;
		e_iterator++;
	}

	symbols[i].Initialize("EOF", false, 0, 0, "", true, false);

	i = 0;

	map< int, list<int> >	left_symbol_map;

	list<CRule *>::const_iterator			r_iterator = r_list.begin();

	while (r_iterator != r_list.end())
	{
		CRule	*r = *r_iterator;

		const CSymbol	*left = r->GetLeft();

		int		left_symbol_index = left->GetSymbolIndex();

		left_symbol_map[left_symbol_index].push_back(r->GetIndex());

		const vector<const CSymbol *>	&right = r->GetRight();

		int		right_symbol_count		= right.size();
		int     *right_symbol_indexes	= new int[right_symbol_count];

		vector<const CSymbol *>::size_type		j;

		for (j=0; j<right_symbol_count; j++)
			right_symbol_indexes[j] = right[j]->GetSymbolIndex();

		rules[i].Initialize(left_symbol_index,
							right_symbol_count,
							right_symbol_indexes);

		delete [] right_symbol_indexes;

		i++;
		r_iterator++;
	}

	list<CNonTerminal *>::const_iterator	n_iterator = n_list.begin();

	if (n_iterator != n_list.end())
		i = (*n_iterator)->GetSymbolIndex();

	while (n_iterator != n_list.end())
	{
		list<int>	&left_symbol_list = left_symbol_map[i];

		int		left_symbol_count		= left_symbol_list.size();
		int		*left_symbol_indexes	= new int[left_symbol_count];
		int		j						= 0;

		list<int>::iterator		l_iterator = left_symbol_list.begin();

		while (l_iterator != left_symbol_list.end())
			left_symbol_indexes[j++] = *l_iterator++;

		symbols[i].Initialize(  (*n_iterator)->GetName().c_str(),
								true,
								left_symbol_count,
								left_symbol_indexes,
								0, false, false);

		delete [] left_symbol_indexes;

		i++;
		n_iterator++;
	}

	int		augmented_left_symbol_indexes[1];

	augmented_left_symbol_indexes[0] = augmented_rule_index;

	symbols[augmented_symbol_index].Initialize(	"ASTART",
												true,
												1,
												augmented_left_symbol_indexes,
												0, false, false);

	int		augmented_right_symbol_indexes[2];

	augmented_right_symbol_indexes[0] = start_symbol_index;
	augmented_right_symbol_indexes[1] = empty_symbol_index;

	rules[augmented_rule_index].Initialize( augmented_symbol_index,
											2,
											augmented_right_symbol_indexes);

	int     *token_indexes			= new int[token_count_e];
	int     *nonterminal_indexes	= new int[nonterminal_count_a];

	for (i=0; i<token_count_e; i++)
		token_indexes[i] = i;

	for (i=0; i<nonterminal_count_a; i++)
		nonterminal_indexes[i] = token_count_e + i;

	int     precedence_count = p_list.size();

	CPrecedenceInformation	*precedences =
								new CPrecedenceInformation[precedence_count];

	i = 0;

	list<CPrecedence *>::const_iterator		p_iterator = p_list.begin();

	while (p_iterator != p_list.end())
	{
		CPrecedence		*p = *p_iterator;

		CPrecedence::Associativity		pa      	= p->GetAssociativity();
		const vector<SPrecedenceEntry>	&entries	= p->GetEntries();

		int		a;
		int		ec		= entries.size();
		int		*type	= new int[ec];
		int		*index	= new int[ec];

		switch (pa)
		{
			case CPrecedence::LeftAssociative:  a = 0; break;
			case CPrecedence::RightAssociative: a = 1; break;
			case CPrecedence::NonAssociative:   a = 2; break;
		}

		for (int j=0; j<ec; j++)
		{
			const SPrecedenceEntry  &pe = entries[j];

			switch (pe.type)
			{
				case SPrecedenceEntry::TerminalEntry:
				type[j]		= 0;
				index[j]	= pe.terminal->GetSymbolIndex();
				break;

				case SPrecedenceEntry::ErrorTokenEntry:
				type[j]		= 0;
				index[j]	= pe.error_token->GetSymbolIndex();
				break;

				case SPrecedenceEntry::RuleEntry:
				type[j]		= 1;
				index[j]	= pe.rule->GetIndex();
				break;
			}
		}

		precedences[i].Initialize(a, ec, type, index);

		delete [] type;
		delete [] index;

		i++;
		p_iterator++;
	}

	parser = new CParse(symbol_count,
						symbol_count_a,
						symbols,
						rule_count,
						rule_count_a,
						rules,
						empty_symbol_index,
						augmented_symbol_index,
						start_symbol_index,
						augmented_rule_index,
						token_count,
						token_count_e,
						token_indexes,
						nonterminal_count,
						nonterminal_count_a,
						nonterminal_indexes,
						precedence_count,
						precedences);

	CParse::ParserType		p_type;

	switch (type)
	{
		case CCompilerData::LR_1: p_type = CParse::LR_1;
		break;

		case CCompilerData::LALR_1: p_type = CParse::LALR_1;
		break;

		case CCompilerData::SLR_1: p_type = CParse::SLR_1;
		break;
	}

	return parser_calculated = parser->CalculateParser(p_type);
}

void	CCompilerCalculate::CalculateErrorEntries()
{
	CCompilerData::ErrorCalculation				c;
	CCompilerData::ErrorFunctionNumbering		n;

	c = compiler_data->GetParserErrorCalculation();
	n = compiler_data->GetParserErrorFunctionNumbering();

	error_entries_numbered				= true;
	unit_rule_elimination_calculated	= false;

	CParse::ErrorCalculation		calculation;
	CParse::ErrorFunctionNumbering	numbering;

	switch (c)
	{
		case CCompilerData::NoCalculation:
		error_entries_calculated	= false;
		calculation					= CParse::NoCalculation;
		break;

		case CCompilerData::ApproximateEssential:
		error_entries_calculated	= true;
		calculation					= CParse::ApproximateEssential;
		break;

		case CCompilerData::ExactEssential:
		error_entries_calculated	= true;
		calculation					= CParse::ExactEssential;
		break;
	}

	switch (n)
	{
		case CCompilerData::OneNumber:
		numbering = CParse::OneNumber;
		break;

		case CCompilerData::NumberPerState:
		numbering = CParse::NumberPerState;
		break;

		case CCompilerData::NumberPerEntry:
		numbering = CParse::NumberPerEntry;
		break;
	}

	parser->CalculateEssentialErrorEntries(calculation, numbering);
}

void	CCompilerCalculate::CalculateUnitRuleElimination()
{
	bool	calculate;

	calculate = compiler_data->GetParserOptimizeUnitRuleReduction();

	unit_rule_elimination_calculated = calculate;

	if (calculate)
	{
		const list<int>		&order_list = compiler_data->GetUnitRuleOrderList();

		int		unit_rule_order_count = order_list.size();
		int		unit_rule_order_list[unit_rule_order_count];

		list<int>::const_iterator		ol_iterator = order_list.begin();
		int								index		= 0;

		while (ol_iterator != order_list.end())
			unit_rule_order_list[index++] = *ol_iterator++;

		parser->CalculateUnitRuleElimination(	unit_rule_order_count,
												unit_rule_order_list);
	}
}

string	&CCompilerCalculate::MachineStatusString(string &s)
{
	ostrstream	os;

	os << "Lexical Calculated       = ";

	if (lexical_calculated)
		os << "true"; else os << "false";

	os << endl << "Parser Calculated        = ";

	if (parser_calculated)
		os << "true"; else os << "false";

	os << endl << "Error Entries Calculated = ";

	if (error_entries_calculated)
		os << "true"; else os << "false";

	os << endl << "Unit Rule Elimination    = ";

	if (unit_rule_elimination_calculated)
		os << "true"; else os << "false";

	os << endl << '\0';

	return s = os.str();
}

string	&CCompilerCalculate::LexicalStatusString(string &s)
{
	ostrstream	os;

	if (lexical_calculated)
	{
		os << "Calculate lexical successful" << endl;
		os << "Finite state machine count = " << automaton->fsm_count << endl;
	}
	else
	{
		if (automaton)
		{
			os << "Lexical calculation error" << endl << endl;

			for (int i=0; i<automaton->regexp_count; i++)
			{
				if (!automaton->context_array[i])
				{
					os << "regular expression " << i << ") ";
					os << automaton->terminals[i].name << endl;
					os << automaton->terminals[i].regexp << endl;
					os << string(automaton->current_char_number[i], '-');
					os << '^' << endl << endl;
				}
			}
		}
		else
			os << "No lexical calculation" << endl;
	}

	os << '\0';

	return s = os.str();
}

string	&CCompilerCalculate::ParserStatusString(string &s)
{
	ostrstream	os;

	if (parser_calculated)
	{
		os << "Calculate parser successful" << endl;

		if (compiler_data->GetParserType() == CCompilerData::LR_1)
			os << "LR(1) machine state count = ";
		else
			os << "LR(0) machine state count = ";
		
		os << parser->machine_count << endl;
	}
	else
	{
		int		i;

		if (parser)
		{
			os << "Parser grammar not in reduced form" << endl << endl;

			if (!parser->derives_terminal_flag)
			{
				os	<< "Nonterminals that do not derive terminal strings"
					<< endl;

				for (i=0; i<parser->symbol_count_a; i++)
				{
					if (!parser->symbol_status[i].derives_terminal_flag)
					{
						os << "symbol index " << i << ") ";
						os << parser->symbols[i].name << endl;
					}
				}

				os << endl;
			}

			if (!parser->accessible_flag)
			{
				os	<< "Symbols that are not accessible from the start symbol"
					<< endl;

				for (i=0; i<parser->symbol_count_a; i++)
				{
					if (!parser->symbol_status[i].accessible_flag)
					{
						os << "symbol index " << i << ") ";
						os << parser->symbols[i].name << endl;
					}
				}

				os << endl;
			}
		}
		else
		{
			if (!undefined_symbol_rules && !undefined_entry_precedences)
			{
				os << "No parser calculation" << endl;
			}
			else
			{
				if (undefined_symbol_rules)
				{
					os << "Undefined symbols in rules" << endl;
					os << "Rules:";

					const list<int>		&u_list =
						compiler_data->GetUndefinedSymbolRulesList();

					list<int>::const_iterator	u_iterator = u_list.begin();

					while (u_iterator != u_list.end())
						os << " " << *u_iterator++;

					os << endl << endl;
				}

				if (undefined_entry_precedences)
				{
					os << "Undefined entries in precedences" << endl;
					os << "Precedences:";

					const list<int>		&u_list =
						compiler_data->GetUndefinedEntryPrecedencesList();

					list<int>::const_iterator	u_iterator = u_list.begin();

					while (u_iterator != u_list.end())
						os << " " << *u_iterator++;

					os << endl << endl;
				}
			}
		}
	}

	os << '\0';

	return s = os.str();
}

string	&CCompilerCalculate::ConflictSectionStatusString(string &s)
{
	ostrstream	os;

	os	<< "Conflict entry count = "
		<< parser->parsing_conflicts->Size() << endl;

	if (conflict_sections_valid)
		os << "Conflict sections valid" << endl;
	else
		os << "Conflict sections invalid" << endl;

	os << '\0';

	return s = os.str();
}

string	&CCompilerCalculate::ErrorEntriesStatusString(string &s)
{
	ostrstream	os;

	if (error_entries_numbered)
	{
		CCompilerData::ErrorCalculation			c;
		CCompilerData::ErrorFunctionNumbering	n;
		CCompilerData::ErrorFunctionAggregation	a;

		c = compiler_data->GetParserErrorCalculation();
		n = compiler_data->GetParserErrorFunctionNumbering();
		a = compiler_data->GetParserErrorFunctionAggregation();

		os << "Error calculation    - ";

		switch (c)
		{
			case CCompilerData::NoCalculation:
			os << "no calculation";
			break;

			case CCompilerData::ApproximateEssential:
			os << "approximate essential";
			break;

			case CCompilerData::ExactEssential:
			os << "exact essential";
			break;
		}

		os << endl;
		os << "Entry numbering      - ";

		switch (n)
		{
			case CCompilerData::OneNumber:
			os << "one number";
			break;

			case CCompilerData::NumberPerState:
			os << "number per state";
			break;

			case CCompilerData::NumberPerEntry:
			os << "number per entry";
			break;
		}

		os << endl;
		os << "Function aggregation - ";

		switch (a)
		{
			case CCompilerData::OneFunction:
			os << "one function";
			break;

			case CCompilerData::FunctionPerState:
			os << "function per state";
			break;

			case CCompilerData::FunctionPerEntry:
			os << "function per entry";
			break;
		}

		os	<< endl;
	}

	os	<< "Default error entry count   = "
		<< parser->default_error_entry_count << endl;

	if (error_entries_calculated)
	{
		os	<< "Essential error entry count = "
			<< parser->essential_error_entry_count << endl;
	}

	os << '\0';

	return s = os.str();
}

string	&CCompilerCalculate::ErrorCodeSectionStatusString(string &s)
{
	ostrstream	os;

	if (error_code_sections_valid)
		os << "Error code sections valid" << endl;
	else
		os << "Error code sections invalid" << endl;

	os << '\0';

	return s = os.str();
}

string	&CCompilerCalculate::UnitRuleEliminationStatusString(string &s)
{
	ostrstream	os;

	os	<< "Default machine state count   = "
		<< parser->machine_count << endl;

	if (unit_rule_elimination_calculated)
	{
		os	<< "Unit rule machine state count = "
			<< parser->unit_machine_count << endl;
	}

	os << '\0';

	return s = os.str();
}

bool	CCompilerCalculate::ValidateConflictSections()
{
	const map<int, SConflictResolution *>
		&c_map = compiler_data->GetConflictResolutionMap();

	map<int, SConflictResolution *>::const_iterator
		c_iterator = c_map.begin();

	while (c_iterator != c_map.end())
	{
		SConflictResolution		*cr = (*c_iterator).second;

		if (!parser->ValidateConflict(	cr->index,
										cr->state,
										cr->symbol,
										cr->action))
			break;

		c_iterator++;
	}

	if (c_iterator == c_map.end())
		conflict_sections_valid = true;
	else
		conflict_sections_valid = false;

	return conflict_sections_valid;
}

void	CCompilerCalculate::ModifyConflictActions()
{
	const map<int, SConflictResolution *>
		&c_map = compiler_data->GetConflictResolutionMap();

	map<int, SConflictResolution *>::const_iterator
		c_iterator = c_map.begin();

	while (c_iterator != c_map.end())
	{
		SConflictResolution		*cr = (*c_iterator++).second;

		parser->ModifyActionTable(	cr->index,
									cr->state,
									cr->symbol,
									cr->action);
	}
}

void	CCompilerCalculate::ClearConflictSections()
{
	compiler_data->ClearConflictResolution();
}

bool	CCompilerCalculate::ValidateErrorCodeSections()
{
	CParse	&p = *parser;

	int		number_of_error_functions;

	CCompilerData::ErrorCalculation
		c = compiler_data->GetParserErrorCalculation();

	CCompilerData::ErrorFunctionAggregation
		aggregation = compiler_data->GetParserErrorFunctionAggregation();
					
	if (c == CCompilerData::NoCalculation)
	{
		if (aggregation == CCompilerData::OneFunction)
		{
			number_of_error_functions = 1;
		}
		else if (aggregation == CCompilerData::FunctionPerState)
		{
			number_of_error_functions = p.machine_count;
		}
		else if (aggregation == CCompilerData::FunctionPerEntry)
		{
			number_of_error_functions = p.default_error_entry_count;
		}
	}
	else if (	c == CCompilerData::ApproximateEssential ||
				c == CCompilerData::ExactEssential)
	{
		if (aggregation == CCompilerData::OneFunction)
		{
			number_of_error_functions = 2;
		}
		else if (aggregation == CCompilerData::FunctionPerState)
		{
			number_of_error_functions = p.machine_count + 1;
		}
		else if (aggregation == CCompilerData::FunctionPerEntry)
		{
			number_of_error_functions = p.essential_error_entry_count + 1;
		}
	}

	const map<int, SErrorCode *>
		&ec_map = compiler_data->GetErrorCodeMap();

	map<int, SErrorCode *>::const_iterator
		ec_iterator = ec_map.begin();

	while (ec_iterator != ec_map.end())
	{
		SErrorCode		*ec = (*ec_iterator).second;

		if (ec->entry < 1 || ec->entry > number_of_error_functions)
			break;

		if (c == CCompilerData::NoCalculation)
		{
			if (aggregation == CCompilerData::OneFunction)
			{
				if (ec->state != -1)
					break;
			}
			else if (aggregation == CCompilerData::FunctionPerState)
			{
				if (ec->state != ec->entry - 1 || ec->symbol != -1)
					break;
			}
			else if (aggregation == CCompilerData::FunctionPerEntry)
			{
				if (ec->state !=
						p.default_error_state_terminals[ec->entry].state ||
					ec->symbol !=
						p.default_error_state_terminals[ec->entry].terminal)
					break;
			}
		}
		else if (	c == CCompilerData::ApproximateEssential ||
					c == CCompilerData::ExactEssential)
		{
			if (aggregation == CCompilerData::OneFunction)
			{
				if (ec->entry == 1)
				{
					if (ec->state != -1)
						break;
				}
				else
				{
					if (ec->state != -2)
						break;
				}
			}
			else if (aggregation == CCompilerData::FunctionPerState)
			{
				if (ec->entry == 1)
				{
					if (ec->state != -1)
						break;
				}
				else
				{
					if (ec->state != ec->entry - 2 || ec->symbol != -1)
						break;
				}
			}
			else if (aggregation == CCompilerData::FunctionPerEntry)
			{
				if (ec->entry == 1)
				{
					if (ec->state != -1)
						break;
				}
				else
				{
					if (ec->state !=
						p.essential_error_state_terminals[ec->entry].state ||
						ec->symbol !=
						p.essential_error_state_terminals[ec->entry].terminal)
						break;
				}
			}
		}

		ec_iterator++;
	}

	if (ec_iterator == ec_map.end())
		error_code_sections_valid = true;
	else
		error_code_sections_valid = false;

	return error_code_sections_valid;
}

void	CCompilerCalculate::ClearErrorCodeSections()
{
	compiler_data->ClearErrorCode();
}

int		IntegerLength(int i);

string		&CCompilerCalculate::LR1MachineStateString(int state, string &s)
{
	ListLR1MS::listptr	lr1ms_listptr =
							parser->lr1_machine_states->FindIndex(state);

	if (!lr1ms_listptr)
		return s = "";

	ostrstream	os;
	int			index_width;
	int			i;
	int			j;

	CLR1MachineState	*ms = parser->lr1_machine_states->Data(lr1ms_listptr);
	VListInt			*is = ms->item_set;
	VListOrdPairInt		*st = ms->state_transitions;

	index_width = IntegerLength(is->Size() - 1);

	os << "item set count = " << is->Size() << endl;

	VListInt::listptr	int_listptr = is->FirstListPtr();

	i = 0;
	while (int_listptr)
	{
		os << setw(index_width) << i++ << "  [ ";

		int		lr1_item_index = is->NextData(int_listptr);

		CLR1Item			&li = parser->lr1_items[lr1_item_index];
		CItemCore			&ic = parser->item_cores[li.item_core_index];
		CRuleInformation	&ri = parser->rules[ic.rule_index];

		string	pad1(compiler_data->GetRuleLeftNameWidth() -
					strlen(parser->symbols[ri.left_symbol_index].name), ' ');

		os << parser->symbols[ri.left_symbol_index].name;
		os << pad1 << "  ->  ";

		for (j=0; j<ic.right_symbol_after_dot_index; j++)
			os << parser->symbols[ri.right_symbol_indexes[j]].name << " ";

		os << "(.) ";

		for (; j<ri.right_symbol_count; j++)
			os << parser->symbols[ri.right_symbol_indexes[j]].name << " ";

		os << ", ";

		if (li.lookahead_terminal_index != -1)
			os << parser->symbols[li.lookahead_terminal_index].name << " ";

		os << "]" << endl;
	}

	index_width = IntegerLength(st->Size() - 1);

	os << endl << "goto transition count = " << st->Size() << endl;

	VListOrdPairInt::listptr	pair_listptr = st->FirstListPtr();

	i = 0;
	while (pair_listptr)
	{
		os << setw(index_width) << i++;

		OrdPairInt	pair = st->NextData(pair_listptr);

		string	pad1(compiler_data->GetSymbolNameWidth() -
						strlen(parser->symbols[pair.First()].name), ' ');

		if (parser->symbols[pair.First()].nonterminal_flag)
			os << " REDUCE ";
		else
			os << "  SHIFT ";

		os << parser->symbols[pair.First()].name;
		os << pad1 << " GOTO " << pair.Second() << endl;
	}

	os << '\0';

	return s = os.str();
}

string		&CCompilerCalculate::LR0MachineStateString(int state, string &s)
{
	ListLR0MS::listptr	lr0ms_listptr =
							parser->lr0_machine_states->FindIndex(state);

	if (!lr0ms_listptr)
		return s = "";

	ostrstream	os;
	int			index_width;
	int			i;
	int			j;

	CLR0MachineState	*ms = parser->lr0_machine_states->Data(lr0ms_listptr);
	VListInt			*is = ms->item_set;
	VListOrdPairInt		*st = ms->state_transitions;

	index_width = IntegerLength(is->Size() - 1);

	os << "item core set count = " << is->Size() << endl;

	VListInt::listptr	int_listptr = is->FirstListPtr();

	i = 0;
	while (int_listptr)
	{
		os << setw(index_width) << i++ << "  [ ";

		int		item_core_index = is->NextData(int_listptr);

		CItemCore			&ic = parser->item_cores[item_core_index];
		CRuleInformation	&ri = parser->rules[ic.rule_index];

		string	pad1(compiler_data->GetRuleLeftNameWidth() -
					strlen(parser->symbols[ri.left_symbol_index].name), ' ');

		os << parser->symbols[ri.left_symbol_index].name;
		os << pad1 << "  ->  ";

		for (j=0; j<ic.right_symbol_after_dot_index; j++)
			os << parser->symbols[ri.right_symbol_indexes[j]].name << " ";

		os << "(.) ";

		for (; j<ri.right_symbol_count; j++)
			os << parser->symbols[ri.right_symbol_indexes[j]].name << " ";

		os << "]" << endl;
	}

	index_width = IntegerLength(st->Size() - 1);

	os << endl << "goto transition count = " << st->Size() << endl;

	VListOrdPairInt::listptr	pair_listptr = st->FirstListPtr();

	i = 0;
	while (pair_listptr)
	{
		os << setw(index_width) << i++;

		OrdPairInt	pair = st->NextData(pair_listptr);

		string	pad1(compiler_data->GetSymbolNameWidth() -
						strlen(parser->symbols[pair.First()].name), ' ');

		if (parser->symbols[pair.First()].nonterminal_flag)
			os << " REDUCE ";
		else
			os << "  SHIFT ";

		os << parser->symbols[pair.First()].name;
		os << pad1 << " GOTO " << pair.Second() << endl;
	}

	os << '\0';

	return s = os.str();
}

string		&CCompilerCalculate::LALR1LookaheadString(int state, string &s)
{
	ostrstream	os;
	int			index_width;
	int			count;
	int			i;
	int			j;

	VListOrdPairInt		*la = parser->has_lalr_lookahead->pairs;

	VListOrdPairInt::listptr	pair_listptr = la->FirstListPtr();

	count = 0;
	while (pair_listptr)
	{
		OrdPairInt	pair = la->NextData(pair_listptr);

		CContainsItem		&ci = parser->contains_item[pair.First()];

		if (ci.machine_state_index == state)
			count++;
	}

	index_width = IntegerLength(count - 1);

	os << "LALR(1) lookahead count = " << count << endl;

	pair_listptr = la->FirstListPtr();

	i = 0;
	while (pair_listptr)
	{
		OrdPairInt	pair = la->NextData(pair_listptr);

		CContainsItem		&ci = parser->contains_item[pair.First()];

		if (ci.machine_state_index != state)
			continue;

		CItemCore			&ic = parser->item_cores[ci.item_core_index];
		CRuleInformation	&ri = parser->rules[ic.rule_index];

		os << setw(index_width) << i++ << "  [ ";

		string	pad1(compiler_data->GetRuleLeftNameWidth() -
					strlen(parser->symbols[ri.left_symbol_index].name), ' ');

		os << parser->symbols[ri.left_symbol_index].name;
		os << pad1 << "  ->  ";

		for (j=0; j<ic.right_symbol_after_dot_index; j++)
			os << parser->symbols[ri.right_symbol_indexes[j]].name << " ";

		os << "(.) ";

		for (; j<ri.right_symbol_count; j++)
			os << parser->symbols[ri.right_symbol_indexes[j]].name << " ";

		os << "] HAS-LALR-LOOKAHEAD ";
		os << parser->symbols[pair.Second()].name << endl;
	}

	os << '\0';

	return s = os.str();
}

string		&CCompilerCalculate::SLR1LookaheadString(int state, string &s)
{
	ostrstream	os;
	int			index_width;
	int			count;
	int			i;
	int			j;

	ListSLR1LookAhead	*la = parser->slr1_lookahead;

	ListSLR1LookAhead::listptr	la_listptr = la->FirstListPtr();

	count = 0;
	while (la_listptr)
	{
		CSLR1LookAhead	&slr_la = la->NextData(la_listptr);

		if (slr_la.state == state)
			count++;
	}

	index_width = IntegerLength(count - 1);

	os << "SLR(1) lookahead count = " << count << endl;

	la_listptr = la->FirstListPtr();

	i = 0;
	while (la_listptr)
	{
		CSLR1LookAhead	&slr_la = la->NextData(la_listptr);

		if (slr_la.state != state)
			continue;

		CRuleInformation	&ri = parser->rules[slr_la.rule_index];

		os << setw(index_width) << i++ << "  [ ";

		string	pad1(compiler_data->GetRuleLeftNameWidth() -
					strlen(parser->symbols[ri.left_symbol_index].name), ' ');

		os << parser->symbols[ri.left_symbol_index].name;
		os << pad1 << "  ->  ";

		for (j=0; j<ri.right_symbol_count; j++)
			os << parser->symbols[ri.right_symbol_indexes[j]].name << " ";

		os << "(.) ";

		os << "] HAS-SLR-LOOKAHEAD ";
		os << parser->symbols[slr_la.terminal].name << endl;
	}

	os << '\0';

	return s = os.str();
}
