/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#ifndef TLIST_H
#define TLIST_H

template <class T>
class TList
{
	public:
	struct SElement
	{
		SElement	*next;
		SElement	*previous;
		T			*data_ptr;
	};
	
	typedef typename TList<T>::SElement*	listptr;

	TList(int b_size = 100);
	~TList();

	T			*&Data(SElement *element)		{ return element->data_ptr; }
	T			*Data(SElement *element) const	{ return element->data_ptr; }
	int			Size() const					{ return list_count; }
	T			*&FirstData()					{ return first->data_ptr; }
	T			*FirstData() const				{ return first->data_ptr; }
	SElement	*FirstListPtr() const			{ return first; }
	T			*&LastData()					{ return last->data_ptr; }
	T			*LastData() const				{ return last->data_ptr; }
	SElement	*LastListPtr() const			{ return last; }
	int			Empty() const					{ return list_count; }

	SElement	*PushFront(T *data);
	SElement	*PushBack(T *data);
	SElement	*FindData(T *value, SElement *start = 0) const;
	SElement	*FindIndex(int index) const;
	SElement	*NextListPtr(SElement *&element) const;
	SElement	*PreviousListPtr(SElement *&element) const;
	T			*&NextData(SElement *&element);
	T			*NextData(SElement *&element) const;
	T			*&PreviousData(SElement *&element);
	T			*PreviousData(SElement *&element) const;
	SElement	*InsertAfter(SElement *element, T *data);
	SElement	*InsertBefore(SElement *element, T *data);
	void		Clear();
	SElement	*Erase(SElement *element);
	SElement	*PopFront();
	SElement	*PopFront(int count);
	SElement	*PopBack();
	SElement	*PopBack(int count);
	SElement	*SetData(SElement *element, T *data);

	private:

	struct SBlock
	{
		SBlock		*next;
		SElement	*element_array;
	};
	
	typedef typename TList<T>::SBlock*		blockptr;

	int			block_size;
	SBlock		*allocate_block();
	SBlock		*first_block;
	SBlock		*last_block;
	SElement	*allocate_element();
	void		free_element(SElement *element);
	SElement	*free_list;
	int			list_count;
	SElement	*first;
	SElement	*last;

	TList	&operator=(const TList &l);
};

template <class T> inline
typename TList<T>::SBlock		*TList<T>::allocate_block()
{
	SBlock	*block = new SBlock;

	block->next				= 0;
	SElement *e_array		= new SElement[block_size];
	block->element_array	= e_array;

	for (int i=0; i<block_size-1; i++)
	{
		e_array[i].next		= e_array + (i + 1);
		e_array[i].data_ptr	= 0;
	}

	e_array[block_size - 1].next		= 0;
	e_array[block_size - 1].data_ptr	= 0;

	free_list = e_array;

	return block;
}

template <class T> inline
typename TList<T>::SElement		*TList<T>::allocate_element()
{
	if (!free_list)
	{
		SBlock	*block		= allocate_block();
		last_block->next	= block;
		last_block			= block;
	}

	SElement	*element = free_list;

	free_list = free_list->next;

	return element;
}

template <class T> inline
void	TList<T>::free_element(SElement *element)
{
	delete element->data_ptr;

	if (!free_list)
		element->next	= 0;
	else
		element->next	= free_list;

	element->data_ptr	= 0;
	free_list			= element;
}

template <class T>
TList<T>::TList(int b_size)
{
	if (b_size < 2)
		block_size = 2;
	else
		block_size = b_size;

	first_block = allocate_block();
	last_block	= first_block;

	list_count	= 0;
	first		= 0;
	last		= 0;
}

template <class T>
TList<T>::~TList()
{
	Clear();

	SBlock	*current_block = first_block;
	SBlock	*next_block;

	while (current_block)
	{
		next_block = current_block->next;
		delete [] current_block->element_array;
		delete current_block;
		current_block = next_block;
	}
}

template <class T>
typename TList<T>::SElement		*TList<T>::PushFront(T *data)
{
	SElement	*element = allocate_element();

	if (!first)
	{
		first			= element;
		last			= element;

		first->next		= 0;
		first->previous	= 0;
		first->data_ptr	= data;
	}
	else
	{
		element->next		= first;
		element->previous	= 0;
		element->data_ptr	= data;

		first->previous	= element;
		first			= element;
	}

	list_count++;

	return first;
}

template <class T>
typename TList<T>::SElement		*TList<T>::PushBack(T *data)
{
	SElement	*element = allocate_element();

	if (!last)
	{
		first			= element;
		last			= element;

		last->next		= 0;
		last->previous	= 0;
		last->data_ptr	= data;
	}
	else
	{
		element->next		= 0;
		element->previous	= last;
		element->data_ptr	= data;

		last->next		= element;
		last			= element;
	}

	list_count++;

	return last;
}

template <class T>
typename TList<T>::SElement
	*TList<T>::FindData(T *value, SElement *start) const
{
	SElement	*current_element;

	if (!start)
		current_element = first;
	else
		current_element = start;

	while (current_element)
	{
		if (value == current_element->data_ptr)
			return current_element;

		current_element = current_element->next;
	}

	return 0;
}

template <class T>
typename TList<T>::SElement		*TList<T>::FindIndex(int index) const
{
	if (index >= list_count || index < 0)
		return 0;

	SElement	*current_element = first;

	for (int i=0; i<index; i++)
		current_element = current_element->next;

	return current_element;
}

template <class T> inline
typename TList<T>::SElement
	*TList<T>::NextListPtr(SElement *&element) const
{
	if (element == last)
		element = 0;
	else
		element = element->next;
		
	return element;
}

template <class T> inline
typename TList<T>::SElement
	*TList<T>::PreviousListPtr(SElement *&element) const
{
	if (element == first)
		element = 0;
	else
		element = element->previous;
		
	return element;
}

template <class T> inline
T	*&TList<T>::NextData(SElement *&element)
{
	T	*&data = element->data_ptr;

	element = element->next;

	return data;
}

template <class T> inline
T	*TList<T>::NextData(SElement *&element) const
{
	T	*data = element->data_ptr;

	element = element->next;

	return data;
}

template <class T> inline
T	*&TList<T>::PreviousData(SElement *&element)
{
	T	*&data = element->data_ptr;

	element = element->previous;

	return data;
}

template <class T> inline
T	*TList<T>::PreviousData(SElement *&element) const
{
	T	*data = element->data_ptr;

	element = element->previous;

	return data;
}

template <class T>
typename TList<T>::SElement
	*TList<T>::InsertAfter(SElement *element, T *data)
{
	if (!element)
		return PushBack(data);

	SElement	*next_element = element->next;

	if (!next_element)
		return PushBack(data);

	SElement	*insert_element = allocate_element();

	element->next				= insert_element;
	insert_element->next		= next_element;
	insert_element->previous	= element;
	insert_element->data_ptr	= data;
	next_element->previous		= insert_element;

	list_count++;

	return insert_element;
}

template <class T>
typename TList<T>::SElement
	*TList<T>::InsertBefore(SElement *element, T *data)
{
	if (!element)
		return PushFront(data);

	SElement	*previous_element = element->previous;

	if (!previous_element)
		return PushFront(data);

	SElement	*insert_element = allocate_element();

	previous_element->next		= insert_element;
	insert_element->next		= element;
	insert_element->previous	= previous_element;
	insert_element->data_ptr	= data;
	element->previous			= insert_element;

	list_count++;

	return insert_element;
}

template <class T> inline
void	TList<T>::Clear()
{
	PopBack(list_count);
}

template <class T>
typename TList<T>::SElement		*TList<T>::Erase(SElement *element)
{
	SElement	*previous_element = element->previous;

	if (!previous_element)
		return PopFront();

	SElement	*next_element = element->next;

	if (!next_element)
		return PopBack();

	free_element(element);

	previous_element->next	= next_element;
	next_element->previous	= previous_element;

	list_count--;

	return next_element;
}

template <class T>
typename TList<T>::SElement		*TList<T>::PopFront()
{
	if (!first)
		return 0;

	SElement	*next_element = first->next;

	free_element(first);

	if (!next_element)
	{
		first	= 0;
		last	= 0;
	}
	else
	{
		first			= next_element;
		first->previous	= 0;
	}

	list_count--;

	return first;
}

template <class T>
typename TList<T>::SElement		*TList<T>::PopFront(int count)
{
	int		remove_count =	count < list_count ?
							count : list_count;

	if (remove_count <= 0)
		return first;

	SElement	*next_element;

	for (int i=0; i<remove_count; i++)
	{
		next_element = first->next;

		free_element(first);

		first = next_element;
	}

	if (!first)
		last = 0;
	else
		first->previous = 0;

	list_count -= remove_count;

	return first;
}

template <class T>
typename TList<T>::SElement		*TList<T>::PopBack()
{
	if (!last)
		return 0;

	SElement	*previous_element = last->previous;

	free_element(last);

	if (!previous_element)
	{
		first	= 0;
		last	= 0;
	}
	else
	{
		last		= previous_element;
		last->next	= 0;
	}

	list_count--;

	return last;
}

template <class T>
typename TList<T>::SElement		*TList<T>::PopBack(int count)
{
	int		remove_count =	count < list_count ?
							count : list_count;

	if (remove_count <= 0)
		return last;

	SElement	*previous_element;

	for (int i=0; i<remove_count; i++)
	{
		previous_element = last->previous;

		free_element(last);

		last = previous_element;
	}

	if (!last)
		first = 0;
	else
		last->next = 0;

	list_count -= remove_count;

	return last;
}

template <class T> inline
typename TList<T>::SElement
	*TList<T>::SetData(SElement *element, T *data)
{
	delete element->data_ptr;

	element->data_ptr = data;

	return element;
}

#endif
