#ifndef CALCULATE_H
#define CALCULATE_H

#include <string>
#include <list>
#include <map>
#include <functional>
#include <iostream>

using namespace std;

class CExpressionValue
{
	public:
	enum ExpressionType { integer, floating };
	enum OperationType	{ add, subtract, multiply, divide };

	ExpressionType	type;

	union Value
	{
		long			i_value;
		double			f_value;
	} value;

	CExpressionValue() { type = floating; value.f_value = 0.0; }

	CExpressionValue(long i)
	: type(integer) { value.i_value = i; }

	CExpressionValue(double f)
	: type(floating) { value.f_value = f; }

	CExpressionValue(const string &s)
	{
		CExpressionValue	ev;

		if (GetSymbolValue(s, ev))
		{ type = ev.type; value = ev.value; }
		else
		{ type = floating; value.f_value = 0.0; }
	}

	CExpressionValue(const CExpressionValue &ev)
	{ type = ev.type; value = ev.value; }

	static bool		AddSymbolValue(const string &s, const CExpressionValue &ev);
	static bool		SetSymbolValue(const string &s, const CExpressionValue &ev);
	static bool		GetSymbolValue(const string &s, CExpressionValue &ev);
	static bool		FindSymbol(const string &s);

	static map<string, CExpressionValue>	symbol_value_map;

	static CExpressionValue		Add(CExpressionValue a, CExpressionValue b);
	static CExpressionValue		Subtract(CExpressionValue a, CExpressionValue b);
	static CExpressionValue		Multiply(CExpressionValue a, CExpressionValue b);
	static CExpressionValue		Divide(CExpressionValue a, CExpressionValue b);

	static ostream	*output_stream;
};

ostream &operator<<(ostream &os, const CExpressionValue &ev);

class CExpressionNode
{
	public:
	CExpressionNode() { }
	CExpressionNode(long i) : value(i) { }
	CExpressionNode(double f) : value(f) { }
	CExpressionNode(const string &s) : value(s), identifier(s) { }
	virtual ~CExpressionNode() { }

	virtual CExpressionValue	EvaluateExpression() = 0;

	void				SetValue(CExpressionValue ev)	{ value = ev; }
	CExpressionValue	GetValue()						{ return value; }
	bool				HasIdentifier()					{ return !identifier.empty(); }
	void				SetIdentifier(const string &s)	{ identifier = s; }
	string				GetIdentifier()					{ return identifier; }

	protected:
	CExpressionValue	value;
	string				identifier;
};

class CUnaryExpression : public CExpressionNode
{
	public:
	virtual ~CUnaryExpression() { }

	CUnaryExpression(long i) : CExpressionNode(i) { }
	CUnaryExpression(double f) : CExpressionNode(f) { }
	CUnaryExpression(const string &s) : CExpressionNode(s) { }

	virtual CExpressionValue	EvaluateExpression();
};

class CBinaryExpression : public CExpressionNode
{
	public:
	CBinaryExpression(CExpressionNode *a, CExpressionNode *b, CExpressionValue::OperationType ot)
		: sub_expression_a(a), sub_expression_b(b), operation_type(ot) { }

	virtual ~CBinaryExpression()
	{ delete sub_expression_a; delete sub_expression_b; }

	virtual CExpressionValue	EvaluateExpression();

	private:
	CExpressionNode						*sub_expression_a;
	CExpressionNode						*sub_expression_b;
	CExpressionValue::OperationType		operation_type;
};

class CAssignmentExpression
{
	public:
	CAssignmentExpression(const string &s, CExpressionNode *a)
		: identifier(s), expression(a) { }

	virtual ~CAssignmentExpression()
	{ delete expression; }

	void	EvaluateAssignment();

	static list<CAssignmentExpression *>	assignment_list;

	static void		AddAssignment(CAssignmentExpression *a)
	{ assignment_list.push_back(a); }

	static void		EvaluateAssignments();
	static void		DestroyAssignments();

	struct evaluate_assignments
		: public unary_function<CAssignmentExpression *, void>
	{
		void	operator()(CAssignmentExpression *a);
	};

	struct destroy_assignments
		: public unary_function<CAssignmentExpression *, void>
	{
		void	operator()(CAssignmentExpression *a);
	};

	private:
	string				identifier;
	CExpressionNode		*expression;
};

#endif
