/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "compilerCalculate.h"

bool	CCompilerCalculate::GenerateCode(	bool generate_l,
											bool generate_p,
											bool generate_b,
											bool unit_rule_flag,
											string &directory)
{
	const string	&l_prefix	= compiler_data->GetLexicalPrefixName();
	const string	&p_prefix	= compiler_data->GetParserPrefixName();
	const string	&b_prefix	= compiler_data->GetBasePrefixName();
	bool			tty			= compiler_data->GetGenerateTTYCode();
	bool			rm_table	= compiler_data->GetRegexpMatchTable();

	generate_lexical_source_files	= generate_l;
	generate_parser_source_files	= generate_p;
	generate_base_source_files		= generate_b;
	generate_tty_source_files		= tty;

	string::size_type	flno = directory.find_last_not_of('/');

	code_generation_directory_name = string(directory, 0, flno + 1);

	bool	r;
	DIR		*dir = opendir(code_generation_directory_name.c_str());

	if (dir)
	{
		code_generation_directory_accesible = true;

		string	&cgdn = code_generation_directory_name;

		lexical_single_char_data_h_name	= l_prefix + "SingleCharData.h";
		lexical_regexp_data_h_name		= l_prefix + "RegexpData.h";
		lexical_driver_h_name			= l_prefix + "LexicalDriver.h";
		lexical_driver_cpp_name			= l_prefix + "LexicalDriver.cpp";

		parser_data_h_name				= p_prefix + "ParserData.h";
		parser_driver_h_name			= p_prefix + "ParserDriver.h";
		parser_driver_cpp_name			= p_prefix + "ParserDriver.cpp";

		if (b_prefix.length())
		{
			lexical_base_h_name			= b_prefix + "LexicalBase.h";
			lexical_base_cpp_name		= b_prefix + "LexicalBase.cpp";
			parser_base_h_name			= b_prefix + "ParserBase.h";
			parser_base_cpp_name		= b_prefix + "ParserBase.cpp";
			stack_element_h_name		= b_prefix + "Stack.h";
		}
		else
		{
			lexical_base_h_name			= "lexicalDriver.h";
			lexical_base_cpp_name		= "lexicalDriver.cpp";
			parser_base_h_name			= "parserDriver.h";
			parser_base_cpp_name		= "parserDriver.cpp";
			stack_element_h_name		= "stackElement.h";
		}

		tty_main_cpp_name				= "ttyMain.cpp";
		tty_makefile_name				= "Makefile";

		lexical_single_char_data_h_path	= cgdn + '/' +
											lexical_single_char_data_h_name;
		lexical_regexp_data_h_path		= cgdn + '/' +
											lexical_regexp_data_h_name;
		lexical_driver_h_path			= cgdn + '/' +
											lexical_driver_h_name;
		lexical_driver_cpp_path			= cgdn + '/' +
											lexical_driver_cpp_name;
		lexical_base_h_path				= cgdn + '/' +
											lexical_base_h_name;
		lexical_base_cpp_path			= cgdn + '/' +
											lexical_base_cpp_name;
		parser_data_h_path				= cgdn + '/' +
											parser_data_h_name;
		parser_driver_h_path			= cgdn + '/' +
											parser_driver_h_name;
		parser_driver_cpp_path			= cgdn + '/' +
											parser_driver_cpp_name;
		parser_base_h_path				= cgdn + '/' +
											parser_base_h_name;
		parser_base_cpp_path			= cgdn + '/' +
											parser_base_cpp_name;
		stack_element_h_path			= cgdn + '/' +
											stack_element_h_name;
		tty_main_cpp_path				= cgdn + '/' +
											tty_main_cpp_name;
		tty_makefile_path				= cgdn + '/' +
											tty_makefile_name;

		bool	files_exist = false;

		if (generate_lexical_source_files)
		{
			if (compiler_data->GetLexicalFiniteStateMachineType() ==
				CCompilerData::SingleCharacter)
			{
				lexical_single_char_data_h_exists =
					FilePath::FileExists(lexical_single_char_data_h_path);

				files_exist = files_exist || lexical_single_char_data_h_exists;
			}
			else if (compiler_data->GetLexicalFiniteStateMachineType() ==
				CCompilerData::RegularExpression)
			{
				lexical_regexp_data_h_exists =
					FilePath::FileExists(lexical_regexp_data_h_path);
				lexical_driver_h_exists =
					FilePath::FileExists(lexical_driver_h_path);
				lexical_driver_cpp_exists =
					FilePath::FileExists(lexical_driver_cpp_path);

				files_exist = files_exist || lexical_regexp_data_h_exists;
				files_exist = files_exist || lexical_driver_h_exists;
				files_exist = files_exist || lexical_driver_cpp_exists;
			}

			if (generate_base_source_files)
			{
				lexical_base_h_exists =
					FilePath::FileExists(lexical_base_h_path);
				lexical_base_cpp_exists =
					FilePath::FileExists(lexical_base_cpp_path);

				files_exist = files_exist || lexical_base_h_exists;
				files_exist = files_exist || lexical_base_cpp_exists;
			}
		}

		if (generate_parser_source_files)
		{
			parser_data_h_exists =
				FilePath::FileExists(parser_data_h_path);
			parser_driver_h_exists =
				FilePath::FileExists(parser_driver_h_path);
			parser_driver_cpp_exists =
				FilePath::FileExists(parser_driver_cpp_path);

			files_exist = files_exist || parser_data_h_exists;
			files_exist = files_exist || parser_driver_h_exists;
			files_exist = files_exist || parser_driver_cpp_exists;

			if (generate_base_source_files)
			{
				parser_base_h_exists =
					FilePath::FileExists(parser_base_h_path);
				parser_base_cpp_exists =
					FilePath::FileExists(parser_base_cpp_path);

				files_exist = files_exist || parser_base_h_exists;
				files_exist = files_exist || parser_base_cpp_exists;
			}
		}

		if (generate_base_source_files)
		{
			stack_element_h_exists =
				FilePath::FileExists(stack_element_h_path);

			files_exist = files_exist || stack_element_h_exists;
		}

		if (generate_tty_source_files)
		{
			tty_main_cpp_exists =
				FilePath::FileExists(tty_main_cpp_path);
			tty_makefile_exists =
				FilePath::FileExists(tty_makefile_path);

			files_exist = files_exist || tty_main_cpp_exists;
			files_exist = files_exist || tty_makefile_exists;
		}

		if (!files_exist)
		{
			source_files_previously_exist = false;

			if (generate_lexical_source_files)
				GenerateLexicalCode(tty, rm_table);

			if (generate_parser_source_files)
				GenerateParserCode(unit_rule_flag, tty);

			if (generate_base_source_files)
				GenerateBaseAndStackCode();

			if (generate_tty_source_files)
				GenerateTTYCode();

			if (b_prefix.length())
				SubstituteIdentifiers();

			r = true;
		}
		else
		{
			source_files_previously_exist = true;

			r = false;
		}

		closedir(dir);
	}
	else
	{
		r = code_generation_directory_accesible = false;
	}

	return r;
}

string	&CCompilerCalculate::GenerateCodeStatusString(string &s)
{
	ostrstream	os;

	if (code_generation_directory_accesible)
	{
		if (!source_files_previously_exist)
		{
			if (generate_lexical_source_files)
			{
				os << "Lexical files generated:" << endl;

				if (compiler_data->GetLexicalFiniteStateMachineType() ==
					CCompilerData::SingleCharacter)
				{
					os << lexical_single_char_data_h_name << endl;
				}
				else if (compiler_data->GetLexicalFiniteStateMachineType() ==
					CCompilerData::RegularExpression)
				{
					os << lexical_regexp_data_h_name << endl;
					os << lexical_driver_h_name << endl;
					os << lexical_driver_cpp_name << endl;
				}

				if (generate_base_source_files)
				{
					os << lexical_base_h_name << endl;
					os << lexical_base_cpp_name << endl;
				}

				os << endl;
			}

			if (generate_parser_source_files)
			{
				os << "Parser files generated:" << endl;
				os << parser_data_h_name << endl;
				os << parser_driver_h_name << endl;
				os << parser_driver_cpp_name << endl;

				if (generate_base_source_files)
				{
					os << parser_base_h_name << endl;
					os << parser_base_cpp_name << endl;
				}

				os << endl;
			}

			if (generate_base_source_files)
			{
				os << "Stack file generated:" << endl;
				os << stack_element_h_name << endl << endl;
			}

			if (generate_tty_source_files)
			{
				os << "TTY files generated:" << endl;
				os << tty_main_cpp_name << endl;
				os << tty_makefile_name << endl;
			}
		}
		else
		{
			os << "Source files currently exist - no code generation" << endl;

			if (generate_lexical_source_files)
			{
				if (compiler_data->GetLexicalFiniteStateMachineType() ==
					CCompilerData::SingleCharacter)
				{
					if (lexical_single_char_data_h_exists)
						os	<< lexical_single_char_data_h_name
							<< " exists" << endl;
				}
				else if (compiler_data->GetLexicalFiniteStateMachineType() ==
					CCompilerData::RegularExpression)
				{
					if (lexical_regexp_data_h_exists)
						os	<< lexical_regexp_data_h_name
							<< " exists" << endl;

					if (lexical_driver_h_exists)
						os	<< lexical_driver_h_name
							<< " exists" << endl;

					if (lexical_driver_cpp_exists)
						os	<< lexical_driver_cpp_name
							<< " exists" << endl;
				}

				if (generate_base_source_files)
				{
					if (lexical_base_h_exists)
						os	<< lexical_base_h_name
							<< " exists" << endl;

					if (lexical_base_cpp_exists)
						os	<< lexical_base_cpp_name
							<< " exists" << endl;
				}
			}

			if (generate_parser_source_files)
			{
				if (parser_data_h_exists)
					os << parser_data_h_name << " exists" << endl;

				if (parser_driver_h_exists)
					os << parser_driver_h_name << " exists" << endl;

				if (parser_driver_cpp_exists)
					os << parser_driver_cpp_name << " exists" << endl;

				if (generate_base_source_files)
				{
					if (parser_base_h_exists)
						os << parser_base_h_name << " exists" << endl;

					if (parser_base_cpp_exists)
						os << parser_base_cpp_name << " exists" << endl;
				}
			}

			if (generate_base_source_files)
			{
				if (stack_element_h_exists)
					os << stack_element_h_name << " exists" << endl;
			}

			if (generate_tty_source_files)
			{
				if (tty_main_cpp_exists)
					os << tty_main_cpp_name << " exists" << endl;

				if (tty_makefile_exists)
					os << tty_makefile_name << " exists" << endl;
			}
		}
	}
	else
		os << "Code generation directory not accessible" << endl;

	os << '\0';

	return s = os.str();
}

int		IntegerLength(int i);

string	&CreateIncludeGuardIdentifier(const string &s, string &gs)
{
	gs = s;

	string::size_type	i;

	for (i=0; i<gs.length(); i++)
	{
		if (gs[i] == '.')
			gs[i] = '_';
		else
			gs[i] = toupper(gs[i]);
	}

	return gs;
}

void	PrintRow(	ostream &os,
					int *data,
					int count,
					int justify_width,
					bool last)
{
	os << "    ";

	int	i;
	for (i=0; i<count-1; i++)
		os << setw(justify_width) << data[i] << ",";

	os << setw(justify_width) << data[i];

	if (!last)
		os << ",";

	os << endl;
}

void	PrintArrayRow(	ostream &os,
						int *data,
						int count,
						int justify_width,
						bool last)
{
	int		maximum_row_count;
	int		number_of_full_rows;
	int		partial_row_count;
	int		*data_ptr;
	int		i;

	switch (justify_width)
	{
		case 1:  maximum_row_count = 10; break;
		case 2:  maximum_row_count = 10; break;
		case 3:  maximum_row_count = 10; break;
		case 4:  maximum_row_count = 10; break;
		case 5:  maximum_row_count = 10; break;
		case 6:  maximum_row_count = 10; break;
		case 7:  maximum_row_count =  5; break;
		case 8:  maximum_row_count =  5; break;
		case 9:  maximum_row_count =  5; break;
		case 10: maximum_row_count =  5; break;
		case 11: maximum_row_count =  5; break;
		case 12: maximum_row_count =  5; break;
	}

	number_of_full_rows	= count / maximum_row_count;
	partial_row_count	= count % maximum_row_count;
	data_ptr			= data;
	
	if (partial_row_count)
	{
		for (i=0; i<number_of_full_rows; i++)
		{
			PrintRow(os, data_ptr, maximum_row_count, justify_width, false);
			data_ptr += maximum_row_count;
		}

		PrintRow(os, data_ptr, partial_row_count, justify_width, last);
	}
	else
	{
		for (i=0; i<number_of_full_rows-1; i++)
		{
			PrintRow(os, data_ptr, maximum_row_count, justify_width, false);
			data_ptr += maximum_row_count;
		}

		PrintRow(os, data_ptr, maximum_row_count, justify_width, last);
	}
}

string	&EscapeString(const string &s1, string &s2)
{
	string::size_type	i;

	s2 = "";

	for (i=0; i<s1.length(); i++)
	{
		if (s1[i] == '\\' || s1[i] == '\"')
			s2 += '\\';

		s2 += s1[i];
	}

	return s2;
}

void	CCompilerCalculate::GenerateLexicalCode(bool tty, bool rm_table)
{
	CAutomaton	&a = *automaton;

	if (compiler_data->GetLexicalFiniteStateMachineType() ==
		CCompilerData::SingleCharacter)
	{
		int		i;
		string	lexical_single_char_data_h_guard;

		CreateIncludeGuardIdentifier(	lexical_single_char_data_h_name,
										lexical_single_char_data_h_guard);

		ofstream	f(lexical_single_char_data_h_path.c_str());

		f << "#ifndef " << lexical_single_char_data_h_guard << endl;
		f << "#define " << lexical_single_char_data_h_guard << endl;
		f << endl;
		f << "//  token names for CLexicalSingleCharDriver" << endl;
		f << endl;
		f << "const char * const  ";
		f	<< compiler_data->GetLexicalPrefixName();
		f	<< "_char_token_names[] =" << endl;
		f << "{" << endl;

		CTerminalInformation	*terminals		= a.terminals;

		for (i=0; i<a.regexp_count; i++)
		{
			char	*name = terminals[i].name;

			f << "    \"";

			if (!strcmp(name, "\\") || !strcmp(name, "\""))
				f << "\\";

			f << name << "\"," << endl;
		}

		f << "    0" << endl;
		f << "};" << endl;
		f << endl;
		f << "const int    " << compiler_data->GetLexicalPrefixName();
		f	<< "_empty_symbol_token = " << a.empty_symbol_index << ";" << endl;
		f << endl;
		f << "#endif" << endl;

		f.close();
	}
	else if (compiler_data->GetLexicalFiniteStateMachineType() ==
			CCompilerData::RegularExpression)
	{
		int		i;
		string	lexical_regexp_data_h_guard;
		string	lexical_driver_h_guard;
		string	lexical_driver_prefix;
		string	lexical_driver_prefix_u;
		int		justify_width;
		int		*data_ptr;
		bool	last;

		list<string>			include_code;
		list<string>			class_code;
		list<string>			source_code;
		list<string>			constructor_code;
		list<string>			destructor_code;
		list<string>::iterator	s_iterator;

		CreateIncludeGuardIdentifier(	lexical_regexp_data_h_name,
										lexical_regexp_data_h_guard);

		CreateIncludeGuardIdentifier(	lexical_driver_h_name,
										lexical_driver_h_guard);

		lexical_driver_prefix	= compiler_data->GetLexicalPrefixName();
		lexical_driver_prefix_u	= lexical_driver_prefix;

		if (lexical_driver_prefix_u.length())
			lexical_driver_prefix_u[0] = toupper(lexical_driver_prefix_u[0]);

		compiler_data->GetLexicalCodeSections(	include_code,
												class_code,
												source_code,
												constructor_code,
												destructor_code);

		ofstream	f(lexical_regexp_data_h_path.c_str());

		f << "#ifndef " << lexical_regexp_data_h_guard << endl;
		f << "#define " << lexical_regexp_data_h_guard << endl;
		f << endl;
		f << "//  finite state machine data tables for CLexicalRegexpDriver";
		f	<< endl << endl;
		f << "const int    " << lexical_driver_prefix_u;
		f	<< "_Token_Count         = " << a.token_count << ";" << endl;
		f << "const int    " << lexical_driver_prefix_u;
		f	<< "_Filter_Count        = " << a.filter_count << ";" << endl;
		f << "const int    " << lexical_driver_prefix_u;
		f	<< "_Terminal_Count      = " << a.regexp_count << ";" << endl;
		f << "const int    " << lexical_driver_prefix_u;
		f	<< "_Empty_Symbol_Token  = " << a.empty_symbol_index << ";" << endl;
		f << "const int    " << lexical_driver_prefix_u;
		f	<< "_Input_Range         = " << a.input_range << ";" << endl;
		f << "const int    " << lexical_driver_prefix_u;
		f	<< "_FSM_Count           = " << a.fsm_count << ";" << endl;
		f << "const int    " << lexical_driver_prefix_u;
		f	<< "_Single_Column_Match = ";

		if (a.StartAssignmentFlag())
			f << "0;" << endl;
		else
			f << "1;" << endl;

		f << endl;

		f << "const int    " << lexical_driver_prefix_u;
		f	<< "_FSM_Table[" << lexical_driver_prefix_u;
		f	<< "_FSM_Count * " << lexical_driver_prefix_u;
		f	<< "_Input_Range] =" << endl;
		f << "{" << endl;

		justify_width	= IntegerLength(a.fsm_count - 1) + 1;
		data_ptr		= a.fsm_table;

		for (i=0; i<a.fsm_count; i++)
		{
			f << "//** fsm state " << i << endl;

			if (i < a.fsm_count - 1)
				last = false;
			else
				last = true;

			PrintArrayRow(f, data_ptr, a.input_range, justify_width, last);

			data_ptr += a.input_range;
		}

		f << "};" << endl << endl;

		if (rm_table)
		{
			f << "const int    " << lexical_driver_prefix_u;
			f	<< "_Regexp_Match[" << lexical_driver_prefix_u;
			f	<< "_FSM_Count * " << lexical_driver_prefix_u;
			f	<< "_Terminal_Count] =" << endl;
			f << "{" << endl;

			data_ptr = a.regexp_match;

			for (i=0; i<a.fsm_count; i++)
			{
				f << "//** fsm state " << i << endl;

				if (i < a.fsm_count - 1)
					last = false;
				else
					last = true;

				PrintArrayRow(f, data_ptr, a.regexp_count, 2, last);

				data_ptr += a.regexp_count;
			}

			f << "};" << endl << endl;
		}
		else
		{
			f << "const int    " << lexical_driver_prefix_u;
			f	<< "_Regexp_Match[1] = { 0 };" << endl << endl;
		}

		if (a.TrailingContextFlag())
		{
			f << "const int    " << lexical_driver_prefix_u;
			f	<< "_Prefix_Match[" << lexical_driver_prefix_u;
			f	<< "_FSM_Count * " << lexical_driver_prefix_u;
			f	<< "_Terminal_Count] =" << endl;
			f << "{" << endl;

			data_ptr = a.prefix_match;

			for (i=0; i<a.fsm_count; i++)
			{
				f << "//** fsm state " << i << endl;

				if (i < a.fsm_count - 1)
					last = false;
				else
					last = true;

				PrintArrayRow(f, data_ptr, a.regexp_count, 2, last);

				data_ptr += a.regexp_count;
			}

			f << "};" << endl << endl;
		}
		else
		{
			f << "const int    " << lexical_driver_prefix_u;
			f	<< "_Prefix_Match[1] = { 0 };" << endl << endl;
		}

		justify_width	= IntegerLength(a.regexp_count - 1) + 1;
		data_ptr		= a.terminal_match;

		if (a.StartAssignmentFlag())
		{
			f << "const int    " << lexical_driver_prefix_u;
			f	<< "_Terminal_Match[" << lexical_driver_prefix_u;
			f	<< "_FSM_Count * (" << lexical_driver_prefix_u;
			f	<< "_Terminal_Count + 1)] =" << endl;
			f << "{" << endl;

			for (i=0; i<a.fsm_count; i++)
			{
				f << "//** fsm state " << i << endl;

				if (i < a.fsm_count - 1)
					last = false;
				else
					last = true;

				PrintArrayRow(f, data_ptr, a.regexp_count + 1,
														justify_width, last);

				data_ptr += a.regexp_count + 1;
			}

			f << "};" << endl << endl;
		}
		else
		{
			f << "const int    " << lexical_driver_prefix_u;
			f	<< "_Terminal_Match[" << lexical_driver_prefix_u;
			f	<< "_FSM_Count] =" << endl;
			f << "{" << endl;

			int		tm[a.fsm_count];

			for (i=0; i<a.fsm_count; i++)
			{
				tm[i] = data_ptr[a.regexp_count];

				data_ptr += a.regexp_count + 1;
			}

			PrintArrayRow(f, tm, a.fsm_count, justify_width, true);

			f << "};" << endl << endl;
		}

		f << "const int    " << lexical_driver_prefix_u;
		f	<< "_Start_Assignment[" << lexical_driver_prefix_u;
		f	<< "_Terminal_Count] =" << endl;
		f << "{" << endl;

		PrintArrayRow(f, a.start_assignment, a.regexp_count, 2, true);

		f << "};" << endl << endl;

		f << "const int    " << lexical_driver_prefix_u;
		f	<< "_Trailing_Context[" << lexical_driver_prefix_u;
		f	<< "_Terminal_Count] =" << endl;
		f << "{" << endl;

		PrintArrayRow(f, a.trailing_context, a.regexp_count, 2, true);

		f << "};" << endl << endl;

		CTerminalInformation	*terminals = a.terminals;
		int						terminal_data[a.regexp_count];

		f << "const int    " << lexical_driver_prefix_u;
		f	<< "_Token_Map[" << lexical_driver_prefix_u;
		f	<< "_Terminal_Count] =" << endl;
		f << "{" << endl;

		for (i=0; i<a.regexp_count; i++)
		{
			if (!terminals[i].filter_flag)
				terminal_data[i] = terminals[i].symbol_index;
			else
				terminal_data[i] = -2;
		}

		PrintArrayRow(f, terminal_data, a.regexp_count, justify_width, true);

		f << "};" << endl << endl;

		f << "const int    " << lexical_driver_prefix_u;
		f	<< "_Filter_Map[" << lexical_driver_prefix_u;
		f	<< "_Terminal_Count] =" << endl;
		f << "{" << endl;

		for (i=0; i<a.regexp_count; i++)
		{
			if (terminals[i].filter_flag)
				terminal_data[i] = -terminals[i].symbol_index - 1;
			else
				terminal_data[i] = -1;
		}

		PrintArrayRow(f, terminal_data, a.regexp_count, justify_width, true);

		f << "};" << endl << endl;

		f << "const int    " << lexical_driver_prefix_u;
		f	<< "_Literal_Terminal[" << lexical_driver_prefix_u;
		f	<< "_Terminal_Count] =" << endl;
		f << "{" << endl;

		for (i=0; i<a.regexp_count; i++)
		{
			if (terminals[i].literal_flag)
				terminal_data[i] = 1;
			else
				terminal_data[i] = 0;
		}

		PrintArrayRow(f, terminal_data, a.regexp_count, 2, true);

		f << "};" << endl;

		f << endl;
		f << "#endif" << endl;

		f.close();

		f.open(lexical_driver_h_path.c_str());

		f << "#ifndef " << lexical_driver_h_guard << endl;
		f << "#define " << lexical_driver_h_guard << endl;
		f << endl;
		f << "#include \"lexicalDriver.h\"" << endl;
		f << "#include \"" << lexical_driver_prefix << "RegexpData.h\"" << endl;
		f << endl;

		if (tty)
		{
			f << "#include \"ttyTree.h\"" << endl << endl;
			f << "extern SttyTreeNode     *parse_tree;" << endl;
		}
		else
		{
			s_iterator = include_code.begin();

			while (s_iterator != include_code.end())
				f << *s_iterator++ << endl;
		}

		f << endl;
		f << "class C" << lexical_driver_prefix_u;
		f	<< "LexicalDriver: public CLexicalRegexpDriver" << endl;
		f << "{" << endl;
		f << "    public:" << endl;
		f << "    C" << lexical_driver_prefix_u << "LexicalDriver();" << endl;
		f << "    virtual ~C" << lexical_driver_prefix_u;
		f	<< "LexicalDriver();" << endl;
		f << endl;
		f << "    virtual void    SetupFunctionTables();" << endl;
		f << "    virtual void    LoadStackElement(CStackElement *se);" << endl;
		f << "    virtual void    ProcessFilter();" << endl;
		f << endl;

		if (tty)
		{
			f << "//  tty parse tree declarations" << endl << endl;
			f << "    void    ttyTreePrintValueOnly(const char *n)" << endl;
			f << "    { parse_tree->shift(n, (const char *)";
			f	<< "GetTerminalString(), true); }" << endl;
			f << "    void    ttyTreePrintNameAndValue(const char *n)" << endl;
			f << "    { parse_tree->shift(n, (const char *)";
			f	<< "GetTerminalString(), false); }" << endl;
		}
		else
		{
			s_iterator = class_code.begin();

			while (s_iterator != class_code.end())
				f << *s_iterator++ << endl;
		}

		f << endl;
		f << "//  terminal function declarations for C";
		f	<< lexical_driver_prefix_u << "LexicalDriver" << endl;
		f << endl;
		f << "    void    (C" << lexical_driver_prefix_u;
		f	<< "LexicalDriver::* *token_functions)(CStackElement *se);" << endl;
		f << "    void    (C" << lexical_driver_prefix_u;
		f	<< "LexicalDriver::* *filter_functions)();" << endl;
		f << endl;

		justify_width = IntegerLength(a.token_count - 1);

		for (i=0; i<a.token_count; i++)
		{
			f << "    void    Token_" << setfill('0');
			f	<< setw(justify_width) << i << "(CStackElement *se);" << endl;
		}

		f << endl;

		justify_width = IntegerLength(a.filter_count - 1);

		for (i=0; i<a.filter_count; i++)
		{
			f << "    void    Filter_" << setfill('0');
			f	<< setw(justify_width) << i << "();" << endl;
		}

		f << "};" << endl;
		f << endl;
		f << "#endif" << endl;

		f.close();

		f.open(lexical_driver_cpp_path.c_str());

		f << "#include \"" << lexical_driver_prefix;
		f	<< "LexicalDriver.h\"" << endl;
		f << endl;

		if (!tty)
		{
			s_iterator = source_code.begin();

			while (s_iterator != source_code.end())
				f << *s_iterator++ << endl;
		}

		f << endl;
		f << "C" << lexical_driver_prefix_u << "LexicalDriver::C";
		f	<< lexical_driver_prefix_u << "LexicalDriver()" << endl;
		f << "    : CLexicalRegexpDriver(" << endl;
		f << "        " << lexical_driver_prefix_u;
		f	<< "_Token_Count," << endl;
		f << "        " << lexical_driver_prefix_u;
		f	<< "_Filter_Count," << endl;
		f << "        " << lexical_driver_prefix_u;
		f	<< "_Terminal_Count," << endl;
		f << "        " << lexical_driver_prefix_u;
		f	<< "_Empty_Symbol_Token," << endl;
		f << "        " << lexical_driver_prefix_u;
		f	<< "_Input_Range," << endl;
		f << "        " << lexical_driver_prefix_u;
		f	<< "_FSM_Count," << endl;
		f << "        " << lexical_driver_prefix_u;
		f	<< "_FSM_Table," << endl;
		f << "        " << lexical_driver_prefix_u;
		f	<< "_Regexp_Match," << endl;
		f << "        " << lexical_driver_prefix_u;
		f	<< "_Prefix_Match," << endl;
		f << "        " << lexical_driver_prefix_u;
		f	<< "_Terminal_Match," << endl;
		f << "        " << lexical_driver_prefix_u;
		f	<< "_Start_Assignment," << endl;
		f << "        " << lexical_driver_prefix_u;
		f	<< "_Trailing_Context," << endl;
		f << "        " << lexical_driver_prefix_u;
		f	<< "_Token_Map," << endl;
		f << "        " << lexical_driver_prefix_u;
		f	<< "_Filter_Map," << endl;
		f << "        " << lexical_driver_prefix_u;
		f	<< "_Literal_Terminal," << endl;
		f << "        " << lexical_driver_prefix_u;
		f	<< "_Single_Column_Match)" << endl;
		f << "{" << endl;
		f << "    token_functions =" << endl;
		f << "        new (void (C" << lexical_driver_prefix_u;
		f	<< "LexicalDriver::*[token_count])(CStackElement *));" << endl;
		f << endl;
		f << "    filter_functions =" << endl;
		f << "        new (void (C" << lexical_driver_prefix_u;
		f	<< "LexicalDriver::*[filter_count])());" << endl;
		f << endl;
		f << "    SetupFunctionTables();" << endl;
		f << endl;

		if (!tty)
		{
			s_iterator = constructor_code.begin();

			while (s_iterator != constructor_code.end())
				f << *s_iterator++ << endl;
		}

		f << "}" << endl;
		f << endl;
		f << "C" << lexical_driver_prefix_u << "LexicalDriver::~C";
		f	<< lexical_driver_prefix_u << "LexicalDriver()" << endl;
		f << "{" << endl;
		f << "    delete [] token_functions;" << endl;
		f << "    delete [] filter_functions;" << endl;
		f << endl;

		if (!tty)
		{
			s_iterator = destructor_code.begin();

			while (s_iterator != destructor_code.end())
				f << *s_iterator++ << endl;
		}

		f << "}" << endl;
		f << endl;
		f << "void    C" << lexical_driver_prefix_u;
		f	<< "LexicalDriver::LoadStackElement(CStackElement *se)" << endl;
		f << "{" << endl;
		f << "    if (0 <= current_token &&";
		f	<< " current_token < token_count)" << endl;
		f << "    {" << endl;

		if (tty)
		{
			f << "        se->type    = StackLiteralToken;" << endl;
			f << "        se->token   = current_token;" << endl;
			f << endl;
			f << "        (this->*token_functions[current_token])(se);" << endl;
		}
		else
		{
			f << "        se->token = current_token;" << endl;
			f << endl;
			f << "        if (!process_terminal_flag ||";
			f	<< " GetCurrentLiteralTerminalFlag())" << endl;
			f << "            se->type = StackLiteralToken;" << endl;
			f << "        else" << endl;
			f << "            (this->*token_functions";
			f	<< "[current_token])(se);" << endl;
		}

		f << "    }" << endl;
		f << "}" << endl;
		f << endl;

		f << "void    C" << lexical_driver_prefix_u;
		f	<< "LexicalDriver::ProcessFilter()" << endl;
		f << "{" << endl;
		f << "    if (process_terminal_flag &&";
		f	<< " !GetCurrentLiteralTerminalFlag())" << endl;
		f << "        (this->*filter_functions[current_filter])();" << endl;
		f << "}" << endl;
		f << endl;
		f << "//  terminal function definitions for C";
		f	<< lexical_driver_prefix_u << "LexicalDriver" << endl;

		compiler_data->CalculateOutputWidths();

		const list<CTerminal *>		&t_list = compiler_data->GetTerminalList();

		list<CTerminal *>::const_iterator	t_iterator = t_list.begin();

		while (t_iterator != t_list.end())
		{
			CTerminal	*t = *t_iterator++;

			f << endl;
			f << "//  " << *t;
			f << "//" << endl;
			f << "void    C" << lexical_driver_prefix_u << "LexicalDriver::";

			if (t->GetFilterFlag())
			{
				justify_width = IntegerLength(a.filter_count - 1);

				f << "Filter_" << setfill('0') << setw(justify_width);
				f	<< -t->GetSymbolIndex() - 1 << "()" << endl;
			}
			else
			{
				justify_width = IntegerLength(a.token_count - 1);

				f << "Token_" << setfill('0') << setw(justify_width);
				f	<< t->GetSymbolIndex() << "(CStackElement *se)" << endl;
			}

			f << "{" << endl;

			if (tty)
			{
				if (!t->GetFilterFlag())
				{
					const string	&name	= t->GetName();
					bool			l_flag	= t->GetLiteralFlag();

					string	escape_string;

					if (l_flag)
						f << "    ttyTreePrintValueOnly(\"";
					else
						f << "    ttyTreePrintNameAndValue(\"";

					f << EscapeString(name, escape_string) << "\");" << endl;
				}
			}
			else
			{
				const list<string>	&c_list = t->GetCodeTextConst();

				if (c_list.size())
				{
					list<string>::const_iterator	c_iterator = c_list.begin();

					while (c_iterator != c_list.end())
						f << *c_iterator++ << endl;
				}
				else if (!t->GetFilterFlag())
					f << "    se->type = StackLiteralToken;" << endl;
			}

			f << "}" << endl;
		}

		f << endl;
		f << "void    C" << lexical_driver_prefix_u;
		f	<< "LexicalDriver::SetupFunctionTables()" << endl;
		f << "{" << endl;

		justify_width = IntegerLength(a.token_count - 1);

		for (i=0; i<a.token_count; i++)
		{
			f << "    token_functions[" << setfill(' ');
			f	<< setw(justify_width) << i;
			f	<< "] = &C" << lexical_driver_prefix_u;
			f	<< "LexicalDriver::Token_";
			f	<< setfill('0') << setw(justify_width) << i << ";" << endl;
		}

		f << endl;

		justify_width = IntegerLength(a.filter_count - 1);

		for (i=0; i<a.filter_count; i++)
		{
			f << "    filter_functions[" << setfill(' ');
			f	<< setw(justify_width) << i;
			f	<< "] = &C" << lexical_driver_prefix_u;
			f	<< "LexicalDriver::Filter_";
			f	<< setfill('0') << setw(justify_width) << i << ";" << endl;
		}

		f << "}" << endl;

		f.close();
	}
}

void	CCompilerCalculate::GenerateParserCode(bool unit_rule_flag, bool tty)
{
	CParse	&p = *parser;
	int		i;
	string	parser_data_h_guard;
	string	parser_driver_h_guard;
	string	parser_driver_prefix;
	string	parser_driver_prefix_u;
	int		justify_width;
	int		*data_ptr;
	bool	last;

	list<string>			include_code;
	list<string>			class_code;
	list<string>			source_code;
	list<string>			constructor_code;
	list<string>			destructor_code;
	list<string>::iterator	s_iterator;

	CreateIncludeGuardIdentifier(	parser_data_h_name,
									parser_data_h_guard);

	CreateIncludeGuardIdentifier(	parser_driver_h_name,
									parser_driver_h_guard);

	compiler_data->CalculateOutputWidths();

	parser_driver_prefix	= compiler_data->GetParserPrefixName();
	parser_driver_prefix_u	= parser_driver_prefix;

	if (parser_driver_prefix_u.length())
		parser_driver_prefix_u[0] = toupper(parser_driver_prefix_u[0]);

	compiler_data->GetParserCodeSections(	include_code,
											class_code,
											source_code,
											constructor_code,
											destructor_code);

	ofstream	f(parser_data_h_path.c_str());

	f << "#ifndef " << parser_data_h_guard << endl;
	f << "#define " << parser_data_h_guard << endl;
	f << endl;
	f << "//  parser data tables for CParserDriver" << endl;
	f << endl;
	f << "const int    " << parser_driver_prefix_u;
	f	<< "_Terminal_Count_E           = ";
	f	<< p.terminal_count_e << ";" << endl;
	f << "const int    " << parser_driver_prefix_u;
	f	<< "_Symbol_Count_A             = ";
	f	<< p.symbol_count_a << ";" << endl;
	f << "const int    " << parser_driver_prefix_u;
	f	<< "_Rule_Count_A               = ";
	f	<< p.rule_count_a << ";" << endl;
	f << "const int    " << parser_driver_prefix_u;
	f	<< "_Parser_Empty_Symbol_Token  = ";
	f	<< p.empty_symbol_index << ";" << endl;
	f << "const int    " << parser_driver_prefix_u;
	f	<< "_Shift_Action               = ";
	f	<< p.shift_action << ";" << endl;
	f << "const int    " << parser_driver_prefix_u;
	f	<< "_Accept_Action              = ";
	f	<< p.accept_action << ";" << endl;
	f << "const int    " << parser_driver_prefix_u;
	f	<< "_Conflict_Entry             = ";
	f	<< p.conflict_entry << ";" << endl;

	f << "const int    " << parser_driver_prefix_u;
	f	<< "_State_Count                = ";

	int		m_count;

	if (!unit_rule_flag)
		m_count = p.machine_count;
	else
		m_count = p.unit_machine_count;

	f << m_count << ";" << endl;

	f << "const int    " << parser_driver_prefix_u;
	f	<< "_Number_Of_Conflict_Entries = ";
	f	<< p.parsing_conflicts->Size() << ";" << endl;

	f << "const int    " << parser_driver_prefix_u;
	f	<< "_Number_Of_Error_Indexes    = ";

	int		number_of_error_indexes;
	int		number_of_error_functions;

	CCompilerData::ErrorCalculation
		c = compiler_data->GetParserErrorCalculation();

	CCompilerData::ErrorFunctionNumbering
		n = compiler_data->GetParserErrorFunctionNumbering();

	CCompilerData::ErrorFunctionAggregation
		aggregation = compiler_data->GetParserErrorFunctionAggregation();
					
	if (!error_entries_numbered)
	{
		number_of_error_indexes		= 1;
		number_of_error_functions	= 1;
	}
	else
	{
		if (c == CCompilerData::NoCalculation)
		{
			if (n == CCompilerData::OneNumber)
			{
				number_of_error_indexes = 1;
			}
			else if (n == CCompilerData::NumberPerState)
			{
				number_of_error_indexes = p.machine_count;
			}
			else if (n == CCompilerData::NumberPerEntry)
			{
				number_of_error_indexes = p.default_error_entry_count;
			}

			if (aggregation == CCompilerData::OneFunction)
			{
				number_of_error_functions = 1;
			}
			else if (aggregation == CCompilerData::FunctionPerState)
			{
				number_of_error_functions = p.machine_count;
			}
			else if (aggregation == CCompilerData::FunctionPerEntry)
			{
				number_of_error_functions = p.default_error_entry_count;
			}
		}
		else if (	c == CCompilerData::ApproximateEssential ||
					c == CCompilerData::ExactEssential)
		{
			if (n == CCompilerData::OneNumber)
			{
				number_of_error_indexes = 2;
			}
			else if (n == CCompilerData::NumberPerState)
			{
				number_of_error_indexes = p.machine_count + 1;
			}
			else if (n == CCompilerData::NumberPerEntry)
			{
				number_of_error_indexes = p.essential_error_entry_count + 1;
			}

			if (aggregation == CCompilerData::OneFunction)
			{
				number_of_error_functions = 2;
			}
			else if (aggregation == CCompilerData::FunctionPerState)
			{
				number_of_error_functions = p.machine_count + 1;
			}
			else if (aggregation == CCompilerData::FunctionPerEntry)
			{
				number_of_error_functions = p.essential_error_entry_count + 1;
			}
		}
	}

	f << number_of_error_indexes << ";" << endl;
	f << endl;

	f << "const int    " << parser_driver_prefix_u;
	f	<< "_Action_Table[" << parser_driver_prefix_u;
	f	<< "_State_Count * " << parser_driver_prefix_u;
	f	<< "_Terminal_Count_E] =" << endl;
	f << "{" << endl;

	int		jw1 = p.conflict_entry + p.parsing_conflicts->Size() - 1;
	int		jw2 =	jw1 > p.default_error_entry_count ?
					jw1 : p.default_error_entry_count;

	justify_width	= IntegerLength(jw2) + 1;

	if (!error_entries_numbered)
		data_ptr = p.action_table;
	else if (!unit_rule_flag)
		data_ptr = p.error_action_table;
	else
		data_ptr = p.unit_action_table;

	for (i=0; i<m_count; i++)
	{
		f << "//** machine state " << i << endl;

		if (i < m_count - 1)
			last = false;
		else
			last = true;

		PrintArrayRow(f, data_ptr, p.terminal_count_e, justify_width, last);

		data_ptr += p.terminal_count_e;
	}

	f << "};" << endl << endl;

	f << "const int    " << parser_driver_prefix_u;
	f	<< "_Goto_Table[" << parser_driver_prefix_u;
	f	<< "_State_Count * " << parser_driver_prefix_u;
	f	<< "_Symbol_Count_A] =" << endl;
	f << "{" << endl;

	justify_width	= IntegerLength(m_count - 1) + 1;

	if (!unit_rule_flag)
		data_ptr = p.goto_table;
	else
		data_ptr = p.unit_goto_table;

	for (i=0; i<m_count; i++)
	{
		f << "//** machine state " << i << endl;

		if (i < m_count - 1)
			last = false;
		else
			last = true;

		PrintArrayRow(f, data_ptr, p.symbol_count_a, justify_width, last);

		data_ptr += p.symbol_count_a;
	}

	f << "};" << endl << endl;

	int		rt[p.rule_count_a];

	f << "const int    " << parser_driver_prefix_u;
	f	<< "_Rule_Left_Symbol_Table[" << parser_driver_prefix_u;
	f	<< "_Rule_Count_A] =" << endl;
	f << "{" << endl;

	justify_width	= IntegerLength(p.symbol_count_a - 1) + 1;

	for (i=0; i<p.rule_count_a; i++)
		rt[i] = p.rules[i].left_symbol_index;

	PrintArrayRow(f, rt, p.rule_count_a, justify_width, true);

	f << "};" << endl << endl;

	f << "const int    " << parser_driver_prefix_u;
	f	<< "_Rule_Right_Length_Table[" << parser_driver_prefix_u;
	f	<< "_Rule_Count_A] =" << endl;
	f << "{" << endl;

	for (i=0; i<p.rule_count_a; i++)
		rt[i] = p.rules[i].right_symbol_count;

	PrintArrayRow(f, rt, p.rule_count_a, justify_width, true);

	f << "};" << endl << endl;

	const list<CErrorToken *>	&et_list = compiler_data->GetErrorTokenList();

	if (!tty && et_list.size())
	{
		list<CErrorToken *>::const_iterator		et_iterator = et_list.begin();

		while (et_iterator != et_list.end())
		{
			CErrorToken		*et = *et_iterator++;

			string	pad1(compiler_data->GetErrorTokenNameWidth() -
							et->GetName().length(), ' ');

			f << "const int    Error_Token_" << et->GetName() << pad1;
			f	<< " = " << et->GetSymbolIndex() << ";" << endl;
		}

		f << endl;

		et_iterator = et_list.begin();

		while (et_iterator != et_list.end())
		{
			CErrorToken		*et = *et_iterator++;

			string	pad1(compiler_data->GetErrorTokenNameWidth() -
							et->GetName().length(), ' ');

			f << "const int    Error_Synchronization_" << et->GetName() << pad1;
			f	<< " = " << et->GetSynchronizationLength() << ";" << endl;
		}

		f << endl;
	}

	f << "#endif" << endl;

	f.close();

	f.open(parser_driver_h_path.c_str());

	f << "#ifndef " << parser_driver_h_guard << endl;
	f << "#define " << parser_driver_h_guard << endl;
	f << endl;
	f << "#include \"" << parser_driver_prefix << "ParserData.h\"" << endl;
	f << "#include \"parserDriver.h\"" << endl;
	f << endl;

	if (tty)
	{
		f << "#include \"ttyTree.h\"" << endl;
		f << endl;
		f << "extern SttyTreeNode     *parse_tree;" << endl;
	}
	else
	{
		s_iterator = include_code.begin();

		while (s_iterator != include_code.end())
			f << *s_iterator++ << endl;
	}

	f << endl;
	f << "class C" << parser_driver_prefix_u;
	f	<< "ParserDriver: public CParserDriver" << endl;
	f << "{" << endl;
	f << "    public:" << endl;
	f << "    C" << parser_driver_prefix_u << "ParserDriver();" << endl;
	f << "    virtual ~C" << parser_driver_prefix_u << "ParserDriver();";
	f << endl;
	f << endl;
	f << "    virtual void    SetupFunctionTables();";
	f	<< endl;
	f << "    virtual int     CallReduceFunction(int action, void *data);";
	f	<< endl;
	f << "    virtual void    CallConflictFunction(int index);";
	f	<< endl;
	f << "    virtual void    CallErrorFunction(int index);";
	f	<< endl;
	f << "    virtual void    DeleteGeneratedCode(CStackElement *s, int c);";
	f	<< endl;
	f << endl;

	if (!tty)
	{
		s_iterator = class_code.begin();

		while (s_iterator != class_code.end())
			f << *s_iterator++ << endl;
	}

	f << endl;
	f << "//  rule reduce function declarations for C";
	f	<< parser_driver_prefix_u << "ParserDriver" << endl;
	f << endl;
	f << "    int     (C" << parser_driver_prefix_u;
	f	<< "ParserDriver::* *reduce_functions)(void *data);" << endl;
	f << endl;

	justify_width = IntegerLength(p.rule_count - 1);

	for (i=0; i<p.rule_count; i++)
	{
		f << "    int     Reduce_" << setfill('0');
		f	<< setw(justify_width) << i << "(void *data);" << endl;
	}

	f << endl;
	f << "//  conflict function declarations for C";
	f	<< parser_driver_prefix_u << "ParserDriver" << endl;
	f << endl;
	f << "    void    (C" << parser_driver_prefix_u;
	f	<< "ParserDriver::* *conflict_functions)();" << endl;
	f << endl;

	justify_width = IntegerLength(p.parsing_conflicts->Size() - 1);

	for (i=0; i<p.parsing_conflicts->Size(); i++)
	{
		f << "    void    Conflict_" << setfill('0');
		f	<< setw(justify_width) << i << "();" << endl;
	}

	f << endl;
	f << "//  parser error function declarations for C";
	f	<< parser_driver_prefix_u << "ParserDriver" << endl;
	f << endl;
	f << "    void    (C" << parser_driver_prefix_u;
	f	<< "ParserDriver::* *error_functions)();" << endl;
	f << endl;

	justify_width = IntegerLength(number_of_error_functions);

	for (i=1; i<=number_of_error_functions; i++)
	{
		f << "    void    Error_" << setfill('0');
		f	<< setw(justify_width) << i << "();" << endl;
	}

	f << "};" << endl;
	f << endl;
	f << "#endif" << endl;

	f.close();

	f.open(parser_driver_cpp_path.c_str());

	f << "#include \"" << parser_driver_prefix << "ParserDriver.h\"" << endl;
	f << endl;

	if (!tty)
	{
		s_iterator = source_code.begin();

		while (s_iterator != source_code.end())
			f << *s_iterator++ << endl;
	}

	f << endl;
	f << "C" << parser_driver_prefix_u << "ParserDriver::C";
	f	<< parser_driver_prefix_u << "ParserDriver()" << endl;
	f << "    : CParserDriver(" <<  endl;
	f << "        " << parser_driver_prefix_u;
	f	<< "_Terminal_Count_E," << endl;
	f << "        " << parser_driver_prefix_u;
	f	<< "_Symbol_Count_A," << endl;
	f << "        " << parser_driver_prefix_u;
	f	<< "_Rule_Count_A," << endl;
	f << "        " << parser_driver_prefix_u;
	f	<< "_Parser_Empty_Symbol_Token," << endl;
	f << "        " << parser_driver_prefix_u;
	f	<< "_Shift_Action," << endl;
	f << "        " << parser_driver_prefix_u;
	f	<< "_Accept_Action," << endl;
	f << "        " << parser_driver_prefix_u;
	f	<< "_Conflict_Entry," << endl;
	f << "        " << parser_driver_prefix_u;
	f	<< "_State_Count," << endl;
	f << "        " << parser_driver_prefix_u;
	f	<< "_Number_Of_Conflict_Entries," << endl;
	f << "        " << parser_driver_prefix_u;
	f	<< "_Number_Of_Error_Indexes," << endl;
	f << "        " << parser_driver_prefix_u;
	f	<< "_Action_Table," << endl;
	f << "        " << parser_driver_prefix_u;
	f	<< "_Goto_Table," << endl;
	f << "        " << parser_driver_prefix_u;
	f	<< "_Rule_Left_Symbol_Table," << endl;
	f << "        " << parser_driver_prefix_u;
	f	<< "_Rule_Right_Length_Table)" << endl;
	f << "{" << endl;
	f << "    reduce_functions =" << endl;
	f << "    new (int (C" << parser_driver_prefix_u << "ParserDriver::*[";
	f	<< parser_driver_prefix_u << "_Rule_Count_A - 1])(void *));" << endl;
	f << endl;

	if (p.parsing_conflicts->Size())
	{
		f << "    conflict_functions =" << endl;
		f << "    new (void (C" << parser_driver_prefix_u << "ParserDriver::*[";
		f	<< parser_driver_prefix_u << "_Number_Of_Conflict_Entries])());";
		f << endl;
		f << endl;
	}

	f << "    error_functions =" << endl;
	f << "    new (void (C" << parser_driver_prefix_u << "ParserDriver::*[";
	f	<< parser_driver_prefix_u << "_Number_Of_Error_Indexes + 1])());";
	f << endl;
	f << endl;
	f << "    SetupFunctionTables();" << endl;
	f << endl;

	if (!tty)
	{
		s_iterator = constructor_code.begin();

		while (s_iterator != constructor_code.end())
			f << *s_iterator++ << endl;
	}

	f << "}" << endl;
	f << endl;
	f << "C" << parser_driver_prefix_u << "ParserDriver::~C";
	f	<< parser_driver_prefix_u << "ParserDriver()" << endl;
	f << "{" << endl;
	f << "    delete [] reduce_functions;" << endl;

	if (p.parsing_conflicts->Size())
	{
		f << "    delete [] conflict_functions;" << endl;
	}

	f << "    delete [] error_functions;" << endl;
	f << endl;

	if (!tty)
	{
		s_iterator = destructor_code.begin();

		while (s_iterator != destructor_code.end())
			f << *s_iterator++ << endl;
	}

	f << "}" << endl;
	f << endl;
	f << "int     C" << parser_driver_prefix_u;
	f	<< "ParserDriver::CallReduceFunction(int action, void *data)" << endl;
	f << "{" << endl;
	f << "    return (this->*reduce_functions[action])(data);" << endl;
	f << "}" << endl;
	f << endl;
	f << "void    C" << parser_driver_prefix_u;
	f	<< "ParserDriver::CallConflictFunction(int index)" << endl;
	f << "{" << endl;
	f << "    (this->*conflict_functions[index])();" << endl;
	f << "}" << endl;
	f << endl;
	f << "void    C" << parser_driver_prefix_u;
	f	<< "ParserDriver::CallErrorFunction(int index)" << endl;
	f << "{" << endl;
	f << "    (this->*error_functions[index])();" << endl;
	f << "}" << endl;
	f << endl;
	f << "void    C" << parser_driver_prefix_u;
	f	<< "ParserDriver::DeleteGeneratedCode(CStackElement *s, int c)" << endl;
	f << "{" << endl;
	f << "    for (int i=c-1; i>=0; i--)" << endl;
	f << "    {" << endl;
	f << "        CStackElement   &se = s[i];" << endl;
	f << endl;
	f << "        if (se.type == StackCharArray)" << endl;
	f << "            delete [] se.char_array;" << endl;
	f << "        else if (se.type == StackData)" << endl;
	f << "            (this->*reduce_functions[se.reduce_action])(se.data);";
	f	<< endl;
	f << "    }" << endl;
	f << "}" << endl;
	f << endl;
	f << "//  rule reduce function definitions for C";
	f	<< parser_driver_prefix_u << "ParserDriver" << endl;

	justify_width = IntegerLength(p.rule_count - 1);

	const list<CRule *>		&r_list = compiler_data->GetRuleList();

	list<CRule *>::const_iterator	r_iterator = r_list.begin();

	while (r_iterator != r_list.end())
	{
		CRule	*r = *r_iterator++;

		f << endl;
		f << "//  " << *r;
		f << "//" << endl;
		f << "int     C" << parser_driver_prefix_u << "ParserDriver::Reduce_";
		f	<< setfill('0') << setw(justify_width);
		f	<< r->GetIndex() << "(void *data)" << endl;

		f << "{" << endl;

		if (tty)
		{
			string	escape_string;

			f << "    return parse_tree->reduce_rule(" << endl;
			f << "        \"";
			f	<< EscapeString(r->GetLeft()->GetName(), escape_string);
			f	<< "\"," << endl;

			const vector<const CSymbol *>		&right = r->GetRight();
			vector<const CSymbol *>::size_type	j;

			for (j=0; j<right.size(); j++)
			{
				f << "        \"";
				f	<< EscapeString(right[j]->GetName(), escape_string);
				f	<< "\"," << endl;
			}

			f << "        0);" << endl;
		}
		else
		{
			const list<string>	&c_list = r->GetCodeTextConst();

			if (c_list.size())
			{
				list<string>::const_iterator	c_iterator = c_list.begin();

				while (c_iterator != c_list.end())
					f << *c_iterator++ << endl;
			}
			else
				f << "    return 0;" << endl;
		}

		f << "}" << endl;
	}

	justify_width = IntegerLength(p.parsing_conflicts->Size() - 1);

	f << endl;
	f << "//  conflict function definitions for C";
	f	<< parser_driver_prefix_u << "ParserDriver" << endl;

	const map<int, SConflictResolution *>
		&cr_map = compiler_data->GetConflictResolutionMap();

	ListConflict::listptr	conflict_listptr =
								p.parsing_conflicts->FirstListPtr();

	for (i=0; i<p.parsing_conflicts->Size(); i++)
	{
		CConflict	*conflict = p.parsing_conflicts->NextData(conflict_listptr);

		f << endl;
		f << "//  conflict entry " << i << endl;
		f << "//  machine state " << conflict->machine_state_index;
		f << "  terminal symbol " << conflict->terminal_symbol_index;
		f << " " << p.symbols[conflict->terminal_symbol_index].name << endl;
		f << "//  action set";

		VListInt	*a_set = conflict->action_set;

		VListInt::listptr	a_listptr = a_set->FirstListPtr();

		while (a_listptr)
		{
			f << " ";

			int		action = a_set->NextData(a_listptr);

			if (action == p.shift_action)
				f << "shift";
			else if (action == p.conflict_entry)
				f << "conflict";
			else
				f << "rule:" << action;
		}

		f << endl;
		f << "//" << endl;
		f << "void    C" << parser_driver_prefix_u << "ParserDriver::Conflict_";
		f	<< setfill('0') << setw(justify_width) << i << "()" << endl;

		f << "{" << endl;

		map<int, SConflictResolution *>::const_iterator
			cr_iterator = cr_map.find(i);

		if (cr_iterator != cr_map.end() && !tty)
		{
			const list<string>	&c_list =
				(*cr_iterator).second->code_text;

			list<string>::const_iterator	c_iterator = c_list.begin();

			while (c_iterator != c_list.end())
				f << *c_iterator++ << endl;
		}

		f << "}" << endl;
	}

	const map<int, SErrorCode *>
		&er_map = compiler_data->GetErrorCodeMap();

	f << endl;
	f << "//  error function definitions for C";
	f	<< parser_driver_prefix_u << "ParserDriver" << endl;

	justify_width = IntegerLength(number_of_error_functions);

	for (i=1; i<=number_of_error_functions; i++)
	{
		f << endl;

		if (!error_entries_numbered)
		{
			f << "// error function" << endl;
		}
		else
		{
			if (c == CCompilerData::NoCalculation)
			{
				if (aggregation == CCompilerData::OneFunction)
				{
					f << "// error function" << endl;
				}
				else if (aggregation == CCompilerData::FunctionPerState)
				{
					f << "// machine state " << i - 1 << endl;
				}
				else if (aggregation == CCompilerData::FunctionPerEntry)
				{
					f << "// machine state ";
					f	<< p.default_error_state_terminals[i].state;
					f	<< "  terminal ";
					f	<< p.default_error_state_terminals[i].terminal << " ";
					f	<< p.symbols[
							p.default_error_state_terminals[i].terminal].name;
					f	<< endl;
				}
			}
			else if (	c == CCompilerData::ApproximateEssential ||
						c == CCompilerData::ExactEssential)
			{
				if (aggregation == CCompilerData::OneFunction)
				{
					if (i == 1)
						f << "// inaccessible error" << endl;
					else
						f << "// accessible error" << endl;
				}
				else if (aggregation == CCompilerData::FunctionPerState)
				{
					if (i == 1)
						f << "// inaccessible error" << endl;
					else
						f << "// machine state " << i - 2 << endl;
				}
				else if (aggregation == CCompilerData::FunctionPerEntry)
				{
					if (i == 1)
						f << "// inaccessible error" << endl;
					else
					{
						f << "// machine state ";
						f	<< p.essential_error_state_terminals[i].state;
						f	<< "  terminal ";
						f	<< p.essential_error_state_terminals[i].terminal;
						f	<< " " << p.symbols[
							p.essential_error_state_terminals[i].terminal].name;
						f	<< endl;
					}
				}
			}
		}

		f << "//" << endl;
		f << "void    C" << parser_driver_prefix_u << "ParserDriver::Error_";
		f	<< setfill('0') << setw(justify_width) << i << "()" << endl;

		f << "{" << endl;

		map<int, SErrorCode *>::const_iterator
			er_iterator = er_map.find(i);

		if (er_iterator != er_map.end() && !tty)
		{
			const list<string>	&c_list =
				(*er_iterator).second->code_text;

			list<string>::const_iterator	c_iterator = c_list.begin();

			while (c_iterator != c_list.end())
				f << *c_iterator++ << endl;
		}

		f << "}" << endl;
	}

	f << endl;
	f << "void    C" << parser_driver_prefix_u;
	f	<< "ParserDriver::SetupFunctionTables()" << endl;
	f << "{" << endl;

	justify_width = IntegerLength(p.rule_count - 1);

	r_iterator = r_list.begin();

	while (r_iterator != r_list.end())
	{
		CRule	*r = *r_iterator++;

		f << "    reduce_functions[" << setfill(' ') << setw(justify_width);
		f	<< r->GetIndex() << "] = &C" << parser_driver_prefix_u;
		f	<< "ParserDriver::Reduce_" << setfill('0') << setw(justify_width);
		f	<< r->GetIndex() << ";" << endl;
	}

	f << endl;

	justify_width = IntegerLength(p.parsing_conflicts->Size() - 1);

	for (i=0; i<p.parsing_conflicts->Size(); i++)
	{
		f << "    conflict_functions[" << setfill(' ') << setw(justify_width);
		f	<< i << "] = &C" << parser_driver_prefix_u;
		f	<< "ParserDriver::Conflict_" << setfill('0') << setw(justify_width);
		f	<< i << ";" << endl;
	}

	f << endl;

	jw1 = IntegerLength(number_of_error_indexes);
	jw2 = IntegerLength(number_of_error_functions);

	for (i=1; i<=number_of_error_indexes; i++)
	{
		f << "    error_functions[" << setfill(' ') << setw(jw1);
		f	<< i << "] = &C" << parser_driver_prefix_u;
		f	<< "ParserDriver::Error_";

		if (!error_entries_numbered)
		{
			f << "1;" << endl;
		}
		else
		{
			if (c == CCompilerData::NoCalculation)
			{
				if (aggregation == CCompilerData::OneFunction)
				{
					f << "1;" << endl;
				}
				else if (aggregation == CCompilerData::FunctionPerState)
				{
					if (n == CCompilerData::NumberPerState)
					{
						f << setfill('0') << setw(jw2) << i << ";" << endl;
					}
					else if (n == CCompilerData::NumberPerEntry)
					{
						f << setfill('0') << setw(jw2);
						f << p.default_error_state_terminals[i].state + 1;
						f << ";" << endl;
					}
				}
				else if (aggregation == CCompilerData::FunctionPerEntry)
				{
					f << setfill('0') << setw(jw2) << i << ";" << endl;
				}
			}
			else if (	c == CCompilerData::ApproximateEssential ||
						c == CCompilerData::ExactEssential)
			{
				if (aggregation == CCompilerData::OneFunction)
				{
					if (i == 1)
						f << "1;" << endl;
					else
						f << "2;" << endl;
				}
				else if (aggregation == CCompilerData::FunctionPerState)
				{
					if (i == 1 || n == CCompilerData::NumberPerState)
					{
						f << setfill('0') << setw(jw2) << i << ";" << endl;
					}
					else if (n == CCompilerData::NumberPerEntry)
					{
						f << setfill('0') << setw(jw2);
						f << p.essential_error_state_terminals[i].state + 2;
						f << ";" << endl;
					}
				}
				else if (aggregation == CCompilerData::FunctionPerEntry)
				{
					f << setfill('0') << setw(jw2) << i << ";" << endl;
				}
			}
		}
	}

	f << "}" << endl;

	f.close();
}

void	CCompilerCalculate::GenerateBaseAndStackCode()
{
	int		index;

	if (generate_lexical_source_files)
	{
		ofstream	f1(lexical_base_h_path.c_str());

		index = 0;

		while (lexical_base_code_h[index])
			f1 << lexical_base_code_h[index++] << endl;

		f1.close();

		ofstream	f2(lexical_base_cpp_path.c_str());

		index = 0;

		while (lexical_base_code_cpp[index])
			f2 << lexical_base_code_cpp[index++] << endl;

		f2.close();
	}

	if (generate_parser_source_files)
	{
		ofstream	f1(parser_base_h_path.c_str());

		index = 0;

		while (parser_base_code_h[index])
			f1 << parser_base_code_h[index++] << endl;

		f1.close();

		ofstream	f2(parser_base_cpp_path.c_str());

		index = 0;

		while (parser_base_code_cpp[index])
			f2 << parser_base_code_cpp[index++] << endl;

		f2.close();
	}

	ofstream	f(stack_element_h_path.c_str());

	f << "#ifndef STACKELEMENT_H" << endl;
	f << "#define STACKELEMENT_H" << endl;

	const list<string>				&sc_list	= compiler_data->GetStackCode();
	list<string>::const_iterator	sc_iterator	= sc_list.begin();

	while (sc_iterator != sc_list.end())
		f << *sc_iterator++ << endl;

	f << "#endif" << endl;

	f.close();
}

void	CCompilerCalculate::GenerateTTYCode()
{
	string	lexical_driver_prefix;
	string	lexical_driver_prefix_u;
	string	parser_driver_prefix;
	string	parser_driver_prefix_u;

	lexical_driver_prefix	= compiler_data->GetLexicalPrefixName();
	lexical_driver_prefix_u	= lexical_driver_prefix;

	if (lexical_driver_prefix_u.length())
		lexical_driver_prefix_u[0] = toupper(lexical_driver_prefix_u[0]);

	parser_driver_prefix	= compiler_data->GetParserPrefixName();
	parser_driver_prefix_u	= parser_driver_prefix;

	if (parser_driver_prefix_u.length())
		parser_driver_prefix_u[0] = toupper(parser_driver_prefix_u[0]);

	ofstream	f(tty_main_cpp_path.c_str());

	f << "#include \"" << lexical_driver_h_name << "\"" << endl;
	f << "#include \"" << parser_driver_h_name << "\"" << endl;
	f << "#include \"ttyTree.h\"" << endl;
	f << endl;
	f << "extern SttyTreeNode     *parse_tree;" << endl;
	f << endl;
	f << "int     main(int argc, char *argv[])" << endl;
	f << "{" << endl;
	f << "    C" << lexical_driver_prefix_u << "LexicalDriver     ";
	f	<< "*lexical    = new C" << lexical_driver_prefix_u;
	f	<< "LexicalDriver();" << endl;
	f << "    C" << parser_driver_prefix_u << "ParserDriver      ";
	f	<< "*parser     = new C" << parser_driver_prefix_u;
	f	<< "ParserDriver();" << endl;
	f << endl;
	f << "    switch (argc)" << endl;
	f << "    {" << endl;
	f << "        case 1:" << endl;
	f << "//      lexical->SetInputIoStream(&cin);" << endl;
	f << "//      lexical->SetInputStdStream(stdin);" << endl;
	f << "        lexical->SetInputDescriptor(0);" << endl;
	f << "        break;" << endl;
	f << endl;
	f << "        default:" << endl;
	f << "//      if (lexical->SetInputFileIoStream(argv[1]) ";
	f	<< "& CLexicalDriver::OpenError)" << endl;
	f << "//      if (lexical->SetInputFileStdStream(argv[1]) ";
	f	<< "& CLexicalDriver::OpenError)" << endl;
	f << "        if (lexical->SetInputFileDescriptor(argv[1]) ";
	f	<< "& CLexicalDriver::OpenError)" << endl;
	f << "        {" << endl;
	f << "            cout << \"input file: OpenError\" << endl;" << endl;
	f << "            return 1;" << endl;
	f << "        }" << endl;
	f << "    }" << endl;
	f << endl;
	f << "    lexical->Initialize();" << endl;
	f << "    parser->Initialize();" << endl;
	f << "    parser->SetLexical(lexical);" << endl;
	f << endl;
	f << "    parse_tree = new SttyTreeNode(0, ";
	f	<< "SttyTreeNode::NonTerminal, \"ASTART\");" << endl;
	f << endl;
	f << "    CParserDriver::StatusCode   sc = parser->ParseStream();" << endl;
	f << endl;
	f << "    int lexical_error   = sc & CParserDriver::LexicalError;" << endl;
	f << "    int parser_error    = sc & CParserDriver::ParserError;" << endl;
	f << endl;
	f << "    if (lexical_error)" << endl;
	f << "        cout << \"LEXICAL ERROR\" << endl;" << endl;
	f << endl;
	f << "    if (parser_error)" << endl;
	f << "        cout << \"PARSER ERROR\" << endl;" << endl;
	f << endl;
	f << "    if (lexical_error || parser_error)" << endl;
	f << "    {" << endl;
	f << "        cout    << \"current line   = \"" << endl;
	f << "                << lexical->GetCurrentLineNumber() + 1 << endl;";
	f	<< endl;
	f << "        cout    << \"current column = \"" << endl;
	f << "                << lexical->GetCurrentColumnNumber() + 1 << endl;";
	f	<< endl;
	f << "    }" << endl;
	f << endl;
	f << "    if (sc == CParserDriver::EndOfFile)" << endl;
	f << "        parse_tree->shift(\"EOF\", \"EOF\", true);" << endl;
	f << endl;
	f << "    print_node  pn;" << endl;
	f << endl;
	f << "    pn(parse_tree);" << endl;
	f << endl;
	f << "    delete parse_tree;" << endl;
	f << "    delete parser;" << endl;
	f << "    delete lexical;" << endl;
	f << "}" << endl;

	f.close();

	f.open(tty_makefile_path.c_str());

	f << "SHELL = /bin/sh" << endl;
	f << endl;
	f << "MAKECONFIG_CXX       = g++" << endl;
	f << "MAKECONFIG_LINK      = g++" << endl;
	f << "MAKECONFIG_LFLAGS    = " << endl;
	f << "MAKECONFIG_LIBS      = " << endl;
	f << "MAKECONFIG_CXXFLAGS  = -pipe -O3 -Wno-deprecated" << endl;
	f << endl;
	f << "CXX       =   $(MAKECONFIG_CXX)" << endl;
	f << "CXXFLAGS  =   $(MAKECONFIG_CXXFLAGS)" << endl;
	f << "INCPATH   =   -I. -I$(CPG_DIR)" << endl;
	f << "LFLAGS    =   $(MAKECONFIG_LFLAGS)" << endl;
	f << "LIBS      =   $(MAKECONFIG_LIBS)" << endl;
	f << endl;
	f << ".SUFFIXES: .cpp" << endl;
	f << endl;
	f << ".cpp.o:" << endl;
	f << "\t$(CXX) -c $(CXXFLAGS) $(INCPATH) -o $@ $<" << endl;
	f << endl;
	f << "VPATH     = .:$(CPG_DIR)" << endl;
	f << "DESTDIR   = ./" << endl;
	f << "TARGET    = tty_parser" << endl;
	f << endl;
	f << "SOURCES = \\" << endl;
	f << "\tttyMain.cpp \\" << endl;
	f << "\tttyTree.cpp \\" << endl;
	f << "\t" << lexical_driver_cpp_name << " \\" << endl;
	f << "\t" << lexical_base_cpp_name << " \\" << endl;
	f << "\t" << parser_driver_cpp_name << " \\" << endl;
	f << "\t" << parser_base_cpp_name << endl;
	f << endl;
	f << "OBJECTS = $(SOURCES:.cpp=.o)" << endl;
	f << endl;
	f << "application: $(DESTDIR)$(TARGET)" << endl;
	f << endl;
	f << "$(DESTDIR)$(TARGET): $(OBJECTS)" << endl;
	f << "\t$(MAKECONFIG_LINK) $(LFLAGS) -o $(TARGET) ";
	f	<< "$(OBJECTS) $(LIBS)" << endl;
	f << endl;
	f << "%.d: %.cpp" << endl;
	f << "\t$(SHELL) -ec \"$(CXX) -MM $(CXXFLAGS) ";
	f	<< "$(INCPATH) $< | sed 's/$*\\\\.o[ :]*/$@ &/g' > $@\"" << endl;
	f << endl;
	f << "clean:" << endl;
	f << "\t-rm -f $(OBJECTS)" << endl;
	f << "\t-rm -f *.d" << endl;
	f << "\t-rm -f core" << endl;
	f << endl;
	f << "ifneq ($(MAKECMDGOALS),clean)" << endl;
	f << endl;
	f << "include $(SOURCES:.cpp=.d)" << endl;
	f << endl;
	f << "endif" << endl;

	f.close();
}

void	CCompilerCalculate::SubstituteIdentifiers()
{
	string	stack_element_h_guard;
	string	lexical_base_h_guard;
	string	parser_base_h_guard;

	CreateIncludeGuardIdentifier(	stack_element_h_name,
									stack_element_h_guard);

	CreateIncludeGuardIdentifier(	lexical_base_h_name,
									lexical_base_h_guard);

	CreateIncludeGuardIdentifier(	parser_base_h_name,
									parser_base_h_guard);

	string	b_prefix_u	= compiler_data->GetBasePrefixName();

	if (b_prefix_u.length())
		b_prefix_u[0] = toupper(b_prefix_u[0]);

	string	b_ = "C" + b_prefix_u;

	string	stack_class_id			= b_ + "Stack";
	string	lexical_base_class_id	= b_ + "LexicalBase";
	string	lexical_single_char_id	= b_ + "LexicalSingleChar";
	string	lexical_regexp_id		= b_ + "LexicalRegexp";
	string	parser_base_class_id	= b_ + "ParserBase";

	if (generate_lexical_source_files)
	{
		if (generate_base_source_files)
		{
			GlobalSubstitute(	lexical_base_h_path,
								"LEXICALDRIVER_H",
								lexical_base_h_guard);

			GlobalSubstitute(	lexical_base_h_path,
								"stackElement.h",
								stack_element_h_name);

			GlobalSubstitute(	lexical_base_h_path,
								"CStackElement",
								stack_class_id);

			GlobalSubstitute(	lexical_base_h_path,
								"CLexicalDriver",
								lexical_base_class_id);

			GlobalSubstitute(	lexical_base_h_path,
								"CLexicalSingleCharDriver",
								lexical_single_char_id);

			GlobalSubstitute(	lexical_base_h_path,
								"CLexicalRegexpDriver",
								lexical_regexp_id);

			GlobalSubstitute(	lexical_base_cpp_path,
								"lexicalDriver.h",
								lexical_base_h_name);

			GlobalSubstitute(	lexical_base_cpp_path,
								"CStackElement",
								stack_class_id);

			GlobalSubstitute(	lexical_base_cpp_path,
								"CLexicalDriver",
								lexical_base_class_id);

			GlobalSubstitute(	lexical_base_cpp_path,
								"CLexicalSingleCharDriver",
								lexical_single_char_id);

			GlobalSubstitute(	lexical_base_cpp_path,
								"CLexicalRegexpDriver",
								lexical_regexp_id);
		}

		if (compiler_data->GetLexicalFiniteStateMachineType() ==
											CCompilerData::RegularExpression)
		{
			GlobalSubstitute(	lexical_driver_h_path,
								"lexicalDriver.h",
								lexical_base_h_name);

			GlobalSubstitute(	lexical_driver_h_path,
								"CStackElement",
								stack_class_id);

			GlobalSubstitute(	lexical_driver_h_path,
								"CLexicalRegexpDriver",
								lexical_regexp_id);

			GlobalSubstitute(	lexical_driver_cpp_path,
								"CStackElement",
								stack_class_id);

			GlobalSubstitute(	lexical_driver_cpp_path,
								"CLexicalRegexpDriver",
								lexical_regexp_id);
		}
	}

	if (generate_parser_source_files)
	{
		if (generate_base_source_files)
		{
			GlobalSubstitute(	parser_base_h_path,
								"PARSERDRIVER_H",
								parser_base_h_guard);

			GlobalSubstitute(	parser_base_h_path,
								"lexicalDriver.h",
								lexical_base_h_name);

			GlobalSubstitute(	parser_base_h_path,
								"CStackElement",
								stack_class_id);

			GlobalSubstitute(	parser_base_h_path,
								"CLexicalDriver",
								lexical_base_class_id);

			GlobalSubstitute(	parser_base_h_path,
								"CParserDriver",
								parser_base_class_id);

			GlobalSubstitute(	parser_base_cpp_path,
								"parserDriver.h",
								parser_base_h_name);

			GlobalSubstitute(	parser_base_cpp_path,
								"CStackElement",
								stack_class_id);

			GlobalSubstitute(	parser_base_cpp_path,
								"CLexicalDriver",
								lexical_base_class_id);

			GlobalSubstitute(	parser_base_cpp_path,
								"CParserDriver",
								parser_base_class_id);
		}

		GlobalSubstitute(	parser_driver_h_path,
							"parserDriver.h",
							parser_base_h_name);

		GlobalSubstitute(	parser_driver_h_path,
							"CStackElement",
							stack_class_id);

		GlobalSubstitute(	parser_driver_h_path,
							"CParserDriver",
							parser_base_class_id);

		GlobalSubstitute(	parser_driver_cpp_path,
							"CStackElement",
							stack_class_id);

		GlobalSubstitute(	parser_driver_cpp_path,
							"CParserDriver",
							parser_base_class_id);
	}

	if (generate_base_source_files)
	{
		GlobalSubstitute(	stack_element_h_path,
							"STACKELEMENT_H",
							stack_element_h_guard);

		GlobalSubstitute(	stack_element_h_path,
							"CStackElement",
							stack_class_id);
	}

	if (generate_tty_source_files)
	{
		GlobalSubstitute(	tty_main_cpp_path,
							"CLexicalDriver",
							lexical_base_class_id);

		GlobalSubstitute(	tty_main_cpp_path,
							"CParserDriver",
							parser_base_class_id);
	}
}

int		CCompilerCalculate::GlobalSubstitute(	const string &file_path,
												const string &identifier,
												const string &replace)
{
	string		temporary_file_path;

	GetTemporaryFilePath(file_path, temporary_file_path);

	ifstream	in_file(file_path.c_str());
	ofstream	out_file(temporary_file_path.c_str());

	string				line;
	string::size_type	id_length = identifier.length();

	while (getline(in_file, line))
	{
		string				sline;
		string::size_type	l_length	= line.length();
		string::size_type	l_index;
		string::size_type	id_index	= 0;

		for (l_index=0; l_index<l_length; l_index++)
		{
			if (line[l_index] == identifier[id_index])
			{
				id_index++;

				if (id_index == id_length)
				{
					sline += replace;

					id_index = 0;
				}
			}
			else
			{
				if (id_index)
					sline += string(identifier, 0, id_index);

				if (line[l_index] == identifier[0])
				{
					id_index = 1;
				}
				else
				{
					id_index = 0;

					sline += line[l_index];
				}
			}
		}

		if (id_index)
			sline += string(identifier, 0, id_index);

		out_file << sline << endl;
	}

	in_file.close();
	out_file.close();

	unlink(file_path.c_str());
	rename(temporary_file_path.c_str(), file_path.c_str());

	return 0;
}

int		CCompilerCalculate::temporary_file_id = 0;

void	CCompilerCalculate::GetTemporaryFilePath(	const string &file_path,
													string &tfp)
{
	temporary_file_id++;

	while (1)
	{
		ostrstream		os;

		os << file_path << ".cpg" << temporary_file_id << '\0';

		if (!FilePath::FileExists(os.str()))
		{
			tfp = os.str();

			break;
		}

		temporary_file_id++;
	}
}
