/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#ifndef LEXICALDRIVER_H
#define LEXICALDRIVER_H

using namespace std;

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <iostream>
#include <fstream>
#include <strstream>
#include <string>
#include <map>

#include "tvaluelist.h"
#include "stackElement.h"

class CLexicalDriver
{
	public:
	enum LexicalScanType
	{
		SingleCharLexical,
		RegexpLexical,
		NullLexical
	};

	enum InputError
	{
		NoError		= 0,
		OpenError	= 1,
		CloseError	= 2,
		ReadError	= 4
	};

	CLexicalDriver(int est = 0);
	virtual ~CLexicalDriver();

	InputError	SetInputBuffer(const unsigned char *i_buffer, int i_size);
	InputError	SetInputString(const string &i_string);
	InputError	SetInputIoStream(		istream *i_stream,
										int b_size = default_block_size);
	InputError	SetInputFileIoStream(	const string &f_name,
										int b_size = default_block_size);
	InputError	SetInputStdStream(		FILE *f,
										int b_size = default_block_size);
	InputError	SetInputFileStdStream(	const string &f_name,
										int b_size = default_block_size);
	InputError	SetInputDescriptor(		int i_d,
										int b_size = default_block_size);
	InputError	SetInputFileDescriptor(	const string &f_name,
										int b_size = default_block_size);

	int		GetInputNull() { return 0; }
	int		GetInputData() { return (this->*input_function)(); }
	int		GetInputIoStreamData();
	int		GetInputStdStreamData();
	int		GetInputDescriptorData();
	int		(CLexicalDriver::* input_function)();

	InputError	CloseNull() { return NoError; }
	InputError	CloseInput();
	InputError	CloseIoStream();
	InputError	CloseFileIoStream();
	InputError	CloseStdStream();
	InputError	CloseFileStdStream();
	InputError	CloseDescriptor();
	InputError	CloseFileDescriptor();
	InputError	(CLexicalDriver::* close_function)();

	private:
	void				AllocateBuffer(int b_size);
	static const int	default_block_size;
	istream				*input_ios;
	FILE				*input_std;
	int					input_fd;

	public:
	virtual void	Initialize(int t_size = 100, bool ptf = true) = 0;
	virtual int		GetFilterOrToken() = 0;
	virtual int		GetFilterOrToken(	unsigned char *&terminal_ptr,
										int &terminal_size) = 0;
	virtual int		GetToken() = 0;
	virtual int		GetToken(unsigned char *&token_ptr, int &token_size) = 0;
	virtual void	LoadStackElement(CStackElement *se) = 0;

	virtual unsigned char	*GetTerminalString() = 0;
	virtual int				GetTerminalLength() = 0;
	virtual int				GetCurrentTerminalMatchIndex() = 0;
	virtual int				GetCurrentTerminalStateIndex() = 0;

	void	PushCurrentTokenBack()
	{ push_token_back_flag = true; }
	void	HaltProcessTerminal()
	{ process_terminal_flag = false; HaltProcessFilter(); }
	void	BreakOnProcessFilter()
	{ process_filter_break_flag = true; }
	void	SetCurrentToken(int t)
	{ current_token = t; }
	string	&GetFileName()
	{ return file_name; }
	void	SetParentLexical(CLexicalDriver *l)
	{ parent_lexical = l; }
	CLexicalDriver	*GetParentLexical()
	{ return parent_lexical; }
	string	GetSourceFileLocationString(bool one_based = true);
	string	GetIdentifierLocationString(const string &id, bool one_based = true);

	struct SSourceLocation
	{
		SSourceLocation(int h, int l, int c)
		: char_number(h), line_number(l), column_number(c) { }
		SSourceLocation() : char_number(), line_number(), column_number() { }

		int		char_number;
		int		line_number;
		int		column_number;
	};

	void	SetIdentifierLocation(const string &id, const SSourceLocation &l)
	{ identifier_location_map[id] = l; }
	void	SetIdentifierLocation(const string &id)
	{ SetIdentifierLocation(id, SSourceLocation(
		GetCurrentCharNumber(), GetCurrentLineNumber(), GetCurrentColumnNumber())); }
	SSourceLocation		&GetIdentifierLocation(const string &id)
	{ return identifier_location_map[id]; }

	virtual int		GetCurrentCharNumber()
	{ return current_char_number; }
	virtual int		GetCurrentLineNumber()
	{ return current_line_number; }
	virtual int		GetCurrentColumnNumber()
	{ return current_column_number; }
	virtual string	&GetLexicalStatusString(string &s, bool one_based = true);

	int		empty_symbol_token;
	int		lexical_error_code;
	int		filter_terminal;

	void	SetImplicitEndToken(const string &s)
	{ implicit_end_token_flag = true; implicit_end_token_string = s; }

	protected:
	virtual void	GetTerminal() = 0;
	virtual void	ProcessFilter() = 0;
	virtual void	HaltProcessFilter() = 0;
	unsigned char	*input_buffer;
	int				input_buffer_size;
	int				input_buffer_offset;
	int				characters_remaining;
	unsigned char	current_char;
	unsigned char	previous_char;
	int				token_count;
	int				filter_count;
	int				terminal_count;
	int				current_token;
	int				current_filter;
	bool			push_token_back_flag;
	bool			process_terminal_flag;
	bool			process_filter_break_flag;
	bool			implicit_end_token_flag;
	string			implicit_end_token_string;
	map<string, SSourceLocation>	identifier_location_map;
	string			file_name;
	CLexicalDriver	*parent_lexical;
	int				current_char_number;
	int				current_line_number;
	int				current_column_number;
};

class CLexicalSingleCharDriver : public CLexicalDriver
{
	public:
	CLexicalSingleCharDriver(	const char * const lex_char_token_names[],
								int est = 0);
	~CLexicalSingleCharDriver();

	virtual void	Initialize(int t_size = 100, bool ptf = true);
	virtual int		GetFilterOrToken();
	virtual int		GetFilterOrToken(	unsigned char *&terminal_ptr,
										int &terminal_size);
	virtual int		GetToken();
	virtual int		GetToken(unsigned char *&token_ptr, int &token_size);
	virtual void	LoadStackElement(CStackElement *se);

	virtual unsigned char	*GetTerminalString()
	{ return single_char_array_buffer; }
	virtual int				GetTerminalLength()
	{ return single_char_array_size; }
	virtual int				GetCurrentTerminalMatchIndex()
	{ return current_token; }
	virtual int				GetCurrentTerminalStateIndex()
	{ return single_char_array_buffer[0]; }

	protected:
	virtual void	ProcessFilter() { }
	virtual void	HaltProcessFilter() { }

	private:
	virtual void	GetTerminal();
	const int		map_size;
	int				*character_map;
	unsigned char	single_char_array_buffer[2];
	int				single_char_array_size;
};

class CLexicalRegexpDriver : public CLexicalDriver
{
	public:
	CLexicalRegexpDriver(	int	tkc, int ftc, int tmc,
							int est, int ir, int fsmc,
							const int *f_table,
							const int *rm_table,
							const int *pm_table,
							const int *tm_table,
							const int *sa_table,
							const int *tc_table,
							const int *t_map_table,
							const int *f_map_table,
							const int *lt_table,
							int sctm);
	~CLexicalRegexpDriver();

	virtual void	SetupFunctionTables() { }

	virtual void	Initialize(int t_size = 100, bool ptf = true);
	virtual int		GetFilterOrToken();
	virtual int		GetFilterOrToken(unsigned char *&terminal_ptr,
										int &terminal_size);
	virtual int		GetToken();
	virtual int		GetToken(unsigned char *&token_ptr, int &token_size);
	virtual void	LoadStackElement(CStackElement *se);

	virtual unsigned char	*GetTerminalString()
	{ return terminal_char_array_buffer; }
	virtual int				GetTerminalLength()
	{ return terminal_char_array_size; }
	virtual int				GetCurrentTerminalMatchIndex()
	{ return terminal_match_index; }
	virtual int				GetCurrentTerminalStateIndex()
	{ return terminal_state_index; }

	int	GetTokenMatchIndex(int token)	{ return inverse_token_map[token]; }
	int	GetFilterMatchIndex(int filter)	{ return inverse_filter_map[filter]; }

	int		GetLiteralTerminalFlag(int t) { return literal_terminal[t]; }
	int		GetCurrentLiteralTerminalFlag()
	{ return literal_terminal[terminal_match_index]; }

	virtual int		GetCurrentCharNumber();
	virtual int		GetCurrentLineNumber();
	virtual int		GetCurrentColumnNumber();

	protected:
	virtual void	ProcessFilter() { }
	virtual void	HaltProcessFilter() { }

	void	SetStartCondition(int sc)
	{ current_start_condition = sc; }

	private:
	int			input_range;
	int			fsm_count;
	const int	*fsm_table;
	const int	*regexp_match;
	const int	*prefix_match;
	const int	*token_match;
	const int	*start_assignment;
	const int	*trailing_context;
	const int	*token_map;
	const int	*filter_map;
	const int	*literal_terminal;
	int			single_column_token_match;

	int		*inverse_token_map;
	int		*inverse_filter_map;
	void	SetupInverseTokenMap();
	void	SetupInverseFilterMap();

	struct STerminalChar
	{
		STerminalChar() { }
		STerminalChar(unsigned char tc, int cn, int line, int column)
			: terminal_char(tc), char_number(cn),
				line_number(line), column_number(column) { }
		unsigned char	terminal_char;
		int				char_number;
		int				line_number;
		int				column_number;
		int				state_register;
		int				match_register;
	};

	virtual void						GetTerminal();
	void								GetTerminalCharList();
	TValueList<STerminalChar>			terminal_char_list;
	TValueList<STerminalChar>::listptr	terminal_char_listptr;
	unsigned char						*terminal_char_array_buffer;
	int									terminal_char_array_maximum;
	int									terminal_char_array_size;
	int									current_start_condition;
	int									terminal_match_index;
	int									terminal_state_index;
};

#endif
