/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#ifndef STRINGTOKEN_H
#define STRINGTOKEN_H

using namespace std;

#include <stdlib.h>
#include <errno.h>
#include <vector>
#include <map>
#include <string>
#include <iostream>

namespace StringToken
{
	enum TokenType
	{
		CharToken,
		IntegerToken,
		DoubleToken,
		MappedToken
	};

	enum TokenStatus
	{
		NoError				= 0,
		BadIntegerType		= 1,
		BadDoubleType		= 2,
		OutOfRange			= 4,
		NoCharMapping		= 8,
		NoTokenMap			= 16,
		NoData				= 32,
		InsufficientFields	= 64,
		ExcessFields		= 128
	};

	class CTokenValue
	{
		public:
		CTokenValue(const string &s, TokenStatus ts):
			token_string(s), token_status(ts) { }
		virtual ~CTokenValue() { }

		virtual TokenType	GetTokenType() const = 0;
		virtual ostream		&OutputData(ostream &os) const
		{ return os << GetTokenString(); }
		const string		&GetTokenString() const { return token_string; }
		TokenStatus			GetTokenStatus() const { return token_status; }
		virtual long int	GetIntegerValue() const { return 0; }
		virtual double		GetDoubleValue() const { return 0.0; }
		virtual long int	GetMappedValue() const { return 0; }

		private:
		string			token_string;
		TokenStatus		token_status;
	};

	class CCharToken: public CTokenValue
	{
		public:
		CCharToken(): CTokenValue("", NoData) { }
		CCharToken(const string &s, TokenStatus ts): CTokenValue(s, ts) { }
		virtual ~CCharToken() { }

		virtual TokenType	GetTokenType() const { return CharToken; }
		virtual ostream		&OutputData(ostream &os) const
		{ return os << GetTokenString(); }
	};

	class CIntegerToken: public CTokenValue
	{
		public:
		CIntegerToken(): CTokenValue("0", NoData), i_value(0) { }
		CIntegerToken(const string &s, TokenStatus ts, long int i):
			CTokenValue(s, ts), i_value(i) { }
		virtual ~CIntegerToken() { }

		virtual TokenType	GetTokenType() const { return IntegerToken; }
		virtual ostream		&OutputData(ostream &os) const
		{ return os << i_value; }
		virtual long int	GetIntegerValue() const { return i_value; }

		private:
		long int	i_value;
	};

	class CDoubleToken: public CTokenValue
	{
		public:
		CDoubleToken(): CTokenValue("0.0", NoData), x_value(0.0) { }
		CDoubleToken(const string &s, TokenStatus ts, double x):
			CTokenValue(s, ts), x_value(x) { }
		virtual ~CDoubleToken() { }

		virtual TokenType	GetTokenType() const { return DoubleToken; }
		virtual ostream		&OutputData(ostream &os) const
		{ return os << x_value; }
		virtual double		GetDoubleValue() const { return x_value; }

		private:
		double		x_value;
	};

	class CMappedToken: public CTokenValue
	{
		public:
		CMappedToken(): CTokenValue("", NoData), m_value(0) { }
		CMappedToken(const string &s, TokenStatus ts, long int m):
			CTokenValue(s, ts), m_value(m) { }
		virtual ~CMappedToken() { }

		virtual TokenType	GetTokenType() const { return MappedToken; }
		virtual ostream		&OutputData(ostream &os) const
		{ return os << m_value; }
		virtual long int	GetMappedValue() const { return m_value; }

		private:
		long int	m_value;
	};

	TokenStatus		GetCharTokenSequence(	const string &s, const string &d,
											vector<CTokenValue *> &tv);

	TokenStatus		GetIntegerTokenSequence(const string &s, const string &d,
											vector<CTokenValue *> &tv);

	TokenStatus		GetDoubleTokenSequence(	const string &s, const string &d,
											vector<CTokenValue *> &tv);

	TokenStatus		GetMappedTokenSequence(	const string &s, const string &d,
											const map<string, long int> &tm,
											vector<CTokenValue *> &tv);

	TokenStatus		GetTokenSequence(	const string &s, const string &d,
										const vector<TokenType> &tt,
										vector<CTokenValue *> &tv);

	TokenStatus		GetTokenSequence(	const string &s, const string &d,
										const vector<TokenType> &tt,
										const map<string, long int> &tm,
										vector<CTokenValue *> &tv);

	void			DeleteTokenVectorElements(vector<CTokenValue *> &tv);

	typedef pair<string::size_type, string::size_type>		TSubstringInfo;
	typedef vector<TSubstringInfo>							TInfoVector;

	void	GetInfoVector(const string &s, const string &d, TInfoVector &iv);

	CCharToken		*CreateNewCharToken(	const string &s,
											const TSubstringInfo &si);

	CIntegerToken	*CreateNewIntegerToken(	const string &s,
											const TSubstringInfo &si);

	CDoubleToken	*CreateNewDoubleToken(	const string &s,
											const TSubstringInfo &si);

	CMappedToken	*CreateNewMappedToken(	const string &s,
											const TSubstringInfo &si,
											const map<string, long int> &tm);
};

ostream		&operator<<(ostream &s, const StringToken::CTokenValue &tv);
ostream		&operator<<(ostream &s,
				const vector<StringToken::CTokenValue *> &tvv);

#endif
