/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#ifndef TVALUELIST_H
#define TVALUELIST_H

template <class T>
class TValueList
{
	public:
	struct SElement
	{
		SElement	*next;
		SElement	*previous;
		T			data;
	};

	typedef typename TValueList<T>::SElement*	listptr;

	TValueList(int b_size = 100);
	~TValueList();

	T			&Data(SElement *element)		{ return element->data; }
	T			Data(SElement *element) const	{ return element->data; }
	int			Size() const					{ return list_count; }
	T			&FirstData()					{ return first->data; }
	T			FirstData() const				{ return first->data; }
	SElement	*FirstListPtr() const			{ return first; }
	T			&LastData()						{ return last->data; }
	T			LastData() const				{ return last->data; }
	SElement	*LastListPtr() const			{ return last; }
	int			Empty() const					{ return list_count; }

	SElement	*PushFront(const T &data);
	SElement	*PushBack(const T &data);
	SElement	*FindData(T &value, SElement *start = 0) const;
	SElement	*FindIndex(int index) const;
	SElement	*NextListPtr(SElement *&element) const ;
	SElement	*PreviousListPtr(SElement *&element) const ;
	T			&NextData(SElement *&element);
	T			NextData(SElement *&element) const;
	T			&PreviousData(SElement *&element);
	T			PreviousData(SElement *&element) const;
	SElement	*InsertAfter(SElement *element, const T &data);
	SElement	*InsertBefore(SElement *element, const T &data);
	void		Clear();
	SElement	*Erase(SElement *element);
	SElement	*PopFront();
	SElement	*PopFront(int count);
	SElement	*PopBack();
	SElement	*PopBack(int count);
	SElement	*SetData(SElement *element, const T &data);

	private:

	struct SBlock
	{
		SBlock		*next;
		SElement	*element_array;
	};

	typedef typename TValueList<T>::SBlock*		blockptr;
	
	int			block_size;
	SBlock		*allocate_block();
	SBlock		*first_block;
	SBlock		*last_block;
	SElement	*allocate_element();
	void		free_element(SElement *element);
	SElement	*free_list;
	int			list_count;
	SElement	*first;
	SElement	*last;

	TValueList	&operator=(const TValueList &l);
};

typedef		TValueList<int>							VListInt;
typedef		TValueList<int>::listptr				VListIntPtr;
typedef		TValueList<char>						VListChar;
typedef		TValueList<char>::listptr				VListCharPtr;
typedef		TValueList<char *>						VListPtrChar;
typedef		TValueList<char *>::listptr				VListPtrCharPtr;
typedef		TValueList<unsigned char>				VListUChar;
typedef		TValueList<unsigned char>::listptr		VListUCharPtr;
typedef		TValueList<unsigned char *>				VListPtrUChar;
typedef		TValueList<unsigned char *>::listptr	VListPtrUCharPtr;

template <class T> inline
typename TValueList<T>::SBlock		*TValueList<T>::allocate_block()
{
	SBlock	*block = new SBlock;

	block->next				= 0;
	SElement *e_array		= new SElement[block_size];
	block->element_array	= e_array;

	for (int i=0; i<block_size-1; i++)
	{
		e_array[i].next	= e_array + (i + 1);
//		e_array[i].data	= T();
	}

	e_array[block_size - 1].next	= 0;
//	e_array[block_size - 1].data	= T();

	free_list = e_array;

	return block;
}

template <class T> inline
typename TValueList<T>::SElement	*TValueList<T>::allocate_element()
{
	if (!free_list)
	{
		SBlock	*block		= allocate_block();
		last_block->next	= block;
		last_block			= block;
	}

	SElement	*element = free_list;

	free_list = free_list->next;

	return element;
}

template <class T> inline
void	TValueList<T>::free_element(SElement *element)
{
	if (!free_list)
		element->next	= 0;
	else
		element->next	= free_list;

	free_list = element;
}

template <class T>
TValueList<T>::TValueList(int b_size)
{
	if (b_size < 2)
		block_size = 2;
	else
		block_size = b_size;

	first_block = allocate_block();
	last_block	= first_block;

	list_count	= 0;
	first		= 0;
	last		= 0;
}

template <class T>
TValueList<T>::~TValueList()
{
	SBlock	*current_block = first_block;
	SBlock	*next_block;

	while (current_block)
	{
		next_block = current_block->next;
		delete [] current_block->element_array;
		delete current_block;
		current_block = next_block;
	}
}

template <class T>
typename TValueList<T>::SElement	*TValueList<T>::PushFront(const T &data)
{
	SElement	*element = allocate_element();

	if (!first)
	{
		first			= element;
		last			= element;

		first->next		= 0;
		first->previous	= 0;
		first->data		= data;
	}
	else
	{
		element->next		= first;
		element->previous	= 0;
		element->data		= data;

		first->previous	= element;
		first			= element;
	}

	list_count++;

	return first;
}

template <class T>
typename TValueList<T>::SElement	*TValueList<T>::PushBack(const T &data)
{
	SElement	*element = allocate_element();

	if (!last)
	{
		first			= element;
		last			= element;

		last->next		= 0;
		last->previous	= 0;
		last->data		= data;
	}
	else
	{
		element->next		= 0;
		element->previous	= last;
		element->data		= data;

		last->next		= element;
		last			= element;
	}

	list_count++;

	return last;
}

template <class T>
typename TValueList<T>::SElement
	*TValueList<T>::FindData(T &value, SElement *start) const
{
	SElement	*current_element;

	if (!start)
		current_element = first;
	else
		current_element = start;

	while (current_element)
	{
		if (value == current_element->data)
			return current_element;

		current_element = current_element->next;
	}

	return 0;
}

template <class T>
typename TValueList<T>::SElement	*TValueList<T>::FindIndex(int index) const
{
	if (index >= list_count || index < 0)
		return 0;

	SElement	*current_element = first;

	for (int i=0; i<index; i++)
		current_element = current_element->next;

	return current_element;
}

template <class T> inline
typename TValueList<T>::SElement
	*TValueList<T>::NextListPtr(SElement *&element) const
{
	if (element == last)
		element = 0;
	else
		element = element->next;
		
	return element;
}

template <class T> inline
typename TValueList<T>::SElement
	*TValueList<T>::PreviousListPtr(SElement *&element) const
{
	if (element == first)
		element = 0;
	else
		element = element->previous;
		
	return element;
}

template <class T> inline
T	&TValueList<T>::NextData(SElement *&element)
{
	T	&data = element->data;

	element = element->next;

	return data;
}

template <class T> inline
T	TValueList<T>::NextData(SElement *&element) const
{
	T	data = element->data;

	element = element->next;

	return data;
}

template <class T> inline
T	&TValueList<T>::PreviousData(SElement *&element)
{
	T	&data = element->data;

	element = element->previous;

	return data;
}

template <class T> inline
T	TValueList<T>::PreviousData(SElement *&element) const
{
	T	data = element->data;

	element = element->previous;

	return data;
}

template <class T>
typename TValueList<T>::SElement
	*TValueList<T>::InsertAfter(SElement *element, const T &data)
{
	if (!element)
		return PushBack(data);

	SElement	*next_element = element->next;

	if (!next_element)
		return PushBack(data);

	SElement	*insert_element = allocate_element();

	element->next				= insert_element;
	insert_element->next		= next_element;
	insert_element->previous	= element;
	insert_element->data		= data;
	next_element->previous		= insert_element;

	list_count++;

	return insert_element;
}

template <class T>
typename TValueList<T>::SElement
	*TValueList<T>::InsertBefore(SElement *element, const T &data)
{
	if (!element)
		return PushFront(data);

	SElement	*previous_element = element->previous;

	if (!previous_element)
		return PushFront(data);

	SElement	*insert_element = allocate_element();

	previous_element->next		= insert_element;
	insert_element->next		= element;
	insert_element->previous	= previous_element;
	insert_element->data		= data;
	element->previous			= insert_element;

	list_count++;

	return insert_element;
}

template <class T> inline
void	TValueList<T>::Clear()
{
	PopBack(list_count);
}

template <class T>
typename TValueList<T>::SElement	*TValueList<T>::Erase(SElement *element)
{
	SElement	*previous_element = element->previous;

	if (!previous_element)
		return PopFront();

	SElement	*next_element = element->next;

	if (!next_element)
		return PopBack();

	free_element(element);

	previous_element->next	= next_element;
	next_element->previous	= previous_element;

	list_count--;

	return next_element;
}

template <class T>
typename TValueList<T>::SElement	*TValueList<T>::PopFront()
{
	if (!first)
		return 0;

	SElement	*next_element = first->next;

	free_element(first);

	if (!next_element)
	{
		first	= 0;
		last	= 0;
	}
	else
	{
		first			= next_element;
		first->previous	= 0;
	}

	list_count--;

	return first;
}

template <class T>
typename TValueList<T>::SElement	*TValueList<T>::PopFront(int count)
{
	int		remove_count =	count < list_count ?
							count : list_count;

	if (remove_count <= 0)
		return first;

	SElement	*next_element;

	for (int i=0; i<remove_count; i++)
	{
		next_element = first->next;

		free_element(first);

		first = next_element;
	}

	if (!first)
		last = 0;
	else
		first->previous = 0;

	list_count -= remove_count;

	return first;
}

template <class T>
typename TValueList<T>::SElement	*TValueList<T>::PopBack()
{
	if (!last)
		return 0;

	SElement	*previous_element = last->previous;

	free_element(last);

	if (!previous_element)
	{
		first	= 0;
		last	= 0;
	}
	else
	{
		last		= previous_element;
		last->next	= 0;
	}

	list_count--;

	return last;
}

template <class T>
typename TValueList<T>::SElement	*TValueList<T>::PopBack(int count)
{
	int		remove_count =	count < list_count ?
							count : list_count;

	if (remove_count <= 0)
		return last;

	SElement	*previous_element;

	for (int i=0; i<remove_count; i++)
	{
		previous_element = last->previous;

		free_element(last);

		last = previous_element;
	}

	if (!last)
		first = 0;
	else
		last->next = 0;

	list_count -= remove_count;

	return last;
}

template <class T> inline
typename TValueList<T>::SElement
	*TValueList<T>::SetData(SElement *element, const T &data)
{
	element->data = data;

	return element;
}

#endif
