/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#ifndef AUTOMATON_H
#define	AUTOMATON_H

#include <string.h>

#include "tlist.h"
#include "tvaluelist.h"
#include "memoryAllocator.h"
#include "regexpSingleCharData.h"
#include "lexicalDriver.h"
#include "regexpParserDriver.h"

class CTerminalInformation
{
	public:
	CTerminalInformation(): name(0), regexp(0) { }
	~CTerminalInformation() { delete [] name; delete [] regexp; }

	void	Initialize(int i, bool ff, bool lf, const char *n, const char *r)
	{
		symbol_index	= i;
		filter_flag		= ff;
		literal_flag	= lf;

		int	length = strlen(n);
		name = new char[length + 1];
		strcpy(name, n);

		length = strlen(r);
		regexp = new char[length + 1];
		strcpy(regexp, r);
	}

	int		symbol_index;
	bool	filter_flag;
	bool	literal_flag;
	char	*name;
	char	*regexp;
};

class CNdTransition
{
	public:
	CNdTransition(int s) : states(10)
	{ symbol = s; }

	CNdTransition(int s, int st) : states(10)
	{ symbol = s; states.PushBack(st); }

	CNdTransition(const CNdTransition &tr) : states(10)
	{
		symbol = tr.symbol;

		VListInt::listptr	tr_st_listptr = tr.states.FirstListPtr();

		while (tr_st_listptr)
			states.PushBack(tr.states.NextData(tr_st_listptr));
	}

	~CNdTransition() { }

	void			AddNewState(int s) { states.PushBack(s); }
	CNdTransition	&operator+=(const CNdTransition &tr);
	bool			operator<(const CNdTransition &tr)
					{ if (symbol < tr.symbol) return true; else return false; }
	bool			operator>(const CNdTransition &tr)
					{ if (symbol > tr.symbol) return true; else return false; }
	bool			operator==(const CNdTransition &tr)
					{ if (symbol == tr.symbol) return true; else return false; }

	int			symbol;
	VListInt	states;
};

typedef		TValueList<CNdTransition *>		VListNdTrans;
typedef		VListNdTrans::listptr			VListNdTransPtr;

class CNdState
{
	public:
	CNdState() : transitions(10), prefix_final_flag(false) { }

	CNdState(const CNdState &st) : transitions(10)
	{
		state_number		= st.state_number;
		final_flag			= st.final_flag;
		prefix_final_flag	= st.prefix_final_flag;

		VListNdTransPtr	st_tr_listptr = st.transitions.FirstListPtr();

		while (st_tr_listptr)
		{
			CNdTransition	*tr = st.transitions.NextData(st_tr_listptr);
			transitions.PushBack(new CNdTransition(*tr));
		}
	}

	~CNdState() { }
	void	DeleteObjects()
	{
		VListNdTransPtr	t_listptr = transitions.FirstListPtr();
		while (t_listptr)
			delete transitions.NextData(t_listptr);
	}

	void		AddNewTransition(CNdTransition *tr) { transitions.PushBack(tr); }
	CNdState	&operator+=(const CNdState &st);

	int				state_number;
	bool			final_flag;
	bool			prefix_final_flag;
	VListNdTrans	transitions;
};

typedef		TValueList<CNdState *>		VListNdState;
typedef		VListNdState::listptr		VListNdStatePtr;

class CNdAutomaton
{
	public:
	CNdAutomaton() { }

	CNdAutomaton(CNdAutomaton &a)
	{
		CNdState	*start_state = a.GetStartState();
		AddNewStartState(new CNdState(*start_state));
		a.RemoveStartState();

		VListNdStatePtr	st_listptr = a.states.FirstListPtr();

		while (st_listptr)
		{
			CNdState	*st = a.states.NextData(st_listptr);
			states.PushBack(new CNdState(*st));
		}

		a.AddNewStartState(start_state);
	}

	~CNdAutomaton() { }
	void	DeleteObjects()
	{
		VListNdStatePtr	s_listptr = states.FirstListPtr();
		while (s_listptr)
		{
			CNdState	*st = states.NextData(s_listptr);
			st->DeleteObjects();
			delete st;
		}
	}

	void		AddNewState(CNdState *st) { states.PushBack(st); }
	void		AddNewStartState(CNdState *st)
				{ start_state_listptr = states.PushFront(st); }
	CNdState	*GetStartState() { return states.Data(start_state_listptr); }
	void		RemoveStartState() { states.Erase(start_state_listptr); }

	void		SetPrefixFinalFlags()
	{
		VListNdStatePtr	s_listptr = states.FirstListPtr();
		while (s_listptr)
		{
			CNdState	*st = states.NextData(s_listptr);

			if (st->final_flag)
				st->prefix_final_flag = true;
		}
	}

	VListNdStatePtr	start_state_listptr;
	VListNdState	states;

	class CStateMap
	{
		public:
		struct SPair
		{
			SPair() { }
			SPair(int o, int n) { old_state = o; new_state = n; }
			int		old_state;
			int		new_state;
		};

		CStateMap(int h_size)
		{
			hash_size	= h_size;
			hash_lists	= new TValueList<SPair>[hash_size];
		}

		~CStateMap()
		{
			delete [] hash_lists;
		}

		void	Clear()
		{
			for (int i=0; i<hash_size; i++)
				hash_lists[i].Clear();
		}

		int		ReturnNewState(int old)
		{
			int		index = old % hash_size;

			TValueList<SPair>			&hl			= hash_lists[index];
			TValueList<SPair>::listptr	hl_listptr	= hl.FirstListPtr();

			while (hl_listptr)
			{
				SPair	&p = hl.NextData(hl_listptr);

				if (p.old_state == old)
					return p.new_state;
			}

			int		ns = new_state++;

			hl.PushBack(SPair(old, ns));

			return ns;
		}

		int					hash_size;
		TValueList<SPair>	*hash_lists;
	};

	static int			new_state;
	static int			input_range;
	static CStateMap	state_map;
};

CNdAutomaton	*NdSymbol(int s);
CNdAutomaton	*NdSymbolString(const VListInt &s_string);
CNdAutomaton	*NdSymbolClass(const VListInt &c_set);
CNdAutomaton	*NdSymbolComplement(const VListInt &c_set);
CNdAutomaton	*NdCat(CNdAutomaton *nd1, CNdAutomaton *nd2);
CNdAutomaton	*NdOr(CNdAutomaton *nd_array[], int count);
CNdAutomaton	*NdClone(CNdAutomaton *nd1);
CNdAutomaton	*NdEmpty();
CNdAutomaton	*NdZeroOrMore(CNdAutomaton *nd1);
CNdAutomaton	*NdMOrMore(CNdAutomaton *nd1, int m);
CNdAutomaton	*NdZeroToN(CNdAutomaton *nd1, int n);
CNdAutomaton	*NdMToN(CNdAutomaton *nd1, int m, int n);

class CContext
{
	public:
	CContext(CNdAutomaton *a)
		: nd(a), start_activations(10), start_assignments(10) { }
	~CContext()
	{
		if (nd)
		{
			nd->DeleteObjects();
			delete nd;
		}

		VListPtrCharPtr		listptr = start_activations.FirstListPtr();

		while (listptr)
			delete [] start_activations.NextData(listptr);

		listptr = start_assignments.FirstListPtr();

		while (listptr)
			delete [] start_assignments.NextData(listptr);
	}

	void	SetStartActivations(VListPtrChar &s_list);
	void	SetStartAssignments(VListPtrChar &s_list);

	CNdAutomaton		*nd;
	VListPtrChar		start_activations;
	VListPtrChar		start_assignments;
};

class CFSMState
{
	public:
	struct SPair
	{
		SPair() { }
		SPair(int s, int f) { symbol = s; fsm_number = f; }
		int		symbol;
		int		fsm_number;
	};

	CFSMState(int fsmn, VListInt &ndsl)
	{
		fsm_number	= fsmn;

		VListIntPtr		listptr = ndsl.FirstListPtr();

		while (listptr)
			nd_state_list.PushBack(ndsl.NextData(listptr));
	}
	~CFSMState() { }

	int					fsm_number;
	VListInt			nd_state_list;
	TValueList<SPair>	fsm_transitions;
};

typedef	TList<CFSMState>				ListFSMS;
typedef	TValueList<ListFSMS::listptr>	VListListFSMSptr;

class CAutomaton
{
	public:
	CAutomaton(CLexicalDriver::LexicalScanType st, int range,
				int rc, CTerminalInformation *t, int esi, VListPtrChar &s_list)
	{
		scan_type			= st;
		input_range			= range;
		regexp_count		= rc;
		terminals			= t;
		empty_symbol_index	= esi;

		SetStartContext(s_list);

		lowest_state_number	= new int[regexp_count + 1];

		context_array	= new (CContext *[regexp_count]);

		int	i;
		for (i=0; i<regexp_count; i++)
			context_array[i] = 0;

		current_char_number = new int[regexp_count];

		nd_or				= 0;
		nd_states			= 0;
		final_states		= 0;
		prefix_final_states	= 0;
		fsm_states			= 0;

		fsm_table			= 0;
		regexp_match		= 0;
		prefix_match		= 0;
		terminal_match		= 0;
		start_assignment	= 0;
		trailing_context	= 0;
		token_map			= 0;
		filter_map			= 0;
		literal_terminal	= 0;
	}

	~CAutomaton()
	{
		VListPtrCharPtr		listptr = start_context.FirstListPtr();

		while (listptr)
			delete [] start_context.NextData(listptr);

		delete [] terminals;
		delete [] lowest_state_number;

		int	i;
		for (i=0; i<regexp_count; i++)
			delete context_array[i];

		delete [] context_array;
		delete [] current_char_number;

		if (nd_or)
		{
			nd_or->DeleteObjects();
			delete nd_or;
		}

		delete [] nd_states;
		delete [] final_states;
		delete [] prefix_final_states;
		delete fsm_states;

		delete [] fsm_table;
		delete [] regexp_match;
		delete [] prefix_match;
		delete [] terminal_match;
		delete [] start_assignment;
		delete [] trailing_context;
		delete [] token_map;
		delete [] filter_map;
		delete [] literal_terminal;
	}

	void	SetStartContext(VListPtrChar &s_list);
	bool	CalculateLexical();
	bool	StartAssignmentFlag();
	bool	TrailingContextFlag();
	bool	ParseNdAutomata();
	void	CalculateAutomata();

	CLexicalDriver::LexicalScanType		scan_type;

	int						input_range;
	int						regexp_count;
	CTerminalInformation	*terminals;
	int						empty_symbol_index;
	VListPtrChar			start_context;

	int					token_count;
	int					filter_count;
	int					*lowest_state_number;
	CContext			**context_array;
	bool				all_tokens_compiled;
	int					*current_char_number;

	CNdAutomaton		*nd_or;
	int					number_of_nd_states;
	CNdState			**nd_states;
	VListInt			*final_states;
	VListInt			*prefix_final_states;
	ListFSMS			*fsm_states;

	int					fsm_count;
	int					*fsm_table;
	int					*regexp_match;
	int					*prefix_match;
	int					*terminal_match;
	int					*start_assignment;
	int					*trailing_context;
	int					*token_map;
	int					*filter_map;
	int					*literal_terminal;
};

bool	IntersectionTestInt(VListInt &s1, VListInt &s2);
bool	IntersectionTestChar(VListPtrChar &s1, VListPtrChar &s2);

class CFSMStateNode
{
	public:
	CFSMStateNode(CFSMStateNode *p, int ndsn, bool fsmsf)
		: initial_list_size(10), descendants(initial_list_size)
	{
		parent				= p;
		nd_state_number		= ndsn;
		fsm_state_flag		= fsmsf;
	}

	~CFSMStateNode() { }

	CFSMStateNode	*InsertOrMatch()
	{
		if (nd_state_listptr)
		{
			int		number = nd_state_list->Data(nd_state_listptr);

			TList<CFSMStateNode>::listptr	descendant_listptr;

			descendant_listptr = descendants.FirstListPtr();

			while (descendant_listptr)
			{
				CFSMStateNode	*dp = descendants.NextData(descendant_listptr);

				if (number == dp->nd_state_number)
				{
					nd_state_list->NextData(nd_state_listptr);
					return dp;
				}
			}

			CFSMStateNode	*child = new CFSMStateNode(this, number, false);

			descendants.PushBack(child);

			nd_state_list->NextData(nd_state_listptr);
			return child;
		}
		else
		{
			final_node			= this;
			existing_fsm_state	= fsm_state_flag;

			return 0;
		}
	}

	void	*operator new(size_t size)
	{ return allocator->allocate_element(size); }

	void	operator delete(void *p)
	{ allocator->release_element(p); }

	int						nd_state_number;
	bool					fsm_state_flag;
	ListFSMS::listptr		fsm_state_listptr;
	CFSMStateNode			*parent;
	const int				initial_list_size;
	TList<CFSMStateNode>	descendants;

	static VListInt 			*nd_state_list;
	static VListInt::listptr	nd_state_listptr;
	static CFSMStateNode		*final_node;
	static bool					existing_fsm_state;
	static CMemoryAllocator		*allocator;
	static void					SetupAllocator(int block_size);
};

#endif
