/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "compilerData.h"

int		CCompilerData::index_upper_bound;

namespace OutputWidths
{
	int		terminal_index_width;
	int		terminal_name_width;
	int		error_token_index_width;
	int		error_token_name_width;
	int		nonterminal_index_width;
	int		nonterminal_name_width;
	int		symbol_index_width;
	int		symbol_name_width;
	int		rule_index_width;
	int		rule_left_name_width;
	int		precedence_index_width;
	int		conflict_index_width;
	int		error_entry_width;
};

CTerminal::CTerminal(const string &s)
	: CSymbol(-1)
{
	StringToken::TokenStatus				token_status;
	vector<StringToken::CTokenValue *>		token_vector;

	token_status = StringToken::GetCharTokenSequence(
						s, string("\t "), token_vector);

	if (token_status == StringToken::NoError && token_vector.size() >= 5)
	{
		int				t_index = -1;
		istrstream		is(token_vector[0]->GetTokenString().c_str());

		is >> t_index;

		if (t_index >= 0 && t_index < CCompilerData::index_upper_bound)
		{
			string	s1(token_vector[1]->GetTokenString());
			string	s2(token_vector[2]->GetTokenString());

			bool	b11 = s1 == "FILTER";
			bool	b12 = s1 == "TOKEN";
			bool	b21 = s2 == "LITERAL";
			bool	b22 = s2 == "REGEXP";

			if ((b11 || b12) && (b21 || b22))
			{
				string	s3(token_vector[3]->GetTokenString());

				StringToken::TInfoVector	info_vector;

				StringToken::GetInfoVector(s, string("\t "), info_vector);

				string::size_type	n_index = info_vector[3].first;

				string::size_type	r_index =
					s.find_first_not_of("\t ", n_index + s3.length());

				SetIndex(t_index);
				SetName(s3);
				SetFilterFlag(b11);
				SetLiteralFlag(b21);
				SetRegexp(s.substr(r_index));
				SetCodeFlag(false);
			}
		}
	}

	StringToken::DeleteTokenVectorElements(token_vector);
}

CErrorToken::CErrorToken(const string &s)
	: CSymbol(-1)
{
	StringToken::TokenStatus				token_status;
	vector<StringToken::TokenType>			token_types;
	vector<StringToken::CTokenValue *>		token_vector;

	token_types.push_back(StringToken::IntegerToken);
	token_types.push_back(StringToken::CharToken);
	token_types.push_back(StringToken::IntegerToken);

	token_status = StringToken::GetTokenSequence(	s, "\t ",
													token_types, token_vector);

	if (token_status == StringToken::NoError)
	{
		int		e_index = token_vector[0]->GetIntegerValue();

		if (e_index >= 0 && e_index < CCompilerData::index_upper_bound)
		{
			SetIndex(e_index);
			SetName(token_vector[1]->GetTokenString());
			SetSynchronizationLength(token_vector[2]->GetIntegerValue());
		}
	}

	StringToken::DeleteTokenVectorElements(token_vector);
}

CNonTerminal::CNonTerminal(const string &s)
	: CSymbol(-1)
{
	StringToken::TokenStatus				token_status;
	vector<StringToken::TokenType>			token_types;
	vector<StringToken::CTokenValue *>		token_vector;

	token_types.push_back(StringToken::IntegerToken);
	token_types.push_back(StringToken::CharToken);
	token_types.push_back(StringToken::CharToken);

	token_status = StringToken::GetTokenSequence(	s, "\t ",
													token_types, token_vector);

	if (token_status == StringToken::NoError)
	{
		int		n_index = token_vector[0]->GetIntegerValue();

		if (n_index >= 0 && n_index < CCompilerData::index_upper_bound &&
			token_vector[2]->GetTokenString() == "START")
		{
			SetIndex(n_index);
			SetName(token_vector[1]->GetTokenString());
			SetStartSymbolFlag(true);
		}
	}
	else if (	token_status ==
					(StringToken::InsufficientFields | StringToken::NoData) &&
				token_vector[0]->GetTokenStatus() == StringToken::NoError &&
				token_vector[1]->GetTokenStatus() == StringToken::NoError)
	{
		int		n_index = token_vector[0]->GetIntegerValue();

		if (n_index >= 0 && n_index < CCompilerData::index_upper_bound)
		{
			SetIndex(n_index);
			SetName(token_vector[1]->GetTokenString());
			SetStartSymbolFlag(false);
		}
	}

	StringToken::DeleteTokenVectorElements(token_vector);
}

CRule::CRule(const string &s, CCompilerData *d)
	: index(-1), unit_rule_elimination_flag(false), left(0),
		undefined_symbol_flag(false), code_flag(false)
{
	StringToken::TokenStatus						token_status;
	vector<StringToken::CTokenValue *>				token_vector;
	vector<StringToken::CTokenValue *>::size_type	v_index;

	token_status = StringToken::GetCharTokenSequence(
						s, string("\t "), token_vector);

	if (token_status == StringToken::NoError && token_vector.size() >= 3)
	{
		int				r_index = -1;
		istrstream		is(token_vector[0]->GetTokenString().c_str());

		is >> r_index;

		if (r_index >= 0 && r_index < CCompilerData::index_upper_bound)
		{
			if (token_vector[2]->GetTokenString() == "->")
			{
				index = r_index;

				left = d->FindSymbolName(token_vector[1]->GetTokenString());

				if (!left)
				{
					left					= &undefined_symbol;
					undefined_symbol_flag	= true;
				}

				for (v_index=3; v_index<token_vector.size(); v_index++)
				{
					const CSymbol	*r_symbol =
						d->FindSymbolName(
								token_vector[v_index]->GetTokenString());

					if (!r_symbol)
					{
						r_symbol				= &undefined_symbol;
						undefined_symbol_flag	= true;
					}

					right.push_back(r_symbol);
				}
			}
			else if (token_vector.size() == 5 &&
						token_vector[1]->GetTokenString() == "*" &&
						token_vector[3]->GetTokenString() == "->")
			{
				index = r_index;

				unit_rule_elimination_flag = true;

				left = d->FindSymbolName(token_vector[2]->GetTokenString());

				if (!left)
				{
					left					= &undefined_symbol;
					undefined_symbol_flag	= true;
				}

				for (v_index=4; v_index<token_vector.size(); v_index++)
				{
					const CSymbol	*r_symbol =
						d->FindSymbolName(
								token_vector[v_index]->GetTokenString());

					if (!r_symbol)
					{
						r_symbol				= &undefined_symbol;
						undefined_symbol_flag	= true;
					}

					right.push_back(r_symbol);
				}
			}
		}
	}

	StringToken::DeleteTokenVectorElements(token_vector);
}

bool	CRule::ReplaceSymbol(const CSymbol *current, const CSymbol *new_symbol)
{
	bool	replace_flag = false;

	if (left == current)
	{
		left			= new_symbol;
		replace_flag	= true;
	}

	vector<const CSymbol *>::size_type		r_index;

	for (r_index=0; r_index<right.size(); r_index++)
	{
		if (right[r_index] == current)
		{
			right[r_index]	= new_symbol;
			replace_flag	= true;
		}
	}

	if (replace_flag && new_symbol->IsUndefined())
		undefined_symbol_flag = true;

	return replace_flag;
}

string	&CRule::GetLeftString(string &s) const
{
	if (left)
		s = left->GetName();
	else
		s = "";

	return s;
}

string	&CRule::GetRightString(string &s) const
{
	if (right.size())
		s = right[0]->GetName();
	else
		s = "";

	vector<const CSymbol *>::size_type		r_index;

	for (r_index=1; r_index<right.size(); r_index++)
	{
		s += " ";
		s += right[r_index]->GetName();
	}

	return s;
}

string	&CRule::GetRightLessOne(string &s)
{
	right.pop_back();

	return GetRightString(s);
}

string	&CRule::GetStringPadded(string &s, int rule_left_name_width) const
{
	if (left)
	{
		s = left->GetName();

		string	pad(rule_left_name_width - s.length() + 1, ' ');

		s += pad;
	}
	else
		s = " ";

	s += "->";

	vector<const CSymbol *>::size_type		r_index;

	for (r_index=0; r_index<right.size(); r_index++)
	{
		s += " ";
		s += right[r_index]->GetName();
	}

	return s;
}

string	&CRule::GetStringIndexed(string &s) const
{
	if (left)
		s = left->GetName();
	else
		s = "";

	s += " ->";

	vector<const CSymbol *>::size_type		r_index;

	for (r_index=0; r_index<right.size(); r_index++)
	{
		ostrstream		os;

		os << r_index << '\0';

		s += " ";
		s += right[r_index]->GetName();
		s += "(";
		s += os.str();
		s += ")";
	}

	return s;
}

CPrecedence::CPrecedence(const string &s, CCompilerData *d)
	: index(-1), associativity(NullAssociative), undefined_entry_flag(false)
{
	StringToken::TokenStatus						token_status;
	vector<StringToken::CTokenValue *>				token_vector;
	vector<StringToken::CTokenValue *>::size_type	v_index;

	token_status = StringToken::GetCharTokenSequence(
						s, string("\t "), token_vector);

	if (token_status == StringToken::NoError && token_vector.size() >= 3)
	{
		int				p_index = -1;
		istrstream		is(token_vector[0]->GetTokenString().c_str());

		is >> p_index;

		if (p_index >= 0 && p_index < CCompilerData::index_upper_bound)
		{
			string	s1(token_vector[1]->GetTokenString());

			bool	b1 = s1 == "LEFT";
			bool	b2 = s1 == "RIGHT";
			bool	b3 = s1 == "NONASSOCIATIVE";

			if (b1 || b2 || b3)
			{
				index = p_index;

				if (b1)
					associativity = LeftAssociative;
				else if (b2)
					associativity = RightAssociative;
				else if (b3)
					associativity = NonAssociative;

				CTerminal		*terminal;
				CErrorToken		*error_token;
				CRule			*rule;

				for (v_index=2; v_index<token_vector.size(); v_index++)
				{
					string	ps(token_vector[v_index]->GetTokenString());

					if (terminal = d->FindTerminalName(ps))
					{
						entries.push_back(SPrecedenceEntry(terminal));
					}
					else if (error_token = d->FindErrorTokenName(ps))
					{
						entries.push_back(SPrecedenceEntry(error_token));
					}
					else
					{
						if (ps.substr(0,5) == "RULE:")
						{
							int				r_index = -1;
							string			index_string(ps.substr(5));
							istrstream		rs(index_string.c_str());

							rs >> r_index;

							if (rule = d->FindRuleIndex(r_index))
							{
								entries.push_back(SPrecedenceEntry(rule));
							}
							else
							{
								entries.push_back(SPrecedenceEntry());
								undefined_entry_flag = true;
							}
						}
						else
						{
							entries.push_back(SPrecedenceEntry());
							undefined_entry_flag = true;
						}
					}
				}
			}
		}
	}

	StringToken::DeleteTokenVectorElements(token_vector);
}

bool    CPrecedence::ReplaceTerminal(	const CTerminal *current,
										const CTerminal *new_terminal)
{
	bool	replace_flag = false;

	vector<SPrecedenceEntry>::size_type		e_index;

	for (e_index=0; e_index<entries.size(); e_index++)
	{
		SPrecedenceEntry	&pe = entries[e_index];

		if (pe.type == SPrecedenceEntry::TerminalEntry &&
			pe.terminal == current)
		{
			if (new_terminal->IsUndefined())
				pe.type = SPrecedenceEntry::UndefinedEntry;
			else
				pe.terminal = new_terminal;

			replace_flag = true;
		}
	}

	if (replace_flag && new_terminal->IsUndefined())
		undefined_entry_flag = true;

	return replace_flag;
}

bool    CPrecedence::ReplaceErrorToken(	const CErrorToken *current,
										const CErrorToken *new_error_token)
{
	bool	replace_flag = false;

	vector<SPrecedenceEntry>::size_type		e_index;

	for (e_index=0; e_index<entries.size(); e_index++)
	{
		SPrecedenceEntry	&pe = entries[e_index];

		if (pe.type == SPrecedenceEntry::ErrorTokenEntry &&
			pe.error_token == current)
		{
			if (new_error_token->IsUndefined())
				pe.type = SPrecedenceEntry::UndefinedEntry;
			else
				pe.error_token = new_error_token;

			replace_flag = true;
		}
	}

	if (replace_flag && new_error_token->IsUndefined())
		undefined_entry_flag = true;

	return replace_flag;
}

bool    CPrecedence::ReplaceRule(	const CRule *current,
									const CRule *new_rule)
{
	bool	replace_flag = false;

	vector<SPrecedenceEntry>::size_type		e_index;

	for (e_index=0; e_index<entries.size(); e_index++)
	{
		SPrecedenceEntry	&pe = entries[e_index];

		if (pe.type == SPrecedenceEntry::RuleEntry &&
			pe.rule == current)
		{
			if (new_rule->IsUndefined())
				pe.type = SPrecedenceEntry::UndefinedEntry;
			else
				pe.rule = new_rule;

			replace_flag = true;
		}
	}

	if (replace_flag && new_rule->IsUndefined())
		undefined_entry_flag = true;

	return replace_flag;
}

string  &CPrecedence::GetAssociativityString(string &s) const
{
	if (associativity == LeftAssociative)
		s = "LEFT";
	else if (associativity == RightAssociative)
		s = "RIGHT";
	else if (associativity == NonAssociative)
		s = "NONASSOCIATIVE";
	else
		s = "NULLASSOCIATIVE";

	return s;
}

string  &CPrecedence::GetEntryString(string &s) const
{
	s = "";

	vector<SPrecedenceEntry>::size_type		e_index;

	for (e_index=0; e_index<entries.size(); e_index++)
	{
		const SPrecedenceEntry	&pe = entries[e_index];

		switch (pe.type)
		{
			case SPrecedenceEntry::TerminalEntry:
			if (e_index) s+= " ";
			s += pe.terminal->GetName();
			break;

			case SPrecedenceEntry::ErrorTokenEntry:
			if (e_index) s+= " ";
			s += pe.error_token->GetName();
			break;

			case SPrecedenceEntry::RuleEntry:
			{
				ostrstream		os;

				os << pe.rule->GetIndex() << '\0';

				if (e_index) s+= " ";
				s += "RULE:";
				s += os.str();
			}
			break;

			case SPrecedenceEntry::UndefinedEntry:
			if (e_index) s+= " ";
			s += undefined_text;
			break;
		}
	}

	return s;
}

string  &CPrecedence::GetEntryLessOne(string &s)
{
	entries.pop_back();

	return GetEntryString(s);
}

SConflictResolution::SConflictResolution(const string &s)
	: index(-1)
{
	StringToken::TokenStatus				token_status;
	vector<StringToken::CTokenValue *>		token_vector;

	token_status = StringToken::GetIntegerTokenSequence(
						s, string("\t "), token_vector);

	if (token_status == StringToken::NoError && token_vector.size() == 4)
	{
		int		c_index = token_vector[0]->GetIntegerValue();

		if (c_index >= 0 && c_index < CCompilerData::index_upper_bound)
		{
			index	= c_index;
			state	= token_vector[1]->GetIntegerValue();
			symbol	= token_vector[2]->GetIntegerValue();
			action	= token_vector[3]->GetIntegerValue();
		}
	}

	StringToken::DeleteTokenVectorElements(token_vector);
}

SErrorCode::SErrorCode(const string &s)
	: entry(0)
{
	StringToken::TokenStatus				token_status;
	vector<StringToken::CTokenValue *>		token_vector;

	token_status = StringToken::GetIntegerTokenSequence(
						s, string("\t "), token_vector);

	if (token_status == StringToken::NoError &&
		(token_vector.size() == 3 || token_vector.size() == 4))
	{
		int		e_entry = token_vector[0]->GetIntegerValue();

		if (e_entry > 0)
		{
			entry	= e_entry;
			state	= token_vector[1]->GetIntegerValue();
			symbol	= token_vector[2]->GetIntegerValue();

			if (token_vector.size() == 3)
				token = -1;
			else
			{
				token = token_vector[3]->GetIntegerValue();

				if (token < 0)
					token = -1;
			}
		}
	}

	StringToken::DeleteTokenVectorElements(token_vector);
}

void	CCompilerData::ReplaceUpdatedTerminal(CTerminal *t)
{
	ReplaceDeletedTerminal(t);
}

void	CCompilerData::ReplaceDeletedTerminal(CTerminal *t)
{
	list<CRule *>::iterator			r_iterator	= rule_list.begin();

	while (r_iterator != rule_list.end())
		(*r_iterator++)->ReplaceSymbol(t, &undefined_symbol);

	list<CPrecedence *>::iterator	p_iterator	= precedence_list.begin();

	while (p_iterator != precedence_list.end())
		(*p_iterator++)->ReplaceTerminal(t, &undefined_terminal);
}

void	CCompilerData::ReplaceDeletedErrorToken(CErrorToken *e)
{
	list<CRule *>::iterator			r_iterator	= rule_list.begin();

	while (r_iterator != rule_list.end())
		(*r_iterator++)->ReplaceSymbol(e, &undefined_symbol);

	list<CPrecedence *>::iterator	p_iterator	= precedence_list.begin();

	while (p_iterator != precedence_list.end())
		(*p_iterator++)->ReplaceErrorToken(e, &undefined_error_token);
}

void	CCompilerData::ReplaceDeletedNonTerminal(CNonTerminal *n)
{
	list<CRule *>::iterator			r_iterator	= rule_list.begin();

	while (r_iterator != rule_list.end())
		(*r_iterator++)->ReplaceSymbol(n, &undefined_symbol);
}

void	CCompilerData::ReplaceDeletedRule(CRule *r)
{
	list<CPrecedence *>::iterator	p_iterator	= precedence_list.begin();

	while (p_iterator != precedence_list.end())
		(*p_iterator++)->ReplaceRule(r, &undefined_rule);
}

CSymbol		*CCompilerData::FindSymbolName(const string &s)
{
	CTerminal		*t;
	CErrorToken		*e;
	CNonTerminal	*n;

	if (t = FindTerminalName(s))
		return t;

	if (e = FindErrorTokenName(s))
		return e;

	if (n = FindNonTerminalName(s))
		return n;

	return 0;
}

CTerminal	*CCompilerData::FindTerminalName(const string &s)
{
	list<CTerminal *>::iterator		t_iterator = terminal_list.begin();

	while (t_iterator != terminal_list.end())
	{
		if ((*t_iterator)->GetName() == s && !(*t_iterator)->GetFilterFlag())
			return *t_iterator;

		t_iterator++;
	}

	return 0;
}

CErrorToken		*CCompilerData::FindErrorTokenName(const string &s)
{
	list<CErrorToken *>::iterator		e_iterator = error_token_list.begin();

	while (e_iterator != error_token_list.end())
	{
		if ((*e_iterator)->GetName() == s)
			return *e_iterator;

		e_iterator++;
	}

	return 0;
}

CNonTerminal	*CCompilerData::FindNonTerminalName(const string &s)
{
	list<CNonTerminal *>::iterator	n_iterator = nonterminal_list.begin();

	while (n_iterator != nonterminal_list.end())
	{
		if ((*n_iterator)->GetName() == s)
			return *n_iterator;

		n_iterator++;
	}

	return 0;
}

list<CTerminal *>::iterator		CCompilerData::FindTerminalIterator(int i)
{
	if (i >= terminal_list.size() || i < 0)
		return terminal_list.end();
	else
	{
		list<CTerminal *>::iterator		t_iterator = terminal_list.begin();
		int								t_index;

		for (t_index=0; t_index<i; t_index++)
			t_iterator++;

		return t_iterator;
	}
}

CTerminal		*CCompilerData::FindTerminalIndex(int i)
{
	list<CTerminal *>::iterator		t_iterator = FindTerminalIterator(i);

	if (t_iterator != terminal_list.end())
		return *t_iterator;
	else
		return 0;
}

list<CErrorToken *>::iterator	CCompilerData::FindErrorTokenIterator(int i)
{
	if (i >= error_token_list.size() || i < 0)
		return error_token_list.end();
	else
	{
		list<CErrorToken *>::iterator	e_iterator = error_token_list.begin();
		int								e_index;

		for (e_index=0; e_index<i; e_index++)
			e_iterator++;

		return e_iterator;
	}
}

CErrorToken		*CCompilerData::FindErrorTokenIndex(int i)
{
	list<CErrorToken *>::iterator		e_iterator = FindErrorTokenIterator(i);

	if (e_iterator != error_token_list.end())
		return *e_iterator;
	else
		return 0;
}

list<CNonTerminal *>::iterator	CCompilerData::FindNonTerminalIterator(int i)
{
	if (i >= nonterminal_list.size() || i < 0)
		return nonterminal_list.end();
	else
	{
		list<CNonTerminal *>::iterator	n_iterator = nonterminal_list.begin();
		int								n_index;

		for (n_index=0; n_index<i; n_index++)
			n_iterator++;

		return n_iterator;
	}
}

CNonTerminal	*CCompilerData::FindNonTerminalIndex(int i)
{
	list<CNonTerminal *>::iterator		n_iterator = FindNonTerminalIterator(i);

	if (n_iterator != nonterminal_list.end())
		return *n_iterator;
	else
		return 0;
}

list<CRule *>::iterator		CCompilerData::FindRuleIterator(int i)
{
	if (i >= rule_list.size() || i < 0)
		return rule_list.end();
	else
	{
		list<CRule *>::iterator		r_iterator = rule_list.begin();
		int							r_index;

		for (r_index=0; r_index<i; r_index++)
			r_iterator++;

		return r_iterator;
	}
}

CRule	*CCompilerData::FindRuleIndex(int i)
{
	list<CRule *>::iterator		r_iterator = FindRuleIterator(i);

	if (r_iterator != rule_list.end())
		return *r_iterator;
	else
		return 0;
}

list<CPrecedence *>::iterator	CCompilerData::FindPrecedenceIterator(int i)
{
	if (i >= precedence_list.size() || i < 0)
		return precedence_list.end();
	else
	{
		list<CPrecedence *>::iterator	p_iterator = precedence_list.begin();
		int								p_index;

		for (p_index=0; p_index<i; p_index++)
			p_iterator++;

		return p_iterator;
	}
}

CPrecedence		*CCompilerData::FindPrecedenceIndex(int i)
{
	list<CPrecedence *>::iterator	p_iterator = FindPrecedenceIterator(i);

	if (p_iterator != precedence_list.end())
		return *p_iterator;
	else
		return 0;
}

SConflictResolution		*CCompilerData::FindConflictResolutionIndex(int i)
{
	map<int, SConflictResolution *>::iterator
		m_iterator = conflict_resolution_map.find(i);

	if (m_iterator != conflict_resolution_map.end())
		return (*m_iterator).second;
	else
		return 0;
}

SErrorCode		*CCompilerData::FindErrorCodeIndex(int i)
{
	map<int, SErrorCode *>::iterator
		m_iterator = error_code_map.find(i);

	if (m_iterator != error_code_map.end())
		return (*m_iterator).second;
	else
		return 0;
}

void	CCompilerData::AppendTerminal(const CTerminal &t)
{
	int		l_size	= terminal_list.size();
	int		i		= t.GetIndex();

	if (i < l_size)
	{
		CTerminal	*current = FindTerminalIndex(i);

		if (current)
			*current = t;
	}
	else
	{
		for (int j=0; j<i-l_size; j++)
			terminal_list.push_back(new CTerminal(l_size + j));

		terminal_list.push_back(new CTerminal(t));
	}
}

void	CCompilerData::InsertBeforeTerminal(int i, const CTerminal &t)
{
	list<CTerminal *>::iterator		t_iterator = FindTerminalIterator(i);

	if (t_iterator != terminal_list.end())
		terminal_list.insert(t_iterator, new CTerminal(t));
	else
		AppendTerminal(t);

	SetupTerminalIndexes();
	SetupSymbolIndexes();
}

void	CCompilerData::InsertAfterTerminal(int i, const CTerminal &t)
{
	list<CTerminal *>::iterator		t_iterator = FindTerminalIterator(i);

	if (t_iterator != terminal_list.end())
		terminal_list.insert(++t_iterator, new CTerminal(t));
	else
		AppendTerminal(t);

	SetupTerminalIndexes();
	SetupSymbolIndexes();
}

void	CCompilerData::UpdateTerminal(int i, const CTerminal &t)
{
	list<CTerminal *>::iterator		t_iterator = FindTerminalIterator(i);

	if (t_iterator != terminal_list.end())
	{
		if (!(*t_iterator)->GetFilterFlag() && t.GetFilterFlag())
			ReplaceUpdatedTerminal(*t_iterator);

		(*t_iterator)->UpdateTerminal(t);

		SetupTerminalIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::DeleteTerminal(int i)
{
	list<CTerminal *>::iterator		t_iterator = FindTerminalIterator(i);

	if (t_iterator != terminal_list.end())
	{
		ReplaceDeletedTerminal(*t_iterator);

		delete *t_iterator;

		terminal_list.erase(t_iterator);

		SetupTerminalIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::MoveTerminalUp(int i)
{
	if (!i)
		return;

	list<CTerminal *>::iterator		t_iterator = FindTerminalIterator(i);

	if (t_iterator != terminal_list.end())
	{
		list<CTerminal *>::iterator		previous_iterator = t_iterator--;

		iter_swap(t_iterator, previous_iterator);

		SetupTerminalIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::MoveTerminalDown(int i)
{
	if (i == terminal_list.size() - 1)
		return;

	list<CTerminal *>::iterator		t_iterator = FindTerminalIterator(i);

	if (t_iterator != terminal_list.end())
	{
		list<CTerminal *>::iterator		next_iterator = t_iterator++;

		iter_swap(t_iterator, next_iterator);

		SetupTerminalIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::AppendErrorToken(const CErrorToken &e)
{
	int		l_size	= error_token_list.size();
	int		i		= e.GetIndex();

	if (i < l_size)
	{
		CErrorToken		*current = FindErrorTokenIndex(i);

		if (current)
			*current = e;
	}
	else
	{
		for (int j=0; j<i-l_size; j++)
			error_token_list.push_back(new CErrorToken(l_size + j));

		error_token_list.push_back(new CErrorToken(e));
	}
}

void	CCompilerData::InsertBeforeErrorToken(int i, const CErrorToken &e)
{
	list<CErrorToken *>::iterator	e_iterator = FindErrorTokenIterator(i);

	if (e_iterator != error_token_list.end())
		error_token_list.insert(e_iterator, new CErrorToken(e));
	else
		AppendErrorToken(e);

	SetupErrorTokenIndexes();
	SetupSymbolIndexes();
}

void	CCompilerData::InsertAfterErrorToken(int i, const CErrorToken &e)
{
	list<CErrorToken *>::iterator	e_iterator = FindErrorTokenIterator(i);

	if (e_iterator != error_token_list.end())
		error_token_list.insert(++e_iterator, new CErrorToken(e));
	else
		AppendErrorToken(e);

	SetupErrorTokenIndexes();
	SetupSymbolIndexes();
}

void	CCompilerData::UpdateErrorToken(int i, const CErrorToken &e)
{
	list<CErrorToken *>::iterator	e_iterator = FindErrorTokenIterator(i);

	if (e_iterator != error_token_list.end())
	{
		**e_iterator = e;

		SetupErrorTokenIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::DeleteErrorToken(int i)
{
	list<CErrorToken *>::iterator	e_iterator = FindErrorTokenIterator(i);

	if (e_iterator != error_token_list.end())
	{
		ReplaceDeletedErrorToken(*e_iterator);

		delete *e_iterator;

		error_token_list.erase(e_iterator);

		SetupErrorTokenIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::MoveErrorTokenUp(int i)
{
	if (!i)
		return;

	list<CErrorToken *>::iterator		e_iterator = FindErrorTokenIterator(i);

	if (e_iterator != error_token_list.end())
	{
		list<CErrorToken *>::iterator		previous_iterator = e_iterator--;

		iter_swap(e_iterator, previous_iterator);

		SetupErrorTokenIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::MoveErrorTokenDown(int i)
{
	if (i == error_token_list.size() - 1)
		return;

	list<CErrorToken *>::iterator		e_iterator = FindErrorTokenIterator(i);

	if (e_iterator != error_token_list.end())
	{
		list<CErrorToken *>::iterator		next_iterator = e_iterator++;

		iter_swap(e_iterator, next_iterator);

		SetupErrorTokenIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::AppendNonTerminal(const CNonTerminal &n)
{
	int		l_size	= nonterminal_list.size();
	int		i		= n.GetIndex();

	if (i < l_size)
	{
		CNonTerminal	*current = FindNonTerminalIndex(i);

		if (current)
			*current = n;
	}
	else
	{
		for (int j=0; j<i-l_size; j++)
			nonterminal_list.push_back(new CNonTerminal(l_size + j));

		nonterminal_list.push_back(new CNonTerminal(n));
	}
}

void	CCompilerData::InsertBeforeNonTerminal(int i, const CNonTerminal &n)
{
	if (n.GetStartSymbolFlag())
		ClearStartSymbolFlag();

	list<CNonTerminal *>::iterator	n_iterator = FindNonTerminalIterator(i);

	if (n_iterator != nonterminal_list.end())
		nonterminal_list.insert(n_iterator, new CNonTerminal(n));
	else
		AppendNonTerminal(n);

	SetupNonTerminalIndexes();
	SetupSymbolIndexes();
}

void	CCompilerData::InsertAfterNonTerminal(int i, const CNonTerminal &n)
{
	if (n.GetStartSymbolFlag())
		ClearStartSymbolFlag();

	list<CNonTerminal *>::iterator	n_iterator = FindNonTerminalIterator(i);

	if (n_iterator != nonterminal_list.end())
		nonterminal_list.insert(++n_iterator, new CNonTerminal(n));
	else
		AppendNonTerminal(n);

	SetupNonTerminalIndexes();
	SetupSymbolIndexes();
}

void	CCompilerData::UpdateNonTerminal(int i, const CNonTerminal &n)
{
	list<CNonTerminal *>::iterator	n_iterator = FindNonTerminalIterator(i);

	if (n_iterator != nonterminal_list.end())
	{
		if (n.GetStartSymbolFlag())
			ClearStartSymbolFlag();

		**n_iterator = n;

		SetupNonTerminalIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::DeleteNonTerminal(int i)
{
	list<CNonTerminal *>::iterator	n_iterator = FindNonTerminalIterator(i);

	if (n_iterator != nonterminal_list.end())
	{
		ReplaceDeletedNonTerminal(*n_iterator);

		delete *n_iterator;

		nonterminal_list.erase(n_iterator);

		SetupNonTerminalIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::MoveNonTerminalUp(int i)
{
	if (!i)
		return;

	list<CNonTerminal *>::iterator		n_iterator = FindNonTerminalIterator(i);

	if (n_iterator != nonterminal_list.end())
	{
		list<CNonTerminal *>::iterator		previous_iterator = n_iterator--;

		iter_swap(n_iterator, previous_iterator);

		SetupNonTerminalIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::MoveNonTerminalDown(int i)
{
	if (i == nonterminal_list.size() - 1)
		return;

	list<CNonTerminal *>::iterator		n_iterator = FindNonTerminalIterator(i);

	if (n_iterator != nonterminal_list.end())
	{
		list<CNonTerminal *>::iterator		next_iterator = n_iterator++;

		iter_swap(n_iterator, next_iterator);

		SetupNonTerminalIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::AppendRule(const CRule &r)
{
	int		l_size	= rule_list.size();
	int		i		= r.GetIndex();

	if (i < l_size)
	{
		CRule	*current = FindRuleIndex(i);

		if (current)
			*current = r;
	}
	else
	{
		for (int j=0; j<i-l_size; j++)
			rule_list.push_back(new CRule(l_size + j));

		rule_list.push_back(new CRule(r));
	}
}

void	CCompilerData::InsertBeforeRule(int i, const CRule &r)
{
	list<CRule *>::iterator		r_iterator = FindRuleIterator(i);

	if (r_iterator != rule_list.end())
		rule_list.insert(r_iterator, new CRule(r));
	else
		AppendRule(r);

	SetupRuleIndexes();
	SetupSymbolIndexes();
}

void	CCompilerData::InsertAfterRule(int i, const CRule &r)
{
	list<CRule *>::iterator		r_iterator = FindRuleIterator(i);

	if (r_iterator != rule_list.end())
		rule_list.insert(++r_iterator, new CRule(r));
	else
		AppendRule(r);

	SetupRuleIndexes();
	SetupSymbolIndexes();
}

void	CCompilerData::UpdateRule(int i, const CRule &r)
{
	list<CRule *>::iterator		r_iterator = FindRuleIterator(i);

	if (r_iterator != rule_list.end())
	{
		(*r_iterator)->UpdateRule(r);

		SetupRuleIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::DeleteRule(int i)
{
	list<CRule *>::iterator		r_iterator = FindRuleIterator(i);

	if (r_iterator != rule_list.end())
	{
		ReplaceDeletedRule(*r_iterator);

		delete *r_iterator;

		rule_list.erase(r_iterator);

		SetupRuleIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::MoveRuleUp(int i)
{
	if (!i)
		return;

	list<CRule *>::iterator		r_iterator = FindRuleIterator(i);

	if (r_iterator != rule_list.end())
	{
		list<CRule *>::iterator		previous_iterator = r_iterator--;

		iter_swap(r_iterator, previous_iterator);

		SetupRuleIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::MoveRuleDown(int i)
{
	if (i == rule_list.size() - 1)
		return;

	list<CRule *>::iterator		r_iterator = FindRuleIterator(i);

	if (r_iterator != rule_list.end())
	{
		list<CRule *>::iterator		next_iterator = r_iterator++;

		iter_swap(r_iterator, next_iterator);

		SetupRuleIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::AppendPrecedence(const CPrecedence &p)
{
	int		l_size	= precedence_list.size();
	int		i		= p.GetIndex();

	if (i < l_size)
	{
		CPrecedence		*current = FindPrecedenceIndex(i);

		if (current)
			*current = p;
	}
	else
	{
		for (int j=0; j<i-l_size; j++)
			precedence_list.push_back(new CPrecedence(l_size + j));

		precedence_list.push_back(new CPrecedence(p));
	}
}

void	CCompilerData::InsertBeforePrecedence(int i, const CPrecedence &p)
{
	list<CPrecedence *>::iterator	p_iterator = FindPrecedenceIterator(i);

	if (p_iterator != precedence_list.end())
		precedence_list.insert(p_iterator, new CPrecedence(p));
	else
		AppendPrecedence(p);

	SetupPrecedenceIndexes();
	SetupSymbolIndexes();
}

void	CCompilerData::InsertAfterPrecedence(int i, const CPrecedence &p)
{
	list<CPrecedence *>::iterator	p_iterator = FindPrecedenceIterator(i);

	if (p_iterator != precedence_list.end())
		precedence_list.insert(++p_iterator, new CPrecedence(p));
	else
		AppendPrecedence(p);

	SetupPrecedenceIndexes();
	SetupSymbolIndexes();
}

void	CCompilerData::UpdatePrecedence(int i, const CPrecedence &p)
{
	list<CPrecedence *>::iterator	p_iterator = FindPrecedenceIterator(i);

	if (p_iterator != precedence_list.end())
	{
		**p_iterator = p;

		SetupPrecedenceIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::DeletePrecedence(int i)
{
	list<CPrecedence *>::iterator	p_iterator = FindPrecedenceIterator(i);

	if (p_iterator != precedence_list.end())
	{
		delete *p_iterator;

		precedence_list.erase(p_iterator);

		SetupPrecedenceIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::MovePrecedenceUp(int i)
{
	if (!i)
		return;

	list<CPrecedence *>::iterator		p_iterator = FindPrecedenceIterator(i);

	if (p_iterator != precedence_list.end())
	{
		list<CPrecedence *>::iterator		previous_iterator = p_iterator--;

		iter_swap(p_iterator, previous_iterator);

		SetupPrecedenceIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::MovePrecedenceDown(int i)
{
	if (i == precedence_list.size() - 1)
		return;

	list<CPrecedence *>::iterator		p_iterator = FindPrecedenceIterator(i);

	if (p_iterator != precedence_list.end())
	{
		list<CPrecedence *>::iterator		next_iterator = p_iterator++;

		iter_swap(p_iterator, next_iterator);

		SetupPrecedenceIndexes();
		SetupSymbolIndexes();
	}
}

void	CCompilerData::UpdateLexicalCode(const char *c)
{
	istrstream		is(c);
	string			line;

	while (getline(is, line))
		lexical_code.push_back(line);
}

void	CCompilerData::AddLexicalCode(const string &s)
{
	lexical_code.push_back(s);
}

void	CCompilerData::ClearLexicalCode()
{
	lexical_code.clear();
}

void	CCompilerData::GetLexicalCodeSections(	list<string> &include_code,
												list<string> &class_code,
												list<string> &source_code,
												list<string> &constructor_code,
												list<string> &destructor_code)
{
	include_code.clear();
	class_code.clear();
	source_code.clear();
	constructor_code.clear();
	destructor_code.clear();

	const int	include_section		= 0;
	const int	class_section		= 1;
	const int	source_section		= 2;
	const int	constructor_section	= 3;
	const int	destructor_section	= 4;

	map<string, int>				section_map;
	map<string, int>::iterator		s_iterator;

	section_map["#INCLUDE"]		= include_section;
	section_map["#CLASS"]		= class_section;
	section_map["#SOURCE"]		= source_section;
	section_map["#CONSTRUCTOR"]	= constructor_section;
	section_map["#DESTRUCTOR"]	= destructor_section;

	int		current_section = source_section;

	list<string>::const_iterator	c_iterator = lexical_code.begin();

	while (c_iterator != lexical_code.end())
	{
		const string	&s = *c_iterator++;

		s_iterator = section_map.find(s);

		if (s_iterator != section_map.end())
		{
			current_section	= (*s_iterator).second;
		}
		else
		{
			switch (current_section)
			{
				case include_section:
				include_code.push_back(s);
				break;

				case class_section:
				class_code.push_back(s);
				break;

				case source_section:
				source_code.push_back(s);
				break;

				case constructor_section:
				constructor_code.push_back(s);
				break;

				case destructor_section:
				destructor_code.push_back(s);
				break;
			}
		}
	}
}

void	CCompilerData::StoreLexicalCodeSections(
										const list<string> &include_code,
										const list<string> &class_code,
										const list<string> &source_code,
										const list<string> &constructor_code,
										const list<string> &destructor_code)
{
	list<string>::const_iterator	c_iterator;

	lexical_code.clear();

	lexical_code.push_back("#INCLUDE");

	c_iterator = include_code.begin();

	while (c_iterator != include_code.end())
		lexical_code.push_back(*c_iterator++);

	lexical_code.push_back("#CLASS");

	c_iterator = class_code.begin();

	while (c_iterator != class_code.end())
		lexical_code.push_back(*c_iterator++);

	lexical_code.push_back("#SOURCE");

	c_iterator = source_code.begin();

	while (c_iterator != source_code.end())
		lexical_code.push_back(*c_iterator++);

	lexical_code.push_back("#CONSTRUCTOR");

	c_iterator = constructor_code.begin();

	while (c_iterator != constructor_code.end())
		lexical_code.push_back(*c_iterator++);

	lexical_code.push_back("#DESTRUCTOR");

	c_iterator = destructor_code.begin();

	while (c_iterator != destructor_code.end())
		lexical_code.push_back(*c_iterator++);
}

void	CCompilerData::UpdateParserCode(const char *c)
{
	istrstream		is(c);
	string			line;

	while (getline(is, line))
		parser_code.push_back(line);
}

void	CCompilerData::AddParserCode(const string &s)
{
	parser_code.push_back(s);
}

void	CCompilerData::ClearParserCode()
{
	parser_code.clear();
}

void	CCompilerData::GetParserCodeSections(	list<string> &include_code,
												list<string> &class_code,
												list<string> &source_code,
												list<string> &constructor_code,
												list<string> &destructor_code)
{
	include_code.clear();
	class_code.clear();
	source_code.clear();
	constructor_code.clear();
	destructor_code.clear();

	const int	include_section		= 0;
	const int	class_section		= 1;
	const int	source_section		= 2;
	const int	constructor_section	= 3;
	const int	destructor_section	= 4;

	map<string, int>				section_map;
	map<string, int>::iterator		s_iterator;

	section_map["#INCLUDE"]		= include_section;
	section_map["#CLASS"]		= class_section;
	section_map["#SOURCE"]		= source_section;
	section_map["#CONSTRUCTOR"]	= constructor_section;
	section_map["#DESTRUCTOR"]	= destructor_section;

	int		current_section = source_section;

	list<string>::const_iterator	c_iterator = parser_code.begin();

	while (c_iterator != parser_code.end())
	{
		const string	&s = *c_iterator++;

		s_iterator = section_map.find(s);

		if (s_iterator != section_map.end())
		{
			current_section	= (*s_iterator).second;
		}
		else
		{
			switch (current_section)
			{
				case include_section:
				include_code.push_back(s);
				break;

				case class_section:
				class_code.push_back(s);
				break;

				case source_section:
				source_code.push_back(s);
				break;

				case constructor_section:
				constructor_code.push_back(s);
				break;

				case destructor_section:
				destructor_code.push_back(s);
				break;
			}
		}
	}
}

void	CCompilerData::StoreParserCodeSections(
										const list<string> &include_code,
										const list<string> &class_code,
										const list<string> &source_code,
										const list<string> &constructor_code,
										const list<string> &destructor_code)
{
	list<string>::const_iterator	c_iterator;

	parser_code.clear();

	parser_code.push_back("#INCLUDE");

	c_iterator = include_code.begin();

	while (c_iterator != include_code.end())
		parser_code.push_back(*c_iterator++);

	parser_code.push_back("#CLASS");

	c_iterator = class_code.begin();

	while (c_iterator != class_code.end())
		parser_code.push_back(*c_iterator++);

	parser_code.push_back("#SOURCE");

	c_iterator = source_code.begin();

	while (c_iterator != source_code.end())
		parser_code.push_back(*c_iterator++);

	parser_code.push_back("#CONSTRUCTOR");

	c_iterator = constructor_code.begin();

	while (c_iterator != constructor_code.end())
		parser_code.push_back(*c_iterator++);

	parser_code.push_back("#DESTRUCTOR");

	c_iterator = destructor_code.begin();

	while (c_iterator != destructor_code.end())
		parser_code.push_back(*c_iterator++);
}

void	CCompilerData::AddStackCode(const string &s)
{
	stack_code.push_back(s);
}

void	CCompilerData::StoreStackCode(const list<string> &code)
{
	stack_code.clear();

	list<string>::const_iterator	c_iterator = code.begin();

	while (c_iterator != code.end())
		stack_code.push_back(*c_iterator++);
}

void	CCompilerData::ClearStackCode()
{
	stack_code.clear();
}

void	CCompilerData::InitializeStackCode()
{
	stack_code.clear();

	stack_code.push_back("");
	stack_code.push_back("#include \"unionScalar.h\"");
	stack_code.push_back("#include \"stackTypes.h\"");
	stack_code.push_back("");
	stack_code.push_back("class CStackElement");
	stack_code.push_back("{");
	stack_code.push_back("\tpublic:");
	stack_code.push_back(
	"\tCStackElement() : type(StackLiteralToken), char_array(0), data(0) { }");
	stack_code.push_back("");
	stack_code.push_back("\tint\t\t\t\t\tstate;");
	stack_code.push_back("\tStackElementType\ttype;");
	stack_code.push_back("\tint\t\t\t\t\ttoken;");
	stack_code.push_back("\tUScalar\t\t\t\tscalar;");
	stack_code.push_back("\tunsigned char\t\t*char_array;");
	stack_code.push_back("\tvoid\t\t\t\t*data;");
	stack_code.push_back("\tint\t\t\t\t\treduce_action;");
	stack_code.push_back("};");
	stack_code.push_back("");
}

void	CCompilerData::AddCommentString(const string &s)
{
	comment_list.push_back(s);
}

void	CCompilerData::AddConflictResolution(const SConflictResolution &c)
{
	DeleteConflictResolution(c.index);

	conflict_resolution_map[c.index] = new SConflictResolution(c);
}

void	CCompilerData::DeleteConflictResolution(int i)
{
	map<int, SConflictResolution *>::iterator
		c_iterator = conflict_resolution_map.find(i);

	if (c_iterator != conflict_resolution_map.end())
	{
		delete (*c_iterator).second;

		conflict_resolution_map.erase(c_iterator);
	}
}

void	CCompilerData::ClearConflictResolution()
{
	map<int, SConflictResolution *>::iterator
		c_iterator = conflict_resolution_map.begin();

	while (c_iterator != conflict_resolution_map.end())
		delete (*c_iterator++).second;

	conflict_resolution_map.clear();
}

void	CCompilerData::AddErrorCode(const SErrorCode &e)
{
	DeleteErrorCode(e.entry);

	error_code_map[e.entry] = new SErrorCode(e);
}

void	CCompilerData::DeleteErrorCode(int i)
{
	map<int, SErrorCode *>::iterator
		e_iterator = error_code_map.find(i);

	if (e_iterator != error_code_map.end())
	{
		delete (*e_iterator).second;

		error_code_map.erase(e_iterator);
	}
}

void	CCompilerData::ClearErrorCode()
{
	map<int, SErrorCode *>::iterator
		e_iterator = error_code_map.begin();

	while (e_iterator != error_code_map.end())
		delete (*e_iterator++).second;

	error_code_map.clear();
}

void	CCompilerData::ClearAllData()
{
	list<CTerminal *>::iterator		t_iterator	= terminal_list.begin();

	while (t_iterator != terminal_list.end())
		delete *t_iterator++;

	list<CErrorToken *>::iterator	e_iterator	= error_token_list.begin();

	while (e_iterator != error_token_list.end())
		delete *e_iterator++;

	list<CNonTerminal *>::iterator	n_iterator	= nonterminal_list.begin();

	while (n_iterator != nonterminal_list.end())
		delete *n_iterator++;

	list<CRule *>::iterator			r_iterator	= rule_list.begin();

	while (r_iterator != rule_list.end())
		delete *r_iterator++;

	list<CPrecedence *>::iterator	p_iterator	= precedence_list.begin();

	while (p_iterator != precedence_list.end())
		delete *p_iterator++;

	map<int, SConflictResolution *>::iterator
		cr_iterator = conflict_resolution_map.begin();

	while (cr_iterator != conflict_resolution_map.end())
		delete (*cr_iterator++).second;

	map<int, SErrorCode *>::iterator
		ec_iterator = error_code_map.begin();

	while (ec_iterator != error_code_map.end())
		delete (*ec_iterator++).second;
}

void	CCompilerData::InitializeUnitRuleOrderList()
{
	unit_rule_order_list.clear();

	list<CRule *>::iterator		r_iterator = rule_list.begin();

	while (r_iterator != rule_list.end())
	{
		if ((*r_iterator)->GetUnitRuleEliminationFlag())
			unit_rule_order_list.push_back((*r_iterator)->GetIndex());

		r_iterator++;
	}
}

bool	CCompilerData::ValidateUnitRuleOrderList()
{
	set<int>				validation_set;
	set<int>::iterator		v_iterator;
	list<int>::iterator		u_iterator = unit_rule_order_list.begin();

	while (u_iterator != unit_rule_order_list.end())
	{
		v_iterator = validation_set.find(*u_iterator);

		if (v_iterator != validation_set.end())
			return false;
		else
			validation_set.insert(*u_iterator);

		CRule	*r = FindRuleIndex(*u_iterator);

		if (!r)
			return false;

		if (!r->GetUnitRuleEliminationFlag())
			return false;

		u_iterator++;
	}

	int		unit_rule_count = 0;

	list<CRule *>::iterator		r_iterator = rule_list.begin();

	while (r_iterator != rule_list.end())
		if ((*r_iterator++)->GetUnitRuleEliminationFlag())
			unit_rule_count++;

	if (unit_rule_count != unit_rule_order_list.size())
		return false;

	return true;
}

void	CCompilerData::ClearStartSymbolFlag()
{
	list<CNonTerminal *>::iterator	n_iterator	= nonterminal_list.begin();

	while (n_iterator != nonterminal_list.end())
		(*n_iterator++)->SetStartSymbolFlag(false);
}

void	CCompilerData::SetupTerminalIndexes()
{
	list<CTerminal *>::iterator		t_iterator	= terminal_list.begin();
	int								index		= 0;

	while (t_iterator != terminal_list.end())
		(*t_iterator++)->SetIndex(index++);
}

void	CCompilerData::SetupErrorTokenIndexes()
{
	list<CErrorToken *>::iterator	e_iterator	= error_token_list.begin();
	int								index		= 0;

	while (e_iterator != error_token_list.end())
		(*e_iterator++)->SetIndex(index++);
}

void	CCompilerData::SetupNonTerminalIndexes()
{
	list<CNonTerminal *>::iterator	n_iterator	= nonterminal_list.begin();
	int								index		= 0;

	while (n_iterator != nonterminal_list.end())
		(*n_iterator++)->SetIndex(index++);
}

void	CCompilerData::SetupRuleIndexes()
{
	list<CRule *>::iterator			r_iterator	= rule_list.begin();
	int								index		= 0;

	while (r_iterator != rule_list.end())
		(*r_iterator++)->SetIndex(index++);
}

void	CCompilerData::SetupPrecedenceIndexes()
{
	list<CPrecedence *>::iterator	p_iterator	= precedence_list.begin();
	int								index		= 0;

	while (p_iterator != precedence_list.end())
		(*p_iterator++)->SetIndex(index++);
}

void	CCompilerData::SetupSymbolIndexes()
{
	int				symbol_index		= 0;
	int				filter_index		= -1;
	bool			start_symbol_flag	= false;

	list<CTerminal *>::iterator		t_iterator = terminal_list.begin();
	list<CErrorToken *>::iterator	e_iterator = error_token_list.begin();
	list<CNonTerminal *>::iterator	n_iterator = nonterminal_list.begin();

	while (t_iterator != terminal_list.end())
	{
		CTerminal	*t = *t_iterator;

		if (t->GetFilterFlag())
			t->SetSymbolIndex(filter_index--);
		else
			t->SetSymbolIndex(symbol_index++);

		t_iterator++;
	}

	while (e_iterator != error_token_list.end())
	{
		CErrorToken		*e = *e_iterator;

		e->SetSymbolIndex(symbol_index++);

		e_iterator++;
	}

	end_of_file_symbol.SetSymbolIndex(symbol_index++);

	start_symbol = 0;

	while (n_iterator != nonterminal_list.end())
	{
		CNonTerminal	*n = *n_iterator;

		n->SetSymbolIndex(symbol_index++);

		if (!start_symbol_flag && n->GetStartSymbolFlag())
		{
			start_symbol_flag	= true;
			start_symbol		= n;
		}
		else
			n->SetStartSymbolFlag(false);

		n_iterator++;
	}

	if (!start_symbol_flag && nonterminal_list.size())
	{
		CNonTerminal	*n = *nonterminal_list.begin();

		n->SetStartSymbolFlag(true);

		start_symbol_flag	= true;
		start_symbol		= n;
	}

	augmented_start_symbol.SetSymbolIndex(symbol_index++);

	augmented_start_rule.SetIndex(rule_list.size());
	augmented_start_rule.SetLeft(&augmented_start_symbol);

	vector<const CSymbol *>		right;

	if (start_symbol_flag)
	{
		right.push_back(start_symbol);
		right.push_back(&end_of_file_symbol);
	}

	augmented_start_rule.SetRight(right);
}

bool	CCompilerData::GetUndefinedSymbolRules()
{
	bool	r_flag = false;

	undefined_symbol_rules.clear();

	list<CRule *>::iterator			r_iterator	= rule_list.begin();

	while (r_iterator != rule_list.end())
	{
		CRule	*r = *r_iterator++;

		if (r->GetUndefinedSymbolFlag())
		{
			undefined_symbol_rules.push_back(r->GetIndex());

			r_flag = true;
		}
	}

	return r_flag;
}

bool	CCompilerData::GetUndefinedEntryPrecedences()
{
	bool	r_flag = false;

	undefined_entry_precedences.clear();

	list<CPrecedence *>::iterator	p_iterator	= precedence_list.begin();

	while (p_iterator != precedence_list.end())
	{
		CPrecedence		*p = *p_iterator++;

		if (p->GetUndefinedEntryFlag())
		{
			undefined_entry_precedences.push_back(p->GetIndex());

			r_flag = true;
		}
	}

	return r_flag;
}

istream		&operator>>(istream &is, CCompilerData &d)
{
	const int	context_section			= 0;
	const int	terminal_section		= 1;
	const int	error_token_section		= 2;
	const int	nonterminal_section		= 3;
	const int	rule_section			= 4;
	const int	precedence_section		= 5;
	const int	lexical_code_section	= 6;
	const int	terminal_code_section	= 7;
	const int	parser_code_section		= 8;
	const int	reduce_code_section		= 9;
	const int	stack_code_section		= 10;
	const int	conflict_section		= 11;
	const int	error_code_section		= 12;
	const int	unit_rule_order_section	= 13;
	const int	configuration_section	= 14;
	const int	comment_section			= 15;

	map<string, int>	section_map;

	section_map["#CONTEXT"]			= context_section;
	section_map["#TERMINAL"]		= terminal_section;
	section_map["#ERROR_TOKEN"]		= error_token_section;
	section_map["#NONTERMINAL"]		= nonterminal_section;
	section_map["#RULE"]			= rule_section;
	section_map["#PRECENDENCE"]		= precedence_section;
	section_map["#LEXICAL_CODE"]	= lexical_code_section;
	section_map["#TERMINAL_CODE"]	= terminal_code_section;
	section_map["#PARSER_CODE"]		= parser_code_section;
	section_map["#REDUCE_CODE"]		= reduce_code_section;
	section_map["#STACK_CODE"]		= stack_code_section;
	section_map["#CONFLICT"]		= conflict_section;
	section_map["#ERROR_CODE"]		= error_code_section;
	section_map["#UNIT_RULE_ORDER"]	= unit_rule_order_section;
	section_map["#CONFIGURATION"]	= configuration_section;
	section_map["#COMMENT"]			= comment_section;

	const int	lexical_automaton_option			= 0;
	const int	lexical_prefix_name_option			= 1;
	const int	lexical_finite_state_machine_option	= 2;
	const int	lexical_input_range_option			= 3;
	const int	parser_automaton_option				= 4;
	const int	parser_prefix_name_option			= 5;
	const int	parser_type_option					= 6;
	const int	base_prefix_name_option				= 7;
	const int	generate_base_classes_option		= 8;
	const int	parser_error_calculation_option		= 9;
	const int	parser_error_numbering_option		= 10;
	const int	parser_error_aggregation_option		= 11;
	const int	parser_optimize_option				= 12;
	const int	generate_tty_code_option			= 13;
	const int	regexp_match_table_option			= 14;

	map<string, int>	configuration_map;

	configuration_map["LEXICAL_AUTOMATON"]
		= lexical_automaton_option;
	configuration_map["LEXICAL_PREFIX_NAME"]
		= lexical_prefix_name_option;
	configuration_map["LEXICAL_FINITE_STATE_MACHINE"]
		= lexical_finite_state_machine_option;
	configuration_map["LEXICAL_INPUT_RANGE"]
		= lexical_input_range_option;
	configuration_map["PARSER_AUTOMATON"]
		= parser_automaton_option;
	configuration_map["PARSER_PREFIX_NAME"]
		= parser_prefix_name_option;
	configuration_map["PARSER_TYPE"]
		= parser_type_option;
	configuration_map["BASE_PREFIX_NAME"]
		= base_prefix_name_option;
	configuration_map["GENERATE_BASE_CLASSES"]
		= generate_base_classes_option;
	configuration_map["PARSER_ERROR_CALCULATION"]
		= parser_error_calculation_option;
	configuration_map["PARSER_ERROR_NUMBERING"]
		= parser_error_numbering_option;
	configuration_map["PARSER_ERROR_AGGREGATION"]
		= parser_error_aggregation_option;
	configuration_map["PARSER_OPTIMIZE"]
		= parser_optimize_option;
	configuration_map["GENERATE_TTY_CODE"]
		= generate_tty_code_option;
	configuration_map["REGEXP_MATCH_TABLE"]
		= regexp_match_table_option;

	int								current_section = comment_section;
	bool							stack_code_flag = false;
	CTerminal						*code_terminal;
	CRule							*code_rule;
	SConflictResolution				*code_conflict;
	SErrorCode						*code_error;
	string							line;
	bool							index_entry_flag;
	map<string, int>::iterator		s_iterator;
	map<string, int>::iterator		c_iterator;

	while (getline(is, line))
	{
		s_iterator = section_map.find(line);

		if (s_iterator != section_map.end())
		{
			current_section		= (*s_iterator).second;
			index_entry_flag	= true;
		}
		else
		{
			switch (current_section)
			{
				case context_section:
				d.SetStartContext(line);
				break;

				case terminal_section:
				{
					CTerminal	t(line);

					if (!t.IsUndefined())
						d.AppendTerminal(t);
				}
				break;

				case error_token_section:
				{
					CErrorToken		e(line);

					if (!e.IsUndefined())
						d.AppendErrorToken(e);
				}
				break;

				case nonterminal_section:
				{
					CNonTerminal	n(line);

					if (!n.IsUndefined())
						d.AppendNonTerminal(n);
				}
				break;

				case rule_section:
				{
					CRule	r(line, &d);

					if (!r.IsUndefined())
						d.AppendRule(r);
				}
				break;

				case precedence_section:
				{
					CPrecedence		p(line, &d);

					if (!p.IsUndefined())
						d.AppendPrecedence(p);
				}
				break;

				case lexical_code_section:
				d.AddLexicalCode(line);
				break;

				case terminal_code_section:
				if (index_entry_flag)
				{
					index_entry_flag = false;

					if (line.substr(0,9) == "TERMINAL:")
					{
						int				token_index;
						string			index_string(line.substr(9));
						istrstream		ti(index_string.c_str());

						ti >> token_index;

						code_terminal = d.FindTerminalIndex(token_index);
					}
					else
						code_terminal = 0;
				}
				else if (code_terminal)
				{
					code_terminal->SetCodeFlag(true);
					code_terminal->GetCodeText().push_back(line);
				}
				break;

				case parser_code_section:
				d.AddParserCode(line);
				break;

				case reduce_code_section:
				if (index_entry_flag)
				{
					index_entry_flag = false;

					if (line.substr(0,5) == "RULE:")
					{
						int				rule_index;
						string			index_string(line.substr(5));
						istrstream		ri(index_string.c_str());

						ri >> rule_index;

						code_rule = d.FindRuleIndex(rule_index);
					}
					else
						code_rule = 0;
				}
				else if (code_rule)
				{
					code_rule->SetCodeFlag(true);
					code_rule->GetCodeText().push_back(line);
				}
				break;

				case stack_code_section:
				if (!stack_code_flag)
				{
					d.ClearStackCode();

					stack_code_flag = true;
				}

				d.AddStackCode(line);
				break;

				case conflict_section:
				if (index_entry_flag)
				{
					index_entry_flag = false;

					SConflictResolution		c(line);

					if (!c.IsUndefined())
					{
						d.AddConflictResolution(c);

						code_conflict = d.FindConflictResolutionIndex(c.index);
					}
					else
						code_conflict = 0;
				}
				else if (code_conflict)
				{
					code_conflict->code_text.push_back(line);
				}
				break;

				case error_code_section:
				if (index_entry_flag)
				{
					index_entry_flag = false;

					SErrorCode		e(line);

					if (!e.IsUndefined())
					{
						d.AddErrorCode(e);

						code_error = d.FindErrorCodeIndex(e.entry);
					}
					else
						code_error = 0;
				}
				else if (code_error)
				{
					code_error->code_text.push_back(line);
				}
				break;

				case unit_rule_order_section:
				{
					StringToken::TokenStatus					token_status;
					vector<StringToken::CTokenValue *>			token_vector;
					vector<StringToken::CTokenValue *>::size_type	v_index;

					token_status = StringToken::GetIntegerTokenSequence(
										line, string("\t "), token_vector);

					if (token_status == StringToken::NoError)
					{
						for (v_index=0; v_index<token_vector.size(); v_index++)
							d.AddUnitRuleOrderIndex(
								token_vector[v_index]->GetIntegerValue());
					}

					StringToken::DeleteTokenVectorElements(token_vector);
				}
				break;

				case configuration_section:
				{
					StringToken::TokenStatus				token_status;
					vector<StringToken::CTokenValue *>		token_vector;

					token_status = StringToken::GetCharTokenSequence(
										line, string("\t :"), token_vector);

					if (token_status == StringToken::NoError &&
						token_vector.size() == 2)
					{
						c_iterator = configuration_map.find(
										token_vector[0]->GetTokenString());

						if (c_iterator != configuration_map.end())
						{
							string	s1(token_vector[1]->GetTokenString());

							switch ((*c_iterator).second)
							{
								case lexical_automaton_option:
								if (s1 == "TRUE")
									d.SetLexicalAutomatonCalculate(true);
								else if (s1 == "FALSE")
									d.SetLexicalAutomatonCalculate(false);
								break;

								case lexical_prefix_name_option:
								d.SetLexicalPrefixName(s1);
								break;

								case lexical_finite_state_machine_option:
								if (s1 == "REGEXP")
									d.SetLexicalFiniteStateMachineType(
											CCompilerData::RegularExpression);
								else if (s1 == "SINGLECHAR")
									d.SetLexicalFiniteStateMachineType(
											CCompilerData::SingleCharacter);
								break;

								case lexical_input_range_option:
								if (s1 == "128")
									d.SetLexicalInputRange(128);
								else if (s1 == "256")
									d.SetLexicalInputRange(256);
								break;

								case parser_automaton_option:
								if (s1 == "TRUE")
									d.SetParserAutomatonCalculate(true);
								else if (s1 == "FALSE")
									d.SetParserAutomatonCalculate(false);
								break;

								case parser_prefix_name_option:
								d.SetParserPrefixName(s1);
								break;

								case parser_type_option:
								if (s1 == "LR(1)")
									d.SetParserType(CCompilerData::LR_1);
								else if (s1 == "LALR(1)")
									d.SetParserType(CCompilerData::LALR_1);
								else if (s1 == "SLR(1)")
									d.SetParserType(CCompilerData::SLR_1);
								break;

								case base_prefix_name_option:
								d.SetBasePrefixName(s1);
								break;

								case generate_base_classes_option:
								if (s1 == "TRUE")
									d.SetGenerateBaseClasses(true);
								else if (s1 == "FALSE")
									d.SetGenerateBaseClasses(false);
								break;

								case parser_error_calculation_option:
								if (s1 == "NONE")
									d.SetParserErrorCalculation(
										CCompilerData::NoCalculation);
								else if (s1 == "APPROXIMATE")
									d.SetParserErrorCalculation(
										CCompilerData::ApproximateEssential);
								else if (s1 == "EXACT")
									d.SetParserErrorCalculation(
										CCompilerData::ExactEssential);
								break;

								case parser_error_numbering_option:
								if (s1 == "ONE")
									d.SetParserErrorFunctionNumbering(
										CCompilerData::OneNumber);
								else if (s1 == "STATE")
									d.SetParserErrorFunctionNumbering(
										CCompilerData::NumberPerState);
								else if (s1 == "ENTRY")
									d.SetParserErrorFunctionNumbering(
										CCompilerData::NumberPerEntry);
								break;

								case parser_error_aggregation_option:
								if (s1 == "ONE")
									d.SetParserErrorFunctionAggregation(
										CCompilerData::OneFunction);
								else if (s1 == "STATE")
									d.SetParserErrorFunctionAggregation(
										CCompilerData::FunctionPerState);
								else if (s1 == "ENTRY")
									d.SetParserErrorFunctionAggregation(
										CCompilerData::FunctionPerEntry);
								break;

								case parser_optimize_option:
								if (s1 == "UNIT_RULE_REDUCTION")
									d.SetParserOptimizeUnitRuleReduction(true);
								break;

								case generate_tty_code_option:
								if (s1 == "TRUE")
									d.SetGenerateTTYCode(true);
								else if (s1 == "FALSE")
									d.SetGenerateTTYCode(false);
								break;

								case regexp_match_table_option:
								if (s1 == "TRUE")
									d.SetRegexpMatchTable(true);
								else if (s1 == "FALSE")
									d.SetRegexpMatchTable(false);
								break;
							}
						}
					}
					else if (	token_status == StringToken::NoError &&
								token_vector.size() == 1 &&
								token_vector[0]->GetTokenString() ==
									"BASE_PREFIX_NAME")
					{
						d.SetBasePrefixName("");
					}

					StringToken::DeleteTokenVectorElements(token_vector);
				}
				break;

				case comment_section:
				d.AddCommentString(line);
				break;
			}
		}
	}

	if (!d.ValidateUnitRuleOrderList())
		d.ClearUnitRuleOrderList();

	d.SetupSymbolIndexes();

	return is;
}

int		IntegerLength(int i)
{
	ostrstream		os;

	os << i << '\0';

	string		s(os.str());

	return s.length();
}

string		&IntegerString(int i, string &s)
{
	ostrstream		os;

	os << i << '\0';

	return s = os.str();
}

void    CCompilerData::CalculateOutputWidths()
{
	terminal_name_width		= 0;
	error_token_name_width	= 0;
	nonterminal_name_width	= 0;
	symbol_name_width		= 0;
	rule_left_name_width	= 0;

	int		terminal_index_upper_bound		= 0;
	int		error_token_index_upper_bound	= 0;
	int		nonterminal_index_upper_bound	= 0;
	int		rule_index_upper_bound			= 0;
	int		precedence_index_upper_bound	= 0;
	int		conflict_index_upper_bound		= 0;
	int		error_entry_lower_bound			= 0;

	list<CTerminal *>::iterator		t_iterator = terminal_list.begin();
	list<CErrorToken *>::iterator	e_iterator = error_token_list.begin();
	list<CNonTerminal *>::iterator	n_iterator = nonterminal_list.begin();
	list<CRule *>::iterator			r_iterator = rule_list.begin();
	list<CPrecedence *>::iterator	p_iterator = precedence_list.begin();

	map<int, SConflictResolution *>::iterator
		cr_iterator = conflict_resolution_map.begin();

	map<int, SErrorCode *>::iterator
		ec_iterator = error_code_map.begin();

	while (t_iterator != terminal_list.end())
	{
		int		t_index	= (*t_iterator)->GetIndex();
		int		length	= (*t_iterator)->GetName().length();

		if (terminal_index_upper_bound < t_index)
			terminal_index_upper_bound = t_index;

		if (terminal_name_width < length)
			terminal_name_width = length;

		t_iterator++;
	}

	if (terminal_name_width < strlen("EOF"))
		terminal_name_width = strlen("EOF");

	while (e_iterator != error_token_list.end())
	{
		int		e_index	= (*e_iterator)->GetIndex();
		int		length	= (*e_iterator)->GetName().length();

		if (error_token_index_upper_bound < e_index)
			error_token_index_upper_bound = e_index;

		if (error_token_name_width < length)
			error_token_name_width = length;

		e_iterator++;
	}

	while (n_iterator != nonterminal_list.end())
	{
		int		n_index	= (*n_iterator)->GetIndex();
		int		length	= (*n_iterator)->GetName().length();

		if (nonterminal_index_upper_bound < n_index)
			nonterminal_index_upper_bound = n_index;

		if (nonterminal_name_width < length)
			nonterminal_name_width = length;

		n_iterator++;
	}

	while (r_iterator != rule_list.end())
	{
		int		r_index	= (*r_iterator)->GetIndex();
		int		length	= (*r_iterator)->GetLeft()->GetName().length();

		if (rule_index_upper_bound < r_index)
			rule_index_upper_bound = r_index;

		if (rule_left_name_width < length)
			rule_left_name_width = length;

		r_iterator++;
	}

	if (rule_left_name_width < nonterminal_name_width)
		rule_left_name_width = nonterminal_name_width;

	if (rule_left_name_width < strlen("ASTART"))
		rule_left_name_width = strlen("ASTART");

	while (p_iterator != precedence_list.end())
	{
		int		p_index = (*p_iterator)->GetIndex();

		if (precedence_index_upper_bound < p_index)
			precedence_index_upper_bound = p_index;

		p_iterator++;
	}

	while (cr_iterator != conflict_resolution_map.end())
	{
		int		cr_index = (*cr_iterator).second->GetIndex();

		if (conflict_index_upper_bound < cr_index)
			conflict_index_upper_bound = cr_index;

		cr_iterator++;
	}

	while (ec_iterator != error_code_map.end())
	{
		int		ec_entry = (*ec_iterator).second->GetEntry();

		if (error_entry_lower_bound > ec_entry)
			error_entry_lower_bound = ec_entry;

		ec_iterator++;
	}

	if (symbol_name_width < terminal_name_width)
		symbol_name_width = terminal_name_width;

	if (symbol_name_width < error_token_name_width)
		symbol_name_width = error_token_name_width;

	if (symbol_name_width < nonterminal_name_width)
		symbol_name_width = nonterminal_name_width;

	terminal_index_width	= IntegerLength(terminal_index_upper_bound);
	error_token_index_width	= IntegerLength(error_token_index_upper_bound);
	nonterminal_index_width	= IntegerLength(nonterminal_index_upper_bound);
	rule_index_width		= IntegerLength(rule_index_upper_bound);
	precedence_index_width	= IntegerLength(precedence_index_upper_bound);
	conflict_index_width	= IntegerLength(conflict_index_upper_bound);
	error_entry_width		= IntegerLength(error_entry_lower_bound);

	if (symbol_index_width < terminal_index_width)
		symbol_index_width = terminal_index_width;

	if (symbol_index_width < error_token_index_width)
		symbol_index_width = error_token_index_width;

	if (symbol_index_width < nonterminal_index_width)
		symbol_index_width = nonterminal_index_width;

	OutputWidths::terminal_index_width		= terminal_index_width;
	OutputWidths::terminal_name_width		= terminal_name_width;
	OutputWidths::error_token_index_width	= error_token_index_width;
	OutputWidths::error_token_name_width	= error_token_name_width;
	OutputWidths::nonterminal_index_width	= nonterminal_index_width;
	OutputWidths::nonterminal_name_width	= nonterminal_name_width;
	OutputWidths::symbol_index_width		= symbol_index_width;
	OutputWidths::symbol_name_width			= symbol_name_width;
	OutputWidths::rule_index_width			= rule_index_width;
	OutputWidths::rule_left_name_width		= rule_left_name_width;
	OutputWidths::precedence_index_width	= precedence_index_width;
	OutputWidths::conflict_index_width		= conflict_index_width;
	OutputWidths::error_entry_width			= error_entry_width;
}

ostream		&operator<<(ostream &os, const CTerminal &d)
{
	os	<< setw(OutputWidths::terminal_index_width)
		<< d.GetIndex() << "  ";

	if (d.GetFilterFlag())
		os << "FILTER  ";
	else
		os << "TOKEN   ";

	if (d.GetLiteralFlag())
		os << "LITERAL  ";
	else
		os << "REGEXP   ";

	string	pad1(OutputWidths::terminal_name_width -
					d.GetName().length() + 2, ' ');

	os	<< d.GetName() << pad1 << d.GetRegexp() << endl;

	return os;
}

ostream		&operator<<(ostream &os, const CErrorToken &d)
{
	os	<< setw(OutputWidths::error_token_index_width)
		<< d.GetIndex() << "  ";

	string	pad1(OutputWidths::error_token_name_width -
					d.GetName().length() + 2, ' ');

	os	<< d.GetName() << pad1 << d.GetSynchronizationLength() << endl;

	return os;
}

ostream		&operator<<(ostream &os, const CNonTerminal &d)
{
	os	<< setw(OutputWidths::nonterminal_index_width)
		<< d.GetIndex() << "  ";

	string	pad1(OutputWidths::nonterminal_name_width -
					d.GetName().length() + 2, ' ');

	os	<< d.GetName() << pad1;
		
	if (d.GetStartSymbolFlag())
		os << "START";

	os << endl;

	return os;
}

ostream		&operator<<(ostream &os, const CRule &d)
{
	os	<< setw(OutputWidths::rule_index_width)
		<< d.GetIndex();

	if (d.GetUnitRuleEliminationFlag())
		os << "  * ";
	else
		os << "    ";

	string	pad1(OutputWidths::rule_left_name_width -
					d.GetLeft()->GetName().length() + 2, ' ');

	os	<< d.GetLeft()->GetName() << pad1 << "->";

	const vector<const CSymbol *>			&sv = d.GetRight();
	vector<const CSymbol *>::size_type		v_index;

	for (v_index=0; v_index<sv.size(); v_index++)
		os << "  " << sv[v_index]->GetName();
		
	os << endl;

	return os;
}

ostream		&operator<<(ostream &os, const CPrecedence &d)
{
	os	<< setw(OutputWidths::precedence_index_width)
		<< d.GetIndex() << "  ";

	CPrecedence::Associativity	a = d.GetAssociativity();

	if (a == CPrecedence::LeftAssociative)
		os << "LEFT          ";
	else if (a == CPrecedence::RightAssociative)
		os << "RIGHT         ";
	else if (a == CPrecedence::NonAssociative)
		os << "NONASSOCIATIVE";
	else
		os << "NULL          ";

	const vector<SPrecedenceEntry>			&pv = d.GetEntries();
	vector<SPrecedenceEntry>::size_type		p_index;

	for (p_index=0; p_index<pv.size(); p_index++)
	{
		SPrecedenceEntry				entry	= pv[p_index];
		SPrecedenceEntry::EntryType		et		= entry.type;

		if (et == SPrecedenceEntry::TerminalEntry)
			os << "  " << entry.terminal->GetName();
		else if (et == SPrecedenceEntry::ErrorTokenEntry)
			os << "  " << entry.error_token->GetName();
		else if (et == SPrecedenceEntry::RuleEntry)
			os << "  RULE:" << entry.rule->GetIndex();
		else
			os << "  " << undefined_text;
	}

	os << endl;

	return os;
}

ostream		&operator<<(ostream &os, const SConflictResolution &d)
{
	os	<< setw(OutputWidths::conflict_index_width)
		<< d.GetIndex() << "  ";

	os	<< setw(10) << d.state;
	os	<< setw(10) << d.symbol;
	os	<< setw(10) << d.action << endl;

	list<string>::const_iterator		c_iterator = d.code_text.begin();

	while (c_iterator != d.code_text.end())
		os << *c_iterator++ << endl;

	return os;
}

ostream		&operator<<(ostream &os, const SErrorCode &d)
{
	os	<< setw(OutputWidths::error_entry_width)
		<< d.GetEntry() << "  ";

	os	<< setw(10) << d.state;
	os	<< setw(10) << d.symbol;
	
	if (d.token != -1)
		os << setw(10) << d.token;
	
	os	<< endl;

	list<string>::const_iterator		c_iterator = d.code_text.begin();

	while (c_iterator != d.code_text.end())
		os << *c_iterator++ << endl;

	return os;
}

ostream		&operator<<(ostream &os, const CCompilerData &d)
{
	os	<< "#COMMENT" << endl;

	const list<string>						&cm_list	= d.GetCommentList();
	list<string>::const_iterator			cm_iterator	= cm_list.begin();

	while (cm_iterator != cm_list.end())
		os << *cm_iterator++ << endl;

	os	<< "#CONTEXT" << endl;

	if (d.GetStartContext().length())
		os	<< d.GetStartContext() << endl;

	os	<< "#TERMINAL" << endl;

	const list<CTerminal *>					&t_list		= d.GetTerminalList();
	list<CTerminal *>::const_iterator		t_iterator	= t_list.begin();

	while (t_iterator != t_list.end())
		os << **t_iterator++;

	os	<< "#ERROR_TOKEN" << endl;

	const list<CErrorToken *>				&e_list		= d.GetErrorTokenList();
	list<CErrorToken *>::const_iterator		e_iterator	= e_list.begin();

	while (e_iterator != e_list.end())
		os << **e_iterator++;

	os	<< "#NONTERMINAL" << endl;

	const list<CNonTerminal *>				&n_list	= d.GetNonTerminalList();
	list<CNonTerminal *>::const_iterator	n_iterator	= n_list.begin();

	while (n_iterator != n_list.end())
		os << **n_iterator++;

	os	<< "#RULE" << endl;

	const list<CRule *>						&r_list		= d.GetRuleList();
	list<CRule *>::const_iterator			r_iterator	= r_list.begin();

	while (r_iterator != r_list.end())
		os << **r_iterator++;

	os	<< "#PRECENDENCE" << endl;

	const list<CPrecedence *>				&p_list		= d.GetPrecedenceList();
	list<CPrecedence *>::const_iterator		p_iterator	= p_list.begin();

	while (p_iterator != p_list.end())
		os << **p_iterator++;

	os	<< "#LEXICAL_CODE" << endl;

	const list<string>						&lc_list	= d.GetLexicalCode();
	list<string>::const_iterator			lc_iterator	= lc_list.begin();

	while (lc_iterator != lc_list.end())
		os << *lc_iterator++ << endl;

	t_iterator	= t_list.begin();

	while (t_iterator != t_list.end())
	{
		const CTerminal		*t = *t_iterator;

		if (t->GetCodeFlag())
		{
			os	<< "#TERMINAL_CODE" << endl;
			os	<< "TERMINAL:" << t->GetIndex() << endl;

			const list<string>				&cl			= t->GetCodeTextConst();
			list<string>::const_iterator	c_iterator	= cl.begin();

			while (c_iterator != cl.end())
				os << *c_iterator++ << endl;
		}

		t_iterator++;
	}

	os	<< "#PARSER_CODE" << endl;

	const list<string>					&pc_list	= d.GetParserCode();
	list<string>::const_iterator		pc_iterator	= pc_list.begin();

	while (pc_iterator != pc_list.end())
		os << *pc_iterator++ << endl;

	r_iterator	= r_list.begin();

	while (r_iterator != r_list.end())
	{
		const CRule		*r = *r_iterator;

		if (r->GetCodeFlag())
		{
			os	<< "#REDUCE_CODE" << endl;
			os	<< "RULE:" << r->GetIndex() << endl;

			const list<string>				&cl			= r->GetCodeTextConst();
			list<string>::const_iterator	c_iterator	= cl.begin();

			while (c_iterator != cl.end())
				os << *c_iterator++ << endl;
		}

		r_iterator++;
	}

	os	<< "#STACK_CODE" << endl;

	const list<string>					&sc_list	= d.GetStackCode();
	list<string>::const_iterator		sc_iterator	= sc_list.begin();

	while (sc_iterator != sc_list.end())
		os << *sc_iterator++ << endl;

	const map<int, SConflictResolution *>
		&cr_map = d.GetConflictResolutionMap();

	map<int, SConflictResolution *>::const_iterator
		cr_iterator = cr_map.begin();

	while (cr_iterator != cr_map.end())
	{
		os	<< "#CONFLICT" << endl;
		os	<< *(*cr_iterator++).second;
	}

	const map<int, SErrorCode *>
		&ec_map = d.GetErrorCodeMap();

	map<int, SErrorCode *>::const_iterator
		ec_iterator = ec_map.begin();

	while (ec_iterator != ec_map.end())
	{
		os	<< "#ERROR_CODE" << endl;
		os	<< *(*ec_iterator++).second;
	}

	os	<< "#UNIT_RULE_ORDER" << endl;

	const list<int>		&u_list = d.GetUnitRuleOrderList();

	if (u_list.size())
	{
		int		line_size	= 10;
		int		l_count		= u_list.size() / line_size;
		int		remainder	= u_list.size() % line_size;
		int		i;
		int		j;

		list<int>::const_iterator	u_iterator = u_list.begin();

		for (i=0; i<l_count; i++)
		{
			for (j=0; j<line_size-1; j++)
				os	<< setw(OutputWidths::rule_index_width)
					<< *u_iterator++ << "  ";

			os	<< setw(OutputWidths::rule_index_width)
				<< *u_iterator++ << endl;
		}

		for (i=0; i<remainder-1; i++)
			os	<< setw(OutputWidths::rule_index_width)
				<< *u_iterator++ << "  ";

		os	<< setw(OutputWidths::rule_index_width)
			<< *u_iterator++ << endl;
	}

	os	<< "#CONFIGURATION" << endl;

	os	<< "LEXICAL_AUTOMATON:";

	if (d.GetLexicalAutomatonCalculate())
		os << "TRUE" << endl;
	else
		os << "FALSE" << endl;

	os	<< "LEXICAL_PREFIX_NAME:" << d.GetLexicalPrefixName() << endl;

	os	<< "LEXICAL_FINITE_STATE_MACHINE:";

	CCompilerData::FiniteStateMachineType
		fsmt = d.GetLexicalFiniteStateMachineType();

	if (fsmt == CCompilerData::RegularExpression)
		os << "REGEXP" << endl;
	else if (fsmt == CCompilerData::SingleCharacter)
		os << "SINGLECHAR" << endl;
	else
		os << endl;

	os	<< "LEXICAL_INPUT_RANGE:" << d.GetLexicalInputRange() << endl;

	os	<< "PARSER_AUTOMATON:";

	if (d.GetParserAutomatonCalculate())
		os << "TRUE" << endl;
	else
		os << "FALSE" << endl;

	os	<< "PARSER_PREFIX_NAME:" << d.GetParserPrefixName() << endl;

	os	<< "PARSER_TYPE:";

	CCompilerData::ParserType
		pt = d.GetParserType();

	if (pt == CCompilerData::LALR_1)
		os << "LALR(1)" << endl;
	else if (pt == CCompilerData::SLR_1)
		os << "SLR(1)" << endl;
	else
		os << endl;

	os	<< "BASE_PREFIX_NAME:" << d.GetBasePrefixName() << endl;

	os	<< "GENERATE_BASE_CLASSES:";

	if (d.GetGenerateBaseClasses())
		os << "TRUE" << endl;
	else
		os << "FALSE" << endl;

	os	<< "PARSER_ERROR_CALCULATION:";

	CCompilerData::ErrorCalculation
		pec = d.GetParserErrorCalculation();

	if (pec == CCompilerData::NoCalculation)
		os << "NONE" << endl;
	else if (pec == CCompilerData::ApproximateEssential)
		os << "APPROXIMATE" << endl;
	else if (pec == CCompilerData::ExactEssential)
		os << "EXACT" << endl;
	else
		os << endl;

	os	<< "PARSER_ERROR_NUMBERING:";

	CCompilerData::ErrorFunctionNumbering
		efn = d.GetParserErrorFunctionNumbering();

	if (efn == CCompilerData::OneNumber)
		os << "ONE" << endl;
	else if (efn == CCompilerData::NumberPerState)
		os << "STATE" << endl;
	else if (efn == CCompilerData::NumberPerEntry)
		os << "ENTRY" << endl;
	else
		os << endl;

	os	<< "PARSER_ERROR_AGGREGATION:";

	CCompilerData::ErrorFunctionAggregation
		efa = d.GetParserErrorFunctionAggregation();

	if (efa == CCompilerData::OneFunction)
		os << "ONE" << endl;
	else if (efa == CCompilerData::FunctionPerState)
		os << "STATE" << endl;
	else if (efa == CCompilerData::FunctionPerEntry)
		os << "ENTRY" << endl;
	else
		os << endl;

	os	<< "PARSER_OPTIMIZE:";

	if (d.GetParserOptimizeUnitRuleReduction())
		os << "UNIT_RULE_REDUCTION" << endl;
	else
		os << endl;

	os	<< "GENERATE_TTY_CODE:";

	if (d.GetGenerateTTYCode())
		os << "TRUE" << endl;
	else
		os << "FALSE" << endl;

	os	<< "REGEXP_MATCH_TABLE:";

	if (d.GetRegexpMatchTable())
		os << "TRUE" << endl;
	else
		os << "FALSE" << endl;

	return os;
}

CCompilerData::~CCompilerData()
{
	ClearAllData();
}
