/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "machineView.h"

CMachineControl::CMachineControl(CMachineVBoxWidget *v)
	: CFrameFc((QWidget *)v, 1, 6), machine_vbox(v)
{
	grid_layout->setRowStretch(0, 1);

	grid_layout->setColStretch(0, 1);
	grid_layout->setColStretch(1, 0);
	grid_layout->setColStretch(2, 0);
	grid_layout->setColStretch(3, 0);
	grid_layout->setColStretch(4, 0);
	grid_layout->setColStretch(5, 0);

	view_label				= new CLabelFc("machine", this);
	parser_data_label		= new CLabelFc("Parser Data Type", this);
	parser_data_combo_box	= new CComboBoxFc(false, this,
									"parser data type string");
	state_label				= new CLabelFc("Machine State", this);
	state_spinbox			= new QSpinBox(0, 99999, 1, this);
	spacer					= new CSpacerFc(this);

	view_label->setFrameStyle(QFrame::Panel | QFrame::Raised);
	view_label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
	parser_data_label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);
	state_label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	grid_layout->addWidget(view_label,				0, 0);
	grid_layout->addWidget(parser_data_label,		0, 1);
	grid_layout->addWidget(parser_data_combo_box,	0, 2);
	grid_layout->addWidget(state_label,				0, 3);
	grid_layout->addWidget(state_spinbox,			0, 4);
	grid_layout->addWidget(spacer,					0, 5);
}

void	CMachineControl::ResetControl(const list<string> &s, int type, int c)
{
	disconnect(	parser_data_combo_box,
				SIGNAL(activated(int)),
				this,
				SLOT(SelectParserData(int)));

	disconnect(	state_spinbox,
				SIGNAL(valueChanged(int)),
				this,
				SLOT(SelectState(int)));

	parser_data_combo_box->clear();

	list<string>::const_iterator	s_iterator = s.begin();

	while (s_iterator != s.end())
		parser_data_combo_box->insertItem((*s_iterator++).c_str());

	parser_data_combo_box->setCurrentItem(type);

	state_spinbox->setMaxValue(c);
	state_spinbox->setValue(0);

	connect(parser_data_combo_box,
			SIGNAL(activated(int)),
			this,
			SLOT(SelectParserData(int)));

	connect(state_spinbox,
			SIGNAL(valueChanged(int)),
			this,
			SLOT(SelectState(int)));
}

void	CMachineControl::SelectParserData(int type)
{
	machine_vbox->SelectParserData(type);
}

void	CMachineControl::SelectState(int state)
{
	machine_vbox->SelectState(state);
}

CMachineVBoxWidget::CMachineVBoxWidget(
									QWidget *parent,
									CCompilerInterface *ci,
									SMainViewControl *mvc,
									COptionData::SSubViewData *d,
									CMachineView *mv)
	: CVBoxWidget(parent, ci, mvc), compiler_interface(ci),
		subview_data(d), machine_view(mv),
		module_ptr_map(mv->GetModulePtrMap())
{
	machine_control	= new CMachineControl(this);
	multi_line_edit = new CMultiLineEditFc(this, "");

	multi_line_edit->setWordWrap(Q3MultiLineEdit::NoWrap);

	setSpacing(0);
	setStretchFactor(machine_control, 0);
	setStretchFactor(multi_line_edit, 1);

	compiler_data		= compiler_interface->GetCompilerData();
	compiler_calculate	= compiler_interface->GetCompilerCalculate();
}

CMachineVBoxWidget::~CMachineVBoxWidget()
{
}

void	CMachineVBoxWidget::InitializeModulePointers()
{
}

void	CMachineVBoxWidget::UpdateView(int type)
{
	data_type		= type;
	machine_state	= 0;

	list<string>	d_types;

	switch (compiler_data->GetParserType())
	{
		case CCompilerData::LR_1:
		d_types.push_back("LR(1) Machine");
		break;

		case CCompilerData::LALR_1:
		d_types.push_back("LR(0) Machine");
		d_types.push_back("LALR(1) Lookahead");
		break;

		case CCompilerData::SLR_1:
		d_types.push_back("LR(0) Machine");
		d_types.push_back("SLR(1) Lookahead");
		break;
	}

	if (compiler_interface->GetParserCalculated())
	{
		int		state_count = compiler_interface->GetParser()->machine_count;

		machine_control->ResetControl(d_types, data_type, state_count - 1);
	}
	else
		machine_control->ResetControl(d_types, data_type, 0);

	DisplayData();
}

void	CMachineVBoxWidget::SetTabStopWidth(int count)
{
	QFontMetrics	fm(font());

	int		w = fm.width(QChar('Z'));

	multi_line_edit->setTabStopWidth(count * w);
}

void	CMachineVBoxWidget::SelectParserData(int type)
{
	data_type = type;

	DisplayData();
}

void	CMachineVBoxWidget::SelectState(int state)
{
	machine_state = state;

	DisplayData();
}

void	CMachineVBoxWidget::DisplayData()
{
	multi_line_edit->clear();

	if (!compiler_interface->GetParserCalculated())
		return;

	string	s;

	switch (compiler_data->GetParserType())
	{
		case CCompilerData::LR_1:
		switch (data_type)
		{
			case 0:
			compiler_calculate->LR1MachineStateString(machine_state, s);
			break;
		}
		break;

		case CCompilerData::LALR_1:
		switch (data_type)
		{
			case 0:
			compiler_calculate->LR0MachineStateString(machine_state, s);
			break;

			case 1:
			compiler_calculate->LALR1LookaheadString(machine_state, s);
			break;
		}
		break;

		case CCompilerData::SLR_1:
		switch (data_type)
		{
			case 0:
			compiler_calculate->LR0MachineStateString(machine_state, s);
			break;

			case 1:
			compiler_calculate->SLR1LookaheadString(machine_state, s);
			break;
		}
		break;
	}

	multi_line_edit->append(s.c_str());
	multi_line_edit->setCursorPosition(0, 0);
}

void	CMachineVBoxWidget::RestoreUndoData()
{
	compiler_data		= compiler_interface->GetCompilerData();
	compiler_calculate	= compiler_interface->GetCompilerCalculate();
}

CMachineView::CMachineView(	QWidget *p,
							CViewNode *n,
							CInterfaceControl *ic,
							SMainViewControl *mvc,
							const string &dfs,
							CPaletteData *dpd,
							COptionData::SSubViewData *d)
	: CSubView(p, n, ic, mvc, dfs, dpd, d)
{
	frame = new CMachineVBoxWidget(
		p, dynamic_cast<CCompilerInterface *>(ic), mvc, d, this);

	QFont		view_node_font;
	QPalette	view_node_palette;

	if (SetupViewNodeFont(view_node_font))
		frame->setFont(view_node_font);
	else
		frame->setFont(frame->font());

	if (SetupViewNodePalette(view_node_palette))
		frame->setPalette(view_node_palette);
	else
		frame->setPalette(frame->palette());
}

CMachineView::~CMachineView()
{
	delete frame;
}

void	CMachineView::SetHighlightPalette(CPaletteData *pd)
{
	restore_palette = frame->palette();

	QPalette	highlight_palette;

	pd->SetupPalette(highlight_palette);

	frame->setPalette(highlight_palette);
}

void	CMachineView::RestorePalette()
{
	frame->setPalette(restore_palette);
}

void	CMachineView::InitializeModulePointers()
{
	frame->InitializeModulePointers();
}

void	CMachineView::UpdateView(int type)
{
	frame->UpdateView(type);
}

void	CMachineView::SetTabStopWidth(int count)
{
	frame->SetTabStopWidth(count);
}

void	CMachineView::RestoreUndoData()
{
	frame->RestoreUndoData();
}
