/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "optionData.h"

bool	FontSpecifier::SetFontSpecifier(QFont &f, const string &s)
{
	bool									raw_name_flag;
	StringToken::TokenStatus				token_status;
	vector<StringToken::TokenType>			token_types;
	vector<StringToken::CTokenValue *>		token_vector;

	token_types.push_back(StringToken::CharToken);
	token_types.push_back(StringToken::CharToken);
	token_types.push_back(StringToken::IntegerToken);

	token_status = StringToken::GetTokenSequence(
						s, string(";"), token_types, token_vector);

	if (token_status == StringToken::NoError)
	{
		QFontDatabase	font_database;
		string			family;
		string			style;

		NormalizeString(family, token_vector[0]->GetTokenString());
		NormalizeString(style, token_vector[1]->GetTokenString());
		
		f = font_database.font(	family.c_str(),
								style.c_str(),
								token_vector[2]->GetIntegerValue());

		raw_name_flag = false;
	}
	else
	{
		f.setRawName(s.c_str());

		raw_name_flag = true;
	}

	StringToken::DeleteTokenVectorElements(token_vector);

	return raw_name_flag;
}

void	FontSpecifier::NormalizeString(string &s1, const string &s2)
{
	int		n1 = s2.find_first_not_of("\t ");
	int		n2 = s2.find_last_not_of("\t ");

	s1 = string(s2, n1, n2 - n1 + 1);
}

QPixmap		*CXpmImage::CreatePixmap()
{
	if (!pixmap)
	{
		char	*char_data[xpm_data.size() + 1];

		int								count		= 0;
		list<string>::const_iterator	s_iterator	= xpm_data.begin();

		while (s_iterator != xpm_data.end())
		{
			int		s_length = (*s_iterator).length();

			char_data[count] = new char[s_length + 1];

			strcpy(char_data[count], (*s_iterator).c_str());
			
			count++;
			s_iterator++;
		}

		char_data[count] = 0;

		if (ValidateXpmData(char_data, count))
			pixmap = new QPixmap(const_cast<const char **>(char_data));
		else
			pixmap = 0;

		for (int i=0; i<count; i++)
			delete [] char_data[i];
	}

	return pixmap;
}

bool    CXpmImage::ValidateXpmData(char *data[], int count)
{
	if (!count)
		return false;

	StringToken::TokenStatus				status;
	vector<StringToken::TokenType>			token_types;
	vector<StringToken::CTokenValue *>		token_values;

	int		width;
	int		height;
	int		number_of_colors;
	int		characters_per_color;
	int		i;

	for (i=0; i<4; i++)
		token_types.push_back(StringToken::IntegerToken);

	status = StringToken::GetTokenSequence(	data[0], " ",
											token_types, token_values);

	if (status != StringToken::NoError)
	{
		StringToken::DeleteTokenVectorElements(token_values);
		return false;
	}
	else
	{
		width					= token_values[0]->GetIntegerValue();
		height					= token_values[1]->GetIntegerValue();
		number_of_colors		= token_values[2]->GetIntegerValue();
		characters_per_color	= token_values[3]->GetIntegerValue();
	}

	StringToken::DeleteTokenVectorElements(token_values);

	if (count != 1 + number_of_colors + height)
		return false;

	for (i=number_of_colors+1; i<number_of_colors+height+1; i++)
		if (strlen(data[i]) != width * characters_per_color)
			return false;

	return true;
}

ostream &operator<<(ostream &os, const CXpmImage &p)
{
	os	<< "XPM_IMAGE" << endl;

	const list<string>	&p_data = p.GetXpmData();

	list<string>::const_iterator	s_iterator	= p_data.begin();

	while (s_iterator != p_data.end())
	{
		os	<< "\"" << (*s_iterator).c_str() << "\"" << endl;

		s_iterator++;
	}

	return os;
}

map<string, QPalette::ColorGroup>		CPaletteData::string_to_color_group_map;
map<string, QColorGroup::ColorRole>		CPaletteData::string_to_color_role_map;
map<string, Qt::BrushStyle>          	CPaletteData::string_to_brush_style_map;
map<QPalette::ColorGroup, string>		CPaletteData::color_group_to_string_map;
map<QColorGroup::ColorRole, string>		CPaletteData::color_role_to_string_map;
map<Qt::BrushStyle, string>          	CPaletteData::brush_style_to_string_map;

CPaletteData::~CPaletteData()
{
	delete precalculate_button_color;
	delete precalculate_background_color;

	if (group_role_specifiers)
		for_each(	group_role_specifiers->begin(),
					group_role_specifiers->end(),
					destroy_map_T<SGroupRoleEntry, SGroupRoleSpecifier *>());

	delete group_role_specifiers;
}

CPaletteData::CPaletteData(const CPaletteData &pd)
{
	if (pd.precalculate_button_color)
		precalculate_button_color = pd.precalculate_button_color->clone();
	else
		precalculate_button_color = 0;

	if (pd.precalculate_background_color)
		precalculate_background_color =
			pd.precalculate_background_color->clone();
	else
		precalculate_background_color = 0;

	if (pd.group_role_specifiers)
	{
		group_role_specifiers = new GroupRoleToSpecifierMap;

		GroupRoleToSpecifierMapIterator		grs_iterator =
			pd.group_role_specifiers->begin();

		while (grs_iterator != pd.group_role_specifiers->end())
		{
			(*group_role_specifiers)[(*grs_iterator).first] =
				(*grs_iterator).second->clone();

			grs_iterator++;
		}
	}
	else
		group_role_specifiers = 0;
}

void	CPaletteData::SetupGroupRoleMaps()
{
	int	i = 0;
	while (group_names[i].name)
	{
		string_to_color_group_map[string(group_names[i].name)] =
													group_names[i].group_id;

		color_group_to_string_map[group_names[i].group_id] =
													string(group_names[i].name);

		i++;
	}

	i = 0;
	while (role_names[i].name)
	{
		string_to_color_role_map[string(role_names[i].name)] =
													role_names[i].role_id;

		color_role_to_string_map[role_names[i].role_id] =
													string(role_names[i].name);

		i++;
	}

	i = 0;
	while (brush_style_names[i].name)
	{
		string_to_brush_style_map[string(brush_style_names[i].name)] =
											brush_style_names[i].style_id;

		brush_style_to_string_map[brush_style_names[i].style_id] =
											string(brush_style_names[i].name);

		i++;
	}
}

void	CPaletteData::SetGroupRoleSpecifierMap(
							map<SGroupRoleEntry, SGroupRoleSpecifier *> *m)
{
	if (group_role_specifiers)
		for_each(	group_role_specifiers->begin(),
					group_role_specifiers->end(),
					destroy_map_T<SGroupRoleEntry, SGroupRoleSpecifier *>());

	delete group_role_specifiers;

	group_role_specifiers = m;
}

void	CPaletteData::SetGroupRoleSpecifier(const SGroupRoleEntry &gre,
											SGroupRoleSpecifier *grs)
{
	if (!group_role_specifiers)
		group_role_specifiers = new map<SGroupRoleEntry, SGroupRoleSpecifier *>;
	else
	{
		GroupRoleToSpecifierMapIterator
			m_iterator = group_role_specifiers->find(gre);

		if (m_iterator != group_role_specifiers->end())
			delete (*m_iterator).second;
	}

	(*group_role_specifiers)[gre] = grs;
}

void    CPaletteData::DeleteGroupRoleSpecifier(const SGroupRoleEntry &gre)
{
	if (!group_role_specifiers)
		return;

	GroupRoleToSpecifierMapIterator
		m_iterator = group_role_specifiers->find(gre);

	if (m_iterator != group_role_specifiers->end())
	{
		delete (*m_iterator).second;

		group_role_specifiers->erase(m_iterator);
	}

	if (!group_role_specifiers->size())
	{
		delete group_role_specifiers;

		group_role_specifiers = 0;
	}
}

void	CPaletteData::SetupPalette(QPalette &p)
{
	if (precalculate_button_color && precalculate_background_color)
	{
		QColor		button;
		QColor		background;

		precalculate_button_color->SetupColor(button);
		precalculate_background_color->SetupColor(background);

		p = QPalette(button, background);
	}

	if (group_role_specifiers)
		for_each(	group_role_specifiers->begin(),
					group_role_specifiers->end(),
					set_group_role_entries(p));
}

ostream     &OutputStringData(ostream &os, const string &s)
{
	static string		escape_string(" \\\",:");
	static QRegExp		regexp("((\\-|\\+)?\\d+)|(#\\S*)");

	if (!s.length())
		return os << "\"\"";
	else if (s.find_first_of(escape_string) == string::npos)
	{
		bool	quote_flag = regexp.exactMatch(s.c_str());

		if (quote_flag)
			os << '\"';

		os << s;

		if (quote_flag)
			os << '\"';

		return os;
	}
	else
	{
		string::size_type	i;

		os << '\"';

		for (i=0; i<s.length(); i++)
			if (s[i] == '\\' || s[i] == '\"' || s[i] == ':')
				os << '\\' << s[i];
			else
				os << s[i];

		return os << '\"';
	}
}

ostream &CPaletteData::SStyleBrushSpecifier::OutputData(ostream &os) const
{
    return os	<< "BRUSH  " << *color_specifier << "  "
				<< brush_style_to_string_map[style];
}

ostream &CPaletteData::SXpmImageBrushSpecifier::OutputData(ostream &os) const
{
    return os	<< "BRUSH  " << *color_specifier << endl << *xpm_image;
}

ostream	&CPaletteData::SGroupRoleEntry::OutputData(ostream &os) const
{
	os << "GROUP_ROLE  ";
	os.setf(ios::left, ios::adjustfield);
	os.width(10);
	os << color_group_to_string_map[group].c_str();
	os.width(17);
	os << color_role_to_string_map[role].c_str();
	os.setf(ios::right, ios::adjustfield);

	return os;
}

ostream	&CPaletteData::SColorGroupRoleSpecifier::OutputData(ostream &os) const
{
	return os << *color_specifier;
}

ostream	&CPaletteData::SBrushGroupRoleSpecifier::OutputData(ostream &os) const
{
	return os << *brush_specifier;
}

ostream &operator<<(ostream &os, const CPaletteData::SColorSpecifier &d)
{
	return d.OutputData(os);
}

ostream &operator<<(ostream &os, const CPaletteData::SBrushSpecifier &d)
{
	return d.OutputData(os);
}

ostream &operator<<(ostream &os, const CPaletteData::SGroupRoleEntry &d)
{
	return d.OutputData(os);
}

ostream &operator<<(ostream &os, const CPaletteData::SGroupRoleSpecifier &d)
{
	return d.OutputData(os);
}

ostream &operator<<(ostream &os, const CPaletteData &d)
{
	os << "PALETTE" << endl;

	if (d.precalculate_button_color || d.precalculate_background_color)
	{
		if (d.precalculate_button_color)
			os	<< "BUTTON_COLOR  "
				<< *d.precalculate_button_color << endl;

		if (d.precalculate_background_color)
			os	<< "BACKGROUND_COLOR  "
				<< *d.precalculate_background_color << endl;
	}

	if (d.group_role_specifiers && d.group_role_specifiers->size())
	{
		for_each(	d.group_role_specifiers->begin(),
					d.group_role_specifiers->end(),
					output_group_role_specifiers(os));
	}

	return os;
}

COptionData::SViewNodeData::SViewNodeData(const SViewNodeData &vnd)
{
	if (vnd.index)
	{
		index = new vector<int>;

		for (vector<int>::size_type i=0; i<vnd.index->size(); i++)
			index->push_back((*vnd.index)[i]);
	}
	else
		index = 0;

	font_string = vnd.font_string;

	if (vnd.palette_data)
		palette_data = new CPaletteData(*vnd.palette_data);
	else
		palette_data = 0;
}

COptionData::SSplitterData::SSplitterData(const SSplitterData &sd)
	: SViewNodeData(sd)
{
	orientation = sd.orientation;

	if (sd.pane_size)
	{
		pane_size = new vector<int>;

		for (vector<int>::size_type i=0; i<sd.pane_size->size(); i++)
			pane_size->push_back((*sd.pane_size)[i]);
	}
	else
		pane_size = 0;
}

COptionData::SSubViewData::SInterfaceSpecifier::
	SInterfaceSpecifier(const SInterfaceSpecifier &is)
{
	main_view_name	= is.main_view_name;
	subview_name	= is.subview_name;

	if (is.index)
	{
		index = new vector<int>;

		for (vector<int>::size_type i=0; i<is.index->size(); i++)
			index->push_back((*is.index)[i]);
	}
	else
		index = 0;
}

COptionData::SSubViewData::SSubViewData(const SSubViewData &svd)
	: SViewNodeData(svd)
{
	subview_name = svd.subview_name;

	if (svd.interface_specifiers)
	{
		interface_specifiers = new vector<SInterfaceSpecifier *>;

		for (	vector<SInterfaceSpecifier *>::size_type i=0;
				i<svd.interface_specifiers->size(); i++)
		{
			SInterfaceSpecifier	*is  =
				new SInterfaceSpecifier(*(*svd.interface_specifiers)[i]);

			interface_specifiers->push_back(is);
		}
	}
	else
		interface_specifiers = 0;
}

COptionData::SSubViewData::~SSubViewData()
{
	if (interface_specifiers)
		for_each(	interface_specifiers->begin(),
					interface_specifiers->end(),
					destroy_T<SInterfaceSpecifier *>());

	delete interface_specifiers;
}

void	COptionData::SSubViewData::SetInterfaceSpecifiers(
										vector<SInterfaceSpecifier *> *v)
{
	if (interface_specifiers)
		for_each(	interface_specifiers->begin(),
					interface_specifiers->end(),
					destroy_T<SInterfaceSpecifier *>());

	delete interface_specifiers;

	interface_specifiers = v;
}

void	COptionData::SSubViewData::AddInterfaceSpecifier(
											SInterfaceSpecifier *is)
{
	if (!interface_specifiers)
		interface_specifiers = new vector<SInterfaceSpecifier *>;

	interface_specifiers->push_back(is);
}

COptionData::SMainViewData::~SMainViewData()
{
	if (view_node_data_ptrs)
		for_each(	view_node_data_ptrs->begin(),
					view_node_data_ptrs->end(),
					destroy_T<SViewNodeData *>());

	delete view_node_data_ptrs;
}

COptionData::SMainViewData::SMainViewData(const SMainViewData &mvd)
{
	main_view_name			= mvd.main_view_name;
	default_display_flag	= mvd.default_display_flag;
	position				= mvd.position;
	size					= mvd.size;

	if (mvd.view_node_data_ptrs)
	{
		view_node_data_ptrs = new vector<SViewNodeData *>;

		for (	vector<SViewNodeData *>::size_type i=0;
				i<mvd.view_node_data_ptrs->size(); i++)
		{
			SViewNodeData	*vnd = (*mvd.view_node_data_ptrs)[i]->clone();

			view_node_data_ptrs->push_back(vnd);
		}
	}
	else
		view_node_data_ptrs = 0;
}

void	COptionData::SMainViewData::SetViewNodeDataPtrs(
											vector<SViewNodeData *> *v)
{
	if (view_node_data_ptrs)
		for_each(	view_node_data_ptrs->begin(),
					view_node_data_ptrs->end(),
					destroy_T<SViewNodeData *>());

	delete view_node_data_ptrs;

	view_node_data_ptrs = v;
}

void	COptionData::SMainViewData::AddViewNodeDataPtr(SViewNodeData *vnd)
{
	if (!view_node_data_ptrs)
		view_node_data_ptrs = new vector<SViewNodeData *>;

	view_node_data_ptrs->push_back(vnd);
}

COptionData::SApplicationData::SApplicationData(const SApplicationData &ad)
{
	position	= ad.position;
	size		= ad.size;
	font_string	= ad.font_string;

	if (ad.palette_data)
		palette_data = new CPaletteData(*ad.palette_data);
	else
		palette_data = 0;
}

COptionData::~COptionData()
{
	delete application_data;
	delete help_data;

	if (main_view_ptrs)
		for_each(	main_view_ptrs->begin(),
					main_view_ptrs->end(),
					destroy_T<SMainViewData *>());
	
	if (auxiliary_palettes)
		for_each(	auxiliary_palettes->begin(),
					auxiliary_palettes->end(),
					destroy_map_T<string, CPaletteData *>());

	if (auxiliary_colors)
		for_each(	auxiliary_colors->begin(),
					auxiliary_colors->end(),
					destroy_map_T<string, CPaletteData::SColorSpecifier *>());

	if (auxiliary_brushes)
		for_each(	auxiliary_brushes->begin(),
					auxiliary_brushes->end(),
					destroy_map_T<string, CPaletteData::SBrushSpecifier *>());

	if (auxiliary_xpm_images)
		for_each(	auxiliary_xpm_images->begin(),
					auxiliary_xpm_images->end(),
					destroy_map_T<string, CXpmImage *>());

	delete main_view_ptrs;
	delete auxiliary_palettes;
	delete auxiliary_colors;
	delete auxiliary_brushes;
	delete auxiliary_xpm_images;
	delete auxiliary_fonts;
	delete auxiliary_strings;
}

void	COptionData::SetMainViewPtrs(vector<SMainViewData *> *v, bool df)
{
	if (df && main_view_ptrs)
		for_each(	main_view_ptrs->begin(),
					main_view_ptrs->end(),
					destroy_T<SMainViewData *>());
	
	delete main_view_ptrs;

	main_view_ptrs = v;
}

void	COptionData::SetAuxiliaryPalettes(map<string, CPaletteData *> *m)
{
	if (auxiliary_palettes)
		for_each(	auxiliary_palettes->begin(),
					auxiliary_palettes->end(),
					destroy_map_T<string, CPaletteData *>());

	delete auxiliary_palettes;

	auxiliary_palettes = m;
}

void	COptionData::SetAuxiliaryColors(map<string,
										CPaletteData::SColorSpecifier *> *m)
{
	if (auxiliary_colors)
		for_each(	auxiliary_colors->begin(),
					auxiliary_colors->end(),
					destroy_map_T<string, CPaletteData::SColorSpecifier *>());

	delete auxiliary_colors;

	auxiliary_colors = m;
}

void	COptionData::SetAuxiliaryBrushes(map<string,
										CPaletteData::SBrushSpecifier *> *m)
{
	if (auxiliary_brushes)
		for_each(	auxiliary_brushes->begin(),
					auxiliary_brushes->end(),
					destroy_map_T<string, CPaletteData::SBrushSpecifier *>());

	delete auxiliary_brushes;

	auxiliary_brushes = m;
}

void	COptionData::SetAuxiliaryXpmImages(map<string, CXpmImage *> *m)
{
	if (auxiliary_xpm_images)
		for_each(	auxiliary_xpm_images->begin(),
					auxiliary_xpm_images->end(),
					destroy_map_T<string, CXpmImage *>());

	delete auxiliary_xpm_images;

	auxiliary_xpm_images = m;
}

void	COptionData::AddMainViewPtr(SMainViewData *p)
{
	if (!main_view_ptrs)
		main_view_ptrs = new vector<SMainViewData *>;

	main_view_ptrs->push_back(p);
}

void	COptionData::AddAuxiliaryPalette(	const string &id,
											CPaletteData *pd)
{
	if (!auxiliary_palettes)
		auxiliary_palettes = new map<string, CPaletteData *>;
	else
	{
		map<string, CPaletteData *>::iterator	m_iterator =
			auxiliary_palettes->find(id);

		if (m_iterator != auxiliary_palettes->end())
			delete (*m_iterator).second;
	}

	(*auxiliary_palettes)[id] = pd;
}

void	COptionData::AddAuxiliaryColor(	const string &id,
										CPaletteData::SColorSpecifier *cs)
{
	if (!auxiliary_colors)
		auxiliary_colors = new map<string, CPaletteData::SColorSpecifier *>;
	else
	{
		map<string, CPaletteData::SColorSpecifier *>::iterator	m_iterator =
			auxiliary_colors->find(id);

		if (m_iterator != auxiliary_colors->end())
			delete (*m_iterator).second;
	}

	(*auxiliary_colors)[id] = cs;
}

void	COptionData::AddAuxiliaryBrush(	const string &id,
										CPaletteData::SBrushSpecifier *bs)
{
	if (!auxiliary_brushes)
		auxiliary_brushes = new map<string, CPaletteData::SBrushSpecifier *>;
	else
	{
		map<string, CPaletteData::SBrushSpecifier *>::iterator	m_iterator =
			auxiliary_brushes->find(id);

		if (m_iterator != auxiliary_brushes->end())
			delete (*m_iterator).second;
	}

	(*auxiliary_brushes)[id] = bs;
}

void	COptionData::AddAuxiliaryXpmImage(	const string &id,
											CXpmImage *xi)
{
	if (!auxiliary_xpm_images)
		auxiliary_xpm_images = new map<string, CXpmImage *>;
	else
	{
		map<string, CXpmImage *>::iterator	m_iterator =
			auxiliary_xpm_images->find(id);

		if (m_iterator != auxiliary_xpm_images->end())
			delete (*m_iterator).second;
	}

	(*auxiliary_xpm_images)[id] = xi;
}

void	COptionData::AddAuxiliaryFont(	const string &id,
										const string &fs)
{
	if (!auxiliary_fonts)
		auxiliary_fonts = new map<string, string>;

	(*auxiliary_fonts)[id] = fs;
}

void	COptionData::AddAuxiliaryString(const string &id,
										const string &s)
{
	if (!auxiliary_strings)
		auxiliary_strings = new map<string, string>;

	(*auxiliary_strings)[id] = s;
}

CPaletteData	*COptionData::GetAuxiliaryPalette(const string &id)
{
	if (!auxiliary_palettes)
		return 0;

	map<string, CPaletteData *>::iterator  m_iterator =
		auxiliary_palettes->find(id);

	if (m_iterator != auxiliary_palettes->end())
		return (*m_iterator).second;
	else
		return 0;
}

CPaletteData::SColorSpecifier	*COptionData::GetAuxiliaryColor(
															const string &id)
{
	if (!auxiliary_colors)
		return 0;

	map<string, CPaletteData::SColorSpecifier *>::iterator  m_iterator =
		auxiliary_colors->find(id);

	if (m_iterator != auxiliary_colors->end())
		return (*m_iterator).second;
	else
		return 0;
}

CPaletteData::SBrushSpecifier	*COptionData::GetAuxiliaryBrush(
															const string &id)
{
	if (!auxiliary_brushes)
		return 0;

	map<string, CPaletteData::SBrushSpecifier *>::iterator  m_iterator =
		auxiliary_brushes->find(id);

	if (m_iterator != auxiliary_brushes->end())
		return (*m_iterator).second;
	else
		return 0;
}

CXpmImage	*COptionData::GetAuxiliaryXpmImage(const string &id)
{
	if (!auxiliary_xpm_images)
		return 0;

	map<string, CXpmImage *>::iterator  m_iterator =
		auxiliary_xpm_images->find(id);

	if (m_iterator != auxiliary_xpm_images->end())
		return (*m_iterator).second;
	else
		return 0;
}

string		*COptionData::GetAuxiliaryFont(const string &id)
{
	if (!auxiliary_fonts)
		return 0;

	map<string, string>::iterator  m_iterator =
		auxiliary_fonts->find(id);

	if (m_iterator != auxiliary_fonts->end())
		return &(*m_iterator).second;
	else
		return 0;
}

string		*COptionData::GetAuxiliaryString(const string &id)
{
	if (!auxiliary_strings)
		return 0;

	map<string, string>::iterator  m_iterator =
		auxiliary_strings->find(id);

	if (m_iterator != auxiliary_strings->end())
		return &(*m_iterator).second;
	else
		return 0;
}

void    COptionData::DeleteAuxiliaryPalette(const string &id)
{
	if (!auxiliary_palettes)
		return;

	map<string, CPaletteData *>::iterator  m_iterator =
		auxiliary_palettes->find(id);

	if (m_iterator != auxiliary_palettes->end())
	{
		delete (*m_iterator).second;

		auxiliary_palettes->erase(m_iterator);
	}

	if (!auxiliary_palettes->size())
	{
		delete auxiliary_palettes;

		auxiliary_palettes = 0;
	}
}

void    COptionData::DeleteAuxiliaryColor(const string &id)
{
	if (!auxiliary_colors)
		return;

	map<string, CPaletteData::SColorSpecifier *>::iterator  m_iterator =
		auxiliary_colors->find(id);

	if (m_iterator != auxiliary_colors->end())
	{
		delete (*m_iterator).second;

		auxiliary_colors->erase(m_iterator);
	}

	if (!auxiliary_colors->size())
	{
		delete auxiliary_colors;

		auxiliary_colors = 0;
	}
}

void    COptionData::DeleteAuxiliaryBrush(const string &id)
{
	if (!auxiliary_brushes)
		return;

	map<string, CPaletteData::SBrushSpecifier *>::iterator  m_iterator =
		auxiliary_brushes->find(id);

	if (m_iterator != auxiliary_brushes->end())
	{
		delete (*m_iterator).second;

		auxiliary_brushes->erase(m_iterator);
	}

	if (!auxiliary_brushes->size())
	{
		delete auxiliary_brushes;

		auxiliary_brushes = 0;
	}
}

void    COptionData::DeleteAuxiliaryXpmImage(const string &id)
{
	if (!auxiliary_xpm_images)
		return;

	map<string, CXpmImage *>::iterator  m_iterator =
		auxiliary_xpm_images->find(id);

	if (m_iterator != auxiliary_xpm_images->end())
	{
		delete (*m_iterator).second;

		auxiliary_xpm_images->erase(m_iterator);
	}

	if (!auxiliary_xpm_images->size())
	{
		delete auxiliary_xpm_images;

		auxiliary_xpm_images = 0;
	}
}

void    COptionData::DeleteAuxiliaryFont(const string &id)
{
	if (!auxiliary_fonts)
		return;

	map<string, string>::iterator  m_iterator =
		auxiliary_fonts->find(id);

	if (m_iterator != auxiliary_fonts->end())
	{
		auxiliary_fonts->erase(m_iterator);
	}

	if (!auxiliary_fonts->size())
	{
		delete auxiliary_fonts;

		auxiliary_fonts = 0;
	}
}

void    COptionData::DeleteAuxiliaryString(const string &id)
{
	if (!auxiliary_strings)
		return;

	map<string, string>::iterator  m_iterator =
		auxiliary_strings->find(id);

	if (m_iterator != auxiliary_strings->end())
	{
		auxiliary_strings->erase(m_iterator);
	}

	if (!auxiliary_strings->size())
	{
		delete auxiliary_strings;

		auxiliary_strings = 0;
	}
}

COptionData::COptionData(const COptionData &od)
{
	if (od.application_data)
		application_data = new SApplicationData(*od.application_data);
	else
		application_data = 0;

	if (od.help_data)
		help_data = new SHelpData(*od.help_data);
	else
		help_data = 0;

	if (od.main_view_ptrs)
	{
		main_view_ptrs = new vector<SMainViewData *>;

		for (	vector<SMainViewData *>::size_type i=0;
				i<od.main_view_ptrs->size(); i++)
		{
			SMainViewData	*mvd = new SMainViewData(*(*od.main_view_ptrs)[i]);

			main_view_ptrs->push_back(mvd);
		}
	}
	else
		main_view_ptrs = 0;

	if (od.auxiliary_palettes)
	{
		auxiliary_palettes = new map<string, CPaletteData *>;

		map<string, CPaletteData *>::iterator
			ap_iterator = od.auxiliary_palettes->begin();

		while (ap_iterator != od.auxiliary_palettes->end())
		{
			(*auxiliary_palettes)[(*ap_iterator).first] =
				new CPaletteData(*(*ap_iterator).second);

			ap_iterator++;
		}
	}
	else
		auxiliary_palettes = 0;

	if (od.auxiliary_colors)
	{
		auxiliary_colors = new map<string, CPaletteData::SColorSpecifier *>;

		map<string, CPaletteData::SColorSpecifier *>::iterator
			scs_iterator = od.auxiliary_colors->begin();

		while (scs_iterator != od.auxiliary_colors->end())
		{
			(*auxiliary_colors)[(*scs_iterator).first] =
				(*scs_iterator).second->clone();

			scs_iterator++;
		}
	}
	else
		auxiliary_colors = 0;

	if (od.auxiliary_brushes)
	{
		auxiliary_brushes = new map<string, CPaletteData::SBrushSpecifier *>;

		map<string, CPaletteData::SBrushSpecifier *>::iterator
			ab_iterator = od.auxiliary_brushes->begin();

		while (ab_iterator != od.auxiliary_brushes->end())
		{
			(*auxiliary_brushes)[(*ab_iterator).first] =
				(*ab_iterator).second->clone();

			ab_iterator++;
		}
	}
	else
		auxiliary_brushes = 0;

	if (od.auxiliary_xpm_images)
	{
		auxiliary_xpm_images = new map<string, CXpmImage *>;

		map<string, CXpmImage *>::iterator
			ax_iterator = od.auxiliary_xpm_images->begin();

		while (ax_iterator != od.auxiliary_xpm_images->end())
		{
			(*auxiliary_xpm_images)[(*ax_iterator).first] =
				new CXpmImage(*(*ax_iterator).second);

			ax_iterator++;
		}
	}
	else
		auxiliary_xpm_images = 0;

	if (od.auxiliary_fonts)
	{
		auxiliary_fonts = new map<string, string>;

		map<string, string>::iterator
			af_iterator = od.auxiliary_fonts->begin();

		while (af_iterator != od.auxiliary_fonts->end())
		{
			(*auxiliary_fonts)[(*af_iterator).first] = (*af_iterator).second;

			af_iterator++;
		}
	}
	else
		auxiliary_fonts = 0;

	if (od.auxiliary_strings)
	{
		auxiliary_strings = new map<string, string>;

		map<string, string>::iterator
			as_iterator = od.auxiliary_strings->begin();

		while (as_iterator != od.auxiliary_strings->end())
		{
			(*auxiliary_strings)[(*as_iterator).first] = (*as_iterator).second;

			as_iterator++;
		}
	}
	else
		auxiliary_strings = 0;
}

ostream		&COptionData::SSplitterData::OutputData(ostream &os) const
{
	os << "SPLITTER" << endl;

	unsigned int	i;

	os << "INDEX";

	if (index && index->size())
	{
		os << "  ";

		for(i=0; i<index->size()-1; i++)
			os << (*index)[i] << ", ";

		os << (*index)[index->size()-1];
	}

	os << endl;

	os << "ORIENTATION  ";
	if (orientation == Qt::Horizontal)
		os << "Horizontal" << endl;
	else if (orientation == Qt::Vertical)
		os << "Vertical" << endl;

	os << "PANE_SIZE";

	if (pane_size && pane_size->size())
	{
		os << "  ";

		for(i=0; i<pane_size->size()-1; i++)
			os << (*pane_size)[i] << ", ";

		os << (*pane_size)[pane_size->size()-1];
	}

	os << endl;

	if (font_string.length())
	{
		os << "FONT" << endl;
		OutputStringData(os, font_string);
		os << endl;
	}

	if (palette_data)
		os << *palette_data;

	return os;
}

ostream		&COptionData::SSubViewData::OutputData(ostream &os) const
{
	os << "SUBVIEW  ";
	OutputStringData(os, subview_name);
	os << endl;

	unsigned int	i;

	os << "INDEX";

	if (index && index->size())
	{
		os << "  ";

		for(i=0; i<index->size()-1; i++)
			os << (*index)[i] << ", ";

		os << (*index)[index->size()-1];
	}

	os << endl;

	if (font_string.length())
	{
		os << "FONT" << endl;
		OutputStringData(os, font_string);
		os << endl;
	}

	if (palette_data)
		os << *palette_data;

	if (interface_specifiers)
		for (i=0; i<interface_specifiers->size(); i++)
			os << *(*interface_specifiers)[i];

	return os;
}

ostream &operator<<(ostream &os,
                    const COptionData::SSubViewData::SInterfaceSpecifier &d)
{
	os << "INTERFACE  ";

	if (!d.main_view_name.length())
	{
		os << "::  ";
		OutputStringData(os, d.subview_name);
		os << endl;
	}
	else
	{
		OutputStringData(os, d.main_view_name);
		os << "::  ";

		if (d.index && d.index->size())
		{
			unsigned int	i;

			os << "INDEX  ";
			for(i=0; i<d.index->size()-1; i++)
				os << (*d.index)[i] << ", ";

			os << (*d.index)[d.index->size()-1];
			os << endl;
		}
		else
		{
			OutputStringData(os, d.subview_name);
			os << endl;
		}
	}

	return os;
}

ostream &operator<<(ostream &os, const COptionData::SViewNodeData &d)
{
	return d.OutputData(os);
}

ostream &operator<<(ostream &os, const COptionData::SMainViewData &d)
{
	os << endl << "MAIN_VIEW  ";
	OutputStringData(os, d.main_view_name);

	if (d.default_display_flag)
		os << "  DEFAULT_DISPLAY";

	os << endl;

	if (d.position != QPoint(-1, -1))
		os	<< "POSITION  " << setw(10) << d.position.x()
			<< "  " << setw(10) << d.position.y() << endl;

	if (d.size != QSize(-1, -1))
		os	<< "SIZE      " << setw(10) << d.size.width()
			<< "  " << setw(10) << d.size.height() << endl;

	int		i;

	if (d.view_node_data_ptrs)
		for (i=0; i<d.view_node_data_ptrs->size(); i++)
			os << *(*d.view_node_data_ptrs)[i];

	return os;
}

ostream &operator<<(ostream &os, const COptionData::SApplicationData &d)
{
	os << "APPLICATION" << endl;

	if (d.position != QPoint(-1, -1))
		os	<< "POSITION  " << setw(10) << d.position.x()
			<< "  " << setw(10) << d.position.y() << endl;

	if (d.size != QSize(-1, -1))
		os	<< "SIZE      " << setw(10) << d.size.width()
			<< "  " << setw(10) << d.size.height() << endl;

	if (d.font_string.length())
	{
		os << "FONT" << endl;
		OutputStringData(os, d.font_string);
		os << endl;
	}

	if (d.palette_data)
		os << *d.palette_data;

	return os;
}

ostream &operator<<(ostream &os, const COptionData::SHelpData &d)
{
	os << "HELP" << endl;

	if (d.position != QPoint(-1, -1))
		os	<< "POSITION  " << setw(10) << d.position.x()
			<< "  " << setw(10) << d.position.y() << endl;

	if (d.size != QSize(-1, -1))
		os	<< "SIZE      " << setw(10) << d.size.width()
			<< "  " << setw(10) << d.size.height() << endl;

	if (d.font_string.length())
	{
		os << "FONT" << endl;
		OutputStringData(os, d.font_string);
		os << endl;
	}

	if (d.palette_data)
		os << *d.palette_data;

	return os;
}

ostream &operator<<(ostream &os, const COptionData &d)
{
	if (d.application_data)
		os << *d.application_data;

	if (d.help_data)
		os << *d.help_data;

	unsigned int	i;

	if (d.main_view_ptrs)
		for (i=0; i<d.main_view_ptrs->size(); i++)
			os << *(*d.main_view_ptrs)[i];

	if (d.auxiliary_palettes)
		for_each(	d.auxiliary_palettes->begin(),
					d.auxiliary_palettes->end(),
					output_auxiliary_palettes(os));

	if (d.auxiliary_colors)
		for_each(	d.auxiliary_colors->begin(),
					d.auxiliary_colors->end(),
					output_auxiliary_colors(os));

	if (d.auxiliary_brushes)
		for_each(	d.auxiliary_brushes->begin(),
					d.auxiliary_brushes->end(),
					output_auxiliary_brushes(os));

	if (d.auxiliary_xpm_images)
		for_each(	d.auxiliary_xpm_images->begin(),
					d.auxiliary_xpm_images->end(),
					output_auxiliary_xpm_images(os));

	if (d.auxiliary_fonts)
		for_each(	d.auxiliary_fonts->begin(),
					d.auxiliary_fonts->end(),
					output_auxiliary_fonts(os));

	if (d.auxiliary_strings)
		for_each(	d.auxiliary_strings->begin(),
					d.auxiliary_strings->end(),
					output_auxiliary_strings(os));

	return os;
}

void	COptionData::OrderViewNodes()
{
	if (main_view_ptrs)
	{
		vector<SMainViewData *>::size_type		i;

		for (i=0; i<main_view_ptrs->size(); i++)
		{
			vector<SViewNodeData *>     *vnd =
				(*main_view_ptrs)[i]->GetViewNodeDataPtrs();

			if (vnd)
				sort(vnd->begin(), vnd->end(), compare_view_nodes());
		}
	}
}

void	COptionData::AdjustMainViewNames(bool omit_last_flag)
{
	if (!main_view_ptrs || !main_view_ptrs->size())
		return;

	multimap<string, int>	name_to_index_multimap;

	vector<SMainViewData *>::size_type		i;

	for (i=0; i<main_view_ptrs->size(); i++)
	{
		name_to_index_multimap.insert(
			pair<string, int>((*main_view_ptrs)[i]->GetMainViewName(), i));
	}

	typedef multimap<string, int>::iterator		Msi_iterator;

	Msi_iterator	m_iterator;
	string			name_key("");

	m_iterator = name_to_index_multimap.lower_bound(name_key);

	while (m_iterator != name_to_index_multimap.end())
	{
		name_key = (*m_iterator).first;
		
		if (name_to_index_multimap.count(name_key) > 1)
		{
			pair<Msi_iterator, Msi_iterator>	range =
				name_to_index_multimap.equal_range(name_key);

			Msi_iterator	r_iterator		= range.first;
			int				suffix_count	= 0;

			if (omit_last_flag)
				range.second--;

			while (r_iterator != range.second)
			{
				string		adjusted_name;

				while (1)
				{
					ostrstream		name;

					suffix_count++;

					name << name_key << ":" << suffix_count << '\0';

					if (name_to_index_multimap.find(name.str()) ==
						name_to_index_multimap.end())
					{
						adjusted_name = name.str();
						break;
					}
				}

				(*main_view_ptrs)[(*r_iterator).second]->SetMainViewName(
																adjusted_name);

				r_iterator++;
			}
		}

		m_iterator = name_to_index_multimap.upper_bound(name_key);
	}
}

CParserDriver::StatusCode
	COptionData::InputOptionData(istream &is, COptionData **ptr_ptr_od)
{
	COptionLexicalDriver	*lexical	= new COptionLexicalDriver();
	COptionParserDriver		*parser		= new COptionParserDriver();

	lexical->SetInputIoStream(&is);

	lexical->Initialize();
	parser->InitializeSyntaxMode();
	parser->SetLexical(lexical);

	CParserDriver::StatusCode	sc = parser->ParseStream();

	if (sc == CParserDriver::EndOfFile)
	{
		CStackElement	*se = parser->CompiledObject();

		*ptr_ptr_od = reinterpret_cast<COptionData *>(se->data);

		(*ptr_ptr_od)->OrderViewNodes();
		(*ptr_ptr_od)->AdjustMainViewNames();
	}
	else
		*ptr_ptr_od = 0;

	delete parser;
	delete lexical;

	return sc;
}

CParserDriver::StatusCode
	COptionData::InputOptionData(	const unsigned char *i_buffer,
									int i_size, COptionData **ptr_ptr_od)
{
	COptionLexicalDriver	*lexical	= new COptionLexicalDriver();
	COptionParserDriver		*parser		= new COptionParserDriver();

	lexical->SetInputBuffer(i_buffer, i_size);

	lexical->Initialize();
	parser->InitializeSyntaxMode();
	parser->SetLexical(lexical);

	CParserDriver::StatusCode	sc = parser->ParseStream();

	if (sc == CParserDriver::EndOfFile)
	{
		CStackElement	*se = parser->CompiledObject();

		*ptr_ptr_od = reinterpret_cast<COptionData *>(se->data);

		(*ptr_ptr_od)->OrderViewNodes();
		(*ptr_ptr_od)->AdjustMainViewNames();
	}
	else
		*ptr_ptr_od = 0;

	delete parser;
	delete lexical;

	return sc;
}

CParserDriver::StatusCode
	COptionData::InputOptionDataFromClipboard(COptionData **ptr_ptr_od)
{
	QClipboard  *cb = QApplication::clipboard();

	string      paste_data((const char *)(cb->text().local8Bit()));

	istrstream  od_text(paste_data.c_str());

	return InputOptionData(od_text, ptr_ptr_od);
}

void    COptionData::CopyOptionDataToClipboard()
{
	ostrstream      copy_data;

	copy_data << *this << '\0';

	QClipboard  *cb = QApplication::clipboard();

	cb->setText(copy_data.str());
}

void    COptionData::GetSubviewCountMap(map<string, int> &cm)
{
	cm.clear();

	vector<SMainViewData *>::size_type		i;
	vector<SViewNodeData *>::size_type		j;
	map<string, int>::iterator				m_iterator;

	if (!main_view_ptrs)
		return;

	for (i=0; i<main_view_ptrs->size(); i++)
	{
		SMainViewData				*mvd = (*main_view_ptrs)[i];
		vector<SViewNodeData *>		*vnd = mvd->GetViewNodeDataPtrs();

		if (!vnd)
			continue;

		for (j=0; j<vnd->size(); j++)
		{
			SViewNodeData	*nd = (*vnd)[j];

			SViewNodeData::ViewNodeType		nt = nd->GetNodeType();

			if (nt == SViewNodeData::SubViewNode)
			{
				SSubViewData	*svd = dynamic_cast<SSubViewData *>(nd);
				string			&svn = svd->GetSubViewName();

				cm[svn]++;
			}
		}
	}
}
