/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "parserDialog.h"

CCalculateDialog			*ParserDialog::calculate_dialog;
CErrorEntriesDialog			*ParserDialog::error_entries_dialog;
CUnitRulesDialog			*ParserDialog::unit_rules_dialog;
CGenerateCodeDialog			*ParserDialog::generate_code_dialog;
CSimulatorErrorDialog		*ParserDialog::simulator_error_dialog;
CSimulatorControlDialog		*ParserDialog::simulator_control_dialog;

void	ParserDialog::CreateParserDialogs(QWidget *parent)
{
	calculate_dialog			= new CCalculateDialog(parent);
	error_entries_dialog		= new CErrorEntriesDialog(parent);
	unit_rules_dialog			= new CUnitRulesDialog(parent);
	generate_code_dialog		= new CGenerateCodeDialog(parent);
	simulator_error_dialog		= new CSimulatorErrorDialog(parent);
	simulator_control_dialog	= new CSimulatorControlDialog(parent);
}

void	ParserDialog::DestroyParserDialogs()
{
	delete calculate_dialog;
	delete error_entries_dialog;
	delete unit_rules_dialog;
	delete generate_code_dialog;
	delete simulator_error_dialog;
	delete simulator_control_dialog;
}

void	ParserDialog::SetCalculatePaletteAndFont(COptionData *d)
{
	static COptionData	*current_option_data = 0;

	if (current_option_data == d)
		return;
	else
		current_option_data = d;

	QFont			dialog_font;
	QPalette		dialog_palette;
	string			*font_string;
	CPaletteData	*palette_data;
	string			id("CALCULATE_DIALOG");

	if (font_string = d->GetAuxiliaryFont(id))
	{
		FontSpecifier::SetFontSpecifier(dialog_font, *font_string);
		calculate_dialog->setFont(dialog_font);
	}

	if (palette_data = d->GetAuxiliaryPalette(id))
	{
		palette_data->SetupPalette(dialog_palette);
		calculate_dialog->setPalette(dialog_palette);
	}
}

void	ParserDialog::SetErrorEntriesPaletteAndFont(COptionData *d)
{
	static COptionData	*current_option_data = 0;

	if (current_option_data == d)
		return;
	else
		current_option_data = d;

	QFont			dialog_font;
	QPalette		dialog_palette;
	string			*font_string;
	CPaletteData	*palette_data;
	string			id("ERROR_ENTRIES_DIALOG");

	if (font_string = d->GetAuxiliaryFont(id))
	{
		FontSpecifier::SetFontSpecifier(dialog_font, *font_string);
		error_entries_dialog->setFont(dialog_font);
	}

	if (palette_data = d->GetAuxiliaryPalette(id))
	{
		palette_data->SetupPalette(dialog_palette);
		error_entries_dialog->setPalette(dialog_palette);
	}
}

void	ParserDialog::SetUnitRulesPaletteAndFont(COptionData *d)
{
	static COptionData	*current_option_data = 0;

	if (current_option_data == d)
		return;
	else
		current_option_data = d;

	QFont			dialog_font;
	QPalette		dialog_palette;
	string			*font_string;
	CPaletteData	*palette_data;
	string			id("UNIT_RULES_DIALOG");

	if (font_string = d->GetAuxiliaryFont(id))
	{
		FontSpecifier::SetFontSpecifier(dialog_font, *font_string);
		unit_rules_dialog->setFont(dialog_font);
	}

	if (palette_data = d->GetAuxiliaryPalette(id))
	{
		palette_data->SetupPalette(dialog_palette);
		unit_rules_dialog->setPalette(dialog_palette);
	}
}

void	ParserDialog::SetGenerateCodePaletteAndFont(COptionData *d)
{
	static COptionData	*current_option_data = 0;

	if (current_option_data == d)
		return;
	else
		current_option_data = d;

	QFont			dialog_font;
	QPalette		dialog_palette;
	string			*font_string;
	CPaletteData	*palette_data;
	string			id("GENERATE_CODE_DIALOG");

	if (font_string = d->GetAuxiliaryFont(id))
	{
		FontSpecifier::SetFontSpecifier(dialog_font, *font_string);
		generate_code_dialog->setFont(dialog_font);
	}

	if (palette_data = d->GetAuxiliaryPalette(id))
	{
		palette_data->SetupPalette(dialog_palette);
		generate_code_dialog->setPalette(dialog_palette);
	}
}

void	ParserDialog::SetSimulatorErrorPaletteAndFont(COptionData *d)
{
	static COptionData	*current_option_data = 0;

	if (current_option_data == d)
		return;
	else
		current_option_data = d;

	QFont			dialog_font;
	QPalette		dialog_palette;
	string			*font_string;
	CPaletteData	*palette_data;
	string			id("SIMULATOR_ERROR_DIALOG");

	if (font_string = d->GetAuxiliaryFont(id))
	{
		FontSpecifier::SetFontSpecifier(dialog_font, *font_string);
		simulator_error_dialog->setFont(dialog_font);
	}

	if (palette_data = d->GetAuxiliaryPalette(id))
	{
		palette_data->SetupPalette(dialog_palette);
		simulator_error_dialog->setPalette(dialog_palette);
	}
}

void	ParserDialog::SetSimulatorControlPaletteAndFont(COptionData *d)
{
	static COptionData	*current_option_data = 0;

	if (current_option_data == d)
		return;
	else
		current_option_data = d;

	QFont			dialog_font;
	QPalette		dialog_palette;
	string			*font_string;
	CPaletteData	*palette_data;
	string			id("SIMULATOR_CONTROL_DIALOG");

	if (font_string = d->GetAuxiliaryFont(id))
	{
		FontSpecifier::SetFontSpecifier(dialog_font, *font_string);
		simulator_control_dialog->setFont(dialog_font);
	}

	if (palette_data = d->GetAuxiliaryPalette(id))
	{
		palette_data->SetupPalette(dialog_palette);
		simulator_control_dialog->setPalette(dialog_palette);
	}
}

CCalculateDialog::CCalculateDialog(QWidget *parent)
	: CDialogFc(parent, 5, 5, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);
	grid_layout->setRowStretch(3, 1);
	grid_layout->setRowStretch(4, 0);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 0);
	grid_layout->setColStretch(2, 0);
	grid_layout->setColStretch(3, 0);
	grid_layout->setColStretch(4, 1);

	QString     size_string("#####");

	spacer1					= new CSpacerFc(this, true);
	lexical_check_box		= new CCheckBoxFc("Calculate Lexical", this);
	calculate_button		= new CPushButtonFc("Calculate", this, size_string);
	close_button			= new CPushButtonFc("Close", this, size_string);
	range_button_group		= new CHButtonGroupFc("Input Range", this);
	r128_radio_button		= new CRadioButtonFc("128", range_button_group);
	r256_radio_button		= new CRadioButtonFc("256", range_button_group);
	lexical_button_group	= new CHButtonGroupFc("Lexical Type", this);
	singlechar_radio_button	= new CRadioButtonFc(	"SingleChar",
													lexical_button_group);
	regexp_radio_button		= new CRadioButtonFc(	"Regexp",
													lexical_button_group);
	spacer2					= new CSpacerFc(this, true);
	parser_check_box		= new CCheckBoxFc("Calculate Parser", this);
	parser_button_group	= new CHButtonGroupFc("Parser Type", this);
	slr1_radio_button	= new CRadioButtonFc("SLR(1)", parser_button_group);
	lalr1_radio_button	= new CRadioButtonFc("LALR(1)", parser_button_group);
	lr1_radio_button	= new CRadioButtonFc("LR(1)", parser_button_group);
	status_multi_line_edit	= new CMultiLineEditFc(this,
								"Parser Calculation #################", 10);
	spacer3					= new CSpacerFc(this);

	grid_layout->addWidget(spacer1,							0, 0);
	grid_layout->addWidget(lexical_check_box,				0, 1);
	grid_layout->addWidget(calculate_button,				0, 2);
	grid_layout->addWidget(close_button,					0, 3);
	grid_layout->addMultiCellWidget(range_button_group,		1, 1, 0, 1);
	grid_layout->addMultiCellWidget(lexical_button_group,	1, 1, 2, 3);
	grid_layout->addWidget(spacer2,							2, 0);
	grid_layout->addWidget(parser_check_box,				2, 1);
	grid_layout->addMultiCellWidget(parser_button_group,	2, 2, 2, 3);
	grid_layout->addMultiCellWidget(status_multi_line_edit,	3, 3, 0, 4);
	grid_layout->addMultiCellWidget(spacer3,				4, 4, 0, 4);

	connect(calculate_button, SIGNAL(clicked()), SLOT(Calculate()));
	connect(close_button, SIGNAL(clicked()), SLOT(reject()));
}

void	CCalculateDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();
}

void	CCalculateDialog::SetCompilerInterfacePointer(CCompilerInterface *c)
{
	compiler_interface_pointer = c;

	CCompilerData	*d = c->GetCompilerData();

	lexical_check_box->setChecked(d->GetLexicalAutomatonCalculate());

	switch (d->GetLexicalInputRange())
	{
		case 128:
		range_button_group->setButton(0);
		break;

		case 256:
		range_button_group->setButton(1);
		break;
	}

	switch (d->GetLexicalFiniteStateMachineType())
	{
		case CCompilerData::SingleCharacter:
		lexical_button_group->setButton(0);
		break;

		case CCompilerData::RegularExpression:
		lexical_button_group->setButton(1);
		break;
	}

	parser_check_box->setChecked(d->GetParserAutomatonCalculate());

	switch (d->GetParserType())
	{
		case CCompilerData::SLR_1:
		parser_button_group->setButton(0);
		break;

		case CCompilerData::LALR_1:
		parser_button_group->setButton(1);
		break;

		case CCompilerData::LR_1:
		parser_button_group->setButton(2);
		break;
	}

	status_multi_line_edit->clear();

	string		status_string;

	compiler_interface_pointer->MachineStatusString(status_string);

	status_multi_line_edit->append(status_string.c_str());
	status_multi_line_edit->append("********************************\n\n");

	if (compiler_interface_pointer->GetLexicalCalculated())
	{
		compiler_interface_pointer->LexicalStatusString(status_string);

		status_multi_line_edit->append(status_string.c_str());
		status_multi_line_edit->append("\n");
	}

	if (compiler_interface_pointer->GetParserCalculated())
	{
		compiler_interface_pointer->ParserStatusString(status_string);

		status_multi_line_edit->append(status_string.c_str());
		status_multi_line_edit->append("\n");

		compiler_interface_pointer->ConflictSectionStatusString(status_string);

		status_multi_line_edit->append(status_string.c_str());
	}
}

void	CCalculateDialog::Calculate()
{
	bool									calculate_lexical;
	CCompilerData::FiniteStateMachineType	lexical_type;
	int										lexical_range;
	bool									calculate_parser;
	CCompilerData::ParserType				parser_type;

	calculate_lexical = lexical_check_box->isChecked();

	if (singlechar_radio_button->isChecked())
		lexical_type = CCompilerData::SingleCharacter;
	else if (regexp_radio_button->isChecked())
		lexical_type = CCompilerData::RegularExpression;

	if (r128_radio_button->isChecked())
		lexical_range = 128;
	else if (r256_radio_button->isChecked())
		lexical_range = 256;

	calculate_parser = parser_check_box->isChecked();

	if (slr1_radio_button->isChecked())
		parser_type = CCompilerData::SLR_1;
	else if (lalr1_radio_button->isChecked())
		parser_type = CCompilerData::LALR_1;
	else if (lr1_radio_button->isChecked())
		parser_type = CCompilerData::LR_1;

	compiler_interface_pointer->CalculateMachine(
					calculate_lexical, lexical_type, lexical_range,
					calculate_parser, parser_type);

	compiler_interface_pointer->EnableParserSubmenu();

	status_multi_line_edit->clear();

	string		status_string;

	compiler_interface_pointer->MachineStatusString(status_string);

	status_multi_line_edit->append(status_string.c_str());
	status_multi_line_edit->append("********************************\n\n");

	compiler_interface_pointer->LexicalStatusString(status_string);

	status_multi_line_edit->append(status_string.c_str());
	status_multi_line_edit->append("\n");

	compiler_interface_pointer->ParserStatusString(status_string);

	status_multi_line_edit->append(status_string.c_str());

	if (compiler_interface_pointer->GetParserCalculated())
	{
		status_multi_line_edit->append("\n");

		compiler_interface_pointer->ConflictSectionStatusString(status_string);

		status_multi_line_edit->append(status_string.c_str());
	}
}

CErrorEntriesDialog::CErrorEntriesDialog(QWidget *parent)
	: CDialogFc(parent, 7, 2, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);
	grid_layout->setRowStretch(3, 0);
	grid_layout->setRowStretch(4, 0);
	grid_layout->setRowStretch(5, 1);
	grid_layout->setRowStretch(6, 0);

	grid_layout->setColStretch(0, 1);
	grid_layout->setColStretch(1, 0);

	QString     size_string("#####");

	calculation_button_group =
		new CHButtonGroupFc("Essential Error Entry Calculation", this);
	no_calculation_radio_button =
		new CRadioButtonFc("No Calculation", calculation_button_group);
	approximate_radio_button =
		new CRadioButtonFc("Approximate", calculation_button_group);
	exact_radio_button =
		new CRadioButtonFc("Exact", calculation_button_group);
	numbering_button_group =
		new CHButtonGroupFc("Error Function Numbering", this);
	one_number_radio_button =
		new CRadioButtonFc("One Number", numbering_button_group);
	number_per_state_radio_button =
		new CRadioButtonFc("Number Per State", numbering_button_group);
	number_per_entry_radio_button =
		new CRadioButtonFc("Number Per Entry", numbering_button_group);
	aggregation_button_group =
		new CHButtonGroupFc("Error Function Aggregation", this);
	one_function_radio_button =
		new CRadioButtonFc("One Function", aggregation_button_group);
	function_per_state_radio_button =
		new CRadioButtonFc("Function Per State", aggregation_button_group);
	function_per_entry_radio_button =
		new CRadioButtonFc("Function Per Entry", aggregation_button_group);
	status_multi_line_edit	= new CMultiLineEditFc(this,
							"Error Entries Calculation #################", 7);
	calculate_button		= new CPushButtonFc("Calculate", this, size_string);
	close_button			= new CPushButtonFc("Close", this, size_string);
	spacer1					= new CSpacerFc(this, true);

	grid_layout->addMultiCellWidget(calculation_button_group,	0, 0, 0, 1);
	grid_layout->addMultiCellWidget(numbering_button_group,		1, 1, 0, 1);
	grid_layout->addMultiCellWidget(aggregation_button_group,	2, 2, 0, 1);
	grid_layout->addMultiCellWidget(status_multi_line_edit,		3, 5, 0, 0);
	grid_layout->addWidget(calculate_button,					3, 1);
	grid_layout->addWidget(close_button,						4, 1);
	grid_layout->addMultiCellWidget(spacer1,					6, 6, 0, 1);

	connect(numbering_button_group,
			SIGNAL(clicked(int)),
			SLOT(NumberingSelected(int)));

	connect(calculate_button, SIGNAL(clicked()), SLOT(Calculate()));
	connect(close_button, SIGNAL(clicked()), SLOT(reject()));
}

void	CErrorEntriesDialog::NumberingSelected(int id)
{
	switch (id)
	{
		case 0:
		aggregation_button_group->setButton(0);
		function_per_state_radio_button->setEnabled(false);
		function_per_entry_radio_button->setEnabled(false);
		break;

		case 1:
		if (function_per_entry_radio_button->isChecked())
		{
			function_per_state_radio_button->setEnabled(true);
			function_per_entry_radio_button->setEnabled(false);
			aggregation_button_group->setButton(1);
		}
		else
		{
			function_per_state_radio_button->setEnabled(true);
			function_per_entry_radio_button->setEnabled(false);
		}
		break;

		case 2:
		function_per_state_radio_button->setEnabled(true);
		function_per_entry_radio_button->setEnabled(true);
		break;
	}
}

void	CErrorEntriesDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();
}

void	CErrorEntriesDialog::SetCompilerInterfacePointer(CCompilerInterface *c)
{
	compiler_interface_pointer = c;

	CCompilerData	*d = c->GetCompilerData();

	switch (d->GetParserErrorCalculation())
	{
		case CCompilerData::NoCalculation:
		calculation_button_group->setButton(0);
		break;

		case CCompilerData::ApproximateEssential:
		calculation_button_group->setButton(1);
		break;

		case CCompilerData::ExactEssential:
		calculation_button_group->setButton(2);
		break;
	}

	switch (d->GetParserErrorFunctionNumbering())
	{
		case CCompilerData::OneNumber:
		numbering_button_group->setButton(0);
		NumberingSelected(0);
		break;

		case CCompilerData::NumberPerState:
		numbering_button_group->setButton(1);
		NumberingSelected(1);
		break;

		case CCompilerData::NumberPerEntry:
		numbering_button_group->setButton(2);
		NumberingSelected(2);
		break;
	}

	switch (d->GetParserErrorFunctionAggregation())
	{
		case CCompilerData::OneFunction:
		aggregation_button_group->setButton(0);
		break;

		case CCompilerData::FunctionPerState:
		if (one_number_radio_button->isChecked())
			aggregation_button_group->setButton(0);
		else
			aggregation_button_group->setButton(1);
		break;

		case CCompilerData::FunctionPerEntry:
		if (one_number_radio_button->isChecked())
			aggregation_button_group->setButton(0);
		else if (number_per_state_radio_button->isChecked())
			aggregation_button_group->setButton(1);
		else
			aggregation_button_group->setButton(2);
		break;
	}

	status_multi_line_edit->clear();

	string		status_string;

	compiler_interface_pointer->ErrorEntriesStatusString(status_string);

	status_multi_line_edit->append(status_string.c_str());

	if (compiler_interface_pointer->GetErrorEntriesNumbered())
	{
		status_multi_line_edit->append("\n");

		compiler_interface_pointer->ErrorCodeSectionStatusString(status_string);

		status_multi_line_edit->append(status_string.c_str());
	}
}

void	CErrorEntriesDialog::Calculate()
{
	CCompilerData::ErrorCalculation				c;
	CCompilerData::ErrorFunctionNumbering		n;
	CCompilerData::ErrorFunctionAggregation		a;

	if (no_calculation_radio_button->isChecked())
		c = CCompilerData::NoCalculation;
	else if (approximate_radio_button->isChecked())
		c = CCompilerData::ApproximateEssential;
	else if (exact_radio_button->isChecked())
		c = CCompilerData::ExactEssential;

	if (one_number_radio_button->isChecked())
		n = CCompilerData::OneNumber;
	else if (number_per_state_radio_button->isChecked())
		n = CCompilerData::NumberPerState;
	else if (number_per_entry_radio_button->isChecked())
		n = CCompilerData::NumberPerEntry;

	if (one_function_radio_button->isChecked())
		a = CCompilerData::OneFunction;
	else if (function_per_state_radio_button->isChecked())
		a = CCompilerData::FunctionPerState;
	else if (function_per_entry_radio_button->isChecked())
		a = CCompilerData::FunctionPerEntry;

	compiler_interface_pointer->CalculateErrorEntries(c, n, a);

	compiler_interface_pointer->EnableParserSubmenu();

	status_multi_line_edit->clear();

	string		status_string;

	compiler_interface_pointer->ErrorEntriesStatusString(status_string);

	status_multi_line_edit->append(status_string.c_str());
	status_multi_line_edit->append("\n");

	compiler_interface_pointer->ErrorCodeSectionStatusString(status_string);

	status_multi_line_edit->append(status_string.c_str());
}

CUnitRulesDialog::CUnitRulesDialog(QWidget *parent)
	: CDialogFc(parent, 9, 2, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);
	grid_layout->setRowStretch(3, 0);
	grid_layout->setRowStretch(4, 1);
	grid_layout->setRowStretch(5, 0);
	grid_layout->setRowStretch(6, 0);
	grid_layout->setRowStretch(7, 0);
	grid_layout->setRowStretch(8, 0);

	grid_layout->setColStretch(0, 1);
	grid_layout->setColStretch(1, 0);

	QString     size_string("#########");

	unit_rule_listbox		= new CListBoxFc(this, "unit rule listbox", 8);
	move_up_button			= new CPushButtonFc("Up", this, size_string);
	move_down_button		= new CPushButtonFc("Down", this, size_string);
	move_to_first_button	= new CPushButtonFc("First", this, size_string);
	move_to_last_button		= new CPushButtonFc("Last", this, size_string);
	status_multi_line_edit	= new CMultiLineEditFc(this,
							"Unit Rules Calculation #################", 2);
	calculate_button		= new CPushButtonFc("Calculate", this, size_string);
	close_button			= new CPushButtonFc("Close", this, size_string);
	spacer1					= new CSpacerFc(this, true);

	grid_layout->addMultiCellWidget(unit_rule_listbox,			0, 4, 0, 0);
	grid_layout->addWidget(move_up_button,						0, 1);
	grid_layout->addWidget(move_down_button,					1, 1);
	grid_layout->addWidget(move_to_first_button,				2, 1);
	grid_layout->addWidget(move_to_last_button,					3, 1);
	grid_layout->addMultiCellWidget(status_multi_line_edit,		5, 7, 0, 0);
	grid_layout->addWidget(calculate_button,					5, 1);
	grid_layout->addWidget(close_button,						6, 1);
	grid_layout->addMultiCellWidget(spacer1,					8, 8, 0, 1);

	connect(calculate_button, SIGNAL(clicked()), SLOT(Calculate()));
	connect(close_button, SIGNAL(clicked()), SLOT(reject()));

	connect(move_up_button,
			SIGNAL(clicked()),
			SLOT(MoveUp()));

	connect(move_down_button,
			SIGNAL(clicked()),
			SLOT(MoveDown()));

	connect(move_to_first_button,
			SIGNAL(clicked()),
			SLOT(MoveToFirst()));

	connect(move_to_last_button,
			SIGNAL(clicked()),
			SLOT(MoveToLast()));
}

void    CUnitRulesDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();
}

void	CUnitRulesDialog::SetCompilerInterfacePointer(CCompilerInterface *c)
{
	compiler_interface_pointer = c;

	CCompilerData	*d = c->GetCompilerData();

	d->CalculateOutputWidths();

	if (!d->ValidateUnitRuleOrderList())
		d->InitializeUnitRuleOrderList();

	const list<int>		&ordering_list = d->GetUnitRuleOrderList();

	ordering_array.clear();

	int		rule_count = d->GetRuleList().size();

	ostrstream	os_width;

	os_width << rule_count - 1 << '\0';

	int		rc_width = string(os_width.str()).length();

	unit_rule_listbox->clear();

	list<int>::const_iterator		ol_iterator = ordering_list.begin();

	while (ol_iterator != ordering_list.end())
	{
		int		rule_index = *ol_iterator++;

		ordering_array.push_back(rule_index);

		CRule	*rule = d->FindRuleIndex(rule_index);
		string	left;
		string	right;

		string	pad1(d->GetRuleLeftNameWidth() -
					rule->GetLeft()->GetName().length() + 1, ' ');

		ostrstream	os;

		os << setw(rc_width) << rule_index << "  ";

		os << rule->GetLeftString(left) << pad1 << "-> ";
		os << rule->GetRightString(right) << '\0';

		unit_rule_listbox->insertItem(os.str());
	}

	unit_rule_listbox->setCurrentItem(0);

	status_multi_line_edit->clear();

	string		status_string;

	compiler_interface_pointer->UnitRuleEliminationStatusString(status_string);

	status_multi_line_edit->append(status_string.c_str());
}

void	CUnitRulesDialog::Calculate()
{
	CCompilerData	*d = compiler_interface_pointer->GetCompilerData();

	d->ClearUnitRuleOrderList();

	vector<int>::size_type		i;

	for (i=0; i<ordering_array.size(); i++)
		d->AddUnitRuleOrderIndex(ordering_array[i]);

	compiler_interface_pointer->CalculateUnitRuleElimination(true);

	status_multi_line_edit->clear();

	string		status_string;

	compiler_interface_pointer->UnitRuleEliminationStatusString(status_string);

	status_multi_line_edit->append(status_string.c_str());
}

void    CUnitRulesDialog::MoveUp()
{
	int		current_item = unit_rule_listbox->currentItem();

	if (current_item <= 0)
		return;

	QString		s1 = unit_rule_listbox->text(current_item - 1);
	QString		s2 = unit_rule_listbox->text(current_item);

	unit_rule_listbox->changeItem(s2, current_item - 1);
	unit_rule_listbox->changeItem(s1, current_item);

	swap(ordering_array[current_item - 1], ordering_array[current_item]);

	unit_rule_listbox->setCurrentItem(current_item - 1);
}

void    CUnitRulesDialog::MoveDown()
{
	int		current_item = unit_rule_listbox->currentItem();

	if (current_item == -1 || current_item == unit_rule_listbox->count() - 1)
		return;

	QString		s1 = unit_rule_listbox->text(current_item);
	QString		s2 = unit_rule_listbox->text(current_item + 1);

	unit_rule_listbox->changeItem(s2, current_item);
	unit_rule_listbox->changeItem(s1, current_item + 1);

	swap(ordering_array[current_item], ordering_array[current_item + 1]);

	unit_rule_listbox->setCurrentItem(current_item + 1);
}

void    CUnitRulesDialog::MoveToFirst()
{
	int		current_item = unit_rule_listbox->currentItem();

	if (current_item <= 0)
		return;

	QString		s = unit_rule_listbox->text(current_item);

	unit_rule_listbox->removeItem(current_item);
	unit_rule_listbox->insertItem(s, 0);

	rotate(	ordering_array.begin(),
			ordering_array.begin() + current_item,
			ordering_array.begin() + current_item + 1);

	unit_rule_listbox->setCurrentItem(0);
}

void    CUnitRulesDialog::MoveToLast()
{
	int		current_item	= unit_rule_listbox->currentItem();
	int		last_index		= unit_rule_listbox->count() - 1;

	if (current_item == -1 || current_item == last_index)
		return;

	QString		s = unit_rule_listbox->text(current_item);

	unit_rule_listbox->removeItem(current_item);
	unit_rule_listbox->insertItem(s, last_index);

	rotate(	ordering_array.begin() + current_item,
			ordering_array.begin() + current_item + 1,
			ordering_array.end());

	unit_rule_listbox->setCurrentItem(last_index);
}

CGenerateCodeDialog::CGenerateCodeDialog(QWidget *parent)
	: CDialogFc(parent, 6, 2, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1 = new QGridLayout(this, 1, 4, 0, s, "subgrid1");
	subgrid_layout2 = new QGridLayout(this, 4, 2, 0, s, "subgrid2");
	subgrid_layout3 = new QGridLayout(this, 2, 6, 0, s, "subgrid3");

	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);
	grid_layout->setRowStretch(3, 0);
	grid_layout->setRowStretch(4, 1);
	grid_layout->setRowStretch(5, 0);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 1);

	subgrid_layout1->setRowStretch(0, 1);

	subgrid_layout1->setColStretch(0, 0);
	subgrid_layout1->setColStretch(1, 1);
	subgrid_layout1->setColStretch(2, 0);
	subgrid_layout1->setColStretch(3, 1);

	subgrid_layout2->setRowStretch(0, 1);
	subgrid_layout2->setRowStretch(1, 1);
	subgrid_layout2->setRowStretch(2, 1);
	subgrid_layout2->setRowStretch(3, 1);

	subgrid_layout2->setColStretch(0, 0);
	subgrid_layout2->setColStretch(1, 1);

	subgrid_layout3->setRowStretch(0, 1);
	subgrid_layout3->setRowStretch(1, 1);

	subgrid_layout3->setColStretch(0, 0);
	subgrid_layout3->setColStretch(1, 0);
	subgrid_layout3->setColStretch(2, 0);
	subgrid_layout3->setColStretch(3, 0);
	subgrid_layout3->setColStretch(4, 1);
	subgrid_layout3->setColStretch(5, 0);

	QString     size_string("#########");

	spacer1						= new CSpacerFc(this, true);
	lexical_check_box			= new CCheckBoxFc("Generate Lexical Code",
									this);
	spacer2						= new CSpacerFc(this, true);
	parser_check_box			= new CCheckBoxFc("Generate Parser Code", this);
	table_type_button_group		=
		new CHButtonGroupFc("Parser Table Unit Rule Eliminations", this);
	without_unit_radio_button 	=
		new CRadioButtonFc("Without Eliminations", table_type_button_group);
	with_unit_radio_button		=
		new CRadioButtonFc("With Eliminations", table_type_button_group);
	directory_label				= new CLabelFc("Code Directory", this);
	directory_line_edit			= new CLineEditFc(this, "directory");
	lexical_prefix_label		= new CLabelFc("Lexical Prefix Name", this);
	lexical_prefix_line_edit	= new CLineEditFc(this, "lexical prefix");
	parser_prefix_label			= new CLabelFc("Parser Prefix Name", this);
	parser_prefix_line_edit		= new CLineEditFc(this, "parser prefix");
	base_prefix_label			= new CLabelFc("Base Class Prefix Name", this);
	base_prefix_line_edit		= new CLineEditFc(this, "base prefix");
	spacer7						= new CSpacerFc(this, true);
	generate_base_classes_check_box =
		new CCheckBoxFc("Generate Base And Stack Classes", this);
	spacer8						= new CSpacerFc(this, false);
	spacer3						= new CSpacerFc(this, true);
	tty_check_box				= new CCheckBoxFc("TTY Code", this);
	spacer4						= new CSpacerFc(this, true);
	regexp_check_box			= new CCheckBoxFc("Regexp Table", this);
	spacer5						= new CSpacerFc(this, false);
	generate_code_button		= new CPushButtonFc("Generate",
									this, size_string);
	close_button				= new CPushButtonFc("Close", this, size_string);
	status_multi_line_edit		= new CMultiLineEditFc(this,
								"Generate Code #################", 9);
	spacer6						= new CSpacerFc(this, true);

	directory_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
	lexical_prefix_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
	parser_prefix_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);
	base_prefix_label->setAlignment(Qt::AlignRight | Qt::AlignVCenter);

	string		directory;

	FilePath::GetCurrentWorkingDirectory(directory);

	directory_line_edit->setText(directory.c_str());

	subgrid_layout1->addWidget(spacer1,						0, 0);
	subgrid_layout1->addWidget(lexical_check_box,			0, 1);
	subgrid_layout1->addWidget(spacer2,						0, 2);
	subgrid_layout1->addWidget(parser_check_box,			0, 3);

	subgrid_layout2->addWidget(directory_label,				0, 0);
	subgrid_layout2->addWidget(directory_line_edit,			0, 1);
	subgrid_layout2->addWidget(lexical_prefix_label,		1, 0);
	subgrid_layout2->addWidget(lexical_prefix_line_edit,	1, 1);
	subgrid_layout2->addWidget(parser_prefix_label,			2, 0);
	subgrid_layout2->addWidget(parser_prefix_line_edit,		2, 1);
	subgrid_layout2->addWidget(base_prefix_label,			3, 0);
	subgrid_layout2->addWidget(base_prefix_line_edit,		3, 1);

	subgrid_layout3->addWidget(spacer7,							0, 0);
	subgrid_layout3->addMultiCellWidget(
							generate_base_classes_check_box,	0, 0, 1, 3);
	subgrid_layout3->addWidget(spacer8,							0, 4);
	subgrid_layout3->addWidget(generate_code_button,			0, 5);

	subgrid_layout3->addWidget(spacer3,						1, 0);
	subgrid_layout3->addWidget(tty_check_box,				1, 1);
	subgrid_layout3->addWidget(spacer4,						1, 2);
	subgrid_layout3->addWidget(regexp_check_box,			1, 3);
	subgrid_layout3->addWidget(spacer5,						1, 4);
	subgrid_layout3->addWidget(close_button,				1, 5);

	grid_layout->addLayout(subgrid_layout1,					0, 0);
	grid_layout->addWidget(table_type_button_group,			1, 0);
	grid_layout->addMultiCellLayout(subgrid_layout2,		2, 2, 0, 1);
	grid_layout->addMultiCellLayout(subgrid_layout3,		3, 3, 0, 1);
	grid_layout->addMultiCellWidget(status_multi_line_edit,	4, 4, 0, 1);
	grid_layout->addMultiCellWidget(spacer6,				5, 5, 0, 1);

	connect(generate_code_button,
			SIGNAL(clicked()),
			SLOT(GenerateCode()));

	connect(close_button, SIGNAL(clicked()), SLOT(reject()));
}

void	CGenerateCodeDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();
}

void	CGenerateCodeDialog::setFont(const QFont &f)
{
	QFontInfo       fi(f);

	int     pixel_size = fi.pixelSize();

	int     s = grid_layout_spacing_data[pixel_size];

	subgrid_layout1->setMargin(0);
	subgrid_layout1->setSpacing(s);
	subgrid_layout1->invalidate();

	subgrid_layout2->setMargin(0);
	subgrid_layout2->setSpacing(s);
	subgrid_layout2->invalidate();

	subgrid_layout3->setMargin(0);
	subgrid_layout3->setSpacing(s);
	subgrid_layout3->invalidate();

	CDialogFc::setFont(f);
}

void	CGenerateCodeDialog::SetCompilerInterfacePointer(CCompilerInterface *c)
{
	compiler_interface_pointer = c;

	CCompilerData	*d = compiler_interface_pointer->GetCompilerData();

	if (compiler_interface_pointer->GetLexicalCalculated())
	{
		lexical_check_box->setEnabled(true);
		lexical_check_box->setChecked(true);
	}
	else
	{
		lexical_check_box->setEnabled(false);
		lexical_check_box->setChecked(false);
	}

	if (compiler_interface_pointer->GetParserCalculated())
	{
		parser_check_box->setEnabled(true);
		parser_check_box->setChecked(true);
		table_type_button_group->setEnabled(true);
	}
	else
	{
		parser_check_box->setEnabled(false);
		parser_check_box->setChecked(false);
		table_type_button_group->setEnabled(false);
	}

	if (compiler_interface_pointer->GetLexicalCalculated() &&
		compiler_interface_pointer->GetParserCalculated() &&
		d->GetLexicalFiniteStateMachineType() ==
			CCompilerData::RegularExpression)
	{
		tty_check_box->setEnabled(true);

		if (d->GetGenerateTTYCode())
			tty_check_box->setChecked(true);
		else
			tty_check_box->setChecked(false);
	}
	else
	{
		tty_check_box->setEnabled(false);
		tty_check_box->setChecked(false);
	}

	if (compiler_interface_pointer->GetLexicalCalculated() &&
		d->GetLexicalFiniteStateMachineType() ==
			CCompilerData::RegularExpression)
	{
		regexp_check_box->setEnabled(true);

		if (d->GetRegexpMatchTable())
			regexp_check_box->setChecked(true);
		else
			regexp_check_box->setChecked(false);
	}
	else
	{
		regexp_check_box->setEnabled(false);
		regexp_check_box->setChecked(false);
	}

	if (compiler_interface_pointer->GetUnitRuleEliminationCalculated())
	{
		without_unit_radio_button->setEnabled(true);
		with_unit_radio_button->setEnabled(true);
		table_type_button_group->setButton(1);
	}
	else
	{
		if (compiler_interface_pointer->GetParserCalculated())
		{
			without_unit_radio_button->setEnabled(true);
			table_type_button_group->setButton(0);
		}
		else
			without_unit_radio_button->setEnabled(false);

		with_unit_radio_button->setEnabled(false);
	}

	if (d->GetGenerateBaseClasses())
		generate_base_classes_check_box->setChecked(true);
	else
		generate_base_classes_check_box->setChecked(false);

	lexical_prefix_line_edit->setText(d->GetLexicalPrefixName().c_str());

	parser_prefix_line_edit->setText(d->GetParserPrefixName().c_str());

	base_prefix_line_edit->setText(d->GetBasePrefixName().c_str());

	status_multi_line_edit->clear();
}

void	CGenerateCodeDialog::GenerateCode()
{
	bool	generate_lexical	= lexical_check_box->isChecked();
	bool	generate_parser		= parser_check_box->isChecked();
	bool	generate_base		= generate_base_classes_check_box->isChecked();
	bool	generate_tty		= tty_check_box->isChecked();
	bool	regexp_match_table	= regexp_check_box->isChecked();
	bool	unit_rule_elimination_flag
								= with_unit_radio_button->isChecked();

	string		directory((const char *)
								(directory_line_edit->text().local8Bit()));
	string		lexical_prefix((const char *)
								(lexical_prefix_line_edit->text().local8Bit()));
	string		parser_prefix((const char *)
								(parser_prefix_line_edit->text().local8Bit()));
	string		base_prefix((const char *)
								(base_prefix_line_edit->text().local8Bit()));

	compiler_interface_pointer->GenerateCode(	generate_lexical,
												generate_parser,
												generate_base,
												unit_rule_elimination_flag,
												directory,
												lexical_prefix,
												parser_prefix,
												base_prefix,
												generate_tty,
												regexp_match_table);

	status_multi_line_edit->clear();

	string		status_string;

	compiler_interface_pointer->GenerateCodeStatusString(status_string);

	status_multi_line_edit->append(status_string.c_str());
}

CSimulatorErrorDialog::CSimulatorErrorDialog(QWidget *parent)
	: CDialogFc(parent, 3, 3, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	grid_layout->setRowStretch(0, 1);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 0);

	grid_layout->setColStretch(0, 1);
	grid_layout->setColStretch(1, 0);
	grid_layout->setColStretch(2, 0);

	QString     size_string("#####");

	error_token_listbox		= new CListBoxFc(this, "error token listbox", 8);
	ok_button				= new CPushButtonFc("Ok", this, size_string);
	close_button			= new CPushButtonFc("Close", this, size_string);
	spacer1					= new CSpacerFc(this, true);

	grid_layout->addMultiCellWidget(error_token_listbox,		0, 0, 0, 2);
	grid_layout->addWidget(ok_button,							1, 1);
	grid_layout->addWidget(close_button,						1, 2);
	grid_layout->addMultiCellWidget(spacer1,					2, 2, 0, 2);

	connect(ok_button, SIGNAL(clicked()), SLOT(SetErrorToken()));
	connect(close_button, SIGNAL(clicked()), SLOT(reject()));
}

void    CSimulatorErrorDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();
}

void	CSimulatorErrorDialog::SetCompilerDataPointer(CCompilerData *d)
{
	error_token_listbox->clear();
	error_token_listbox->insertItem("(none)");

	const list<CErrorToken *>	&et_list = d->GetErrorTokenList();

	list<CErrorToken *>::const_iterator		et_iterator = et_list.begin();

	while (et_iterator != et_list.end())
		error_token_listbox->insertItem((*et_iterator++)->GetName().c_str());

	error_token_listbox->setCurrentItem(*value_pointer + 1);
}

void	CSimulatorErrorDialog::SetErrorToken()
{
	int		item = error_token_listbox->currentItem();

	*value_pointer = item - 1;

	accept();
}

CSimulatorControlDialog::CSimulatorControlDialog(QWidget *parent)
	: CDialogFc(parent, 5, 4, true, 0,
		Qt::WType_Dialog 		|
		Qt::WShowModal			|
		Qt::WStyle_Customize	|
		Qt::WStyle_DialogBorder	|
		Qt::WStyle_Title		|
		Qt::WStyle_Dialog
		)
{
	QFontInfo		fi(font());

	int		pixel_size = fi.pixelSize();

	int		s = grid_layout_spacing_data[pixel_size];

	subgrid_layout = new QGridLayout(this, 1, 3, 0, s, "subgrid");

	grid_layout->setRowStretch(0, 0);
	grid_layout->setRowStretch(1, 0);
	grid_layout->setRowStretch(2, 1);
	grid_layout->setRowStretch(3, 0);
	grid_layout->setRowStretch(4, 0);

	grid_layout->setColStretch(0, 0);
	grid_layout->setColStretch(1, 0);
	grid_layout->setColStretch(2, 0);
	grid_layout->setColStretch(3, 1);

	subgrid_layout->setRowStretch(0, 1);

	subgrid_layout->setColStretch(0, 1);
	subgrid_layout->setColStretch(1, 0);
	subgrid_layout->setColStretch(2, 0);

	QString     size_string("######");

	simulate_type_button_group	= new CVButtonGroupFc("Simulate", this);
	lexical_radio_button	=
		new CRadioButtonFc("Lexical", simulate_type_button_group);
	parser_radio_button		=
		new CRadioButtonFc("Parser", simulate_type_button_group);
	spacer1					= new CSpacerFc(this, true);
	discard_lexical_error_check_box =
		new CCheckBoxFc("Discard Lexical Error", this);
	spacer2					= new CSpacerFc(this, true);
	with_unit_rule_eliminations_check_box =
		new CCheckBoxFc("With Unit Rule Eliminations", this);
	trap_symbol_list_view	= new Q3ListView(this);
	ok_button				= new CPushButtonFc("Ok", this, size_string);
	cancel_button			= new CPushButtonFc("Cancel", this, size_string);
	spacer3					= new CSpacerFc(this, true);

	trap_symbol_list_view->header()->setMovingEnabled(false);
	trap_symbol_list_view->setSorting(-1);
	trap_symbol_list_view->addColumn("trap symbol");

	trap_symbol_list_view->setColumnWidth(0, 160);

	subgrid_layout->addWidget(ok_button,			0, 1);
	subgrid_layout->addWidget(cancel_button,		0, 2);

	grid_layout->addMultiCellWidget(simulate_type_button_group,		0, 1, 0, 0);
	grid_layout->addWidget(spacer1,									0, 1);
	grid_layout->addWidget(discard_lexical_error_check_box,			0, 2);
	grid_layout->addWidget(spacer2,									1, 1);
	grid_layout->addWidget(with_unit_rule_eliminations_check_box,	1, 2);
	grid_layout->addMultiCellWidget(trap_symbol_list_view,			2, 2, 0, 3);
	grid_layout->addMultiCellLayout(subgrid_layout,					3, 3, 0, 3);
	grid_layout->addMultiCellWidget(spacer3,						4, 4, 0, 3);

	connect(ok_button, SIGNAL(clicked()), SLOT(SetOptions()));
	connect(cancel_button, SIGNAL(clicked()), SLOT(reject()));
}

void	CSimulatorControlDialog::SetTitleAndResize(const string &title)
{
	setCaption(title.c_str());
	updateGeometry();
}

void	CSimulatorControlDialog::setFont(const QFont &f)
{
	QFontInfo       fi(f);

	int     pixel_size = fi.pixelSize();

	int     s = grid_layout_spacing_data[pixel_size];

	subgrid_layout->setMargin(0);
	subgrid_layout->setSpacing(s);
	subgrid_layout->invalidate();

	CDialogFc::setFont(f);
}

void	CSimulatorControlDialog::SetCompilerInterfacePointer(
											CCompilerInterface *c)
{
	disconnect(	lexical_radio_button,
				SIGNAL(toggled(bool)),
				this,
				SLOT(LexicalRadioToggled(bool)));

	disconnect(	parser_radio_button,
				SIGNAL(toggled(bool)),
				this,
				SLOT(ParserRadioToggled(bool)));

	compiler_interface_pointer = c;

	CCompilerData	*d = compiler_interface_pointer->GetCompilerData();

	if (compiler_interface_pointer->GetLexicalCalculated())
	{
		lexical_radio_button->setEnabled(true);
		discard_lexical_error_check_box->setEnabled(true);
		discard_lexical_error_check_box->setChecked(
			*discard_lexical_error_flag_pointer);

		if (compiler_interface_pointer->GetParserCalculated())
		{
			parser_radio_button->setEnabled(true);

			if (*lexical_flag_pointer)
				simulate_type_button_group->setButton(0);
			else if (*parser_flag_pointer)
				simulate_type_button_group->setButton(1);
		}
		else
		{
			parser_radio_button->setEnabled(false);
			simulate_type_button_group->setButton(0);
		}

		if (compiler_interface_pointer->GetUnitRuleEliminationCalculated())
		{
			with_unit_rule_eliminations_check_box->setEnabled(true);
			with_unit_rule_eliminations_check_box->setChecked(
				*with_unit_rule_eliminations_flag_pointer);
		}
		else
		{
			with_unit_rule_eliminations_check_box->setEnabled(false);
			with_unit_rule_eliminations_check_box->setChecked(false);
		}
	}
	else
	{
		lexical_radio_button->setEnabled(false);
		parser_radio_button->setEnabled(false);
		discard_lexical_error_check_box->setEnabled(false);
		discard_lexical_error_check_box->setChecked(false);
		with_unit_rule_eliminations_check_box->setEnabled(false);
		with_unit_rule_eliminations_check_box->setChecked(false);
	}

	lexical_list_set_flag	= false;
	parser_list_set_flag	= false;

	if (*lexical_flag_pointer)
		SetTrapLexicalList();
	else if (*parser_flag_pointer)
		SetTrapParserList();
	else
		trap_symbol_list_view->clear();

	connect(lexical_radio_button,
			SIGNAL(toggled(bool)),
			this,
			SLOT(LexicalRadioToggled(bool)));

	connect(parser_radio_button,
			SIGNAL(toggled(bool)),
			this,
			SLOT(ParserRadioToggled(bool)));
}

void	CSimulatorControlDialog::LexicalRadioToggled(bool on)
{
	if (on)
		SetTrapLexicalList();
}

void	CSimulatorControlDialog::ParserRadioToggled(bool on)
{
	if (on)
		SetTrapParserList();
}

int		IntegerLength(int i);

void	CSimulatorControlDialog::SetTrapLexicalList()
{
	if (parser_list_set_flag)
		StoreTrapParserList();

	lexical_list_set_flag	= true;
	parser_list_set_flag	= false;

	trap_symbol_list_view->clear();

	list< pair<bool, string> >::reverse_iterator
		t_iterator = trap_lexical_list_pointer->rbegin();

	int		i = trap_lexical_list_pointer->size() - 1;
	int		w = IntegerLength(trap_lexical_list_pointer->size() - 1);

	while (t_iterator != trap_lexical_list_pointer->rend())
	{
		pair<bool, string>	&p = *t_iterator;

		ostrstream	os;

		os << setw(w) << i << ") " << p.second << '\0';

		Q3CheckListItem	*item =
			new Q3CheckListItem(trap_symbol_list_view,
								os.str(),
								Q3CheckListItem::CheckBox);

		item->setOn(p.first);

		t_iterator++;
		i--;
	}
}

void	CSimulatorControlDialog::SetTrapParserList()
{
	if (lexical_list_set_flag)
		StoreTrapLexicalList();

	lexical_list_set_flag	= false;
	parser_list_set_flag	= true;

	trap_symbol_list_view->clear();

	list< pair<bool, string> >::reverse_iterator
		t_iterator = trap_parser_list_pointer->rbegin();

	int		i = trap_parser_list_pointer->size() - 1;
	int		w = IntegerLength(trap_parser_list_pointer->size() - 1);

	while (t_iterator != trap_parser_list_pointer->rend())
	{
		pair<bool, string>	&p = *t_iterator;

		ostrstream	os;

		os << setw(w) << i << ") " << p.second << '\0';

		Q3CheckListItem	*item =
			new Q3CheckListItem(trap_symbol_list_view,
								os.str(),
								Q3CheckListItem::CheckBox);

		item->setOn(p.first);

		t_iterator++;
		i--;
	}
}

void	CSimulatorControlDialog::StoreTrapLexicalList()
{
	list< pair<bool, string> >::iterator
		t_iterator = trap_lexical_list_pointer->begin();

	Q3ListViewItemIterator	l_iterator(trap_symbol_list_view);

	while (t_iterator != trap_lexical_list_pointer->end())
	{
		pair<bool, string>	&p = *t_iterator;

		Q3ListViewItem	*item = l_iterator.current();

		Q3CheckListItem	*check = dynamic_cast<Q3CheckListItem *>(item);

		if (check->isOn())
			p.first = true;
		else
			p.first = false;

		t_iterator++;
		l_iterator++;
	}
}

void	CSimulatorControlDialog::StoreTrapParserList()
{
	list< pair<bool, string> >::iterator
		t_iterator = trap_parser_list_pointer->begin();

	Q3ListViewItemIterator	l_iterator(trap_symbol_list_view);

	while (t_iterator != trap_parser_list_pointer->end())
	{
		pair<bool, string>	&p = *t_iterator;

		Q3ListViewItem	*item = l_iterator.current();

		Q3CheckListItem	*check = dynamic_cast<Q3CheckListItem *>(item);

		if (check->isOn())
			p.first = true;
		else
			p.first = false;

		t_iterator++;
		l_iterator++;
	}
}

void	CSimulatorControlDialog::SetOptions()
{
	*lexical_flag_pointer =
		lexical_radio_button->isChecked();
	*parser_flag_pointer =
		parser_radio_button->isChecked();
	*discard_lexical_error_flag_pointer =
		discard_lexical_error_check_box->isChecked();
	*with_unit_rule_eliminations_flag_pointer =
		with_unit_rule_eliminations_check_box->isChecked();

	if (lexical_list_set_flag)
		StoreTrapLexicalList();
	else if (parser_list_set_flag)
		StoreTrapParserList();

	accept();
}
