/*****************************************************************************/
/*                                                                           */
/*  Compiler - a Parser Generator Program, Version 2.0                       */
/*  Copyright (c) 2000, 2003  Charles M. Fayle  All Rights Reserved.         */
/*                                                                           */
/*  This software is distributed under the terms of the GNU General Public   */
/*  License as specified in the file gpl.txt included with the distribution. */
/*                                                                           */
/*****************************************************************************/
//
//  $Id$
//

#include "terminalSelectView.h"

void	CTerminalSelectListView::InitializePalette(const QPalette &p)
{
	active_highlight			= p.color(	QPalette::Active,
											QColorGroup::Highlight);
	active_highlighted_text		= p.color(	QPalette::Active,
											QColorGroup::HighlightedText);
	inactive_highlight			= p.color(	QPalette::Inactive,
											QColorGroup::Highlight);
	inactive_highlighted_text	= p.color(	QPalette::Inactive,
											QColorGroup::HighlightedText);

	QPalette	focus_palette = p;

	focus_palette.setColor(	QPalette::Active,
							QColorGroup::Highlight,
							inactive_highlight);

	focus_palette.setColor(	QPalette::Active,
							QColorGroup::HighlightedText,
							inactive_highlighted_text);

	setPalette(focus_palette);
}

void	CTerminalSelectListView::focusInEvent(QFocusEvent *e)
{
	QPalette	p = palette();

	p.setColor(	QPalette::Active,
				QColorGroup::Highlight,
				active_highlight);

	p.setColor(	QPalette::Active,
				QColorGroup::HighlightedText,
				active_highlighted_text);

	setPalette(p);

	Q3ListView::focusInEvent(e);
}

void	CTerminalSelectListView::focusOutEvent(QFocusEvent *e)
{
	QPalette	p = palette();

	p.setColor(	QPalette::Active,
				QColorGroup::Highlight,
				inactive_highlight);

	p.setColor(	QPalette::Active,
				QColorGroup::HighlightedText,
				inactive_highlighted_text);

	setPalette(p);

	Q3ListView::focusOutEvent(e);
}

CTerminalSelectVBoxWidget::CTerminalSelectVBoxWidget(
									QWidget *parent,
									CCompilerInterface *ci,
									SMainViewControl *mvc,
									COptionData::SSubViewData *d,
									CTerminalSelectView *tsv)
	: CVBoxWidget(parent, ci, mvc), compiler_interface(ci),
		subview_data(d), terminal_select_view(tsv),
		module_ptr_map(tsv->GetModulePtrMap())
{
	label = new CLabelFc("terminal select", this, true);

	label->setAlignment(Qt::AlignHCenter | Qt::AlignVCenter);

	list_view = new CTerminalSelectListView(this);

	list_view->header()->setMovingEnabled(false);
	list_view->setSorting(-1);
	list_view->addColumn("index");
	list_view->addColumn("function");

	setSpacing(0);
	setStretchFactor(label, 0);
	setStretchFactor(list_view, 1);

	compiler_data = compiler_interface->GetCompilerData();

	tcv = 0;

	UpdateListView();
}

CTerminalSelectVBoxWidget::~CTerminalSelectVBoxWidget()
{
}

void	CTerminalSelectVBoxWidget::InitializeModulePointers()
{
	vector<CSubView *>		&v1 = module_ptr_map["TERMINAL_CODE_VIEW"];

	if (v1.size())
		tcv = dynamic_cast<CTerminalCodeView *>(v1[0]);
	else
		tcv = 0;

	UpdateTerminalViews();
}

string	&IntegerString(int i, string &s);

void	CTerminalSelectVBoxWidget::IndexString(int index, string &s)
{
	IntegerString(index, s);
}

void	CTerminalSelectVBoxWidget::FunctionString(const CTerminal *t, string &s)
{
	ostrstream	os;

	if (t->GetFilterFlag())
		os << "FILTER  ";
	else
		os << "TOKEN   ";

	if (t->GetLiteralFlag())
		os << "LITERAL  ";
	else
		os << "REGEXP   ";

	string  pad1(compiler_data->GetTerminalNameWidth() -
					t->GetName().length() + 2, ' ');

	os << t->GetName() << pad1 << t->GetRegexp() << '\0';

	s = os.str();
}

void	CTerminalSelectVBoxWidget::UpdateListView()
{
	QObject::disconnect(list_view,
						SIGNAL(currentChanged(Q3ListViewItem *)),
						this,
						SLOT(SelectTerminalFunction(Q3ListViewItem *)));

	compiler_data->CalculateOutputWidths();

	const list<CTerminal *>		&t_list = compiler_data->GetTerminalList();

	int		t_count = t_list.size();
	int		l_count = list_view->childCount();
	int		i;

	if (t_count < l_count)
	{
		for (i=0; i<l_count-t_count; i++)
			delete list_view->firstChild();
	}
	else if (t_count > l_count)
	{
		for (i=0; i<t_count-l_count; i++)
			new Q3ListViewItem(list_view);
	}

	i = 0;

	list<CTerminal *>::const_iterator	t_iterator  = t_list.begin();

	Q3ListViewItemIterator	l_iterator(list_view);

	while (t_iterator != t_list.end())
	{
		const CTerminal		*t = *t_iterator;

		Q3ListViewItem	*item = l_iterator.current();

		string	s0;
		string	s1;

		IndexString(i, s0);
		FunctionString(t, s1);

		item->setText(0, s0.c_str());
		item->setText(1, s1.c_str());

		i++;
		t_iterator++;
		l_iterator++;
	}

	if (t_count)
	{
		list_view->setSelected(list_view->firstChild(), true);

		current_function = 0;
	}
	else
		current_function = -1;

	QObject::connect(	list_view,
						SIGNAL(currentChanged(Q3ListViewItem *)),
						this,
						SLOT(SelectTerminalFunction(Q3ListViewItem *)));

	UpdateTerminalViews();
}

void	CTerminalSelectVBoxWidget::UpdateTerminalViews()
{
	if (current_function != -1)
	{
		if (tcv)
		{
			CTerminal	*t = compiler_data->FindTerminalIndex(current_function);

			if (t && t->GetCodeFlag())
				tcv->SetMultiLineEditText(t->GetCodeText());
			else
				tcv->ClearMultiLineEditText();
		}
	}
	else
	{
		if (tcv)
			tcv->ClearMultiLineEditText();
	}
}

void	CTerminalSelectVBoxWidget::GetCurrentFunctionString(string &s)
{
	if (current_function != -1)
	{
		string		index_string;
		string		function_string;

		IndexString(current_function, index_string);

		CTerminal	*t = compiler_data->FindTerminalIndex(current_function);

		FunctionString(t, function_string);

		s  = index_string;
		s += " ";
		s += function_string;
	}
	else
		s = "";
}

void	CTerminalSelectVBoxWidget::PostCurrentFunctionChanges()
{
	if (current_function != -1)
	{
		CTerminal	*t = compiler_data->FindTerminalIndex(current_function);

		if (!t)
			return;

		list<string>	&c_list = t->GetCodeText();

		if (tcv)
			tcv->GetMultiLineEditText(c_list);

		list<string>::const_iterator	c_iterator = c_list.begin();

		while (c_iterator != c_list.end())
		{
			if ((*c_iterator).find_first_not_of("\t\n ") != string::npos)
				break;

			c_iterator++;
		}

		if (c_iterator == c_list.end())
		{
			c_list.clear();

			t->SetCodeFlag(false);
		}
		else
			t->SetCodeFlag(true);
	}
}

void	CTerminalSelectVBoxWidget::SelectTerminalFunction(Q3ListViewItem *item)
{
	PostCurrentFunctionChanges();

	if (item)
		current_function = item->text(0).toInt();
	else
		current_function = -1;

	UpdateTerminalViews();
}

void	CTerminalSelectVBoxWidget::RestoreUndoData()
{
	compiler_data = compiler_interface->GetCompilerData();
}

void	CTerminalSelectVBoxWidget::CodeTextChanged()
{
	compiler_interface->SetDataModifiedFlag(true);
}

CTerminalSelectView::CTerminalSelectView(	QWidget *p,
											CViewNode *n,
											CInterfaceControl *ic,
											SMainViewControl *mvc,
											const string &dfs,
											CPaletteData *dpd,
											COptionData::SSubViewData *d)
	: CSubView(p, n, ic, mvc, dfs, dpd, d)
{
	frame = new CTerminalSelectVBoxWidget(
		p, dynamic_cast<CCompilerInterface *>(ic), mvc, d, this);

	QFont		view_node_font;
	QPalette	view_node_palette;

	if (SetupViewNodeFont(view_node_font))
		frame->setFont(view_node_font);
	else
		frame->setFont(frame->font());

	if (SetupViewNodePalette(view_node_palette))
	{
		frame->setPalette(view_node_palette);
		frame->InitializePalette(view_node_palette);
	}
	else
	{
		frame->setPalette(frame->palette());
		frame->InitializePalette(frame->palette());
	}
}

CTerminalSelectView::~CTerminalSelectView()
{
	delete frame;
}

void	CTerminalSelectView::SetHighlightPalette(CPaletteData *pd)
{
	restore_palette = frame->palette();

	QPalette	highlight_palette;

	pd->SetupPalette(highlight_palette);

	frame->setPalette(highlight_palette);
}

void	CTerminalSelectView::RestorePalette()
{
	frame->setPalette(restore_palette);
}

void	CTerminalSelectView::InitializeModulePointers()
{
	frame->InitializeModulePointers();
}

void	CTerminalSelectView::UpdateListView()
{
	frame->UpdateListView();
}

void	CTerminalSelectView::GetCurrentFunctionString(string &s)
{
	frame->GetCurrentFunctionString(s);
}

void	CTerminalSelectView::PostCurrentFunctionChanges()
{
	frame->PostCurrentFunctionChanges();
}

void	CTerminalSelectView::RestoreUndoData()
{
	frame->RestoreUndoData();
}

void	CTerminalSelectView::CodeTextChanged()
{
	frame->CodeTextChanged();
}
